!>  \file radsw_datatb.f
!!  This file contains the following:
!!    - module_radsw_ref         (reference temperature and pressure)
!!    - module_radsw_cldprtb     (cloud property coefficients table)
!!    - module_radsw_sflux       (spectral solar flux distribution)
!!    - module_radsw_kgbnn       (absorption coeffients for 14 bands, 
!!                                where nn = 16-29)

!  ==============================================================  !!!!!
!              sw-rrtm3 radiation package description              !!!!!
!  ==============================================================  !!!!!
!                                                                          !
!   this package includes ncep's modifications of the rrtm-sw radiation    !
!   code from aer inc.                                                     !
!                                                                          !
!    the sw-rrtm3 package includes these parts:                            !
!                                                                          !
!       'radsw_rrtm3_param.f'                                              !
!       'radsw_rrtm3_datatb.f'                                             !
!       'radsw_rrtm3_main.f'                                               !
!                                                                          !
!    the 'radsw_rrtm3_param.f' contains:                                   !
!                                                                          !
!       'module_radsw_parameters'  -- band parameters set up               !
!                                                                          !
!    the 'radsw_rrtm3_datatb.f' contains:                                  !
!                                                                          !
!       'module_radsw_ref'         -- reference temperature and pressure   !
!       'module_radsw_cldprtb'     -- cloud property coefficients table    !
!       'module_radsw_sflux'       -- spectral solar flux distribution     !
!       'module_radsw_kgbnn'       -- absorption coeffients for 14         !
!                                     bands, where nn = 16-29              !
!                                                                          !
!    the 'radsw_rrtm3_main.f' contains:                                    !
!                                                                          !
!       'module_radsw_main'        -- main sw radiation transfer           !
!                                                                          !
!    in the main module 'module_radsw_main' there are only two             !
!    externally callable subroutines:                                      !
!                                                                          !
!       'swrad'      -- main rrtm3 sw radiation routine                    !
!       'rswinit'    -- initialization routine                             !
!                                                                          !
!    all the sw radiation subprograms become contained subprograms         !
!    in module 'module_radsw_main' and many of them are not directly       !
!    accessable from places outside the module.                            !
!                                                                          !
!    compilation sequence is:                                              !
!                                                                          !
!       'radsw_rrtm3_param.f'                                              !
!       'radsw_rrtm3_datatb.f'                                             !
!       'radsw_rrtm3_main.f'                                               !
!                                                                          !
!    and all should be put in front of routines that use sw modules        !
!                                                                          !
!   ncep modifications history log:                                        !
!                                                                          !
!       see list in program "radsw_rrtm3_main.f"                           !
!                                                                          !
!!!!!  ==============================================================  !!!!!
!!!!!                         end descriptions                         !!!!!
!!!!!  ==============================================================  !!!!!


!> \ingroup module_radlw_main
!> This module contains reference temperature and pressure.
!!
!! - These pressures are chosen such that the ln of the first pressure
!!   has only a few non-zero digits (i.e. ln(pref(1)) = 6.96000) and
!!   each subsequent ln(pressure) differs from the previous one by 0.2.
!! - These temperatures are associated with the respective
!!   pressures for the MLS standard atmosphere.
!========================================!
      module module_radsw_ref            !
!........................................!
!
      use physparam,               only : kind_phys
!
      implicit none
!
      public

!> Reference pressure and temperature
      real (kind=kind_phys), dimension(59) :: preflog, tref

!  ...  these pressures are chosen such that the ln of the first pressure
!       has only a few non-zero digits (i.e. ln(pref(1)) = 6.96000) and
!       each subsequent ln(pressure) differs from the previous one by 0.2.

!     data  pref(:)  /                                                  &
!    &     1.05363e+03,8.62642e+02,7.06272e+02,5.78246e+02,4.73428e+02, &
!    &     3.87610e+02,3.17348e+02,2.59823e+02,2.12725e+02,1.74164e+02, &
!    &     1.42594e+02,1.16746e+02,9.55835e+01,7.82571e+01,6.40715e+01, &
!    &     5.24573e+01,4.29484e+01,3.51632e+01,2.87892e+01,2.35706e+01, &
!    &     1.92980e+01,1.57998e+01,1.29358e+01,1.05910e+01,8.67114e+00, &
!    &     7.09933e+00,5.81244e+00,4.75882e+00,3.89619e+00,3.18993e+00, &
!    &     2.61170e+00,2.13828e+00,1.75067e+00,1.43333e+00,1.17351e+00, &
!    &     9.60789e-01,7.86628e-01,6.44036e-01,5.27292e-01,4.31710e-01, &
!    &     3.53455e-01,2.89384e-01,2.36928e-01,1.93980e-01,1.58817e-01, &
!    &     1.30029e-01,1.06458e-01,8.71608e-02,7.13612e-02,5.84256e-02, &
!    &     4.78349e-02,3.91639e-02,3.20647e-02,2.62523e-02,2.14936e-02, &
!    &     1.75975e-02,1.44076e-02,1.17959e-02,9.65769e-03 /

      data  preflog(:)  /                                               &
     &      6.9600e+00, 6.7600e+00, 6.5600e+00, 6.3600e+00, 6.1600e+00, &
     &      5.9600e+00, 5.7600e+00, 5.5600e+00, 5.3600e+00, 5.1600e+00, &
     &      4.9600e+00, 4.7600e+00, 4.5600e+00, 4.3600e+00, 4.1600e+00, &
     &      3.9600e+00, 3.7600e+00, 3.5600e+00, 3.3600e+00, 3.1600e+00, &
     &      2.9600e+00, 2.7600e+00, 2.5600e+00, 2.3600e+00, 2.1600e+00, &
     &      1.9600e+00, 1.7600e+00, 1.5600e+00, 1.3600e+00, 1.1600e+00, &
     &      9.6000e-01, 7.6000e-01, 5.6000e-01, 3.6000e-01, 1.6000e-01, &
     &     -4.0000e-02,-2.4000e-01,-4.4000e-01,-6.4000e-01,-8.4000e-01, &
     &     -1.0400e+00,-1.2400e+00,-1.4400e+00,-1.6400e+00,-1.8400e+00, &
     &     -2.0400e+00,-2.2400e+00,-2.4400e+00,-2.6400e+00,-2.8400e+00, &
     &     -3.0400e+00,-3.2400e+00,-3.4400e+00,-3.6400e+00,-3.8400e+00, &
     &     -4.0400e+00,-4.2400e+00,-4.4400e+00,-4.6400e+00 /

!  ...  these are the temperatures associated with the respective
!       pressures for the MLS standard atmosphere.
      data  tref(:)  /                                                  &
     &      2.9420e+02, 2.8799e+02, 2.7894e+02, 2.6925e+02, 2.5983e+02, &
     &      2.5017e+02, 2.4077e+02, 2.3179e+02, 2.2306e+02, 2.1578e+02, &
     &      2.1570e+02, 2.1570e+02, 2.1570e+02, 2.1706e+02, 2.1858e+02, &
     &      2.2018e+02, 2.2174e+02, 2.2328e+02, 2.2479e+02, 2.2655e+02, &
     &      2.2834e+02, 2.3113e+02, 2.3401e+02, 2.3703e+02, 2.4022e+02, &
     &      2.4371e+02, 2.4726e+02, 2.5085e+02, 2.5457e+02, 2.5832e+02, &
     &      2.6216e+02, 2.6606e+02, 2.6999e+02, 2.7340e+02, 2.7536e+02, &
     &      2.7568e+02, 2.7372e+02, 2.7163e+02, 2.6955e+02, 2.6593e+02, &
     &      2.6211e+02, 2.5828e+02, 2.5360e+02, 2.4854e+02, 2.4348e+02, &
     &      2.3809e+02, 2.3206e+02, 2.2603e+02, 2.2000e+02, 2.1435e+02, &
     &      2.0887e+02, 2.0340e+02, 1.9792e+02, 1.9290e+02, 1.8809e+02, &
     &      1.8329e+02, 1.7849e+02, 1.7394e+02, 1.7212e+02 /

!........................................!
      end module module_radsw_ref        !
!========================================!

!> \ingroup module_radlw_main
!> This module contains coefficients of cloud optical properties for
!! each of the spectral bands.
!========================================!
      module module_radsw_cldprtb        !
!........................................!
!
! ***********************  module descriptions  ********************** !
!                                                                      !
!     this module contains coefficients of cloud-optical properties    !
!     for each of the spectral bands.  modified from aer/ecmwf rrtm    !
!     sw radiation package subroutine "susrtop".                       !
!                                                                      !
!     modify history:                                                  !
!       jan 2004   --  yu-tai hou    convert subroutine to data        !
!                      module for ncep models.                         !
!       jun 2008   --  yu-tai hou    modified to use aer's newer       !
!                      release v3.5 data table for cloud ice particles !
!                                                                      !
! *********         the original program descriptions        ********* !
!                                                                      !
!     adapted from j. delamere,  atmospheric & environmental research. !
!     by jjmorcrette,  ecmwf    02/10/29                               !
!     revision:  j.d. 2.6 2002/04/04 18:29:47                          !
!     reformatted for f90 by jjmorcrette, ecmwf                        !
!                                                                      !
!  name     type     purpose
!  ----   : ----   : --------------------------------------------------!
! xxxliq1 : real   : optical properties (extinction coefficient, single!
!                    scattering albedo, assymetry factor) from         !
!                    hu & stamnes, 1993, j. clim., 6, 728-742          !
! xbari   : real   : optical properties (extinction coefficient, single!
!                    scattering albedo, assymetry factor) calculated   !
!                    based on ebert and curry method, jgr, 1992        !
! xxxice2 : real   : optical properties (extinction coefficient, single!
!                    scattering albedo, assymetry factor) from         !
!                    streamer v3.0, key.j.streamer user's guide,       !
!                    cooperative institute for meteorological satellite!
!                    studies, 2001, 96 pp.                             !
! xxxice3 : real   : optical properties (extinction coefficient, single!
!                    scattering albedo, assymetry factor) from         !
!                    fu, 1996, j. clim., 9,                            !
!                                                                      !
! *************************  end description  ************************ !

      use physparam,               only : kind_phys
      use module_radsw_parameters, only : nblow, nbhgh
!
      implicit none
!
      private

!
! === for iflagliq = 0,
!     cloud optical properties are external inputs to the main program

! === everything below is for iflagliq >= 1.
!> extinction coefficient from hu and stamnes
      real (kind=kind_phys), dimension(58,nblow:nbhgh), public ::       &
     &       extliq1
!> single scattering albedo from hu and stamnes
      real (kind=kind_phys), dimension(58,nblow:nbhgh), public ::       &
     &       ssaliq1
!> asymmetry parameter from hu and stamnes
      real (kind=kind_phys), dimension(58,nblow:nbhgh), public ::       &
     &       asyliq1
!> spherical ice particle parameterization from streamer v3,
!! extinction units (ext coef/iwc): \f$\frac{m^{-1}}{gm^{-3}}\f$
      real (kind=kind_phys), dimension(43,nblow:nbhgh), public ::       &
     &       extice2
!> single-scattering albedo from streamer v3, unitless
      real (kind=kind_phys), dimension(43,nblow:nbhgh), public ::       &
     &       ssaice2
!> asymmetry factor from streamer v3, unitless
      real (kind=kind_phys), dimension(43,nblow:nbhgh), public ::       &
     &       asyice2   
!> hexagonal ice particle parameterization from fu,
!! extinction units (ext coef/iwc): \f$\frac{m^{-1}}{gm^{-3}}\f$
      real (kind=kind_phys), dimension(46,nblow:nbhgh), public ::       &
     &       extice3
!> single-scattering albedo from fu, unitless
      real (kind=kind_phys), dimension(46,nblow:nbhgh), public ::       &
     &       ssaice3
!> asymmetry factor from fu, unitless
      real (kind=kind_phys), dimension(46,nblow:nbhgh), public ::       &
     &       asyice3
!> fdelta from fu, unitless
      real (kind=kind_phys), dimension(46,nblow:nbhgh), public ::       &
     &       fdlice3

!> \name coefficients from ebert and curry method
!!@{
      real (kind=kind_phys), dimension(5), public ::                    &
     &       abari, bbari, cbari, dbari, ebari, fbari
!!@}

!  --- ...  coefficients from ebert and curry method
      data abari(:)/ 3.448e-03,3.448e-03,3.448e-03,3.448e-03,3.448e-03 /
      data bbari(:)/ 2.431e+00,2.431e+00,2.431e+00,2.431e+00,2.431e+00 /
      data cbari(:)/ 1.000e-05,1.100e-04,1.240e-02,3.779e-02,4.666e-01 /
      data dbari(:)/ 0.000e+00,1.405e-05,6.867e-04,1.284e-03,2.050e-05 /
      data ebari(:)/ 7.661e-01,7.730e-01,7.865e-01,8.172e-01,9.595e-01 /
      data fbari(:)/ 5.851e-04,5.665e-04,7.204e-04,7.463e-04,1.076e-04 /

!  --- ...  extinction coefficient from hu and stamnes
      data  extliq1(:, 16)   /                                          &
     & 8.981463e-01,6.317895e-01,4.557508e-01,3.481624e-01,2.797950e-01,&
     & 2.342753e-01,2.026934e-01,1.800102e-01,1.632408e-01,1.505384e-01,&
     & 1.354524e-01,1.246520e-01,1.154342e-01,1.074756e-01,1.005353e-01,&
     & 9.442987e-02,8.901760e-02,8.418693e-02,7.984904e-02,7.593229e-02,&
     & 7.237827e-02,6.913887e-02,6.617415e-02,6.345061e-02,6.094001e-02,&
     & 5.861834e-02,5.646506e-02,5.446250e-02,5.249596e-02,5.081114e-02,&
     & 4.922243e-02,4.772189e-02,4.630243e-02,4.495766e-02,4.368189e-02,&
     & 4.246995e-02,4.131720e-02,4.021941e-02,3.917276e-02,3.817376e-02,&
     & 3.721926e-02,3.630635e-02,3.543237e-02,3.459491e-02,3.379171e-02,&
     & 3.302073e-02,3.228007e-02,3.156798e-02,3.088284e-02,3.022315e-02,&
     & 2.958753e-02,2.897468e-02,2.838340e-02,2.781258e-02,2.726117e-02,&
     & 2.672821e-02,2.621278e-02,2.5714e-02 /
      data  extliq1(:, 17)   /                                          &
     & 8.293797e-01,6.048371e-01,4.465706e-01,3.460387e-01,2.800064e-01,&
     & 2.346584e-01,2.022399e-01,1.782626e-01,1.600153e-01,1.457903e-01,&
     & 1.334061e-01,1.228548e-01,1.138396e-01,1.060486e-01,9.924856e-02,&
     & 9.326208e-02,8.795158e-02,8.320883e-02,7.894750e-02,7.509792e-02,&
     & 7.160323e-02,6.841653e-02,6.549889e-02,6.281763e-02,6.034516e-02,&
     & 5.805802e-02,5.593615e-02,5.396226e-02,5.202302e-02,5.036246e-02,&
     & 4.879606e-02,4.731610e-02,4.591565e-02,4.458852e-02,4.332912e-02,&
     & 4.213243e-02,4.099390e-02,3.990941e-02,3.887522e-02,3.788792e-02,&
     & 3.694440e-02,3.604183e-02,3.517760e-02,3.434934e-02,3.355485e-02,&
     & 3.279211e-02,3.205925e-02,3.135458e-02,3.067648e-02,3.002349e-02,&
     & 2.939425e-02,2.878748e-02,2.820200e-02,2.763673e-02,2.709062e-02,&
     & 2.656272e-02,2.605214e-02,2.5558e-02 /
      data  extliq1(:, 18)   /                                          &
     & 9.193685e-01,6.128292e-01,4.344150e-01,3.303048e-01,2.659500e-01,&
     & 2.239727e-01,1.953457e-01,1.751012e-01,1.603515e-01,1.493360e-01,&
     & 1.323791e-01,1.219335e-01,1.130076e-01,1.052926e-01,9.855839e-02,&
     & 9.262925e-02,8.736918e-02,8.267112e-02,7.844965e-02,7.463585e-02,&
     & 7.117343e-02,6.801601e-02,6.512503e-02,6.246815e-02,6.001806e-02,&
     & 5.775154e-02,5.564872e-02,5.369250e-02,5.176284e-02,5.011536e-02,&
     & 4.856099e-02,4.709211e-02,4.570193e-02,4.438430e-02,4.313375e-02,&
     & 4.194529e-02,4.081443e-02,3.973712e-02,3.870966e-02,3.772866e-02,&
     & 3.679108e-02,3.589409e-02,3.503514e-02,3.421185e-02,3.342206e-02,&
     & 3.266377e-02,3.193513e-02,3.123447e-02,3.056018e-02,2.991081e-02,&
     & 2.928502e-02,2.868154e-02,2.809920e-02,2.753692e-02,2.699367e-02,&
     & 2.646852e-02,2.596057e-02,2.5469e-02 /
      data  extliq1(:, 19)   /                                          &
     & 9.136931e-01,5.743244e-01,4.080708e-01,3.150572e-01,2.577261e-01,&
     & 2.197900e-01,1.933037e-01,1.740212e-01,1.595056e-01,1.482756e-01,&
     & 1.312164e-01,1.209246e-01,1.121227e-01,1.045095e-01,9.785967e-02,&
     & 9.200149e-02,8.680170e-02,8.215531e-02,7.797850e-02,7.420361e-02,&
     & 7.077530e-02,6.764798e-02,6.478369e-02,6.215063e-02,5.972189e-02,&
     & 5.747458e-02,5.538913e-02,5.344866e-02,5.153216e-02,4.989745e-02,&
     & 4.835476e-02,4.689661e-02,4.551629e-02,4.420777e-02,4.296563e-02,&
     & 4.178497e-02,4.066137e-02,3.959081e-02,3.856963e-02,3.759452e-02,&
     & 3.666244e-02,3.577061e-02,3.491650e-02,3.409777e-02,3.331227e-02,&
     & 3.255803e-02,3.183322e-02,3.113617e-02,3.046530e-02,2.981918e-02,&
     & 2.919646e-02,2.859591e-02,2.801635e-02,2.745671e-02,2.691599e-02,&
     & 2.639324e-02,2.588759e-02,2.5398e-02 /
      data  extliq1(:, 20)   /                                          &
     & 8.447548e-01,5.326840e-01,3.921523e-01,3.119082e-01,2.597055e-01,&
     & 2.228737e-01,1.954157e-01,1.741155e-01,1.570881e-01,1.431520e-01,&
     & 1.302034e-01,1.200491e-01,1.113571e-01,1.038330e-01,9.725657e-02,&
     & 9.145949e-02,8.631112e-02,8.170840e-02,7.756901e-02,7.382641e-02,&
     & 7.042616e-02,6.732338e-02,6.448069e-02,6.186672e-02,5.945494e-02,&
     & 5.722277e-02,5.515089e-02,5.322262e-02,5.132153e-02,4.969799e-02,&
     & 4.816556e-02,4.671686e-02,4.534525e-02,4.404480e-02,4.281014e-02,&
     & 4.163643e-02,4.051930e-02,3.945479e-02,3.843927e-02,3.746945e-02,&
     & 3.654234e-02,3.565518e-02,3.480547e-02,3.399088e-02,3.320930e-02,&
     & 3.245876e-02,3.173745e-02,3.104371e-02,3.037600e-02,2.973287e-02,&
     & 2.911300e-02,2.851516e-02,2.793818e-02,2.738101e-02,2.684264e-02,&
     & 2.632214e-02,2.581863e-02,2.5331e-02 /
      data  extliq1(:, 21)   /                                          &
     & 7.727642e-01,5.034865e-01,3.808673e-01,3.080333e-01,2.586453e-01,&
     & 2.224989e-01,1.947060e-01,1.725821e-01,1.545096e-01,1.394456e-01,&
     & 1.288683e-01,1.188852e-01,1.103317e-01,1.029214e-01,9.643967e-02,&
     & 9.072239e-02,8.564194e-02,8.109758e-02,7.700875e-02,7.331026e-02,&
     & 6.994879e-02,6.688028e-02,6.406807e-02,6.148133e-02,5.909400e-02,&
     & 5.688388e-02,5.483197e-02,5.292185e-02,5.103763e-02,4.942905e-02,&
     & 4.791039e-02,4.647438e-02,4.511453e-02,4.382497e-02,4.260043e-02,&
     & 4.143616e-02,4.032784e-02,3.927155e-02,3.826375e-02,3.730117e-02,&
     & 3.638087e-02,3.550013e-02,3.465646e-02,3.384759e-02,3.307141e-02,&
     & 3.232598e-02,3.160953e-02,3.092040e-02,3.025706e-02,2.961810e-02,&
     & 2.900220e-02,2.840814e-02,2.783478e-02,2.728106e-02,2.674599e-02,&
     & 2.622864e-02,2.572816e-02,2.5244e-02 /
      data  extliq1(:, 22)   /                                          &
     & 7.416833e-01,4.959591e-01,3.775057e-01,3.056353e-01,2.565943e-01,&
     & 2.206935e-01,1.931479e-01,1.712860e-01,1.534837e-01,1.386906e-01,&
     & 1.281198e-01,1.182344e-01,1.097595e-01,1.024137e-01,9.598552e-02,&
     & 9.031320e-02,8.527093e-02,8.075927e-02,7.669869e-02,7.302481e-02,&
     & 6.968491e-02,6.663542e-02,6.384008e-02,6.126838e-02,5.889452e-02,&
     & 5.669654e-02,5.465558e-02,5.275540e-02,5.087937e-02,4.927904e-02,&
     & 4.776796e-02,4.633895e-02,4.498557e-02,4.370202e-02,4.248306e-02,&
     & 4.132399e-02,4.022052e-02,3.916878e-02,3.816523e-02,3.720665e-02,&
     & 3.629011e-02,3.541290e-02,3.457257e-02,3.376685e-02,3.299365e-02,&
     & 3.225105e-02,3.153728e-02,3.085069e-02,3.018977e-02,2.955310e-02,&
     & 2.893940e-02,2.834742e-02,2.777606e-02,2.722424e-02,2.669099e-02,&
     & 2.617539e-02,2.567658e-02,2.5194e-02 /
      data  extliq1(:, 23)   /                                          &
     & 7.058580e-01,4.866573e-01,3.712238e-01,2.998638e-01,2.513441e-01,&
     & 2.161972e-01,1.895576e-01,1.686669e-01,1.518437e-01,1.380046e-01,&
     & 1.267564e-01,1.170399e-01,1.087026e-01,1.014704e-01,9.513729e-02,&
     & 8.954555e-02,8.457221e-02,8.012009e-02,7.611136e-02,7.248294e-02,&
     & 6.918317e-02,6.616934e-02,6.340584e-02,6.086273e-02,5.851465e-02,&
     & 5.634001e-02,5.432027e-02,5.243946e-02,5.058070e-02,4.899628e-02,&
     & 4.749975e-02,4.608411e-02,4.474303e-02,4.347082e-02,4.226237e-02,&
     & 4.111303e-02,4.001861e-02,3.897528e-02,3.797959e-02,3.702835e-02,&
     & 3.611867e-02,3.524791e-02,3.441364e-02,3.361360e-02,3.284577e-02,&
     & 3.210823e-02,3.139923e-02,3.071716e-02,3.006052e-02,2.942791e-02,&
     & 2.881806e-02,2.822974e-02,2.766185e-02,2.711335e-02,2.658326e-02,&
     & 2.607066e-02,2.557473e-02,2.5095e-02 /
      data  extliq1(:, 24)   /                                          &
     & 6.822779e-01,4.750373e-01,3.634834e-01,2.940726e-01,2.468060e-01,&
     & 2.125768e-01,1.866586e-01,1.663588e-01,1.500326e-01,1.366192e-01,&
     & 1.253472e-01,1.158052e-01,1.076101e-01,1.004954e-01,9.426089e-02,&
     & 8.875268e-02,8.385090e-02,7.946063e-02,7.550578e-02,7.192466e-02,&
     & 6.866669e-02,6.569001e-02,6.295971e-02,6.044642e-02,5.812526e-02,&
     & 5.597500e-02,5.397746e-02,5.211690e-02,5.027505e-02,4.870703e-02,&
     & 4.722555e-02,4.582373e-02,4.449540e-02,4.323497e-02,4.203742e-02,&
     & 4.089821e-02,3.981321e-02,3.877867e-02,3.779118e-02,3.684762e-02,&
     & 3.594514e-02,3.508114e-02,3.425322e-02,3.345917e-02,3.269698e-02,&
     & 3.196477e-02,3.126082e-02,3.058352e-02,2.993141e-02,2.930310e-02,&
     & 2.869732e-02,2.811289e-02,2.754869e-02,2.700371e-02,2.647698e-02,&
     & 2.596760e-02,2.547473e-02,2.4998e-02 /
      data  extliq1(:, 25)   /                                          &
     & 6.666233e-01,4.662044e-01,3.579517e-01,2.902984e-01,2.440475e-01,&
     & 2.104431e-01,1.849277e-01,1.648970e-01,1.487555e-01,1.354714e-01,&
     & 1.244173e-01,1.149913e-01,1.068903e-01,9.985323e-02,9.368351e-02,&
     & 8.823009e-02,8.337507e-02,7.902511e-02,7.510529e-02,7.155482e-02,&
     & 6.832386e-02,6.537113e-02,6.266218e-02,6.016802e-02,5.786408e-02,&
     & 5.572939e-02,5.374598e-02,5.189830e-02,5.006825e-02,4.851081e-02,&
     & 4.703906e-02,4.564623e-02,4.432621e-02,4.307349e-02,4.188312e-02,&
     & 4.075060e-02,3.967183e-02,3.864313e-02,3.766111e-02,3.672269e-02,&
     & 3.582505e-02,3.496559e-02,3.414196e-02,3.335198e-02,3.259362e-02,&
     & 3.186505e-02,3.116454e-02,3.049052e-02,2.984152e-02,2.921617e-02,&
     & 2.861322e-02,2.803148e-02,2.746986e-02,2.692733e-02,2.640295e-02,&
     & 2.589582e-02,2.540510e-02,2.4930e-02 /
      data  extliq1(:, 26)   /                                          &
     & 6.535669e-01,4.585865e-01,3.529226e-01,2.867245e-01,2.413848e-01,&
     & 2.083956e-01,1.833191e-01,1.636150e-01,1.477247e-01,1.346392e-01,&
     & 1.236449e-01,1.143095e-01,1.062828e-01,9.930773e-02,9.319029e-02,&
     & 8.778150e-02,8.296497e-02,7.864847e-02,7.475799e-02,7.123343e-02,&
     & 6.802549e-02,6.509332e-02,6.240285e-02,5.992538e-02,5.763657e-02,&
     & 5.551566e-02,5.354483e-02,5.170870e-02,4.988866e-02,4.834061e-02,&
     & 4.687751e-02,4.549264e-02,4.417999e-02,4.293410e-02,4.175006e-02,&
     & 4.062344e-02,3.955019e-02,3.852663e-02,3.754943e-02,3.661553e-02,&
     & 3.572214e-02,3.486669e-02,3.404683e-02,3.326040e-02,3.250542e-02,&
     & 3.178003e-02,3.108254e-02,3.041139e-02,2.976511e-02,2.914235e-02,&
     & 2.854187e-02,2.796247e-02,2.740309e-02,2.686271e-02,2.634038e-02,&
     & 2.583520e-02,2.534636e-02,2.4873e-02 /
      data  extliq1(:, 27)   /                                          &
     & 6.448790e-01,4.541425e-01,3.503348e-01,2.850494e-01,2.401966e-01,&
     & 2.074811e-01,1.825631e-01,1.629515e-01,1.471142e-01,1.340574e-01,&
     & 1.231462e-01,1.138628e-01,1.058802e-01,9.894286e-02,9.285818e-02,&
     & 8.747802e-02,8.268676e-02,7.839271e-02,7.452230e-02,7.101580e-02,&
     & 6.782418e-02,6.490685e-02,6.222991e-02,5.976484e-02,5.748742e-02,&
     & 5.537703e-02,5.341593e-02,5.158883e-02,4.977355e-02,4.823172e-02,&
     & 4.677430e-02,4.539465e-02,4.408680e-02,4.284533e-02,4.166539e-02,&
     & 4.054257e-02,3.947283e-02,3.845256e-02,3.747842e-02,3.654737e-02,&
     & 3.565665e-02,3.480370e-02,3.398620e-02,3.320198e-02,3.244908e-02,&
     & 3.172566e-02,3.103002e-02,3.036062e-02,2.971600e-02,2.909482e-02,&
     & 2.849582e-02,2.791785e-02,2.735982e-02,2.682072e-02,2.629960e-02,&
     & 2.579559e-02,2.530786e-02,2.4836e-02 /
      data  extliq1(:, 28)   /                                          &
     & 6.422688e-01,4.528453e-01,3.497232e-01,2.847724e-01,2.400815e-01,&
     & 2.074403e-01,1.825502e-01,1.629415e-01,1.470934e-01,1.340183e-01,&
     & 1.230935e-01,1.138049e-01,1.058201e-01,9.888245e-02,9.279878e-02,&
     & 8.742053e-02,8.263175e-02,7.834058e-02,7.447327e-02,7.097000e-02,&
     & 6.778167e-02,6.486765e-02,6.219400e-02,5.973215e-02,5.745790e-02,&
     & 5.535059e-02,5.339250e-02,5.156831e-02,4.975308e-02,4.821235e-02,&
     & 4.675596e-02,4.537727e-02,4.407030e-02,4.282968e-02,4.165053e-02,&
     & 4.052845e-02,3.945941e-02,3.843980e-02,3.746628e-02,3.653583e-02,&
     & 3.564567e-02,3.479326e-02,3.397626e-02,3.319253e-02,3.244008e-02,&
     & 3.171711e-02,3.102189e-02,3.035289e-02,2.970866e-02,2.908784e-02,&
     & 2.848920e-02,2.791156e-02,2.735385e-02,2.681507e-02,2.629425e-02,&
     & 2.579053e-02,2.530308e-02,2.4831e-02 /
      data  extliq1(:, 29)   /                                          &
     & 4.614710e-01,4.556116e-01,4.056568e-01,3.529833e-01,3.060334e-01,&
     & 2.658127e-01,2.316095e-01,2.024325e-01,1.773749e-01,1.556867e-01,&
     & 1.455558e-01,1.332882e-01,1.229052e-01,1.140067e-01,1.062981e-01,&
     & 9.955703e-02,9.361333e-02,8.833420e-02,8.361467e-02,7.937071e-02,&
     & 7.553420e-02,7.204942e-02,6.887031e-02,6.595851e-02,6.328178e-02,&
     & 6.081286e-02,5.852854e-02,5.640892e-02,5.431269e-02,5.252561e-02,&
     & 5.084345e-02,4.925727e-02,4.775910e-02,4.634182e-02,4.499907e-02,&
     & 4.372512e-02,4.251484e-02,4.136357e-02,4.026710e-02,3.922162e-02,&
     & 3.822365e-02,3.727004e-02,3.635790e-02,3.548457e-02,3.464764e-02,&
     & 3.384488e-02,3.307424e-02,3.233384e-02,3.162192e-02,3.093688e-02,&
     & 3.027723e-02,2.964158e-02,2.902864e-02,2.843722e-02,2.786621e-02,&
     & 2.731457e-02,2.678133e-02,2.6266e-02 /

!  --- ...  single scattering albedo from hu and stamnes
      data  ssaliq1(:, 16)   /                                          &
     & 8.143821e-01,7.836739e-01,7.550722e-01,7.306269e-01,7.105612e-01,&
     & 6.946649e-01,6.825556e-01,6.737762e-01,6.678448e-01,6.642830e-01,&
     & 6.679741e-01,6.584607e-01,6.505598e-01,6.440951e-01,6.388901e-01,&
     & 6.347689e-01,6.315549e-01,6.290718e-01,6.271432e-01,6.255928e-01,&
     & 6.242441e-01,6.229207e-01,6.214464e-01,6.196445e-01,6.173388e-01,&
     & 6.143527e-01,6.105099e-01,6.056339e-01,6.108290e-01,6.073939e-01,&
     & 6.043073e-01,6.015473e-01,5.990913e-01,5.969173e-01,5.950028e-01,&
     & 5.933257e-01,5.918636e-01,5.905944e-01,5.894957e-01,5.885453e-01,&
     & 5.877209e-01,5.870003e-01,5.863611e-01,5.857811e-01,5.852381e-01,&
     & 5.847098e-01,5.841738e-01,5.836081e-01,5.829901e-01,5.822979e-01,&
     & 5.815089e-01,5.806011e-01,5.795521e-01,5.783396e-01,5.769413e-01,&
     & 5.753351e-01,5.734986e-01,5.7141e-01 /
      data  ssaliq1(:, 17)   /                                          &
     & 8.165821e-01,8.002015e-01,7.816921e-01,7.634131e-01,7.463721e-01,&
     & 7.312469e-01,7.185883e-01,7.088975e-01,7.026671e-01,7.004020e-01,&
     & 7.042138e-01,6.960930e-01,6.894243e-01,6.840459e-01,6.797957e-01,&
     & 6.765119e-01,6.740325e-01,6.721955e-01,6.708391e-01,6.698013e-01,&
     & 6.689201e-01,6.680339e-01,6.669805e-01,6.655982e-01,6.637250e-01,&
     & 6.611992e-01,6.578588e-01,6.535420e-01,6.584449e-01,6.553992e-01,&
     & 6.526547e-01,6.501917e-01,6.479905e-01,6.460313e-01,6.442945e-01,&
     & 6.427605e-01,6.414094e-01,6.402217e-01,6.391775e-01,6.382573e-01,&
     & 6.374413e-01,6.367099e-01,6.360433e-01,6.354218e-01,6.348257e-01,&
     & 6.342355e-01,6.336313e-01,6.329935e-01,6.323023e-01,6.315383e-01,&
     & 6.306814e-01,6.297122e-01,6.286110e-01,6.273579e-01,6.259333e-01,&
     & 6.243176e-01,6.224910e-01,6.2043e-01 /
      data  ssaliq1(:, 18)   /                                          &
     & 9.900163e-01,9.854307e-01,9.797730e-01,9.733113e-01,9.664245e-01,&
     & 9.594976e-01,9.529055e-01,9.470112e-01,9.421695e-01,9.387304e-01,&
     & 9.344918e-01,9.305302e-01,9.267048e-01,9.230072e-01,9.194289e-01,&
     & 9.159616e-01,9.125968e-01,9.093260e-01,9.061409e-01,9.030330e-01,&
     & 8.999940e-01,8.970154e-01,8.940888e-01,8.912058e-01,8.883579e-01,&
     & 8.855368e-01,8.827341e-01,8.799413e-01,8.777423e-01,8.749566e-01,&
     & 8.722298e-01,8.695605e-01,8.669469e-01,8.643875e-01,8.618806e-01,&
     & 8.594246e-01,8.570179e-01,8.546589e-01,8.523459e-01,8.500773e-01,&
     & 8.478516e-01,8.456670e-01,8.435219e-01,8.414148e-01,8.393439e-01,&
     & 8.373078e-01,8.353047e-01,8.333330e-01,8.313911e-01,8.294774e-01,&
     & 8.275904e-01,8.257282e-01,8.238893e-01,8.220721e-01,8.202751e-01,&
     & 8.184965e-01,8.167346e-01,8.1499e-01 /
      data  ssaliq1(:, 19)   /                                          &
     & 9.999916e-01,9.987396e-01,9.966900e-01,9.950738e-01,9.937531e-01,&
     & 9.925912e-01,9.914525e-01,9.902018e-01,9.887046e-01,9.868263e-01,&
     & 9.849039e-01,9.832372e-01,9.815265e-01,9.797770e-01,9.779940e-01,&
     & 9.761827e-01,9.743481e-01,9.724955e-01,9.706303e-01,9.687575e-01,&
     & 9.668823e-01,9.650100e-01,9.631457e-01,9.612947e-01,9.594622e-01,&
     & 9.576534e-01,9.558734e-01,9.541275e-01,9.522059e-01,9.504258e-01,&
     & 9.486459e-01,9.468676e-01,9.450921e-01,9.433208e-01,9.415548e-01,&
     & 9.397955e-01,9.380441e-01,9.363022e-01,9.345706e-01,9.328510e-01,&
     & 9.311445e-01,9.294524e-01,9.277761e-01,9.261167e-01,9.244755e-01,&
     & 9.228540e-01,9.212534e-01,9.196748e-01,9.181197e-01,9.165894e-01,&
     & 9.150851e-01,9.136080e-01,9.121596e-01,9.107410e-01,9.093536e-01,&
     & 9.079987e-01,9.066775e-01,9.0539e-01 /
      data  ssaliq1(:, 20)   /                                          &
     & 9.979493e-01,9.964113e-01,9.950014e-01,9.937045e-01,9.924964e-01,&
     & 9.913546e-01,9.902575e-01,9.891843e-01,9.881136e-01,9.870238e-01,&
     & 9.859934e-01,9.849372e-01,9.838873e-01,9.828434e-01,9.818052e-01,&
     & 9.807725e-01,9.797450e-01,9.787225e-01,9.777047e-01,9.766914e-01,&
     & 9.756823e-01,9.746771e-01,9.736756e-01,9.726775e-01,9.716827e-01,&
     & 9.706907e-01,9.697014e-01,9.687145e-01,9.678060e-01,9.668108e-01,&
     & 9.658218e-01,9.648391e-01,9.638629e-01,9.628936e-01,9.619313e-01,&
     & 9.609763e-01,9.600287e-01,9.590888e-01,9.581569e-01,9.572330e-01,&
     & 9.563176e-01,9.554108e-01,9.545128e-01,9.536239e-01,9.527443e-01,&
     & 9.518741e-01,9.510137e-01,9.501633e-01,9.493230e-01,9.484931e-01,&
     & 9.476740e-01,9.468656e-01,9.460683e-01,9.452824e-01,9.445080e-01,&
     & 9.437454e-01,9.429948e-01,9.4226e-01 /
      data  ssaliq1(:, 21)   /                                          &
     & 9.988742e-01,9.982668e-01,9.976935e-01,9.971497e-01,9.966314e-01,&
     & 9.961344e-01,9.956545e-01,9.951873e-01,9.947286e-01,9.942741e-01,&
     & 9.938457e-01,9.933947e-01,9.929473e-01,9.925032e-01,9.920621e-01,&
     & 9.916237e-01,9.911875e-01,9.907534e-01,9.903209e-01,9.898898e-01,&
     & 9.894597e-01,9.890304e-01,9.886015e-01,9.881726e-01,9.877435e-01,&
     & 9.873138e-01,9.868833e-01,9.864516e-01,9.860698e-01,9.856317e-01,&
     & 9.851957e-01,9.847618e-01,9.843302e-01,9.839008e-01,9.834739e-01,&
     & 9.830494e-01,9.826275e-01,9.822083e-01,9.817918e-01,9.813782e-01,&
     & 9.809675e-01,9.805598e-01,9.801552e-01,9.797538e-01,9.793556e-01,&
     & 9.789608e-01,9.785695e-01,9.781817e-01,9.777975e-01,9.774171e-01,&
     & 9.770404e-01,9.766676e-01,9.762988e-01,9.759340e-01,9.755733e-01,&
     & 9.752169e-01,9.748649e-01,9.7452e-01 /
      data  ssaliq1(:, 22)   /                                          &
     & 9.994441e-01,9.991608e-01,9.988949e-01,9.986439e-01,9.984054e-01,&
     & 9.981768e-01,9.979557e-01,9.977396e-01,9.975258e-01,9.973120e-01,&
     & 9.971011e-01,9.968852e-01,9.966708e-01,9.964578e-01,9.962462e-01,&
     & 9.960357e-01,9.958264e-01,9.956181e-01,9.954108e-01,9.952043e-01,&
     & 9.949987e-01,9.947937e-01,9.945892e-01,9.943853e-01,9.941818e-01,&
     & 9.939786e-01,9.937757e-01,9.935728e-01,9.933922e-01,9.931825e-01,&
     & 9.929739e-01,9.927661e-01,9.925592e-01,9.923534e-01,9.921485e-01,&
     & 9.919447e-01,9.917421e-01,9.915406e-01,9.913403e-01,9.911412e-01,&
     & 9.909435e-01,9.907470e-01,9.905519e-01,9.903581e-01,9.901659e-01,&
     & 9.899751e-01,9.897858e-01,9.895981e-01,9.894120e-01,9.892276e-01,&
     & 9.890447e-01,9.888637e-01,9.886845e-01,9.885070e-01,9.883314e-01,&
     & 9.881576e-01,9.879859e-01,9.8782e-01 /
      data  ssaliq1(:, 23)   /                                          &
     & 9.999138e-01,9.998730e-01,9.998338e-01,9.997965e-01,9.997609e-01,&
     & 9.997270e-01,9.996944e-01,9.996629e-01,9.996321e-01,9.996016e-01,&
     & 9.995690e-01,9.995372e-01,9.995057e-01,9.994744e-01,9.994433e-01,&
     & 9.994124e-01,9.993817e-01,9.993510e-01,9.993206e-01,9.992903e-01,&
     & 9.992600e-01,9.992299e-01,9.991998e-01,9.991698e-01,9.991398e-01,&
     & 9.991098e-01,9.990799e-01,9.990499e-01,9.990231e-01,9.989920e-01,&
     & 9.989611e-01,9.989302e-01,9.988996e-01,9.988690e-01,9.988386e-01,&
     & 9.988084e-01,9.987783e-01,9.987485e-01,9.987187e-01,9.986891e-01,&
     & 9.986598e-01,9.986306e-01,9.986017e-01,9.985729e-01,9.985443e-01,&
     & 9.985160e-01,9.984879e-01,9.984600e-01,9.984324e-01,9.984050e-01,&
     & 9.983778e-01,9.983509e-01,9.983243e-01,9.982980e-01,9.982719e-01,&
     & 9.982461e-01,9.982206e-01,9.9820e-01 /
      data  ssaliq1(:, 24)   /                                          &
     & 9.999985e-01,9.999979e-01,9.999972e-01,9.999966e-01,9.999961e-01,&
     & 9.999955e-01,9.999950e-01,9.999944e-01,9.999938e-01,9.999933e-01,&
     & 9.999927e-01,9.999921e-01,9.999915e-01,9.999910e-01,9.999904e-01,&
     & 9.999899e-01,9.999893e-01,9.999888e-01,9.999882e-01,9.999877e-01,&
     & 9.999871e-01,9.999866e-01,9.999861e-01,9.999855e-01,9.999850e-01,&
     & 9.999844e-01,9.999839e-01,9.999833e-01,9.999828e-01,9.999823e-01,&
     & 9.999817e-01,9.999812e-01,9.999807e-01,9.999801e-01,9.999796e-01,&
     & 9.999791e-01,9.999786e-01,9.999781e-01,9.999776e-01,9.999770e-01,&
     & 9.999765e-01,9.999761e-01,9.999756e-01,9.999751e-01,9.999746e-01,&
     & 9.999741e-01,9.999736e-01,9.999732e-01,9.999727e-01,9.999722e-01,&
     & 9.999718e-01,9.999713e-01,9.999709e-01,9.999705e-01,9.999701e-01,&
     & 9.999697e-01,9.999692e-01,9.9997e-01 /
      data  ssaliq1(:, 25)   /                                          &
     & 9.999999e-01,9.999998e-01,9.999997e-01,9.999997e-01,9.999997e-01,&
     & 9.999996e-01,9.999996e-01,9.999995e-01,9.999995e-01,9.999994e-01,&
     & 9.999994e-01,9.999993e-01,9.999993e-01,9.999992e-01,9.999992e-01,&
     & 9.999991e-01,9.999991e-01,9.999991e-01,9.999990e-01,9.999989e-01,&
     & 9.999989e-01,9.999989e-01,9.999988e-01,9.999988e-01,9.999987e-01,&
     & 9.999987e-01,9.999986e-01,9.999986e-01,9.999985e-01,9.999985e-01,&
     & 9.999984e-01,9.999984e-01,9.999984e-01,9.999983e-01,9.999983e-01,&
     & 9.999982e-01,9.999982e-01,9.999982e-01,9.999981e-01,9.999980e-01,&
     & 9.999980e-01,9.999980e-01,9.999979e-01,9.999979e-01,9.999978e-01,&
     & 9.999978e-01,9.999977e-01,9.999977e-01,9.999977e-01,9.999976e-01,&
     & 9.999976e-01,9.999975e-01,9.999975e-01,9.999974e-01,9.999974e-01,&
     & 9.999974e-01,9.999973e-01,1.0000e+00 /
      data  ssaliq1(:, 26)   /                                          &
     & 9.999997e-01,9.999995e-01,9.999993e-01,9.999992e-01,9.999990e-01,&
     & 9.999989e-01,9.999988e-01,9.999987e-01,9.999986e-01,9.999985e-01,&
     & 9.999984e-01,9.999983e-01,9.999982e-01,9.999981e-01,9.999980e-01,&
     & 9.999978e-01,9.999977e-01,9.999976e-01,9.999975e-01,9.999974e-01,&
     & 9.999973e-01,9.999972e-01,9.999970e-01,9.999969e-01,9.999968e-01,&
     & 9.999967e-01,9.999966e-01,9.999965e-01,9.999964e-01,9.999963e-01,&
     & 9.999962e-01,9.999961e-01,9.999959e-01,9.999958e-01,9.999957e-01,&
     & 9.999956e-01,9.999955e-01,9.999954e-01,9.999953e-01,9.999952e-01,&
     & 9.999951e-01,9.999949e-01,9.999949e-01,9.999947e-01,9.999946e-01,&
     & 9.999945e-01,9.999944e-01,9.999943e-01,9.999942e-01,9.999941e-01,&
     & 9.999940e-01,9.999939e-01,9.999938e-01,9.999937e-01,9.999936e-01,&
     & 9.999935e-01,9.999934e-01,9.9999e-01 /
      data  ssaliq1(:, 27)   /                                          &
     & 9.999984e-01,9.999976e-01,9.999969e-01,9.999962e-01,9.999956e-01,&
     & 9.999950e-01,9.999945e-01,9.999940e-01,9.999935e-01,9.999931e-01,&
     & 9.999926e-01,9.999920e-01,9.999914e-01,9.999908e-01,9.999903e-01,&
     & 9.999897e-01,9.999891e-01,9.999886e-01,9.999880e-01,9.999874e-01,&
     & 9.999868e-01,9.999863e-01,9.999857e-01,9.999851e-01,9.999846e-01,&
     & 9.999840e-01,9.999835e-01,9.999829e-01,9.999824e-01,9.999818e-01,&
     & 9.999812e-01,9.999806e-01,9.999800e-01,9.999795e-01,9.999789e-01,&
     & 9.999783e-01,9.999778e-01,9.999773e-01,9.999767e-01,9.999761e-01,&
     & 9.999756e-01,9.999750e-01,9.999745e-01,9.999739e-01,9.999734e-01,&
     & 9.999729e-01,9.999723e-01,9.999718e-01,9.999713e-01,9.999708e-01,&
     & 9.999703e-01,9.999697e-01,9.999692e-01,9.999687e-01,9.999683e-01,&
     & 9.999678e-01,9.999673e-01,9.9997e-01 /
      data  ssaliq1(:, 28)   /                                          &
     & 9.999981e-01,9.999973e-01,9.999965e-01,9.999958e-01,9.999951e-01,&
     & 9.999943e-01,9.999937e-01,9.999930e-01,9.999924e-01,9.999918e-01,&
     & 9.999912e-01,9.999905e-01,9.999897e-01,9.999890e-01,9.999883e-01,&
     & 9.999876e-01,9.999869e-01,9.999862e-01,9.999855e-01,9.999847e-01,&
     & 9.999840e-01,9.999834e-01,9.999827e-01,9.999819e-01,9.999812e-01,&
     & 9.999805e-01,9.999799e-01,9.999791e-01,9.999785e-01,9.999778e-01,&
     & 9.999771e-01,9.999764e-01,9.999757e-01,9.999750e-01,9.999743e-01,&
     & 9.999736e-01,9.999729e-01,9.999722e-01,9.999715e-01,9.999709e-01,&
     & 9.999701e-01,9.999695e-01,9.999688e-01,9.999682e-01,9.999675e-01,&
     & 9.999669e-01,9.999662e-01,9.999655e-01,9.999649e-01,9.999642e-01,&
     & 9.999636e-01,9.999630e-01,9.999624e-01,9.999618e-01,9.999612e-01,&
     & 9.999606e-01,9.999600e-01,9.9996e-01 /
      data  ssaliq1(:, 29)   /                                          &
     & 8.505737e-01,8.465102e-01,8.394829e-01,8.279508e-01,8.110806e-01,&
     & 7.900397e-01,7.669615e-01,7.444422e-01,7.253055e-01,7.124831e-01,&
     & 7.016434e-01,6.885485e-01,6.767340e-01,6.661029e-01,6.565577e-01,&
     & 6.480013e-01,6.403373e-01,6.334697e-01,6.273034e-01,6.217440e-01,&
     & 6.166983e-01,6.120740e-01,6.077796e-01,6.037249e-01,5.998207e-01,&
     & 5.959788e-01,5.921123e-01,5.881354e-01,5.891285e-01,5.851143e-01,&
     & 5.814653e-01,5.781606e-01,5.751792e-01,5.724998e-01,5.701016e-01,&
     & 5.679634e-01,5.660642e-01,5.643829e-01,5.628984e-01,5.615898e-01,&
     & 5.604359e-01,5.594158e-01,5.585083e-01,5.576924e-01,5.569470e-01,&
     & 5.562512e-01,5.555838e-01,5.549239e-01,5.542503e-01,5.535420e-01,&
     & 5.527781e-01,5.519374e-01,5.509989e-01,5.499417e-01,5.487445e-01,&
     & 5.473865e-01,5.458466e-01,5.4410e-01 /

!  --- ...  asymmetry parameter from hu and stamnes
      data  asyliq1(:, 16)   /                                          &
     & 8.133297e-01,8.133528e-01,8.173865e-01,8.243205e-01,8.333063e-01,&
     & 8.436317e-01,8.546611e-01,8.657934e-01,8.764345e-01,8.859837e-01,&
     & 8.627394e-01,8.824569e-01,8.976887e-01,9.089541e-01,9.167699e-01,&
     & 9.216517e-01,9.241147e-01,9.246743e-01,9.238469e-01,9.221504e-01,&
     & 9.201045e-01,9.182299e-01,9.170491e-01,9.170862e-01,9.188653e-01,&
     & 9.229111e-01,9.297468e-01,9.398950e-01,9.203269e-01,9.260693e-01,&
     & 9.309373e-01,9.349918e-01,9.382935e-01,9.409030e-01,9.428809e-01,&
     & 9.442881e-01,9.451851e-01,9.456331e-01,9.456926e-01,9.454247e-01,&
     & 9.448902e-01,9.441503e-01,9.432661e-01,9.422987e-01,9.413094e-01,&
     & 9.403594e-01,9.395102e-01,9.388230e-01,9.383594e-01,9.381810e-01,&
     & 9.383489e-01,9.389251e-01,9.399707e-01,9.415475e-01,9.437167e-01,&
     & 9.465399e-01,9.500786e-01,9.5439e-01 /
      data  asyliq1(:, 17)   /                                          &
     & 8.794448e-01,8.819306e-01,8.837667e-01,8.853832e-01,8.871010e-01,&
     & 8.892675e-01,8.922584e-01,8.964666e-01,9.022940e-01,9.101456e-01,&
     & 8.839999e-01,9.035610e-01,9.184568e-01,9.292315e-01,9.364282e-01,&
     & 9.405887e-01,9.422554e-01,9.419703e-01,9.402759e-01,9.377159e-01,&
     & 9.348345e-01,9.321769e-01,9.302888e-01,9.297166e-01,9.310075e-01,&
     & 9.347080e-01,9.413643e-01,9.515216e-01,9.306286e-01,9.361781e-01,&
     & 9.408374e-01,9.446692e-01,9.477363e-01,9.501013e-01,9.518268e-01,&
     & 9.529756e-01,9.536105e-01,9.537938e-01,9.535886e-01,9.530574e-01,&
     & 9.522633e-01,9.512688e-01,9.501370e-01,9.489306e-01,9.477126e-01,&
     & 9.465459e-01,9.454934e-01,9.446183e-01,9.439833e-01,9.436519e-01,&
     & 9.436866e-01,9.441508e-01,9.451073e-01,9.466195e-01,9.487501e-01,&
     & 9.515621e-01,9.551185e-01,9.5948e-01 /
      data  asyliq1(:, 18)   /                                          &
     & 8.478817e-01,8.269312e-01,8.161352e-01,8.135960e-01,8.173586e-01,&
     & 8.254167e-01,8.357072e-01,8.461167e-01,8.544952e-01,8.586776e-01,&
     & 8.335562e-01,8.524273e-01,8.669052e-01,8.775014e-01,8.847277e-01,&
     & 8.890958e-01,8.911173e-01,8.913038e-01,8.901669e-01,8.882182e-01,&
     & 8.859692e-01,8.839315e-01,8.826164e-01,8.825356e-01,8.842004e-01,&
     & 8.881223e-01,8.948131e-01,9.047837e-01,8.855951e-01,8.911796e-01,&
     & 8.959229e-01,8.998837e-01,9.031209e-01,9.056939e-01,9.076609e-01,&
     & 9.090812e-01,9.100134e-01,9.105167e-01,9.106496e-01,9.104712e-01,&
     & 9.100404e-01,9.094159e-01,9.086568e-01,9.078218e-01,9.069697e-01,&
     & 9.061595e-01,9.054499e-01,9.048999e-01,9.045683e-01,9.045142e-01,&
     & 9.047962e-01,9.054730e-01,9.066037e-01,9.082472e-01,9.104623e-01,&
     & 9.133079e-01,9.168427e-01,9.2113e-01 /
      data  asyliq1(:, 19)   /                                          &
     & 8.216697e-01,7.982871e-01,7.891147e-01,7.909083e-01,8.003833e-01,&
     & 8.142516e-01,8.292290e-01,8.420356e-01,8.493945e-01,8.480316e-01,&
     & 8.212381e-01,8.394984e-01,8.534095e-01,8.634813e-01,8.702242e-01,&
     & 8.741483e-01,8.757638e-01,8.755808e-01,8.741095e-01,8.718604e-01,&
     & 8.693433e-01,8.670686e-01,8.655464e-01,8.652872e-01,8.668006e-01,&
     & 8.705973e-01,8.771874e-01,8.870809e-01,8.678284e-01,8.732315e-01,&
     & 8.778084e-01,8.816166e-01,8.847146e-01,8.871603e-01,8.890116e-01,&
     & 8.903266e-01,8.911632e-01,8.915796e-01,8.916337e-01,8.913834e-01,&
     & 8.908869e-01,8.902022e-01,8.893873e-01,8.885001e-01,8.875986e-01,&
     & 8.867411e-01,8.859852e-01,8.853891e-01,8.850111e-01,8.849089e-01,&
     & 8.851405e-01,8.857639e-01,8.868372e-01,8.884185e-01,8.905656e-01,&
     & 8.933368e-01,8.967899e-01,9.0098e-01 /
      data  asyliq1(:, 20)   /                                          &
     & 8.063610e-01,7.938147e-01,7.921304e-01,7.985092e-01,8.101339e-01,&
     & 8.242175e-01,8.379913e-01,8.486920e-01,8.535547e-01,8.498083e-01,&
     & 8.224849e-01,8.405509e-01,8.542436e-01,8.640770e-01,8.705653e-01,&
     & 8.742227e-01,8.755630e-01,8.751004e-01,8.733491e-01,8.708231e-01,&
     & 8.680365e-01,8.655035e-01,8.637381e-01,8.632544e-01,8.645665e-01,&
     & 8.681885e-01,8.746346e-01,8.844188e-01,8.648180e-01,8.700563e-01,&
     & 8.744672e-01,8.781087e-01,8.810393e-01,8.833174e-01,8.850011e-01,&
     & 8.861485e-01,8.868183e-01,8.870687e-01,8.869579e-01,8.865441e-01,&
     & 8.858857e-01,8.850412e-01,8.840686e-01,8.830263e-01,8.819726e-01,&
     & 8.809658e-01,8.800642e-01,8.793260e-01,8.788099e-01,8.785737e-01,&
     & 8.786758e-01,8.791746e-01,8.801283e-01,8.815955e-01,8.836340e-01,&
     & 8.863024e-01,8.896592e-01,8.9376e-01 /
      data  asyliq1(:, 21)   /                                          &
     & 7.885899e-01,7.937172e-01,8.020658e-01,8.123971e-01,8.235502e-01,&
     & 8.343776e-01,8.437336e-01,8.504711e-01,8.534421e-01,8.514978e-01,&
     & 8.238888e-01,8.417463e-01,8.552057e-01,8.647853e-01,8.710038e-01,&
     & 8.743798e-01,8.754319e-01,8.746786e-01,8.726386e-01,8.698303e-01,&
     & 8.667724e-01,8.639836e-01,8.619823e-01,8.612870e-01,8.624165e-01,&
     & 8.658893e-01,8.722241e-01,8.819394e-01,8.620216e-01,8.671239e-01,&
     & 8.713983e-01,8.749032e-01,8.776970e-01,8.798385e-01,8.813860e-01,&
     & 8.823980e-01,8.829332e-01,8.830500e-01,8.828068e-01,8.822623e-01,&
     & 8.814750e-01,8.805031e-01,8.794056e-01,8.782407e-01,8.770672e-01,&
     & 8.759432e-01,8.749275e-01,8.740784e-01,8.734547e-01,8.731146e-01,&
     & 8.731170e-01,8.735199e-01,8.743823e-01,8.757625e-01,8.777191e-01,&
     & 8.803105e-01,8.835953e-01,8.8763e-01 /
      data  asyliq1(:, 22)   /                                          &
     & 7.811516e-01,7.962229e-01,8.096199e-01,8.212996e-01,8.312212e-01,&
     & 8.393430e-01,8.456236e-01,8.500214e-01,8.524950e-01,8.530031e-01,&
     & 8.251485e-01,8.429043e-01,8.562461e-01,8.656954e-01,8.717737e-01,&
     & 8.750020e-01,8.759022e-01,8.749953e-01,8.728027e-01,8.698461e-01,&
     & 8.666466e-01,8.637257e-01,8.616047e-01,8.608051e-01,8.618483e-01,&
     & 8.652557e-01,8.715487e-01,8.812485e-01,8.611645e-01,8.662052e-01,&
     & 8.704173e-01,8.738594e-01,8.765901e-01,8.786678e-01,8.801517e-01,&
     & 8.810999e-01,8.815713e-01,8.816246e-01,8.813185e-01,8.807114e-01,&
     & 8.798621e-01,8.788290e-01,8.776713e-01,8.764470e-01,8.752152e-01,&
     & 8.740343e-01,8.729631e-01,8.720602e-01,8.713842e-01,8.709936e-01,&
     & 8.709475e-01,8.713041e-01,8.721221e-01,8.734602e-01,8.753774e-01,&
     & 8.779319e-01,8.811825e-01,8.8519e-01 /
      data  asyliq1(:, 23)   /                                          &
     & 7.865744e-01,8.093340e-01,8.257596e-01,8.369940e-01,8.441574e-01,&
     & 8.483602e-01,8.507096e-01,8.523139e-01,8.542834e-01,8.577321e-01,&
     & 8.288960e-01,8.465308e-01,8.597175e-01,8.689830e-01,8.748542e-01,&
     & 8.778584e-01,8.785222e-01,8.773728e-01,8.749370e-01,8.717419e-01,&
     & 8.683145e-01,8.651816e-01,8.628704e-01,8.619077e-01,8.628205e-01,&
     & 8.661356e-01,8.723803e-01,8.820815e-01,8.616715e-01,8.666389e-01,&
     & 8.707753e-01,8.741398e-01,8.767912e-01,8.787885e-01,8.801908e-01,&
     & 8.810570e-01,8.814460e-01,8.814167e-01,8.810283e-01,8.803395e-01,&
     & 8.794095e-01,8.782971e-01,8.770613e-01,8.757610e-01,8.744553e-01,&
     & 8.732031e-01,8.720634e-01,8.710951e-01,8.703572e-01,8.699086e-01,&
     & 8.698084e-01,8.701155e-01,8.708887e-01,8.721872e-01,8.740698e-01,&
     & 8.765957e-01,8.798235e-01,8.8381e-01 /
      data  asyliq1(:, 24)   /                                          &
     & 8.069513e-01,8.262939e-01,8.398241e-01,8.486352e-01,8.538213e-01,&
     & 8.564743e-01,8.576854e-01,8.585455e-01,8.601452e-01,8.635755e-01,&
     & 8.337383e-01,8.512655e-01,8.643049e-01,8.733896e-01,8.790535e-01,&
     & 8.818295e-01,8.822518e-01,8.808533e-01,8.781676e-01,8.747284e-01,&
     & 8.710690e-01,8.677229e-01,8.652236e-01,8.641047e-01,8.648993e-01,&
     & 8.681413e-01,8.743640e-01,8.841007e-01,8.633558e-01,8.682719e-01,&
     & 8.723543e-01,8.756621e-01,8.782547e-01,8.801915e-01,8.815318e-01,&
     & 8.823347e-01,8.826598e-01,8.825663e-01,8.821135e-01,8.813608e-01,&
     & 8.803674e-01,8.791928e-01,8.778960e-01,8.765366e-01,8.751738e-01,&
     & 8.738670e-01,8.726755e-01,8.716585e-01,8.708755e-01,8.703856e-01,&
     & 8.702483e-01,8.705229e-01,8.712687e-01,8.725448e-01,8.744109e-01,&
     & 8.769260e-01,8.801496e-01,8.8414e-01 /
      data  asyliq1(:, 25)   /                                          &
     & 8.252182e-01,8.379244e-01,8.471709e-01,8.535760e-01,8.577540e-01,&
     & 8.603183e-01,8.618820e-01,8.630578e-01,8.644587e-01,8.666970e-01,&
     & 8.362159e-01,8.536817e-01,8.666387e-01,8.756240e-01,8.811746e-01,&
     & 8.838273e-01,8.841191e-01,8.825871e-01,8.797681e-01,8.761992e-01,&
     & 8.724174e-01,8.689593e-01,8.663623e-01,8.651632e-01,8.658988e-01,&
     & 8.691064e-01,8.753226e-01,8.850847e-01,8.641620e-01,8.690500e-01,&
     & 8.731026e-01,8.763795e-01,8.789400e-01,8.808438e-01,8.821503e-01,&
     & 8.829191e-01,8.832095e-01,8.830813e-01,8.825938e-01,8.818064e-01,&
     & 8.807787e-01,8.795704e-01,8.782408e-01,8.768493e-01,8.754557e-01,&
     & 8.741193e-01,8.728995e-01,8.718561e-01,8.710484e-01,8.705360e-01,&
     & 8.703782e-01,8.706347e-01,8.713650e-01,8.726285e-01,8.744849e-01,&
     & 8.769933e-01,8.802136e-01,8.8421e-01 /
      data  asyliq1(:, 26)   /                                          &
     & 8.370583e-01,8.467920e-01,8.537769e-01,8.585136e-01,8.615034e-01,&
     & 8.632474e-01,8.642468e-01,8.650026e-01,8.660161e-01,8.677882e-01,&
     & 8.369760e-01,8.543821e-01,8.672699e-01,8.761782e-01,8.816454e-01,&
     & 8.842103e-01,8.844114e-01,8.827872e-01,8.798766e-01,8.762179e-01,&
     & 8.723500e-01,8.688112e-01,8.661403e-01,8.648758e-01,8.655563e-01,&
     & 8.687206e-01,8.749072e-01,8.846546e-01,8.636289e-01,8.684849e-01,&
     & 8.725054e-01,8.757501e-01,8.782785e-01,8.801503e-01,8.814249e-01,&
     & 8.821620e-01,8.824211e-01,8.822620e-01,8.817440e-01,8.809268e-01,&
     & 8.798699e-01,8.786330e-01,8.772756e-01,8.758572e-01,8.744374e-01,&
     & 8.730760e-01,8.718323e-01,8.707660e-01,8.699366e-01,8.694039e-01,&
     & 8.692271e-01,8.694661e-01,8.701803e-01,8.714293e-01,8.732727e-01,&
     & 8.757702e-01,8.789811e-01,8.8297e-01 /
      data  asyliq1(:, 27)   /                                          &
     & 8.430819e-01,8.510060e-01,8.567270e-01,8.606533e-01,8.631934e-01,&
     & 8.647554e-01,8.657471e-01,8.665760e-01,8.676496e-01,8.693754e-01,&
     & 8.384298e-01,8.557913e-01,8.686214e-01,8.774605e-01,8.828495e-01,&
     & 8.853287e-01,8.854393e-01,8.837215e-01,8.807161e-01,8.769639e-01,&
     & 8.730053e-01,8.693812e-01,8.666321e-01,8.652988e-01,8.659219e-01,&
     & 8.690419e-01,8.751999e-01,8.849360e-01,8.638013e-01,8.686371e-01,&
     & 8.726369e-01,8.758605e-01,8.783674e-01,8.802176e-01,8.814705e-01,&
     & 8.821859e-01,8.824234e-01,8.822429e-01,8.817038e-01,8.808658e-01,&
     & 8.797887e-01,8.785323e-01,8.771560e-01,8.757196e-01,8.742828e-01,&
     & 8.729052e-01,8.716467e-01,8.705666e-01,8.697250e-01,8.691812e-01,&
     & 8.689950e-01,8.692264e-01,8.699346e-01,8.711795e-01,8.730209e-01,&
     & 8.755181e-01,8.787312e-01,8.8272e-01 /
      data  asyliq1(:, 28)   /                                          &
     & 8.452284e-01,8.522700e-01,8.572973e-01,8.607031e-01,8.628802e-01,&
     & 8.642215e-01,8.651198e-01,8.659679e-01,8.671588e-01,8.690853e-01,&
     & 8.383803e-01,8.557485e-01,8.685851e-01,8.774303e-01,8.828245e-01,&
     & 8.853077e-01,8.854207e-01,8.837034e-01,8.806962e-01,8.769398e-01,&
     & 8.729740e-01,8.693393e-01,8.665761e-01,8.652247e-01,8.658253e-01,&
     & 8.689182e-01,8.750438e-01,8.847424e-01,8.636140e-01,8.684449e-01,&
     & 8.724400e-01,8.756589e-01,8.781613e-01,8.800072e-01,8.812559e-01,&
     & 8.819671e-01,8.822007e-01,8.820165e-01,8.814737e-01,8.806322e-01,&
     & 8.795518e-01,8.782923e-01,8.769129e-01,8.754737e-01,8.740342e-01,&
     & 8.726542e-01,8.713934e-01,8.703111e-01,8.694677e-01,8.689222e-01,&
     & 8.687344e-01,8.689646e-01,8.696715e-01,8.709156e-01,8.727563e-01,&
     & 8.752531e-01,8.784659e-01,8.8245e-01 /
      data  asyliq1(:, 29)   /                                          &
     & 7.800869e-01,8.091120e-01,8.325369e-01,8.466266e-01,8.515495e-01,&
     & 8.499371e-01,8.456203e-01,8.430521e-01,8.470286e-01,8.625431e-01,&
     & 8.402261e-01,8.610822e-01,8.776608e-01,8.904485e-01,8.999294e-01,&
     & 9.065860e-01,9.108995e-01,9.133503e-01,9.144187e-01,9.145855e-01,&
     & 9.143320e-01,9.141402e-01,9.144933e-01,9.158754e-01,9.187716e-01,&
     & 9.236677e-01,9.310503e-01,9.414058e-01,9.239108e-01,9.300719e-01,&
     & 9.353612e-01,9.398378e-01,9.435609e-01,9.465895e-01,9.489829e-01,&
     & 9.508000e-01,9.521002e-01,9.529424e-01,9.533860e-01,9.534902e-01,&
     & 9.533143e-01,9.529177e-01,9.523596e-01,9.516997e-01,9.509973e-01,&
     & 9.503121e-01,9.497037e-01,9.492317e-01,9.489558e-01,9.489356e-01,&
     & 9.492311e-01,9.499019e-01,9.510077e-01,9.526084e-01,9.547636e-01,&
     & 9.575331e-01,9.609766e-01,9.6515e-01 /

!  --- ...  spherical ice particle parameterization from streamer v3
!           extinction units (ext coef/iwc): [(m^-1)/(g m^-3)]
      data  extice2(:, 16)   /                                          &
     & 4.101824e-01,2.435514e-01,1.713697e-01,1.314865e-01,1.063406e-01,&
     & 8.910701e-02,7.659480e-02,6.711784e-02,5.970353e-02,5.375249e-02,&
     & 4.887577e-02,4.481025e-02,4.137171e-02,3.842744e-02,3.587948e-02,&
     & 3.365396e-02,3.169419e-02,2.995593e-02,2.840419e-02,2.701091e-02,&
     & 2.575336e-02,2.461293e-02,2.357423e-02,2.262443e-02,2.175276e-02,&
     & 2.095012e-02,2.020875e-02,1.952199e-02,1.888412e-02,1.829018e-02,&
     & 1.773586e-02,1.721738e-02,1.673144e-02,1.627510e-02,1.584579e-02,&
     & 1.544122e-02,1.505934e-02,1.469833e-02,1.435654e-02,1.403251e-02,&
     & 1.372492e-02,1.343255e-02,1.315433e-02 /
      data  extice2(:, 17)   /                                          &
     & 3.836650e-01,2.304055e-01,1.637265e-01,1.266681e-01,1.031602e-01,&
     & 8.695191e-02,7.511544e-02,6.610009e-02,5.900909e-02,5.328833e-02,&
     & 4.857728e-02,4.463133e-02,4.127880e-02,3.839567e-02,3.589013e-02,&
     & 3.369280e-02,3.175027e-02,3.002079e-02,2.847121e-02,2.707493e-02,&
     & 2.581031e-02,2.465962e-02,2.360815e-02,2.264363e-02,2.175571e-02,&
     & 2.093563e-02,2.017592e-02,1.947015e-02,1.881278e-02,1.819901e-02,&
     & 1.762463e-02,1.708598e-02,1.657982e-02,1.610330e-02,1.565390e-02,&
     & 1.522937e-02,1.482768e-02,1.444706e-02,1.408588e-02,1.374270e-02,&
     & 1.341619e-02,1.310517e-02,1.280857e-02 /
      data  extice2(:, 18)   /                                          &
     & 4.152673e-01,2.436816e-01,1.702243e-01,1.299704e-01,1.047528e-01,&
     & 8.756039e-02,7.513327e-02,6.575690e-02,5.844616e-02,5.259609e-02,&
     & 4.781531e-02,4.383980e-02,4.048517e-02,3.761891e-02,3.514342e-02,&
     & 3.298525e-02,3.108814e-02,2.940825e-02,2.791096e-02,2.656858e-02,&
     & 2.535869e-02,2.426297e-02,2.326627e-02,2.235602e-02,2.152164e-02,&
     & 2.075420e-02,2.004613e-02,1.939091e-02,1.878296e-02,1.821744e-02,&
     & 1.769015e-02,1.719741e-02,1.673600e-02,1.630308e-02,1.589615e-02,&
     & 1.551298e-02,1.515159e-02,1.481021e-02,1.448726e-02,1.418131e-02,&
     & 1.389109e-02,1.361544e-02,1.335330e-02 /
      data  extice2(:, 19)   /                                          &
     & 3.873250e-01,2.331609e-01,1.655002e-01,1.277753e-01,1.038247e-01,&
     & 8.731780e-02,7.527638e-02,6.611873e-02,5.892850e-02,5.313885e-02,&
     & 4.838068e-02,4.440356e-02,4.103167e-02,3.813804e-02,3.562870e-02,&
     & 3.343269e-02,3.149539e-02,2.977414e-02,2.823510e-02,2.685112e-02,&
     & 2.560015e-02,2.446411e-02,2.342805e-02,2.247948e-02,2.160789e-02,&
     & 2.080438e-02,2.006139e-02,1.937238e-02,1.873177e-02,1.813469e-02,&
     & 1.757689e-02,1.705468e-02,1.656479e-02,1.610435e-02,1.567081e-02,&
     & 1.526192e-02,1.487565e-02,1.451020e-02,1.416396e-02,1.383546e-02,&
     & 1.352339e-02,1.322657e-02,1.294392e-02 /
      data  extice2(:, 20)   /                                          &
     & 3.784280e-01,2.291396e-01,1.632551e-01,1.263775e-01,1.028944e-01,&
     & 8.666975e-02,7.480952e-02,6.577335e-02,5.866714e-02,5.293694e-02,&
     & 4.822153e-02,4.427547e-02,4.092626e-02,3.804918e-02,3.555184e-02,&
     & 3.336440e-02,3.143307e-02,2.971577e-02,2.817912e-02,2.679632e-02,&
     & 2.554558e-02,2.440903e-02,2.337187e-02,2.242173e-02,2.154821e-02,&
     & 2.074249e-02,1.999706e-02,1.930546e-02,1.866212e-02,1.806221e-02,&
     & 1.750152e-02,1.697637e-02,1.648352e-02,1.602010e-02,1.558358e-02,&
     & 1.517172e-02,1.478250e-02,1.441413e-02,1.406498e-02,1.373362e-02,&
     & 1.341872e-02,1.311911e-02,1.283371e-02 /
      data  extice2(:, 21)   /                                          &
     & 3.719909e-01,2.259490e-01,1.613144e-01,1.250648e-01,1.019462e-01,&
     & 8.595358e-02,7.425064e-02,6.532618e-02,5.830218e-02,5.263421e-02,&
     & 4.796697e-02,4.405891e-02,4.074013e-02,3.788776e-02,3.541071e-02,&
     & 3.324008e-02,3.132280e-02,2.961733e-02,2.809071e-02,2.671645e-02,&
     & 2.547302e-02,2.434276e-02,2.331102e-02,2.236558e-02,2.149614e-02,&
     & 2.069397e-02,1.995163e-02,1.926272e-02,1.862174e-02,1.802389e-02,&
     & 1.746500e-02,1.694142e-02,1.644994e-02,1.598772e-02,1.555225e-02,&
     & 1.514129e-02,1.475286e-02,1.438515e-02,1.403659e-02,1.370572e-02,&
     & 1.339124e-02,1.309197e-02,1.280685e-02 /
      data  extice2(:, 22)   /                                          &
     & 3.713158e-01,2.253816e-01,1.608461e-01,1.246718e-01,1.016109e-01,&
     & 8.566332e-02,7.399666e-02,6.510199e-02,5.810290e-02,5.245608e-02,&
     & 4.780702e-02,4.391478e-02,4.060989e-02,3.776982e-02,3.530374e-02,&
     & 3.314296e-02,3.123458e-02,2.953719e-02,2.801794e-02,2.665043e-02,&
     & 2.541321e-02,2.428868e-02,2.326224e-02,2.232173e-02,2.145688e-02,&
     & 2.065899e-02,1.992067e-02,1.923552e-02,1.859808e-02,1.800356e-02,&
     & 1.744782e-02,1.692721e-02,1.643855e-02,1.597900e-02,1.554606e-02,&
     & 1.513751e-02,1.475137e-02,1.438586e-02,1.403938e-02,1.371050e-02,&
     & 1.339793e-02,1.310050e-02,1.281713e-02 /
      data  extice2(:, 23)   /                                          &
     & 3.605883e-01,2.204388e-01,1.580431e-01,1.229033e-01,1.004203e-01,&
     & 8.482616e-02,7.338941e-02,6.465105e-02,5.776176e-02,5.219398e-02,&
     & 4.760288e-02,4.375369e-02,4.048111e-02,3.766539e-02,3.521771e-02,&
     & 3.307079e-02,3.117277e-02,2.948303e-02,2.796929e-02,2.660560e-02,&
     & 2.537086e-02,2.424772e-02,2.322182e-02,2.228114e-02,2.141556e-02,&
     & 2.061649e-02,1.987661e-02,1.918962e-02,1.855009e-02,1.795330e-02,&
     & 1.739514e-02,1.687199e-02,1.638069e-02,1.591845e-02,1.548276e-02,&
     & 1.507143e-02,1.468249e-02,1.431416e-02,1.396486e-02,1.363318e-02,&
     & 1.331781e-02,1.301759e-02,1.273147e-02 /
      data  extice2(:, 24)   /                                          &
     & 3.527890e-01,2.168469e-01,1.560090e-01,1.216216e-01,9.955787e-02,&
     & 8.421942e-02,7.294827e-02,6.432192e-02,5.751081e-02,5.199888e-02,&
     & 4.744835e-02,4.362899e-02,4.037847e-02,3.757910e-02,3.514351e-02,&
     & 3.300546e-02,3.111382e-02,2.942853e-02,2.791775e-02,2.655584e-02,&
     & 2.532195e-02,2.419892e-02,2.317255e-02,2.223092e-02,2.136402e-02,&
     & 2.056334e-02,1.982160e-02,1.913258e-02,1.849087e-02,1.789178e-02,&
     & 1.733124e-02,1.680565e-02,1.631187e-02,1.584711e-02,1.540889e-02,&
     & 1.499502e-02,1.460354e-02,1.423269e-02,1.388088e-02,1.354670e-02,&
     & 1.322887e-02,1.292620e-02,1.263767e-02 /
      data  extice2(:, 25)   /                                          &
     & 3.477874e-01,2.143515e-01,1.544887e-01,1.205942e-01,9.881779e-02,&
     & 8.366261e-02,7.251586e-02,6.397790e-02,5.723183e-02,5.176908e-02,&
     & 4.725658e-02,4.346715e-02,4.024055e-02,3.746055e-02,3.504080e-02,&
     & 3.291583e-02,3.103507e-02,2.935891e-02,2.785582e-02,2.650042e-02,&
     & 2.527206e-02,2.415376e-02,2.313142e-02,2.219326e-02,2.132934e-02,&
     & 2.053122e-02,1.979169e-02,1.910456e-02,1.846448e-02,1.786680e-02,&
     & 1.730745e-02,1.678289e-02,1.628998e-02,1.582595e-02,1.538835e-02,&
     & 1.497499e-02,1.458393e-02,1.421341e-02,1.386187e-02,1.352788e-02,&
     & 1.321019e-02,1.290762e-02,1.261913e-02 /
      data  extice2(:, 26)   /                                          &
     & 3.453721e-01,2.130744e-01,1.536698e-01,1.200140e-01,9.838078e-02,&
     & 8.331940e-02,7.223803e-02,6.374775e-02,5.703770e-02,5.160290e-02,&
     & 4.711259e-02,4.334110e-02,4.012923e-02,3.736150e-02,3.495208e-02,&
     & 3.283589e-02,3.096267e-02,2.929302e-02,2.779560e-02,2.644517e-02,&
     & 2.522119e-02,2.410677e-02,2.308788e-02,2.215281e-02,2.129165e-02,&
     & 2.049602e-02,1.975874e-02,1.907365e-02,1.843542e-02,1.783943e-02,&
     & 1.728162e-02,1.675847e-02,1.626685e-02,1.580401e-02,1.536750e-02,&
     & 1.495515e-02,1.456502e-02,1.419537e-02,1.384463e-02,1.351139e-02,&
     & 1.319438e-02,1.289246e-02,1.260456e-02 /
      data  extice2(:, 27)   /                                          &
     & 3.417883e-01,2.113379e-01,1.526395e-01,1.193347e-01,9.790253e-02,&
     & 8.296715e-02,7.196979e-02,6.353806e-02,5.687024e-02,5.146670e-02,&
     & 4.700001e-02,4.324667e-02,4.004894e-02,3.729233e-02,3.489172e-02,&
     & 3.278257e-02,3.091499e-02,2.924987e-02,2.775609e-02,2.640859e-02,&
     & 2.518695e-02,2.407439e-02,2.305697e-02,2.212303e-02,2.126273e-02,&
     & 2.046774e-02,1.973090e-02,1.904610e-02,1.840801e-02,1.781204e-02,&
     & 1.725417e-02,1.673086e-02,1.623902e-02,1.577590e-02,1.533906e-02,&
     & 1.492634e-02,1.453580e-02,1.416571e-02,1.381450e-02,1.348078e-02,&
     & 1.316327e-02,1.286082e-02,1.257240e-02 /
      data  extice2(:, 28)   /                                          &
     & 3.416111e-01,2.114124e-01,1.527734e-01,1.194809e-01,9.804612e-02,&
     & 8.310287e-02,7.209595e-02,6.365442e-02,5.697710e-02,5.156460e-02,&
     & 4.708957e-02,4.332850e-02,4.012361e-02,3.736037e-02,3.495364e-02,&
     & 3.283879e-02,3.096593e-02,2.929589e-02,2.779751e-02,2.644571e-02,&
     & 2.522004e-02,2.410369e-02,2.308271e-02,2.214542e-02,2.128195e-02,&
     & 2.048396e-02,1.974429e-02,1.905679e-02,1.841614e-02,1.781774e-02,&
     & 1.725754e-02,1.673203e-02,1.623807e-02,1.577293e-02,1.533416e-02,&
     & 1.491958e-02,1.452727e-02,1.415547e-02,1.380262e-02,1.346732e-02,&
     & 1.314830e-02,1.284439e-02,1.255456e-02 /
      data  extice2(:, 29)   /                                          &
     & 4.196611e-01,2.493642e-01,1.761261e-01,1.357197e-01,1.102161e-01,&
     & 9.269376e-02,7.992985e-02,7.022538e-02,6.260168e-02,5.645603e-02,&
     & 5.139732e-02,4.716088e-02,4.356133e-02,4.046498e-02,3.777303e-02,&
     & 3.541094e-02,3.332137e-02,3.145954e-02,2.978998e-02,2.828419e-02,&
     & 2.691905e-02,2.567559e-02,2.453811e-02,2.349350e-02,2.253072e-02,&
     & 2.164042e-02,2.081464e-02,2.004652e-02,1.933015e-02,1.866041e-02,&
     & 1.803283e-02,1.744348e-02,1.688894e-02,1.636616e-02,1.587244e-02,&
     & 1.540539e-02,1.496287e-02,1.454295e-02,1.414392e-02,1.376423e-02,&
     & 1.340247e-02,1.305739e-02,1.272784e-02 /

!  --- ...  single-scattering albedo from streamer v3, unitless
      data  ssaice2(:, 16)   /                                          &
     & 6.630615e-01,6.451169e-01,6.333696e-01,6.246927e-01,6.178420e-01,&
     & 6.121976e-01,6.074069e-01,6.032505e-01,5.995830e-01,5.963030e-01,&
     & 5.933372e-01,5.906311e-01,5.881427e-01,5.858395e-01,5.836955e-01,&
     & 5.816896e-01,5.798046e-01,5.780264e-01,5.763429e-01,5.747441e-01,&
     & 5.732213e-01,5.717672e-01,5.703754e-01,5.690403e-01,5.677571e-01,&
     & 5.665215e-01,5.653297e-01,5.641782e-01,5.630643e-01,5.619850e-01,&
     & 5.609381e-01,5.599214e-01,5.589328e-01,5.579707e-01,5.570333e-01,&
     & 5.561193e-01,5.552272e-01,5.543558e-01,5.535041e-01,5.526708e-01,&
     & 5.518551e-01,5.510561e-01,5.502729e-01 /
      data  ssaice2(:, 17)   /                                          &
     & 7.689749e-01,7.398171e-01,7.205819e-01,7.065690e-01,6.956928e-01,&
     & 6.868989e-01,6.795813e-01,6.733606e-01,6.679838e-01,6.632742e-01,&
     & 6.591036e-01,6.553766e-01,6.520197e-01,6.489757e-01,6.461991e-01,&
     & 6.436531e-01,6.413075e-01,6.391375e-01,6.371221e-01,6.352438e-01,&
     & 6.334876e-01,6.318406e-01,6.302918e-01,6.288315e-01,6.274512e-01,&
     & 6.261436e-01,6.249022e-01,6.237211e-01,6.225953e-01,6.215201e-01,&
     & 6.204914e-01,6.195055e-01,6.185592e-01,6.176492e-01,6.167730e-01,&
     & 6.159280e-01,6.151120e-01,6.143228e-01,6.135587e-01,6.128177e-01,&
     & 6.120984e-01,6.113993e-01,6.107189e-01 /
      data  ssaice2(:, 18)   /                                          &
     & 9.956167e-01,9.814770e-01,9.716104e-01,9.639746e-01,9.577179e-01,&
     & 9.524010e-01,9.477672e-01,9.436527e-01,9.399467e-01,9.365708e-01,&
     & 9.334672e-01,9.305921e-01,9.279118e-01,9.253993e-01,9.230330e-01,&
     & 9.207954e-01,9.186719e-01,9.166501e-01,9.147199e-01,9.128722e-01,&
     & 9.110997e-01,9.093956e-01,9.077544e-01,9.061708e-01,9.046406e-01,&
     & 9.031598e-01,9.017248e-01,9.003326e-01,8.989804e-01,8.976655e-01,&
     & 8.963857e-01,8.951389e-01,8.939233e-01,8.927370e-01,8.915785e-01,&
     & 8.904464e-01,8.893392e-01,8.882559e-01,8.871951e-01,8.861559e-01,&
     & 8.851373e-01,8.841383e-01,8.831581e-01 /
      data  ssaice2(:, 19)   /                                          &
     & 9.723177e-01,9.452119e-01,9.267592e-01,9.127393e-01,9.014238e-01,&
     & 8.919334e-01,8.837584e-01,8.765773e-01,8.701736e-01,8.643950e-01,&
     & 8.591299e-01,8.542942e-01,8.498230e-01,8.456651e-01,8.417794e-01,&
     & 8.381324e-01,8.346964e-01,8.314484e-01,8.283687e-01,8.254408e-01,&
     & 8.226505e-01,8.199854e-01,8.174348e-01,8.149891e-01,8.126403e-01,&
     & 8.103808e-01,8.082041e-01,8.061044e-01,8.040765e-01,8.021156e-01,&
     & 8.002174e-01,7.983781e-01,7.965941e-01,7.948622e-01,7.931795e-01,&
     & 7.915432e-01,7.899508e-01,7.884002e-01,7.868891e-01,7.854156e-01,&
     & 7.839779e-01,7.825742e-01,7.812031e-01 /
      data  ssaice2(:, 20)   /                                          &
     & 9.933294e-01,9.860917e-01,9.811564e-01,9.774008e-01,9.743652e-01,&
     & 9.718155e-01,9.696159e-01,9.676810e-01,9.659531e-01,9.643915e-01,&
     & 9.629667e-01,9.616561e-01,9.604426e-01,9.593125e-01,9.582548e-01,&
     & 9.572607e-01,9.563227e-01,9.554347e-01,9.545915e-01,9.537888e-01,&
     & 9.530226e-01,9.522898e-01,9.515874e-01,9.509130e-01,9.502643e-01,&
     & 9.496394e-01,9.490366e-01,9.484542e-01,9.478910e-01,9.473456e-01,&
     & 9.468169e-01,9.463039e-01,9.458056e-01,9.453212e-01,9.448499e-01,&
     & 9.443910e-01,9.439438e-01,9.435077e-01,9.430821e-01,9.426666e-01,&
     & 9.422607e-01,9.418638e-01,9.414756e-01 /
      data  ssaice2(:, 21)   /                                          &
     & 9.900787e-01,9.828880e-01,9.779258e-01,9.741173e-01,9.710184e-01,&
     & 9.684012e-01,9.661332e-01,9.641301e-01,9.623352e-01,9.607083e-01,&
     & 9.592198e-01,9.578474e-01,9.565739e-01,9.553856e-01,9.542715e-01,&
     & 9.532226e-01,9.522314e-01,9.512919e-01,9.503986e-01,9.495472e-01,&
     & 9.487337e-01,9.479549e-01,9.472077e-01,9.464897e-01,9.457985e-01,&
     & 9.451322e-01,9.444890e-01,9.438673e-01,9.432656e-01,9.426826e-01,&
     & 9.421173e-01,9.415684e-01,9.410351e-01,9.405164e-01,9.400115e-01,&
     & 9.395198e-01,9.390404e-01,9.385728e-01,9.381164e-01,9.376707e-01,&
     & 9.372350e-01,9.368091e-01,9.363923e-01 /
      data  ssaice2(:, 22)   /                                          &
     & 9.986793e-01,9.985239e-01,9.983911e-01,9.982715e-01,9.981606e-01,&
     & 9.980562e-01,9.979567e-01,9.978613e-01,9.977691e-01,9.976798e-01,&
     & 9.975929e-01,9.975081e-01,9.974251e-01,9.973438e-01,9.972640e-01,&
     & 9.971855e-01,9.971083e-01,9.970322e-01,9.969571e-01,9.968830e-01,&
     & 9.968099e-01,9.967375e-01,9.966660e-01,9.965951e-01,9.965250e-01,&
     & 9.964555e-01,9.963867e-01,9.963185e-01,9.962508e-01,9.961836e-01,&
     & 9.961170e-01,9.960508e-01,9.959851e-01,9.959198e-01,9.958550e-01,&
     & 9.957906e-01,9.957266e-01,9.956629e-01,9.955997e-01,9.955367e-01,&
     & 9.954742e-01,9.954119e-01,9.953500e-01 /
      data  ssaice2(:, 23)   /                                          &
     & 9.997944e-01,9.997791e-01,9.997664e-01,9.997547e-01,9.997436e-01,&
     & 9.997327e-01,9.997219e-01,9.997110e-01,9.996999e-01,9.996886e-01,&
     & 9.996771e-01,9.996653e-01,9.996533e-01,9.996409e-01,9.996282e-01,&
     & 9.996152e-01,9.996019e-01,9.995883e-01,9.995743e-01,9.995599e-01,&
     & 9.995453e-01,9.995302e-01,9.995149e-01,9.994992e-01,9.994831e-01,&
     & 9.994667e-01,9.994500e-01,9.994329e-01,9.994154e-01,9.993976e-01,&
     & 9.993795e-01,9.993610e-01,9.993422e-01,9.993230e-01,9.993035e-01,&
     & 9.992837e-01,9.992635e-01,9.992429e-01,9.992221e-01,9.992008e-01,&
     & 9.991793e-01,9.991574e-01,9.991352e-01 /
      data  ssaice2(:, 24)   /                                          &
     & 9.999949e-01,9.999947e-01,9.999943e-01,9.999939e-01,9.999934e-01,&
     & 9.999927e-01,9.999920e-01,9.999913e-01,9.999904e-01,9.999895e-01,&
     & 9.999885e-01,9.999874e-01,9.999863e-01,9.999851e-01,9.999838e-01,&
     & 9.999824e-01,9.999810e-01,9.999795e-01,9.999780e-01,9.999764e-01,&
     & 9.999747e-01,9.999729e-01,9.999711e-01,9.999692e-01,9.999673e-01,&
     & 9.999653e-01,9.999632e-01,9.999611e-01,9.999589e-01,9.999566e-01,&
     & 9.999543e-01,9.999519e-01,9.999495e-01,9.999470e-01,9.999444e-01,&
     & 9.999418e-01,9.999392e-01,9.999364e-01,9.999336e-01,9.999308e-01,&
     & 9.999279e-01,9.999249e-01,9.999219e-01 /
      data  ssaice2(:, 25)   /                                          &
     & 9.999997e-01,9.999997e-01,9.999997e-01,9.999996e-01,9.999996e-01,&
     & 9.999995e-01,9.999994e-01,9.999993e-01,9.999993e-01,9.999992e-01,&
     & 9.999991e-01,9.999989e-01,9.999988e-01,9.999987e-01,9.999986e-01,&
     & 9.999984e-01,9.999983e-01,9.999981e-01,9.999980e-01,9.999978e-01,&
     & 9.999976e-01,9.999974e-01,9.999972e-01,9.999971e-01,9.999969e-01,&
     & 9.999966e-01,9.999964e-01,9.999962e-01,9.999960e-01,9.999957e-01,&
     & 9.999955e-01,9.999953e-01,9.999950e-01,9.999947e-01,9.999945e-01,&
     & 9.999942e-01,9.999939e-01,9.999936e-01,9.999934e-01,9.999931e-01,&
     & 9.999928e-01,9.999925e-01,9.999921e-01 /
      data  ssaice2(:, 26)   /                                          &
     & 9.999997e-01,9.999996e-01,9.999996e-01,9.999995e-01,9.999994e-01,&
     & 9.999993e-01,9.999992e-01,9.999991e-01,9.999990e-01,9.999989e-01,&
     & 9.999987e-01,9.999986e-01,9.999984e-01,9.999982e-01,9.999980e-01,&
     & 9.999978e-01,9.999976e-01,9.999974e-01,9.999972e-01,9.999970e-01,&
     & 9.999967e-01,9.999965e-01,9.999962e-01,9.999959e-01,9.999956e-01,&
     & 9.999954e-01,9.999951e-01,9.999947e-01,9.999944e-01,9.999941e-01,&
     & 9.999938e-01,9.999934e-01,9.999931e-01,9.999927e-01,9.999923e-01,&
     & 9.999920e-01,9.999916e-01,9.999912e-01,9.999908e-01,9.999904e-01,&
     & 9.999899e-01,9.999895e-01,9.999891e-01 /
      data  ssaice2(:, 27)   /                                          &
     & 9.999987e-01,9.999987e-01,9.999985e-01,9.999984e-01,9.999982e-01,&
     & 9.999980e-01,9.999978e-01,9.999976e-01,9.999973e-01,9.999970e-01,&
     & 9.999967e-01,9.999964e-01,9.999960e-01,9.999956e-01,9.999952e-01,&
     & 9.999948e-01,9.999944e-01,9.999939e-01,9.999934e-01,9.999929e-01,&
     & 9.999924e-01,9.999918e-01,9.999913e-01,9.999907e-01,9.999901e-01,&
     & 9.999894e-01,9.999888e-01,9.999881e-01,9.999874e-01,9.999867e-01,&
     & 9.999860e-01,9.999853e-01,9.999845e-01,9.999837e-01,9.999829e-01,&
     & 9.999821e-01,9.999813e-01,9.999804e-01,9.999796e-01,9.999787e-01,&
     & 9.999778e-01,9.999768e-01,9.999759e-01 /
      data  ssaice2(:, 28)   /                                          &
     & 9.999989e-01,9.999989e-01,9.999987e-01,9.999986e-01,9.999984e-01,&
     & 9.999982e-01,9.999980e-01,9.999978e-01,9.999975e-01,9.999972e-01,&
     & 9.999969e-01,9.999966e-01,9.999962e-01,9.999958e-01,9.999954e-01,&
     & 9.999950e-01,9.999945e-01,9.999941e-01,9.999936e-01,9.999931e-01,&
     & 9.999925e-01,9.999920e-01,9.999914e-01,9.999908e-01,9.999902e-01,&
     & 9.999896e-01,9.999889e-01,9.999883e-01,9.999876e-01,9.999869e-01,&
     & 9.999861e-01,9.999854e-01,9.999846e-01,9.999838e-01,9.999830e-01,&
     & 9.999822e-01,9.999814e-01,9.999805e-01,9.999796e-01,9.999787e-01,&
     & 9.999778e-01,9.999769e-01,9.999759e-01 /
      data  ssaice2(:, 29)   /                                          &
     & 7.042143e-01,6.691161e-01,6.463240e-01,6.296590e-01,6.166381e-01,&
     & 6.060183e-01,5.970908e-01,5.894144e-01,5.826968e-01,5.767343e-01,&
     & 5.713804e-01,5.665256e-01,5.620867e-01,5.579987e-01,5.542101e-01,&
     & 5.506794e-01,5.473727e-01,5.442620e-01,5.413239e-01,5.385389e-01,&
     & 5.358901e-01,5.333633e-01,5.309460e-01,5.286277e-01,5.263988e-01,&
     & 5.242512e-01,5.221777e-01,5.201719e-01,5.182280e-01,5.163410e-01,&
     & 5.145062e-01,5.127197e-01,5.109776e-01,5.092766e-01,5.076137e-01,&
     & 5.059860e-01,5.043911e-01,5.028266e-01,5.012904e-01,4.997805e-01,&
     & 4.982951e-01,4.968326e-01,4.953913e-01 /

!  --- ...  asymmetry factor from streamer v3, unitless
      data  asyice2(:, 16)   /                                          &
     & 7.946655e-01,8.547685e-01,8.806016e-01,8.949880e-01,9.041676e-01,&
     & 9.105399e-01,9.152249e-01,9.188160e-01,9.216573e-01,9.239620e-01,&
     & 9.258695e-01,9.274745e-01,9.288441e-01,9.300267e-01,9.310584e-01,&
     & 9.319665e-01,9.327721e-01,9.334918e-01,9.341387e-01,9.347236e-01,&
     & 9.352551e-01,9.357402e-01,9.361850e-01,9.365942e-01,9.369722e-01,&
     & 9.373225e-01,9.376481e-01,9.379516e-01,9.382352e-01,9.385010e-01,&
     & 9.387505e-01,9.389854e-01,9.392070e-01,9.394163e-01,9.396145e-01,&
     & 9.398024e-01,9.399809e-01,9.401508e-01,9.403126e-01,9.404670e-01,&
     & 9.406144e-01,9.407555e-01,9.408906e-01 /
      data  asyice2(:, 17)   /                                          &
     & 9.078091e-01,9.195850e-01,9.267250e-01,9.317083e-01,9.354632e-01,&
     & 9.384323e-01,9.408597e-01,9.428935e-01,9.446301e-01,9.461351e-01,&
     & 9.474555e-01,9.486259e-01,9.496722e-01,9.506146e-01,9.514688e-01,&
     & 9.522476e-01,9.529612e-01,9.536181e-01,9.542251e-01,9.547883e-01,&
     & 9.553124e-01,9.558019e-01,9.562601e-01,9.566904e-01,9.570953e-01,&
     & 9.574773e-01,9.578385e-01,9.581806e-01,9.585054e-01,9.588142e-01,&
     & 9.591083e-01,9.593888e-01,9.596569e-01,9.599135e-01,9.601593e-01,&
     & 9.603952e-01,9.606219e-01,9.608399e-01,9.610499e-01,9.612523e-01,&
     & 9.614477e-01,9.616365e-01,9.618192e-01 /
      data  asyice2(:, 18)   /                                          &
     & 8.322045e-01,8.528693e-01,8.648167e-01,8.729163e-01,8.789054e-01,&
     & 8.835845e-01,8.873819e-01,8.905511e-01,8.932532e-01,8.955965e-01,&
     & 8.976567e-01,8.994887e-01,9.011334e-01,9.026221e-01,9.039791e-01,&
     & 9.052237e-01,9.063715e-01,9.074349e-01,9.084245e-01,9.093489e-01,&
     & 9.102154e-01,9.110303e-01,9.117987e-01,9.125253e-01,9.132140e-01,&
     & 9.138682e-01,9.144910e-01,9.150850e-01,9.156524e-01,9.161955e-01,&
     & 9.167160e-01,9.172157e-01,9.176959e-01,9.181581e-01,9.186034e-01,&
     & 9.190330e-01,9.194478e-01,9.198488e-01,9.202368e-01,9.206126e-01,&
     & 9.209768e-01,9.213301e-01,9.216731e-01 /
      data  asyice2(:, 19)   /                                          &
     & 8.116560e-01,8.488278e-01,8.674331e-01,8.788148e-01,8.865810e-01,&
     & 8.922595e-01,8.966149e-01,9.000747e-01,9.028980e-01,9.052513e-01,&
     & 9.072468e-01,9.089632e-01,9.104574e-01,9.117713e-01,9.129371e-01,&
     & 9.139793e-01,9.149174e-01,9.157668e-01,9.165400e-01,9.172473e-01,&
     & 9.178970e-01,9.184962e-01,9.190508e-01,9.195658e-01,9.200455e-01,&
     & 9.204935e-01,9.209130e-01,9.213067e-01,9.216771e-01,9.220262e-01,&
     & 9.223560e-01,9.226680e-01,9.229636e-01,9.232443e-01,9.235112e-01,&
     & 9.237652e-01,9.240074e-01,9.242385e-01,9.244594e-01,9.246708e-01,&
     & 9.248733e-01,9.250674e-01,9.252536e-01 /
      data  asyice2(:, 20)   /                                          &
     & 8.047113e-01,8.402864e-01,8.570332e-01,8.668455e-01,8.733206e-01,&
     & 8.779272e-01,8.813796e-01,8.840676e-01,8.862225e-01,8.879904e-01,&
     & 8.894682e-01,8.907228e-01,8.918019e-01,8.927404e-01,8.935645e-01,&
     & 8.942943e-01,8.949452e-01,8.955296e-01,8.960574e-01,8.965366e-01,&
     & 8.969736e-01,8.973740e-01,8.977422e-01,8.980820e-01,8.983966e-01,&
     & 8.986889e-01,8.989611e-01,8.992153e-01,8.994533e-01,8.996766e-01,&
     & 8.998865e-01,9.000843e-01,9.002709e-01,9.004474e-01,9.006146e-01,&
     & 9.007731e-01,9.009237e-01,9.010670e-01,9.012034e-01,9.013336e-01,&
     & 9.014579e-01,9.015767e-01,9.016904e-01 /
      data  asyice2(:, 21)   /                                          &
     & 8.179122e-01,8.480726e-01,8.621945e-01,8.704354e-01,8.758555e-01,&
     & 8.797007e-01,8.825750e-01,8.848078e-01,8.865939e-01,8.880564e-01,&
     & 8.892765e-01,8.903105e-01,8.911982e-01,8.919689e-01,8.926446e-01,&
     & 8.932419e-01,8.937738e-01,8.942506e-01,8.946806e-01,8.950702e-01,&
     & 8.954251e-01,8.957497e-01,8.960477e-01,8.963223e-01,8.965762e-01,&
     & 8.968116e-01,8.970306e-01,8.972347e-01,8.974255e-01,8.976042e-01,&
     & 8.977720e-01,8.979298e-01,8.980784e-01,8.982188e-01,8.983515e-01,&
     & 8.984771e-01,8.985963e-01,8.987095e-01,8.988171e-01,8.989195e-01,&
     & 8.990172e-01,8.991104e-01,8.991994e-01 /
      data  asyice2(:, 22)   /                                          &
     & 8.169789e-01,8.455024e-01,8.586925e-01,8.663283e-01,8.713217e-01,&
     & 8.748488e-01,8.774765e-01,8.795122e-01,8.811370e-01,8.824649e-01,&
     & 8.835711e-01,8.845073e-01,8.853103e-01,8.860068e-01,8.866170e-01,&
     & 8.871560e-01,8.876358e-01,8.880658e-01,8.884533e-01,8.888044e-01,&
     & 8.891242e-01,8.894166e-01,8.896851e-01,8.899324e-01,8.901612e-01,&
     & 8.903733e-01,8.905706e-01,8.907545e-01,8.909265e-01,8.910876e-01,&
     & 8.912388e-01,8.913812e-01,8.915153e-01,8.916419e-01,8.917617e-01,&
     & 8.918752e-01,8.919829e-01,8.920851e-01,8.921824e-01,8.922751e-01,&
     & 8.923635e-01,8.924478e-01,8.925284e-01 /
      data  asyice2(:, 23)   /                                          &
     & 8.387642e-01,8.569979e-01,8.658630e-01,8.711825e-01,8.747605e-01,&
     & 8.773472e-01,8.793129e-01,8.808621e-01,8.821179e-01,8.831583e-01,&
     & 8.840361e-01,8.847875e-01,8.854388e-01,8.860094e-01,8.865138e-01,&
     & 8.869634e-01,8.873668e-01,8.877310e-01,8.880617e-01,8.883635e-01,&
     & 8.886401e-01,8.888947e-01,8.891298e-01,8.893477e-01,8.895504e-01,&
     & 8.897393e-01,8.899159e-01,8.900815e-01,8.902370e-01,8.903833e-01,&
     & 8.905214e-01,8.906518e-01,8.907753e-01,8.908924e-01,8.910036e-01,&
     & 8.911094e-01,8.912101e-01,8.913062e-01,8.913979e-01,8.914856e-01,&
     & 8.915695e-01,8.916498e-01,8.917269e-01 /
      data  asyice2(:, 24)   /                                          &
     & 8.522208e-01,8.648132e-01,8.711224e-01,8.749901e-01,8.776354e-01,&
     & 8.795743e-01,8.810649e-01,8.822518e-01,8.832225e-01,8.840333e-01,&
     & 8.847224e-01,8.853162e-01,8.858342e-01,8.862906e-01,8.866962e-01,&
     & 8.870595e-01,8.873871e-01,8.876842e-01,8.879551e-01,8.882032e-01,&
     & 8.884316e-01,8.886425e-01,8.888380e-01,8.890199e-01,8.891895e-01,&
     & 8.893481e-01,8.894968e-01,8.896366e-01,8.897683e-01,8.898926e-01,&
     & 8.900102e-01,8.901215e-01,8.902272e-01,8.903276e-01,8.904232e-01,&
     & 8.905144e-01,8.906014e-01,8.906845e-01,8.907640e-01,8.908402e-01,&
     & 8.909132e-01,8.909834e-01,8.910507e-01 /
      data  asyice2(:, 25)   /                                          &
     & 8.578202e-01,8.683033e-01,8.735431e-01,8.767488e-01,8.789378e-01,&
     & 8.805399e-01,8.817701e-01,8.827485e-01,8.835480e-01,8.842152e-01,&
     & 8.847817e-01,8.852696e-01,8.856949e-01,8.860694e-01,8.864020e-01,&
     & 8.866997e-01,8.869681e-01,8.872113e-01,8.874330e-01,8.876360e-01,&
     & 8.878227e-01,8.879951e-01,8.881548e-01,8.883033e-01,8.884418e-01,&
     & 8.885712e-01,8.886926e-01,8.888066e-01,8.889139e-01,8.890152e-01,&
     & 8.891110e-01,8.892017e-01,8.892877e-01,8.893695e-01,8.894473e-01,&
     & 8.895214e-01,8.895921e-01,8.896597e-01,8.897243e-01,8.897862e-01,&
     & 8.898456e-01,8.899025e-01,8.899572e-01 /
      data  asyice2(:, 26)   /                                          &
     & 8.625615e-01,8.713831e-01,8.755799e-01,8.780560e-01,8.796983e-01,&
     & 8.808714e-01,8.817534e-01,8.824420e-01,8.829953e-01,8.834501e-01,&
     & 8.838310e-01,8.841549e-01,8.844338e-01,8.846767e-01,8.848902e-01,&
     & 8.850795e-01,8.852484e-01,8.854002e-01,8.855374e-01,8.856620e-01,&
     & 8.857758e-01,8.858800e-01,8.859759e-01,8.860644e-01,8.861464e-01,&
     & 8.862225e-01,8.862935e-01,8.863598e-01,8.864218e-01,8.864800e-01,&
     & 8.865347e-01,8.865863e-01,8.866349e-01,8.866809e-01,8.867245e-01,&
     & 8.867658e-01,8.868050e-01,8.868423e-01,8.868778e-01,8.869117e-01,&
     & 8.869440e-01,8.869749e-01,8.870044e-01 /
      data  asyice2(:, 27)   /                                          &
     & 8.587495e-01,8.684764e-01,8.728189e-01,8.752872e-01,8.768846e-01,&
     & 8.780060e-01,8.788386e-01,8.794824e-01,8.799960e-01,8.804159e-01,&
     & 8.807660e-01,8.810626e-01,8.813175e-01,8.815390e-01,8.817335e-01,&
     & 8.819057e-01,8.820593e-01,8.821973e-01,8.823220e-01,8.824353e-01,&
     & 8.825387e-01,8.826336e-01,8.827209e-01,8.828016e-01,8.828764e-01,&
     & 8.829459e-01,8.830108e-01,8.830715e-01,8.831283e-01,8.831817e-01,&
     & 8.832320e-01,8.832795e-01,8.833244e-01,8.833668e-01,8.834071e-01,&
     & 8.834454e-01,8.834817e-01,8.835164e-01,8.835495e-01,8.835811e-01,&
     & 8.836113e-01,8.836402e-01,8.836679e-01 /
      data  asyice2(:, 28)   /                                          &
     & 8.561110e-01,8.678583e-01,8.727554e-01,8.753892e-01,8.770154e-01,&
     & 8.781109e-01,8.788949e-01,8.794812e-01,8.799348e-01,8.802952e-01,&
     & 8.805880e-01,8.808300e-01,8.810331e-01,8.812058e-01,8.813543e-01,&
     & 8.814832e-01,8.815960e-01,8.816956e-01,8.817839e-01,8.818629e-01,&
     & 8.819339e-01,8.819979e-01,8.820560e-01,8.821089e-01,8.821573e-01,&
     & 8.822016e-01,8.822425e-01,8.822801e-01,8.823150e-01,8.823474e-01,&
     & 8.823775e-01,8.824056e-01,8.824318e-01,8.824564e-01,8.824795e-01,&
     & 8.825011e-01,8.825215e-01,8.825408e-01,8.825589e-01,8.825761e-01,&
     & 8.825924e-01,8.826078e-01,8.826224e-01 /
      data  asyice2(:, 29)   /                                          &
     & 8.311124e-01,8.688197e-01,8.900274e-01,9.040696e-01,9.142334e-01,&
     & 9.220181e-01,9.282195e-01,9.333048e-01,9.375689e-01,9.412085e-01,&
     & 9.443604e-01,9.471230e-01,9.495694e-01,9.517549e-01,9.537224e-01,&
     & 9.555057e-01,9.571316e-01,9.586222e-01,9.599952e-01,9.612656e-01,&
     & 9.624458e-01,9.635461e-01,9.645756e-01,9.655418e-01,9.664513e-01,&
     & 9.673098e-01,9.681222e-01,9.688928e-01,9.696256e-01,9.703237e-01,&
     & 9.709903e-01,9.716280e-01,9.722391e-01,9.728258e-01,9.733901e-01,&
     & 9.739336e-01,9.744579e-01,9.749645e-01,9.754546e-01,9.759294e-01,&
     & 9.763901e-01,9.768376e-01,9.772727e-01 /

!  --- ...  hexagonal ice particle parameterization from fu
!           extinction units (ext coef/iwc): [(m^-1)/(g m^-3)]
      data  extice3(:, 16)   /                                          &
     & 5.194013e-01,3.215089e-01,2.327917e-01,1.824424e-01,1.499977e-01,&
     & 1.273492e-01,1.106421e-01,9.780982e-02,8.764435e-02,7.939266e-02,&
     & 7.256081e-02,6.681137e-02,6.190600e-02,5.767154e-02,5.397915e-02,&
     & 5.073102e-02,4.785151e-02,4.528125e-02,4.297296e-02,4.088853e-02,&
     & 3.899690e-02,3.727251e-02,3.569411e-02,3.424393e-02,3.290694e-02,&
     & 3.167040e-02,3.052340e-02,2.945654e-02,2.846172e-02,2.753188e-02,&
     & 2.666085e-02,2.584322e-02,2.507423e-02,2.434967e-02,2.366579e-02,&
     & 2.301926e-02,2.240711e-02,2.182666e-02,2.127551e-02,2.075150e-02,&
     & 2.025267e-02,1.977725e-02,1.932364e-02,1.889035e-02,1.847607e-02,&
     & 1.807956e-02 /
      data  extice3(:, 17)   /                                          &
     & 4.901155e-01,3.065286e-01,2.230800e-01,1.753951e-01,1.445402e-01,&
     & 1.229417e-01,1.069777e-01,9.469760e-02,8.495824e-02,7.704501e-02,&
     & 7.048834e-02,6.496693e-02,6.025353e-02,5.618286e-02,5.263186e-02,&
     & 4.950698e-02,4.673585e-02,4.426164e-02,4.203904e-02,4.003153e-02,&
     & 3.820932e-02,3.654790e-02,3.502688e-02,3.362919e-02,3.234041e-02,&
     & 3.114829e-02,3.004234e-02,2.901356e-02,2.805413e-02,2.715727e-02,&
     & 2.631705e-02,2.552828e-02,2.478637e-02,2.408725e-02,2.342734e-02,&
     & 2.280343e-02,2.221264e-02,2.165242e-02,2.112043e-02,2.061461e-02,&
     & 2.013308e-02,1.967411e-02,1.923616e-02,1.881783e-02,1.841781e-02,&
     & 1.803494e-02 /
      data  extice3(:, 18)   /                                          &
     & 5.056264e-01,3.160261e-01,2.298442e-01,1.805973e-01,1.487318e-01,&
     & 1.264258e-01,1.099389e-01,9.725656e-02,8.719819e-02,7.902576e-02,&
     & 7.225433e-02,6.655206e-02,6.168427e-02,5.748028e-02,5.381296e-02,&
     & 5.058572e-02,4.772383e-02,4.516857e-02,4.287317e-02,4.079990e-02,&
     & 3.891801e-02,3.720217e-02,3.563133e-02,3.418786e-02,3.285686e-02,&
     & 3.162569e-02,3.048352e-02,2.942104e-02,2.843018e-02,2.750395e-02,&
     & 2.663621e-02,2.582160e-02,2.505539e-02,2.433337e-02,2.365185e-02,&
     & 2.300750e-02,2.239736e-02,2.181878e-02,2.126937e-02,2.074699e-02,&
     & 2.024968e-02,1.977567e-02,1.932338e-02,1.889134e-02,1.847823e-02,&
     & 1.808281e-02 /
      data  extice3(:, 19)   /                                          &
     & 4.881605e-01,3.055237e-01,2.225070e-01,1.750688e-01,1.443736e-01,&
     & 1.228869e-01,1.070054e-01,9.478893e-02,8.509997e-02,7.722769e-02,&
     & 7.070495e-02,6.521211e-02,6.052311e-02,5.647351e-02,5.294088e-02,&
     & 4.983217e-02,4.707539e-02,4.461398e-02,4.240288e-02,4.040575e-02,&
     & 3.859298e-02,3.694016e-02,3.542701e-02,3.403655e-02,3.275444e-02,&
     & 3.156849e-02,3.046827e-02,2.944481e-02,2.849034e-02,2.759812e-02,&
     & 2.676226e-02,2.597757e-02,2.523949e-02,2.454400e-02,2.388750e-02,&
     & 2.326682e-02,2.267909e-02,2.212176e-02,2.159253e-02,2.108933e-02,&
     & 2.061028e-02,2.015369e-02,1.971801e-02,1.930184e-02,1.890389e-02,&
     & 1.852300e-02 /
      data  extice3(:, 20)   /                                          &
     & 5.103703e-01,3.188144e-01,2.317435e-01,1.819887e-01,1.497944e-01,&
     & 1.272584e-01,1.106013e-01,9.778822e-02,8.762610e-02,7.936938e-02,&
     & 7.252809e-02,6.676701e-02,6.184901e-02,5.760165e-02,5.389651e-02,&
     & 5.063598e-02,4.774457e-02,4.516295e-02,4.284387e-02,4.074922e-02,&
     & 3.884792e-02,3.711438e-02,3.552734e-02,3.406898e-02,3.272425e-02,&
     & 3.148038e-02,3.032643e-02,2.925299e-02,2.825191e-02,2.731612e-02,&
     & 2.643943e-02,2.561642e-02,2.484230e-02,2.411284e-02,2.342429e-02,&
     & 2.277329e-02,2.215686e-02,2.157231e-02,2.101724e-02,2.048946e-02,&
     & 1.998702e-02,1.950813e-02,1.905118e-02,1.861468e-02,1.819730e-02,&
     & 1.779781e-02 /
      data  extice3(:, 21)   /                                          &
     & 5.031161e-01,3.144511e-01,2.286942e-01,1.796903e-01,1.479819e-01,&
     & 1.257860e-01,1.093803e-01,9.676059e-02,8.675183e-02,7.861971e-02,&
     & 7.188168e-02,6.620754e-02,6.136376e-02,5.718050e-02,5.353127e-02,&
     & 5.031995e-02,4.747218e-02,4.492952e-02,4.264544e-02,4.058240e-02,&
     & 3.870979e-02,3.700242e-02,3.543933e-02,3.400297e-02,3.267854e-02,&
     & 3.145345e-02,3.031691e-02,2.925967e-02,2.827370e-02,2.735203e-02,&
     & 2.648858e-02,2.567798e-02,2.491555e-02,2.419710e-02,2.351893e-02,&
     & 2.287776e-02,2.227063e-02,2.169491e-02,2.114821e-02,2.062840e-02,&
     & 2.013354e-02,1.966188e-02,1.921182e-02,1.878191e-02,1.837083e-02,&
     & 1.797737e-02 /
      data  extice3(:, 22)   /                                          &
     & 4.949453e-01,3.095918e-01,2.253402e-01,1.771964e-01,1.460446e-01,&
     & 1.242383e-01,1.081206e-01,9.572235e-02,8.588928e-02,7.789990e-02,&
     & 7.128013e-02,6.570559e-02,6.094684e-02,5.683701e-02,5.325183e-02,&
     & 5.009688e-02,4.729909e-02,4.480106e-02,4.255708e-02,4.053025e-02,&
     & 3.869051e-02,3.701310e-02,3.547745e-02,3.406631e-02,3.276512e-02,&
     & 3.156153e-02,3.044494e-02,2.940626e-02,2.843759e-02,2.753211e-02,&
     & 2.668381e-02,2.588744e-02,2.513839e-02,2.443255e-02,2.376629e-02,&
     & 2.313637e-02,2.253990e-02,2.197428e-02,2.143718e-02,2.092649e-02,&
     & 2.044032e-02,1.997694e-02,1.953478e-02,1.911241e-02,1.870855e-02,&
     & 1.832199e-02 /
      data  extice3(:, 23)   /                                          &
     & 5.052816e-01,3.157665e-01,2.296233e-01,1.803986e-01,1.485473e-01,&
     & 1.262514e-01,1.097718e-01,9.709524e-02,8.704139e-02,7.887264e-02,&
     & 7.210424e-02,6.640454e-02,6.153894e-02,5.733683e-02,5.367116e-02,&
     & 5.044537e-02,4.758477e-02,4.503066e-02,4.273629e-02,4.066395e-02,&
     & 3.878291e-02,3.706784e-02,3.549771e-02,3.405488e-02,3.272448e-02,&
     & 3.149387e-02,3.035221e-02,2.929020e-02,2.829979e-02,2.737397e-02,&
     & 2.650663e-02,2.569238e-02,2.492651e-02,2.420482e-02,2.352361e-02,&
     & 2.287954e-02,2.226968e-02,2.169136e-02,2.114220e-02,2.062005e-02,&
     & 2.012296e-02,1.964917e-02,1.919709e-02,1.876524e-02,1.835231e-02,&
     & 1.795707e-02 /
      data  extice3(:, 24)   /                                          &
     & 5.042067e-01,3.151195e-01,2.291708e-01,1.800573e-01,1.482779e-01,&
     & 1.260324e-01,1.095900e-01,9.694202e-02,8.691087e-02,7.876056e-02,&
     & 7.200745e-02,6.632062e-02,6.146600e-02,5.727338e-02,5.361599e-02,&
     & 5.039749e-02,4.754334e-02,4.499500e-02,4.270580e-02,4.063815e-02,&
     & 3.876135e-02,3.705016e-02,3.548357e-02,3.404400e-02,3.271661e-02,&
     & 3.148877e-02,3.034969e-02,2.929008e-02,2.830191e-02,2.737818e-02,&
     & 2.651279e-02,2.570039e-02,2.493624e-02,2.421618e-02,2.353650e-02,&
     & 2.289390e-02,2.228541e-02,2.170840e-02,2.116048e-02,2.063950e-02,&
     & 2.014354e-02,1.967082e-02,1.921975e-02,1.878888e-02,1.837688e-02,&
     & 1.798254e-02 /
      data  extice3(:, 25)   /                                          &
     & 5.022507e-01,3.139246e-01,2.283218e-01,1.794059e-01,1.477544e-01,&
     & 1.255984e-01,1.092222e-01,9.662516e-02,8.663439e-02,7.851688e-02,&
     & 7.179095e-02,6.612700e-02,6.129193e-02,5.711618e-02,5.347351e-02,&
     & 5.026796e-02,4.742530e-02,4.488721e-02,4.260724e-02,4.054790e-02,&
     & 3.867866e-02,3.697435e-02,3.541407e-02,3.398029e-02,3.265824e-02,&
     & 3.143535e-02,3.030085e-02,2.924551e-02,2.826131e-02,2.734130e-02,&
     & 2.647939e-02,2.567026e-02,2.490919e-02,2.419203e-02,2.351509e-02,&
     & 2.287507e-02,2.226903e-02,2.169434e-02,2.114862e-02,2.062975e-02,&
     & 2.013578e-02,1.966496e-02,1.921571e-02,1.878658e-02,1.837623e-02,&
     & 1.798348e-02 /
      data  extice3(:, 26)   /                                          &
     & 5.068316e-01,3.166869e-01,2.302576e-01,1.808693e-01,1.489122e-01,&
     & 1.265423e-01,1.100080e-01,9.728926e-02,8.720201e-02,7.900612e-02,&
     & 7.221524e-02,6.649660e-02,6.161484e-02,5.739877e-02,5.372093e-02,&
     & 5.048442e-02,4.761431e-02,4.505172e-02,4.274972e-02,4.067050e-02,&
     & 3.878321e-02,3.706244e-02,3.548710e-02,3.403948e-02,3.270466e-02,&
     & 3.146995e-02,3.032450e-02,2.925897e-02,2.826527e-02,2.733638e-02,&
     & 2.646615e-02,2.564920e-02,2.488078e-02,2.415670e-02,2.347322e-02,&
     & 2.282702e-02,2.221513e-02,2.163489e-02,2.108390e-02,2.056002e-02,&
     & 2.006128e-02,1.958591e-02,1.913232e-02,1.869904e-02,1.828474e-02,&
     & 1.788819e-02 /
      data  extice3(:, 27)   /                                          &
     & 5.077707e-01,3.172636e-01,2.306695e-01,1.811871e-01,1.491691e-01,&
     & 1.267565e-01,1.101907e-01,9.744773e-02,8.734125e-02,7.912973e-02,&
     & 7.232591e-02,6.659637e-02,6.170530e-02,5.748120e-02,5.379634e-02,&
     & 5.055367e-02,4.767809e-02,4.511061e-02,4.280423e-02,4.072104e-02,&
     & 3.883015e-02,3.710611e-02,3.552776e-02,3.407738e-02,3.274002e-02,&
     & 3.150296e-02,3.035532e-02,2.928776e-02,2.829216e-02,2.736150e-02,&
     & 2.648961e-02,2.567111e-02,2.490123e-02,2.417576e-02,2.349098e-02,&
     & 2.284354e-02,2.223049e-02,2.164914e-02,2.109711e-02,2.057222e-02,&
     & 2.007253e-02,1.959626e-02,1.914181e-02,1.870770e-02,1.829261e-02,&
     & 1.789531e-02 /
      data  extice3(:, 28)   /                                          &
     & 5.062281e-01,3.163402e-01,2.300275e-01,1.807060e-01,1.487921e-01,&
     & 1.264523e-01,1.099403e-01,9.723879e-02,8.716516e-02,7.898034e-02,&
     & 7.219863e-02,6.648771e-02,6.161254e-02,5.740217e-02,5.372929e-02,&
     & 5.049716e-02,4.763092e-02,4.507179e-02,4.277290e-02,4.069649e-02,&
     & 3.881175e-02,3.709331e-02,3.552008e-02,3.407442e-02,3.274141e-02,&
     & 3.150837e-02,3.036447e-02,2.930037e-02,2.830801e-02,2.738037e-02,&
     & 2.651132e-02,2.569547e-02,2.492810e-02,2.420499e-02,2.352243e-02,&
     & 2.287710e-02,2.226604e-02,2.168658e-02,2.113634e-02,2.061316e-02,&
     & 2.011510e-02,1.964038e-02,1.918740e-02,1.875471e-02,1.834096e-02,&
     & 1.794495e-02 /
      data  extice3(:, 29)   /                                          &
     & 1.338834e-01,1.924912e-01,1.755523e-01,1.534793e-01,1.343937e-01,&
     & 1.187883e-01,1.060654e-01,9.559106e-02,8.685880e-02,7.948698e-02,&
     & 7.319086e-02,6.775669e-02,6.302215e-02,5.886236e-02,5.517996e-02,&
     & 5.189810e-02,4.895539e-02,4.630225e-02,4.389823e-02,4.171002e-02,&
     & 3.970998e-02,3.787493e-02,3.618537e-02,3.462471e-02,3.317880e-02,&
     & 3.183547e-02,3.058421e-02,2.941590e-02,2.832256e-02,2.729724e-02,&
     & 2.633377e-02,2.542675e-02,2.457136e-02,2.376332e-02,2.299882e-02,&
     & 2.227443e-02,2.158707e-02,2.093400e-02,2.031270e-02,1.972091e-02,&
     & 1.915659e-02,1.861787e-02,1.810304e-02,1.761055e-02,1.713899e-02,&
     & 1.668704e-02 /

!  --- ...  single-scattering albedo from fu, unitless
      data  ssaice3(:, 16)   /                                          &
     & 6.749442e-01,6.649947e-01,6.565828e-01,6.489928e-01,6.420046e-01,&
     & 6.355231e-01,6.294964e-01,6.238901e-01,6.186783e-01,6.138395e-01,&
     & 6.093543e-01,6.052049e-01,6.013742e-01,5.978457e-01,5.946030e-01,&
     & 5.916302e-01,5.889115e-01,5.864310e-01,5.841731e-01,5.821221e-01,&
     & 5.802624e-01,5.785785e-01,5.770549e-01,5.756759e-01,5.744262e-01,&
     & 5.732901e-01,5.722524e-01,5.712974e-01,5.704097e-01,5.695739e-01,&
     & 5.687747e-01,5.679964e-01,5.672238e-01,5.664415e-01,5.656340e-01,&
     & 5.647860e-01,5.638821e-01,5.629070e-01,5.618452e-01,5.606815e-01,&
     & 5.594006e-01,5.579870e-01,5.564255e-01,5.547008e-01,5.527976e-01,&
     & 5.507005e-01 /
      data  ssaice3(:, 17)   /                                          &
     & 7.628550e-01,7.567297e-01,7.508463e-01,7.451972e-01,7.397745e-01,&
     & 7.345705e-01,7.295775e-01,7.247881e-01,7.201945e-01,7.157894e-01,&
     & 7.115652e-01,7.075145e-01,7.036300e-01,6.999044e-01,6.963304e-01,&
     & 6.929007e-01,6.896083e-01,6.864460e-01,6.834067e-01,6.804833e-01,&
     & 6.776690e-01,6.749567e-01,6.723397e-01,6.698109e-01,6.673637e-01,&
     & 6.649913e-01,6.626870e-01,6.604441e-01,6.582561e-01,6.561163e-01,&
     & 6.540182e-01,6.519554e-01,6.499215e-01,6.479099e-01,6.459145e-01,&
     & 6.439289e-01,6.419468e-01,6.399621e-01,6.379686e-01,6.359601e-01,&
     & 6.339306e-01,6.318740e-01,6.297845e-01,6.276559e-01,6.254825e-01,&
     & 6.232583e-01 /
      data  ssaice3(:, 18)   /                                          &
     & 9.924147e-01,9.882792e-01,9.842257e-01,9.802522e-01,9.763566e-01,&
     & 9.725367e-01,9.687905e-01,9.651157e-01,9.615104e-01,9.579725e-01,&
     & 9.544997e-01,9.510901e-01,9.477416e-01,9.444520e-01,9.412194e-01,&
     & 9.380415e-01,9.349165e-01,9.318421e-01,9.288164e-01,9.258373e-01,&
     & 9.229027e-01,9.200106e-01,9.171589e-01,9.143457e-01,9.115688e-01,&
     & 9.088263e-01,9.061161e-01,9.034362e-01,9.007846e-01,8.981592e-01,&
     & 8.955581e-01,8.929792e-01,8.904206e-01,8.878803e-01,8.853562e-01,&
     & 8.828464e-01,8.803488e-01,8.778616e-01,8.753827e-01,8.729102e-01,&
     & 8.704421e-01,8.679764e-01,8.655112e-01,8.630445e-01,8.605744e-01,&
     & 8.580989e-01 /
      data  ssaice3(:, 19)   /                                          &
     & 9.629413e-01,9.517182e-01,9.409209e-01,9.305366e-01,9.205529e-01,&
     & 9.109569e-01,9.017362e-01,8.928780e-01,8.843699e-01,8.761992e-01,&
     & 8.683536e-01,8.608204e-01,8.535873e-01,8.466417e-01,8.399712e-01,&
     & 8.335635e-01,8.274062e-01,8.214868e-01,8.157932e-01,8.103129e-01,&
     & 8.050336e-01,7.999432e-01,7.950294e-01,7.902798e-01,7.856825e-01,&
     & 7.812250e-01,7.768954e-01,7.726815e-01,7.685711e-01,7.645522e-01,&
     & 7.606126e-01,7.567404e-01,7.529234e-01,7.491498e-01,7.454074e-01,&
     & 7.416844e-01,7.379688e-01,7.342485e-01,7.305118e-01,7.267468e-01,&
     & 7.229415e-01,7.190841e-01,7.151628e-01,7.111657e-01,7.070811e-01,&
     & 7.028972e-01 /
      data  ssaice3(:, 20)   /                                          &
     & 9.942270e-01,9.909206e-01,9.876775e-01,9.844960e-01,9.813746e-01,&
     & 9.783114e-01,9.753049e-01,9.723535e-01,9.694553e-01,9.666088e-01,&
     & 9.638123e-01,9.610641e-01,9.583626e-01,9.557060e-01,9.530928e-01,&
     & 9.505211e-01,9.479895e-01,9.454961e-01,9.430393e-01,9.406174e-01,&
     & 9.382288e-01,9.358717e-01,9.335446e-01,9.312456e-01,9.289731e-01,&
     & 9.267255e-01,9.245010e-01,9.222980e-01,9.201147e-01,9.179496e-01,&
     & 9.158008e-01,9.136667e-01,9.115457e-01,9.094359e-01,9.073358e-01,&
     & 9.052436e-01,9.031577e-01,9.010763e-01,8.989977e-01,8.969203e-01,&
     & 8.948423e-01,8.927620e-01,8.906778e-01,8.885879e-01,8.864907e-01,&
     & 8.843843e-01 /
      data  ssaice3(:, 21)   /                                          &
     & 9.934014e-01,9.899331e-01,9.865537e-01,9.832610e-01,9.800523e-01,&
     & 9.769254e-01,9.738777e-01,9.709069e-01,9.680106e-01,9.651862e-01,&
     & 9.624315e-01,9.597439e-01,9.571212e-01,9.545608e-01,9.520605e-01,&
     & 9.496177e-01,9.472301e-01,9.448954e-01,9.426111e-01,9.403749e-01,&
     & 9.381843e-01,9.360370e-01,9.339307e-01,9.318629e-01,9.298313e-01,&
     & 9.278336e-01,9.258673e-01,9.239302e-01,9.220198e-01,9.201338e-01,&
     & 9.182700e-01,9.164258e-01,9.145991e-01,9.127874e-01,9.109884e-01,&
     & 9.091999e-01,9.074194e-01,9.056447e-01,9.038735e-01,9.021033e-01,&
     & 9.003320e-01,8.985572e-01,8.967766e-01,8.949879e-01,8.931888e-01,&
     & 8.913770e-01 /
      data  ssaice3(:, 22)   /                                          &
     & 9.994833e-01,9.992055e-01,9.989278e-01,9.986500e-01,9.983724e-01,&
     & 9.980947e-01,9.978172e-01,9.975397e-01,9.972623e-01,9.969849e-01,&
     & 9.967077e-01,9.964305e-01,9.961535e-01,9.958765e-01,9.955997e-01,&
     & 9.953230e-01,9.950464e-01,9.947699e-01,9.944936e-01,9.942174e-01,&
     & 9.939414e-01,9.936656e-01,9.933899e-01,9.931144e-01,9.928390e-01,&
     & 9.925639e-01,9.922889e-01,9.920141e-01,9.917396e-01,9.914652e-01,&
     & 9.911911e-01,9.909171e-01,9.906434e-01,9.903700e-01,9.900967e-01,&
     & 9.898237e-01,9.895510e-01,9.892784e-01,9.890062e-01,9.887342e-01,&
     & 9.884625e-01,9.881911e-01,9.879199e-01,9.876490e-01,9.873784e-01,&
     & 9.871081e-01 /
      data  ssaice3(:, 23)   /                                          &
     & 9.999343e-01,9.998917e-01,9.998492e-01,9.998067e-01,9.997642e-01,&
     & 9.997218e-01,9.996795e-01,9.996372e-01,9.995949e-01,9.995528e-01,&
     & 9.995106e-01,9.994686e-01,9.994265e-01,9.993845e-01,9.993426e-01,&
     & 9.993007e-01,9.992589e-01,9.992171e-01,9.991754e-01,9.991337e-01,&
     & 9.990921e-01,9.990505e-01,9.990089e-01,9.989674e-01,9.989260e-01,&
     & 9.988846e-01,9.988432e-01,9.988019e-01,9.987606e-01,9.987194e-01,&
     & 9.986782e-01,9.986370e-01,9.985959e-01,9.985549e-01,9.985139e-01,&
     & 9.984729e-01,9.984319e-01,9.983910e-01,9.983502e-01,9.983094e-01,&
     & 9.982686e-01,9.982279e-01,9.981872e-01,9.981465e-01,9.981059e-01,&
     & 9.980653e-01 /
      data  ssaice3(:, 24)   /                                          &
     & 9.999978e-01,9.999965e-01,9.999952e-01,9.999939e-01,9.999926e-01,&
     & 9.999913e-01,9.999900e-01,9.999887e-01,9.999873e-01,9.999860e-01,&
     & 9.999847e-01,9.999834e-01,9.999821e-01,9.999808e-01,9.999795e-01,&
     & 9.999782e-01,9.999769e-01,9.999756e-01,9.999743e-01,9.999730e-01,&
     & 9.999717e-01,9.999704e-01,9.999691e-01,9.999678e-01,9.999665e-01,&
     & 9.999652e-01,9.999639e-01,9.999626e-01,9.999613e-01,9.999600e-01,&
     & 9.999587e-01,9.999574e-01,9.999561e-01,9.999548e-01,9.999535e-01,&
     & 9.999522e-01,9.999509e-01,9.999496e-01,9.999483e-01,9.999470e-01,&
     & 9.999457e-01,9.999444e-01,9.999431e-01,9.999418e-01,9.999405e-01,&
     & 9.999392e-01 /
      data  ssaice3(:, 25)   /                                          &
     & 9.999994e-01,9.999993e-01,9.999991e-01,9.999990e-01,9.999989e-01,&
     & 9.999987e-01,9.999986e-01,9.999984e-01,9.999983e-01,9.999982e-01,&
     & 9.999980e-01,9.999979e-01,9.999977e-01,9.999976e-01,9.999975e-01,&
     & 9.999973e-01,9.999972e-01,9.999970e-01,9.999969e-01,9.999967e-01,&
     & 9.999966e-01,9.999965e-01,9.999963e-01,9.999962e-01,9.999960e-01,&
     & 9.999959e-01,9.999957e-01,9.999956e-01,9.999954e-01,9.999953e-01,&
     & 9.999952e-01,9.999950e-01,9.999949e-01,9.999947e-01,9.999946e-01,&
     & 9.999944e-01,9.999943e-01,9.999941e-01,9.999940e-01,9.999939e-01,&
     & 9.999937e-01,9.999936e-01,9.999934e-01,9.999933e-01,9.999931e-01,&
     & 9.999930e-01 /
      data  ssaice3(:, 26)   /                                          &
     & 9.999997e-01,9.999995e-01,9.999992e-01,9.999990e-01,9.999987e-01,&
     & 9.999985e-01,9.999983e-01,9.999980e-01,9.999978e-01,9.999976e-01,&
     & 9.999973e-01,9.999971e-01,9.999969e-01,9.999967e-01,9.999965e-01,&
     & 9.999963e-01,9.999960e-01,9.999958e-01,9.999956e-01,9.999954e-01,&
     & 9.999952e-01,9.999950e-01,9.999948e-01,9.999946e-01,9.999944e-01,&
     & 9.999942e-01,9.999939e-01,9.999937e-01,9.999935e-01,9.999933e-01,&
     & 9.999931e-01,9.999929e-01,9.999927e-01,9.999925e-01,9.999923e-01,&
     & 9.999920e-01,9.999918e-01,9.999916e-01,9.999914e-01,9.999911e-01,&
     & 9.999909e-01,9.999907e-01,9.999905e-01,9.999902e-01,9.999900e-01,&
     & 9.999897e-01 /
      data  ssaice3(:, 27)   /                                          &
     & 9.999991e-01,9.999985e-01,9.999980e-01,9.999974e-01,9.999968e-01,&
     & 9.999963e-01,9.999957e-01,9.999951e-01,9.999946e-01,9.999940e-01,&
     & 9.999934e-01,9.999929e-01,9.999923e-01,9.999918e-01,9.999912e-01,&
     & 9.999907e-01,9.999901e-01,9.999896e-01,9.999891e-01,9.999885e-01,&
     & 9.999880e-01,9.999874e-01,9.999869e-01,9.999863e-01,9.999858e-01,&
     & 9.999853e-01,9.999847e-01,9.999842e-01,9.999836e-01,9.999831e-01,&
     & 9.999826e-01,9.999820e-01,9.999815e-01,9.999809e-01,9.999804e-01,&
     & 9.999798e-01,9.999793e-01,9.999787e-01,9.999782e-01,9.999776e-01,&
     & 9.999770e-01,9.999765e-01,9.999759e-01,9.999754e-01,9.999748e-01,&
     & 9.999742e-01 /
      data  ssaice3(:, 28)   /                                          &
     & 9.999975e-01,9.999961e-01,9.999946e-01,9.999931e-01,9.999917e-01,&
     & 9.999903e-01,9.999888e-01,9.999874e-01,9.999859e-01,9.999845e-01,&
     & 9.999831e-01,9.999816e-01,9.999802e-01,9.999788e-01,9.999774e-01,&
     & 9.999759e-01,9.999745e-01,9.999731e-01,9.999717e-01,9.999702e-01,&
     & 9.999688e-01,9.999674e-01,9.999660e-01,9.999646e-01,9.999631e-01,&
     & 9.999617e-01,9.999603e-01,9.999589e-01,9.999574e-01,9.999560e-01,&
     & 9.999546e-01,9.999532e-01,9.999517e-01,9.999503e-01,9.999489e-01,&
     & 9.999474e-01,9.999460e-01,9.999446e-01,9.999431e-01,9.999417e-01,&
     & 9.999403e-01,9.999388e-01,9.999374e-01,9.999359e-01,9.999345e-01,&
     & 9.999330e-01 /
      data  ssaice3(:, 29)   /                                          &
     & 4.526500e-01,5.287890e-01,5.410487e-01,5.459865e-01,5.485149e-01,&
     & 5.498914e-01,5.505895e-01,5.508310e-01,5.507364e-01,5.503793e-01,&
     & 5.498090e-01,5.490612e-01,5.481637e-01,5.471395e-01,5.460083e-01,&
     & 5.447878e-01,5.434946e-01,5.421442e-01,5.407514e-01,5.393309e-01,&
     & 5.378970e-01,5.364641e-01,5.350464e-01,5.336582e-01,5.323140e-01,&
     & 5.310283e-01,5.298158e-01,5.286914e-01,5.276704e-01,5.267680e-01,&
     & 5.260000e-01,5.253823e-01,5.249311e-01,5.246629e-01,5.245946e-01,&
     & 5.247434e-01,5.251268e-01,5.257626e-01,5.266693e-01,5.278653e-01,&
     & 5.293698e-01,5.312022e-01,5.333823e-01,5.359305e-01,5.388676e-01,&
     & 5.422146e-01 /

!  --- ...  asymmetry factor from fu, unitless
      data  asyice3(:, 16)   /                                          &
     & 8.340752e-01,8.435170e-01,8.517487e-01,8.592064e-01,8.660387e-01,&
     & 8.723204e-01,8.780997e-01,8.834137e-01,8.882934e-01,8.927662e-01,&
     & 8.968577e-01,9.005914e-01,9.039899e-01,9.070745e-01,9.098659e-01,&
     & 9.123836e-01,9.146466e-01,9.166734e-01,9.184817e-01,9.200886e-01,&
     & 9.215109e-01,9.227648e-01,9.238661e-01,9.248304e-01,9.256727e-01,&
     & 9.264078e-01,9.270505e-01,9.276150e-01,9.281156e-01,9.285662e-01,&
     & 9.289806e-01,9.293726e-01,9.297557e-01,9.301435e-01,9.305491e-01,&
     & 9.309859e-01,9.314671e-01,9.320055e-01,9.326140e-01,9.333053e-01,&
     & 9.340919e-01,9.349861e-01,9.360000e-01,9.371451e-01,9.384329e-01,&
     & 9.398744e-01 /
      data  asyice3(:, 17)   /                                          &
     & 8.728160e-01,8.777333e-01,8.823754e-01,8.867535e-01,8.908785e-01,&
     & 8.947611e-01,8.984118e-01,9.018408e-01,9.050582e-01,9.080739e-01,&
     & 9.108976e-01,9.135388e-01,9.160068e-01,9.183106e-01,9.204595e-01,&
     & 9.224620e-01,9.243271e-01,9.260632e-01,9.276788e-01,9.291822e-01,&
     & 9.305817e-01,9.318853e-01,9.331012e-01,9.342372e-01,9.353013e-01,&
     & 9.363013e-01,9.372450e-01,9.381400e-01,9.389939e-01,9.398145e-01,&
     & 9.406092e-01,9.413856e-01,9.421511e-01,9.429131e-01,9.436790e-01,&
     & 9.444561e-01,9.452517e-01,9.460729e-01,9.469270e-01,9.478209e-01,&
     & 9.487617e-01,9.497562e-01,9.508112e-01,9.519335e-01,9.531294e-01,&
     & 9.544055e-01 /
      data  asyice3(:, 18)   /                                          &
     & 7.897566e-01,7.948704e-01,7.998041e-01,8.045623e-01,8.091495e-01,&
     & 8.135702e-01,8.178290e-01,8.219305e-01,8.258790e-01,8.296792e-01,&
     & 8.333355e-01,8.368524e-01,8.402343e-01,8.434856e-01,8.466108e-01,&
     & 8.496143e-01,8.525004e-01,8.552737e-01,8.579384e-01,8.604990e-01,&
     & 8.629597e-01,8.653250e-01,8.675992e-01,8.697867e-01,8.718916e-01,&
     & 8.739185e-01,8.758715e-01,8.777551e-01,8.795734e-01,8.813308e-01,&
     & 8.830315e-01,8.846799e-01,8.862802e-01,8.878366e-01,8.893534e-01,&
     & 8.908350e-01,8.922854e-01,8.937090e-01,8.951099e-01,8.964925e-01,&
     & 8.978609e-01,8.992192e-01,9.005718e-01,9.019229e-01,9.032765e-01,&
     & 9.046369e-01 /
      data  asyice3(:, 19)   /                                          &
     & 7.812615e-01,7.887764e-01,7.959664e-01,8.028413e-01,8.094109e-01,&
     & 8.156849e-01,8.216730e-01,8.273846e-01,8.328294e-01,8.380166e-01,&
     & 8.429556e-01,8.476556e-01,8.521258e-01,8.563753e-01,8.604131e-01,&
     & 8.642481e-01,8.678893e-01,8.713455e-01,8.746254e-01,8.777378e-01,&
     & 8.806914e-01,8.834948e-01,8.861566e-01,8.886854e-01,8.910897e-01,&
     & 8.933779e-01,8.955586e-01,8.976402e-01,8.996311e-01,9.015398e-01,&
     & 9.033745e-01,9.051436e-01,9.068555e-01,9.085185e-01,9.101410e-01,&
     & 9.117311e-01,9.132972e-01,9.148476e-01,9.163905e-01,9.179340e-01,&
     & 9.194864e-01,9.210559e-01,9.226505e-01,9.242784e-01,9.259476e-01,&
     & 9.276661e-01 /
      data  asyice3(:, 20)   /                                          &
     & 7.640720e-01,7.691119e-01,7.739941e-01,7.787222e-01,7.832998e-01,&
     & 7.877304e-01,7.920177e-01,7.961652e-01,8.001765e-01,8.040551e-01,&
     & 8.078044e-01,8.114280e-01,8.149294e-01,8.183119e-01,8.215791e-01,&
     & 8.247344e-01,8.277812e-01,8.307229e-01,8.335629e-01,8.363046e-01,&
     & 8.389514e-01,8.415067e-01,8.439738e-01,8.463560e-01,8.486568e-01,&
     & 8.508795e-01,8.530274e-01,8.551039e-01,8.571122e-01,8.590558e-01,&
     & 8.609378e-01,8.627618e-01,8.645309e-01,8.662485e-01,8.679178e-01,&
     & 8.695423e-01,8.711251e-01,8.726697e-01,8.741792e-01,8.756571e-01,&
     & 8.771065e-01,8.785307e-01,8.799331e-01,8.813169e-01,8.826854e-01,&
     & 8.840419e-01 /
      data  asyice3(:, 21)   /                                          &
     & 7.602598e-01,7.651572e-01,7.699014e-01,7.744962e-01,7.789452e-01,&
     & 7.832522e-01,7.874205e-01,7.914538e-01,7.953555e-01,7.991290e-01,&
     & 8.027777e-01,8.063049e-01,8.097140e-01,8.130081e-01,8.161906e-01,&
     & 8.192645e-01,8.222331e-01,8.250993e-01,8.278664e-01,8.305374e-01,&
     & 8.331153e-01,8.356030e-01,8.380037e-01,8.403201e-01,8.425553e-01,&
     & 8.447121e-01,8.467935e-01,8.488022e-01,8.507412e-01,8.526132e-01,&
     & 8.544210e-01,8.561675e-01,8.578554e-01,8.594875e-01,8.610665e-01,&
     & 8.625951e-01,8.640760e-01,8.655119e-01,8.669055e-01,8.682594e-01,&
     & 8.695763e-01,8.708587e-01,8.721094e-01,8.733308e-01,8.745255e-01,&
     & 8.756961e-01 /
      data  asyice3(:, 22)   /                                          &
     & 7.568957e-01,7.606995e-01,7.644072e-01,7.680204e-01,7.715402e-01,&
     & 7.749682e-01,7.783057e-01,7.815541e-01,7.847148e-01,7.877892e-01,&
     & 7.907786e-01,7.936846e-01,7.965084e-01,7.992515e-01,8.019153e-01,&
     & 8.045011e-01,8.070103e-01,8.094444e-01,8.118048e-01,8.140927e-01,&
     & 8.163097e-01,8.184571e-01,8.205364e-01,8.225488e-01,8.244958e-01,&
     & 8.263789e-01,8.281993e-01,8.299586e-01,8.316580e-01,8.332991e-01,&
     & 8.348831e-01,8.364115e-01,8.378857e-01,8.393071e-01,8.406770e-01,&
     & 8.419969e-01,8.432682e-01,8.444923e-01,8.456706e-01,8.468044e-01,&
     & 8.478952e-01,8.489444e-01,8.499533e-01,8.509234e-01,8.518561e-01,&
     & 8.527528e-01 /
      data  asyice3(:, 23)   /                                          &
     & 7.575066e-01,7.606912e-01,7.638236e-01,7.669035e-01,7.699306e-01,&
     & 7.729046e-01,7.758254e-01,7.786926e-01,7.815060e-01,7.842654e-01,&
     & 7.869705e-01,7.896211e-01,7.922168e-01,7.947574e-01,7.972428e-01,&
     & 7.996726e-01,8.020466e-01,8.043646e-01,8.066262e-01,8.088313e-01,&
     & 8.109796e-01,8.130709e-01,8.151049e-01,8.170814e-01,8.190001e-01,&
     & 8.208608e-01,8.226632e-01,8.244071e-01,8.260924e-01,8.277186e-01,&
     & 8.292856e-01,8.307932e-01,8.322411e-01,8.336291e-01,8.349570e-01,&
     & 8.362244e-01,8.374312e-01,8.385772e-01,8.396621e-01,8.406856e-01,&
     & 8.416476e-01,8.425479e-01,8.433861e-01,8.441620e-01,8.448755e-01,&
     & 8.455263e-01 /
      data  asyice3(:, 24)   /                                          &
     & 7.568829e-01,7.597947e-01,7.626745e-01,7.655212e-01,7.683337e-01,&
     & 7.711111e-01,7.738523e-01,7.765565e-01,7.792225e-01,7.818494e-01,&
     & 7.844362e-01,7.869819e-01,7.894854e-01,7.919459e-01,7.943623e-01,&
     & 7.967337e-01,7.990590e-01,8.013373e-01,8.035676e-01,8.057488e-01,&
     & 8.078802e-01,8.099605e-01,8.119890e-01,8.139645e-01,8.158862e-01,&
     & 8.177530e-01,8.195641e-01,8.213183e-01,8.230149e-01,8.246527e-01,&
     & 8.262308e-01,8.277483e-01,8.292042e-01,8.305976e-01,8.319275e-01,&
     & 8.331929e-01,8.343929e-01,8.355265e-01,8.365928e-01,8.375909e-01,&
     & 8.385197e-01,8.393784e-01,8.401659e-01,8.408815e-01,8.415240e-01,&
     & 8.420926e-01 /
      data  asyice3(:, 25)   /                                          &
     & 7.548616e-01,7.575454e-01,7.602153e-01,7.628696e-01,7.655067e-01,&
     & 7.681249e-01,7.707225e-01,7.732978e-01,7.758492e-01,7.783750e-01,&
     & 7.808735e-01,7.833430e-01,7.857819e-01,7.881886e-01,7.905612e-01,&
     & 7.928983e-01,7.951980e-01,7.974588e-01,7.996789e-01,8.018567e-01,&
     & 8.039905e-01,8.060787e-01,8.081196e-01,8.101115e-01,8.120527e-01,&
     & 8.139416e-01,8.157764e-01,8.175557e-01,8.192776e-01,8.209405e-01,&
     & 8.225427e-01,8.240826e-01,8.255585e-01,8.269688e-01,8.283117e-01,&
     & 8.295856e-01,8.307889e-01,8.319198e-01,8.329767e-01,8.339579e-01,&
     & 8.348619e-01,8.356868e-01,8.364311e-01,8.370930e-01,8.376710e-01,&
     & 8.381633e-01 /
      data  asyice3(:, 26)   /                                          &
     & 7.491854e-01,7.518523e-01,7.545089e-01,7.571534e-01,7.597839e-01,&
     & 7.623987e-01,7.649959e-01,7.675737e-01,7.701303e-01,7.726639e-01,&
     & 7.751727e-01,7.776548e-01,7.801084e-01,7.825318e-01,7.849230e-01,&
     & 7.872804e-01,7.896020e-01,7.918862e-01,7.941309e-01,7.963345e-01,&
     & 7.984951e-01,8.006109e-01,8.026802e-01,8.047009e-01,8.066715e-01,&
     & 8.085900e-01,8.104546e-01,8.122636e-01,8.140150e-01,8.157072e-01,&
     & 8.173382e-01,8.189063e-01,8.204096e-01,8.218464e-01,8.232148e-01,&
     & 8.245130e-01,8.257391e-01,8.268915e-01,8.279682e-01,8.289675e-01,&
     & 8.298875e-01,8.307264e-01,8.314824e-01,8.321537e-01,8.327385e-01,&
     & 8.332350e-01 /
      data  asyice3(:, 27)   /                                          &
     & 7.397086e-01,7.424069e-01,7.450955e-01,7.477725e-01,7.504362e-01,&
     & 7.530846e-01,7.557159e-01,7.583283e-01,7.609199e-01,7.634888e-01,&
     & 7.660332e-01,7.685512e-01,7.710411e-01,7.735009e-01,7.759288e-01,&
     & 7.783229e-01,7.806814e-01,7.830024e-01,7.852841e-01,7.875246e-01,&
     & 7.897221e-01,7.918748e-01,7.939807e-01,7.960380e-01,7.980449e-01,&
     & 7.999995e-01,8.019000e-01,8.037445e-01,8.055311e-01,8.072581e-01,&
     & 8.089235e-01,8.105255e-01,8.120623e-01,8.135319e-01,8.149326e-01,&
     & 8.162626e-01,8.175198e-01,8.187025e-01,8.198089e-01,8.208371e-01,&
     & 8.217852e-01,8.226514e-01,8.234338e-01,8.241306e-01,8.247399e-01,&
     & 8.252599e-01 /
      data  asyice3(:, 28)   /                                          &
     & 7.224533e-01,7.251681e-01,7.278728e-01,7.305654e-01,7.332444e-01,&
     & 7.359078e-01,7.385539e-01,7.411808e-01,7.437869e-01,7.463702e-01,&
     & 7.489291e-01,7.514616e-01,7.539661e-01,7.564408e-01,7.588837e-01,&
     & 7.612933e-01,7.636676e-01,7.660049e-01,7.683034e-01,7.705612e-01,&
     & 7.727767e-01,7.749480e-01,7.770733e-01,7.791509e-01,7.811789e-01,&
     & 7.831556e-01,7.850791e-01,7.869478e-01,7.887597e-01,7.905131e-01,&
     & 7.922062e-01,7.938372e-01,7.954044e-01,7.969059e-01,7.983399e-01,&
     & 7.997047e-01,8.009985e-01,8.022195e-01,8.033658e-01,8.044357e-01,&
     & 8.054275e-01,8.063392e-01,8.071692e-01,8.079157e-01,8.085768e-01,&
     & 8.091507e-01 /
      data  asyice3(:, 29)   /                                          &
     & 8.850026e-01,9.005489e-01,9.069242e-01,9.121799e-01,9.168987e-01,&
     & 9.212259e-01,9.252176e-01,9.289028e-01,9.323000e-01,9.354235e-01,&
     & 9.382858e-01,9.408985e-01,9.432734e-01,9.454218e-01,9.473557e-01,&
     & 9.490871e-01,9.506282e-01,9.519917e-01,9.531904e-01,9.542374e-01,&
     & 9.551461e-01,9.559298e-01,9.566023e-01,9.571775e-01,9.576692e-01,&
     & 9.580916e-01,9.584589e-01,9.587853e-01,9.590851e-01,9.593729e-01,&
     & 9.596632e-01,9.599705e-01,9.603096e-01,9.606954e-01,9.611427e-01,&
     & 9.616667e-01,9.622826e-01,9.630060e-01,9.638524e-01,9.648379e-01,&
     & 9.659788e-01,9.672916e-01,9.687933e-01,9.705014e-01,9.724337e-01,&
     & 9.746084e-01 /

!  --- ...  fdelta from fu, unitless
      data  fdlice3(:, 16)   /                                          &
     & 4.959277e-02,4.685292e-02,4.426104e-02,4.181231e-02,3.950191e-02,&
     & 3.732500e-02,3.527675e-02,3.335235e-02,3.154697e-02,2.985578e-02,&
     & 2.827395e-02,2.679666e-02,2.541909e-02,2.413640e-02,2.294378e-02,&
     & 2.183639e-02,2.080940e-02,1.985801e-02,1.897736e-02,1.816265e-02,&
     & 1.740905e-02,1.671172e-02,1.606585e-02,1.546661e-02,1.490917e-02,&
     & 1.438870e-02,1.390038e-02,1.343939e-02,1.300089e-02,1.258006e-02,&
     & 1.217208e-02,1.177212e-02,1.137536e-02,1.097696e-02,1.057210e-02,&
     & 1.015596e-02,9.723704e-03,9.270516e-03,8.791565e-03,8.282026e-03,&
     & 7.737072e-03,7.151879e-03,6.521619e-03,5.841467e-03,5.106597e-03,&
     & 4.312183e-03 /
      data  fdlice3(:, 17)   /                                          &
     & 5.071224e-02,5.000217e-02,4.933872e-02,4.871992e-02,4.814380e-02,&
     & 4.760839e-02,4.711170e-02,4.665177e-02,4.622662e-02,4.583426e-02,&
     & 4.547274e-02,4.514007e-02,4.483428e-02,4.455340e-02,4.429544e-02,&
     & 4.405844e-02,4.384041e-02,4.363939e-02,4.345340e-02,4.328047e-02,&
     & 4.311861e-02,4.296586e-02,4.282024e-02,4.267977e-02,4.254248e-02,&
     & 4.240640e-02,4.226955e-02,4.212995e-02,4.198564e-02,4.183462e-02,&
     & 4.167494e-02,4.150462e-02,4.132167e-02,4.112413e-02,4.091003e-02,&
     & 4.067737e-02,4.042420e-02,4.014854e-02,3.984840e-02,3.952183e-02,&
     & 3.916683e-02,3.878144e-02,3.836368e-02,3.791158e-02,3.742316e-02,&
     & 3.689645e-02 /
      data  fdlice3(:, 18)   /                                          &
     & 1.062938e-01,1.065234e-01,1.067822e-01,1.070682e-01,1.073793e-01,&
     & 1.077137e-01,1.080693e-01,1.084442e-01,1.088364e-01,1.092439e-01,&
     & 1.096647e-01,1.100970e-01,1.105387e-01,1.109878e-01,1.114423e-01,&
     & 1.119004e-01,1.123599e-01,1.128190e-01,1.132757e-01,1.137279e-01,&
     & 1.141738e-01,1.146113e-01,1.150385e-01,1.154534e-01,1.158540e-01,&
     & 1.162383e-01,1.166045e-01,1.169504e-01,1.172741e-01,1.175738e-01,&
     & 1.178472e-01,1.180926e-01,1.183080e-01,1.184913e-01,1.186405e-01,&
     & 1.187538e-01,1.188291e-01,1.188645e-01,1.188580e-01,1.188076e-01,&
     & 1.187113e-01,1.185672e-01,1.183733e-01,1.181277e-01,1.178282e-01,&
     & 1.174731e-01 /
      data  fdlice3(:, 19)   /                                          &
     & 1.076195e-01,1.065195e-01,1.054696e-01,1.044673e-01,1.035099e-01,&
     & 1.025951e-01,1.017203e-01,1.008831e-01,1.000808e-01,9.931116e-02,&
     & 9.857151e-02,9.785939e-02,9.717230e-02,9.650774e-02,9.586322e-02,&
     & 9.523623e-02,9.462427e-02,9.402484e-02,9.343544e-02,9.285358e-02,&
     & 9.227675e-02,9.170245e-02,9.112818e-02,9.055144e-02,8.996974e-02,&
     & 8.938056e-02,8.878142e-02,8.816981e-02,8.754323e-02,8.689919e-02,&
     & 8.623517e-02,8.554869e-02,8.483724e-02,8.409832e-02,8.332943e-02,&
     & 8.252807e-02,8.169175e-02,8.081795e-02,7.990419e-02,7.894796e-02,&
     & 7.794676e-02,7.689809e-02,7.579945e-02,7.464834e-02,7.344227e-02,&
     & 7.217872e-02 /
      data  fdlice3(:, 20)   /                                          &
     & 1.119014e-01,1.122706e-01,1.126690e-01,1.130947e-01,1.135456e-01,&
     & 1.140199e-01,1.145154e-01,1.150302e-01,1.155623e-01,1.161096e-01,&
     & 1.166703e-01,1.172422e-01,1.178233e-01,1.184118e-01,1.190055e-01,&
     & 1.196025e-01,1.202008e-01,1.207983e-01,1.213931e-01,1.219832e-01,&
     & 1.225665e-01,1.231411e-01,1.237050e-01,1.242561e-01,1.247926e-01,&
     & 1.253122e-01,1.258132e-01,1.262934e-01,1.267509e-01,1.271836e-01,&
     & 1.275896e-01,1.279669e-01,1.283134e-01,1.286272e-01,1.289063e-01,&
     & 1.291486e-01,1.293522e-01,1.295150e-01,1.296351e-01,1.297104e-01,&
     & 1.297390e-01,1.297189e-01,1.296480e-01,1.295244e-01,1.293460e-01,&
     & 1.291109e-01 /
      data  fdlice3(:, 21)   /                                          &
     & 1.133298e-01,1.136777e-01,1.140556e-01,1.144615e-01,1.148934e-01,&
     & 1.153492e-01,1.158269e-01,1.163243e-01,1.168396e-01,1.173706e-01,&
     & 1.179152e-01,1.184715e-01,1.190374e-01,1.196108e-01,1.201897e-01,&
     & 1.207720e-01,1.213558e-01,1.219389e-01,1.225194e-01,1.230951e-01,&
     & 1.236640e-01,1.242241e-01,1.247733e-01,1.253096e-01,1.258309e-01,&
     & 1.263352e-01,1.268205e-01,1.272847e-01,1.277257e-01,1.281415e-01,&
     & 1.285300e-01,1.288893e-01,1.292173e-01,1.295118e-01,1.297710e-01,&
     & 1.299927e-01,1.301748e-01,1.303154e-01,1.304124e-01,1.304637e-01,&
     & 1.304673e-01,1.304212e-01,1.303233e-01,1.301715e-01,1.299638e-01,&
     & 1.296983e-01 /
      data  fdlice3(:, 22)   /                                          &
     & 1.145360e-01,1.153256e-01,1.161453e-01,1.169929e-01,1.178666e-01,&
     & 1.187641e-01,1.196835e-01,1.206227e-01,1.215796e-01,1.225522e-01,&
     & 1.235383e-01,1.245361e-01,1.255433e-01,1.265579e-01,1.275779e-01,&
     & 1.286011e-01,1.296257e-01,1.306494e-01,1.316703e-01,1.326862e-01,&
     & 1.336951e-01,1.346950e-01,1.356838e-01,1.366594e-01,1.376198e-01,&
     & 1.385629e-01,1.394866e-01,1.403889e-01,1.412678e-01,1.421212e-01,&
     & 1.429469e-01,1.437430e-01,1.445074e-01,1.452381e-01,1.459329e-01,&
     & 1.465899e-01,1.472069e-01,1.477819e-01,1.483128e-01,1.487976e-01,&
     & 1.492343e-01,1.496207e-01,1.499548e-01,1.502346e-01,1.504579e-01,&
     & 1.506227e-01 /
      data  fdlice3(:, 23)   /                                          &
     & 1.153263e-01,1.161445e-01,1.169932e-01,1.178703e-01,1.187738e-01,&
     & 1.197016e-01,1.206516e-01,1.216217e-01,1.226099e-01,1.236141e-01,&
     & 1.246322e-01,1.256621e-01,1.267017e-01,1.277491e-01,1.288020e-01,&
     & 1.298584e-01,1.309163e-01,1.319736e-01,1.330281e-01,1.340778e-01,&
     & 1.351207e-01,1.361546e-01,1.371775e-01,1.381873e-01,1.391820e-01,&
     & 1.401593e-01,1.411174e-01,1.420540e-01,1.429671e-01,1.438547e-01,&
     & 1.447146e-01,1.455449e-01,1.463433e-01,1.471078e-01,1.478364e-01,&
     & 1.485270e-01,1.491774e-01,1.497857e-01,1.503497e-01,1.508674e-01,&
     & 1.513367e-01,1.517554e-01,1.521216e-01,1.524332e-01,1.526880e-01,&
     & 1.528840e-01 /
      data  fdlice3(:, 24)   /                                          &
     & 1.160842e-01,1.169118e-01,1.177697e-01,1.186556e-01,1.195676e-01,&
     & 1.205036e-01,1.214616e-01,1.224394e-01,1.234349e-01,1.244463e-01,&
     & 1.254712e-01,1.265078e-01,1.275539e-01,1.286075e-01,1.296664e-01,&
     & 1.307287e-01,1.317923e-01,1.328550e-01,1.339149e-01,1.349699e-01,&
     & 1.360179e-01,1.370567e-01,1.380845e-01,1.390991e-01,1.400984e-01,&
     & 1.410803e-01,1.420429e-01,1.429840e-01,1.439016e-01,1.447936e-01,&
     & 1.456579e-01,1.464925e-01,1.472953e-01,1.480642e-01,1.487972e-01,&
     & 1.494923e-01,1.501472e-01,1.507601e-01,1.513287e-01,1.518511e-01,&
     & 1.523252e-01,1.527489e-01,1.531201e-01,1.534368e-01,1.536969e-01,&
     & 1.538984e-01 /
      data  fdlice3(:, 25)   /                                          &
     & 1.168725e-01,1.177088e-01,1.185747e-01,1.194680e-01,1.203867e-01,&
     & 1.213288e-01,1.222923e-01,1.232750e-01,1.242750e-01,1.252903e-01,&
     & 1.263187e-01,1.273583e-01,1.284069e-01,1.294626e-01,1.305233e-01,&
     & 1.315870e-01,1.326517e-01,1.337152e-01,1.347756e-01,1.358308e-01,&
     & 1.368788e-01,1.379175e-01,1.389449e-01,1.399590e-01,1.409577e-01,&
     & 1.419389e-01,1.429007e-01,1.438410e-01,1.447577e-01,1.456488e-01,&
     & 1.465123e-01,1.473461e-01,1.481483e-01,1.489166e-01,1.496492e-01,&
     & 1.503439e-01,1.509988e-01,1.516118e-01,1.521808e-01,1.527038e-01,&
     & 1.531788e-01,1.536037e-01,1.539764e-01,1.542951e-01,1.545575e-01,&
     & 1.547617e-01 /
      data  fdlice3(:, 26)   /                                          &
     & 1.180509e-01,1.189025e-01,1.197820e-01,1.206875e-01,1.216171e-01,&
     & 1.225687e-01,1.235404e-01,1.245303e-01,1.255363e-01,1.265564e-01,&
     & 1.275888e-01,1.286313e-01,1.296821e-01,1.307392e-01,1.318006e-01,&
     & 1.328643e-01,1.339284e-01,1.349908e-01,1.360497e-01,1.371029e-01,&
     & 1.381486e-01,1.391848e-01,1.402095e-01,1.412208e-01,1.422165e-01,&
     & 1.431949e-01,1.441539e-01,1.450915e-01,1.460058e-01,1.468947e-01,&
     & 1.477564e-01,1.485888e-01,1.493900e-01,1.501580e-01,1.508907e-01,&
     & 1.515864e-01,1.522428e-01,1.528582e-01,1.534305e-01,1.539578e-01,&
     & 1.544380e-01,1.548692e-01,1.552494e-01,1.555767e-01,1.558490e-01,&
     & 1.560645e-01 /
      data  fdlice3(:, 27)   /                                          &
     & 1.200480e-01,1.209267e-01,1.218304e-01,1.227575e-01,1.237059e-01,&
     & 1.246739e-01,1.256595e-01,1.266610e-01,1.276765e-01,1.287041e-01,&
     & 1.297420e-01,1.307883e-01,1.318412e-01,1.328988e-01,1.339593e-01,&
     & 1.350207e-01,1.360813e-01,1.371393e-01,1.381926e-01,1.392396e-01,&
     & 1.402783e-01,1.413069e-01,1.423235e-01,1.433263e-01,1.443134e-01,&
     & 1.452830e-01,1.462332e-01,1.471622e-01,1.480681e-01,1.489490e-01,&
     & 1.498032e-01,1.506286e-01,1.514236e-01,1.521863e-01,1.529147e-01,&
     & 1.536070e-01,1.542614e-01,1.548761e-01,1.554491e-01,1.559787e-01,&
     & 1.564629e-01,1.568999e-01,1.572879e-01,1.576249e-01,1.579093e-01,&
     & 1.581390e-01 /
      data  fdlice3(:, 28)   /                                          &
     & 1.247813e-01,1.256496e-01,1.265417e-01,1.274560e-01,1.283905e-01,&
     & 1.293436e-01,1.303135e-01,1.312983e-01,1.322964e-01,1.333060e-01,&
     & 1.343252e-01,1.353523e-01,1.363855e-01,1.374231e-01,1.384632e-01,&
     & 1.395042e-01,1.405441e-01,1.415813e-01,1.426140e-01,1.436404e-01,&
     & 1.446587e-01,1.456672e-01,1.466640e-01,1.476475e-01,1.486157e-01,&
     & 1.495671e-01,1.504997e-01,1.514117e-01,1.523016e-01,1.531673e-01,&
     & 1.540073e-01,1.548197e-01,1.556026e-01,1.563545e-01,1.570734e-01,&
     & 1.577576e-01,1.584054e-01,1.590149e-01,1.595843e-01,1.601120e-01,&
     & 1.605962e-01,1.610349e-01,1.614266e-01,1.617693e-01,1.620614e-01,&
     & 1.623011e-01 /
      data  fdlice3(:, 29)   /                                          &
     & 1.006055e-01,9.549582e-02,9.063960e-02,8.602900e-02,8.165612e-02,&
     & 7.751308e-02,7.359199e-02,6.988496e-02,6.638412e-02,6.308156e-02,&
     & 5.996942e-02,5.703979e-02,5.428481e-02,5.169657e-02,4.926719e-02,&
     & 4.698880e-02,4.485349e-02,4.285339e-02,4.098061e-02,3.922727e-02,&
     & 3.758547e-02,3.604733e-02,3.460497e-02,3.325051e-02,3.197604e-02,&
     & 3.077369e-02,2.963558e-02,2.855381e-02,2.752050e-02,2.652776e-02,&
     & 2.556772e-02,2.463247e-02,2.371415e-02,2.280485e-02,2.189670e-02,&
     & 2.098180e-02,2.005228e-02,1.910024e-02,1.811781e-02,1.709709e-02,&
     & 1.603020e-02,1.490925e-02,1.372635e-02,1.247363e-02,1.114319e-02,&
     & 9.727157e-03 /

!> \name coefficients to compute tau, ssa, asy for rain drop and snowflake 
!!@{
      real (kind=kind_phys), public  :: a0r, a1r, a0s, a1s
      data a0r,a1r / 3.07e-3, 0.0 /,    a0s,a1s / 0.0,     1.5 /  ! fu's coeff

      real (kind=kind_phys), dimension(nblow:nbhgh), public ::          &
     &       b0r, b0s, b1s, c0r, c0s
      data b0r  / 0.466, 0.437, 0.416, 0.391, 0.374, 0.352, 0.183,      &
     &            0.048, 0.012, 0.000, 0.000, 0.000, 0.000, 0.496 /
      data c0r  / 0.975, 0.965, 0.960, 0.955, 0.952, 0.950, 0.944,      &
     &            0.894, 0.884, 0.883, 0.883, 0.883, 0.883, 0.980 /
      data b0s  / 7*0.460, 2*0.000,   4*0.000, 0.460 /
      data b1s  / 7*0.000, 2*1.62e-5, 4*0.000, 0.000 /
      data c0s  / 7*0.970, 2*0.970,   4*0.700, 0.970 /
!!@}


!........................................!
      end module module_radsw_cldprtb    !
!========================================!

!> \ingroup module_radlw_main
!> This module contains spectral distribution of solar radiation flux 
!! used to obtain the incoming solar flux at toa.
!========================================!
      module module_radsw_sflux          !
!........................................!
!
! ***********************  module descriptions  ********************** !
!                                                                      !
!     this module contains spectral distribution of solar radiation    !
!     flux used to obtain the incoming solar flux at toa.              !
!                                                                      !
!     modify history:                                                  !
!      originally by j.delamere, atmospheric & environmental research  !
!                          in 14 bands kgb data table.                 !
!      feb. 18, 2004   --  yu-tai hou    move the reference data table !
!                          to a new data module.                       !
!                                                                      !
! *************************  end description  ************************ !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NGMAX, NG16, NG17, NG18, NG19,&
     &                                    NG20, NG21, NG22, NG23, NG24, &
     &                                    NG25, NG26, NG27, NG28, NG29, &
     &                                    nblow, nbhgh
!
      implicit none
!
      public
!
      integer, parameter :: MFS01 = 1      !
      integer, parameter :: MFS02 = 5      !
      integer, parameter :: MFS03 = 9      !
      integer, parameter :: MFB01 = 7      !
      integer, parameter :: MFB02 = 2      !
      integer, parameter :: MFB03 = 5      !
!
      real (kind=kind_phys), dimension(nblow:nbhgh), public ::          &
     &       strrat, specwt

!  ---  original strrat
!     data  strrat  / 2.52131e+2, 3.64641e-1, 3.89589e+1, 5.49281e+0,   &
!    &    0.00000e+0, 4.53210e-3, 2.27080e-2, 0.00000e+0, 1.24692e-1,   &
!    &    0.00000e+0, 0.00000e+0, 0.00000e+0, 6.67029e-7, 0.00000e+0 /
!  ---  strrat(22) has been multified by factor o2adj=1.6
      data  strrat  / 2.52131e+2, 3.64641e-1, 3.89589e+1, 5.49281e+0,   &
     &    0.00000e+0, 4.53210e-3, 3.63328e-2, 0.00000e+0, 1.24692e-1,   &
     &    0.00000e+0, 0.00000e+0, 0.00000e+0, 6.67029e-7, 0.00000e+0 /

      data  specwt  / 8.,4.,8.,8.,0.,8.,8.,0.,8.,0.,0.,0.,4.,0. /
!
      integer, dimension(nblow:nbhgh), public :: layreffr, ix1, ix2, ibx

      data  layreffr/ 18,30, 6, 3, 3, 8, 2, 6, 1, 2, 0,32,58,49 /
      data  ix1     /  1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 0, 0, 3, 0 /
      data  ix2     /  5, 2, 5, 2, 0, 2, 6, 0, 6, 0, 0, 0, 6, 0 /
      data  ibx     /  1, 1, 1, 2, 2, 3, 4, 3, 5, 4, 5, 6, 2, 7 /

      real (kind=kind_phys), parameter, public :: scalekur=50.15/48.37
!
      real (kind=kind_phys), target, public ::                          &
     &       sfluxref01(NGMAX,MFS01,MFB01),                             &
     &       sfluxref02(NGMAX,MFS02,MFB02),                             &
     &       sfluxref03(NGMAX,MFS03,MFB03)

!  ---  setup solar sfluxref01
!  ...  band 16, NG16=6
      data   sfluxref01(:,:,1)  /                          .36511300e+1,&
     & .32277700e+1,.26913900e+1,.18039990e+1,.68068900e+0,.54590270e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ...  band 20, NG20=10
      data   sfluxref01(:,:,2)  /                          .93408100e+1,&
     & .89372000e+1,.81934600e+1,.73919600e+1,.61212700e+1,.52395600e+1,&
     & .42494100e+1,.32001300e+1,.23949790e+1,.55783362e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ...  band 23, NG23=10
      data   sfluxref01(:,:,3)  /                          .10462440e+3,&
     & .94796000e+2,.40829400e+2,.35180100e+2,.28694700e+2,.21575100e+2,&
     & .14638800e+2,.15911100e+1,.23587800e+1,.14541238e+1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ...  band 25, NG25=6
      data   sfluxref01(:,:,4)  /                          .42685800e+2,&
     & .45772000e+2,.90953400e+2,.88238400e+2,.75011180e+2,.45315347e+1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ...  band 26, NG26=6
      data   sfluxref01(:,:,5)  /                          .29007900e+2,&
     & .28408800e+2,.33338200e+2,.21820300e+2,.15982813e+2,.93700512e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!     the following (NG27) values were obtained using the "low resolution"
!     version of the kurucz solar source function.  for unknown reasons,
!     the total irradiance in this band differs from the corresponding
!     total in the "high-resolution" version of the Kurucz function.
!     therefore, below these values are scaled by the factor scalekur.

!  ...  band 27, NG27=8
      data   sfluxref01(:,:,6)  /                          .14052600e+2,&
     & .11479400e+2,.87259000e+1,.55696600e+1,.38092700e+1,.15769000e+1,&
     & .29680079e+1,.19043253e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ...  band 29, NG29=12
      data   sfluxref01(:,:,7)  /                          .13288000e+1,&
     & .21401800e+1,.19761200e+1,.17900000e+1,.27421900e+1,.18615160e+1,&
     & .83448990e+0,.12731000e+0,.44698800e-1,.30744100e-1,.11672800e-1,&
     & .16557300e-2,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ---  setup solar sfluxref02
!  ...  band 17, NG17=12
      data   sfluxref02(:,:,1)  /                                       &
     & .31561300e+1,.30344900e+1,.29206900e+1,.26387400e+1,.23458100e+1,&
     & .37790500e+1,.12908500e+1,.97039020e+0,.78781300e-1,.10810270e+0,&
     & .29412900e-1,.12635380e-1,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.28314700e+1,.29591900e+1,.29667400e+1,.27767700e+1,&
     & .24682600e+1,.38472400e+1,.13027900e+1,.97894900e+0,.80212200e-1,&
     & .11088140e+0,.29905100e-1,.12675180e-1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.28230000e+1,.29484500e+1,.29588700e+1,&
     & .27759300e+1,.24709600e+1,.38644300e+1,.13079600e+1,.98108860e+0,&
     & .80199600e-1,.11130020e+0,.30051500e-1,.12844740e-1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.28171500e+1,.29378900e+1,&
     & .29509100e+1,.27704600e+1,.24771600e+1,.38795600e+1,.13127700e+1,&
     & .98414460e+0,.80339100e-1,.11153510e+0,.30072000e-1,.13100450e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.28233500e+1,&
     & .29316800e+1,.29145500e+1,.27521300e+1,.24916800e+1,.39013400e+1,&
     & .13240100e+1,.99059630e+0,.80519700e-1,.11150350e+0,.30566700e-1,&
     & .13155950e-1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ...  band 28, NG28=6
      data   sfluxref02(:,:,2)  /                                       &
     & .10615600e+1,.59991000e+0,.82253900e+0,.47011400e+0,.12054496e+0,&
     & .52716294e-2,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.10759800e+1,.58509900e+0,.82292900e+0,.47011400e+0,&
     & .12059320e+0,.52233221e-2,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.46164700e+0,.40611300e+0,.64001400e+0,&
     & .44662400e+0,.10729748e+1,.52565480e-1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.13266900e+0,.17505800e+0,&
     & .74740500e+0,.82625100e+0,.11385304e+1,.60026497e-1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.75480000e-1,&
     & .23224600e+0,.74740500e+0,.77667600e+0,.11821991e+1,.65932601e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0/

!  ---  setup solar sfluxref03
!  ...  band 18, NG18=8
      data   sfluxref03(:,:,1)  /                                       &
     & .36584000e+1,.35437500e+1,.33448100e+1,.31053400e+1,.52272000e+1,&
     & .34812500e+1,.12703318e+1,.98650010e-1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.38637200e+1,.34852100e+1,.33079000e+1,.30810300e+1,&
     & .51827400e+1,.34533000e+1,.12591244e+1,.96720520e-1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.39037000e+1,.35065700e+1,.33062900e+1,&
     & .30604600e+1,.51688900e+1,.34381600e+1,.12512870e+1,.94381020e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.39316500e+1,.35205800e+1,&
     & .33134600e+1,.30494400e+1,.51550700e+1,.34226800e+1,.12456526e+1,&
     & .91203220e-1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.39408200e+1,&
     & .35522100e+1,.33186300e+1,.30473000e+1,.51324600e+1,.34110100e+1,&
     & .12380195e+1,.89290320e-1,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .39419800e+1,.35874300e+1,.33210600e+1,.30586600e+1,.51105400e+1,&
     & .33948500e+1,.12272675e+1,.87955720e-1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.39359600e+1,.36336600e+1,.33314400e+1,.30625200e+1,&
     & .50954600e+1,.33706900e+1,.12121929e+1,.87814620e-1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.39252000e+1,.36907800e+1,.33565600e+1,&
     & .30705500e+1,.50829200e+1,.33242900e+1,.11916105e+1,.87812520e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.38072100e+1,.37443700e+1,&
     & .35020500e+1,.31800900e+1,.50494500e+1,.32007600e+1,.11580073e+1,&
     & .87812920e-1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0 /

!  ...  band 19, NG19=8
      data   sfluxref03(:,:,2)  /                                       &
     & .32579100e+1,.32969700e+1,.31603100e+1,.29611500e+1,.50305700e+1,&
     & .33767800e+1,.12463848e+1,.97623090e-1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.32276900e+1,.32881700e+1,.31668700e+1,.29766200e+1,&
     & .50388700e+1,.33829100e+1,.12493943e+1,.97174500e-1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.32229400e+1,.32778000e+1,.31742400e+1,&
     & .29714300e+1,.50477800e+1,.33835100e+1,.12527049e+1,.97276400e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.32244500e+1,.32611300e+1,&
     & .31843800e+1,.29692100e+1,.50516500e+1,.33840300e+1,.12553459e+1,&
     & .97505500e-1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.32249700e+1,&
     & .32510900e+1,.31874100e+1,.29697000e+1,.50548000e+1,.33852500e+1,&
     & .12565675e+1,.97922400e-1,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .32263200e+1,.32417400e+1,.31852400e+1,.29740200e+1,.50554900e+1,&
     & .33899800e+1,.12562855e+1,.98616200e-1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.32279300e+1,.32358900e+1,.31772000e+1,.29786900e+1,&
     & .50572900e+1,.33942500e+1,.12571685e+1,.99276100e-1,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.32296600e+1,.32408700e+1,.31567600e+1,&
     & .29817100e+1,.50586900e+1,.34020900e+1,.12586610e+1,.99259200e-1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.32724000e+1,.32466600e+1,&
     & .31388600e+1,.29523800e+1,.50465000e+1,.34105900e+1,.12610396e+1,&
     & .99262600e-1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0 /

!  ...  band 21, NG21=10
      data   sfluxref03(:,:,3)  /                                       &
     & .16164300e+2,.15580600e+2,.14725400e+2,.13554100e+2,.11951900e+2,&
     & .10444100e+2,.83788400e+1,.62638400e+1,.47495780e+1,.11187392e+1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.15645100e+2,.15317000e+2,.14698700e+2,.13735000e+2,&
     & .12226700e+2,.10516460e+2,.84715000e+1,.63887300e+1,.48059700e+1,&
     & .11263635e+1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.15609200e+2,.15329300e+2,.14688100e+2,&
     & .13669300e+2,.12234200e+2,.10520100e+2,.84944200e+1,.64213800e+1,&
     & .48319990e+1,.11333414e+1,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.15578600e+2,.15342200e+2,&
     & .14689400e+2,.13604000e+2,.12256700e+2,.10494000e+2,.85352100e+1,&
     & .64442700e+1,.48477890e+1,.11392313e+1,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.15538000e+2,&
     & .15382600e+2,.14657500e+2,.13572200e+2,.12264600e+2,.10476720e+2,&
     & .85715800e+1,.64634300e+1,.48602370e+1,.114454305e+1,.0000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .15512400e+2,.15398600e+2,.14624000e+2,.13553500e+2,.12246800e+2,&
     & .10488910e+2,.86043400e+1,.6479850e+1,.48727470e+1,.115032015e+1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.15491000e+2,.15402800e+2,.14577200e+2,.13550700e+2,&
     & .12212200e+2,.10527350e+2,.86265000e+1,.64964400e+1,.48903570e+1,&
     & .115687915e+1,.0000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.15456200e+2,.15392800e+2,.14551000e+2,&
     & .13512200e+2,.12189000e+2,.10582600e+2,.86584200e+1,.65155800e+1,&
     & .49081390e+1,.116541315e+1,.0000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.15006900e+2,.15147900e+2,&
     & .14780200e+2,.13608500e+2,.12279300e+2,.10692900e+2,.87272300e+1,&
     & .65711400e+1,.49500240e+1,.116747315e+1,.0000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0 /

!  ...  band 22, NG22=2
      data   sfluxref03(:,:,4)  /                                       &
     & .22870690e+2,.142292348e+1,.0000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.22872700e+2,.142092788e+1,.0000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.22877410e+2,.141623238e+1,.0000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.0000000e+0,.22879360e+2,.141425888e+1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.22880890e+2,&
     & .14127222e+1,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .22882260e+2,.141135631e+1,.0000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.22883070e+2,.141054121e+1,.0000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.22884580e+2,.140903474e+1,.0000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.0000000e+0,.22885890e+2,.140772281e+1,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0 /

!  ...  band 24, NG24=8
      data   sfluxref03(:,:,5)  /                                       &
     & .67485000e+2,.60119600e+2,.46277600e+2,.31190000e+2,.10594640e+2,&
     & .15530850e+1,.82783000e+0,.13921180e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.66669400e+2,.58294800e+2,.47994500e+2,.32163300e+2,&
     & .10551730e+2,.15462830e+1,.82782300e+0,.13922180e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.66436200e+2,.58556400e+2,.47978900e+2,&
     & .32183500e+2,.10528620e+2,.15397570e+1,.82447300e+0,.13921180e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.66368100e+2,.58663000e+2,&
     & .47964300e+2,.32188200e+2,.10508680e+2,.15344650e+1,.82105100e+0,&
     & .13921180e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.66325500e+2,&
     & .58727000e+2,.47967600e+2,.32196900e+2,.10482800e+2,.15295910e+1,&
     & .81841600e+0,.13922180e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .66303600e+2,.58785900e+2,.47960400e+2,.32194700e+2,.10462370e+2,&
     & .15250500e+1,.81580900e+0,.13921180e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.66332200e+2,.58793500e+2,.47946700e+2,.32196300e+2,&
     & .10445690e+2,.15221860e+1,.81413300e+0,.13631770e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.66389100e+2,.58828200e+2,.47920000e+2,&
     & .32155700e+2,.10430340e+2,.15208810e+1,.81012300e+0,.13260880e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.67319100e+2,.58382700e+2,&
     & .47610800e+2,.32032700e+2,.10390010e+2,.15156730e+1,.80297600e+0,&
     & .13316830e+0,.00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0,&
     & .00000000e+0,.00000000e+0,.00000000e+0,.00000000e+0 /

!........................................!
      end module module_radsw_sflux      !
!========================================!

!>\defgroup module_radsw_kgbnn module_radsw_kgbnn
!>\ingroup  module_radsw_main
!! @{

!> This module sets up absorption coefficients for band 16: 2600-3250
!! cm-1 (low - h2o, ch4; high - ch4)
!========================================!
      module module_radsw_kgb16          !
!........................................!
!
! ***************  the original program descriptions  *************** !
!                                                                     !
!     originally by j.delamere, atmospheric & environmental research. !
!     revision: 2.4                                                   !
!     band 16:  2600-3250 cm-1 (low - h2o,ch4; high - ch4)            !
!     reformatted for f90 by jjmorcrette, ecmwf                       !
!                                                                     !
!     this table has been re-generated for reduced number of g-point  !
!     by y.t.hou, ncep  Feb 2004                                      !
!                                                                     !
! ************************  end description  ************************ !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NG16

!
      implicit none
!
      private
!
!> msa16=585
      integer,  public :: MSA16
!> msb16=235
      integer,  public :: MSB16
!> msf16=10
      integer,  public :: MSF16
!> mfr16=3
      integer,  public :: MFR16
      parameter (MSA16=585, MSB16=235, MSF16=10, MFR16=3)

!>    the array selfref contains the coefficient of the water vapor
!!    self-continuum (including the energy term).  the first index
!!    refers to temperature in 7.2 degree increments.  for instance,
!!    jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!!    etc.  the second index runs over the g-channel (1 to 6).
      real (kind=kind_phys), public ::  selfref(MSF16,NG16)

!>    the array absa(585,NG16) (ka(9,5,13,NG16)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels> ~100mb,
!!    temperatures, and binary species parameters (see taumol.f for definition).
!!    the first  index in the array, js, runs from 1 to 9, and corresponds
!!    to different values of the binary species parameter.  for instance,
!!    js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!!    js = 3 corresponds to the parameter value 2/8, etc.  the second index
!!    in the array, jt, which runs from 1 to 5, corresponds to different
!!    temperatures.  more specifically, jt = 3 means that the data are for
!!    the reference temperature tref for this  pressure level, jt = 2 refers
!!    to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!!    is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!!    to the jpth reference pressure level (see taumol.f for these levels
!!    in mb).  the fourth index, ig, goes from 1 to 6, and indicates
!!    which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public ::  absa(MSA16,NG16)

!>    the array absb(235,6) (kb(5,13:59,6)) contains absorption coefs at
!!     the 16 chosen g-values for a range of pressure levels < ~100mb and
!!     temperatures. the first index in the array, jt, which runs from 1 to 5,
!!     corresponds to different temperatures.  more specifically, jt = 3 means
!!     that the data are for the reference temperature tref for this pressure
!!     level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!!     tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!!     the second index, jp, runs from 13 to 59 and refers to the jpth
!!     reference pressure level (see taumol.f for the value of these
!!     pressure levels in mb).  the third index, ig, goes from 1 to 6,
!!     and tells us which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public ::  absb(MSB16,NG16)

      real (kind=kind_phys), public ::  forref(MFR16,NG16)

!> rayleigh extinction coefficient at v = \f$2925 cm^{-1}\f$
      real (kind=kind_phys), parameter, public :: rayl = 2.91e-10

!     the array absa(585,NG16) (ka(9,5,13,NG16)) contains absorption coefs at
!     the 16 chosen g-values for a range of pressure levels> ~100mb,
!     temperatures, and binary species parameters (see taumol.f for definition).
!     the first  index in the array, js, runs from 1 to 9, and corresponds
!     to different values of the binary species parameter.  for instance,
!     js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!     js = 3 corresponds to the parameter value 2/8, etc.  the second index
!     in the array, jt, which runs from 1 to 5, corresponds to different
!     temperatures.  more specifically, jt = 3 means that the data are for
!     the reference temperature tref for this  pressure level, jt = 2 refers
!     to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!     is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!     to the jpth reference pressure level (see taumol.f for these levels
!     in mb).  the fourth index, ig, goes from 1 to 6, and indicates
!     which g-interval the absorption coefficients are for.

      data   absa(  1:180, 1)   /                                       &
     & .6164391E-04,.1393981E-03,.1485874E-03,.1470456E-03,.1378359E-03,&
     & .1251049E-03,.1080809E-03,.8380100E-04,.4375937E-04,.6303123E-04,&
     & .1387829E-03,.1476732E-03,.1465675E-03,.1376116E-03,.1247762E-03,&
     & .1082831E-03,.8452911E-04,.4545314E-04,.6421152E-04,.1383722E-03,&
     & .1469969E-03,.1464177E-03,.1376712E-03,.1247454E-03,.1085938E-03,&
     & .8527181E-04,.4714073E-04,.6487941E-04,.1376604E-03,.1463450E-03,&
     & .1456654E-03,.1375752E-03,.1247169E-03,.1088288E-03,.8599534E-04,&
     & .4882216E-04,.6543285E-04,.1371565E-03,.1459883E-03,.1450442E-03,&
     & .1376047E-03,.1246488E-03,.1084938E-03,.8677500E-04,.5041239E-04,&
     & .5313622E-04,.1212259E-03,.1294107E-03,.1289969E-03,.1215056E-03,&
     & .1099177E-03,.9560988E-04,.7413197E-04,.3741326E-04,.5429633E-04,&
     & .1208023E-03,.1288524E-03,.1287190E-03,.1215251E-03,.1095731E-03,&
     & .9593147E-04,.7493945E-04,.3890369E-04,.5532831E-04,.1205422E-03,&
     & .1285240E-03,.1281017E-03,.1216844E-03,.1097557E-03,.9586096E-04,&
     & .7569726E-04,.4033143E-04,.5592005E-04,.1201155E-03,.1280667E-03,&
     & .1275655E-03,.1215386E-03,.1099924E-03,.9556109E-04,.7641040E-04,&
     & .4181164E-04,.5617841E-04,.1196101E-03,.1275438E-03,.1270047E-03,&
     & .1210925E-03,.1101989E-03,.9565779E-04,.7709991E-04,.4321251E-04,&
     & .4531290E-04,.1048278E-03,.1125651E-03,.1126461E-03,.1064403E-03,&
     & .9599592E-04,.8416227E-04,.6509075E-04,.3260217E-04,.4640094E-04,&
     & .1045811E-03,.1122478E-03,.1120739E-03,.1065744E-03,.9607551E-04,&
     & .8395460E-04,.6589635E-04,.3382481E-04,.4717230E-04,.1043514E-03,&
     & .1118439E-03,.1116296E-03,.1065479E-03,.9636785E-04,.8372961E-04,&
     & .6667225E-04,.3501452E-04,.4770015E-04,.1042243E-03,.1115519E-03,&
     & .1111144E-03,.1064012E-03,.9685136E-04,.8410622E-04,.6743058E-04,&
     & .3619213E-04,.4786332E-04,.1037274E-03,.1110955E-03,.1107897E-03,&
     & .1063675E-03,.9719954E-04,.8449192E-04,.6808715E-04,.3740757E-04,&
     & .3843039E-04,.9043110E-04,.9770352E-04,.9799335E-04,.9304240E-04,&
     & .8389000E-04,.7321734E-04,.5677836E-04,.2994598E-04,.3931732E-04,&
     & .9024615E-04,.9732057E-04,.9745928E-04,.9306943E-04,.8414269E-04,&
     & .7304152E-04,.5752626E-04,.3089971E-04,.3999765E-04,.9015208E-04,&
     & .9690713E-04,.9702354E-04,.9292115E-04,.8443883E-04,.7337197E-04,&
     & .5829869E-04,.3187537E-04,.4043089E-04,.8996634E-04,.9668335E-04,&
     & .9670285E-04,.9286755E-04,.8481230E-04,.7380442E-04,.5906049E-04,&
     & .3283818E-04,.4073202E-04,.8972393E-04,.9645513E-04,.9660020E-04,&
     & .9303706E-04,.8536068E-04,.7423279E-04,.5981517E-04,.3379450E-04/

      data   absa(181:315, 1)   /                                       &
     & .3239990E-04,.7798454E-04,.8455791E-04,.8504346E-04,.8081384E-04,&
     & .7294914E-04,.6337302E-04,.4924460E-04,.2913833E-04,.3322912E-04,&
     & .7774352E-04,.8406820E-04,.8454204E-04,.8094382E-04,.7323153E-04,&
     & .6358291E-04,.4994292E-04,.2993189E-04,.3380830E-04,.7760287E-04,&
     & .8375655E-04,.8415223E-04,.8078676E-04,.7365225E-04,.6386151E-04,&
     & .5066949E-04,.3071453E-04,.3417420E-04,.7744463E-04,.8351059E-04,&
     & .8385781E-04,.8079307E-04,.7414802E-04,.6412908E-04,.5142302E-04,&
     & .3151682E-04,.3444762E-04,.7723103E-04,.8340026E-04,.8386295E-04,&
     & .8097163E-04,.7469899E-04,.6465444E-04,.5217398E-04,.3235645E-04,&
     & .2715767E-04,.6701770E-04,.7283633E-04,.7341902E-04,.6985462E-04,&
     & .6319692E-04,.5481578E-04,.4247223E-04,.2909771E-04,.2792146E-04,&
     & .6685295E-04,.7241685E-04,.7300079E-04,.6986342E-04,.6356893E-04,&
     & .5491074E-04,.4310004E-04,.2966308E-04,.2845821E-04,.6661135E-04,&
     & .7209651E-04,.7266374E-04,.6978860E-04,.6395350E-04,.5508302E-04,&
     & .4376855E-04,.3030854E-04,.2875713E-04,.6641988E-04,.7186205E-04,&
     & .7238777E-04,.6998777E-04,.6431932E-04,.5551791E-04,.4448282E-04,&
     & .3094504E-04,.2891644E-04,.6615229E-04,.7169260E-04,.7229235E-04,&
     & .7001252E-04,.6472305E-04,.5607907E-04,.4487260E-04,.3162719E-04,&
     & .2261551E-04,.5728902E-04,.6237413E-04,.6297079E-04,.5987738E-04,&
     & .5427552E-04,.4703020E-04,.3639661E-04,.2945807E-04,.2329813E-04,&
     & .5709378E-04,.6201885E-04,.6258230E-04,.5996529E-04,.5461263E-04,&
     & .4701194E-04,.3698989E-04,.3029693E-04,.2379683E-04,.5686371E-04,&
     & .6178860E-04,.6234127E-04,.5990707E-04,.5500772E-04,.4729286E-04,&
     & .3760923E-04,.3119772E-04,.2405414E-04,.5664356E-04,.6153544E-04,&
     & .6214186E-04,.6002313E-04,.5529452E-04,.4777815E-04,.3824621E-04,&
     & .3201199E-04,.2417921E-04,.5638188E-04,.6131618E-04,.6200411E-04,&
     & .5999630E-04,.5566905E-04,.4832075E-04,.3841731E-04,.3273083E-04/

      data   absa(316:450, 1)   /                                       &
     & .1878381E-04,.4894749E-04,.5331015E-04,.5389221E-04,.5125674E-04,&
     & .4647673E-04,.4004232E-04,.3106983E-04,.2910087E-04,.1940228E-04,&
     & .4877469E-04,.5302872E-04,.5356337E-04,.5126792E-04,.4678873E-04,&
     & .4012482E-04,.3162012E-04,.3075290E-04,.1985918E-04,.4855059E-04,&
     & .5279719E-04,.5333080E-04,.5126898E-04,.4707665E-04,.4046924E-04,&
     & .3217345E-04,.3150876E-04,.2014062E-04,.4829423E-04,.5255189E-04,&
     & .5319514E-04,.5130651E-04,.4737582E-04,.4091784E-04,.3262566E-04,&
     & .3216455E-04,.2025329E-04,.4801466E-04,.5234235E-04,.5308048E-04,&
     & .5128923E-04,.4759043E-04,.4140204E-04,.3278753E-04,.3333379E-04,&
     & .1546471E-04,.4162009E-04,.4540125E-04,.4587423E-04,.4356518E-04,&
     & .3940508E-04,.3397440E-04,.2638191E-04,.3148426E-04,.1606324E-04,&
     & .4144674E-04,.4510294E-04,.4559038E-04,.4360139E-04,.3960990E-04,&
     & .3411097E-04,.2689026E-04,.3297831E-04,.1647624E-04,.4124716E-04,&
     & .4486029E-04,.4540089E-04,.4359331E-04,.3994622E-04,.3443250E-04,&
     & .2738815E-04,.3466130E-04,.1674831E-04,.4101176E-04,.4463266E-04,&
     & .4523420E-04,.4362159E-04,.4026871E-04,.3484293E-04,.2762312E-04,&
     & .3608726E-04,.1688109E-04,.4072506E-04,.4442676E-04,.4508760E-04,&
     & .4359419E-04,.4044664E-04,.3522608E-04,.2789812E-04,.3743236E-04,&
     & .1262626E-04,.3515542E-04,.3839650E-04,.3883879E-04,.3680058E-04,&
     & .3322886E-04,.2866957E-04,.2229743E-04,.3542581E-04,.1323548E-04,&
     & .3501861E-04,.3816682E-04,.3862136E-04,.3691204E-04,.3341643E-04,&
     & .2882784E-04,.2274500E-04,.3810724E-04,.1360997E-04,.3480644E-04,&
     & .3790261E-04,.3840467E-04,.3688940E-04,.3369037E-04,.2909482E-04,&
     & .2319725E-04,.4051484E-04,.1386341E-04,.3459077E-04,.3769098E-04,&
     & .3821868E-04,.3684829E-04,.3399846E-04,.2945016E-04,.2334606E-04,&
     & .4250812E-04,.1401007E-04,.3434705E-04,.3748890E-04,.3806620E-04,&
     & .3681278E-04,.3416954E-04,.2981567E-04,.2357415E-04,.4469942E-04/

      data   absa(451:585, 1)   /                                       &
     & .1054345E-04,.2952630E-04,.3222392E-04,.3265638E-04,.3106790E-04,&
     & .2796314E-04,.2415008E-04,.1892216E-04,.3133027E-04,.1103113E-04,&
     & .2937313E-04,.3202245E-04,.3244622E-04,.3107680E-04,.2818612E-04,&
     & .2432663E-04,.1931576E-04,.3420418E-04,.1132648E-04,.2917577E-04,&
     & .3179835E-04,.3224295E-04,.3105314E-04,.2839919E-04,.2458295E-04,&
     & .1960995E-04,.3745408E-04,.1151773E-04,.2897659E-04,.3158949E-04,&
     & .3206591E-04,.3099578E-04,.2862811E-04,.2491522E-04,.1969533E-04,&
     & .4018024E-04,.1162646E-04,.2873280E-04,.3141795E-04,.3192928E-04,&
     & .3092122E-04,.2876193E-04,.2527302E-04,.1991005E-04,.4272251E-04,&
     & .8773859E-05,.2472331E-04,.2698123E-04,.2731286E-04,.2605313E-04,&
     & .2347524E-04,.2028179E-04,.1599354E-04,.2710338E-04,.9175082E-05,&
     & .2455642E-04,.2677932E-04,.2714788E-04,.2603439E-04,.2367042E-04,&
     & .2044068E-04,.1632502E-04,.2976471E-04,.9405766E-05,.2437939E-04,&
     & .2657517E-04,.2695641E-04,.2600791E-04,.2387449E-04,.2068968E-04,&
     & .1643613E-04,.3198691E-04,.9551613E-05,.2419222E-04,.2638939E-04,&
     & .2678440E-04,.2590383E-04,.2398600E-04,.2098502E-04,.1655507E-04,&
     & .3463054E-04,.9634553E-05,.2396313E-04,.2622876E-04,.2664925E-04,&
     & .2582504E-04,.2410681E-04,.2125060E-04,.1672669E-04,.3742745E-04,&
     & .7283549E-05,.2064472E-04,.2252697E-04,.2280007E-04,.2176915E-04,&
     & .1965230E-04,.1697791E-04,.1344921E-04,.2262985E-04,.7614582E-05,&
     & .2049273E-04,.2234335E-04,.2265284E-04,.2175650E-04,.1980433E-04,&
     & .1713132E-04,.1373534E-04,.2489034E-04,.7799854E-05,.2032918E-04,&
     & .2216475E-04,.2247253E-04,.2168451E-04,.1997844E-04,.1730534E-04,&
     & .1371476E-04,.2674112E-04,.7917832E-05,.2015404E-04,.2198881E-04,&
     & .2232428E-04,.2159377E-04,.2002799E-04,.1753711E-04,.1382698E-04,&
     & .2881746E-04,.7981504E-05,.1995296E-04,.2184003E-04,.2217179E-04,&
     & .2152204E-04,.2013182E-04,.1776050E-04,.1402084E-04,.3110145E-04/

      data   absa(  1:180, 2)   /                                       &
     & .2930165E-03,.4205176E-03,.4603190E-03,.4714149E-03,.4679470E-03,&
     & .4501949E-03,.4185459E-03,.3688035E-03,.2684515E-03,.2936114E-03,&
     & .4252087E-03,.4672427E-03,.4798112E-03,.4769734E-03,.4593456E-03,&
     & .4281470E-03,.3781642E-03,.2764789E-03,.2946377E-03,.4315576E-03,&
     & .4755024E-03,.4896418E-03,.4876102E-03,.4700407E-03,.4395838E-03,&
     & .3880213E-03,.2860406E-03,.2952749E-03,.4370625E-03,.4830699E-03,&
     & .4992608E-03,.4981682E-03,.4812406E-03,.4500990E-03,.3988892E-03,&
     & .2947867E-03,.2963643E-03,.4435447E-03,.4919681E-03,.5108647E-03,&
     & .5100757E-03,.4946687E-03,.4633694E-03,.4116274E-03,.3068946E-03,&
     & .2564192E-03,.3741028E-03,.4121845E-03,.4222514E-03,.4178533E-03,&
     & .4017900E-03,.3723948E-03,.3265092E-03,.2280897E-03,.2572159E-03,&
     & .3784588E-03,.4184094E-03,.4294373E-03,.4260348E-03,.4103047E-03,&
     & .3806553E-03,.3334951E-03,.2357658E-03,.2585085E-03,.3843346E-03,&
     & .4259985E-03,.4392394E-03,.4361957E-03,.4206279E-03,.3905943E-03,&
     & .3429498E-03,.2431183E-03,.2596736E-03,.3898247E-03,.4335148E-03,&
     & .4488328E-03,.4466516E-03,.4312067E-03,.4012724E-03,.3532432E-03,&
     & .2517667E-03,.2607431E-03,.3954864E-03,.4412810E-03,.4585918E-03,&
     & .4573106E-03,.4422671E-03,.4128750E-03,.3642066E-03,.2619245E-03,&
     & .2235048E-03,.3314541E-03,.3652461E-03,.3744704E-03,.3699739E-03,&
     & .3548398E-03,.3279534E-03,.2865252E-03,.1924155E-03,.2243217E-03,&
     & .3351441E-03,.3704468E-03,.3812002E-03,.3769484E-03,.3624388E-03,&
     & .3347617E-03,.2927657E-03,.1990504E-03,.2253955E-03,.3397624E-03,&
     & .3767056E-03,.3889755E-03,.3852321E-03,.3709575E-03,.3432829E-03,&
     & .3003296E-03,.2050004E-03,.2269437E-03,.3455225E-03,.3848163E-03,&
     & .3985839E-03,.3955107E-03,.3808205E-03,.3527054E-03,.3092164E-03,&
     & .2125103E-03,.2280141E-03,.3507535E-03,.3920562E-03,.4069340E-03,&
     & .4049634E-03,.3903833E-03,.3628870E-03,.3187283E-03,.2207668E-03,&
     & .1945146E-03,.2918639E-03,.3214502E-03,.3303679E-03,.3254231E-03,&
     & .3116270E-03,.2876070E-03,.2498710E-03,.1637253E-03,.1954096E-03,&
     & .2951961E-03,.3261050E-03,.3361412E-03,.3314562E-03,.3181672E-03,&
     & .2938352E-03,.2559070E-03,.1693706E-03,.1965264E-03,.2992768E-03,&
     & .3318268E-03,.3430041E-03,.3386437E-03,.3253725E-03,.3007759E-03,&
     & .2623706E-03,.1739993E-03,.1975452E-03,.3039553E-03,.3382589E-03,&
     & .3504664E-03,.3467920E-03,.3335199E-03,.3082884E-03,.2693916E-03,&
     & .1801702E-03,.1992020E-03,.3099779E-03,.3465283E-03,.3595055E-03,&
     & .3565400E-03,.3430057E-03,.3175611E-03,.2774855E-03,.1867273E-03/

      data   absa(181:315, 2)   /                                       &
     & .1692110E-03,.2560783E-03,.2816239E-03,.2892751E-03,.2854340E-03,&
     & .2729454E-03,.2512636E-03,.2176013E-03,.1406685E-03,.1700158E-03,&
     & .2591285E-03,.2857505E-03,.2942600E-03,.2906779E-03,.2782649E-03,&
     & .2565024E-03,.2227555E-03,.1451134E-03,.1711402E-03,.2627639E-03,&
     & .2908575E-03,.3002244E-03,.2969592E-03,.2845936E-03,.2626853E-03,&
     & .2280973E-03,.1489243E-03,.1723113E-03,.2667721E-03,.2966815E-03,&
     & .3070349E-03,.3041790E-03,.2912783E-03,.2694000E-03,.2338131E-03,&
     & .1538692E-03,.1740563E-03,.2722405E-03,.3040666E-03,.3151440E-03,&
     & .3128365E-03,.2994324E-03,.2773169E-03,.2405489E-03,.1587634E-03,&
     & .1467247E-03,.2239121E-03,.2459341E-03,.2521661E-03,.2491285E-03,&
     & .2381498E-03,.2185025E-03,.1886728E-03,.1245807E-03,.1473884E-03,&
     & .2265685E-03,.2495294E-03,.2564125E-03,.2535594E-03,.2425776E-03,&
     & .2232033E-03,.1930135E-03,.1274953E-03,.1482897E-03,.2296257E-03,&
     & .2540362E-03,.2615549E-03,.2589723E-03,.2479233E-03,.2285370E-03,&
     & .1976049E-03,.1306204E-03,.1494392E-03,.2332170E-03,.2591337E-03,&
     & .2675927E-03,.2648872E-03,.2537659E-03,.2341528E-03,.2022251E-03,&
     & .1344888E-03,.1508093E-03,.2374102E-03,.2648426E-03,.2740852E-03,&
     & .2718576E-03,.2604781E-03,.2406433E-03,.2081661E-03,.1378196E-03,&
     & .1263109E-03,.1951428E-03,.2144975E-03,.2195749E-03,.2170080E-03,&
     & .2069188E-03,.1894328E-03,.1633783E-03,.1200887E-03,.1271134E-03,&
     & .1973810E-03,.2175147E-03,.2232648E-03,.2204590E-03,.2109194E-03,&
     & .1937750E-03,.1670597E-03,.1213160E-03,.1278627E-03,.2000293E-03,&
     & .2212521E-03,.2276502E-03,.2250313E-03,.2152307E-03,.1982160E-03,&
     & .1708010E-03,.1229746E-03,.1288762E-03,.2031265E-03,.2255973E-03,&
     & .2327902E-03,.2301752E-03,.2203853E-03,.2029333E-03,.1746039E-03,&
     & .1249396E-03,.1300624E-03,.2067082E-03,.2305851E-03,.2383986E-03,&
     & .2363202E-03,.2260966E-03,.2085247E-03,.1797897E-03,.1276706E-03/

      data   absa(316:450, 2)   /                                       &
     & .1087530E-03,.1695227E-03,.1869845E-03,.1912855E-03,.1886016E-03,&
     & .1794107E-03,.1640395E-03,.1407348E-03,.1388109E-03,.1095657E-03,&
     & .1712971E-03,.1895043E-03,.1945158E-03,.1916236E-03,.1827598E-03,&
     & .1678635E-03,.1438343E-03,.1366925E-03,.1103222E-03,.1736665E-03,&
     & .1926247E-03,.1982051E-03,.1955376E-03,.1864091E-03,.1715841E-03,&
     & .1469470E-03,.1364063E-03,.1111854E-03,.1763283E-03,.1961855E-03,&
     & .2023728E-03,.2000777E-03,.1907608E-03,.1757564E-03,.1506354E-03,&
     & .1369112E-03,.1122634E-03,.1793268E-03,.2003957E-03,.2070334E-03,&
     & .2051943E-03,.1958970E-03,.1805692E-03,.1551262E-03,.1374744E-03,&
     & .9300705E-04,.1464182E-03,.1620111E-03,.1662152E-03,.1638421E-03,&
     & .1556181E-03,.1415254E-03,.1207125E-03,.2197460E-03,.9379197E-04,&
     & .1478848E-03,.1642219E-03,.1688554E-03,.1664207E-03,.1584541E-03,&
     & .1446859E-03,.1233342E-03,.2178811E-03,.9457710E-04,.1498107E-03,&
     & .1668492E-03,.1719850E-03,.1697382E-03,.1615171E-03,.1478254E-03,&
     & .1260245E-03,.2169464E-03,.9536836E-04,.1520740E-03,.1699345E-03,&
     & .1755822E-03,.1735872E-03,.1652759E-03,.1515057E-03,.1293800E-03,&
     & .2166357E-03,.9627944E-04,.1547433E-03,.1735275E-03,.1795990E-03,&
     & .1780373E-03,.1697434E-03,.1556408E-03,.1331356E-03,.2164129E-03,&
     & .7917379E-04,.1259529E-03,.1400562E-03,.1440600E-03,.1421290E-03,&
     & .1348884E-03,.1219352E-03,.1031837E-03,.3271209E-03,.7977861E-04,&
     & .1271040E-03,.1418834E-03,.1462764E-03,.1442107E-03,.1372758E-03,&
     & .1246562E-03,.1054164E-03,.3261360E-03,.8053422E-04,.1287862E-03,&
     & .1442112E-03,.1488988E-03,.1470409E-03,.1399784E-03,.1274996E-03,&
     & .1079546E-03,.3247606E-03,.8124880E-04,.1308583E-03,.1468362E-03,&
     & .1519274E-03,.1504180E-03,.1431923E-03,.1307815E-03,.1110513E-03,&
     & .3234572E-03,.8201452E-04,.1331325E-03,.1497495E-03,.1553833E-03,&
     & .1542308E-03,.1471022E-03,.1343535E-03,.1143694E-03,.3218489E-03/

      data   absa(451:585, 2)   /                                       &
     & .6728851E-04,.1080906E-03,.1208580E-03,.1247323E-03,.1232827E-03,&
     & .1175266E-03,.1060571E-03,.8883780E-04,.3501958E-03,.6791800E-04,&
     & .1091898E-03,.1225378E-03,.1268241E-03,.1253901E-03,.1196653E-03,&
     & .1084469E-03,.9082183E-04,.3442870E-03,.6851447E-04,.1107863E-03,&
     & .1246494E-03,.1293024E-03,.1281246E-03,.1222490E-03,.1111838E-03,&
     & .9324574E-04,.3420183E-03,.6908353E-04,.1126804E-03,.1269928E-03,&
     & .1321156E-03,.1312354E-03,.1252083E-03,.1140710E-03,.9617117E-04,&
     & .3390113E-03,.6957312E-04,.1148165E-03,.1296320E-03,.1352293E-03,&
     & .1346612E-03,.1287087E-03,.1172723E-03,.9910752E-04,.3385633E-03,&
     & .5686255E-04,.9216035E-04,.1034858E-03,.1071516E-03,.1060652E-03,&
     & .1014159E-03,.9163836E-04,.7623805E-04,.3310437E-03,.5734770E-04,&
     & .9329051E-04,.1049670E-03,.1090060E-03,.1081383E-03,.1034188E-03,&
     & .9390315E-04,.7812498E-04,.3244905E-03,.5781298E-04,.9473394E-04,&
     & .1068652E-03,.1112574E-03,.1106069E-03,.1057391E-03,.9631206E-04,&
     & .8054235E-04,.3210568E-03,.5822217E-04,.9640942E-04,.1089655E-03,&
     & .1137847E-03,.1134036E-03,.1085277E-03,.9896999E-04,.8309374E-04,&
     & .3173487E-03,.5853550E-04,.9825196E-04,.1113394E-03,.1165990E-03,&
     & .1165067E-03,.1117023E-03,.1018552E-03,.8573085E-04,.3168446E-03,&
     & .4783824E-04,.7831477E-04,.8819578E-04,.9148052E-04,.9081639E-04,&
     & .8696960E-04,.7879106E-04,.6529420E-04,.2796540E-03,.4821354E-04,&
     & .7934773E-04,.8956662E-04,.9314137E-04,.9269390E-04,.8882919E-04,&
     & .8080982E-04,.6704726E-04,.2741660E-03,.4859146E-04,.8061493E-04,&
     & .9116325E-04,.9515541E-04,.9486227E-04,.9088828E-04,.8295551E-04,&
     & .6935405E-04,.2709349E-03,.4883862E-04,.8204290E-04,.9304806E-04,&
     & .9741929E-04,.9740051E-04,.9346855E-04,.8534014E-04,.7158356E-04,&
     & .2681460E-03,.4903521E-04,.8353294E-04,.9510393E-04,.9997200E-04,&
     & .1001833E-03,.9622958E-04,.8790924E-04,.7395389E-04,.2673945E-03/

      data   absa(  1:180, 3)   /                                       &
     & .8184572E-03,.1288796E-02,.1495187E-02,.1632713E-02,.1719945E-02,&
     & .1761605E-02,.1759635E-02,.1698408E-02,.1367318E-02,.8243726E-03,&
     & .1301794E-02,.1514658E-02,.1658937E-02,.1753464E-02,.1802877E-02,&
     & .1805416E-02,.1745195E-02,.1405620E-02,.8348880E-03,.1320873E-02,&
     & .1542641E-02,.1692748E-02,.1795401E-02,.1852780E-02,.1858131E-02,&
     & .1800382E-02,.1449500E-02,.8418718E-03,.1340423E-02,.1571743E-02,&
     & .1728235E-02,.1837120E-02,.1903874E-02,.1915521E-02,.1859941E-02,&
     & .1503155E-02,.8531253E-03,.1366891E-02,.1607512E-02,.1771740E-02,&
     & .1886777E-02,.1961375E-02,.1981251E-02,.1927613E-02,.1561788E-02,&
     & .7302326E-03,.1151907E-02,.1334524E-02,.1456839E-02,.1533051E-02,&
     & .1567636E-02,.1559387E-02,.1499443E-02,.1187382E-02,.7372791E-03,&
     & .1164596E-02,.1352581E-02,.1480270E-02,.1561840E-02,.1601428E-02,&
     & .1596993E-02,.1539888E-02,.1219763E-02,.7483530E-03,.1183063E-02,&
     & .1378407E-02,.1510357E-02,.1596909E-02,.1642217E-02,.1642410E-02,&
     & .1586010E-02,.1257897E-02,.7577399E-03,.1203485E-02,.1406414E-02,&
     & .1543415E-02,.1634037E-02,.1686222E-02,.1692333E-02,.1636545E-02,&
     & .1301186E-02,.7660501E-03,.1226282E-02,.1437446E-02,.1580408E-02,&
     & .1675024E-02,.1733502E-02,.1745352E-02,.1693309E-02,.1350969E-02,&
     & .6502794E-03,.1026616E-02,.1186960E-02,.1295393E-02,.1364677E-02,&
     & .1394889E-02,.1381785E-02,.1317502E-02,.1017363E-02,.6567559E-03,&
     & .1037505E-02,.1202583E-02,.1315591E-02,.1388459E-02,.1421306E-02,&
     & .1413547E-02,.1351496E-02,.1044406E-02,.6651689E-03,.1050207E-02,&
     & .1223023E-02,.1339830E-02,.1416067E-02,.1452943E-02,.1449484E-02,&
     & .1390425E-02,.1076530E-02,.6772722E-03,.1071331E-02,.1249580E-02,&
     & .1370662E-02,.1449539E-02,.1490615E-02,.1491899E-02,.1434093E-02,&
     & .1111777E-02,.6855463E-03,.1091648E-02,.1276403E-02,.1402863E-02,&
     & .1483596E-02,.1529594E-02,.1535466E-02,.1481180E-02,.1152638E-02,&
     & .5768479E-03,.9115621E-03,.1053538E-02,.1150084E-02,.1211847E-02,&
     & .1236742E-02,.1220110E-02,.1153457E-02,.8691713E-03,.5830135E-03,&
     & .9209284E-03,.1066914E-02,.1166765E-02,.1232185E-02,.1259290E-02,&
     & .1246437E-02,.1180938E-02,.8912098E-03,.5910480E-03,.9345088E-03,&
     & .1084575E-02,.1187678E-02,.1255979E-02,.1286682E-02,.1276512E-02,&
     & .1213240E-02,.9180720E-03,.6008298E-03,.9515132E-03,.1106486E-02,&
     & .1212563E-02,.1283329E-02,.1317431E-02,.1311088E-02,.1249258E-02,&
     & .9471810E-03,.6138388E-03,.9709949E-03,.1132455E-02,.1242798E-02,&
     & .1315868E-02,.1353280E-02,.1350757E-02,.1291032E-02,.9816019E-03/

      data   absa(181:315, 3)   /                                       &
     & .5093571E-03,.8080237E-03,.9381643E-03,.1019789E-02,.1071192E-02,&
     & .1090471E-02,.1073488E-02,.1005758E-02,.7418709E-03,.5151626E-03,&
     & .8165236E-03,.9496789E-03,.1034266E-02,.1088284E-02,.1110067E-02,&
     & .1094940E-02,.1028625E-02,.7599047E-03,.5224485E-03,.8286429E-03,&
     & .9624412E-03,.1052107E-02,.1109000E-02,.1133457E-02,.1120200E-02,&
     & .1055094E-02,.7812729E-03,.5316862E-03,.8438037E-03,.9798289E-03,&
     & .1073448E-02,.1132718E-02,.1160426E-02,.1149517E-02,.1085478E-02,&
     & .8053947E-03,.5444831E-03,.8638786E-03,.1002962E-02,.1099774E-02,&
     & .1161623E-02,.1191798E-02,.1183243E-02,.1121002E-02,.8340163E-03,&
     & .4476728E-03,.7152034E-03,.8303044E-03,.9027280E-03,.9422368E-03,&
     & .9566089E-03,.9387208E-03,.8743692E-03,.6315594E-03,.4531060E-03,&
     & .7229985E-03,.8401962E-03,.9128749E-03,.9571607E-03,.9729227E-03,&
     & .9565139E-03,.8930810E-03,.6455309E-03,.4601132E-03,.7340161E-03,&
     & .8531068E-03,.9280096E-03,.9748476E-03,.9926457E-03,.9777088E-03,&
     & .9149552E-03,.6618730E-03,.4687606E-03,.7470819E-03,.8686951E-03,&
     & .9463629E-03,.9951022E-03,.1015485E-02,.1002244E-02,.9403643E-03,&
     & .6808008E-03,.4785595E-03,.7635702E-03,.8869579E-03,.9676686E-03,&
     & .1018859E-02,.1041246E-02,.1029537E-02,.9684597E-03,.7044125E-03,&
     & .3929615E-03,.6307616E-03,.7313469E-03,.7929589E-03,.8247908E-03,&
     & .8356659E-03,.8173907E-03,.7575170E-03,.5383971E-03,.3978536E-03,&
     & .6382438E-03,.7406018E-03,.8039178E-03,.8378859E-03,.8488038E-03,&
     & .8315811E-03,.7727276E-03,.5497760E-03,.4043732E-03,.6478463E-03,&
     & .7517647E-03,.8168558E-03,.8526219E-03,.8656075E-03,.8491574E-03,&
     & .7907891E-03,.5627384E-03,.4122128E-03,.6599763E-03,.7657659E-03,&
     & .8304335E-03,.8701303E-03,.8849440E-03,.8695717E-03,.8115473E-03,&
     & .5782796E-03,.4216792E-03,.6747622E-03,.7823998E-03,.8487735E-03,&
     & .8905701E-03,.9067120E-03,.8925209E-03,.8346838E-03,.5962302E-03/

      data   absa(316:450, 3)   /                                       &
     & .3452083E-03,.5538022E-03,.6406411E-03,.6932554E-03,.7213690E-03,&
     & .7269628E-03,.7089993E-03,.6545119E-03,.4847209E-03,.3494976E-03,&
     & .5608628E-03,.6492121E-03,.7026395E-03,.7319618E-03,.7381217E-03,&
     & .7204878E-03,.6670675E-03,.4924614E-03,.3555168E-03,.5695884E-03,&
     & .6593390E-03,.7138651E-03,.7429187E-03,.7522301E-03,.7349547E-03,&
     & .6817461E-03,.5010009E-03,.3629721E-03,.5805421E-03,.6721512E-03,&
     & .7278957E-03,.7573896E-03,.7683553E-03,.7514496E-03,.6984177E-03,&
     & .5116719E-03,.3715828E-03,.5938995E-03,.6869808E-03,.7439941E-03,&
     & .7750335E-03,.7864892E-03,.7704733E-03,.7174510E-03,.5246067E-03,&
     & .3053224E-03,.4856259E-03,.5588804E-03,.6031584E-03,.6264927E-03,&
     & .6296159E-03,.6127370E-03,.5636624E-03,.6772975E-03,.3090662E-03,&
     & .4924553E-03,.5671508E-03,.6120265E-03,.6363728E-03,.6394116E-03,&
     & .6225004E-03,.5739117E-03,.6640269E-03,.3140506E-03,.5003281E-03,&
     & .5762886E-03,.6218129E-03,.6467350E-03,.6509293E-03,.6346838E-03,&
     & .5862457E-03,.6537700E-03,.3203596E-03,.5098203E-03,.5876621E-03,&
     & .6339281E-03,.6587374E-03,.6641130E-03,.6482634E-03,.5998107E-03,&
     & .6477927E-03,.3280678E-03,.5213843E-03,.6008020E-03,.6482465E-03,&
     & .6724424E-03,.6792561E-03,.6641693E-03,.6153933E-03,.6458298E-03,&
     & .2733291E-03,.4289828E-03,.4894421E-03,.5250706E-03,.5435356E-03,&
     & .5455983E-03,.5286069E-03,.4846966E-03,.1525269E-02,.2767436E-03,&
     & .4347254E-03,.4967461E-03,.5327268E-03,.5517500E-03,.5540458E-03,&
     & .5369648E-03,.4938952E-03,.1462778E-02,.2810846E-03,.4416485E-03,&
     & .5047342E-03,.5418403E-03,.5610033E-03,.5634743E-03,.5472960E-03,&
     & .5038891E-03,.1415701E-02,.2866866E-03,.4502441E-03,.5146077E-03,&
     & .5523415E-03,.5718914E-03,.5740346E-03,.5587333E-03,.5149488E-03,&
     & .1375472E-02,.2934865E-03,.4607363E-03,.5263019E-03,.5647079E-03,&
     & .5847436E-03,.5869370E-03,.5719471E-03,.5277797E-03,.1341507E-02/

      data   absa(451:585, 3)   /                                       &
     & .2437828E-03,.3835942E-03,.4355111E-03,.4643060E-03,.4763641E-03,&
     & .4751451E-03,.4583198E-03,.4193574E-03,.1825342E-02,.2475557E-03,&
     & .3898469E-03,.4421366E-03,.4710873E-03,.4839473E-03,.4831741E-03,&
     & .4661299E-03,.4275995E-03,.1756933E-02,.2523063E-03,.3973500E-03,&
     & .4497777E-03,.4789579E-03,.4924096E-03,.4921150E-03,.4752010E-03,&
     & .4363639E-03,.1708388E-02,.2580622E-03,.4058625E-03,.4589941E-03,&
     & .4884064E-03,.5025671E-03,.5020326E-03,.4855219E-03,.4461901E-03,&
     & .1669299E-02,.2648941E-03,.4157912E-03,.4698491E-03,.4998329E-03,&
     & .5143630E-03,.5136245E-03,.4973797E-03,.4575998E-03,.1623002E-02,&
     & .2144367E-03,.3401170E-03,.3854223E-03,.4099266E-03,.4196610E-03,&
     & .4161366E-03,.3989091E-03,.3620378E-03,.1821655E-02,.2181060E-03,&
     & .3463069E-03,.3922259E-03,.4165637E-03,.4265488E-03,.4233797E-03,&
     & .4058075E-03,.3692157E-03,.1752071E-02,.2229514E-03,.3537530E-03,&
     & .4002355E-03,.4242150E-03,.4345415E-03,.4315545E-03,.4136194E-03,&
     & .3770147E-03,.1694072E-02,.2286334E-03,.3622049E-03,.4092201E-03,&
     & .4334592E-03,.4440346E-03,.4406362E-03,.4225253E-03,.3857130E-03,&
     & .1645638E-02,.2354630E-03,.3723794E-03,.4193491E-03,.4443949E-03,&
     & .4547391E-03,.4513798E-03,.4332059E-03,.3959499E-03,.1604607E-02,&
     & .1873934E-03,.2995165E-03,.3394711E-03,.3611157E-03,.3690992E-03,&
     & .3648908E-03,.3479275E-03,.3123745E-03,.1574790E-02,.1910454E-03,&
     & .3056861E-03,.3461932E-03,.3679395E-03,.3758939E-03,.3716238E-03,&
     & .3543646E-03,.3186636E-03,.1513357E-02,.1954531E-03,.3127911E-03,&
     & .3540282E-03,.3758084E-03,.3835301E-03,.3790064E-03,.3616108E-03,&
     & .3254549E-03,.1464333E-02,.2011139E-03,.3213571E-03,.3628469E-03,&
     & .3848293E-03,.3922803E-03,.3876951E-03,.3697429E-03,.3333521E-03,&
     & .1424128E-02,.2075183E-03,.3308876E-03,.3732484E-03,.3952328E-03,&
     & .4025443E-03,.3978826E-03,.3793694E-03,.3424789E-03,.1389880E-02/

      data   absa(  1:180, 4)   /                                       &
     & .3736324E-02,.6100560E-02,.7728410E-02,.8959576E-02,.9913897E-02,&
     & .1060534E-01,.1095043E-01,.1073709E-01,.9408838E-02,.3792238E-02,&
     & .6178500E-02,.7827205E-02,.9076284E-02,.1004324E-01,.1073477E-01,&
     & .1108085E-01,.1083940E-01,.9519649E-02,.3871754E-02,.6278718E-02,&
     & .7949468E-02,.9221394E-02,.1019996E-01,.1088941E-01,.1123148E-01,&
     & .1097176E-01,.9657396E-02,.3934975E-02,.6363492E-02,.8064193E-02,&
     & .9370476E-02,.1036793E-01,.1105745E-01,.1140542E-01,.1112862E-01,&
     & .9816845E-02,.4020724E-02,.6468817E-02,.8206395E-02,.9541906E-02,&
     & .1055918E-01,.1126073E-01,.1160795E-01,.1130758E-01,.9996192E-02,&
     & .3402273E-02,.5620650E-02,.7030748E-02,.8098078E-02,.8924472E-02,&
     & .9508944E-02,.9801255E-02,.9592074E-02,.8097867E-02,.3451518E-02,&
     & .5698898E-02,.7132591E-02,.8219355E-02,.9056752E-02,.9642702E-02,&
     & .9930623E-02,.9700146E-02,.8198722E-02,.3525436E-02,.5795932E-02,&
     & .7258246E-02,.8364978E-02,.9212267E-02,.9796289E-02,.1008263E-01,&
     & .9835560E-02,.8322568E-02,.3594242E-02,.5888324E-02,.7386430E-02,&
     & .8518910E-02,.9379100E-02,.9970774E-02,.1025951E-01,.9989379E-02,&
     & .8466107E-02,.3660490E-02,.5981362E-02,.7519542E-02,.8678334E-02,&
     & .9565719E-02,.1017025E-01,.1045322E-01,.1016255E-01,.8625539E-02,&
     & .3102796E-02,.5141091E-02,.6344760E-02,.7250524E-02,.7949175E-02,&
     & .8434987E-02,.8669628E-02,.8481084E-02,.6903275E-02,.3148146E-02,&
     & .5218469E-02,.6443866E-02,.7368450E-02,.8075641E-02,.8560889E-02,&
     & .8792122E-02,.8585149E-02,.6988914E-02,.3201454E-02,.5303035E-02,&
     & .6555873E-02,.7497631E-02,.8217284E-02,.8706091E-02,.8936316E-02,&
     & .8708232E-02,.7092276E-02,.3275760E-02,.5404005E-02,.6690045E-02,&
     & .7649565E-02,.8382414E-02,.8880227E-02,.9103836E-02,.8853150E-02,&
     & .7213900E-02,.3333969E-02,.5494628E-02,.6816793E-02,.7800860E-02,&
     & .8559397E-02,.9065543E-02,.9281723E-02,.9013731E-02,.7356112E-02,&
     & .2831427E-02,.4682414E-02,.5696173E-02,.6452550E-02,.7031440E-02,&
     & .7425488E-02,.7615078E-02,.7445178E-02,.5905547E-02,.2873398E-02,&
     & .4758548E-02,.5791245E-02,.6565599E-02,.7146886E-02,.7543948E-02,&
     & .7728470E-02,.7539994E-02,.5975231E-02,.2922042E-02,.4836804E-02,&
     & .5896068E-02,.6683182E-02,.7278970E-02,.7678598E-02,.7858498E-02,&
     & .7652488E-02,.6063670E-02,.2979003E-02,.4922733E-02,.6010505E-02,&
     & .6817378E-02,.7426977E-02,.7832832E-02,.8004073E-02,.7783099E-02,&
     & .6166540E-02,.3051509E-02,.5034845E-02,.6147088E-02,.6974054E-02,&
     & .7599704E-02,.8009148E-02,.8169023E-02,.7931032E-02,.6289201E-02/

      data   absa(181:315, 4)   /                                       &
     & .2581422E-02,.4252396E-02,.5095766E-02,.5725588E-02,.6193716E-02,&
     & .6511713E-02,.6657529E-02,.6499613E-02,.5066125E-02,.2622021E-02,&
     & .4323928E-02,.5186112E-02,.5827234E-02,.6301641E-02,.6620209E-02,&
     & .6758605E-02,.6586843E-02,.5124677E-02,.2667849E-02,.4398967E-02,&
     & .5285995E-02,.5935349E-02,.6423419E-02,.6740405E-02,.6875009E-02,&
     & .6687854E-02,.5199137E-02,.2719644E-02,.4480894E-02,.5391504E-02,&
     & .6059082E-02,.6557515E-02,.6877697E-02,.7004932E-02,.6806128E-02,&
     & .5284911E-02,.2786609E-02,.4583687E-02,.5519294E-02,.6203484E-02,&
     & .6711011E-02,.7035046E-02,.7152458E-02,.6938895E-02,.5385742E-02,&
     & .2339609E-02,.3846094E-02,.4554211E-02,.5067608E-02,.5443590E-02,&
     & .5691422E-02,.5796061E-02,.5651788E-02,.4350403E-02,.2380923E-02,&
     & .3911794E-02,.4637326E-02,.5161799E-02,.5541208E-02,.5788128E-02,&
     & .5888423E-02,.5729114E-02,.4399487E-02,.2424975E-02,.3982801E-02,&
     & .4723535E-02,.5261943E-02,.5649668E-02,.5897396E-02,.5992483E-02,&
     & .5819539E-02,.4460491E-02,.2474162E-02,.4061310E-02,.4818668E-02,&
     & .5372412E-02,.5769446E-02,.6018440E-02,.6107034E-02,.5923338E-02,&
     & .4530683E-02,.2528477E-02,.4146106E-02,.4927901E-02,.5493040E-02,&
     & .5902213E-02,.6153417E-02,.6234974E-02,.6038909E-02,.4613086E-02,&
     & .2108102E-02,.3463767E-02,.4067283E-02,.4486100E-02,.4783508E-02,&
     & .4968566E-02,.5039362E-02,.4903349E-02,.3736969E-02,.2149169E-02,&
     & .3526123E-02,.4141192E-02,.4568003E-02,.4870426E-02,.5056936E-02,&
     & .5123533E-02,.4973057E-02,.3776037E-02,.2192220E-02,.3593626E-02,&
     & .4218969E-02,.4658171E-02,.4967303E-02,.5154377E-02,.5214872E-02,&
     & .5052489E-02,.3823705E-02,.2239930E-02,.3665815E-02,.4305224E-02,&
     & .4760112E-02,.5074184E-02,.5261541E-02,.5315916E-02,.5141203E-02,&
     & .3879448E-02,.2291457E-02,.3745249E-02,.4403192E-02,.4867213E-02,&
     & .5191513E-02,.5380406E-02,.5427689E-02,.5240706E-02,.3946442E-02/

      data   absa(316:450, 4)   /                                       &
     & .1888206E-02,.3101989E-02,.3625968E-02,.3971182E-02,.4202567E-02,&
     & .4344619E-02,.4384538E-02,.4251354E-02,.3181821E-02,.1928325E-02,&
     & .3162679E-02,.3693527E-02,.4045345E-02,.4282687E-02,.4424622E-02,&
     & .4458656E-02,.4312656E-02,.3213796E-02,.1970231E-02,.3226941E-02,&
     & .3764384E-02,.4127675E-02,.4371933E-02,.4510091E-02,.4538985E-02,&
     & .4381451E-02,.3253041E-02,.2016368E-02,.3294141E-02,.3842606E-02,&
     & .4216009E-02,.4467372E-02,.4604053E-02,.4627489E-02,.4457971E-02,&
     & .3298935E-02,.2065463E-02,.3369431E-02,.3930153E-02,.4311858E-02,&
     & .4569900E-02,.4708801E-02,.4723164E-02,.4542339E-02,.3355283E-02,&
     & .1679110E-02,.2762441E-02,.3217170E-02,.3511424E-02,.3699441E-02,&
     & .3807111E-02,.3820305E-02,.3685883E-02,.2768596E-02,.1718111E-02,&
     & .2820629E-02,.3280981E-02,.3579835E-02,.3771324E-02,.3877601E-02,&
     & .3885055E-02,.3740150E-02,.2792302E-02,.1759096E-02,.2880777E-02,&
     & .3347449E-02,.3654997E-02,.3849489E-02,.3952528E-02,.3955472E-02,&
     & .3800453E-02,.2822088E-02,.1804159E-02,.2943869E-02,.3420297E-02,&
     & .3734906E-02,.3934456E-02,.4035234E-02,.4031563E-02,.3866511E-02,&
     & .2856445E-02,.1852018E-02,.3014249E-02,.3500208E-02,.3821348E-02,&
     & .4026208E-02,.4126369E-02,.4113254E-02,.3937685E-02,.2897579E-02,&
     & .1485209E-02,.2447369E-02,.2842719E-02,.3098235E-02,.3261684E-02,&
     & .3343620E-02,.3333197E-02,.3190317E-02,.4072685E-02,.1522667E-02,&
     & .2503774E-02,.2902412E-02,.3163342E-02,.3327673E-02,.3406943E-02,&
     & .3392715E-02,.3239642E-02,.4013646E-02,.1562338E-02,.2561154E-02,&
     & .2966329E-02,.3232897E-02,.3398043E-02,.3474218E-02,.3456041E-02,&
     & .3293328E-02,.3960211E-02,.1605822E-02,.2619463E-02,.3035388E-02,&
     & .3307311E-02,.3472770E-02,.3548127E-02,.3523210E-02,.3351454E-02,&
     & .3918895E-02,.1652396E-02,.2684419E-02,.3109623E-02,.3386474E-02,&
     & .3553371E-02,.3627950E-02,.3594923E-02,.3414383E-02,.3895295E-02/

      data   absa(451:585, 4)   /                                       &
     & .1327826E-02,.2178249E-02,.2523484E-02,.2748178E-02,.2891129E-02,&
     & .2953926E-02,.2925626E-02,.2774151E-02,.5032627E-02,.1363836E-02,&
     & .2231507E-02,.2582331E-02,.2811256E-02,.2953861E-02,.3013652E-02,&
     & .2982332E-02,.2820761E-02,.4894171E-02,.1402555E-02,.2284583E-02,&
     & .2644926E-02,.2877676E-02,.3020940E-02,.3077742E-02,.3041427E-02,&
     & .2870356E-02,.4763815E-02,.1444850E-02,.2341577E-02,.2711315E-02,&
     & .2948500E-02,.3091276E-02,.3147103E-02,.3103922E-02,.2923523E-02,&
     & .4657359E-02,.1491126E-02,.2405746E-02,.2783605E-02,.3023985E-02,&
     & .3167614E-02,.3221278E-02,.3170403E-02,.2981808E-02,.4593950E-02,&
     & .1198258E-02,.1938520E-02,.2240507E-02,.2433881E-02,.2549827E-02,&
     & .2596355E-02,.2561600E-02,.2410177E-02,.5165038E-02,.1231984E-02,&
     & .1988536E-02,.2296997E-02,.2493517E-02,.2610218E-02,.2654628E-02,&
     & .2615348E-02,.2452955E-02,.4998695E-02,.1267303E-02,.2040113E-02,&
     & .2355817E-02,.2557194E-02,.2673983E-02,.2716145E-02,.2670737E-02,&
     & .2497659E-02,.4849281E-02,.1307299E-02,.2095435E-02,.2418826E-02,&
     & .2623945E-02,.2740846E-02,.2781831E-02,.2728441E-02,.2545893E-02,&
     & .4735365E-02,.1350768E-02,.2157279E-02,.2487383E-02,.2694912E-02,&
     & .2813081E-02,.2851205E-02,.2790976E-02,.2599947E-02,.4638902E-02,&
     & .1094611E-02,.1729442E-02,.1987775E-02,.2148646E-02,.2243224E-02,&
     & .2274670E-02,.2235199E-02,.2090113E-02,.4614858E-02,.1125402E-02,&
     & .1775981E-02,.2041820E-02,.2205584E-02,.2300560E-02,.2330200E-02,&
     & .2284917E-02,.2128857E-02,.4463987E-02,.1159259E-02,.1824840E-02,&
     & .2098113E-02,.2265637E-02,.2360409E-02,.2388227E-02,.2336076E-02,&
     & .2169572E-02,.4338182E-02,.1196616E-02,.1877338E-02,.2157749E-02,&
     & .2328747E-02,.2423999E-02,.2448955E-02,.2390335E-02,.2214290E-02,&
     & .4232022E-02,.1237500E-02,.1937731E-02,.2222832E-02,.2395916E-02,&
     & .2492122E-02,.2513855E-02,.2450198E-02,.2264343E-02,.4147518E-02/

      data   absa(  1:180, 5)   /                                       &
     & .7569383E-01,.7127726E-01,.7072055E-01,.7167894E-01,.7293550E-01,&
     & .7244916E-01,.7056707E-01,.6926245E-01,.7313178E-01,.7613347E-01,&
     & .7151997E-01,.7077247E-01,.7153549E-01,.7263614E-01,.7201405E-01,&
     & .7009892E-01,.6889573E-01,.7285660E-01,.7647835E-01,.7168221E-01,&
     & .7076014E-01,.7134341E-01,.7231870E-01,.7159794E-01,.6969923E-01,&
     & .6859261E-01,.7264748E-01,.7670016E-01,.7175750E-01,.7068442E-01,&
     & .7110712E-01,.7197697E-01,.7120664E-01,.6933990E-01,.6836243E-01,&
     & .7255382E-01,.7691373E-01,.7183982E-01,.7062341E-01,.7088895E-01,&
     & .7164992E-01,.7087382E-01,.6904623E-01,.6822798E-01,.7257501E-01,&
     & .7421380E-01,.7089850E-01,.7145266E-01,.7330346E-01,.7466460E-01,&
     & .7395589E-01,.7188620E-01,.7007745E-01,.7342812E-01,.7470315E-01,&
     & .7115943E-01,.7149768E-01,.7312749E-01,.7437682E-01,.7354337E-01,&
     & .7146237E-01,.6973763E-01,.7320381E-01,.7511576E-01,.7137057E-01,&
     & .7150213E-01,.7293552E-01,.7408060E-01,.7316493E-01,.7110190E-01,&
     & .6944710E-01,.7305691E-01,.7543079E-01,.7151417E-01,.7146139E-01,&
     & .7272686E-01,.7376978E-01,.7280501E-01,.7076206E-01,.6923458E-01,&
     & .7300795E-01,.7568786E-01,.7161383E-01,.7139697E-01,.7250907E-01,&
     & .7343069E-01,.7248294E-01,.7048791E-01,.6909555E-01,.7306492E-01,&
     & .7230673E-01,.7021426E-01,.7189722E-01,.7451771E-01,.7579082E-01,&
     & .7503288E-01,.7282416E-01,.7049154E-01,.7299906E-01,.7286315E-01,&
     & .7051801E-01,.7197408E-01,.7438907E-01,.7552182E-01,.7466506E-01,&
     & .7245855E-01,.7016762E-01,.7281308E-01,.7332177E-01,.7075910E-01,&
     & .7199437E-01,.7424541E-01,.7524477E-01,.7430774E-01,.7210437E-01,&
     & .6988899E-01,.7270099E-01,.7375460E-01,.7098234E-01,.7200912E-01,&
     & .7408957E-01,.7497762E-01,.7399034E-01,.7179310E-01,.6966381E-01,&
     & .7266033E-01,.7410607E-01,.7113688E-01,.7196716E-01,.7387325E-01,&
     & .7467008E-01,.7369947E-01,.7152362E-01,.6950511E-01,.7269904E-01,&
     & .7006148E-01,.6923816E-01,.7203176E-01,.7518932E-01,.7630010E-01,&
     & .7558723E-01,.7324666E-01,.7030687E-01,.7173847E-01,.7070776E-01,&
     & .6960480E-01,.7215580E-01,.7510526E-01,.7608714E-01,.7528596E-01,&
     & .7291553E-01,.7001785E-01,.7160799E-01,.7124096E-01,.6989489E-01,&
     & .7221590E-01,.7501608E-01,.7584481E-01,.7498936E-01,.7259196E-01,&
     & .6976034E-01,.7152087E-01,.7170772E-01,.7013752E-01,.7222754E-01,&
     & .7487010E-01,.7559690E-01,.7471465E-01,.7231022E-01,.6954421E-01,&
     & .7149408E-01,.7217297E-01,.7036437E-01,.7224475E-01,.7470987E-01,&
     & .7536578E-01,.7448130E-01,.7207643E-01,.6939518E-01,.7153955E-01/

      data   absa(181:315, 5)   /                                       &
     & .6746543E-01,.6789511E-01,.7175811E-01,.7512073E-01,.7623306E-01,&
     & .7554875E-01,.7299624E-01,.6947460E-01,.6972139E-01,.6817808E-01,&
     & .6831863E-01,.7191880E-01,.7511721E-01,.7607693E-01,.7533587E-01,&
     & .7272951E-01,.6921640E-01,.6962219E-01,.6878127E-01,.6866239E-01,&
     & .7201451E-01,.7507768E-01,.7588347E-01,.7512346E-01,.7245380E-01,&
     & .6898486E-01,.6955882E-01,.6931769E-01,.6894903E-01,.7206127E-01,&
     & .7499506E-01,.7568911E-01,.7490453E-01,.7221264E-01,.6878897E-01,&
     & .6955676E-01,.6984062E-01,.6921967E-01,.7210380E-01,.7490218E-01,&
     & .7552190E-01,.7471412E-01,.7202114E-01,.6866058E-01,.6962597E-01,&
     & .6453797E-01,.6619435E-01,.7096321E-01,.7435120E-01,.7557506E-01,&
     & .7489358E-01,.7213608E-01,.6797549E-01,.6697223E-01,.6531448E-01,&
     & .6667806E-01,.7120136E-01,.7443855E-01,.7550608E-01,.7477578E-01,&
     & .7192188E-01,.6776652E-01,.6691692E-01,.6599601E-01,.6708647E-01,&
     & .7137864E-01,.7446953E-01,.7539664E-01,.7463387E-01,.7169803E-01,&
     & .6757635E-01,.6690442E-01,.6658455E-01,.6741621E-01,.7148953E-01,&
     & .7446547E-01,.7527439E-01,.7447360E-01,.7150646E-01,.6741051E-01,&
     & .6692725E-01,.6710838E-01,.6769670E-01,.7154133E-01,.7442810E-01,&
     & .7515746E-01,.7430715E-01,.7133898E-01,.6729366E-01,.6700931E-01,&
     & .6127817E-01,.6414812E-01,.6958749E-01,.7293419E-01,.7429145E-01,&
     & .7361146E-01,.7059124E-01,.6583890E-01,.6355716E-01,.6212572E-01,&
     & .6469369E-01,.6993766E-01,.7312422E-01,.7433484E-01,.7357556E-01,&
     & .7045275E-01,.6569828E-01,.6357688E-01,.6288706E-01,.6516584E-01,&
     & .7021271E-01,.7326619E-01,.7432982E-01,.7348213E-01,.7031560E-01,&
     & .6555824E-01,.6360213E-01,.6354801E-01,.6556305E-01,.7040342E-01,&
     & .7334105E-01,.7430526E-01,.7337063E-01,.7018757E-01,.6542809E-01,&
     & .6365427E-01,.6412664E-01,.6589381E-01,.7053384E-01,.7337186E-01,&
     & .7426386E-01,.7326847E-01,.7006536E-01,.6533352E-01,.6375247E-01/

      data   absa(316:450, 5)   /                                       &
     & .5769446E-01,.6178260E-01,.6765439E-01,.7096937E-01,.7236097E-01,&
     & .7162253E-01,.6841692E-01,.6322137E-01,.5970130E-01,.5863799E-01,&
     & .6240216E-01,.6811545E-01,.7126851E-01,.7252091E-01,.7168532E-01,&
     & .6838432E-01,.6313347E-01,.5975507E-01,.5948073E-01,.6294022E-01,&
     & .6847433E-01,.7150605E-01,.7262755E-01,.7166820E-01,.6832826E-01,&
     & .6301598E-01,.5980255E-01,.6019384E-01,.6338337E-01,.6874609E-01,&
     & .7165133E-01,.7270983E-01,.7163959E-01,.6825300E-01,.6292421E-01,&
     & .5988086E-01,.6082893E-01,.6376171E-01,.6894902E-01,.7174796E-01,&
     & .7275284E-01,.7161814E-01,.6819849E-01,.6284853E-01,.5997885E-01,&
     & .5384218E-01,.5913297E-01,.6512641E-01,.6847022E-01,.6981558E-01,&
     & .6898244E-01,.6574451E-01,.6014063E-01,.5493742E-01,.5486164E-01,&
     & .5981373E-01,.6568933E-01,.6889734E-01,.7011656E-01,.6915288E-01,&
     & .6580592E-01,.6008501E-01,.5501675E-01,.5576632E-01,.6040455E-01,&
     & .6615529E-01,.6921938E-01,.7034252E-01,.6924287E-01,.6580255E-01,&
     & .6001888E-01,.5509485E-01,.5653718E-01,.6090010E-01,.6649772E-01,&
     & .6944333E-01,.7051698E-01,.6931819E-01,.6580285E-01,.5995294E-01,&
     & .5518924E-01,.5724006E-01,.6133632E-01,.6675529E-01,.6962363E-01,&
     & .7062688E-01,.6937140E-01,.6580880E-01,.5991082E-01,.5530789E-01,&
     & .4986911E-01,.5625720E-01,.6219195E-01,.6550333E-01,.6673087E-01,&
     & .6585114E-01,.6260934E-01,.5664518E-01,.4665462E-01,.5094960E-01,&
     & .5701871E-01,.6284240E-01,.6603147E-01,.6716332E-01,.6612920E-01,&
     & .6275302E-01,.5663929E-01,.4676883E-01,.5188210E-01,.5764984E-01,&
     & .6336343E-01,.6646529E-01,.6749570E-01,.6632743E-01,.6282719E-01,&
     & .5662832E-01,.4689733E-01,.5269455E-01,.5818682E-01,.6376950E-01,&
     & .6678498E-01,.6774711E-01,.6647465E-01,.6289082E-01,.5661653E-01,&
     & .4705226E-01,.5340814E-01,.5863202E-01,.6410648E-01,.6704487E-01,&
     & .6795408E-01,.6661171E-01,.6295124E-01,.5663255E-01,.4724050E-01/

      data   absa(451:585, 5)   /                                       &
     & .4634040E-01,.5350954E-01,.5924519E-01,.6235764E-01,.6342928E-01,&
     & .6250075E-01,.5915701E-01,.5294180E-01,.3981576E-01,.4736892E-01,&
     & .5426407E-01,.5988551E-01,.6297853E-01,.6392000E-01,.6284030E-01,&
     & .5933938E-01,.5297399E-01,.3998733E-01,.4826061E-01,.5489369E-01,&
     & .6040498E-01,.6345651E-01,.6430714E-01,.6308766E-01,.5949295E-01,&
     & .5300748E-01,.4016986E-01,.4903376E-01,.5542677E-01,.6085024E-01,&
     & .6383044E-01,.6465029E-01,.6331517E-01,.5962492E-01,.5307512E-01,&
     & .4039950E-01,.4974119E-01,.5589358E-01,.6123797E-01,.6415006E-01,&
     & .6494753E-01,.6353101E-01,.5976611E-01,.5317271E-01,.4065617E-01,&
     & .4271897E-01,.5052183E-01,.5599851E-01,.5891678E-01,.5984422E-01,&
     & .5881804E-01,.5542417E-01,.4913608E-01,.3713988E-01,.4368596E-01,&
     & .5124809E-01,.5664483E-01,.5957881E-01,.6039071E-01,.5919218E-01,&
     & .5567250E-01,.4923797E-01,.3723585E-01,.4453270E-01,.5188770E-01,&
     & .5720401E-01,.6010232E-01,.6087078E-01,.5953237E-01,.5588506E-01,&
     & .4936407E-01,.3745797E-01,.4529021E-01,.5243444E-01,.5768876E-01,&
     & .6053783E-01,.6126963E-01,.5984905E-01,.5610497E-01,.4950780E-01,&
     & .3755909E-01,.4600687E-01,.5293281E-01,.5813232E-01,.6093595E-01,&
     & .6163456E-01,.6016722E-01,.5635091E-01,.4968371E-01,.3773583E-01,&
     & .3908364E-01,.4730786E-01,.5250316E-01,.5524959E-01,.5601420E-01,&
     & .5485012E-01,.5154058E-01,.4532019E-01,.3566372E-01,.3998663E-01,&
     & .4803344E-01,.5319099E-01,.5592987E-01,.5663652E-01,.5530572E-01,&
     & .5183777E-01,.4553398E-01,.3583037E-01,.4078498E-01,.4865307E-01,&
     & .5379331E-01,.5649229E-01,.5715869E-01,.5574035E-01,.5214488E-01,&
     & .4574777E-01,.3594123E-01,.4154329E-01,.4922239E-01,.5434666E-01,&
     & .5701428E-01,.5763600E-01,.5616885E-01,.5247198E-01,.4597428E-01,&
     & .3604552E-01,.4229775E-01,.4978726E-01,.5486411E-01,.5751694E-01,&
     & .5809005E-01,.5659205E-01,.5281871E-01,.4623884E-01,.3624776E-01/

      data   absa(  1:180, 6)   /                                       &
     & .4846620E+00,.4254449E+00,.3666979E+00,.3085003E+00,.2601346E+00,&
     & .2464956E+00,.2660087E+00,.3021063E+00,.3287759E+00,.4745246E+00,&
     & .4166752E+00,.3592695E+00,.3025580E+00,.2553943E+00,.2432268E+00,&
     & .2629249E+00,.2987363E+00,.3245474E+00,.4658709E+00,.4092568E+00,&
     & .3530642E+00,.2977163E+00,.2515626E+00,.2405793E+00,.2602783E+00,&
     & .2957536E+00,.3209745E+00,.4575145E+00,.4020922E+00,.3470681E+00,&
     & .2930793E+00,.2481591E+00,.2381801E+00,.2579191E+00,.2931017E+00,&
     & .3178371E+00,.4497966E+00,.3954548E+00,.3415131E+00,.2891031E+00,&
     & .2456319E+00,.2359968E+00,.2560087E+00,.2910169E+00,.3153507E+00,&
     & .5233277E+00,.4592284E+00,.3955317E+00,.3326601E+00,.2881387E+00,&
     & .2814594E+00,.3054605E+00,.3475919E+00,.3858173E+00,.5126637E+00,&
     & .4500477E+00,.3878258E+00,.3267077E+00,.2828332E+00,.2774387E+00,&
     & .3014139E+00,.3430855E+00,.3803776E+00,.5032811E+00,.4419694E+00,&
     & .3810629E+00,.3215530E+00,.2784520E+00,.2739908E+00,.2977829E+00,&
     & .3391699E+00,.3757057E+00,.4943912E+00,.4343294E+00,.3746578E+00,&
     & .3166441E+00,.2747404E+00,.2708931E+00,.2946970E+00,.3358529E+00,&
     & .3717479E+00,.4856925E+00,.4268266E+00,.3684341E+00,.3122038E+00,&
     & .2717769E+00,.2679905E+00,.2921861E+00,.3332832E+00,.3686250E+00,&
     & .5659721E+00,.4964838E+00,.4273442E+00,.3613333E+00,.3229778E+00,&
     & .3207717E+00,.3487909E+00,.3970188E+00,.4483696E+00,.5545946E+00,&
     & .4866491E+00,.4190737E+00,.3544937E+00,.3171184E+00,.3157821E+00,&
     & .3434204E+00,.3913987E+00,.4416557E+00,.5441901E+00,.4776455E+00,&
     & .4115043E+00,.3481950E+00,.3119787E+00,.3113749E+00,.3389006E+00,&
     & .3865528E+00,.4358447E+00,.5345476E+00,.4693294E+00,.4045767E+00,&
     & .3427926E+00,.3075766E+00,.3073312E+00,.3350898E+00,.3825859E+00,&
     & .4311541E+00,.5246854E+00,.4609334E+00,.3978727E+00,.3380061E+00,&
     & .3038003E+00,.3036818E+00,.3320521E+00,.3795333E+00,.4274614E+00,&
     & .6114156E+00,.5360856E+00,.4610841E+00,.3944939E+00,.3631225E+00,&
     & .3640797E+00,.3957193E+00,.4512458E+00,.5140119E+00,.5990211E+00,&
     & .5253614E+00,.4520735E+00,.3868300E+00,.3563118E+00,.3576976E+00,&
     & .3893012E+00,.4443146E+00,.5057705E+00,.5878031E+00,.5156587E+00,&
     & .4439389E+00,.3797822E+00,.3504051E+00,.3520245E+00,.3838216E+00,&
     & .4383993E+00,.4987384E+00,.5771932E+00,.5065469E+00,.4365968E+00,&
     & .3737334E+00,.3451671E+00,.3469143E+00,.3791776E+00,.4335158E+00,&
     & .4929659E+00,.5672493E+00,.4980854E+00,.4298759E+00,.3686864E+00,&
     & .3406738E+00,.3425037E+00,.3753823E+00,.4296076E+00,.4882770E+00/

      data   absa(181:315, 6)   /                                       &
     & .6599457E+00,.5784248E+00,.4971773E+00,.4337450E+00,.4071444E+00,&
     & .4110564E+00,.4472427E+00,.5099709E+00,.5830348E+00,.6468482E+00,&
     & .5670871E+00,.4876631E+00,.4250777E+00,.3993382E+00,.4031458E+00,&
     & .4395792E+00,.5018776E+00,.5735256E+00,.6348396E+00,.5566698E+00,&
     & .4790978E+00,.4172804E+00,.3925206E+00,.3961110E+00,.4330389E+00,&
     & .4948939E+00,.5653109E+00,.6234166E+00,.5469278E+00,.4713504E+00,&
     & .4102887E+00,.3864066E+00,.3900014E+00,.4273461E+00,.4889966E+00,&
     & .5583254E+00,.6128604E+00,.5379244E+00,.4642610E+00,.4042901E+00,&
     & .3809946E+00,.3847891E+00,.4225472E+00,.4840732E+00,.5525507E+00,&
     & .7118405E+00,.6237026E+00,.5373473E+00,.4788483E+00,.4552052E+00,&
     & .4622830E+00,.5030364E+00,.5735317E+00,.6569249E+00,.6980451E+00,&
     & .6117423E+00,.5269042E+00,.4690257E+00,.4461057E+00,.4528737E+00,&
     & .4940744E+00,.5641178E+00,.6458719E+00,.6851401E+00,.6005735E+00,&
     & .5173938E+00,.4602048E+00,.4380661E+00,.4445391E+00,.4863645E+00,&
     & .5558444E+00,.6361244E+00,.6731826E+00,.5903092E+00,.5088902E+00,&
     & .4521990E+00,.4308774E+00,.4373808E+00,.4795420E+00,.5488504E+00,&
     & .6279339E+00,.6618869E+00,.5806518E+00,.5011956E+00,.4451372E+00,&
     & .4243261E+00,.4313958E+00,.4738124E+00,.5429837E+00,.6210544E+00,&
     & .7668972E+00,.6717524E+00,.5822052E+00,.5289402E+00,.5074279E+00,&
     & .5174762E+00,.5633483E+00,.6412958E+00,.7351146E+00,.7523724E+00,&
     & .6591380E+00,.5705110E+00,.5178052E+00,.4967826E+00,.5066462E+00,&
     & .5528830E+00,.6302350E+00,.7222090E+00,.7385919E+00,.6472212E+00,&
     & .5598496E+00,.5075245E+00,.4872782E+00,.4974170E+00,.5435990E+00,&
     & .6206660E+00,.7110675E+00,.7257857E+00,.6361589E+00,.5503268E+00,&
     & .4984685E+00,.4786719E+00,.4893603E+00,.5354629E+00,.6125751E+00,&
     & .7016399E+00,.7138973E+00,.6259486E+00,.5417002E+00,.4905437E+00,&
     & .4710327E+00,.4822383E+00,.5286586E+00,.6057803E+00,.6935878E+00/

      data   absa(316:450, 6)   /                                       &
     & .8249134E+00,.7223579E+00,.6318367E+00,.5828461E+00,.5642052E+00,&
     & .5774889E+00,.6274321E+00,.7112179E+00,.8154752E+00,.8094743E+00,&
     & .7089425E+00,.6188253E+00,.5703294E+00,.5519242E+00,.5651410E+00,&
     & .6152177E+00,.6988740E+00,.8012054E+00,.7948100E+00,.6962635E+00,&
     & .6071335E+00,.5587642E+00,.5408447E+00,.5546817E+00,.6044782E+00,&
     & .6883629E+00,.7888678E+00,.7814861E+00,.6847483E+00,.5966332E+00,&
     & .5488036E+00,.5307506E+00,.5452968E+00,.5952041E+00,.6791251E+00,&
     & .7781169E+00,.7689261E+00,.6739251E+00,.5871415E+00,.5399440E+00,&
     & .5219107E+00,.5368668E+00,.5871449E+00,.6714540E+00,.7691532E+00,&
     & .8852145E+00,.7750456E+00,.6871055E+00,.6408984E+00,.6254709E+00,&
     & .6421647E+00,.6941504E+00,.7835789E+00,.8985979E+00,.8691028E+00,&
     & .7609767E+00,.6728330E+00,.6267061E+00,.6112664E+00,.6281600E+00,&
     & .6803225E+00,.7701312E+00,.8830032E+00,.8539650E+00,.7478668E+00,&
     & .6598673E+00,.6140672E+00,.5985796E+00,.6161528E+00,.6684674E+00,&
     & .7583442E+00,.8692469E+00,.8400620E+00,.7358241E+00,.6484499E+00,&
     & .6030312E+00,.5871483E+00,.6051785E+00,.6578534E+00,.7481252E+00,&
     & .8573340E+00,.8267308E+00,.7243453E+00,.6382877E+00,.5930271E+00,&
     & .5773599E+00,.5955257E+00,.6486068E+00,.7393848E+00,.8471526E+00,&
     & .9457456E+00,.8285939E+00,.7451511E+00,.7019917E+00,.6899693E+00,&
     & .7089637E+00,.7625561E+00,.8572460E+00,.9824613E+00,.9291323E+00,&
     & .8136971E+00,.7298541E+00,.6864425E+00,.6740084E+00,.6934007E+00,&
     & .7472875E+00,.8425945E+00,.9655705E+00,.9139136E+00,.8002614E+00,&
     & .7162229E+00,.6723573E+00,.6598915E+00,.6797955E+00,.7340436E+00,&
     & .8294611E+00,.9506098E+00,.8996396E+00,.7878323E+00,.7040173E+00,&
     & .6600638E+00,.6474290E+00,.6677087E+00,.7222527E+00,.8180613E+00,&
     & .9373243E+00,.8863343E+00,.7763696E+00,.6928464E+00,.6489803E+00,&
     & .6362699E+00,.6568098E+00,.7118980E+00,.8079921E+00,.9257131E+00/

      data   absa(451:585, 6)   /                                       &
     & .9984509E+00,.8767023E+00,.7981236E+00,.7590051E+00,.7500479E+00,&
     & .7704148E+00,.8258274E+00,.9236821E+00,.1057116E+01,.9825580E+00,&
     & .8620354E+00,.7830513E+00,.7424078E+00,.7334532E+00,.7542144E+00,&
     & .8101460E+00,.9086138E+00,.1040043E+01,.9678310E+00,.8486729E+00,&
     & .7694638E+00,.7279026E+00,.7187504E+00,.7400787E+00,.7961034E+00,&
     & .8951540E+00,.1024689E+01,.9541401E+00,.8363966E+00,.7569500E+00,&
     & .7151152E+00,.7053524E+00,.7272043E+00,.7836771E+00,.8829254E+00,&
     & .1010610E+01,.9409238E+00,.8247336E+00,.7452605E+00,.7034806E+00,&
     & .6932724E+00,.7155504E+00,.7725713E+00,.8720814E+00,.9982035E+00,&
     & .1051209E+01,.9265442E+00,.8532619E+00,.8179703E+00,.8119095E+00,&
     & .8341371E+00,.8903756E+00,.9888756E+00,.1095956E+01,.1036235E+01,&
     & .9123511E+00,.8382101E+00,.8010032E+00,.7947542E+00,.8176096E+00,&
     & .8740008E+00,.9731309E+00,.1080421E+01,.1022216E+01,.8989889E+00,&
     & .8242808E+00,.7860446E+00,.7790076E+00,.8024437E+00,.8593689E+00,&
     & .9586866E+00,.1065303E+01,.1008831E+01,.8865583E+00,.8113619E+00,&
     & .7726307E+00,.7650415E+00,.7885957E+00,.8460657E+00,.9457463E+00,&
     & .1053284E+01,.9956237E+00,.8746029E+00,.7991663E+00,.7602091E+00,&
     & .7523241E+00,.7758775E+00,.8338469E+00,.9341164E+00,.1042323E+01,&
     & .1103094E+01,.9779325E+00,.9101471E+00,.8783812E+00,.8752086E+00,&
     & .8997168E+00,.9549270E+00,.1052106E+01,.1130777E+01,.1089042E+01,&
     & .9638679E+00,.8946771E+00,.8612419E+00,.8571856E+00,.8822649E+00,&
     & .9380724E+00,.1035137E+01,.1114750E+01,.1075742E+01,.9508309E+00,&
     & .8802630E+00,.8459304E+00,.8411000E+00,.8660581E+00,.9224269E+00,&
     & .1019820E+01,.1101053E+01,.1062410E+01,.9382231E+00,.8666081E+00,&
     & .8315446E+00,.8263075E+00,.8509882E+00,.9079071E+00,.1005967E+01,&
     & .1089316E+01,.1048876E+01,.9255641E+00,.8536249E+00,.8179717E+00,&
     & .8126191E+00,.8371360E+00,.8945910E+00,.9933411E+00,.1078307E+01/

!     the array absb(235,6) (kb(5,13:59,6)) contains absorption coefs at
!     the 16 chosen g-values for a range of pressure levels < ~100mb and
!     temperatures. the first index in the array, jt, which runs from 1 to 5,
!     corresponds to different temperatures.  more specifically, jt = 3 means
!     that the data are for the reference temperature tref for this pressure
!     level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!     tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!     the second index, jp, runs from 13 to 59 and refers to the jpth
!     reference pressure level (see taumol.f for the value of these
!     pressure levels in mb).  the third index, ig, goes from 1 to 6,
!     and tells us which g-interval the absorption coefficients are for.
!     -----------------------------------------------------------------

      data   absb(  1:120, 1)   /                                       &
     & .2455108E-02,.2494284E-02,.2520068E-02,.2540222E-02,.2547453E-02,&
     & .2053818E-02,.2083023E-02,.2101215E-02,.2116145E-02,.2123172E-02,&
     & .1734639E-02,.1758680E-02,.1772416E-02,.1781767E-02,.1788224E-02,&
     & .1482314E-02,.1503678E-02,.1512899E-02,.1519630E-02,.1524757E-02,&
     & .1284745E-02,.1304855E-02,.1312793E-02,.1316311E-02,.1318005E-02,&
     & .1129086E-02,.1146597E-02,.1156453E-02,.1156435E-02,.1156401E-02,&
     & .9921713E-03,.1008990E-02,.1018291E-02,.1020213E-02,.1018365E-02,&
     & .8471850E-03,.8618578E-03,.8693996E-03,.8719597E-03,.8699465E-03,&
     & .7178765E-03,.7312448E-03,.7368809E-03,.7383122E-03,.7377363E-03,&
     & .6069271E-03,.6192294E-03,.6230026E-03,.6249060E-03,.6240653E-03,&
     & .5123512E-03,.5240166E-03,.5274574E-03,.5289784E-03,.5283974E-03,&
     & .4327672E-03,.4433865E-03,.4469695E-03,.4481477E-03,.4479432E-03,&
     & .3651769E-03,.3741862E-03,.3784929E-03,.3797155E-03,.3799202E-03,&
     & .3081802E-03,.3145936E-03,.3193752E-03,.3209357E-03,.3216224E-03,&
     & .2600564E-03,.2641825E-03,.2692918E-03,.2713207E-03,.2719491E-03,&
     & .2189889E-03,.2226081E-03,.2271490E-03,.2294557E-03,.2302383E-03,&
     & .1846009E-03,.1885356E-03,.1916569E-03,.1944705E-03,.1953784E-03,&
     & .1569872E-03,.1601747E-03,.1622966E-03,.1649634E-03,.1666332E-03,&
     & .1341940E-03,.1363878E-03,.1381676E-03,.1401546E-03,.1421984E-03,&
     & .1150202E-03,.1162012E-03,.1182134E-03,.1193516E-03,.1212431E-03,&
     & .9900466E-04,.9989409E-04,.1008343E-03,.1022803E-03,.1033529E-03,&
     & .8523323E-04,.8592559E-04,.8646982E-04,.8752563E-04,.8834547E-04,&
     & .7355247E-04,.7410040E-04,.7455079E-04,.7505627E-04,.7561834E-04,&
     & .6386391E-04,.6399156E-04,.6452062E-04,.6507966E-04,.6511678E-04/

      data   absb(121:235, 1)   /                                       &
     & .5378377E-04,.5383807E-04,.5419748E-04,.5461584E-04,.5475942E-04,&
     & .4529476E-04,.4536607E-04,.4551849E-04,.4582121E-04,.4594236E-04,&
     & .3805002E-04,.3835500E-04,.3837164E-04,.3841587E-04,.3870224E-04,&
     & .3134721E-04,.3153158E-04,.3158800E-04,.3159241E-04,.3173210E-04,&
     & .2572678E-04,.2581660E-04,.2592636E-04,.2593449E-04,.2604293E-04,&
     & .2111340E-04,.2114297E-04,.2123906E-04,.2126974E-04,.2132702E-04,&
     & .1724042E-04,.1721678E-04,.1726492E-04,.1735856E-04,.1734777E-04,&
     & .1400853E-04,.1398539E-04,.1400256E-04,.1407486E-04,.1407744E-04,&
     & .1138599E-04,.1138052E-04,.1136754E-04,.1138325E-04,.1144462E-04,&
     & .9234875E-05,.9228129E-05,.9225511E-05,.9219129E-05,.9274721E-05,&
     & .7450018E-05,.7494669E-05,.7491514E-05,.7465424E-05,.7498018E-05,&
     & .5987641E-05,.6060252E-05,.6077441E-05,.6062453E-05,.6062401E-05,&
     & .4810640E-05,.4877970E-05,.4932441E-05,.4916522E-05,.4908248E-05,&
     & .3869095E-05,.3916912E-05,.3981052E-05,.3989636E-05,.3985085E-05,&
     & .3125584E-05,.3158316E-05,.3201951E-05,.3234304E-05,.3229423E-05,&
     & .2526771E-05,.2539982E-05,.2565764E-05,.2606671E-05,.2620099E-05,&
     & .2033741E-05,.2048230E-05,.2064945E-05,.2094217E-05,.2117279E-05,&
     & .1636949E-05,.1651544E-05,.1664769E-05,.1682082E-05,.1707938E-05,&
     & .1320644E-05,.1329014E-05,.1342957E-05,.1355072E-05,.1374367E-05,&
     & .1063144E-05,.1069247E-05,.1082985E-05,.1092178E-05,.1104704E-05,&
     & .8523022E-06,.8613254E-06,.8703544E-06,.8805758E-06,.8889748E-06,&
     & .6821306E-06,.6939226E-06,.7000980E-06,.7103676E-06,.7161581E-06,&
     & .5473000E-06,.5610035E-06,.5662359E-06,.5739016E-06,.5795604E-06/

      data   absb(  1:120, 2)   /                                       &
     & .1286743E-01,.1298016E-01,.1307688E-01,.1314196E-01,.1318687E-01,&
     & .1084483E-01,.1095286E-01,.1104372E-01,.1107916E-01,.1110333E-01,&
     & .9173853E-02,.9249031E-02,.9308829E-02,.9349446E-02,.9373813E-02,&
     & .7773824E-02,.7815445E-02,.7855507E-02,.7887027E-02,.7894494E-02,&
     & .6635671E-02,.6658973E-02,.6677297E-02,.6693544E-02,.6694290E-02,&
     & .5714053E-02,.5714835E-02,.5716276E-02,.5728622E-02,.5719424E-02,&
     & .4953027E-02,.4935260E-02,.4928449E-02,.4921707E-02,.4912914E-02,&
     & .4202051E-02,.4179936E-02,.4170445E-02,.4154196E-02,.4145879E-02,&
     & .3546756E-02,.3522656E-02,.3509107E-02,.3496226E-02,.3483466E-02,&
     & .2986324E-02,.2962134E-02,.2946586E-02,.2935316E-02,.2924589E-02,&
     & .2520037E-02,.2493976E-02,.2477958E-02,.2467618E-02,.2459527E-02,&
     & .2128530E-02,.2102617E-02,.2086920E-02,.2077359E-02,.2068955E-02,&
     & .1796044E-02,.1774099E-02,.1757802E-02,.1748933E-02,.1740649E-02,&
     & .1511042E-02,.1493208E-02,.1478896E-02,.1469321E-02,.1461744E-02,&
     & .1272579E-02,.1259504E-02,.1246012E-02,.1235685E-02,.1229631E-02,&
     & .1074471E-02,.1063160E-02,.1050792E-02,.1041724E-02,.1036092E-02,&
     & .9106300E-03,.8993222E-03,.8888252E-03,.8805708E-03,.8768352E-03,&
     & .7732921E-03,.7632043E-03,.7552638E-03,.7485823E-03,.7491950E-03,&
     & .6597355E-03,.6512362E-03,.6451463E-03,.6404871E-03,.6392845E-03,&
     & .5654918E-03,.5596233E-03,.5545424E-03,.5500160E-03,.5486305E-03,&
     & .4880680E-03,.4838686E-03,.4792352E-03,.4746633E-03,.4742358E-03,&
     & .4240329E-03,.4198366E-03,.4158480E-03,.4147475E-03,.4118226E-03,&
     & .3707047E-03,.3670166E-03,.3635404E-03,.3627127E-03,.3598610E-03,&
     & .3269270E-03,.3238737E-03,.3206578E-03,.3192890E-03,.3166364E-03/

      data   absb(121:235, 2)   /                                       &
     & .2789970E-03,.2765079E-03,.2739652E-03,.2724815E-03,.2694007E-03,&
     & .2383170E-03,.2362195E-03,.2338966E-03,.2321906E-03,.2293814E-03,&
     & .2043421E-03,.2022653E-03,.1998851E-03,.1974199E-03,.1958820E-03,&
     & .1703638E-03,.1685921E-03,.1664446E-03,.1644332E-03,.1632693E-03,&
     & .1414648E-03,.1398222E-03,.1381078E-03,.1364105E-03,.1354620E-03,&
     & .1171714E-03,.1160394E-03,.1146163E-03,.1132446E-03,.1124903E-03,&
     & .9653894E-04,.9566019E-04,.9459747E-04,.9340503E-04,.9220628E-04,&
     & .7932991E-04,.7856317E-04,.7774927E-04,.7682321E-04,.7581347E-04,&
     & .6515421E-04,.6441758E-04,.6392018E-04,.6313941E-04,.6221003E-04,&
     & .5347743E-04,.5297072E-04,.5245890E-04,.5184866E-04,.5107466E-04,&
     & .4396154E-04,.4347687E-04,.4303733E-04,.4259063E-04,.4196826E-04,&
     & .3615915E-04,.3568995E-04,.3529007E-04,.3492298E-04,.3444102E-04,&
     & .2966497E-04,.2925157E-04,.2889217E-04,.2855828E-04,.2821985E-04,&
     & .2430508E-04,.2403992E-04,.2370451E-04,.2340581E-04,.2311943E-04,&
     & .1995845E-04,.1971625E-04,.1943619E-04,.1919411E-04,.1893202E-04,&
     & .1630421E-04,.1613833E-04,.1595861E-04,.1573142E-04,.1549637E-04,&
     & .1329543E-04,.1321689E-04,.1306432E-04,.1287719E-04,.1268925E-04,&
     & .1082302E-04,.1082655E-04,.1071501E-04,.1058492E-04,.1042620E-04,&
     & .8829908E-05,.8878449E-05,.8804000E-05,.8713675E-05,.8582871E-05,&
     & .7229747E-05,.7274906E-05,.7243006E-05,.7161434E-05,.7060580E-05,&
     & .5920720E-05,.5930473E-05,.5943800E-05,.5878317E-05,.5803165E-05,&
     & .4850374E-05,.4842730E-05,.4872181E-05,.4827114E-05,.4773736E-05,&
     & .3998381E-05,.3995704E-05,.4020867E-05,.3990354E-05,.3948657E-05/

      data   absb(  1:120, 3)   /                                       &
     & .5002398E-01,.5100799E-01,.5223444E-01,.5366230E-01,.5534006E-01,&
     & .4416861E-01,.4512229E-01,.4624963E-01,.4763751E-01,.4926106E-01,&
     & .3889418E-01,.3989178E-01,.4102991E-01,.4233451E-01,.4377504E-01,&
     & .3413273E-01,.3507125E-01,.3618746E-01,.3742424E-01,.3870722E-01,&
     & .2995928E-01,.3086478E-01,.3189959E-01,.3307320E-01,.3425277E-01,&
     & .2638019E-01,.2725271E-01,.2821585E-01,.2927539E-01,.3033342E-01,&
     & .2335920E-01,.2414637E-01,.2503317E-01,.2599890E-01,.2699514E-01,&
     & .2029208E-01,.2105095E-01,.2183638E-01,.2270657E-01,.2359078E-01,&
     & .1752887E-01,.1822677E-01,.1893693E-01,.1971216E-01,.2049024E-01,&
     & .1514301E-01,.1576265E-01,.1644949E-01,.1710856E-01,.1777047E-01,&
     & .1307907E-01,.1369568E-01,.1429327E-01,.1484508E-01,.1540994E-01,&
     & .1135002E-01,.1190407E-01,.1240533E-01,.1289830E-01,.1336006E-01,&
     & .9861342E-02,.1032874E-01,.1078971E-01,.1119067E-01,.1159988E-01,&
     & .8543786E-02,.8972343E-02,.9361670E-02,.9707599E-02,.1007270E-01,&
     & .7399186E-02,.7765826E-02,.8086950E-02,.8410397E-02,.8722569E-02,&
     & .6411375E-02,.6715139E-02,.6991916E-02,.7280002E-02,.7586032E-02,&
     & .5526867E-02,.5785983E-02,.6041283E-02,.6303233E-02,.6579648E-02,&
     & .4774311E-02,.4997352E-02,.5227375E-02,.5480339E-02,.5877541E-02,&
     & .4116828E-02,.4325497E-02,.4545463E-02,.4770394E-02,.5153971E-02,&
     & .3578769E-02,.3764998E-02,.3969666E-02,.4197341E-02,.4545465E-02,&
     & .3124717E-02,.3305659E-02,.3500480E-02,.3709794E-02,.4038550E-02,&
     & .2748728E-02,.2923454E-02,.3112152E-02,.3399419E-02,.3626701E-02,&
     & .2422516E-02,.2586643E-02,.2766372E-02,.3049770E-02,.3276646E-02,&
     & .2131791E-02,.2289101E-02,.2457066E-02,.2736600E-02,.2953943E-02/

      data   absb(121:235, 3)   /                                       &
     & .1828665E-02,.1969373E-02,.2131950E-02,.2386197E-02,.2591184E-02,&
     & .1565325E-02,.1696357E-02,.1846360E-02,.2083995E-02,.2278216E-02,&
     & .1339532E-02,.1462299E-02,.1603891E-02,.1761979E-02,.2010098E-02,&
     & .1123112E-02,.1231996E-02,.1361344E-02,.1506360E-02,.1731174E-02,&
     & .9386379E-03,.1036085E-02,.1150679E-02,.1280160E-02,.1487488E-02,&
     & .7829286E-03,.8681007E-03,.9712611E-03,.1088601E-02,.1279327E-02,&
     & .6439906E-03,.7179533E-03,.8068447E-03,.9121069E-03,.1038346E-02,&
     & .5251100E-03,.5895043E-03,.6665253E-03,.7582909E-03,.8728263E-03,&
     & .4272101E-03,.4822133E-03,.5483728E-03,.6283465E-03,.7297410E-03,&
     & .3455385E-03,.3919227E-03,.4490025E-03,.5186610E-03,.6080615E-03,&
     & .2772875E-03,.3163585E-03,.3648993E-03,.4245665E-03,.5029312E-03,&
     & .2215243E-03,.2545308E-03,.2950909E-03,.3456400E-03,.4140863E-03,&
     & .1762697E-03,.2034714E-03,.2368499E-03,.2802197E-03,.3391266E-03,&
     & .1409510E-03,.1632263E-03,.1911441E-03,.2283913E-03,.2789978E-03,&
     & .1124638E-03,.1310401E-03,.1540205E-03,.1857656E-03,.2289863E-03,&
     & .8954236E-04,.1047453E-03,.1237097E-03,.1505540E-03,.1870030E-03,&
     & .7098317E-04,.8306387E-04,.9912454E-04,.1212505E-03,.1519558E-03,&
     & .5681762E-04,.6665919E-04,.7982178E-04,.9805318E-04,.1239971E-03,&
     & .4562520E-04,.5359104E-04,.6438194E-04,.7935790E-04,.1011203E-03,&
     & .3653132E-04,.4295380E-04,.5170112E-04,.6400146E-04,.8208662E-04,&
     & .2913090E-04,.3439039E-04,.4140146E-04,.5133409E-04,.6632514E-04,&
     & .2332055E-04,.2755437E-04,.3319705E-04,.4115994E-04,.5350282E-04,&
     & .1945717E-04,.2306888E-04,.2793905E-04,.3476686E-04,.4551954E-04/

      data   absb(  1:120, 4)   /                                       &
     & .2878234E+00,.2961252E+00,.3050441E+00,.3147630E+00,.3253959E+00,&
     & .2743822E+00,.2825368E+00,.2911238E+00,.3003062E+00,.3104320E+00,&
     & .2632621E+00,.2715744E+00,.2807089E+00,.2905758E+00,.3014280E+00,&
     & .2470711E+00,.2556028E+00,.2648564E+00,.2753382E+00,.2872254E+00,&
     & .2312754E+00,.2397378E+00,.2488918E+00,.2593077E+00,.2714135E+00,&
     & .2159198E+00,.2248574E+00,.2346644E+00,.2452257E+00,.2571951E+00,&
     & .2025752E+00,.2115209E+00,.2212761E+00,.2319679E+00,.2443755E+00,&
     & .1870918E+00,.1958476E+00,.2056213E+00,.2168964E+00,.2297077E+00,&
     & .1719062E+00,.1808017E+00,.1908654E+00,.2024728E+00,.2155574E+00,&
     & .1580850E+00,.1674409E+00,.1780281E+00,.1900585E+00,.2031710E+00,&
     & .1462403E+00,.1558822E+00,.1666091E+00,.1789235E+00,.1922377E+00,&
     & .1362161E+00,.1459406E+00,.1567969E+00,.1692597E+00,.1829005E+00,&
     & .1274087E+00,.1370931E+00,.1484132E+00,.1609035E+00,.1746367E+00,&
     & .1194990E+00,.1297509E+00,.1410742E+00,.1539492E+00,.1679681E+00,&
     & .1123619E+00,.1230023E+00,.1347556E+00,.1477261E+00,.1617005E+00,&
     & .1061193E+00,.1170526E+00,.1291360E+00,.1421656E+00,.1564595E+00,&
     & .1006760E+00,.1119479E+00,.1241710E+00,.1376384E+00,.1522244E+00,&
     & .9607904E-01,.1075948E+00,.1202361E+00,.1342448E+00,.1520624E+00,&
     & .9236167E-01,.1042029E+00,.1173986E+00,.1321571E+00,.1502113E+00,&
     & .8989190E-01,.1021932E+00,.1159955E+00,.1310628E+00,.1496169E+00,&
     & .8834869E-01,.1013779E+00,.1156618E+00,.1310736E+00,.1499564E+00,&
     & .8767587E-01,.1011140E+00,.1157497E+00,.1341003E+00,.1509449E+00,&
     & .8642676E-01,.1002241E+00,.1151696E+00,.1339998E+00,.1509668E+00,&
     & .8444357E-01,.9839768E-01,.1137899E+00,.1328639E+00,.1500538E+00/

      data   absb(121:235, 4)   /                                       &
     & .8008552E-01,.9397404E-01,.1092312E+00,.1284086E+00,.1456923E+00,&
     & .7606488E-01,.8974941E-01,.1049552E+00,.1241503E+00,.1414966E+00,&
     & .7241439E-01,.8585292E-01,.1009126E+00,.1172255E+00,.1376072E+00,&
     & .6737037E-01,.8040867E-01,.9512233E-01,.1111766E+00,.1313617E+00,&
     & .6249839E-01,.7518420E-01,.8941538E-01,.1051429E+00,.1250058E+00,&
     & .5796755E-01,.7023093E-01,.8399825E-01,.9941523E-01,.1191081E+00,&
     & .5290880E-01,.6463286E-01,.7799108E-01,.9283777E-01,.1092238E+00,&
     & .4784261E-01,.5904001E-01,.7178305E-01,.8613347E-01,.1019882E+00,&
     & .4314406E-01,.5383664E-01,.6605552E-01,.7975409E-01,.9510036E-01,&
     & .3853322E-01,.4870478E-01,.6029343E-01,.7337748E-01,.8815869E-01,&
     & .3389054E-01,.4343173E-01,.5438033E-01,.6682346E-01,.8091510E-01,&
     & .2963117E-01,.3852479E-01,.4883852E-01,.6062832E-01,.7400013E-01,&
     & .2572970E-01,.3391246E-01,.4359826E-01,.5471276E-01,.6736039E-01,&
     & .2233416E-01,.2991585E-01,.3901273E-01,.4954145E-01,.6156569E-01,&
     & .1939210E-01,.2638719E-01,.3485100E-01,.4478336E-01,.5620570E-01,&
     & .1668291E-01,.2307789E-01,.3098475E-01,.4031918E-01,.5116350E-01,&
     & .1420262E-01,.2005395E-01,.2735039E-01,.3608786E-01,.4625548E-01,&
     & .1217748E-01,.1754282E-01,.2430328E-01,.3256355E-01,.4217429E-01,&
     & .1044921E-01,.1536327E-01,.2163899E-01,.2944083E-01,.3855156E-01,&
     & .8884509E-02,.1335574E-01,.1918440E-01,.2646426E-01,.3516363E-01,&
     & .7483675E-02,.1152870E-01,.1691180E-01,.2368487E-01,.3190062E-01,&
     & .6305579E-02,.9961440E-02,.1489973E-01,.2122728E-01,.2897708E-01,&
     & .5828592E-02,.9332797E-02,.1413857E-01,.2030539E-01,.2791717E-01/

      data   absb(  1:120, 5)   /                                       &
     & .9803355E+01,.1002921E+02,.1022928E+02,.1041986E+02,.1060893E+02,&
     & .8901187E+01,.9109354E+01,.9307795E+01,.9507481E+01,.9704484E+01,&
     & .8037550E+01,.8238379E+01,.8440990E+01,.8639971E+01,.8841430E+01,&
     & .7256235E+01,.7456728E+01,.7656282E+01,.7858569E+01,.8063361E+01,&
     & .6549121E+01,.6747154E+01,.6948541E+01,.7154546E+01,.7363018E+01,&
     & .5920127E+01,.6112982E+01,.6313644E+01,.6520192E+01,.6735708E+01,&
     & .5391811E+01,.5564840E+01,.5751524E+01,.5963164E+01,.6187711E+01,&
     & .4979063E+01,.5141211E+01,.5319204E+01,.5518536E+01,.5724852E+01,&
     & .4685729E+01,.4829264E+01,.4999371E+01,.5176536E+01,.5377588E+01,&
     & .4509733E+01,.4627971E+01,.4769025E+01,.4939138E+01,.5138382E+01,&
     & .4441848E+01,.4547145E+01,.4654076E+01,.4777266E+01,.4966192E+01,&
     & .4392672E+01,.4508636E+01,.4631472E+01,.4750610E+01,.4893241E+01,&
     & .4379451E+01,.4509691E+01,.4639368E+01,.4771997E+01,.4917485E+01,&
     & .4383266E+01,.4532070E+01,.4679856E+01,.4811910E+01,.4982307E+01,&
     & .4373711E+01,.4538212E+01,.4701877E+01,.4859289E+01,.5032829E+01,&
     & .4371956E+01,.4552831E+01,.4722242E+01,.4907609E+01,.5087424E+01,&
     & .4332160E+01,.4518631E+01,.4697783E+01,.4891344E+01,.5092684E+01,&
     & .4305857E+01,.4488111E+01,.4683919E+01,.4875401E+01,.5098906E+01,&
     & .4247319E+01,.4442550E+01,.4629624E+01,.4824728E+01,.5055952E+01,&
     & .4214744E+01,.4395695E+01,.4595960E+01,.4805492E+01,.5017792E+01,&
     & .4194067E+01,.4374577E+01,.4586487E+01,.4793725E+01,.5028709E+01,&
     & .4199133E+01,.4388763E+01,.4601898E+01,.4846199E+01,.5088486E+01,&
     & .4205331E+01,.4415572E+01,.4636671E+01,.4889061E+01,.5185128E+01,&
     & .4239611E+01,.4448023E+01,.4670143E+01,.4942226E+01,.5271353E+01/

      data   absb(121:235, 5)   /                                       &
     & .4176844E+01,.4378265E+01,.4607587E+01,.4898813E+01,.5254033E+01,&
     & .4099489E+01,.4293386E+01,.4533269E+01,.4877379E+01,.5230256E+01,&
     & .4007521E+01,.4213195E+01,.4480775E+01,.4835666E+01,.5209135E+01,&
     & .3883977E+01,.4090925E+01,.4376746E+01,.4733176E+01,.5114008E+01,&
     & .3759955E+01,.3970704E+01,.4272560E+01,.4618469E+01,.4997992E+01,&
     & .3631179E+01,.3846064E+01,.4161525E+01,.4505341E+01,.4888289E+01,&
     & .3481107E+01,.3699160E+01,.4022359E+01,.4359341E+01,.4726661E+01,&
     & .3305633E+01,.3543449E+01,.3873804E+01,.4204009E+01,.4566058E+01,&
     & .3130039E+01,.3398800E+01,.3728888E+01,.4056038E+01,.4401957E+01,&
     & .2968975E+01,.3253192E+01,.3573124E+01,.3904086E+01,.4239970E+01,&
     & .2811010E+01,.3105040E+01,.3413054E+01,.3741891E+01,.4067138E+01,&
     & .2663058E+01,.2951303E+01,.3256042E+01,.3581236E+01,.3900989E+01,&
     & .2515451E+01,.2806289E+01,.3104377E+01,.3428028E+01,.3740708E+01,&
     & .2378861E+01,.2672260E+01,.2967412E+01,.3283792E+01,.3592902E+01,&
     & .2250954E+01,.2549314E+01,.2842896E+01,.3146414E+01,.3454348E+01,&
     & .2126384E+01,.2425968E+01,.2718676E+01,.3012722E+01,.3313565E+01,&
     & .2000566E+01,.2300064E+01,.2594150E+01,.2881717E+01,.3180487E+01,&
     & .1893973E+01,.2183036E+01,.2478401E+01,.2762266E+01,.3053944E+01,&
     & .1797191E+01,.2073967E+01,.2365713E+01,.2650269E+01,.2928594E+01,&
     & .1699063E+01,.1967053E+01,.2254958E+01,.2539431E+01,.2812870E+01,&
     & .1602483E+01,.1870209E+01,.2142381E+01,.2424027E+01,.2698355E+01,&
     & .1508607E+01,.1774262E+01,.2038629E+01,.2313372E+01,.2585830E+01,&
     & .1519868E+01,.1744553E+01,.1991354E+01,.2260467E+01,.2531035E+01/

      data   absb(  1:120, 6)   /                                       &
     & .2781168E+03,.2745845E+03,.2712239E+03,.2678704E+03,.2644459E+03,&
     & .2904946E+03,.2871655E+03,.2838470E+03,.2803941E+03,.2768948E+03,&
     & .3022413E+03,.2989983E+03,.2955817E+03,.2921152E+03,.2885458E+03,&
     & .3130565E+03,.3098089E+03,.3064324E+03,.3028900E+03,.2992469E+03,&
     & .3228674E+03,.3196371E+03,.3162446E+03,.3126653E+03,.3089740E+03,&
     & .3316375E+03,.3284418E+03,.3250265E+03,.3214442E+03,.3176680E+03,&
     & .3390155E+03,.3360705E+03,.3328242E+03,.3291721E+03,.3252655E+03,&
     & .3449713E+03,.3421655E+03,.3390332E+03,.3355221E+03,.3318374E+03,&
     & .3493969E+03,.3468038E+03,.3437556E+03,.3405122E+03,.3368916E+03,&
     & .3522054E+03,.3499115E+03,.3472051E+03,.3440347E+03,.3404337E+03,&
     & .3535539E+03,.3513906E+03,.3491088E+03,.3465269E+03,.3430700E+03,&
     & .3545730E+03,.3522557E+03,.3497503E+03,.3472267E+03,.3443498E+03,&
     & .3550447E+03,.3525630E+03,.3499373E+03,.3472388E+03,.3443261E+03,&
     & .3552526E+03,.3524843E+03,.3496510E+03,.3469402E+03,.3437094E+03,&
     & .3555960E+03,.3526031E+03,.3495395E+03,.3465118E+03,.3432573E+03,&
     & .3557797E+03,.3525640E+03,.3494123E+03,.3460217E+03,.3426940E+03,&
     & .3564043E+03,.3530960E+03,.3498075E+03,.3463239E+03,.3427123E+03,&
     & .3568211E+03,.3535487E+03,.3500420E+03,.3465557E+03,.3428692E+03,&
     & .3576037E+03,.3541394E+03,.3507368E+03,.3471845E+03,.3434088E+03,&
     & .3580188E+03,.3547035E+03,.3511129E+03,.3473748E+03,.3438292E+03,&
     & .3581982E+03,.3548803E+03,.3511249E+03,.3474255E+03,.3435849E+03,&
     & .3580668E+03,.3546083E+03,.3508286E+03,.3468622E+03,.3427171E+03,&
     & .3579811E+03,.3542519E+03,.3503648E+03,.3462908E+03,.3414558E+03,&
     & .3576096E+03,.3539086E+03,.3499965E+03,.3456525E+03,.3403969E+03/

      data   absb(121:235, 6)   /                                       &
     & .3586604E+03,.3550335E+03,.3510399E+03,.3464481E+03,.3408584E+03,&
     & .3598800E+03,.3563594E+03,.3522228E+03,.3469653E+03,.3413783E+03,&
     & .3612682E+03,.3576057E+03,.3531275E+03,.3475159E+03,.3418607E+03,&
     & .3631576E+03,.3595023E+03,.3547940E+03,.3491532E+03,.3434219E+03,&
     & .3650737E+03,.3613720E+03,.3564614E+03,.3509635E+03,.3452500E+03,&
     & .3670026E+03,.3632776E+03,.3582252E+03,.3527572E+03,.3469855E+03,&
     & .3692557E+03,.3655325E+03,.3603844E+03,.3550208E+03,.3492473E+03,&
     & .3718397E+03,.3678903E+03,.3626881E+03,.3574203E+03,.3517227E+03,&
     & .3744095E+03,.3701124E+03,.3649258E+03,.3597189E+03,.3542337E+03,&
     & .3767816E+03,.3723343E+03,.3673166E+03,.3620799E+03,.3567366E+03,&
     & .3791321E+03,.3746009E+03,.3697970E+03,.3646097E+03,.3594172E+03,&
     & .3813238E+03,.3769336E+03,.3722063E+03,.3670953E+03,.3619918E+03,&
     & .3834849E+03,.3791057E+03,.3745237E+03,.3694845E+03,.3645086E+03,&
     & .3854281E+03,.3811015E+03,.3766032E+03,.3716952E+03,.3667988E+03,&
     & .3872332E+03,.3829179E+03,.3785031E+03,.3738180E+03,.3689635E+03,&
     & .3889780E+03,.3847170E+03,.3803790E+03,.3758553E+03,.3711179E+03,&
     & .3907123E+03,.3865429E+03,.3822340E+03,.3778545E+03,.3731887E+03,&
     & .3921534E+03,.3881790E+03,.3839339E+03,.3796445E+03,.3751081E+03,&
     & .3934377E+03,.3896848E+03,.3855532E+03,.3813151E+03,.3769936E+03,&
     & .3947370E+03,.3911513E+03,.3871310E+03,.3829493E+03,.3787443E+03,&
     & .3959555E+03,.3924751E+03,.3887103E+03,.3846245E+03,.3804545E+03,&
     & .3971711E+03,.3937408E+03,.3901626E+03,.3862220E+03,.3821316E+03,&
     & .3970289E+03,.3941655E+03,.3908061E+03,.3869572E+03,.3829149E+03/

!  ---
      data   forref(1:3,1: 6)  / .6585271E-05,.9433300E-05,.7977931E-04,&
     & .8154837E-04,.1197092E-03,.1424472E-03,.2535780E-03,.2329964E-03,&
     & .1885349E-03,.3380323E-03,.3052715E-03,.2150602E-03,.3685495E-03,&
     & .3314392E-03,.2543625E-03,.3806938E-03,.3823779E-03,.2995716E-03/

!     the array selfref contains the coefficient of the water vapor
!     self-continuum (including the energy term).  the first index
!     refers to temperature in 7.2 degree increments.  for instance,
!     jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!     etc.  the second index runs over the g-channel (1 to 6).

      data   selfref(1:10,1: 6)  /                                      &
     & .1352797E-02,.1118049E-02,.9240558E-03,.7637369E-03,.6312454E-03,&
     & .5217486E-03,.4312539E-03,.3564616E-03,.2946471E-03,.2435560E-03,&
     & .3427292E-02,.2856171E-02,.2390076E-02,.2008096E-02,.1693734E-02,&
     & .1433911E-02,.1218251E-02,.1038495E-02,.8880442E-03,.7616162E-03,&
     & .4545629E-02,.4185794E-02,.3854444E-02,.3549326E-02,.3268367E-02,&
     & .3009646E-02,.2771406E-02,.2552033E-02,.2350020E-02,.2163998E-02,&
     & .5477838E-02,.5082542E-02,.4715844E-02,.4375667E-02,.4060089E-02,&
     & .3767331E-02,.3495734E-02,.3243771E-02,.3010013E-02,.2793137E-02,&
     & .5682360E-02,.5292443E-02,.4929650E-02,.4592058E-02,.4277891E-02,&
     & .3985497E-02,.3713349E-02,.3460014E-02,.3224177E-02,.3004612E-02,&
     & .7004786E-02,.6372419E-02,.5797303E-02,.5274236E-02,.4798502E-02,&
     & .4365805E-02,.3972237E-02,.3614256E-02,.3288634E-02,.2992436E-02/

!........................................!
      end module module_radsw_kgb16      !
!========================================!


!> This module sets up absorption coeffients for band 17: 3250-4000
!! cm-1 (low - h2o,co2; high - h2o,co2)
!========================================!
      module module_radsw_kgb17          !
!........................................!
!
! *********        the original program descriptions        ********* !
!                                                                     !
!     originally by j.delamere, atmospheric & environmental research. !
!     revision: 2.4                                                   !
!     band 17:  3250-4000 cm-1 (low - h2o,co2; high - h2o, co2)       !
!     reformatted for f90 by jjmorcrette, ecmwf                       !
!                                                                     !
!     this table has been re-generated for reduced number of g-point  !
!     by y.t.hou, ncep                                                !
!                                                                     !
! *********    *********    end description    *********    ********* !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NG17

!
      implicit none
!
      private
!
!> msa17=585
      integer,  public :: MSA17
!> msb17=1175
      integer,  public :: MSB17
!> msf17=10
      integer,  public :: MSF17
!> mfr17=4
      integer,  public :: MFR17
      parameter (MSA17=585, MSB17=1175, MSF17=10, MFR17=4)

!>    the array selfref contains the coefficient of the water vapor
!!    self-continuum (including the energy term).  the first index
!!     refers to temperature in 7.2 degree increments.  for instance,
!!     jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!!     etc.  the second index runs over the g-channel (1 to NG17).
      real (kind=kind_phys), public ::   selfref(MSF17,NG17)

!>    the array absa(585,NG17) (ka((9,5,13,NG17)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels> ~100mb,
!!    temperatures, and binary species parameters (see taumol.f for definition).
!!    the first index in the array, js, runs from 1 to 9, and corresponds to
!!    different values of the binary species parameter.  for instance,
!!    js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!!    js = 3 corresponds to the parameter value 2/8, etc.  the second index
!!    in the array, jt, which runs from 1 to 5, corresponds to different
!!    temperatures.  more specifically, jt = 3 means that the data are for
!!    the reference temperature tref for this  pressure level, jt = 2 refers
!!    to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!!    is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!!    to the jpth reference pressure level (see taumol.f for these levels
!!    in mb).  the fourth index, ig, goes from 1 to 12, and indicates
!!    which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public ::   absa(MSA17,NG17)

!>    the array absb(1175,12) (kb(5,5,13:59,12)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels < ~100mb and
!!    temperatures. the first index in the array, jt, which runs from 1 to 5,
!!    corresponds to different temperatures.  more specifically, jt = 3 means
!!    that the data are for the reference temperature tref for this pressure
!!    level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!!    tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!!    the second index, jp, runs from 13 to 59 and refers to the jpth
!!    reference pressure level (see taumol.f for the value of these
!!    pressure levels in mb).  the third index, ig, goes from 1 to 12,
!!    and tells us which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public ::   absb(MSB17,NG17)

      real (kind=kind_phys), public ::   forref(MFR17,NG17)

!> rayleigh extinction coefficient at v = 3625 \f$cm^{-1}\f$
      real (kind=kind_phys), parameter, public :: rayl = 6.86e-10

      data   absa(  1:180, 1)   /                                       &
     & .1113400E-05,.3209200E-03,.5566300E-03,.7822700E-03,.9942100E-03,&
     & .1204500E-02,.1411600E-02,.1606900E-02,.1933000E-02,.1054400E-05,&
     & .3543400E-03,.6148000E-03,.8642400E-03,.1105500E-02,.1343100E-02,&
     & .1575400E-02,.1795100E-02,.2128000E-02,.9834500E-06,.3898900E-03,&
     & .6774700E-03,.9528900E-03,.1219600E-02,.1486400E-02,.1744900E-02,&
     & .1989700E-02,.2356400E-02,.9501700E-06,.4258200E-03,.7441900E-03,&
     & .1046300E-02,.1342400E-02,.1635300E-02,.1921500E-02,.2193900E-02,&
     & .2599300E-02,.9187000E-06,.4637100E-03,.8139400E-03,.1146400E-02,&
     & .1471500E-02,.1793700E-02,.2107200E-02,.2407700E-02,.2854000E-02,&
     & .9418400E-06,.2653200E-03,.4615700E-03,.6377600E-03,.8108700E-03,&
     & .9772800E-03,.1141900E-02,.1299100E-02,.1512100E-02,.8906000E-06,&
     & .2941700E-03,.5096700E-03,.7075900E-03,.9020300E-03,.1092000E-02,&
     & .1276400E-02,.1454400E-02,.1679500E-02,.8272000E-06,.3236100E-03,&
     & .5612400E-03,.7829900E-03,.9971800E-03,.1209400E-02,.1417200E-02,&
     & .1616500E-02,.1863700E-02,.7884200E-06,.3549500E-03,.6168400E-03,&
     & .8617600E-03,.1099500E-02,.1334500E-02,.1565200E-02,.1785600E-02,&
     & .2062100E-02,.7661000E-06,.3872800E-03,.6753800E-03,.9461300E-03,&
     & .1208200E-02,.1467500E-02,.1720600E-02,.1964000E-02,.2272000E-02,&
     & .7966300E-06,.2136900E-03,.3708800E-03,.5122200E-03,.6454900E-03,&
     & .7732200E-03,.8993900E-03,.1020500E-02,.1156800E-02,.7595500E-06,&
     & .2378000E-03,.4115300E-03,.5690100E-03,.7200900E-03,.8673500E-03,&
     & .1009200E-02,.1147600E-02,.1291300E-02,.7169200E-06,.2627600E-03,&
     & .4548800E-03,.6292200E-03,.7983600E-03,.9625900E-03,.1123600E-02,&
     & .1279600E-02,.1439800E-02,.6634800E-06,.2889600E-03,.5022100E-03,&
     & .6941100E-03,.8822600E-03,.1065800E-02,.1245600E-02,.1419100E-02,&
     & .1599100E-02,.6401300E-06,.3164300E-03,.5516600E-03,.7643400E-03,&
     & .9721600E-03,.1174500E-02,.1373100E-02,.1565700E-02,.1767100E-02,&
     & .6621700E-06,.1696600E-03,.2935800E-03,.4068800E-03,.5111400E-03,&
     & .6073900E-03,.7019100E-03,.7944600E-03,.8883200E-03,.6411100E-06,&
     & .1890200E-03,.3274200E-03,.4536400E-03,.5708600E-03,.6840100E-03,&
     & .7911600E-03,.8950700E-03,.9965500E-03,.6072900E-06,.2100700E-03,&
     & .3635400E-03,.5035400E-03,.6345900E-03,.7611300E-03,.8847300E-03,&
     & .1003500E-02,.1116300E-02,.5725000E-06,.2320300E-03,.4024500E-03,&
     & .5575800E-03,.7029500E-03,.8451500E-03,.9837900E-03,.1116500E-02,&
     & .1244400E-02,.5352400E-06,.2552900E-03,.4439900E-03,.6156400E-03,&
     & .7762600E-03,.9340500E-03,.1089200E-02,.1236900E-02,.1379500E-02/

      data   absa(181:315, 1)   /                                       &
     & .5511200E-06,.1342200E-03,.2316600E-03,.3211900E-03,.4026500E-03,&
     & .4781200E-03,.5479700E-03,.6187300E-03,.6891200E-03,.5366900E-06,&
     & .1499000E-03,.2594900E-03,.3588600E-03,.4523900E-03,.5393600E-03,&
     & .6189400E-03,.6971800E-03,.7773200E-03,.5149500E-06,.1670600E-03,&
     & .2896000E-03,.4002000E-03,.5046100E-03,.6024900E-03,.6967700E-03,&
     & .7846400E-03,.8733400E-03,.4863300E-06,.1854300E-03,.3215900E-03,&
     & .4449900E-03,.5609500E-03,.6705500E-03,.7757500E-03,.8771700E-03,&
     & .9769400E-03,.4516200E-06,.2047200E-03,.3559100E-03,.4935000E-03,&
     & .6216300E-03,.7425300E-03,.8624700E-03,.9751800E-03,.1086400E-02,&
     & .4505500E-06,.1055600E-03,.1817600E-03,.2513700E-03,.3143800E-03,&
     & .3736300E-03,.4275900E-03,.4788300E-03,.5343700E-03,.4450200E-06,&
     & .1181400E-03,.2038300E-03,.2816200E-03,.3554700E-03,.4226100E-03,&
     & .4834000E-03,.5406600E-03,.6035200E-03,.4306600E-06,.1321700E-03,&
     & .2283600E-03,.3155200E-03,.3976200E-03,.4749300E-03,.5454300E-03,&
     & .6106600E-03,.6817600E-03,.4111600E-06,.1473300E-03,.2547200E-03,&
     & .3520400E-03,.4433400E-03,.5299700E-03,.6096600E-03,.6862200E-03,&
     & .7660700E-03,.3874800E-06,.1634400E-03,.2828100E-03,.3917300E-03,&
     & .4930500E-03,.5888000E-03,.6795400E-03,.7648500E-03,.8540100E-03,&
     & .3658200E-06,.8295800E-04,.1421000E-03,.1957400E-03,.2444000E-03,&
     & .2903900E-03,.3328100E-03,.3717700E-03,.4217800E-03,.3670700E-06,&
     & .9290900E-04,.1597700E-03,.2203400E-03,.2774200E-03,.3295500E-03,&
     & .3772700E-03,.4195600E-03,.4785900E-03,.3578700E-06,.1043200E-03,&
     & .1793000E-03,.2476900E-03,.3115500E-03,.3724200E-03,.4269900E-03,&
     & .4747400E-03,.5408300E-03,.3448200E-06,.1168500E-03,.2005500E-03,&
     & .2772500E-03,.3486600E-03,.4167700E-03,.4790300E-03,.5351800E-03,&
     & .6081200E-03,.3259800E-06,.1300100E-03,.2234100E-03,.3090000E-03,&
     & .3891900E-03,.4646500E-03,.5346700E-03,.5992600E-03,.6773300E-03/

      data   absa(316:450, 1)   /                                       &
     & .2983400E-06,.6500900E-04,.1118300E-03,.1520000E-03,.1896900E-03,&
     & .2251700E-03,.2583200E-03,.2873000E-03,.3395500E-03,.2974500E-06,&
     & .7287200E-04,.1248200E-03,.1722100E-03,.2158300E-03,.2559600E-03,&
     & .2932800E-03,.3252000E-03,.3862000E-03,.2942100E-06,.8203600E-04,&
     & .1401000E-03,.1934400E-03,.2436200E-03,.2907100E-03,.3330800E-03,&
     & .3686700E-03,.4390100E-03,.2859600E-06,.9225600E-04,.1573800E-03,&
     & .2175200E-03,.2732500E-03,.3264300E-03,.3757800E-03,.4171000E-03,&
     & .4935800E-03,.2729600E-06,.1032000E-03,.1759800E-03,.2433900E-03,&
     & .3059100E-03,.3650500E-03,.4206100E-03,.4687200E-03,.5499300E-03,&
     & .2425900E-06,.5062600E-04,.8694500E-04,.1177800E-03,.1469200E-03,&
     & .1743100E-03,.2005700E-03,.2213100E-03,.2991900E-03,.2405800E-06,&
     & .5696600E-04,.9746700E-04,.1340500E-03,.1675000E-03,.1981400E-03,&
     & .2273900E-03,.2528100E-03,.3402200E-03,.2409100E-06,.6432400E-04,&
     & .1097100E-03,.1506100E-03,.1899600E-03,.2258000E-03,.2588500E-03,&
     & .2869400E-03,.3804500E-03,.2355000E-06,.7245100E-04,.1234500E-03,&
     & .1699100E-03,.2136200E-03,.2548400E-03,.2934100E-03,.3252300E-03,&
     & .4283600E-03,.2269600E-06,.8135100E-04,.1383900E-03,.1909000E-03,&
     & .2398600E-03,.2860800E-03,.3291000E-03,.3663900E-03,.4792500E-03,&
     & .1967900E-06,.3976000E-04,.6805200E-04,.9229500E-04,.1149800E-03,&
     & .1365600E-03,.1560400E-03,.1717100E-03,.2967900E-03,.1956700E-06,&
     & .4484700E-04,.7682900E-04,.1051800E-03,.1310200E-03,.1550900E-03,&
     & .1776400E-03,.1984300E-03,.3286300E-03,.1957400E-06,.5056500E-04,&
     & .8646600E-04,.1187800E-03,.1492000E-03,.1769800E-03,.2025800E-03,&
     & .2252600E-03,.3695900E-03,.1926200E-06,.5712400E-04,.9754600E-04,&
     & .1339800E-03,.1680300E-03,.2004600E-03,.2302200E-03,.2557800E-03,&
     & .4154300E-03,.1874700E-06,.6426400E-04,.1097100E-03,.1508600E-03,&
     & .1892300E-03,.2256500E-03,.2593400E-03,.2892100E-03,.4646300E-03/

      data   absa(451:585, 1)   /                                       &
     & .1601500E-06,.3302500E-04,.5656200E-04,.7662500E-04,.9564500E-04,&
     & .1135800E-03,.1295500E-03,.1426900E-03,.2754500E-03,.1591800E-06,&
     & .3714800E-04,.6377500E-04,.8760000E-04,.1089700E-03,.1289300E-03,&
     & .1475600E-03,.1646500E-03,.3079900E-03,.1589700E-06,.4199300E-04,&
     & .7204000E-04,.9892800E-04,.1243300E-03,.1473300E-03,.1682400E-03,&
     & .1869400E-03,.3434000E-03,.1557800E-06,.4744100E-04,.8142600E-04,&
     & .1116500E-03,.1398500E-03,.1667300E-03,.1915600E-03,.2126800E-03,&
     & .3817600E-03,.1521000E-06,.5339800E-04,.9165700E-04,.1259200E-03,&
     & .1576900E-03,.1878100E-03,.2156900E-03,.2406900E-03,.4222800E-03,&
     & .1302100E-06,.2727200E-04,.4683200E-04,.6351100E-04,.7920500E-04,&
     & .9394900E-04,.1071900E-03,.1182700E-03,.2394400E-03,.1293000E-06,&
     & .3068600E-04,.5292500E-04,.7278200E-04,.9052700E-04,.1070800E-03,&
     & .1223600E-03,.1362000E-03,.2712100E-03,.1290000E-06,.3472300E-04,&
     & .5986700E-04,.8220500E-04,.1032600E-03,.1223800E-03,.1395800E-03,&
     & .1549800E-03,.2993000E-03,.1264800E-06,.3925200E-04,.6761800E-04,&
     & .9294900E-04,.1163600E-03,.1383500E-03,.1592400E-03,.1765300E-03,&
     & .3305000E-03,.1229900E-06,.4421300E-04,.7624400E-04,.1048800E-03,&
     & .1313500E-03,.1560100E-03,.1790200E-03,.1997600E-03,.3657900E-03,&
     & .1055400E-06,.2238800E-04,.3859700E-04,.5237800E-04,.6540400E-04,&
     & .7756700E-04,.8854300E-04,.9780100E-04,.1959900E-03,.1048000E-06,&
     & .2518100E-04,.4367000E-04,.6020500E-04,.7488300E-04,.8860500E-04,&
     & .1011700E-03,.1124200E-03,.2214600E-03,.1042900E-06,.2853000E-04,&
     & .4942200E-04,.6803900E-04,.8543500E-04,.1013600E-03,.1155700E-03,&
     & .1282800E-03,.2444700E-03,.1023200E-06,.3230100E-04,.5579700E-04,&
     & .7694800E-04,.9644700E-04,.1146200E-03,.1316600E-03,.1461600E-03,&
     & .2699900E-03,.9901400E-07,.3639400E-04,.6299400E-04,.8694600E-04,&
     & .1090000E-03,.1293700E-03,.1483300E-03,.1654000E-03,.2991600E-03/

      data   absa(  1:180, 2)   /                                       &
     & .2348800E-04,.1668400E-02,.2957500E-02,.4008300E-02,.4995300E-02,&
     & .5902100E-02,.6744600E-02,.7600800E-02,.8703200E-02,.2515600E-04,&
     & .1846200E-02,.3271100E-02,.4500800E-02,.5618200E-02,.6628700E-02,&
     & .7590900E-02,.8555800E-02,.9754300E-02,.2708400E-04,.2032500E-02,&
     & .3603400E-02,.4999200E-02,.6279600E-02,.7413400E-02,.8499400E-02,&
     & .9569400E-02,.1084700E-01,.2814700E-04,.2234700E-02,.3962800E-02,&
     & .5500500E-02,.6945300E-02,.8242900E-02,.9469900E-02,.1066000E-01,&
     & .1199100E-01,.2871900E-04,.2447500E-02,.4336900E-02,.6036000E-02,&
     & .7618300E-02,.9087800E-02,.1049300E-01,.1180500E-01,.1320400E-01,&
     & .2094700E-04,.1401500E-02,.2472800E-02,.3367600E-02,.4180100E-02,&
     & .4924400E-02,.5578700E-02,.6220600E-02,.6974100E-02,.2235200E-04,&
     & .1553500E-02,.2739600E-02,.3787000E-02,.4712100E-02,.5557100E-02,&
     & .6289200E-02,.7013200E-02,.7809200E-02,.2388200E-04,.1721300E-02,&
     & .3034500E-02,.4202000E-02,.5278200E-02,.6228500E-02,.7055600E-02,&
     & .7881900E-02,.8733000E-02,.2465000E-04,.1895400E-02,.3338400E-02,&
     & .4633600E-02,.5833300E-02,.6931100E-02,.7875100E-02,.8814900E-02,&
     & .9691100E-02,.2513600E-04,.2080600E-02,.3664100E-02,.5096600E-02,&
     & .6419300E-02,.7628100E-02,.8740300E-02,.9794300E-02,.1070400E-01,&
     & .1772000E-04,.1141400E-02,.2004500E-02,.2728800E-02,.3389100E-02,&
     & .3975500E-02,.4496700E-02,.4938900E-02,.5385800E-02,.1908700E-04,&
     & .1274500E-02,.2233600E-02,.3084100E-02,.3833300E-02,.4498300E-02,&
     & .5089500E-02,.5593000E-02,.6066600E-02,.2012500E-04,.1417800E-02,&
     & .2479500E-02,.3427800E-02,.4310600E-02,.5060700E-02,.5728000E-02,&
     & .6309200E-02,.6828500E-02,.2089500E-04,.1566600E-02,.2734800E-02,&
     & .3793400E-02,.4776900E-02,.5662100E-02,.6404600E-02,.7086400E-02,&
     & .7602600E-02,.2150800E-04,.1725100E-02,.3015300E-02,.4180900E-02,&
     & .5268900E-02,.6246200E-02,.7112500E-02,.7910400E-02,.8438900E-02,&
     & .1466100E-04,.9226100E-03,.1608800E-02,.2178800E-02,.2695600E-02,&
     & .3161200E-02,.3562600E-02,.3887800E-02,.4152500E-02,.1560700E-04,&
     & .1032900E-02,.1797300E-02,.2473500E-02,.3069600E-02,.3595300E-02,&
     & .4051100E-02,.4416200E-02,.4699000E-02,.1655900E-04,.1155000E-02,&
     & .2001900E-02,.2757700E-02,.3462500E-02,.4066200E-02,.4583100E-02,&
     & .4999500E-02,.5307300E-02,.1726300E-04,.1279400E-02,.2218400E-02,&
     & .3061400E-02,.3851000E-02,.4566400E-02,.5153200E-02,.5629000E-02,&
     & .5943600E-02,.1789500E-04,.1413000E-02,.2452600E-02,.3380500E-02,&
     & .4262200E-02,.5055600E-02,.5748900E-02,.6309400E-02,.6625600E-02/

      data   absa(181:315, 2)   /                                       &
     & .1188500E-04,.7488800E-03,.1282600E-02,.1725800E-02,.2134600E-02,&
     & .2497200E-02,.2804600E-02,.3060600E-02,.3232300E-02,.1245200E-04,&
     & .8408400E-03,.1438000E-02,.1969800E-02,.2438600E-02,.2853600E-02,&
     & .3207500E-02,.3493700E-02,.3656200E-02,.1343900E-04,.9427200E-03,&
     & .1606500E-02,.2207900E-02,.2766000E-02,.3238900E-02,.3644000E-02,&
     & .3967300E-02,.4139300E-02,.1405300E-04,.1047300E-02,.1789900E-02,&
     & .2460300E-02,.3086100E-02,.3652100E-02,.4115600E-02,.4482500E-02,&
     & .4652900E-02,.1458600E-04,.1158500E-02,.1988600E-02,.2725700E-02,&
     & .3427200E-02,.4059900E-02,.4611500E-02,.5028600E-02,.5208500E-02,&
     & .9487200E-05,.5979300E-03,.1021000E-02,.1357200E-02,.1671600E-02,&
     & .1951200E-02,.2189000E-02,.2382700E-02,.2534700E-02,.1006000E-04,&
     & .6740000E-03,.1143200E-02,.1556400E-02,.1918800E-02,.2238900E-02,&
     & .2516100E-02,.2730800E-02,.2877000E-02,.1074300E-04,.7594800E-03,&
     & .1280600E-02,.1755100E-02,.2186200E-02,.2554500E-02,.2872900E-02,&
     & .3114900E-02,.3271600E-02,.1132200E-04,.8513400E-03,.1432000E-02,&
     & .1963000E-02,.2451400E-02,.2893000E-02,.3257800E-02,.3537500E-02,&
     & .3691200E-02,.1175500E-04,.9461100E-03,.1598600E-02,.2180400E-02,&
     & .2729100E-02,.3230600E-02,.3669500E-02,.3991300E-02,.4143900E-02,&
     & .7593100E-05,.4713000E-03,.8121100E-03,.1065800E-02,.1300400E-02,&
     & .1517500E-02,.1700200E-02,.1847300E-02,.1986100E-02,.8072400E-05,&
     & .5335800E-03,.9116300E-03,.1226100E-02,.1501400E-02,.1748300E-02,&
     & .1961200E-02,.2120100E-02,.2259100E-02,.8440400E-05,.6051700E-03,&
     & .1023600E-02,.1388700E-02,.1720400E-02,.2003100E-02,.2249700E-02,&
     & .2432100E-02,.2582300E-02,.9051100E-05,.6803700E-03,.1147500E-02,&
     & .1559600E-02,.1940200E-02,.2278900E-02,.2563300E-02,.2775300E-02,&
     & .2924700E-02,.9440500E-05,.7610300E-03,.1283600E-02,.1742200E-02,&
     & .2169800E-02,.2558500E-02,.2898900E-02,.3144400E-02,.3301600E-02/

      data   absa(316:450, 2)   /                                       &
     & .5808000E-05,.3678200E-03,.6351600E-03,.8379900E-03,.1011900E-02,&
     & .1175400E-02,.1316100E-02,.1438200E-02,.1583700E-02,.6315600E-05,&
     & .4190900E-03,.7197400E-03,.9695200E-03,.1171100E-02,.1361700E-02,&
     & .1523400E-02,.1644100E-02,.1811800E-02,.6677600E-05,.4782900E-03,&
     & .8115000E-03,.1105600E-02,.1349300E-02,.1564600E-02,.1752400E-02,&
     & .1893700E-02,.2069800E-02,.7058200E-05,.5401700E-03,.9133000E-03,&
     & .1243400E-02,.1541100E-02,.1788400E-02,.2005700E-02,.2170400E-02,&
     & .2354800E-02,.7490200E-05,.6071700E-03,.1026700E-02,.1390800E-02,&
     & .1723100E-02,.2028500E-02,.2281000E-02,.2471000E-02,.2663100E-02,&
     & .4506100E-05,.2862600E-03,.4944800E-03,.6531700E-03,.7908100E-03,&
     & .9106600E-03,.1018200E-02,.1110700E-02,.1459200E-02,.4939200E-05,&
     & .3273400E-03,.5655800E-03,.7607700E-03,.9184600E-03,.1057800E-02,&
     & .1177600E-02,.1272000E-02,.1677800E-02,.5258000E-05,.3752200E-03,&
     & .6384500E-03,.8780500E-03,.1061900E-02,.1219400E-02,.1360600E-02,&
     & .1468000E-02,.1925900E-02,.5499100E-05,.4265900E-03,.7210300E-03,&
     & .9875600E-03,.1216300E-02,.1400700E-02,.1564700E-02,.1690000E-02,&
     & .2177200E-02,.5876900E-05,.4819900E-03,.8146600E-03,.1107200E-02,&
     & .1368400E-02,.1598400E-02,.1788700E-02,.1933300E-02,.2433100E-02,&
     & .3415400E-05,.2253400E-03,.3878800E-03,.5111300E-03,.6219400E-03,&
     & .7143200E-03,.7974300E-03,.8628900E-03,.1475800E-02,.3808500E-05,&
     & .2583200E-03,.4474800E-03,.5983500E-03,.7251400E-03,.8328400E-03,&
     & .9194100E-03,.9958300E-03,.1735800E-02,.4157800E-05,.2965500E-03,&
     & .5055300E-03,.6951100E-03,.8410700E-03,.9637400E-03,.1066600E-02,&
     & .1147800E-02,.1962500E-02,.4391700E-05,.3390800E-03,.5735100E-03,&
     & .7864000E-03,.9686200E-03,.1110900E-02,.1230900E-02,.1326600E-02,&
     & .2210300E-02,.4544600E-05,.3854800E-03,.6488300E-03,.8840100E-03,&
     & .1095200E-02,.1270600E-02,.1413600E-02,.1523900E-02,.2475400E-02/

      data   absa(451:585, 2)   /                                       &
     & .2760400E-05,.1884800E-03,.3253200E-03,.4294900E-03,.5235000E-03,&
     & .6008400E-03,.6664000E-03,.7207000E-03,.1369200E-02,.3090800E-05,&
     & .2167900E-03,.3725900E-03,.5036600E-03,.6107400E-03,.7015800E-03,&
     & .7727100E-03,.8303700E-03,.1568900E-02,.3367400E-05,.2496500E-03,&
     & .4224400E-03,.5835300E-03,.7092300E-03,.8133000E-03,.8975000E-03,&
     & .9613100E-03,.1797100E-02,.3556800E-05,.2860200E-03,.4804400E-03,&
     & .6586800E-03,.8181600E-03,.9385700E-03,.1035700E-02,.1111900E-02,&
     & .2050900E-02,.3692100E-05,.3248400E-03,.5462300E-03,.7447500E-03,&
     & .9230700E-03,.1075800E-02,.1189500E-02,.1277700E-02,.2325300E-02,&
     & .2227200E-05,.1580700E-03,.2716300E-03,.3593500E-03,.4378100E-03,&
     & .5032100E-03,.5577600E-03,.6004600E-03,.1185200E-02,.2489800E-05,&
     & .1820100E-03,.3100400E-03,.4220600E-03,.5122600E-03,.5882400E-03,&
     & .6491900E-03,.6929100E-03,.1362300E-02,.2702100E-05,.2095800E-03,&
     & .3532200E-03,.4867600E-03,.5958700E-03,.6836400E-03,.7552700E-03,&
     & .8043600E-03,.1565400E-02,.2852600E-05,.2403800E-03,.4026200E-03,&
     & .5509600E-03,.6867100E-03,.7910300E-03,.8721100E-03,.9312400E-03,&
     & .1789800E-02,.2989900E-05,.2728900E-03,.4590400E-03,.6244800E-03,&
     & .7749600E-03,.9021200E-03,.1001800E-02,.1070600E-02,.2026500E-02,&
     & .1757600E-05,.1315100E-03,.2251800E-03,.2989000E-03,.3639800E-03,&
     & .4188400E-03,.4650500E-03,.5008600E-03,.9791600E-03,.1952400E-05,&
     & .1518900E-03,.2568600E-03,.3512300E-03,.4272300E-03,.4906100E-03,&
     & .5429300E-03,.5781500E-03,.1126000E-02,.2113400E-05,.1755800E-03,&
     & .2938200E-03,.4045700E-03,.4977700E-03,.5715300E-03,.6318500E-03,&
     & .6721300E-03,.1296000E-02,.2268100E-05,.2010600E-03,.3358300E-03,&
     & .4597700E-03,.5709400E-03,.6627500E-03,.7314900E-03,.7779900E-03,&
     & .1485600E-02,.2394000E-05,.2285300E-03,.3832300E-03,.5212600E-03,&
     & .6468400E-03,.7545500E-03,.8411500E-03,.8957500E-03,.1681600E-02/

      data   absa(  1:180, 3)   /                                       &
     & .8715400E-04,.7045600E-02,.1162400E-01,.1600700E-01,.2027700E-01,&
     & .2434900E-01,.2790200E-01,.3095600E-01,.3315900E-01,.9267800E-04,&
     & .7532400E-02,.1255200E-01,.1744400E-01,.2196700E-01,.2625000E-01,&
     & .3024100E-01,.3334800E-01,.3616400E-01,.9795500E-04,.8015300E-02,&
     & .1350200E-01,.1881200E-01,.2386200E-01,.2848000E-01,.3251100E-01,&
     & .3571300E-01,.3834900E-01,.1040400E-03,.8472400E-02,.1442900E-01,&
     & .2013600E-01,.2567200E-01,.3068800E-01,.3501100E-01,.3837900E-01,&
     & .4067300E-01,.1111000E-03,.8912300E-02,.1534300E-01,.2147900E-01,&
     & .2729000E-01,.3275100E-01,.3745500E-01,.4093100E-01,.4246300E-01,&
     & .7321400E-04,.6141000E-02,.1012500E-01,.1365000E-01,.1715800E-01,&
     & .2062700E-01,.2362800E-01,.2625100E-01,.2748100E-01,.7793600E-04,&
     & .6575100E-02,.1096600E-01,.1487700E-01,.1869300E-01,.2227200E-01,&
     & .2563000E-01,.2839500E-01,.2989400E-01,.8306100E-04,.6982800E-02,&
     & .1177400E-01,.1606800E-01,.2031700E-01,.2424400E-01,.2774800E-01,&
     & .3043200E-01,.3185100E-01,.8856400E-04,.7398500E-02,.1258100E-01,&
     & .1722600E-01,.2184200E-01,.2617700E-01,.2996900E-01,.3274800E-01,&
     & .3380500E-01,.9493300E-04,.7798800E-02,.1337200E-01,.1836700E-01,&
     & .2332300E-01,.2792800E-01,.3196400E-01,.3501300E-01,.3532700E-01,&
     & .6094700E-04,.5243600E-02,.8617400E-02,.1152000E-01,.1423100E-01,&
     & .1702700E-01,.1946900E-01,.2163400E-01,.2198100E-01,.6492100E-04,&
     & .5597500E-02,.9351200E-02,.1252600E-01,.1551600E-01,.1847100E-01,&
     & .2127000E-01,.2351200E-01,.2411400E-01,.6952100E-04,.5964800E-02,&
     & .1006000E-01,.1358200E-01,.1690300E-01,.2017700E-01,.2305500E-01,&
     & .2538000E-01,.2582200E-01,.7434800E-04,.6310400E-02,.1076700E-01,&
     & .1459900E-01,.1827300E-01,.2183600E-01,.2498400E-01,.2735800E-01,&
     & .2745000E-01,.7978000E-04,.6667100E-02,.1145800E-01,.1559000E-01,&
     & .1957900E-01,.2335800E-01,.2680600E-01,.2933500E-01,.2878600E-01,&
     & .5079900E-04,.4414800E-02,.7222500E-02,.9648300E-02,.1176700E-01,&
     & .1382800E-01,.1584400E-01,.1758800E-01,.1739100E-01,.5422700E-04,&
     & .4723100E-02,.7852700E-02,.1049300E-01,.1283500E-01,.1512600E-01,&
     & .1746600E-01,.1921900E-01,.1935100E-01,.5815100E-04,.5027700E-02,&
     & .8482300E-02,.1141800E-01,.1396900E-01,.1657600E-01,.1890300E-01,&
     & .2084000E-01,.2085300E-01,.6243300E-04,.5337200E-02,.9081300E-02,&
     & .1232500E-01,.1516300E-01,.1797100E-01,.2059000E-01,.2252900E-01,&
     & .2227300E-01,.6685200E-04,.5642900E-02,.9683600E-02,.1318900E-01,&
     & .1628000E-01,.1931200E-01,.2213600E-01,.2426300E-01,.2346600E-01/

      data   absa(181:315, 3)   /                                       &
     & .4259300E-04,.3676000E-02,.5991300E-02,.7981200E-02,.9772900E-02,&
     & .1126500E-01,.1287400E-01,.1415000E-01,.1365100E-01,.4577900E-04,&
     & .3934400E-02,.6540100E-02,.8703000E-02,.1065000E-01,.1237400E-01,&
     & .1413500E-01,.1552900E-01,.1550700E-01,.4886800E-04,.4201300E-02,&
     & .7085900E-02,.9491200E-02,.1160900E-01,.1355700E-01,.1538100E-01,&
     & .1699300E-01,.1678500E-01,.5250900E-04,.4467400E-02,.7602200E-02,&
     & .1027100E-01,.1263000E-01,.1473600E-01,.1683000E-01,.1838700E-01,&
     & .1804000E-01,.5619600E-04,.4738100E-02,.8100500E-02,.1102600E-01,&
     & .1358600E-01,.1594000E-01,.1816000E-01,.1992000E-01,.1909200E-01,&
     & .3611200E-04,.3031600E-02,.4903800E-02,.6532500E-02,.7960600E-02,&
     & .9170600E-02,.1028500E-01,.1125200E-01,.1055600E-01,.3850200E-04,&
     & .3257500E-02,.5385500E-02,.7130200E-02,.8720000E-02,.1013800E-01,&
     & .1131900E-01,.1250400E-01,.1210900E-01,.4107000E-04,.3486900E-02,&
     & .5864300E-02,.7802000E-02,.9523200E-02,.1108100E-01,.1241300E-01,&
     & .1370000E-01,.1335000E-01,.4403400E-04,.3714200E-02,.6292400E-02,&
     & .8486700E-02,.1040000E-01,.1209100E-01,.1361400E-01,.1485000E-01,&
     & .1439600E-01,.4702100E-04,.3938700E-02,.6717100E-02,.9131900E-02,&
     & .1123100E-01,.1312900E-01,.1474500E-01,.1615500E-01,.1539200E-01,&
     & .3000100E-04,.2488400E-02,.3978700E-02,.5326200E-02,.6425400E-02,&
     & .7442200E-02,.8241800E-02,.8765100E-02,.8324100E-02,.3206700E-04,&
     & .2683100E-02,.4387300E-02,.5800900E-02,.7088100E-02,.8230300E-02,&
     & .9111500E-02,.9953000E-02,.9584300E-02,.3441400E-04,.2876200E-02,&
     & .4798100E-02,.6365000E-02,.7755500E-02,.9013000E-02,.1005100E-01,&
     & .1094400E-01,.1068700E-01,.3654300E-04,.3074900E-02,.5169700E-02,&
     & .6954900E-02,.8493900E-02,.9862100E-02,.1101200E-01,.1195000E-01,&
     & .1159600E-01,.3898600E-04,.3267200E-02,.5526800E-02,.7495000E-02,&
     & .9220600E-02,.1075000E-01,.1197000E-01,.1302900E-01,.1246100E-01/

      data   absa(316:450, 3)   /                                       &
     & .2471800E-04,.2030400E-02,.3225300E-02,.4295700E-02,.5162100E-02,&
     & .5947700E-02,.6554300E-02,.6790100E-02,.6657800E-02,.2622600E-04,&
     & .2202700E-02,.3552800E-02,.4684400E-02,.5746300E-02,.6607000E-02,&
     & .7356600E-02,.7908600E-02,.7680700E-02,.2809500E-04,.2361400E-02,&
     & .3896000E-02,.5152900E-02,.6277300E-02,.7291900E-02,.8147500E-02,&
     & .8734400E-02,.8650900E-02,.2985600E-04,.2534600E-02,.4218000E-02,&
     & .5650700E-02,.6886700E-02,.7994800E-02,.8916900E-02,.9622600E-02,&
     & .9421200E-02,.3143200E-04,.2698900E-02,.4528800E-02,.6115800E-02,&
     & .7523100E-02,.8731900E-02,.9744700E-02,.1047000E-01,.1017200E-01,&
     & .2015100E-04,.1662200E-02,.2610200E-02,.3434500E-02,.4151800E-02,&
     & .4735600E-02,.5119800E-02,.5290100E-02,.5560800E-02,.2129000E-04,&
     & .1811200E-02,.2862200E-02,.3777000E-02,.4609800E-02,.5282500E-02,&
     & .5871900E-02,.6211500E-02,.6428700E-02,.2260400E-04,.1948300E-02,&
     & .3153600E-02,.4141800E-02,.5051300E-02,.5888300E-02,.6519900E-02,&
     & .7000000E-02,.7164100E-02,.2403900E-04,.2090900E-02,.3432100E-02,&
     & .4563300E-02,.5557500E-02,.6445300E-02,.7180000E-02,.7706200E-02,&
     & .7782100E-02,.2527500E-04,.2232500E-02,.3697600E-02,.4972700E-02,&
     & .6094500E-02,.7063200E-02,.7869600E-02,.8404800E-02,.8457100E-02,&
     & .1652000E-04,.1357900E-02,.2121700E-02,.2765000E-02,.3320900E-02,&
     & .3765700E-02,.4026300E-02,.4163200E-02,.5901900E-02,.1730700E-04,&
     & .1486800E-02,.2323000E-02,.3061600E-02,.3700000E-02,.4262900E-02,&
     & .4695800E-02,.4895600E-02,.6472900E-02,.1821000E-04,.1608600E-02,&
     & .2563500E-02,.3348000E-02,.4082700E-02,.4759000E-02,.5242400E-02,&
     & .5608800E-02,.7107200E-02,.1926300E-04,.1730800E-02,.2803800E-02,&
     & .3701600E-02,.4493300E-02,.5211400E-02,.5822400E-02,.6205300E-02,&
     & .7644500E-02,.2045000E-04,.1851400E-02,.3028600E-02,.4056900E-02,&
     & .4941700E-02,.5729200E-02,.6380900E-02,.6816200E-02,.8180000E-02/

      data   absa(451:585, 3)   /                                       &
     & .1374300E-04,.1154400E-02,.1800500E-02,.2338000E-02,.2812700E-02,&
     & .3201900E-02,.3418400E-02,.3513700E-02,.5513200E-02,.1431400E-04,&
     & .1264800E-02,.1981100E-02,.2579400E-02,.3128300E-02,.3583500E-02,&
     & .3963500E-02,.4152400E-02,.6064000E-02,.1498000E-04,.1366700E-02,&
     & .2184600E-02,.2831600E-02,.3439000E-02,.4006900E-02,.4428300E-02,&
     & .4728200E-02,.6566200E-02,.1584300E-04,.1469100E-02,.2384400E-02,&
     & .3134600E-02,.3791500E-02,.4392600E-02,.4910800E-02,.5243100E-02,&
     & .7052100E-02,.1680100E-04,.1570900E-02,.2573800E-02,.3425000E-02,&
     & .4178100E-02,.4825800E-02,.5377600E-02,.5740000E-02,.7491300E-02,&
     & .1126500E-04,.9747300E-03,.1519000E-02,.1977900E-02,.2370800E-02,&
     & .2700200E-02,.2895400E-02,.2968900E-02,.4905200E-02,.1170300E-04,&
     & .1066000E-02,.1680600E-02,.2174200E-02,.2637000E-02,.3018400E-02,&
     & .3346100E-02,.3508100E-02,.5351500E-02,.1227500E-04,.1152400E-02,&
     & .1853200E-02,.2401400E-02,.2893200E-02,.3361500E-02,.3733200E-02,&
     & .3975900E-02,.5780600E-02,.1296300E-04,.1240500E-02,.2020000E-02,&
     & .2657300E-02,.3197800E-02,.3690400E-02,.4122500E-02,.4415100E-02,&
     & .6195600E-02,.1369600E-04,.1326600E-02,.2181400E-02,.2893900E-02,&
     & .3528400E-02,.4062900E-02,.4519100E-02,.4828300E-02,.6580000E-02,&
     & .9117900E-05,.8153100E-03,.1275400E-02,.1659800E-02,.1990000E-02,&
     & .2263400E-02,.2444100E-02,.2498400E-02,.4057600E-02,.9503900E-05,&
     & .8909300E-03,.1415000E-02,.1828700E-02,.2214300E-02,.2532900E-02,&
     & .2809200E-02,.2954600E-02,.4437400E-02,.9976200E-05,.9649400E-03,&
     & .1562500E-02,.2028800E-02,.2431900E-02,.2810800E-02,.3137300E-02,&
     & .3317700E-02,.4809400E-02,.1050900E-04,.1039800E-02,.1699600E-02,&
     & .2244300E-02,.2692500E-02,.3094200E-02,.3448100E-02,.3700300E-02,&
     & .5155000E-02,.1103300E-04,.1113400E-02,.1837200E-02,.2440800E-02,&
     & .2968200E-02,.3411400E-02,.3785900E-02,.4042400E-02,.5489300E-02/

      data   absa(  1:180, 4)   /                                       &
     & .2556400E-03,.2394000E-01,.4153500E-01,.5627600E-01,.6856100E-01,&
     & .7863000E-01,.8699300E-01,.9179400E-01,.8422900E-01,.2919400E-03,&
     & .2517000E-01,.4365900E-01,.5901600E-01,.7230200E-01,.8316700E-01,&
     & .9173900E-01,.9677400E-01,.8683800E-01,.3393400E-03,.2643000E-01,&
     & .4583500E-01,.6192200E-01,.7561400E-01,.8728500E-01,.9660600E-01,&
     & .1015600E+00,.9008400E-01,.3887100E-03,.2773300E-01,.4805800E-01,&
     & .6497300E-01,.7909000E-01,.9115600E-01,.1008400E+00,.1058900E+00,&
     & .9258300E-01,.4447300E-03,.2902800E-01,.5028700E-01,.6801000E-01,&
     & .8280500E-01,.9519300E-01,.1049200E+00,.1099500E+00,.9541900E-01,&
     & .2340000E-03,.2075300E-01,.3574200E-01,.4870900E-01,.5954800E-01,&
     & .6820600E-01,.7504500E-01,.7891500E-01,.7018000E-01,.2677500E-03,&
     & .2183200E-01,.3761100E-01,.5115600E-01,.6276800E-01,.7226700E-01,&
     & .7942200E-01,.8326200E-01,.7273500E-01,.3003300E-03,.2296100E-01,&
     & .3961800E-01,.5377400E-01,.6577300E-01,.7581900E-01,.8353200E-01,&
     & .8749600E-01,.7545900E-01,.3436600E-03,.2411600E-01,.4159700E-01,&
     & .5650600E-01,.6894400E-01,.7923100E-01,.8724500E-01,.9133600E-01,&
     & .7766900E-01,.3917900E-03,.2528900E-01,.4357500E-01,.5928200E-01,&
     & .7213200E-01,.8295600E-01,.9103800E-01,.9493700E-01,.8027200E-01,&
     & .2073000E-03,.1804400E-01,.3022700E-01,.4120600E-01,.5062600E-01,&
     & .5809500E-01,.6372500E-01,.6672100E-01,.5778000E-01,.2400700E-03,&
     & .1897200E-01,.3188300E-01,.4348600E-01,.5357200E-01,.6174200E-01,&
     & .6748900E-01,.7055200E-01,.6000900E-01,.2693100E-03,.1993400E-01,&
     & .3368100E-01,.4577500E-01,.5625500E-01,.6486900E-01,.7125400E-01,&
     & .7423200E-01,.6235000E-01,.3006400E-03,.2096500E-01,.3544800E-01,&
     & .4819900E-01,.5903600E-01,.6792100E-01,.7459400E-01,.7768900E-01,&
     & .6441000E-01,.3396700E-03,.2197200E-01,.3719200E-01,.5067600E-01,&
     & .6188400E-01,.7120700E-01,.7788800E-01,.8094700E-01,.6666700E-01,&
     & .1820100E-03,.1559500E-01,.2537200E-01,.3443100E-01,.4240100E-01,&
     & .4891500E-01,.5361300E-01,.5595300E-01,.4772700E-01,.2088300E-03,&
     & .1642200E-01,.2685100E-01,.3648100E-01,.4507400E-01,.5207800E-01,&
     & .5686000E-01,.5928700E-01,.4955300E-01,.2387400E-03,.1728500E-01,&
     & .2842500E-01,.3851400E-01,.4754200E-01,.5490400E-01,.6028400E-01,&
     & .6256100E-01,.5159500E-01,.2670700E-03,.1820000E-01,.3000700E-01,&
     & .4061300E-01,.4998200E-01,.5766200E-01,.6323100E-01,.6566600E-01,&
     & .5345000E-01,.3010200E-03,.1911300E-01,.3155200E-01,.4280500E-01,&
     & .5253500E-01,.6055900E-01,.6619500E-01,.6854900E-01,.5541500E-01/

      data   absa(181:315, 4)   /                                       &
     & .1580100E-03,.1329500E-01,.2152400E-01,.2866600E-01,.3514200E-01,&
     & .4075700E-01,.4471500E-01,.4674000E-01,.3938400E-01,.1799400E-03,&
     & .1405700E-01,.2278600E-01,.3048700E-01,.3756300E-01,.4350900E-01,&
     & .4767600E-01,.4963800E-01,.4084900E-01,.2053500E-03,.1481100E-01,&
     & .2411600E-01,.3226600E-01,.3976400E-01,.4612200E-01,.5066900E-01,&
     & .5241300E-01,.4268300E-01,.2351100E-03,.1559800E-01,.2548100E-01,&
     & .3410300E-01,.4192400E-01,.4859200E-01,.5328300E-01,.5523500E-01,&
     & .4433500E-01,.2664000E-03,.1640200E-01,.2684400E-01,.3600000E-01,&
     & .4418000E-01,.5108600E-01,.5591800E-01,.5779600E-01,.4599100E-01,&
     & .1380500E-03,.1122200E-01,.1812500E-01,.2387700E-01,.2893900E-01,&
     & .3350600E-01,.3694400E-01,.3867200E-01,.3233500E-01,.1543900E-03,&
     & .1188300E-01,.1926200E-01,.2552100E-01,.3106800E-01,.3588200E-01,&
     & .3958900E-01,.4113200E-01,.3368400E-01,.1756100E-03,.1255800E-01,&
     & .2040900E-01,.2703300E-01,.3305600E-01,.3826000E-01,.4215500E-01,&
     & .4363900E-01,.3517500E-01,.1996700E-03,.1323600E-01,.2162800E-01,&
     & .2855400E-01,.3493900E-01,.4045900E-01,.4455000E-01,.4614700E-01,&
     & .3664300E-01,.2311700E-03,.1393600E-01,.2283600E-01,.3018300E-01,&
     & .3690900E-01,.4265200E-01,.4686400E-01,.4844000E-01,.3799200E-01,&
     & .1198700E-03,.9398300E-02,.1513000E-01,.1980800E-01,.2379600E-01,&
     & .2728500E-01,.3017600E-01,.3190600E-01,.2619300E-01,.1316400E-03,&
     & .9974300E-02,.1611800E-01,.2131900E-01,.2560000E-01,.2940400E-01,&
     & .3248500E-01,.3390900E-01,.2743900E-01,.1486200E-03,.1055900E-01,&
     & .1712400E-01,.2267400E-01,.2737300E-01,.3150100E-01,.3470500E-01,&
     & .3611200E-01,.2866700E-01,.1694100E-03,.1114200E-01,.1816300E-01,&
     & .2399200E-01,.2902300E-01,.3346100E-01,.3688200E-01,.3828800E-01,&
     & .2999000E-01,.1944400E-03,.1174000E-01,.1923100E-01,.2541000E-01,&
     & .3070200E-01,.3538000E-01,.3893300E-01,.4032500E-01,.3116300E-01/

      data   absa(316:450, 4)   /                                       &
     & .1026600E-03,.7813200E-02,.1249500E-01,.1629400E-01,.1955200E-01,&
     & .2220300E-01,.2452200E-01,.2615800E-01,.2147700E-01,.1130800E-03,&
     & .8299600E-02,.1338500E-01,.1765000E-01,.2111300E-01,.2402600E-01,&
     & .2642400E-01,.2777900E-01,.2266100E-01,.1243100E-03,.8811800E-02,&
     & .1425400E-01,.1885200E-01,.2271700E-01,.2582200E-01,.2835200E-01,&
     & .2966700E-01,.2367400E-01,.1414600E-03,.9308000E-02,.1515200E-01,&
     & .2001300E-01,.2415800E-01,.2756900E-01,.3030600E-01,.3149500E-01,&
     & .2483800E-01,.1627500E-03,.9828700E-02,.1608000E-01,.2122700E-01,&
     & .2559200E-01,.2922600E-01,.3207500E-01,.3335700E-01,.2588700E-01,&
     & .8579700E-04,.6441200E-02,.1023100E-01,.1332700E-01,.1592300E-01,&
     & .1810300E-01,.1990300E-01,.2118900E-01,.1797500E-01,.9437000E-04,&
     & .6851400E-02,.1104100E-01,.1447200E-01,.1729800E-01,.1966100E-01,&
     & .2142100E-01,.2263100E-01,.1899000E-01,.1060300E-03,.7287000E-02,&
     & .1178600E-01,.1557000E-01,.1871000E-01,.2119500E-01,.2309200E-01,&
     & .2415700E-01,.1993400E-01,.1181900E-03,.7721700E-02,.1255900E-01,&
     & .1658900E-01,.1999100E-01,.2273800E-01,.2476900E-01,.2577100E-01,&
     & .2094600E-01,.1350000E-03,.8162400E-02,.1334600E-01,.1761000E-01,&
     & .2122700E-01,.2414000E-01,.2633900E-01,.2738800E-01,.2181900E-01,&
     & .7327500E-04,.5314700E-02,.8383200E-02,.1089000E-01,.1299800E-01,&
     & .1477200E-01,.1623700E-01,.1716500E-01,.1597200E-01,.8030300E-04,&
     & .5659400E-02,.9096400E-02,.1186900E-01,.1418100E-01,.1606500E-01,&
     & .1749300E-01,.1844700E-01,.1707500E-01,.8882100E-04,.6028600E-02,&
     & .9749100E-02,.1285200E-01,.1538700E-01,.1739400E-01,.1890900E-01,&
     & .1970800E-01,.1807800E-01,.1014200E-03,.6396200E-02,.1039200E-01,&
     & .1373000E-01,.1652200E-01,.1877100E-01,.2030800E-01,.2109600E-01,&
     & .1903700E-01,.1140500E-03,.6775400E-02,.1106500E-01,.1460500E-01,&
     & .1758100E-01,.1999100E-01,.2168700E-01,.2245800E-01,.1987500E-01/

      data   absa(451:585, 4)   /                                       &
     & .6451700E-04,.4513500E-02,.7116500E-02,.9238400E-02,.1099800E-01,&
     & .1247400E-01,.1370000E-01,.1438800E-01,.1463100E-01,.7153500E-04,&
     & .4807000E-02,.7713700E-02,.1009200E-01,.1202300E-01,.1362300E-01,&
     & .1479200E-01,.1544900E-01,.1548100E-01,.7950400E-04,.5119900E-02,&
     & .8260000E-02,.1091100E-01,.1307600E-01,.1475500E-01,.1600200E-01,&
     & .1654600E-01,.1629800E-01,.9043300E-04,.5435100E-02,.8810100E-02,&
     & .1164700E-01,.1401800E-01,.1592200E-01,.1720600E-01,.1771500E-01,&
     & .1699100E-01,.1014600E-03,.5759600E-02,.9386500E-02,.1240700E-01,&
     & .1492200E-01,.1695100E-01,.1838400E-01,.1889300E-01,.1766600E-01,&
     & .5691300E-04,.3836100E-02,.6023800E-02,.7810100E-02,.9294300E-02,&
     & .1051500E-01,.1151400E-01,.1205100E-01,.1270800E-01,.6298400E-04,&
     & .4088600E-02,.6515300E-02,.8546400E-02,.1016800E-01,.1149800E-01,&
     & .1245200E-01,.1293800E-01,.1347300E-01,.7058800E-04,.4356100E-02,&
     & .6976100E-02,.9213700E-02,.1106800E-01,.1248200E-01,.1349700E-01,&
     & .1389200E-01,.1409000E-01,.7955600E-04,.4622200E-02,.7447200E-02,&
     & .9838200E-02,.1185200E-01,.1346100E-01,.1453900E-01,.1486800E-01,&
     & .1468400E-01,.8933000E-04,.4903700E-02,.7933200E-02,.1049200E-01,&
     & .1261300E-01,.1433100E-01,.1552300E-01,.1586700E-01,.1526600E-01,&
     & .4904000E-04,.3247000E-02,.5078200E-02,.6581300E-02,.7825400E-02,&
     & .8839800E-02,.9648300E-02,.1009000E-01,.1061400E-01,.5422500E-04,&
     & .3461600E-02,.5490600E-02,.7195100E-02,.8571100E-02,.9678100E-02,&
     & .1046000E-01,.1084000E-01,.1120600E-01,.6113600E-04,.3687800E-02,&
     & .5877700E-02,.7745800E-02,.9318300E-02,.1052800E-01,.1134700E-01,&
     & .1168400E-01,.1173000E-01,.6878800E-04,.3921300E-02,.6279100E-02,&
     & .8277400E-02,.9976100E-02,.1133200E-01,.1224700E-01,.1249500E-01,&
     & .1225000E-01,.7742700E-04,.4164000E-02,.6689000E-02,.8835800E-02,&
     & .1062400E-01,.1206900E-01,.1306400E-01,.1334100E-01,.1273000E-01/

      data   absa(  1:180, 5)   /                                       &
     & .6909800E-02,.8465000E-01,.1300600E+00,.1665200E+00,.1960800E+00,&
     & .2197500E+00,.2349600E+00,.2327700E+00,.1800700E+00,.7820000E-02,&
     & .8927900E-01,.1363000E+00,.1740900E+00,.2039800E+00,.2271800E+00,&
     & .2412900E+00,.2375000E+00,.1843800E+00,.8978700E-02,.9442800E-01,&
     & .1429600E+00,.1817000E+00,.2120500E+00,.2345000E+00,.2472600E+00,&
     & .2417200E+00,.1887800E+00,.1042600E-01,.1000300E+00,.1500900E+00,&
     & .1894300E+00,.2201500E+00,.2420600E+00,.2534100E+00,.2457100E+00,&
     & .1932500E+00,.1220900E-01,.1060300E+00,.1578100E+00,.1975600E+00,&
     & .2283200E+00,.2496300E+00,.2595700E+00,.2495800E+00,.1974800E+00,&
     & .6402000E-02,.7459700E-01,.1145800E+00,.1458100E+00,.1710300E+00,&
     & .1910500E+00,.2041200E+00,.2013300E+00,.1524200E+00,.7258600E-02,&
     & .7885700E-01,.1202400E+00,.1525700E+00,.1779600E+00,.1976500E+00,&
     & .2097200E+00,.2054800E+00,.1562000E+00,.8290300E-02,.8360200E-01,&
     & .1262300E+00,.1593100E+00,.1851200E+00,.2042300E+00,.2150300E+00,&
     & .2093700E+00,.1599300E+00,.9457800E-02,.8866800E-01,.1327700E+00,&
     & .1663500E+00,.1923300E+00,.2109800E+00,.2205000E+00,.2129400E+00,&
     & .1639700E+00,.1086500E-01,.9411300E-01,.1398700E+00,.1737500E+00,&
     & .1997500E+00,.2177100E+00,.2259200E+00,.2163900E+00,.1674500E+00,&
     & .5694300E-02,.6408400E-01,.9929400E-01,.1261400E+00,.1474600E+00,&
     & .1641400E+00,.1750300E+00,.1725500E+00,.1273800E+00,.6470200E-02,&
     & .6796700E-01,.1042700E+00,.1319700E+00,.1535000E+00,.1698500E+00,&
     & .1800800E+00,.1763100E+00,.1308700E+00,.7420000E-02,.7225200E-01,&
     & .1095600E+00,.1380100E+00,.1597800E+00,.1756700E+00,.1847600E+00,&
     & .1797700E+00,.1342100E+00,.8490700E-02,.7677200E-01,.1153900E+00,&
     & .1442700E+00,.1661500E+00,.1816800E+00,.1894700E+00,.1830000E+00,&
     & .1376100E+00,.9707800E-02,.8167100E-01,.1217200E+00,.1508200E+00,&
     & .1727700E+00,.1876900E+00,.1942700E+00,.1860800E+00,.1406700E+00,&
     & .4882900E-02,.5438300E-01,.8507500E-01,.1081500E+00,.1263800E+00,&
     & .1402000E+00,.1490300E+00,.1467000E+00,.1061400E+00,.5574000E-02,&
     & .5777900E-01,.8943600E-01,.1132300E+00,.1316000E+00,.1451400E+00,&
     & .1534900E+00,.1502700E+00,.1094900E+00,.6401500E-02,.6154600E-01,&
     & .9413900E-01,.1185000E+00,.1370100E+00,.1501700E+00,.1575700E+00,&
     & .1534400E+00,.1125700E+00,.7371900E-02,.6552100E-01,.9923200E-01,&
     & .1240300E+00,.1425900E+00,.1554100E+00,.1617500E+00,.1563400E+00,&
     & .1155700E+00,.8478900E-02,.6981000E-01,.1047800E+00,.1298500E+00,&
     & .1484000E+00,.1607100E+00,.1659700E+00,.1591000E+00,.1183300E+00/

      data   absa(181:315, 5)   /                                       &
     & .4135700E-02,.4587800E-01,.7186600E-01,.9213000E-01,.1077200E+00,&
     & .1193500E+00,.1262700E+00,.1238200E+00,.8849000E-01,.4727800E-02,&
     & .4880900E-01,.7572400E-01,.9648500E-01,.1121800E+00,.1236300E+00,&
     & .1302300E+00,.1272300E+00,.9164600E-01,.5423400E-02,.5210800E-01,&
     & .7992500E-01,.1010500E+00,.1168400E+00,.1279500E+00,.1338900E+00,&
     & .1302400E+00,.9438400E-01,.6267200E-02,.5564700E-01,.8443800E-01,&
     & .1059100E+00,.1216700E+00,.1325500E+00,.1375900E+00,.1328800E+00,&
     & .9703700E-01,.7238700E-02,.5939300E-01,.8932600E-01,.1110200E+00,&
     & .1268000E+00,.1372300E+00,.1412800E+00,.1353000E+00,.9958500E-01,&
     & .3482000E-02,.3845500E-01,.6010600E-01,.7751500E-01,.9114300E-01,&
     & .1009400E+00,.1064900E+00,.1040300E+00,.7350900E-01,.3950700E-02,&
     & .4092700E-01,.6340200E-01,.8124000E-01,.9496800E-01,.1046900E+00,&
     & .1100100E+00,.1071700E+00,.7642600E-01,.4526600E-02,.4372600E-01,&
     & .6704100E-01,.8523400E-01,.9891100E-01,.1084500E+00,.1132900E+00,&
     & .1099600E+00,.7886100E-01,.5224200E-02,.4680700E-01,.7093900E-01,&
     & .8954700E-01,.1031200E+00,.1124700E+00,.1165300E+00,.1123700E+00,&
     & .8125000E-01,.6044900E-02,.5008300E-01,.7517500E-01,.9404100E-01,&
     & .1076000E+00,.1165700E+00,.1198000E+00,.1144900E+00,.8354900E-01,&
     & .2871100E-02,.3252300E-01,.4997100E-01,.6459700E-01,.7636000E-01,&
     & .8491800E-01,.8952300E-01,.8720000E-01,.6061800E-01,.3255000E-02,&
     & .3461400E-01,.5280300E-01,.6774000E-01,.7971300E-01,.8820700E-01,&
     & .9268100E-01,.9006000E-01,.6333100E-01,.3726300E-02,.3696800E-01,&
     & .5590400E-01,.7115300E-01,.8314300E-01,.9150900E-01,.9560700E-01,&
     & .9256400E-01,.6559500E-01,.4295300E-02,.3953800E-01,.5928100E-01,&
     & .7491200E-01,.8682200E-01,.9495600E-01,.9843800E-01,.9472800E-01,&
     & .6768000E-01,.4978800E-02,.4226400E-01,.6295300E-01,.7884200E-01,&
     & .9074200E-01,.9851000E-01,.1012900E+00,.9664100E-01,.6974600E-01/

      data   absa(316:450, 5)   /                                       &
     & .2348000E-02,.2723000E-01,.4167100E-01,.5356500E-01,.6343600E-01,&
     & .7099900E-01,.7502000E-01,.7290600E-01,.4938600E-01,.2651700E-02,&
     & .2906500E-01,.4401700E-01,.5622500E-01,.6634100E-01,.7393500E-01,&
     & .7787900E-01,.7546800E-01,.5175000E-01,.3028500E-02,.3111200E-01,&
     & .4659800E-01,.5916100E-01,.6930200E-01,.7680900E-01,.8043900E-01,&
     & .7772900E-01,.5390400E-01,.3481600E-02,.3334800E-01,.4944800E-01,&
     & .6238000E-01,.7250200E-01,.7978600E-01,.8288700E-01,.7968300E-01,&
     & .5573100E-01,.4044400E-02,.3573000E-01,.5254500E-01,.6579400E-01,&
     & .7587800E-01,.8288900E-01,.8537900E-01,.8139100E-01,.5750900E-01,&
     & .1924700E-02,.2263000E-01,.3475200E-01,.4438700E-01,.5245500E-01,&
     & .5883300E-01,.6246800E-01,.6071900E-01,.4188000E-01,.2155700E-02,&
     & .2420400E-01,.3675000E-01,.4663600E-01,.5497600E-01,.6142100E-01,&
     & .6505200E-01,.6299400E-01,.4401000E-01,.2447700E-02,.2595800E-01,&
     & .3898100E-01,.4910800E-01,.5752600E-01,.6390700E-01,.6735100E-01,&
     & .6503800E-01,.4592900E-01,.2819600E-02,.2788200E-01,.4143100E-01,&
     & .5180200E-01,.6024900E-01,.6648000E-01,.6946700E-01,.6678800E-01,&
     & .4755500E-01,.3265400E-02,.2993900E-01,.4407200E-01,.5471400E-01,&
     & .6316300E-01,.6920700E-01,.7160000E-01,.6832800E-01,.4907700E-01,&
     & .1620000E-02,.1881400E-01,.2891100E-01,.3702500E-01,.4347900E-01,&
     & .4869100E-01,.5176700E-01,.5051700E-01,.3656500E-01,.1803100E-02,&
     & .2015000E-01,.3063200E-01,.3896700E-01,.4560900E-01,.5094000E-01,&
     & .5404600E-01,.5251500E-01,.3842500E-01,.2023200E-02,.2164000E-01,&
     & .3257200E-01,.4107900E-01,.4778800E-01,.5310100E-01,.5611400E-01,&
     & .5433800E-01,.4016200E-01,.2303800E-02,.2329800E-01,.3470300E-01,&
     & .4339300E-01,.5011000E-01,.5529800E-01,.5798200E-01,.5587300E-01,&
     & .4178200E-01,.2664900E-02,.2507300E-01,.3698300E-01,.4586800E-01,&
     & .5261700E-01,.5764100E-01,.5983300E-01,.5722200E-01,.4324300E-01/

      data   absa(451:585, 5)   /                                       &
     & .1424100E-02,.1606500E-01,.2458600E-01,.3146500E-01,.3696300E-01,&
     & .4109300E-01,.4361700E-01,.4267300E-01,.3234300E-01,.1587800E-02,&
     & .1726800E-01,.2614400E-01,.3316900E-01,.3881600E-01,.4298800E-01,&
     & .4554600E-01,.4432700E-01,.3405400E-01,.1800300E-02,.1860900E-01,&
     & .2788800E-01,.3506600E-01,.4071700E-01,.4486100E-01,.4727700E-01,&
     & .4584500E-01,.3556200E-01,.2051300E-02,.2006900E-01,.2979400E-01,&
     & .3714700E-01,.4278100E-01,.4677400E-01,.4886900E-01,.4710600E-01,&
     & .3688100E-01,.2361000E-02,.2163600E-01,.3178600E-01,.3934100E-01,&
     & .4496400E-01,.4883200E-01,.5047800E-01,.4823700E-01,.3813600E-01,&
     & .1236700E-02,.1366400E-01,.2084100E-01,.2661900E-01,.3125800E-01,&
     & .3476600E-01,.3666500E-01,.3587300E-01,.2781100E-01,.1389800E-02,&
     & .1474700E-01,.2223800E-01,.2813400E-01,.3287600E-01,.3639200E-01,&
     & .3828600E-01,.3725700E-01,.2921200E-01,.1582200E-02,.1592000E-01,&
     & .2380800E-01,.2983000E-01,.3455800E-01,.3801400E-01,.3972800E-01,&
     & .3849300E-01,.3051700E-01,.1811700E-02,.1719500E-01,.2547900E-01,&
     & .3166400E-01,.3637800E-01,.3967500E-01,.4109800E-01,.3955300E-01,&
     & .3170500E-01,.2099600E-02,.1856700E-01,.2722500E-01,.3357100E-01,&
     & .3830200E-01,.4143700E-01,.4250100E-01,.4052500E-01,.3274800E-01,&
     & .1060600E-02,.1159400E-01,.1761600E-01,.2245600E-01,.2633700E-01,&
     & .2928200E-01,.3083100E-01,.3003700E-01,.2325100E-01,.1197500E-02,&
     & .1254100E-01,.1885000E-01,.2379500E-01,.2774500E-01,.3067600E-01,&
     & .3219100E-01,.3118600E-01,.2446100E-01,.1368600E-02,.1355700E-01,&
     & .2023600E-01,.2528700E-01,.2922400E-01,.3208500E-01,.3341100E-01,&
     & .3219000E-01,.2553900E-01,.1578700E-02,.1465500E-01,.2168500E-01,&
     & .2688700E-01,.3082100E-01,.3353300E-01,.3458100E-01,.3309100E-01,&
     & .2648300E-01,.1842200E-02,.1584700E-01,.2320100E-01,.2853500E-01,&
     & .3248600E-01,.3505900E-01,.3579100E-01,.3394200E-01,.2734000E-01/

      data   absa(  1:180, 6)   /                                       &
     & .3839378E+00,.6279180E+00,.7318272E+00,.7920608E+00,.8216785E+00,&
     & .8230008E+00,.7947126E+00,.7302326E+00,.5811177E+00,.4128871E+00,&
     & .6485557E+00,.7485293E+00,.8043995E+00,.8308183E+00,.8293298E+00,&
     & .7988096E+00,.7334673E+00,.5866991E+00,.4408435E+00,.6690805E+00,&
     & .7652913E+00,.8177871E+00,.8406473E+00,.8361296E+00,.8033672E+00,&
     & .7370020E+00,.5912609E+00,.4672892E+00,.6894163E+00,.7820258E+00,&
     & .8313814E+00,.8509556E+00,.8436351E+00,.8085369E+00,.7402632E+00,&
     & .5956103E+00,.4927709E+00,.7092005E+00,.7986459E+00,.8452246E+00,&
     & .8615155E+00,.8516325E+00,.8138231E+00,.7437317E+00,.5997054E+00,&
     & .3325679E+00,.5548358E+00,.6461479E+00,.6995525E+00,.7253118E+00,&
     & .7266814E+00,.7024839E+00,.6480527E+00,.5065275E+00,.3575484E+00,&
     & .5737759E+00,.6620339E+00,.7123154E+00,.7348811E+00,.7336335E+00,&
     & .7074669E+00,.6519501E+00,.5119171E+00,.3814205E+00,.5923353E+00,&
     & .6775822E+00,.7253565E+00,.7450096E+00,.7412995E+00,.7131653E+00,&
     & .6558395E+00,.5167702E+00,.4042236E+00,.6103975E+00,.6930426E+00,&
     & .7383493E+00,.7554787E+00,.7494988E+00,.7190075E+00,.6598135E+00,&
     & .5211434E+00,.4265822E+00,.6282698E+00,.7082747E+00,.7513798E+00,&
     & .7663755E+00,.7576413E+00,.7252801E+00,.6640475E+00,.5252205E+00,&
     & .2819028E+00,.4836690E+00,.5649400E+00,.6121709E+00,.6347883E+00,&
     & .6367827E+00,.6170448E+00,.5707116E+00,.4371090E+00,.3032609E+00,&
     & .5006309E+00,.5795105E+00,.6243936E+00,.6444057E+00,.6442898E+00,&
     & .6228572E+00,.5750819E+00,.4422485E+00,.3236861E+00,.5170268E+00,&
     & .5937689E+00,.6365222E+00,.6544181E+00,.6524051E+00,.6289736E+00,&
     & .5795020E+00,.4469729E+00,.3434421E+00,.5330792E+00,.6078864E+00,&
     & .6485227E+00,.6647491E+00,.6605735E+00,.6354043E+00,.5840256E+00,&
     & .4512681E+00,.3630007E+00,.5490413E+00,.6217655E+00,.6606764E+00,&
     & .6752251E+00,.6688803E+00,.6420320E+00,.5886154E+00,.4550732E+00,&
     & .2364425E+00,.4182373E+00,.4906262E+00,.5320011E+00,.5523509E+00,&
     & .5548703E+00,.5383899E+00,.4987899E+00,.3768887E+00,.2546801E+00,&
     & .4331987E+00,.5037554E+00,.5432869E+00,.5616903E+00,.5626459E+00,&
     & .5445887E+00,.5034773E+00,.3818393E+00,.2721315E+00,.4476600E+00,&
     & .5164741E+00,.5543752E+00,.5713544E+00,.5708581E+00,.5510757E+00,&
     & .5081946E+00,.3862452E+00,.2892043E+00,.4618692E+00,.5290503E+00,&
     & .5654304E+00,.5811711E+00,.5789470E+00,.5577046E+00,.5130234E+00,&
     & .3902090E+00,.3062774E+00,.4761272E+00,.5414198E+00,.5765335E+00,&
     & .5909044E+00,.5871126E+00,.5644694E+00,.5178294E+00,.3937875E+00/

      data   absa(181:315, 6)   /                                       &
     & .1976097E+00,.3599556E+00,.4238261E+00,.4601500E+00,.4784937E+00,&
     & .4808783E+00,.4668807E+00,.4328773E+00,.3243113E+00,.2131404E+00,&
     & .3730680E+00,.4353477E+00,.4703740E+00,.4874113E+00,.4885689E+00,&
     & .4731711E+00,.4377759E+00,.3288869E+00,.2280961E+00,.3857179E+00,&
     & .4465292E+00,.4804726E+00,.4964950E+00,.4964579E+00,.4796823E+00,&
     & .4426578E+00,.3330323E+00,.2427415E+00,.3981437E+00,.4574729E+00,&
     & .4906220E+00,.5055377E+00,.5041976E+00,.4863363E+00,.4474608E+00,&
     & .3367502E+00,.2574916E+00,.4108035E+00,.4685214E+00,.5005586E+00,&
     & .5144711E+00,.5119125E+00,.4929884E+00,.4523415E+00,.3400646E+00,&
     & .1644599E+00,.3082155E+00,.3641066E+00,.3958133E+00,.4120577E+00,&
     & .4144583E+00,.4024002E+00,.3731548E+00,.2772332E+00,.1776815E+00,&
     & .3195976E+00,.3741572E+00,.4049902E+00,.4203131E+00,.4217397E+00,&
     & .4086238E+00,.3780822E+00,.2815278E+00,.1904154E+00,.3305911E+00,&
     & .3838882E+00,.4140681E+00,.4286082E+00,.4290731E+00,.4150352E+00,&
     & .3829532E+00,.2854673E+00,.2029594E+00,.3414094E+00,.3935449E+00,&
     & .4230885E+00,.4367885E+00,.4362952E+00,.4213065E+00,.3877874E+00,&
     & .2889571E+00,.2156141E+00,.3525176E+00,.4033758E+00,.4319990E+00,&
     & .4448634E+00,.4434057E+00,.4275549E+00,.3926902E+00,.2921021E+00,&
     & .1365354E+00,.2627938E+00,.3116032E+00,.3390470E+00,.3531982E+00,&
     & .3553931E+00,.3450681E+00,.3195057E+00,.2355642E+00,.1476945E+00,&
     & .2725931E+00,.3203697E+00,.3471668E+00,.3606270E+00,.3621563E+00,&
     & .3510351E+00,.3243608E+00,.2395644E+00,.1584917E+00,.2820807E+00,&
     & .3288732E+00,.3551503E+00,.3679817E+00,.3688657E+00,.3570843E+00,&
     & .3290971E+00,.2432894E+00,.1692002E+00,.2914586E+00,.3373668E+00,&
     & .3630273E+00,.3752511E+00,.3754063E+00,.3629045E+00,.3338612E+00,&
     & .2465948E+00,.1800234E+00,.3011255E+00,.3460817E+00,.3709814E+00,&
     & .3824342E+00,.3817964E+00,.3686830E+00,.3386098E+00,.2495277E+00/

      data   absa(316:450, 6)   /                                       &
     & .1130524E+00,.2235733E+00,.2655973E+00,.2894880E+00,.3015165E+00,&
     & .3032589E+00,.2943333E+00,.2719817E+00,.1988483E+00,.1224152E+00,&
     & .2318861E+00,.2733317E+00,.2965534E+00,.3081331E+00,.3094246E+00,&
     & .2999446E+00,.2766238E+00,.2026511E+00,.1315214E+00,.2399609E+00,&
     & .2807287E+00,.3034743E+00,.3145900E+00,.3154081E+00,.3054644E+00,&
     & .2811905E+00,.2060707E+00,.1406060E+00,.2479709E+00,.2881554E+00,&
     & .3103325E+00,.3209441E+00,.3212333E+00,.3107742E+00,.2857474E+00,&
     & .2091635E+00,.1498138E+00,.2563308E+00,.2958019E+00,.3173622E+00,&
     & .3271974E+00,.3269371E+00,.3159983E+00,.2902250E+00,.2119359E+00,&
     & .9301188E-01,.1895129E+00,.2254431E+00,.2461449E+00,.2564128E+00,&
     & .2576877E+00,.2498578E+00,.2303767E+00,.1659328E+00,.1009065E+00,&
     & .1965526E+00,.2321100E+00,.2522877E+00,.2622065E+00,.2632244E+00,&
     & .2549826E+00,.2347053E+00,.1694537E+00,.1085578E+00,.2034287E+00,&
     & .2384465E+00,.2583058E+00,.2678405E+00,.2684545E+00,.2599040E+00,&
     & .2390279E+00,.1725782E+00,.1162137E+00,.2102907E+00,.2448387E+00,&
     & .2642935E+00,.2733148E+00,.2735764E+00,.2646745E+00,.2432888E+00,&
     & .1754745E+00,.1240561E+00,.2175246E+00,.2515388E+00,.2703899E+00,&
     & .2788204E+00,.2785885E+00,.2693407E+00,.2473703E+00,.1781073E+00,&
     & .7640796E-01,.1603808E+00,.1911818E+00,.2086778E+00,.2175552E+00,&
     & .2186307E+00,.2116034E+00,.1944515E+00,.1456540E+00,.8306947E-01,&
     & .1663339E+00,.1968047E+00,.2139806E+00,.2226600E+00,.2234720E+00,&
     & .2162217E+00,.1984391E+00,.1487611E+00,.8952871E-01,.1721996E+00,&
     & .2022331E+00,.2191642E+00,.2275327E+00,.2280233E+00,.2205353E+00,&
     & .2024797E+00,.1513787E+00,.9600912E-01,.1780570E+00,.2077289E+00,&
     & .2243376E+00,.2322476E+00,.2324865E+00,.2247285E+00,.2063451E+00,&
     & .1537312E+00,.1026970E+00,.1842930E+00,.2135640E+00,.2296382E+00,&
     & .2370510E+00,.2368348E+00,.2288014E+00,.2100442E+00,.1559266E+00/

      data   absa(451:585, 6)   /                                       &
     & .6498651E-01,.1372893E+00,.1636307E+00,.1784508E+00,.1858229E+00,&
     & .1867364E+00,.1805030E+00,.1651282E+00,.1253243E+00,.7051877E-01,&
     & .1422924E+00,.1682882E+00,.1829691E+00,.1901293E+00,.1907854E+00,&
     & .1844056E+00,.1688531E+00,.1278858E+00,.7594246E-01,.1472381E+00,&
     & .1729539E+00,.1873767E+00,.1942330E+00,.1947083E+00,.1880933E+00,&
     & .1724213E+00,.1301325E+00,.8152013E-01,.1524039E+00,.1778017E+00,&
     & .1918315E+00,.1983471E+00,.1985237E+00,.1916908E+00,.1757953E+00,&
     & .1322663E+00,.8732427E-01,.1579555E+00,.1829727E+00,.1965737E+00,&
     & .2026361E+00,.2023437E+00,.1952484E+00,.1791155E+00,.1342170E+00,&
     & .5506321E-01,.1168681E+00,.1392111E+00,.1518638E+00,.1581057E+00,&
     & .1587078E+00,.1534476E+00,.1400850E+00,.1080712E+00,.5967938E-01,&
     & .1210769E+00,.1431802E+00,.1556769E+00,.1617277E+00,.1621446E+00,&
     & .1567559E+00,.1434036E+00,.1102933E+00,.6432180E-01,.1253570E+00,&
     & .1472032E+00,.1594570E+00,.1652341E+00,.1654796E+00,.1599072E+00,&
     & .1464712E+00,.1123711E+00,.6912124E-01,.1299527E+00,.1515000E+00,&
     & .1634309E+00,.1688516E+00,.1687937E+00,.1630152E+00,.1494484E+00,&
     & .1143749E+00,.7411322E-01,.1349524E+00,.1561843E+00,.1676813E+00,&
     & .1726787E+00,.1722628E+00,.1662240E+00,.1523717E+00,.1162707E+00,&
     & .4671501E-01,.9906607E-01,.1179218E+00,.1285633E+00,.1338684E+00,&
     & .1344469E+00,.1299950E+00,.1186902E+00,.9160652E-01,.5062345E-01,&
     & .1026419E+00,.1213067E+00,.1318081E+00,.1369356E+00,.1373840E+00,&
     & .1328006E+00,.1215065E+00,.9359446E-01,.5459974E-01,.1064175E+00,&
     & .1248279E+00,.1351247E+00,.1399943E+00,.1402452E+00,.1355160E+00,&
     & .1241771E+00,.9556170E-01,.5873868E-01,.1105265E+00,.1286741E+00,&
     & .1386392E+00,.1431974E+00,.1431960E+00,.1382828E+00,.1267833E+00,&
     & .9738167E-01,.6302475E-01,.1149971E+00,.1329313E+00,.1425338E+00,&
     & .1467358E+00,.1463643E+00,.1411782E+00,.1293360E+00,.9929140E-01/

      data   absa(  1:180, 7)   /                                       &
     & .2084400E+01,.2334400E+01,.2557600E+01,.2670100E+01,.2701000E+01,&
     & .2668100E+01,.2568300E+01,.2365500E+01,.2081000E+01,.2080700E+01,&
     & .2330400E+01,.2548900E+01,.2661100E+01,.2692000E+01,.2660900E+01,&
     & .2564700E+01,.2364500E+01,.2085900E+01,.2087700E+01,.2334200E+01,&
     & .2547700E+01,.2654800E+01,.2684300E+01,.2654100E+01,.2560500E+01,&
     & .2363700E+01,.2090300E+01,.2105200E+01,.2345400E+01,.2552000E+01,&
     & .2652900E+01,.2677800E+01,.2647800E+01,.2556500E+01,.2363000E+01,&
     & .2093500E+01,.2130300E+01,.2363300E+01,.2561900E+01,.2654600E+01,&
     & .2674900E+01,.2642900E+01,.2554200E+01,.2361900E+01,.2095800E+01,&
     & .1812000E+01,.2110100E+01,.2320400E+01,.2426600E+01,.2465700E+01,&
     & .2446900E+01,.2370200E+01,.2193100E+01,.1866900E+01,.1818700E+01,&
     & .2113700E+01,.2318800E+01,.2422300E+01,.2462000E+01,.2443600E+01,&
     & .2368700E+01,.2194400E+01,.1872600E+01,.1835600E+01,.2125500E+01,&
     & .2325600E+01,.2423200E+01,.2459000E+01,.2440700E+01,.2367400E+01,&
     & .2196000E+01,.1877600E+01,.1861400E+01,.2144700E+01,.2338600E+01,&
     & .2428700E+01,.2459500E+01,.2438700E+01,.2367200E+01,.2197000E+01,&
     & .1881600E+01,.1892700E+01,.2168300E+01,.2357900E+01,.2439200E+01,&
     & .2463600E+01,.2440700E+01,.2366500E+01,.2197400E+01,.1886400E+01,&
     & .1570200E+01,.1894200E+01,.2083700E+01,.2185400E+01,.2232400E+01,&
     & .2226600E+01,.2164900E+01,.2013400E+01,.1661700E+01,.1582700E+01,&
     & .1903800E+01,.2088600E+01,.2186500E+01,.2232900E+01,.2226600E+01,&
     & .2164900E+01,.2016700E+01,.1669500E+01,.1604600E+01,.1921300E+01,&
     & .2101900E+01,.2194200E+01,.2236000E+01,.2227100E+01,.2166500E+01,&
     & .2019500E+01,.1676200E+01,.1632900E+01,.1944300E+01,.2121300E+01,&
     & .2207000E+01,.2242400E+01,.2231100E+01,.2168300E+01,.2021400E+01,&
     & .1683000E+01,.1665800E+01,.1971700E+01,.2145200E+01,.2225100E+01,&
     & .2253000E+01,.2237100E+01,.2170800E+01,.2023800E+01,.1690400E+01,&
     & .1356400E+01,.1686900E+01,.1855000E+01,.1953700E+01,.2004900E+01,&
     & .2008500E+01,.1960100E+01,.1828700E+01,.1475600E+01,.1372000E+01,&
     & .1700000E+01,.1865500E+01,.1960700E+01,.2010000E+01,.2010500E+01,&
     & .1962200E+01,.1833900E+01,.1485400E+01,.1395200E+01,.1720400E+01,&
     & .1883700E+01,.1974000E+01,.2018200E+01,.2015400E+01,.1965400E+01,&
     & .1838400E+01,.1494800E+01,.1422900E+01,.1746100E+01,.1907200E+01,&
     & .1992600E+01,.2030300E+01,.2023400E+01,.1969100E+01,.1842000E+01,&
     & .1504000E+01,.1453400E+01,.1775100E+01,.1934500E+01,.2015400E+01,&
     & .2046700E+01,.2033700E+01,.1974200E+01,.1846500E+01,.1513100E+01/

      data   absa(181:315, 7)   /                                       &
     & .1167500E+01,.1490600E+01,.1642200E+01,.1736400E+01,.1787500E+01,&
     & .1797000E+01,.1758200E+01,.1645900E+01,.1305700E+01,.1183900E+01,&
     & .1506200E+01,.1657300E+01,.1748300E+01,.1795600E+01,.1802000E+01,&
     & .1762300E+01,.1652900E+01,.1317600E+01,.1206100E+01,.1528100E+01,&
     & .1679000E+01,.1766000E+01,.1808200E+01,.1810600E+01,.1767800E+01,&
     & .1658700E+01,.1328900E+01,.1231100E+01,.1554900E+01,.1704800E+01,&
     & .1787700E+01,.1825000E+01,.1822200E+01,.1774200E+01,.1664400E+01,&
     & .1339800E+01,.1258200E+01,.1583600E+01,.1733800E+01,.1813100E+01,&
     & .1845700E+01,.1835900E+01,.1782600E+01,.1671000E+01,.1350700E+01,&
     & .9998100E+00,.1308000E+01,.1447700E+01,.1535600E+01,.1582900E+01,&
     & .1593600E+01,.1563600E+01,.1468300E+01,.1148200E+01,.1015100E+01,&
     & .1325100E+01,.1465500E+01,.1550400E+01,.1594100E+01,.1601500E+01,&
     & .1569800E+01,.1476700E+01,.1161900E+01,.1034400E+01,.1347600E+01,&
     & .1488700E+01,.1570200E+01,.1609700E+01,.1613000E+01,.1577200E+01,&
     & .1484700E+01,.1174300E+01,.1056000E+01,.1373500E+01,.1515500E+01,&
     & .1593900E+01,.1629300E+01,.1627600E+01,.1587000E+01,.1492400E+01,&
     & .1186400E+01,.1079400E+01,.1399600E+01,.1543800E+01,.1620500E+01,&
     & .1652100E+01,.1645100E+01,.1599100E+01,.1500300E+01,.1198200E+01,&
     & .8523900E+00,.1141900E+01,.1271400E+01,.1350700E+01,.1393000E+01,&
     & .1403900E+01,.1379000E+01,.1297100E+01,.1003400E+01,.8654200E+00,&
     & .1159100E+01,.1289800E+01,.1367000E+01,.1406200E+01,.1413900E+01,&
     & .1387100E+01,.1307200E+01,.1017900E+01,.8810800E+00,.1180900E+01,&
     & .1313000E+01,.1388000E+01,.1424100E+01,.1427800E+01,.1397500E+01,&
     & .1317200E+01,.1031000E+01,.8990000E+00,.1204100E+01,.1339000E+01,&
     & .1412300E+01,.1445500E+01,.1445300E+01,.1410500E+01,.1326600E+01,&
     & .1043400E+01,.9186200E+00,.1227900E+01,.1364900E+01,.1438600E+01,&
     & .1469200E+01,.1466400E+01,.1425400E+01,.1336700E+01,.1055800E+01/

      data   absa(316:450, 7)   /                                       &
     & .7249700E+00,.9920100E+00,.1110700E+01,.1181300E+01,.1219500E+01,&
     & .1228500E+01,.1205800E+01,.1135900E+01,.8711000E+00,.7351700E+00,&
     & .1008600E+01,.1128500E+01,.1198800E+01,.1234200E+01,.1240500E+01,&
     & .1216000E+01,.1147900E+01,.8856800E+00,.7475200E+00,.1028500E+01,&
     & .1150700E+01,.1219800E+01,.1253200E+01,.1257000E+01,.1228900E+01,&
     & .1158900E+01,.8992700E+00,.7621300E+00,.1048900E+01,.1174700E+01,&
     & .1243600E+01,.1275700E+01,.1276900E+01,.1244500E+01,.1170000E+01,&
     & .9120300E+00,.7785400E+00,.1069700E+01,.1198000E+01,.1267900E+01,&
     & .1300200E+01,.1299400E+01,.1262100E+01,.1181600E+01,.9239200E+00,&
     & .6168200E+00,.8584100E+00,.9640300E+00,.1027200E+01,.1060600E+01,&
     & .1067900E+01,.1047200E+01,.9871500E+00,.7482500E+00,.6241500E+00,&
     & .8740100E+00,.9814000E+00,.1044600E+01,.1076300E+01,.1081100E+01,&
     & .1059300E+01,.1000300E+01,.7629400E+00,.6338700E+00,.8910400E+00,&
     & .1002200E+01,.1065100E+01,.1096200E+01,.1099700E+01,.1074000E+01,&
     & .1012000E+01,.7765900E+00,.6454200E+00,.9087200E+00,.1023000E+01,&
     & .1087800E+01,.1118800E+01,.1120700E+01,.1091100E+01,.1024300E+01,&
     & .7890500E+00,.6590000E+00,.9267400E+00,.1043600E+01,.1110300E+01,&
     & .1142400E+01,.1143500E+01,.1110200E+01,.1037500E+01,.8003400E+00,&
     & .5254700E+00,.7422000E+00,.8343300E+00,.8889000E+00,.9174100E+00,&
     & .9233600E+00,.9053800E+00,.8531100E+00,.6469600E+00,.5306400E+00,&
     & .7560400E+00,.8510700E+00,.9058400E+00,.9336100E+00,.9383600E+00,&
     & .9186100E+00,.8662600E+00,.6606200E+00,.5380900E+00,.7703700E+00,&
     & .8694200E+00,.9256200E+00,.9540900E+00,.9577400E+00,.9342300E+00,&
     & .8787100E+00,.6739100E+00,.5473400E+00,.7857600E+00,.8873400E+00,&
     & .9463400E+00,.9761100E+00,.9789000E+00,.9525300E+00,.8919100E+00,&
     & .6852600E+00,.5584200E+00,.8012900E+00,.9054200E+00,.9667500E+00,&
     & .9978100E+00,.1001300E+01,.9725300E+00,.9059700E+00,.6958300E+00/

      data   absa(451:585, 7)   /                                       &
     & .4490200E+00,.6464100E+00,.7268200E+00,.7729500E+00,.7967900E+00,&
     & .8009500E+00,.7847800E+00,.7388000E+00,.5750000E+00,.4534300E+00,&
     & .6581900E+00,.7425900E+00,.7903000E+00,.8150900E+00,.8186600E+00,&
     & .7996000E+00,.7513700E+00,.5878400E+00,.4596800E+00,.6709400E+00,&
     & .7581200E+00,.8089800E+00,.8352200E+00,.8385400E+00,.8171100E+00,&
     & .7646100E+00,.5993900E+00,.4676400E+00,.6841800E+00,.7740200E+00,&
     & .8271100E+00,.8551800E+00,.8596800E+00,.8363400E+00,.7785900E+00,&
     & .6095400E+00,.4775000E+00,.6979800E+00,.7902100E+00,.8453400E+00,&
     & .8748000E+00,.8801100E+00,.8559500E+00,.7926900E+00,.6189600E+00,&
     & .3825200E+00,.5612300E+00,.6322100E+00,.6720700E+00,.6919100E+00,&
     & .6945100E+00,.6785300E+00,.6367100E+00,.4962400E+00,.3861000E+00,&
     & .5715200E+00,.6455900E+00,.6885700E+00,.7099200E+00,.7125000E+00,&
     & .6944000E+00,.6494800E+00,.5077200E+00,.3913500E+00,.5825100E+00,&
     & .6595300E+00,.7045900E+00,.7279300E+00,.7318800E+00,.7125100E+00,&
     & .6631400E+00,.5177100E+00,.3983800E+00,.5940600E+00,.6735700E+00,&
     & .7206600E+00,.7457000E+00,.7507500E+00,.7310800E+00,.6769600E+00,&
     & .5268000E+00,.4078500E+00,.6062800E+00,.6880200E+00,.7368500E+00,&
     & .7634200E+00,.7684200E+00,.7485700E+00,.6913500E+00,.5340800E+00,&
     & .3229700E+00,.4855600E+00,.5476000E+00,.5835000E+00,.6007900E+00,&
     & .6011600E+00,.5851800E+00,.5469900E+00,.4227600E+00,.3259100E+00,&
     & .4944500E+00,.5596300E+00,.5976400E+00,.6165400E+00,.6184400E+00,&
     & .6014900E+00,.5597300E+00,.4328000E+00,.3305700E+00,.5040500E+00,&
     & .5718400E+00,.6118600E+00,.6323900E+00,.6355700E+00,.6186400E+00,&
     & .5730500E+00,.4415100E+00,.3370500E+00,.5142700E+00,.5842800E+00,&
     & .6261400E+00,.6482500E+00,.6517500E+00,.6349200E+00,.5868900E+00,&
     & .4489300E+00,.3463600E+00,.5254400E+00,.5974000E+00,.6405500E+00,&
     & .6634200E+00,.6673600E+00,.6500200E+00,.6005500E+00,.4548700E+00/

      data   absa(  1:180, 8)   /                                       &
     & .8454475E+01,.7946981E+01,.8106902E+01,.8293415E+01,.8302744E+01,&
     & .8132473E+01,.7748332E+01,.7379908E+01,.7837216E+01,.8369020E+01,&
     & .7875895E+01,.8045177E+01,.8244563E+01,.8270709E+01,.8116300E+01,&
     & .7749859E+01,.7393894E+01,.7847366E+01,.8285211E+01,.7804773E+01,&
     & .7981608E+01,.8194058E+01,.8236878E+01,.8095655E+01,.7745523E+01,&
     & .7401744E+01,.7853486E+01,.8206631E+01,.7736200E+01,.7920325E+01,&
     & .8142313E+01,.8201279E+01,.8071188E+01,.7732988E+01,.7403543E+01,&
     & .7855384E+01,.8131353E+01,.7669212E+01,.7859419E+01,.8089006E+01,&
     & .8160968E+01,.8041936E+01,.7714112E+01,.7402770E+01,.7854506E+01,&
     & .8035329E+01,.7711984E+01,.8054326E+01,.8312552E+01,.8382878E+01,&
     & .8255716E+01,.7865615E+01,.7362355E+01,.7659280E+01,.7952578E+01,&
     & .7639845E+01,.7991063E+01,.8263177E+01,.8347848E+01,.8239605E+01,&
     & .7870529E+01,.7384107E+01,.7682523E+01,.7876313E+01,.7571250E+01,&
     & .7929061E+01,.8212430E+01,.8313538E+01,.8219780E+01,.7866208E+01,&
     & .7395265E+01,.7697467E+01,.7805455E+01,.7505993E+01,.7868255E+01,&
     & .8161157E+01,.8275309E+01,.8195732E+01,.7853413E+01,.7398979E+01,&
     & .7704855E+01,.7742281E+01,.7447306E+01,.7808392E+01,.8111167E+01,&
     & .8233033E+01,.8164826E+01,.7836884E+01,.7399520E+01,.7708591E+01,&
     & .7532706E+01,.7422731E+01,.7921038E+01,.8241749E+01,.8361773E+01,&
     & .8264953E+01,.7883764E+01,.7273843E+01,.7366162E+01,.7456853E+01,&
     & .7352022E+01,.7858416E+01,.8192991E+01,.8327704E+01,.8251231E+01,&
     & .7894283E+01,.7297668E+01,.7395152E+01,.7388143E+01,.7286773E+01,&
     & .7797671E+01,.8142220E+01,.8291520E+01,.8232710E+01,.7892900E+01,&
     & .7312768E+01,.7415601E+01,.7329897E+01,.7229298E+01,.7738967E+01,&
     & .8093725E+01,.8253763E+01,.8209025E+01,.7884574E+01,.7322979E+01,&
     & .7429711E+01,.7281243E+01,.7179080E+01,.7684292E+01,.8043638E+01,&
     & .8212576E+01,.8180221E+01,.7872560E+01,.7328406E+01,.7440260E+01,&
     & .6960506E+01,.7084860E+01,.7692419E+01,.8072395E+01,.8231420E+01,&
     & .8158941E+01,.7802620E+01,.7115961E+01,.6985525E+01,.6890460E+01,&
     & .7017186E+01,.7632657E+01,.8022920E+01,.8197430E+01,.8148687E+01,&
     & .7816871E+01,.7144322E+01,.7020806E+01,.6833009E+01,.6958106E+01,&
     & .7575897E+01,.7974036E+01,.8162581E+01,.8133025E+01,.7819855E+01,&
     & .7167278E+01,.7050824E+01,.6788341E+01,.6907823E+01,.7522755E+01,&
     & .7926643E+01,.8125775E+01,.8111753E+01,.7815827E+01,.7185676E+01,&
     & .7075641E+01,.6757259E+01,.6867282E+01,.7475452E+01,.7880340E+01,&
     & .8087458E+01,.8084863E+01,.7806464E+01,.7198807E+01,.7097443E+01/

      data   absa(181:315, 8)   /                                       &
     & .6340185E+01,.6707242E+01,.7374216E+01,.7801587E+01,.7986307E+01,&
     & .7935884E+01,.7614843E+01,.6902656E+01,.6547120E+01,.6278416E+01,&
     & .6644090E+01,.7317234E+01,.7752832E+01,.7957040E+01,.7930429E+01,&
     & .7630569E+01,.6936644E+01,.6589898E+01,.6232993E+01,.6592439E+01,&
     & .7264591E+01,.7707509E+01,.7925533E+01,.7918826E+01,.7635078E+01,&
     & .6967951E+01,.6630153E+01,.6204934E+01,.6551194E+01,.7219795E+01,&
     & .7664391E+01,.7893004E+01,.7899675E+01,.7635240E+01,.6992659E+01,&
     & .6663746E+01,.6191620E+01,.6521893E+01,.7181052E+01,.7625747E+01,&
     & .7860409E+01,.7878371E+01,.7631859E+01,.7008584E+01,.6691435E+01,&
     & .5701027E+01,.6282840E+01,.6983217E+01,.7436614E+01,.7637531E+01,&
     & .7613654E+01,.7322878E+01,.6631103E+01,.6068872E+01,.5649363E+01,&
     & .6225960E+01,.6931874E+01,.7392920E+01,.7615331E+01,.7612305E+01,&
     & .7340268E+01,.6671266E+01,.6121220E+01,.5616215E+01,.6181108E+01,&
     & .6886667E+01,.7353790E+01,.7592183E+01,.7603387E+01,.7349200E+01,&
     & .6709222E+01,.6171471E+01,.5602211E+01,.6150585E+01,.6849555E+01,&
     & .7318623E+01,.7568758E+01,.7589509E+01,.7355024E+01,.6738001E+01,&
     & .6212573E+01,.5606495E+01,.6135466E+01,.6822413E+01,.7289641E+01,&
     & .7542848E+01,.7572934E+01,.7357114E+01,.6757793E+01,.6246128E+01,&
     & .5069814E+01,.5819946E+01,.6539483E+01,.6990273E+01,.7204566E+01,&
     & .7201912E+01,.6945088E+01,.6305330E+01,.5574348E+01,.5026679E+01,&
     & .5770074E+01,.6492443E+01,.6954397E+01,.7190065E+01,.7204039E+01,&
     & .6965377E+01,.6352224E+01,.5636607E+01,.5006597E+01,.5734952E+01,&
     & .6454389E+01,.6924197E+01,.7174636E+01,.7200149E+01,.6978142E+01,&
     & .6394579E+01,.5692813E+01,.5006429E+01,.5716643E+01,.6426495E+01,&
     & .6900048E+01,.7157117E+01,.7191321E+01,.6989772E+01,.6426021E+01,&
     & .5738457E+01,.5025396E+01,.5715867E+01,.6411783E+01,.6882764E+01,&
     & .7139311E+01,.7183117E+01,.6997950E+01,.6450090E+01,.5778441E+01/

      data   absa(316:450, 8)   /                                       &
     & .4462234E+01,.5335655E+01,.6051411E+01,.6480333E+01,.6706714E+01,&
     & .6719742E+01,.6501878E+01,.5937436E+01,.5082003E+01,.4429084E+01,&
     & .5293651E+01,.6011247E+01,.6455537E+01,.6695967E+01,.6727139E+01,&
     & .6525443E+01,.5987665E+01,.5149466E+01,.4418913E+01,.5268426E+01,&
     & .5981934E+01,.6436217E+01,.6685886E+01,.6728405E+01,.6544585E+01,&
     & .6031729E+01,.5208404E+01,.4431057E+01,.5262879E+01,.5965884E+01,&
     & .6423698E+01,.6675347E+01,.6726852E+01,.6562473E+01,.6066482E+01,&
     & .5259098E+01,.4463604E+01,.5273803E+01,.5964332E+01,.6418269E+01,&
     & .6667516E+01,.6727383E+01,.6575984E+01,.6094410E+01,.5304372E+01,&
     & .3893017E+01,.4852214E+01,.5530986E+01,.5941479E+01,.6161053E+01,&
     & .6192794E+01,.6013485E+01,.5527651E+01,.4603839E+01,.3868292E+01,&
     & .4816432E+01,.5499970E+01,.5924581E+01,.6155187E+01,.6204798E+01,&
     & .6043116E+01,.5578514E+01,.4673568E+01,.3867315E+01,.4800668E+01,&
     & .5481603E+01,.5912694E+01,.6151550E+01,.6211560E+01,.6068531E+01,&
     & .5623715E+01,.4735833E+01,.3889089E+01,.4804843E+01,.5478401E+01,&
     & .5908385E+01,.6149321E+01,.6219065E+01,.6091671E+01,.5661853E+01,&
     & .4790546E+01,.3931954E+01,.4826464E+01,.5491042E+01,.5912928E+01,&
     & .6152912E+01,.6229107E+01,.6110084E+01,.5693512E+01,.4840222E+01,&
     & .3370800E+01,.4371741E+01,.4996837E+01,.5386824E+01,.5594880E+01,&
     & .5641906E+01,.5505250E+01,.5094350E+01,.4102630E+01,.3354024E+01,&
     & .4346187E+01,.4976577E+01,.5374389E+01,.5596429E+01,.5658447E+01,&
     & .5539483E+01,.5146897E+01,.4175662E+01,.3361375E+01,.4340181E+01,&
     & .4971228E+01,.5370928E+01,.5598179E+01,.5672261E+01,.5570116E+01,&
     & .5193502E+01,.4240227E+01,.3392497E+01,.4352744E+01,.4980679E+01,&
     & .5377112E+01,.5606443E+01,.5688563E+01,.5596813E+01,.5234384E+01,&
     & .4299382E+01,.3445749E+01,.4384734E+01,.5004059E+01,.5393218E+01,&
     & .5623134E+01,.5706956E+01,.5620707E+01,.5270254E+01,.4350295E+01/

      data   absa(451:585, 8)   /                                       &
     & .2896690E+01,.3891858E+01,.4467847E+01,.4828220E+01,.5035892E+01,&
     & .5103952E+01,.5010639E+01,.4674849E+01,.3670107E+01,.2897792E+01,&
     & .3883888E+01,.4462186E+01,.4828076E+01,.5043823E+01,.5124270E+01,&
     & .5046967E+01,.4727706E+01,.3740688E+01,.2923572E+01,.3895453E+01,&
     & .4471923E+01,.4837551E+01,.5057813E+01,.5145441E+01,.5079272E+01,&
     & .4774390E+01,.3805013E+01,.2972461E+01,.3927241E+01,.4497456E+01,&
     & .4859388E+01,.5080493E+01,.5168972E+01,.5109386E+01,.4816374E+01,&
     & .3861922E+01,.3043384E+01,.3977838E+01,.4537492E+01,.4893037E+01,&
     & .5111190E+01,.5197160E+01,.5139056E+01,.4854089E+01,.3912075E+01,&
     & .2490281E+01,.3445349E+01,.3966257E+01,.4299151E+01,.4503210E+01,&
     & .4585981E+01,.4527887E+01,.4253375E+01,.3285333E+01,.2507718E+01,&
     & .3454661E+01,.3975474E+01,.4311153E+01,.4520893E+01,.4611552E+01,&
     & .4565739E+01,.4306696E+01,.3353145E+01,.2549112E+01,.3484385E+01,&
     & .4000658E+01,.4335871E+01,.4546759E+01,.4639747E+01,.4601392E+01,&
     & .4354901E+01,.3414904E+01,.2613534E+01,.3533116E+01,.4041205E+01,&
     & .4373302E+01,.4580520E+01,.4672532E+01,.4636635E+01,.4399031E+01,&
     & .3469339E+01,.2696279E+01,.3599603E+01,.4096892E+01,.4422225E+01,&
     & .4622150E+01,.4711302E+01,.4672921E+01,.4436995E+01,.3520495E+01,&
     & .2147723E+01,.3043091E+01,.3506523E+01,.3813045E+01,.4007672E+01,&
     & .4097762E+01,.4063337E+01,.3839253E+01,.2937518E+01,.2178826E+01,&
     & .3066802E+01,.3529383E+01,.3837562E+01,.4035183E+01,.4129716E+01,&
     & .4103957E+01,.3892792E+01,.3003047E+01,.2233352E+01,.3109540E+01,&
     & .3568948E+01,.3874858E+01,.4071147E+01,.4166460E+01,.4144730E+01,&
     & .3941453E+01,.3061753E+01,.2307024E+01,.3171315E+01,.3623581E+01,&
     & .3923493E+01,.4115241E+01,.4208646E+01,.4185941E+01,.3984848E+01,&
     & .3115620E+01,.2388613E+01,.3250521E+01,.3691760E+01,.3982688E+01,&
     & .4168532E+01,.4255448E+01,.4227404E+01,.4025674E+01,.3167770E+01/

      data   absa(  1:180, 9)   /                                       &
     & .2529000E+02,.2237800E+02,.1968200E+02,.1909200E+02,.1840100E+02,&
     & .1733000E+02,.1719400E+02,.1951300E+02,.2154600E+02,.2515300E+02,&
     & .2225100E+02,.1957400E+02,.1900200E+02,.1830900E+02,.1734600E+02,&
     & .1716800E+02,.1946900E+02,.2151600E+02,.2499000E+02,.2210800E+02,&
     & .1945700E+02,.1889100E+02,.1819700E+02,.1734400E+02,.1715300E+02,&
     & .1943500E+02,.2151600E+02,.2478500E+02,.2192600E+02,.1931200E+02,&
     & .1877400E+02,.1811100E+02,.1730000E+02,.1716800E+02,.1942600E+02,&
     & .2152400E+02,.2456200E+02,.2172900E+02,.1914500E+02,.1865700E+02,&
     & .1804300E+02,.1723900E+02,.1719000E+02,.1940000E+02,.2151700E+02,&
     & .2673800E+02,.2363500E+02,.2116400E+02,.2088200E+02,.2013300E+02,&
     & .1895200E+02,.1839100E+02,.2039700E+02,.2275200E+02,.2657900E+02,&
     & .2349300E+02,.2102600E+02,.2077000E+02,.2004600E+02,.1892400E+02,&
     & .1835100E+02,.2030200E+02,.2267300E+02,.2636900E+02,.2330400E+02,&
     & .2084700E+02,.2063400E+02,.1995500E+02,.1886700E+02,.1835500E+02,&
     & .2027400E+02,.2264300E+02,.2614700E+02,.2310800E+02,.2066300E+02,&
     & .2049100E+02,.1987000E+02,.1879000E+02,.1836500E+02,.2027100E+02,&
     & .2263800E+02,.2587600E+02,.2286700E+02,.2045700E+02,.2030800E+02,&
     & .1978800E+02,.1872300E+02,.1835900E+02,.2024200E+02,.2261200E+02,&
     & .2780100E+02,.2455800E+02,.2262500E+02,.2245600E+02,.2176600E+02,&
     & .2072300E+02,.1978500E+02,.2119300E+02,.2381100E+02,.2758600E+02,&
     & .2436600E+02,.2244400E+02,.2234100E+02,.2167000E+02,.2065100E+02,&
     & .1973200E+02,.2116300E+02,.2377800E+02,.2736200E+02,.2416600E+02,&
     & .2223600E+02,.2220000E+02,.2157500E+02,.2055700E+02,.1971800E+02,&
     & .2115500E+02,.2378300E+02,.2708400E+02,.2391900E+02,.2201300E+02,&
     & .2201800E+02,.2147100E+02,.2045700E+02,.1970800E+02,.2113600E+02,&
     & .2376900E+02,.2679500E+02,.2366600E+02,.2179400E+02,.2182700E+02,&
     & .2135500E+02,.2037400E+02,.1965500E+02,.2109200E+02,.2372400E+02,&
     & .2832700E+02,.2502600E+02,.2391000E+02,.2372000E+02,.2326500E+02,&
     & .2239600E+02,.2108600E+02,.2185900E+02,.2464900E+02,.2809500E+02,&
     & .2481600E+02,.2368300E+02,.2359400E+02,.2315600E+02,.2230400E+02,&
     & .2102300E+02,.2187300E+02,.2467400E+02,.2781700E+02,.2456900E+02,&
     & .2345400E+02,.2344600E+02,.2303100E+02,.2218200E+02,.2102200E+02,&
     & .2187800E+02,.2468000E+02,.2752500E+02,.2431000E+02,.2322000E+02,&
     & .2326800E+02,.2290800E+02,.2209000E+02,.2102200E+02,.2182400E+02,&
     & .2462000E+02,.2719100E+02,.2402300E+02,.2299600E+02,.2307200E+02,&
     & .2276900E+02,.2202200E+02,.2098200E+02,.2173700E+02,.2451400E+02/

      data   absa(181:315, 9)   /                                       &
     & .2825900E+02,.2501400E+02,.2490700E+02,.2470600E+02,.2464200E+02,&
     & .2391800E+02,.2233000E+02,.2221900E+02,.2509100E+02,.2799300E+02,&
     & .2477100E+02,.2466600E+02,.2459600E+02,.2448700E+02,.2381800E+02,&
     & .2234800E+02,.2226900E+02,.2514900E+02,.2769100E+02,.2449900E+02,&
     & .2442100E+02,.2443200E+02,.2434000E+02,.2370500E+02,.2242000E+02,&
     & .2224200E+02,.2510900E+02,.2735700E+02,.2421400E+02,.2417800E+02,&
     & .2425300E+02,.2419100E+02,.2364200E+02,.2242200E+02,.2217400E+02,&
     & .2503000E+02,.2700800E+02,.2391600E+02,.2393500E+02,.2406300E+02,&
     & .2402000E+02,.2357200E+02,.2235500E+02,.2213200E+02,.2495700E+02,&
     & .2762700E+02,.2484100E+02,.2541400E+02,.2551300E+02,.2578800E+02,&
     & .2516100E+02,.2356700E+02,.2238400E+02,.2509100E+02,.2731700E+02,&
     & .2455000E+02,.2516500E+02,.2538400E+02,.2557800E+02,.2508300E+02,&
     & .2366200E+02,.2242300E+02,.2510200E+02,.2699400E+02,.2425900E+02,&
     & .2492600E+02,.2520600E+02,.2539100E+02,.2501500E+02,.2375800E+02,&
     & .2236300E+02,.2502500E+02,.2664000E+02,.2394500E+02,.2469200E+02,&
     & .2500200E+02,.2519300E+02,.2497300E+02,.2373300E+02,.2232800E+02,&
     & .2496500E+02,.2626500E+02,.2362500E+02,.2444600E+02,.2478500E+02,&
     & .2503700E+02,.2490600E+02,.2365900E+02,.2235200E+02,.2496500E+02,&
     & .2643000E+02,.2458100E+02,.2543900E+02,.2614000E+02,.2652400E+02,&
     & .2597400E+02,.2455200E+02,.2244100E+02,.2449800E+02,.2610500E+02,&
     & .2426300E+02,.2522500E+02,.2597400E+02,.2631500E+02,.2594600E+02,&
     & .2467700E+02,.2247100E+02,.2449000E+02,.2575200E+02,.2393800E+02,&
     & .2499600E+02,.2574900E+02,.2611300E+02,.2592500E+02,.2475800E+02,&
     & .2244200E+02,.2445400E+02,.2538900E+02,.2360900E+02,.2476100E+02,&
     & .2549900E+02,.2594000E+02,.2591300E+02,.2472000E+02,.2247300E+02,&
     & .2447500E+02,.2501000E+02,.2326000E+02,.2451000E+02,.2526200E+02,&
     & .2580400E+02,.2581100E+02,.2464000E+02,.2254700E+02,.2453400E+02/

      data   absa(316:450, 9)   /                                       &
     & .2476200E+02,.2399700E+02,.2520900E+02,.2646700E+02,.2674800E+02,&
     & .2636500E+02,.2509400E+02,.2235700E+02,.2340600E+02,.2442100E+02,&
     & .2367200E+02,.2498100E+02,.2621700E+02,.2659500E+02,.2637500E+02,&
     & .2519500E+02,.2243500E+02,.2345300E+02,.2406700E+02,.2333600E+02,&
     & .2475400E+02,.2595600E+02,.2645500E+02,.2636100E+02,.2524100E+02,&
     & .2245800E+02,.2347500E+02,.2369100E+02,.2299600E+02,.2450200E+02,&
     & .2569800E+02,.2633000E+02,.2634300E+02,.2519700E+02,.2253300E+02,&
     & .2357500E+02,.2331100E+02,.2266000E+02,.2423400E+02,.2545000E+02,&
     & .2621500E+02,.2620900E+02,.2512300E+02,.2264200E+02,.2370900E+02,&
     & .2280000E+02,.2301100E+02,.2479900E+02,.2618100E+02,.2653300E+02,&
     & .2630100E+02,.2504400E+02,.2207700E+02,.2190700E+02,.2243600E+02,&
     & .2269700E+02,.2454600E+02,.2594500E+02,.2646000E+02,.2631400E+02,&
     & .2512200E+02,.2220400E+02,.2202900E+02,.2207900E+02,.2236700E+02,&
     & .2429500E+02,.2570600E+02,.2637000E+02,.2628800E+02,.2516200E+02,&
     & .2231400E+02,.2215100E+02,.2171900E+02,.2202900E+02,.2401700E+02,&
     & .2548700E+02,.2627300E+02,.2621700E+02,.2511000E+02,.2243500E+02,&
     & .2233400E+02,.2134100E+02,.2170700E+02,.2370200E+02,.2528200E+02,&
     & .2615900E+02,.2608600E+02,.2506500E+02,.2257800E+02,.2250600E+02,&
     & .2060100E+02,.2167800E+02,.2405800E+02,.2535200E+02,.2593100E+02,&
     & .2571800E+02,.2443200E+02,.2160700E+02,.2026200E+02,.2025400E+02,&
     & .2134500E+02,.2379600E+02,.2518300E+02,.2587500E+02,.2569100E+02,&
     & .2450800E+02,.2177400E+02,.2044500E+02,.1989900E+02,.2104800E+02,&
     & .2349400E+02,.2500000E+02,.2580000E+02,.2566200E+02,.2453300E+02,&
     & .2192200E+02,.2063800E+02,.1954400E+02,.2076600E+02,.2320700E+02,&
     & .2482600E+02,.2568600E+02,.2559300E+02,.2452700E+02,.2208500E+02,&
     & .2083200E+02,.1919300E+02,.2047000E+02,.2293600E+02,.2467300E+02,&
     & .2554400E+02,.2548000E+02,.2453700E+02,.2221800E+02,.2104400E+02/

      data   absa(451:585, 9)   /                                       &
     & .1817000E+02,.2015500E+02,.2272100E+02,.2414500E+02,.2479800E+02,&
     & .2455900E+02,.2340300E+02,.2089500E+02,.1874000E+02,.1783000E+02,&
     & .1986000E+02,.2247900E+02,.2401800E+02,.2472100E+02,.2455600E+02,&
     & .2346600E+02,.2106900E+02,.1894900E+02,.1749600E+02,.1957600E+02,&
     & .2224200E+02,.2388600E+02,.2462100E+02,.2452900E+02,.2352500E+02,&
     & .2123100E+02,.1914700E+02,.1716800E+02,.1929200E+02,.2200500E+02,&
     & .2373400E+02,.2451400E+02,.2447600E+02,.2358000E+02,.2137600E+02,&
     & .1936800E+02,.1684400E+02,.1901300E+02,.2177100E+02,.2357000E+02,&
     & .2439000E+02,.2444000E+02,.2365100E+02,.2151700E+02,.1961200E+02,&
     & .1576200E+02,.1861500E+02,.2113300E+02,.2263900E+02,.2321400E+02,&
     & .2302200E+02,.2206000E+02,.1995800E+02,.1731400E+02,.1545500E+02,&
     & .1834100E+02,.2094400E+02,.2251000E+02,.2314800E+02,.2304200E+02,&
     & .2216400E+02,.2012700E+02,.1754300E+02,.1515600E+02,.1806800E+02,&
     & .2076800E+02,.2237400E+02,.2308400E+02,.2306000E+02,.2225700E+02,&
     & .2027500E+02,.1777000E+02,.1486500E+02,.1779400E+02,.2058200E+02,&
     & .2223700E+02,.2301800E+02,.2307900E+02,.2235000E+02,.2042700E+02,&
     & .1802600E+02,.1461200E+02,.1753400E+02,.2039900E+02,.2208600E+02,&
     & .2295600E+02,.2311200E+02,.2244300E+02,.2060700E+02,.1828000E+02,&
     & .1351700E+02,.1691400E+02,.1942500E+02,.2079000E+02,.2136700E+02,&
     & .2131500E+02,.2060700E+02,.1887100E+02,.1596700E+02,.1324900E+02,&
     & .1668800E+02,.1927900E+02,.2069300E+02,.2134400E+02,.2137200E+02,&
     & .2072500E+02,.1904300E+02,.1621700E+02,.1299400E+02,.1646700E+02,&
     & .1910800E+02,.2060200E+02,.2134300E+02,.2143600E+02,.2083900E+02,&
     & .1921200E+02,.1649400E+02,.1278400E+02,.1625000E+02,.1895100E+02,&
     & .2050600E+02,.2133500E+02,.2150300E+02,.2096100E+02,.1941500E+02,&
     & .1677000E+02,.1273400E+02,.1604900E+02,.1881900E+02,.2043800E+02,&
     & .2132900E+02,.2157300E+02,.2110400E+02,.1962300E+02,.1698300E+02/

      data   absa(  1:180,10)   /                                       &
     & .3587728E+02,.3163298E+02,.2744141E+02,.2457089E+02,.2397093E+02,&
     & .2424633E+02,.2723997E+02,.3121916E+02,.3444458E+02,.3583320E+02,&
     & .3160263E+02,.2742318E+02,.2450739E+02,.2389581E+02,.2409779E+02,&
     & .2700065E+02,.3092695E+02,.3408907E+02,.3574205E+02,.3152702E+02,&
     & .2736303E+02,.2442532E+02,.2382519E+02,.2395943E+02,.2677589E+02,&
     & .3065092E+02,.3375501E+02,.3559971E+02,.3140018E+02,.2725046E+02,&
     & .2431487E+02,.2372351E+02,.2384507E+02,.2657448E+02,.3041293E+02,&
     & .3346529E+02,.3540621E+02,.3122776E+02,.2710481E+02,.2417902E+02,&
     & .2360279E+02,.2374616E+02,.2639615E+02,.3021059E+02,.3322457E+02,&
     & .4028856E+02,.3550073E+02,.3077033E+02,.2797279E+02,.2711240E+02,&
     & .2689686E+02,.2944080E+02,.3377959E+02,.3768881E+02,.4023214E+02,&
     & .3545689E+02,.3073830E+02,.2787552E+02,.2703385E+02,.2679085E+02,&
     & .2921417E+02,.3349222E+02,.3733705E+02,.4012534E+02,.3536466E+02,&
     & .3066284E+02,.2776145E+02,.2693419E+02,.2667684E+02,.2898061E+02,&
     & .3321866E+02,.3700637E+02,.3994773E+02,.3520594E+02,.3052931E+02,&
     & .2761549E+02,.2681235E+02,.2656436E+02,.2878066E+02,.3297874E+02,&
     & .3672473E+02,.3972581E+02,.3501021E+02,.3036408E+02,.2745552E+02,&
     & .2667155E+02,.2645065E+02,.2861110E+02,.3278003E+02,.3649274E+02,&
     & .4476125E+02,.3941959E+02,.3422706E+02,.3167904E+02,.3052481E+02,&
     & .2973002E+02,.3165313E+02,.3629435E+02,.4090093E+02,.4471130E+02,&
     & .3937886E+02,.3418630E+02,.3155677E+02,.3043113E+02,.2965728E+02,&
     & .3143493E+02,.3600307E+02,.4054030E+02,.4457943E+02,.3926295E+02,&
     & .3408654E+02,.3140866E+02,.3032055E+02,.2958250E+02,.3123380E+02,&
     & .3573596E+02,.4022035E+02,.4439236E+02,.3909796E+02,.3394316E+02,&
     & .3123639E+02,.3018674E+02,.2947741E+02,.3105394E+02,.3550064E+02,&
     & .3995406E+02,.4412856E+02,.3886401E+02,.3374559E+02,.3104184E+02,&
     & .3002428E+02,.2936062E+02,.3091031E+02,.3530951E+02,.3974076E+02,&
     & .4910432E+02,.4321511E+02,.3779078E+02,.3556693E+02,.3409226E+02,&
     & .3283110E+02,.3395610E+02,.3866004E+02,.4376828E+02,.4902356E+02,&
     & .4314962E+02,.3771527E+02,.3544117E+02,.3399674E+02,.3277233E+02,&
     & .3377629E+02,.3837960E+02,.4342838E+02,.4888388E+02,.4302756E+02,&
     & .3758282E+02,.3527975E+02,.3389250E+02,.3269755E+02,.3359135E+02,&
     & .3810582E+02,.4311667E+02,.4865644E+02,.4282619E+02,.3740141E+02,&
     & .3507775E+02,.3374861E+02,.3258381E+02,.3342561E+02,.3789331E+02,&
     & .4287514E+02,.4836706E+02,.4257131E+02,.3716473E+02,.3485086E+02,&
     & .3357022E+02,.3245299E+02,.3329940E+02,.3773038E+02,.4269789E+02/

      data   absa(181:315,10)   /                                       &
     & .5307594E+02,.4667483E+02,.4130685E+02,.3949369E+02,.3769718E+02,&
     & .3615626E+02,.3643522E+02,.4090265E+02,.4638012E+02,.5296984E+02,&
     & .4659097E+02,.4119161E+02,.3933623E+02,.3762920E+02,.3609076E+02,&
     & .3626818E+02,.4063453E+02,.4606837E+02,.5278655E+02,.4643245E+02,&
     & .4102506E+02,.3914923E+02,.3752838E+02,.3600030E+02,.3607944E+02,&
     & .4039533E+02,.4580658E+02,.5252049E+02,.4620146E+02,.4079758E+02,&
     & .3892166E+02,.3738707E+02,.3586925E+02,.3592354E+02,.4020885E+02,&
     & .4560089E+02,.5218739E+02,.4591040E+02,.4053117E+02,.3866135E+02,&
     & .3721768E+02,.3572048E+02,.3579033E+02,.4004272E+02,.4542333E+02,&
     & .5645060E+02,.4959535E+02,.4476419E+02,.4329540E+02,.4136555E+02,&
     & .3958949E+02,.3901626E+02,.4295054E+02,.4873997E+02,.5630605E+02,&
     & .4947453E+02,.4458064E+02,.4311329E+02,.4131649E+02,.3950972E+02,&
     & .3886403E+02,.4274257E+02,.4850859E+02,.5608434E+02,.4928505E+02,&
     & .4434892E+02,.4288710E+02,.4120465E+02,.3940634E+02,.3867749E+02,&
     & .4253840E+02,.4828479E+02,.5577797E+02,.4902368E+02,.4406590E+02,&
     & .4263870E+02,.4106511E+02,.3925611E+02,.3851716E+02,.4234391E+02,&
     & .4807363E+02,.5538906E+02,.4869020E+02,.4373996E+02,.4235043E+02,&
     & .4087593E+02,.3908727E+02,.3836821E+02,.4215132E+02,.4786703E+02,&
     & .5901771E+02,.5183601E+02,.4798533E+02,.4682270E+02,.4499590E+02,&
     & .4308761E+02,.4172136E+02,.4474695E+02,.5079194E+02,.5882554E+02,&
     & .5166631E+02,.4772612E+02,.4661016E+02,.4492409E+02,.4299263E+02,&
     & .4159694E+02,.4458623E+02,.5061389E+02,.5854583E+02,.5141979E+02,&
     & .4742883E+02,.4637327E+02,.4480255E+02,.4286883E+02,.4142281E+02,&
     & .4438577E+02,.5039888E+02,.5817816E+02,.5110231E+02,.4708389E+02,&
     & .4609738E+02,.4463949E+02,.4268850E+02,.4126044E+02,.4418534E+02,&
     & .5018101E+02,.5773974E+02,.5072435E+02,.4670644E+02,.4577129E+02,&
     & .4443478E+02,.4250650E+02,.4111702E+02,.4397705E+02,.4997641E+02/

      data   absa(316:450,10)   /                                       &
     & .6060573E+02,.5344936E+02,.5077793E+02,.4997885E+02,.4845492E+02,&
     & .4650628E+02,.4444661E+02,.4611186E+02,.5223587E+02,.6035565E+02,&
     & .5321162E+02,.5047679E+02,.4976412E+02,.4835297E+02,.4640277E+02,&
     & .4437648E+02,.4597962E+02,.5210068E+02,.6001386E+02,.5290034E+02,&
     & .5010907E+02,.4949538E+02,.4818206E+02,.4628144E+02,.4423845E+02,&
     & .4582350E+02,.5193692E+02,.5960595E+02,.5253493E+02,.4971769E+02,&
     & .4917510E+02,.4797986E+02,.4609629E+02,.4410837E+02,.4565103E+02,&
     & .5175207E+02,.5910403E+02,.5209755E+02,.4928800E+02,.4883659E+02,&
     & .4774249E+02,.4591174E+02,.4398757E+02,.4546040E+02,.5157175E+02,&
     & .6109714E+02,.5440970E+02,.5300408E+02,.5268782E+02,.5157437E+02,&
     & .4964597E+02,.4705595E+02,.4702696E+02,.5286716E+02,.6078367E+02,&
     & .5409377E+02,.5266455E+02,.5243664E+02,.5140681E+02,.4958118E+02,&
     & .4706770E+02,.4700927E+02,.5284723E+02,.6037380E+02,.5371502E+02,&
     & .5225748E+02,.5215540E+02,.5119129E+02,.4948761E+02,.4697878E+02,&
     & .4691301E+02,.5274423E+02,.5989651E+02,.5328949E+02,.5183758E+02,&
     & .5181450E+02,.5096397E+02,.4932968E+02,.4689935E+02,.4677701E+02,&
     & .5261482E+02,.5936966E+02,.5280219E+02,.5138276E+02,.5145230E+02,&
     & .5070975E+02,.4914571E+02,.4680586E+02,.4662819E+02,.5250407E+02,&
     & .6042976E+02,.5467511E+02,.5454288E+02,.5481038E+02,.5409097E+02,&
     & .5235706E+02,.4927445E+02,.4742032E+02,.5252935E+02,.6001878E+02,&
     & .5427532E+02,.5413495E+02,.5452142E+02,.5389531E+02,.5233375E+02,&
     & .4936959E+02,.4752824E+02,.5263857E+02,.5955852E+02,.5382395E+02,&
     & .5371774E+02,.5419930E+02,.5366829E+02,.5225894E+02,.4936460E+02,&
     & .4753260E+02,.5265555E+02,.5902075E+02,.5331468E+02,.5326153E+02,&
     & .5383822E+02,.5344237E+02,.5210269E+02,.4932635E+02,.4747318E+02,&
     & .5266176E+02,.5843636E+02,.5278414E+02,.5278267E+02,.5345038E+02,&
     & .5319472E+02,.5193593E+02,.4925399E+02,.4743021E+02,.5267090E+02/

      data   absa(451:585,10)   /                                       &
     & .5841299E+02,.5394652E+02,.5521374E+02,.5605247E+02,.5585908E+02,&
     & .5438236E+02,.5102018E+02,.4757287E+02,.5144594E+02,.5792304E+02,&
     & .5345752E+02,.5477573E+02,.5572296E+02,.5566296E+02,.5437881E+02,&
     & .5115010E+02,.4772352E+02,.5165224E+02,.5737807E+02,.5293116E+02,&
     & .5431199E+02,.5537393E+02,.5546638E+02,.5430287E+02,.5118313E+02,&
     & .4781700E+02,.5183934E+02,.5678698E+02,.5237535E+02,.5381877E+02,&
     & .5500766E+02,.5523934E+02,.5416503E+02,.5117650E+02,.4791639E+02,&
     & .5201561E+02,.5615143E+02,.5178837E+02,.5328954E+02,.5460865E+02,&
     & .5497459E+02,.5398659E+02,.5113103E+02,.4798503E+02,.5214722E+02,&
     & .5536220E+02,.5249053E+02,.5510606E+02,.5651628E+02,.5687411E+02,&
     & .5558754E+02,.5214421E+02,.4740469E+02,.4972236E+02,.5479927E+02,&
     & .5192667E+02,.5465501E+02,.5620526E+02,.5671006E+02,.5563285E+02,&
     & .5229549E+02,.4766073E+02,.5010788E+02,.5421071E+02,.5135180E+02,&
     & .5415133E+02,.5586559E+02,.5652158E+02,.5556841E+02,.5239599E+02,&
     & .4791820E+02,.5046754E+02,.5358519E+02,.5076397E+02,.5362234E+02,&
     & .5548685E+02,.5629675E+02,.5544739E+02,.5247314E+02,.4811189E+02,&
     & .5074908E+02,.5294036E+02,.5016394E+02,.5307965E+02,.5511021E+02,&
     & .5600283E+02,.5528953E+02,.5251126E+02,.4826838E+02,.5100487E+02,&
     & .5148338E+02,.5053874E+02,.5414066E+02,.5624827E+02,.5702373E+02,&
     & .5589687E+02,.5255754E+02,.4705268E+02,.4767006E+02,.5089925E+02,&
     & .4993871E+02,.5366605E+02,.5594429E+02,.5690194E+02,.5597842E+02,&
     & .5278229E+02,.4742535E+02,.4816570E+02,.5028772E+02,.4933321E+02,&
     & .5319737E+02,.5560811E+02,.5674811E+02,.5596622E+02,.5298860E+02,&
     & .4773920E+02,.4858121E+02,.4966712E+02,.4872563E+02,.5271026E+02,&
     & .5528046E+02,.5651701E+02,.5590277E+02,.5315749E+02,.4798366E+02,&
     & .4895540E+02,.4904239E+02,.4811862E+02,.5220900E+02,.5494835E+02,&
     & .5626643E+02,.5584337E+02,.5327664E+02,.4820949E+02,.4934904E+02/

      data   absa(  1:180,11)   /                                       &
     & .5140400E+02,.4521200E+02,.3907800E+02,.3297800E+02,.3121300E+02,&
     & .3568900E+02,.4223900E+02,.4910900E+02,.5355500E+02,.5145600E+02,&
     & .4526300E+02,.3908500E+02,.3297500E+02,.3103400E+02,.3523700E+02,&
     & .4175600E+02,.4853100E+02,.5287700E+02,.5141900E+02,.4522600E+02,&
     & .3901300E+02,.3293200E+02,.3082600E+02,.3487100E+02,.4137200E+02,&
     & .4807000E+02,.5231900E+02,.5131500E+02,.4512200E+02,.3892500E+02,&
     & .3288600E+02,.3064300E+02,.3456500E+02,.4105400E+02,.4769000E+02,&
     & .5185300E+02,.5113400E+02,.4495800E+02,.3878700E+02,.3281900E+02,&
     & .3047600E+02,.3433000E+02,.4079900E+02,.4738400E+02,.5151300E+02,&
     & .5957500E+02,.5234300E+02,.4516800E+02,.3815600E+02,.3654900E+02,&
     & .4079800E+02,.4843300E+02,.5632000E+02,.6249900E+02,.5965600E+02,&
     & .5241500E+02,.4519600E+02,.3815900E+02,.3635600E+02,.4023200E+02,&
     & .4780600E+02,.5560300E+02,.6169200E+02,.5963500E+02,.5239500E+02,&
     & .4514600E+02,.3811600E+02,.3613900E+02,.3980500E+02,.4733900E+02,&
     & .5504600E+02,.6106800E+02,.5952200E+02,.5228500E+02,.4505100E+02,&
     & .3806600E+02,.3588900E+02,.3947000E+02,.4698100E+02,.5462100E+02,&
     & .6059200E+02,.5930700E+02,.5209400E+02,.4488900E+02,.3796000E+02,&
     & .3567700E+02,.3921600E+02,.4668700E+02,.5426800E+02,.6019300E+02,&
     & .6880900E+02,.6039700E+02,.5202500E+02,.4416100E+02,.4270100E+02,&
     & .4666700E+02,.5552000E+02,.6449300E+02,.7266900E+02,.6892400E+02,&
     & .6049900E+02,.5209800E+02,.4413300E+02,.4252100E+02,.4596600E+02,&
     & .5473200E+02,.6359400E+02,.7170500E+02,.6895000E+02,.6053100E+02,&
     & .5210100E+02,.4410000E+02,.4228500E+02,.4541800E+02,.5409500E+02,&
     & .6288300E+02,.7091900E+02,.6885700E+02,.6043400E+02,.5201400E+02,&
     & .4404100E+02,.4203100E+02,.4500500E+02,.5356400E+02,.6229100E+02,&
     & .7024700E+02,.6866600E+02,.6026500E+02,.5187100E+02,.4394200E+02,&
     & .4184200E+02,.4469000E+02,.5313400E+02,.6180600E+02,.6968600E+02,&
     & .7931100E+02,.6956100E+02,.5988100E+02,.5127600E+02,.4976100E+02,&
     & .5319300E+02,.6339600E+02,.7365600E+02,.8365200E+02,.7951100E+02,&
     & .6973600E+02,.6001900E+02,.5117300E+02,.4959400E+02,.5246600E+02,&
     & .6246100E+02,.7257800E+02,.8242600E+02,.7956200E+02,.6978900E+02,&
     & .6004300E+02,.5107500E+02,.4934600E+02,.5184700E+02,.6163900E+02,&
     & .7163200E+02,.8136900E+02,.7951200E+02,.6973300E+02,.5998900E+02,&
     & .5097900E+02,.4908800E+02,.5136000E+02,.6094700E+02,.7083600E+02,&
     & .8046800E+02,.7932500E+02,.6956900E+02,.5985400E+02,.5087100E+02,&
     & .4888200E+02,.5099100E+02,.6036200E+02,.7016100E+02,.7970100E+02/

      data   absa(181:315,11)   /                                       &
     & .9111200E+02,.7989200E+02,.6878100E+02,.5949600E+02,.5798800E+02,&
     & .6041500E+02,.7159500E+02,.8322700E+02,.9485400E+02,.9144700E+02,&
     & .8017300E+02,.6902300E+02,.5941600E+02,.5774300E+02,.5967300E+02,&
     & .7048300E+02,.8194200E+02,.9339400E+02,.9158300E+02,.8028800E+02,&
     & .6910600E+02,.5927600E+02,.5741600E+02,.5903500E+02,.6952800E+02,&
     & .8082900E+02,.9213100E+02,.9156600E+02,.8025700E+02,.6907300E+02,&
     & .5913600E+02,.5709200E+02,.5852700E+02,.6872400E+02,.7988500E+02,&
     & .9105300E+02,.9139000E+02,.8010400E+02,.6893700E+02,.5897300E+02,&
     & .5677600E+02,.5810700E+02,.6807900E+02,.7907200E+02,.9012200E+02,&
     & .1041900E+03,.9139400E+02,.7865600E+02,.6891600E+02,.6712800E+02,&
     & .6846100E+02,.7989400E+02,.9292600E+02,.1060700E+03,.1046300E+03,&
     & .9178200E+02,.7900200E+02,.6881900E+02,.6684500E+02,.6773800E+02,&
     & .7862800E+02,.9145800E+02,.1044000E+03,.1048600E+03,.9197000E+02,&
     & .7915600E+02,.6869800E+02,.6647000E+02,.6706600E+02,.7762800E+02,&
     & .9024200E+02,.1030200E+03,.1049400E+03,.9202500E+02,.7920700E+02,&
     & .6855500E+02,.6608100E+02,.6654300E+02,.7684300E+02,.8925300E+02,&
     & .1018900E+03,.1048100E+03,.9189200E+02,.7910500E+02,.6834400E+02,&
     & .6566800E+02,.6610900E+02,.7619000E+02,.8838600E+02,.1009000E+03,&
     & .1185000E+03,.1039400E+03,.8942300E+02,.7950900E+02,.7720700E+02,&
     & .7719600E+02,.8794300E+02,.1023600E+03,.1169400E+03,.1190500E+03,&
     & .1044300E+03,.8986400E+02,.7952800E+02,.7690500E+02,.7649400E+02,&
     & .8665000E+02,.1007800E+03,.1151500E+03,.1193900E+03,.1047300E+03,&
     & .9011500E+02,.7940100E+02,.7652300E+02,.7585300E+02,.8571500E+02,&
     & .9959500E+02,.1138000E+03,.1195100E+03,.1048400E+03,.9022200E+02,&
     & .7922000E+02,.7609000E+02,.7540500E+02,.8502200E+02,.9865300E+02,&
     & .1127100E+03,.1194200E+03,.1047600E+03,.9016900E+02,.7897600E+02,&
     & .7560700E+02,.7501800E+02,.8435200E+02,.9778300E+02,.1116700E+03/

      data   absa(316:450,11)   /                                       &
     & .1338100E+03,.1173500E+03,.1009500E+03,.9122600E+02,.8818800E+02,&
     & .8637900E+02,.9582800E+02,.1115500E+03,.1275500E+03,.1344600E+03,&
     & .1179400E+03,.1014500E+03,.9127100E+02,.8794700E+02,.8576600E+02,&
     & .9465200E+02,.1100600E+03,.1258300E+03,.1349200E+03,.1183400E+03,&
     & .1018000E+03,.9123300E+02,.8761900E+02,.8522800E+02,.9381100E+02,&
     & .1089200E+03,.1245200E+03,.1350500E+03,.1184600E+03,.1019200E+03,&
     & .9109000E+02,.8715000E+02,.8488700E+02,.9311100E+02,.1079700E+03,&
     & .1233700E+03,.1350000E+03,.1184400E+03,.1019200E+03,.9079700E+02,&
     & .8662600E+02,.8457000E+02,.9242100E+02,.1071000E+03,.1222600E+03,&
     & .1495200E+03,.1311000E+03,.1131600E+03,.1041000E+03,.9992700E+02,&
     & .9607400E+02,.1037400E+03,.1206700E+03,.1380100E+03,.1502800E+03,&
     & .1317900E+03,.1136000E+03,.1041500E+03,.9987000E+02,.9558900E+02,&
     & .1025700E+03,.1191200E+03,.1362300E+03,.1508200E+03,.1322700E+03,&
     & .1139200E+03,.1040500E+03,.9957100E+02,.9518900E+02,.1018200E+03,&
     & .1180300E+03,.1349600E+03,.1510300E+03,.1324700E+03,.1140300E+03,&
     & .1038500E+03,.9912600E+02,.9498000E+02,.1012200E+03,.1172000E+03,&
     & .1338800E+03,.1509400E+03,.1324500E+03,.1140100E+03,.1034800E+03,&
     & .9855000E+02,.9474600E+02,.1005500E+03,.1163600E+03,.1327600E+03,&
     & .1652400E+03,.1448600E+03,.1259400E+03,.1178500E+03,.1122500E+03,&
     & .1065400E+03,.1120200E+03,.1296500E+03,.1482900E+03,.1661000E+03,&
     & .1456400E+03,.1263600E+03,.1178700E+03,.1123300E+03,.1063000E+03,&
     & .1109300E+03,.1281100E+03,.1465300E+03,.1665900E+03,.1460800E+03,&
     & .1265600E+03,.1176900E+03,.1121400E+03,.1060900E+03,.1103200E+03,&
     & .1271300E+03,.1453200E+03,.1668100E+03,.1463000E+03,.1265900E+03,&
     & .1173600E+03,.1116900E+03,.1059700E+03,.1097300E+03,.1262900E+03,&
     & .1441600E+03,.1667100E+03,.1462300E+03,.1263900E+03,.1168800E+03,&
     & .1111100E+03,.1057600E+03,.1090900E+03,.1254200E+03,.1430300E+03/

      data   absa(451:585,11)   /                                       &
     & .1806500E+03,.1583200E+03,.1390300E+03,.1320100E+03,.1251900E+03,&
     & .1182000E+03,.1204700E+03,.1376000E+03,.1573900E+03,.1813100E+03,&
     & .1589300E+03,.1391600E+03,.1318300E+03,.1252000E+03,.1180000E+03,&
     & .1197900E+03,.1366000E+03,.1561200E+03,.1816200E+03,.1592500E+03,&
     & .1391200E+03,.1314500E+03,.1248700E+03,.1178900E+03,.1192200E+03,&
     & .1356600E+03,.1548100E+03,.1816100E+03,.1592700E+03,.1388800E+03,&
     & .1309100E+03,.1243200E+03,.1177000E+03,.1185800E+03,.1346200E+03,&
     & .1535000E+03,.1812900E+03,.1590100E+03,.1384900E+03,.1302700E+03,&
     & .1237100E+03,.1173800E+03,.1180000E+03,.1337800E+03,.1524800E+03,&
     & .1943900E+03,.1703100E+03,.1518100E+03,.1457700E+03,.1382800E+03,&
     & .1307000E+03,.1297200E+03,.1454600E+03,.1661800E+03,.1948300E+03,&
     & .1707300E+03,.1515600E+03,.1454300E+03,.1381200E+03,.1304300E+03,&
     & .1291300E+03,.1444300E+03,.1647600E+03,.1949400E+03,.1708700E+03,&
     & .1512400E+03,.1449300E+03,.1377100E+03,.1302000E+03,.1284700E+03,&
     & .1433000E+03,.1633500E+03,.1947400E+03,.1707300E+03,.1507800E+03,&
     & .1442500E+03,.1371800E+03,.1298600E+03,.1277900E+03,.1424000E+03,&
     & .1622900E+03,.1942000E+03,.1702900E+03,.1501100E+03,.1433700E+03,&
     & .1366600E+03,.1294200E+03,.1272500E+03,.1417100E+03,.1614700E+03,&
     & .2056200E+03,.1801400E+03,.1639500E+03,.1588500E+03,.1513100E+03,&
     & .1434400E+03,.1392800E+03,.1520100E+03,.1733600E+03,.2058300E+03,&
     & .1803400E+03,.1634600E+03,.1584400E+03,.1510200E+03,.1431000E+03,&
     & .1386700E+03,.1510100E+03,.1721000E+03,.2057900E+03,.1803400E+03,&
     & .1628200E+03,.1578200E+03,.1505100E+03,.1427100E+03,.1380300E+03,&
     & .1502600E+03,.1711900E+03,.2053900E+03,.1800200E+03,.1619900E+03,&
     & .1569600E+03,.1500500E+03,.1422600E+03,.1374600E+03,.1497700E+03,&
     & .1706100E+03,.2047100E+03,.1794600E+03,.1610100E+03,.1558000E+03,&
     & .1494200E+03,.1416300E+03,.1368500E+03,.1492200E+03,.1699800E+03/

      data   absa(  1:180,12)   /                                       &
     & .6180979E+02,.5415023E+02,.4652334E+02,.3927453E+02,.3636930E+02,&
     & .4458440E+02,.5332753E+02,.6199779E+02,.6686984E+02,.6201467E+02,&
     & .5432712E+02,.4674137E+02,.3945307E+02,.3621804E+02,.4416739E+02,&
     & .5281294E+02,.6138502E+02,.6622974E+02,.6212587E+02,.5443223E+02,&
     & .4690627E+02,.3958570E+02,.3609178E+02,.4380068E+02,.5235331E+02,&
     & .6084698E+02,.6567797E+02,.6210514E+02,.5448391E+02,.4698891E+02,&
     & .3965214E+02,.3594726E+02,.4347273E+02,.5194539E+02,.6034581E+02,&
     & .6519221E+02,.6202833E+02,.5446118E+02,.4698444E+02,.3963823E+02,&
     & .3578608E+02,.4315548E+02,.5153608E+02,.5985270E+02,.6466696E+02,&
     & .7408612E+02,.6488258E+02,.5568066E+02,.4672809E+02,.4299401E+02,&
     & .5300946E+02,.6345952E+02,.7381337E+02,.8120038E+02,.7431987E+02,&
     & .6508647E+02,.5591094E+02,.4696983E+02,.4270620E+02,.5245020E+02,&
     & .6277983E+02,.7301412E+02,.8031789E+02,.7445016E+02,.6522132E+02,&
     & .5612423E+02,.4716489E+02,.4247167E+02,.5192560E+02,.6214199E+02,&
     & .7225979E+02,.7950285E+02,.7449398E+02,.6532501E+02,.5623682E+02,&
     & .4725560E+02,.4229442E+02,.5140668E+02,.6150959E+02,.7151980E+02,&
     & .7870069E+02,.7445650E+02,.6532487E+02,.5624935E+02,.4725682E+02,&
     & .4210652E+02,.5092387E+02,.6091509E+02,.7081995E+02,.7792811E+02,&
     & .8862835E+02,.7760021E+02,.6657195E+02,.5566578E+02,.5128990E+02,&
     & .6327373E+02,.7579325E+02,.8820522E+02,.9907097E+02,.8897752E+02,&
     & .7790504E+02,.6685198E+02,.5596711E+02,.5087545E+02,.6255790E+02,&
     & .7492859E+02,.8717970E+02,.9792393E+02,.8915502E+02,.7806522E+02,&
     & .6707372E+02,.5618767E+02,.5053795E+02,.6185942E+02,.7408065E+02,&
     & .8619344E+02,.9681481E+02,.8920224E+02,.7817532E+02,.6720276E+02,&
     & .5630420E+02,.5024670E+02,.6121136E+02,.7329693E+02,.8526792E+02,&
     & .9576831E+02,.8916248E+02,.7817526E+02,.6722423E+02,.5632955E+02,&
     & .4997369E+02,.6057311E+02,.7252164E+02,.8436173E+02,.9475516E+02,&
     & .1056142E+03,.9245626E+02,.7929792E+02,.6618040E+02,.6147280E+02,&
     & .7551544E+02,.9049492E+02,.1053424E+03,.1195913E+03,.1061542E+03,&
     & .9293152E+02,.7971544E+02,.6661696E+02,.6092308E+02,.7453632E+02,&
     & .8931434E+02,.1039588E+03,.1180265E+03,.1064863E+03,.9323304E+02,&
     & .8004289E+02,.6692374E+02,.6046019E+02,.7370615E+02,.8831649E+02,&
     & .1027830E+03,.1166845E+03,.1066437E+03,.9342567E+02,.8024158E+02,&
     & .6710433E+02,.6006174E+02,.7289075E+02,.8732560E+02,.1016334E+03,&
     & .1153732E+03,.1066187E+03,.9342970E+02,.8026290E+02,.6712941E+02,&
     & .5968287E+02,.7205721E+02,.8632392E+02,.1004504E+03,.1140396E+03/

      data   absa(181:315,12)   /                                       &
     & .1255319E+03,.1098733E+03,.9422394E+02,.7856981E+02,.7364879E+02,&
     & .8993455E+02,.1078090E+03,.1255219E+03,.1431747E+03,.1261993E+03,&
     & .1104607E+03,.9472274E+02,.7906163E+02,.7299959E+02,.8871931E+02,&
     & .1063433E+03,.1238040E+03,.1412246E+03,.1266478E+03,.1108818E+03,&
     & .9515186E+02,.7946770E+02,.7245329E+02,.8760934E+02,.1050151E+03,&
     & .1222508E+03,.1394428E+03,.1269276E+03,.1111655E+03,.9543063E+02,&
     & .7971950E+02,.7193392E+02,.8655075E+02,.1037357E+03,.1207616E+03,&
     & .1377350E+03,.1269774E+03,.1112303E+03,.9549436E+02,.7977777E+02,&
     & .7145906E+02,.8552064E+02,.1024909E+03,.1193094E+03,.1360690E+03,&
     & .1486316E+03,.1300858E+03,.1115387E+03,.9299465E+02,.8823269E+02,&
     & .1068891E+03,.1281501E+03,.1492401E+03,.1705961E+03,.1495263E+03,&
     & .1308668E+03,.1122085E+03,.9359013E+02,.8736883E+02,.1053484E+03,&
     & .1263026E+03,.1470657E+03,.1681173E+03,.1501396E+03,.1314414E+03,&
     & .1127520E+03,.9410033E+02,.8669616E+02,.1039401E+03,.1246072E+03,&
     & .1450931E+03,.1658588E+03,.1504832E+03,.1317688E+03,.1130645E+03,&
     & .9437770E+02,.8607984E+02,.1025821E+03,.1229795E+03,.1431843E+03,&
     & .1636739E+03,.1505906E+03,.1318738E+03,.1131670E+03,.9446367E+02,&
     & .8553451E+02,.1012614E+03,.1213902E+03,.1413315E+03,.1615525E+03,&
     & .1749535E+03,.1531197E+03,.1312770E+03,.1098627E+03,.1054310E+03,&
     & .1267118E+03,.1519465E+03,.1769451E+03,.2025356E+03,.1761867E+03,&
     & .1541979E+03,.1321991E+03,.1103506E+03,.1043907E+03,.1247740E+03,&
     & .1496095E+03,.1742366E+03,.1994180E+03,.1770334E+03,.1549823E+03,&
     & .1329299E+03,.1109139E+03,.1035393E+03,.1229433E+03,.1474132E+03,&
     & .1716583E+03,.1964792E+03,.1775830E+03,.1554757E+03,.1333697E+03,&
     & .1112711E+03,.1027772E+03,.1211079E+03,.1452057E+03,.1690988E+03,&
     & .1935313E+03,.1778130E+03,.1556757E+03,.1335484E+03,.1114211E+03,&
     & .1020906E+03,.1193757E+03,.1431352E+03,.1666861E+03,.1907553E+03/

      data   absa(316:450,12)   /                                       &
     & .2050280E+03,.1794206E+03,.1538132E+03,.1296698E+03,.1256113E+03,&
     & .1498126E+03,.1795895E+03,.2091417E+03,.2395094E+03,.2065326E+03,&
     & .1807328E+03,.1549517E+03,.1300986E+03,.1242724E+03,.1472448E+03,&
     & .1765192E+03,.2055840E+03,.2354240E+03,.2076189E+03,.1817305E+03,&
     & .1558434E+03,.1304997E+03,.1232100E+03,.1448216E+03,.1736269E+03,&
     & .2021988E+03,.2315773E+03,.2083544E+03,.1823886E+03,.1564141E+03,&
     & .1307551E+03,.1223058E+03,.1424212E+03,.1707823E+03,.1989036E+03,&
     & .2278090E+03,.2086842E+03,.1826772E+03,.1566702E+03,.1308824E+03,&
     & .1214212E+03,.1402293E+03,.1681510E+03,.1958442E+03,.2243003E+03,&
     & .2393883E+03,.2094845E+03,.1795806E+03,.1526649E+03,.1492182E+03,&
     & .1761707E+03,.2111935E+03,.2459442E+03,.2817236E+03,.2413366E+03,&
     & .2111878E+03,.1810490E+03,.1530843E+03,.1474621E+03,.1729020E+03,&
     & .2072851E+03,.2414036E+03,.2765008E+03,.2426640E+03,.2123804E+03,&
     & .1821044E+03,.1534075E+03,.1461365E+03,.1697315E+03,.2034720E+03,&
     & .2369666E+03,.2714312E+03,.2435891E+03,.2131944E+03,.1828085E+03,&
     & .1535710E+03,.1449502E+03,.1666087E+03,.1997639E+03,.2326282E+03,&
     & .2664788E+03,.2439749E+03,.2135440E+03,.1831044E+03,.1535327E+03,&
     & .1438260E+03,.1637945E+03,.1963970E+03,.2287448E+03,.2620139E+03,&
     & .2786275E+03,.2438158E+03,.2090742E+03,.1791980E+03,.1764044E+03,&
     & .2051521E+03,.2459740E+03,.2864840E+03,.3281621E+03,.2809503E+03,&
     & .2458500E+03,.2108110E+03,.1796731E+03,.1742226E+03,.2010652E+03,&
     & .2410665E+03,.2807744E+03,.3216182E+03,.2826742E+03,.2473903E+03,&
     & .2121402E+03,.1799408E+03,.1725339E+03,.1970313E+03,.2361865E+03,&
     & .2750847E+03,.3150950E+03,.2837444E+03,.2483293E+03,.2129494E+03,&
     & .1799627E+03,.1710146E+03,.1933057E+03,.2316484E+03,.2698242E+03,&
     & .3090609E+03,.2842653E+03,.2487778E+03,.2133179E+03,.1797427E+03,&
     & .1695361E+03,.1899175E+03,.2275176E+03,.2649935E+03,.3035329E+03/

      data   absa(451:585,12)   /                                       &
     & .3246164E+03,.2841571E+03,.2437591E+03,.2099009E+03,.2061338E+03,&
     & .2346079E+03,.2807869E+03,.3270860E+03,.3746416E+03,.3270160E+03,&
     & .2862343E+03,.2455753E+03,.2101750E+03,.2037891E+03,.2298452E+03,&
     & .2746768E+03,.3199854E+03,.3664519E+03,.3287071E+03,.2876768E+03,&
     & .2468055E+03,.2102385E+03,.2017300E+03,.2252550E+03,.2691398E+03,&
     & .3135254E+03,.3590802E+03,.3296547E+03,.2884880E+03,.2474892E+03,&
     & .2100045E+03,.1997651E+03,.2211596E+03,.2640071E+03,.3075507E+03,&
     & .3522449E+03,.3298874E+03,.2886932E+03,.2476393E+03,.2093907E+03,&
     & .1977676E+03,.2173310E+03,.2589645E+03,.3016760E+03,.3454882E+03,&
     & .3764185E+03,.3295629E+03,.2827412E+03,.2448362E+03,.2395515E+03,&
     & .2667523E+03,.3170130E+03,.3693427E+03,.4229547E+03,.3787898E+03,&
     & .3316408E+03,.2845544E+03,.2447605E+03,.2368444E+03,.2613950E+03,&
     & .3103293E+03,.3615554E+03,.4140351E+03,.3801676E+03,.3328413E+03,&
     & .2856113E+03,.2443688E+03,.2342033E+03,.2564579E+03,.3041029E+03,&
     & .3543252E+03,.4057425E+03,.3807335E+03,.3332994E+03,.2860069E+03,&
     & .2436761E+03,.2315207E+03,.2518766E+03,.2980292E+03,.3472163E+03,&
     & .3976049E+03,.3804795E+03,.3330315E+03,.2857763E+03,.2425900E+03,&
     & .2287390E+03,.2473661E+03,.2919475E+03,.3401341E+03,.3894698E+03,&
     & .4337936E+03,.3798110E+03,.3258724E+03,.2839162E+03,.2765829E+03,&
     & .3019915E+03,.3551500E+03,.4138455E+03,.4738475E+03,.4358679E+03,&
     & .3816444E+03,.3274735E+03,.2832262E+03,.2732493E+03,.2960468E+03,&
     & .3477195E+03,.4051747E+03,.4639479E+03,.4367543E+03,.3824308E+03,&
     & .3281687E+03,.2821996E+03,.2697997E+03,.2905362E+03,.3403748E+03,&
     & .3966217E+03,.4541170E+03,.4367336E+03,.3824073E+03,.3281688E+03,&
     & .2807593E+03,.2662071E+03,.2851390E+03,.3330291E+03,.3880528E+03,&
     & .4442749E+03,.4356281E+03,.3814241E+03,.3273328E+03,.2789760E+03,&
     & .2624596E+03,.2797763E+03,.3260499E+03,.3798827E+03,.4349494E+03/


      data   absb(  1:175, 1)   /                                       &
     & .1055400E-06,.3859600E-04,.6540200E-04,.8854000E-04,.1960100E-03,&
     & .1047900E-06,.4367200E-04,.7488900E-04,.1011800E-03,.2214600E-03,&
     & .1042900E-06,.4941800E-04,.8542900E-04,.1155700E-03,.2444800E-03,&
     & .1023200E-06,.5580200E-04,.9645900E-04,.1316700E-03,.2700100E-03,&
     & .9901300E-07,.6299600E-04,.1090000E-03,.1483400E-03,.2991700E-03,&
     & .8513600E-07,.3194300E-04,.5440200E-04,.7388700E-04,.1589500E-03,&
     & .8459700E-07,.3619600E-04,.6242900E-04,.8447900E-04,.1774900E-03,&
     & .8362900E-07,.4098200E-04,.7112600E-04,.9657300E-04,.1974000E-03,&
     & .8231000E-07,.4634400E-04,.8045800E-04,.1098400E-03,.2184700E-03,&
     & .7870900E-07,.5225200E-04,.9098900E-04,.1239400E-03,.2428100E-03,&
     & .6849700E-07,.2639300E-04,.4516400E-04,.6157000E-04,.1249300E-03,&
     & .6806200E-07,.2993600E-04,.5191900E-04,.7038700E-04,.1395300E-03,&
     & .6676600E-07,.3386300E-04,.5905600E-04,.8069300E-04,.1565300E-03,&
     & .6563600E-07,.3829400E-04,.6688500E-04,.9155500E-04,.1751000E-03,&
     & .6271100E-07,.4318900E-04,.7575600E-04,.1035500E-03,.1943900E-03,&
     & .5508300E-07,.2176100E-04,.3748400E-04,.5130100E-04,.9787500E-04,&
     & .5409500E-07,.2467000E-04,.4308400E-04,.5863900E-04,.1099600E-03,&
     & .5334000E-07,.2792500E-04,.4897600E-04,.6723400E-04,.1241500E-03,&
     & .5141200E-07,.3158800E-04,.5557600E-04,.7635200E-04,.1393900E-03,&
     & .4934100E-07,.3567700E-04,.6293700E-04,.8636300E-04,.1554300E-03,&
     & .4402200E-07,.1796300E-04,.3110200E-04,.4260400E-04,.7829000E-04,&
     & .4316700E-07,.2038200E-04,.3575400E-04,.4887500E-04,.8794100E-04,&
     & .4221000E-07,.2309100E-04,.4067400E-04,.5605900E-04,.9925700E-04,&
     & .4040200E-07,.2612500E-04,.4616200E-04,.6358600E-04,.1116600E-03,&
     & .3853200E-07,.2949400E-04,.5232700E-04,.7196000E-04,.1248000E-03,&
     & .3493700E-07,.1490300E-04,.2586800E-04,.3550300E-04,.6336800E-04,&
     & .3430800E-07,.1692300E-04,.2978100E-04,.4078900E-04,.7111500E-04,&
     & .3299500E-07,.1917100E-04,.3387200E-04,.4681000E-04,.8030200E-04,&
     & .3165400E-07,.2167900E-04,.3846800E-04,.5304800E-04,.9043100E-04,&
     & .3028200E-07,.2449100E-04,.4359600E-04,.6007800E-04,.1013300E-03,&
     & .2797100E-07,.1238500E-04,.2154900E-04,.2961800E-04,.5173200E-04,&
     & .2712800E-07,.1404200E-04,.2482600E-04,.3404700E-04,.5806400E-04,&
     & .2601800E-07,.1592000E-04,.2822100E-04,.3907200E-04,.6549500E-04,&
     & .2471700E-07,.1801700E-04,.3205600E-04,.4428500E-04,.7385900E-04,&
     & .2385900E-07,.2036000E-04,.3634100E-04,.5017000E-04,.8294400E-04/

      data   absb(176:350, 1)   /                                       &
     & .2252900E-07,.1033600E-04,.1799900E-04,.2477600E-04,.4260200E-04,&
     & .2173300E-07,.1168500E-04,.2076900E-04,.2850300E-04,.4783300E-04,&
     & .2079100E-07,.1328000E-04,.2357100E-04,.3265400E-04,.5399500E-04,&
     & .1966300E-07,.1502900E-04,.2678800E-04,.3706600E-04,.6094400E-04,&
     & .1916900E-07,.1697500E-04,.3035300E-04,.4199600E-04,.6834400E-04,&
     & .1814800E-07,.8603900E-05,.1504100E-04,.2071700E-04,.3514300E-04,&
     & .1746500E-07,.9728700E-05,.1734800E-04,.2386200E-04,.3950800E-04,&
     & .1653700E-07,.1107300E-04,.1968700E-04,.2727100E-04,.4465200E-04,&
     & .1586000E-07,.1252600E-04,.2238800E-04,.3101500E-04,.5026900E-04,&
     & .1545200E-07,.1415800E-04,.2533900E-04,.3514100E-04,.5632700E-04,&
     & .1454200E-07,.7180700E-05,.1268200E-04,.1747300E-04,.2917900E-04,&
     & .1396700E-07,.8157400E-05,.1455000E-04,.2014700E-04,.3279500E-04,&
     & .1324100E-07,.9283900E-05,.1654700E-04,.2297100E-04,.3703600E-04,&
     & .1274600E-07,.1052200E-04,.1884000E-04,.2611400E-04,.4161700E-04,&
     & .1242600E-07,.1188600E-04,.2131300E-04,.2960400E-04,.4670900E-04,&
     & .1172400E-07,.6019600E-05,.1070300E-04,.1475300E-04,.2414300E-04,&
     & .1112400E-07,.6849900E-05,.1223300E-04,.1700900E-04,.2714700E-04,&
     & .1055200E-07,.7789700E-05,.1392700E-04,.1936200E-04,.3066100E-04,&
     & .1021300E-07,.8839800E-05,.1585000E-04,.2202400E-04,.3451900E-04,&
     & .9962700E-08,.9975500E-05,.1791900E-04,.2494400E-04,.3873000E-04,&
     & .9383500E-08,.5050500E-05,.9029800E-05,.1245000E-04,.1994000E-04,&
     & .8872200E-08,.5749400E-05,.1028700E-04,.1435100E-04,.2245000E-04,&
     & .8471900E-08,.6537400E-05,.1172700E-04,.1632900E-04,.2537800E-04,&
     & .8192400E-08,.7418600E-05,.1333400E-04,.1856700E-04,.2856000E-04,&
     & .7919400E-08,.8371900E-05,.1505600E-04,.2100300E-04,.3199000E-04,&
     & .7453500E-08,.4243700E-05,.7622000E-05,.1052300E-04,.1653100E-04,&
     & .7051600E-08,.4830000E-05,.8666700E-05,.1208900E-04,.1861600E-04,&
     & .6774600E-08,.5494100E-05,.9892700E-05,.1379000E-04,.2102400E-04,&
     & .6537500E-08,.6235700E-05,.1123300E-04,.1566200E-04,.2364200E-04,&
     & .6300100E-08,.7028200E-05,.1266900E-04,.1769500E-04,.2647800E-04,&
     & .5952000E-08,.3573900E-05,.6435000E-05,.8933000E-05,.1375400E-04,&
     & .5664400E-08,.4079700E-05,.7329100E-05,.1022800E-04,.1548900E-04,&
     & .5422900E-08,.4636600E-05,.8366400E-05,.1165900E-04,.1748600E-04,&
     & .5215500E-08,.5254600E-05,.9482900E-05,.1324600E-04,.1964300E-04,&
     & .5002600E-08,.5913400E-05,.1067600E-04,.1493300E-04,.2197900E-04/

      data   absb(351:525, 1)   /                                       &
     & .4734700E-08,.3011700E-05,.5422300E-05,.7581000E-05,.1142400E-04,&
     & .4521500E-08,.3442100E-05,.6195900E-05,.8661900E-05,.1288400E-04,&
     & .4333400E-08,.3914300E-05,.7070200E-05,.9876100E-05,.1454300E-04,&
     & .4135000E-08,.4427500E-05,.8000500E-05,.1119500E-04,.1631900E-04,&
     & .3965300E-08,.4976900E-05,.8993100E-05,.1259800E-04,.1816000E-04,&
     & .3794800E-08,.2541500E-05,.4581900E-05,.6433600E-05,.9495600E-05,&
     & .3608300E-08,.2903800E-05,.5242600E-05,.7340000E-05,.1071900E-04,&
     & .3441800E-08,.3301700E-05,.5973300E-05,.8357500E-05,.1206600E-04,&
     & .3263600E-08,.3729700E-05,.6749300E-05,.9455800E-05,.1346800E-04,&
     & .3134000E-08,.4187300E-05,.7575900E-05,.1062000E-04,.1495300E-04,&
     & .3012100E-08,.2153600E-05,.3884100E-05,.5449100E-05,.7909000E-05,&
     & .2877500E-08,.2455700E-05,.4446700E-05,.6224700E-05,.8914800E-05,&
     & .2719600E-08,.2790000E-05,.5054300E-05,.7078200E-05,.9994300E-05,&
     & .2579800E-08,.3147300E-05,.5701100E-05,.7993600E-05,.1113500E-04,&
     & .2478500E-08,.3528700E-05,.6389400E-05,.8962400E-05,.1235700E-04,&
     & .2403400E-08,.1823100E-05,.3294500E-05,.4618300E-05,.6583700E-05,&
     & .2270200E-08,.2076600E-05,.3767100E-05,.5273500E-05,.7387500E-05,&
     & .2155100E-08,.2355900E-05,.4274400E-05,.5993200E-05,.8266500E-05,&
     & .2039700E-08,.2654100E-05,.4813700E-05,.6754900E-05,.9207100E-05,&
     & .1920800E-08,.2971100E-05,.5388000E-05,.7559800E-05,.1021100E-04,&
     & .1911600E-08,.1545400E-05,.2795700E-05,.3922800E-05,.5466400E-05,&
     & .1804500E-08,.1757900E-05,.3193800E-05,.4478000E-05,.6120500E-05,&
     & .1701500E-08,.1991100E-05,.3616700E-05,.5075300E-05,.6852300E-05,&
     & .1604200E-08,.2239300E-05,.4066600E-05,.5709400E-05,.7626000E-05,&
     & .1518900E-08,.2503500E-05,.4543900E-05,.6378500E-05,.8456200E-05,&
     & .1513400E-08,.1308000E-05,.2376300E-05,.3336000E-05,.4538200E-05,&
     & .1425400E-08,.1488600E-05,.2708600E-05,.3799500E-05,.5093800E-05,&
     & .1347800E-08,.1683100E-05,.3060500E-05,.4297100E-05,.5693300E-05,&
     & .1272600E-08,.1890400E-05,.3435300E-05,.4824700E-05,.6335300E-05,&
     & .1202700E-08,.2109800E-05,.3831800E-05,.5380900E-05,.7030200E-05,&
     & .1202400E-08,.1108400E-05,.2019300E-05,.2831000E-05,.3784900E-05,&
     & .1133800E-08,.1260500E-05,.2294900E-05,.3222100E-05,.4245800E-05,&
     & .1067800E-08,.1423100E-05,.2588100E-05,.3636500E-05,.4746000E-05,&
     & .1005200E-08,.1596300E-05,.2901000E-05,.4072800E-05,.5280400E-05,&
     & .9594600E-09,.1777900E-05,.3230100E-05,.4538100E-05,.5854700E-05/

      data   absb(526:700, 1)   /                                       &
     & .9575000E-09,.9366500E-06,.1706800E-05,.2395400E-05,.3161100E-05,&
     & .8999100E-09,.1063600E-05,.1936600E-05,.2720900E-05,.3544000E-05,&
     & .8494400E-09,.1199400E-05,.2180400E-05,.3065100E-05,.3960000E-05,&
     & .8086300E-09,.1343100E-05,.2440600E-05,.3428500E-05,.4404500E-05,&
     & .7669700E-09,.1493700E-05,.2713800E-05,.3815300E-05,.4873900E-05,&
     & .7693100E-09,.7814500E-06,.1424300E-05,.2000400E-05,.2608800E-05,&
     & .7255400E-09,.8865200E-06,.1614000E-05,.2268400E-05,.2925500E-05,&
     & .6867700E-09,.9987200E-06,.1815900E-05,.2553700E-05,.3267200E-05,&
     & .6498600E-09,.1117800E-05,.2031200E-05,.2854200E-05,.3630400E-05,&
     & .6217700E-09,.1242000E-05,.2256600E-05,.3173700E-05,.4018700E-05,&
     & .6224500E-09,.6423500E-06,.1170900E-05,.1644500E-05,.2121300E-05,&
     & .5869600E-09,.7287600E-06,.1326600E-05,.1864600E-05,.2379800E-05,&
     & .5569300E-09,.8208900E-06,.1492700E-05,.2099100E-05,.2660000E-05,&
     & .5293400E-09,.9187400E-06,.1669500E-05,.2345700E-05,.2957500E-05,&
     & .5058500E-09,.1021000E-05,.1854900E-05,.2608500E-05,.3275800E-05,&
     & .5115600E-09,.5187200E-06,.9455000E-06,.1328700E-05,.1708900E-05,&
     & .4825800E-09,.5890800E-06,.1072200E-05,.1507900E-05,.1918700E-05,&
     & .4569400E-09,.6642100E-06,.1207800E-05,.1699200E-05,.2145600E-05,&
     & .4350200E-09,.7440800E-06,.1352000E-05,.1900000E-05,.2387600E-05,&
     & .4151700E-09,.8275800E-06,.1503400E-05,.2114600E-05,.2647300E-05,&
     & .4209400E-09,.4183700E-06,.7626000E-06,.1071900E-05,.1375600E-05,&
     & .3970000E-09,.4755700E-06,.8657200E-06,.1218000E-05,.1546200E-05,&
     & .3756300E-09,.5368400E-06,.9761300E-06,.1373800E-05,.1728900E-05,&
     & .3578300E-09,.6019300E-06,.1093600E-05,.1537400E-05,.1925200E-05,&
     & .3410700E-09,.6701400E-06,.1217300E-05,.1712700E-05,.2136900E-05,&
     & .3463500E-09,.3373200E-06,.6149200E-06,.8644700E-06,.1105500E-05,&
     & .3265400E-09,.3838300E-06,.6986900E-06,.9835500E-06,.1244000E-05,&
     & .3090900E-09,.4337100E-06,.7886900E-06,.1110500E-05,.1392400E-05,&
     & .2942500E-09,.4868100E-06,.8844600E-06,.1243300E-05,.1552600E-05,&
     & .2800000E-09,.5425600E-06,.9855000E-06,.1386800E-05,.1723800E-05,&
     & .2873100E-09,.2684300E-06,.4891500E-06,.6881000E-06,.8786700E-06,&
     & .2708900E-09,.3060000E-06,.5569800E-06,.7843000E-06,.9907500E-06,&
     & .2560700E-09,.3462400E-06,.6296600E-06,.8867700E-06,.1110700E-05,&
     & .2437200E-09,.3892600E-06,.7073200E-06,.9943000E-06,.1240200E-05,&
     & .2317600E-09,.4346500E-06,.7894300E-06,.1110700E-05,.1378700E-05/

      data   absb(701:875, 1)   /                                       &
     & .2390300E-09,.2132300E-06,.3885500E-06,.5470100E-06,.6978100E-06,&
     & .2248700E-09,.2433200E-06,.4432800E-06,.6239700E-06,.7877400E-06,&
     & .2123600E-09,.2758100E-06,.5017000E-06,.7066400E-06,.8842700E-06,&
     & .2020400E-09,.3106600E-06,.5643500E-06,.7938400E-06,.9889900E-06,&
     & .1920900E-09,.3475300E-06,.6311900E-06,.8878700E-06,.1100700E-05,&
     & .1986800E-09,.1695000E-06,.3088500E-06,.4342600E-06,.5533600E-06,&
     & .1868100E-09,.1933800E-06,.3525000E-06,.4962200E-06,.6262800E-06,&
     & .1764900E-09,.2196300E-06,.3995000E-06,.5626900E-06,.7039600E-06,&
     & .1675300E-09,.2477800E-06,.4501300E-06,.6334200E-06,.7877000E-06,&
     & .1593000E-09,.2777200E-06,.5043900E-06,.7094500E-06,.8778300E-06,&
     & .1657700E-09,.1333200E-06,.2432300E-06,.3418400E-06,.4359400E-06,&
     & .1559700E-09,.1524500E-06,.2780100E-06,.3913000E-06,.4946000E-06,&
     & .1472100E-09,.1735400E-06,.3158400E-06,.4449900E-06,.5570500E-06,&
     & .1393600E-09,.1962500E-06,.3566100E-06,.5018900E-06,.6239800E-06,&
     & .1325500E-09,.2204400E-06,.4003400E-06,.5630800E-06,.6963400E-06,&
     & .1386100E-09,.1044500E-06,.1905800E-06,.2684100E-06,.3423300E-06,&
     & .1302700E-09,.1197400E-06,.2183600E-06,.3074400E-06,.3889700E-06,&
     & .1229000E-09,.1366200E-06,.2486400E-06,.3506500E-06,.4391600E-06,&
     & .1161100E-09,.1548400E-06,.2815500E-06,.3962700E-06,.4927300E-06,&
     & .1104200E-09,.1743100E-06,.3166600E-06,.4454000E-06,.5508700E-06,&
     & .1158100E-09,.8191500E-07,.1491500E-06,.2114000E-06,.2698700E-06,&
     & .1090600E-09,.9407700E-07,.1714300E-06,.2417300E-06,.3058600E-06,&
     & .1025400E-09,.1074500E-06,.1958200E-06,.2760600E-06,.3459300E-06,&
     & .9675700E-10,.1220500E-06,.2220100E-06,.3127100E-06,.3887600E-06,&
     & .9197900E-10,.1377200E-06,.2501500E-06,.3520800E-06,.4356100E-06,&
     & .9688900E-10,.6411600E-07,.1170800E-06,.1645500E-06,.2103600E-06,&
     & .9128100E-10,.7364100E-07,.1343600E-06,.1889500E-06,.2395100E-06,&
     & .8583000E-10,.8416100E-07,.1536000E-06,.2164300E-06,.2717600E-06,&
     & .8104800E-10,.9586700E-07,.1744700E-06,.2458200E-06,.3061300E-06,&
     & .7671200E-10,.1084400E-06,.1970600E-06,.2774300E-06,.3432900E-06,&
     & .8151600E-10,.4969800E-07,.9059600E-07,.1272700E-06,.1631300E-06,&
     & .7660900E-10,.5713200E-07,.1043300E-06,.1469400E-06,.1867200E-06,&
     & .7197500E-10,.6553400E-07,.1195500E-06,.1685000E-06,.2121500E-06,&
     & .6790800E-10,.7482900E-07,.1361900E-06,.1920900E-06,.2395900E-06,&
     & .6411700E-10,.8487200E-07,.1543300E-06,.2173300E-06,.2691500E-06/

      data   absb(876:1050, 1)   /                                      &
     & .6842900E-10,.3871500E-07,.6985400E-07,.9837700E-07,.1261900E-06,&
     & .6423600E-10,.4437500E-07,.8089500E-07,.1147100E-06,.1460400E-06,&
     & .6051000E-10,.5102500E-07,.9297700E-07,.1312300E-06,.1654200E-06,&
     & .5685100E-10,.5834500E-07,.1063300E-06,.1499100E-06,.1874500E-06,&
     & .5367400E-10,.6634600E-07,.1207000E-06,.1700100E-06,.2109500E-06,&
     & .5738700E-10,.2980000E-07,.5381700E-07,.7590600E-07,.9771000E-07,&
     & .5396000E-10,.3450000E-07,.6320500E-07,.8865500E-07,.1131000E-06,&
     & .5074000E-10,.3968400E-07,.7250200E-07,.1019300E-06,.1289200E-06,&
     & .4771000E-10,.4541800E-07,.8294100E-07,.1169200E-06,.1464900E-06,&
     & .4505000E-10,.5179700E-07,.9432000E-07,.1329600E-06,.1652600E-06,&
     & .4801000E-10,.2299900E-07,.4168200E-07,.5897500E-07,.7589800E-07,&
     & .4528200E-10,.2691700E-07,.4892900E-07,.6876500E-07,.8798300E-07,&
     & .4253100E-10,.3087900E-07,.5640600E-07,.7959300E-07,.1010900E-06,&
     & .4000200E-10,.3547000E-07,.6470600E-07,.9131200E-07,.1147300E-06,&
     & .3767400E-10,.4054200E-07,.7383400E-07,.1041200E-06,.1296700E-06,&
     & .4009400E-10,.1781000E-07,.3241200E-07,.4598500E-07,.5904100E-07,&
     & .3790100E-10,.2101300E-07,.3792100E-07,.5347900E-07,.6847900E-07,&
     & .3558400E-10,.2410000E-07,.4398800E-07,.6240900E-07,.7931100E-07,&
     & .3351600E-10,.2773700E-07,.5054400E-07,.7143600E-07,.8991200E-07,&
     & .3150300E-10,.3173800E-07,.5788900E-07,.8159100E-07,.1019500E-06,&
     & .3346200E-10,.1382900E-07,.2522700E-07,.3549800E-07,.4579300E-07,&
     & .3170300E-10,.1627200E-07,.2939300E-07,.4147600E-07,.5334000E-07,&
     & .2985600E-10,.1883600E-07,.3453700E-07,.4849000E-07,.6179100E-07,&
     & .2803800E-10,.2168400E-07,.3957800E-07,.5575100E-07,.7042200E-07,&
     & .2637500E-10,.2482400E-07,.4538900E-07,.6389800E-07,.8007100E-07,&
     & .2785900E-10,.1075700E-07,.1943600E-07,.2736800E-07,.3540100E-07,&
     & .2648200E-10,.1257500E-07,.2278600E-07,.3227500E-07,.4150400E-07,&
     & .2501300E-10,.1471800E-07,.2678600E-07,.3765700E-07,.4815000E-07,&
     & .2348100E-10,.1689000E-07,.3087500E-07,.4356100E-07,.5528400E-07,&
     & .2210300E-10,.1940500E-07,.3543500E-07,.5004600E-07,.6282400E-07,&
     & .2326200E-10,.8353400E-08,.1503300E-07,.2114200E-07,.2749100E-07,&
     & .2207500E-10,.9782100E-08,.1779100E-07,.2529800E-07,.3244700E-07,&
     & .2088700E-10,.1154100E-07,.2085200E-07,.2939200E-07,.3764300E-07,&
     & .1962900E-10,.1322800E-07,.2416400E-07,.3433000E-07,.4362200E-07,&
     & .1847500E-10,.1523300E-07,.2777900E-07,.3930400E-07,.4943200E-07/

      data   absb(1051:1175, 1)   /                                     &
     & .1936000E-10,.6488800E-08,.1164400E-07,.1638700E-07,.2134000E-07,&
     & .1837200E-10,.7636700E-08,.1394700E-07,.1967200E-07,.2540100E-07,&
     & .1742100E-10,.9008300E-08,.1627500E-07,.2297100E-07,.2952200E-07,&
     & .1641300E-10,.1040200E-07,.1908100E-07,.2685400E-07,.3423300E-07,&
     & .1541700E-10,.1198400E-07,.2183500E-07,.3085600E-07,.3895400E-07,&
     & .1609400E-10,.5017400E-08,.9028300E-08,.1271800E-07,.1657600E-07,&
     & .1526500E-10,.5974700E-08,.1085000E-07,.1529700E-07,.1978300E-07,&
     & .1451500E-10,.7014200E-08,.1269900E-07,.1796000E-07,.2317100E-07,&
     & .1371500E-10,.8162600E-08,.1494000E-07,.2099400E-07,.2686500E-07,&
     & .1288000E-10,.9405400E-08,.1720600E-07,.2422100E-07,.3073900E-07,&
     & .1336800E-10,.3883800E-08,.6999100E-08,.9837200E-08,.1282100E-07,&
     & .1274300E-10,.4675100E-08,.8424300E-08,.1185600E-07,.1540600E-07,&
     & .1208700E-10,.5464200E-08,.9929500E-08,.1406900E-07,.1814400E-07,&
     & .1144200E-10,.6446100E-08,.1165700E-07,.1640700E-07,.2107200E-07,&
     & .1075900E-10,.7371100E-08,.1347800E-07,.1910600E-07,.2431400E-07,&
     & .1108100E-10,.3014800E-08,.5416500E-08,.7642100E-08,.9949600E-08,&
     & .1059800E-10,.3633700E-08,.6537500E-08,.9204100E-08,.1202500E-07,&
     & .1005200E-10,.4271200E-08,.7784100E-08,.1105300E-07,.1430600E-07,&
     & .9530800E-11,.5042900E-08,.9113300E-08,.1286500E-07,.1655400E-07,&
     & .8979500E-11,.5797900E-08,.1061500E-07,.1505100E-07,.1921600E-07,&
     & .9119400E-11,.2421400E-08,.4349600E-08,.6153100E-08,.8013600E-08,&
     & .8728700E-11,.2923200E-08,.5253500E-08,.7396600E-08,.9683000E-08,&
     & .8276500E-11,.3441200E-08,.6278800E-08,.8899200E-08,.1153400E-07,&
     & .7860500E-11,.4063200E-08,.7345400E-08,.1038200E-07,.1337200E-07,&
     & .7412700E-11,.4680100E-08,.8591000E-08,.1213700E-07,.1553000E-07/

      data   absb(  1:175, 2)   /                                       &
     & .1757600E-05,.2251800E-03,.3639800E-03,.4650500E-03,.9792000E-03,&
     & .1952200E-05,.2568800E-03,.4272400E-03,.5429300E-03,.1126100E-02,&
     & .2113400E-05,.2938200E-03,.4977800E-03,.6318600E-03,.1296100E-02,&
     & .2268000E-05,.3358500E-03,.5709500E-03,.7314900E-03,.1485700E-02,&
     & .2393900E-05,.3832600E-03,.6468800E-03,.8412000E-03,.1681500E-02,&
     & .1386600E-05,.1881600E-03,.3054100E-03,.3908200E-03,.8061400E-03,&
     & .1482500E-05,.2152500E-03,.3590600E-03,.4577200E-03,.9297600E-03,&
     & .1642800E-05,.2466400E-03,.4189100E-03,.5334100E-03,.1068900E-02,&
     & .1780700E-05,.2821800E-03,.4802100E-03,.6186100E-03,.1223600E-02,&
     & .1868500E-05,.3220400E-03,.5438500E-03,.7122300E-03,.1384400E-02,&
     & .1048300E-05,.1569900E-03,.2557100E-03,.3280300E-03,.6587300E-03,&
     & .1131000E-05,.1801600E-03,.3007100E-03,.3853400E-03,.7591600E-03,&
     & .1233000E-05,.2067200E-03,.3518100E-03,.4494000E-03,.8704100E-03,&
     & .1342600E-05,.2365000E-03,.4029200E-03,.5224600E-03,.9944500E-03,&
     & .1451400E-05,.2697900E-03,.4568400E-03,.6019500E-03,.1117300E-02,&
     & .8066300E-06,.1309800E-03,.2138200E-03,.2753900E-03,.5372700E-03,&
     & .8574700E-06,.1505800E-03,.2520100E-03,.3240600E-03,.6157300E-03,&
     & .9243500E-06,.1728300E-03,.2954200E-03,.3787900E-03,.7014000E-03,&
     & .9974700E-06,.1979300E-03,.3380700E-03,.4407600E-03,.7946800E-03,&
     & .1081900E-05,.2258200E-03,.3835400E-03,.5066300E-03,.8898000E-03,&
     & .6177200E-06,.1094100E-03,.1792600E-03,.2314400E-03,.4355400E-03,&
     & .6571800E-06,.1257600E-03,.2117400E-03,.2727500E-03,.4975000E-03,&
     & .6994900E-06,.1446100E-03,.2485200E-03,.3193700E-03,.5646200E-03,&
     & .7538700E-06,.1658000E-03,.2837400E-03,.3721300E-03,.6350700E-03,&
     & .7971500E-06,.1892600E-03,.3223600E-03,.4258800E-03,.7072800E-03,&
     & .4718800E-06,.9195200E-04,.1510000E-03,.1949500E-03,.3554300E-03,&
     & .5093700E-06,.1051300E-03,.1785200E-03,.2299900E-03,.4025000E-03,&
     & .5420600E-06,.1213000E-03,.2095900E-03,.2697900E-03,.4566200E-03,&
     & .5732800E-06,.1393600E-03,.2381500E-03,.3145100E-03,.5111300E-03,&
     & .6107700E-06,.1591600E-03,.2707500E-03,.3596700E-03,.5707600E-03,&
     & .3600500E-06,.7698400E-04,.1274300E-03,.1642200E-03,.2888500E-03,&
     & .3898800E-06,.8802900E-04,.1506700E-03,.1939600E-03,.3285400E-03,&
     & .4198900E-06,.1017200E-03,.1753900E-03,.2278400E-03,.3704500E-03,&
     & .4409200E-06,.1172700E-03,.2000400E-03,.2659200E-03,.4159800E-03,&
     & .4688300E-06,.1338600E-03,.2276900E-03,.3028500E-03,.4655000E-03/

      data   absb(176:350, 2)   /                                       &
     & .2790800E-06,.6445000E-04,.1079500E-03,.1388900E-03,.2387800E-03,&
     & .3093400E-06,.7418700E-04,.1277300E-03,.1641700E-03,.2701700E-03,&
     & .3308600E-06,.8580100E-04,.1478200E-03,.1930700E-03,.3050800E-03,&
     & .3505800E-06,.9886100E-04,.1686800E-03,.2250800E-03,.3424900E-03,&
     & .3706900E-06,.1130600E-03,.1918800E-03,.2560100E-03,.3851900E-03,&
     & .2241800E-06,.5409500E-04,.9141400E-04,.1175700E-03,.1969700E-03,&
     & .2455900E-06,.6256700E-04,.1081900E-03,.1390800E-03,.2228400E-03,&
     & .2616900E-06,.7238300E-04,.1248100E-03,.1636800E-03,.2520500E-03,&
     & .2788500E-06,.8344600E-04,.1425800E-03,.1891000E-03,.2838600E-03,&
     & .2934400E-06,.9554000E-04,.1619800E-03,.2160900E-03,.3194700E-03,&
     & .1794100E-06,.4583000E-04,.7817600E-04,.1006100E-03,.1631300E-03,&
     & .1939700E-06,.5313900E-04,.9255600E-04,.1190100E-03,.1848200E-03,&
     & .2070700E-06,.6159300E-04,.1057900E-03,.1400800E-03,.2096500E-03,&
     & .2207100E-06,.7101600E-04,.1210000E-03,.1612800E-03,.2366100E-03,&
     & .2314800E-06,.8128600E-04,.1379500E-03,.1840100E-03,.2658000E-03,&
     & .1429100E-06,.3882500E-04,.6686600E-04,.8614000E-04,.1351100E-03,&
     & .1517900E-06,.4513500E-04,.7845000E-04,.1019400E-03,.1536900E-03,&
     & .1632600E-06,.5239900E-04,.8991900E-04,.1200100E-03,.1744100E-03,&
     & .1730700E-06,.6044000E-04,.1029300E-03,.1373500E-03,.1966800E-03,&
     & .1813700E-06,.6918900E-04,.1175900E-03,.1565100E-03,.2199800E-03,&
     & .1124500E-06,.3286200E-04,.5716600E-04,.7382800E-04,.1118500E-03,&
     & .1195800E-06,.3828100E-04,.6663500E-04,.8739000E-04,.1274000E-03,&
     & .1270200E-06,.4451900E-04,.7660600E-04,.1019000E-03,.1441800E-03,&
     & .1346700E-06,.5137300E-04,.8762200E-04,.1170000E-03,.1620500E-03,&
     & .1412000E-06,.5887400E-04,.1001900E-03,.1334800E-03,.1818800E-03,&
     & .8824200E-07,.2795000E-04,.4901000E-04,.6343400E-04,.9287700E-04,&
     & .9438300E-07,.3261000E-04,.5669000E-04,.7511400E-04,.1053100E-03,&
     & .9971200E-07,.3792900E-04,.6520400E-04,.8711300E-04,.1193100E-03,&
     & .1047300E-06,.4377800E-04,.7480500E-04,.9995000E-04,.1342700E-03,&
     & .1098800E-06,.5022900E-04,.8559800E-04,.1140600E-03,.1505000E-03,&
     & .6969600E-07,.2389500E-04,.4216600E-04,.5473300E-04,.7714600E-04,&
     & .7439000E-07,.2792100E-04,.4838700E-04,.6481700E-04,.8760700E-04,&
     & .7885600E-07,.3249700E-04,.5569200E-04,.7469100E-04,.9938700E-04,&
     & .8325300E-07,.3749100E-04,.6413000E-04,.8560900E-04,.1119300E-03,&
     & .8767500E-07,.4301100E-04,.7339800E-04,.9782800E-04,.1257400E-03/

      data   absb(351:525, 2)   /                                       &
     & .5461900E-07,.2046900E-04,.3586900E-04,.4725900E-04,.6415100E-04,&
     & .5865700E-07,.2392500E-04,.4146800E-04,.5549900E-04,.7291100E-04,&
     & .6253200E-07,.2783900E-04,.4773500E-04,.6398500E-04,.8295500E-04,&
     & .6618400E-07,.3214100E-04,.5500100E-04,.7345600E-04,.9356700E-04,&
     & .6930500E-07,.3660500E-04,.6296800E-04,.8393800E-04,.1051700E-03,&
     & .4334800E-07,.1747600E-04,.3069200E-04,.4080300E-04,.5327100E-04,&
     & .4634800E-07,.2049600E-04,.3549800E-04,.4762900E-04,.6080900E-04,&
     & .4950700E-07,.2385700E-04,.4097400E-04,.5494700E-04,.6922200E-04,&
     & .5241900E-07,.2756000E-04,.4720500E-04,.6304400E-04,.7812300E-04,&
     & .5484200E-07,.3109400E-04,.5402800E-04,.7211600E-04,.8783400E-04,&
     & .3448000E-07,.1502700E-04,.2626800E-04,.3535800E-04,.4471100E-04,&
     & .3698500E-07,.1763800E-04,.3040500E-04,.4093800E-04,.5105400E-04,&
     & .3947600E-07,.2051200E-04,.3523700E-04,.4720700E-04,.5803400E-04,&
     & .4165000E-07,.2369400E-04,.4058600E-04,.5428600E-04,.6570800E-04,&
     & .4369300E-07,.2643800E-04,.4646700E-04,.6203500E-04,.7387200E-04,&
     & .2740200E-07,.1293200E-04,.2258500E-04,.3036500E-04,.3747100E-04,&
     & .2937700E-07,.1518300E-04,.2615800E-04,.3519500E-04,.4286700E-04,&
     & .3135300E-07,.1765800E-04,.3034300E-04,.4060600E-04,.4885800E-04,&
     & .3305600E-07,.2028600E-04,.3492400E-04,.4673500E-04,.5532900E-04,&
     & .3467300E-07,.2245100E-04,.3971700E-04,.5332900E-04,.6215100E-04,&
     & .2171600E-07,.1115600E-04,.1942700E-04,.2616900E-04,.3151200E-04,&
     & .2331300E-07,.1309100E-04,.2256800E-04,.3037700E-04,.3621600E-04,&
     & .2480300E-07,.1522100E-04,.2615700E-04,.3505000E-04,.4125600E-04,&
     & .2626000E-07,.1727700E-04,.3009100E-04,.4028800E-04,.4669300E-04,&
     & .2733400E-07,.1915900E-04,.3377300E-04,.4589200E-04,.5231400E-04,&
     & .1712300E-07,.9645300E-05,.1668800E-04,.2252800E-04,.2664800E-04,&
     & .1848900E-07,.1130500E-04,.1946400E-04,.2613700E-04,.3061700E-04,&
     & .1962100E-07,.1314300E-04,.2255400E-04,.3023800E-04,.3487600E-04,&
     & .2065300E-07,.1469800E-04,.2592800E-04,.3471200E-04,.3939800E-04,&
     & .2150000E-07,.1631500E-04,.2865100E-04,.3930200E-04,.4409700E-04,&
     & .1341400E-07,.8335500E-05,.1440900E-04,.1946100E-04,.2258200E-04,&
     & .1445600E-07,.9764700E-05,.1681800E-04,.2257800E-04,.2592300E-04,&
     & .1541600E-07,.1125000E-04,.1945100E-04,.2611900E-04,.2948200E-04,&
     & .1618400E-07,.1252000E-04,.2206400E-04,.2991100E-04,.3323000E-04,&
     & .1686100E-07,.1381000E-04,.2433200E-04,.3336300E-04,.3713500E-04/

      data   absb(526:700, 2)   /                                       &
     & .1057500E-07,.7178600E-05,.1240700E-04,.1671900E-04,.1915800E-04,&
     & .1132800E-07,.8409300E-05,.1445700E-04,.1943300E-04,.2194000E-04,&
     & .1204900E-07,.9586000E-05,.1671800E-04,.2242500E-04,.2493200E-04,&
     & .1268900E-07,.1065500E-04,.1877100E-04,.2564700E-04,.2806500E-04,&
     & .1324100E-07,.1170300E-04,.2062800E-04,.2818400E-04,.3132300E-04,&
     & .8248900E-08,.6085000E-05,.1050500E-04,.1414000E-04,.1600400E-04,&
     & .8853200E-08,.7130600E-05,.1225000E-04,.1646300E-04,.1833300E-04,&
     & .9346800E-08,.8026900E-05,.1415800E-04,.1898900E-04,.2083500E-04,&
     & .9847800E-08,.8945000E-05,.1569100E-04,.2157600E-04,.2342800E-04,&
     & .1033200E-07,.9835400E-05,.1729300E-04,.2361800E-04,.2612800E-04,&
     & .6395100E-08,.5054500E-05,.8737800E-05,.1177800E-04,.1317600E-04,&
     & .6825200E-08,.5934800E-05,.1019300E-04,.1373000E-04,.1510400E-04,&
     & .7244500E-08,.6648000E-05,.1178300E-04,.1583200E-04,.1716000E-04,&
     & .7587100E-08,.7396000E-05,.1300500E-04,.1786600E-04,.1929500E-04,&
     & .7907500E-08,.8155800E-05,.1430600E-04,.1961700E-04,.2153200E-04,&
     & .5057400E-08,.4121500E-05,.7124900E-05,.9592300E-05,.1068600E-04,&
     & .5396700E-08,.4851800E-05,.8328600E-05,.1120600E-04,.1226900E-04,&
     & .5751700E-08,.5441100E-05,.9638300E-05,.1294200E-04,.1395800E-04,&
     & .6027300E-08,.6059000E-05,.1064400E-04,.1461900E-04,.1571400E-04,&
     & .6277500E-08,.6690000E-05,.1172300E-04,.1607700E-04,.1756000E-04,&
     & .3994800E-08,.3353500E-05,.5797600E-05,.7798900E-05,.8654800E-05,&
     & .4264200E-08,.3958400E-05,.6793200E-05,.9129000E-05,.9950700E-05,&
     & .4552900E-08,.4447800E-05,.7874600E-05,.1056300E-04,.1133800E-04,&
     & .4791100E-08,.4959200E-05,.8701900E-05,.1195000E-04,.1278300E-04,&
     & .4995900E-08,.5482000E-05,.9596400E-05,.1315500E-04,.1430800E-04,&
     & .3160700E-08,.2726000E-05,.4714300E-05,.6338100E-05,.7009400E-05,&
     & .3377100E-08,.3226300E-05,.5535200E-05,.7432200E-05,.8069500E-05,&
     & .3590700E-08,.3631600E-05,.6425600E-05,.8613300E-05,.9201900E-05,&
     & .3792000E-08,.4055500E-05,.7112800E-05,.9767700E-05,.1039200E-04,&
     & .3968700E-08,.4488200E-05,.7849900E-05,.1075300E-04,.1164700E-04,&
     & .2513700E-08,.2178200E-05,.3770000E-05,.5063600E-05,.5587200E-05,&
     & .2695300E-08,.2587200E-05,.4441100E-05,.5953800E-05,.6469300E-05,&
     & .2873800E-08,.2941500E-05,.5169700E-05,.6921700E-05,.7380600E-05,&
     & .3036500E-08,.3288700E-05,.5757400E-05,.7911500E-05,.8354700E-05,&
     & .3182700E-08,.3639100E-05,.6373000E-05,.8711400E-05,.9385400E-05/

      data   absb(701:875, 2)   /                                       &
     & .2004900E-08,.1735000E-05,.3008900E-05,.4034200E-05,.4443700E-05,&
     & .2152200E-08,.2068500E-05,.3553200E-05,.4754200E-05,.5157700E-05,&
     & .2297800E-08,.2371600E-05,.4149000E-05,.5545900E-05,.5902900E-05,&
     & .2427700E-08,.2660800E-05,.4658500E-05,.6392800E-05,.6701500E-05,&
     & .2559200E-08,.2943400E-05,.5161400E-05,.7037600E-05,.7547200E-05,&
     & .1600700E-08,.1379300E-05,.2393900E-05,.3220300E-05,.3534700E-05,&
     & .1718200E-08,.1650900E-05,.2838800E-05,.3795000E-05,.4106900E-05,&
     & .1836400E-08,.1905800E-05,.3325000E-05,.4438600E-05,.4716900E-05,&
     & .1945900E-08,.2143600E-05,.3767600E-05,.5138900E-05,.5371000E-05,&
     & .2055500E-08,.2378100E-05,.4172200E-05,.5691500E-05,.6061300E-05,&
     & .1276300E-08,.1085500E-05,.1884200E-05,.2535500E-05,.2789400E-05,&
     & .1369300E-08,.1303800E-05,.2243400E-05,.2999100E-05,.3245200E-05,&
     & .1466900E-08,.1522700E-05,.2639400E-05,.3519500E-05,.3740300E-05,&
     & .1562100E-08,.1714500E-05,.3018800E-05,.4088100E-05,.4275700E-05,&
     & .1646700E-08,.1909300E-05,.3348100E-05,.4576800E-05,.4835700E-05,&
     & .1016400E-08,.8490800E-06,.1474100E-05,.1986400E-05,.2174400E-05,&
     & .1092600E-08,.1023200E-05,.1764200E-05,.2353600E-05,.2556200E-05,&
     & .1171700E-08,.1215600E-05,.2083600E-05,.2775000E-05,.2954700E-05,&
     & .1250200E-08,.1367100E-05,.2414900E-05,.3237400E-05,.3385600E-05,&
     & .1320400E-08,.1529000E-05,.2680100E-05,.3663100E-05,.3843000E-05,&
     & .8096700E-09,.6624700E-06,.1152500E-05,.1555700E-05,.1691100E-05,&
     & .8709100E-09,.8011400E-06,.1383800E-05,.1845700E-05,.2007100E-05,&
     & .9347100E-09,.9606300E-06,.1641400E-05,.2186200E-05,.2335700E-05,&
     & .9999600E-09,.1089800E-05,.1928700E-05,.2558700E-05,.2676500E-05,&
     & .1059400E-08,.1225500E-05,.2143700E-05,.2923900E-05,.3049100E-05,&
     & .6443800E-09,.5140800E-06,.8969200E-06,.1217600E-05,.1309400E-05,&
     & .6933800E-09,.6233500E-06,.1078300E-05,.1440800E-05,.1569700E-05,&
     & .7424900E-09,.7519700E-06,.1285900E-05,.1711600E-05,.1828600E-05,&
     & .7971000E-09,.8678500E-06,.1518200E-05,.2011700E-05,.2107200E-05,&
     & .8496600E-09,.9792300E-06,.1715600E-05,.2335400E-05,.2409600E-05,&
     & .5123600E-09,.3948300E-06,.6910000E-06,.9487100E-06,.1005000E-05,&
     & .5507500E-09,.4807500E-06,.8332600E-06,.1117100E-05,.1216900E-05,&
     & .5915300E-09,.5827900E-06,.9992200E-06,.1330700E-05,.1422700E-05,&
     & .6348500E-09,.6834700E-06,.1184500E-05,.1569600E-05,.1647400E-05,&
     & .6783300E-09,.7735900E-06,.1359300E-05,.1832900E-05,.1893300E-05/

      data   absb(876:1050, 2)   /                                      &
     & .4074900E-09,.3016200E-06,.5340800E-06,.7277700E-06,.7676800E-06,&
     & .4368200E-09,.3697500E-06,.6420100E-06,.8617400E-06,.9333400E-06,&
     & .4698100E-09,.4501500E-06,.7746000E-06,.1030600E-05,.1109200E-05,&
     & .5030200E-09,.5391200E-06,.9217100E-06,.1222500E-05,.1287400E-05,&
     & .5399900E-09,.6101300E-06,.1079000E-05,.1433400E-05,.1482800E-05,&
     & .3228800E-09,.2311500E-06,.4162500E-06,.5531800E-06,.5847600E-06,&
     & .3473100E-09,.2835500E-06,.4949300E-06,.6685000E-06,.7160000E-06,&
     & .3731000E-09,.3468200E-06,.5982000E-06,.7961700E-06,.8598500E-06,&
     & .3997500E-09,.4203400E-06,.7152700E-06,.9504600E-06,.1004100E-05,&
     & .4297000E-09,.4834100E-06,.8482900E-06,.1119100E-05,.1158200E-05,&
     & .2564600E-09,.1774300E-06,.3242300E-06,.4217500E-06,.4475400E-06,&
     & .2766900E-09,.2185600E-06,.3822000E-06,.5210200E-06,.5514100E-06,&
     & .2965400E-09,.2680800E-06,.4637500E-06,.6192100E-06,.6702500E-06,&
     & .3187200E-09,.3266200E-06,.5568700E-06,.7398500E-06,.7829100E-06,&
     & .3419800E-09,.3813600E-06,.6638500E-06,.8759100E-06,.9077100E-06,&
     & .2047200E-09,.1369400E-06,.2472600E-06,.3221600E-06,.3422900E-06,&
     & .2205400E-09,.1684200E-06,.2964600E-06,.4063500E-06,.4240200E-06,&
     & .2365300E-09,.2073100E-06,.3591300E-06,.4802700E-06,.5171400E-06,&
     & .2540200E-09,.2537900E-06,.4337700E-06,.5768900E-06,.6122800E-06,&
     & .2723300E-09,.3020200E-06,.5194000E-06,.6862600E-06,.7114600E-06,&
     & .1642800E-09,.1068700E-06,.1880300E-06,.2451700E-06,.2609000E-06,&
     & .1752000E-09,.1297800E-06,.2302500E-06,.3110600E-06,.3251200E-06,&
     & .1886500E-09,.1599400E-06,.2772600E-06,.3726100E-06,.3988800E-06,&
     & .2026500E-09,.1967000E-06,.3370800E-06,.4484500E-06,.4777900E-06,&
     & .2171100E-09,.2373900E-06,.4052800E-06,.5366800E-06,.5582800E-06,&
     & .1320600E-09,.8267700E-07,.1428400E-06,.1864000E-06,.1989800E-06,&
     & .1393800E-09,.9959700E-07,.1809500E-06,.2372500E-06,.2491100E-06,&
     & .1505600E-09,.1230900E-06,.2149400E-06,.2902600E-06,.3072000E-06,&
     & .1613600E-09,.1520500E-06,.2614700E-06,.3477400E-06,.3733100E-06,&
     & .1731800E-09,.1854700E-06,.3156600E-06,.4185700E-06,.4364900E-06,&
     & .1062800E-09,.6352900E-07,.1093800E-06,.1428500E-06,.1528100E-06,&
     & .1118100E-09,.7706800E-07,.1403600E-06,.1820100E-06,.1916300E-06,&
     & .1199200E-09,.9538700E-07,.1669200E-06,.2287800E-06,.2382400E-06,&
     & .1290000E-09,.1180900E-06,.2040600E-06,.2716300E-06,.2907800E-06,&
     & .1382800E-09,.1451000E-06,.2467800E-06,.3275300E-06,.3421700E-06/

      data   absb(1051:1175, 2)   /                                     &
     & .8566400E-10,.4844400E-07,.8393200E-07,.1095000E-06,.1178500E-06,&
     & .8999900E-10,.5988500E-07,.1077600E-06,.1402300E-06,.1476300E-06,&
     & .9583600E-10,.7394100E-07,.1302300E-06,.1787600E-06,.1842600E-06,&
     & .1032300E-09,.9179100E-07,.1588200E-06,.2120200E-06,.2261100E-06,&
     & .1108100E-09,.1133600E-06,.1931600E-06,.2564700E-06,.2691800E-06,&
     & .6909300E-10,.3683200E-07,.6412300E-07,.8403000E-07,.9097400E-07,&
     & .7243300E-10,.4708400E-07,.8261300E-07,.1075800E-06,.1134600E-06,&
     & .7661600E-10,.5738800E-07,.1018200E-06,.1376800E-06,.1422400E-06,&
     & .8270900E-10,.7129400E-07,.1235500E-06,.1653000E-06,.1754900E-06,&
     & .8863600E-10,.8844500E-07,.1509400E-06,.2006700E-06,.2111800E-06,&
     & .5598000E-10,.2801800E-07,.4886700E-07,.6485700E-07,.7005500E-07,&
     & .5839700E-10,.3662000E-07,.6323200E-07,.8242800E-07,.8745700E-07,&
     & .6150900E-10,.4444600E-07,.8051200E-07,.1058000E-06,.1098800E-06,&
     & .6601500E-10,.5522200E-07,.9640100E-07,.1293500E-06,.1361400E-06,&
     & .7097600E-10,.6882100E-07,.1178000E-06,.1563200E-06,.1658600E-06,&
     & .4572200E-10,.2150600E-07,.3732600E-07,.5033300E-07,.5485500E-07,&
     & .4709000E-10,.2824900E-07,.4861200E-07,.6344200E-07,.6755000E-07,&
     & .4955000E-10,.3443000E-07,.6276900E-07,.8150900E-07,.8491700E-07,&
     & .5281700E-10,.4295800E-07,.7517300E-07,.1020400E-06,.1060300E-06,&
     & .5686400E-10,.5363700E-07,.9204400E-07,.1222900E-06,.1301100E-06,&
     & .3749600E-10,.1725500E-07,.2996700E-07,.4039400E-07,.4405900E-07,&
     & .3836700E-10,.2278200E-07,.3914400E-07,.5101900E-07,.5440400E-07,&
     & .4034600E-10,.2784200E-07,.5069200E-07,.6571600E-07,.6844000E-07,&
     & .4290800E-10,.3476100E-07,.6084600E-07,.8283400E-07,.8571300E-07,&
     & .4621300E-10,.4346700E-07,.7462600E-07,.9914400E-07,.1051800E-06/

      data   absb(  1:175, 3)   /                                       &
     & .9118000E-05,.1275400E-02,.1990000E-02,.2444000E-02,.4057600E-02,&
     & .9503700E-05,.1415000E-02,.2214500E-02,.2809400E-02,.4437500E-02,&
     & .9975700E-05,.1562600E-02,.2432000E-02,.3137500E-02,.4809700E-02,&
     & .1050800E-04,.1699700E-02,.2692600E-02,.3448300E-02,.5155400E-02,&
     & .1103300E-04,.1837300E-02,.2968400E-02,.3786100E-02,.5489600E-02,&
     & .7320900E-05,.1072700E-02,.1678700E-02,.2088100E-02,.3304100E-02,&
     & .7679500E-05,.1193700E-02,.1868300E-02,.2361600E-02,.3624600E-02,&
     & .8041700E-05,.1317700E-02,.2056700E-02,.2646500E-02,.3943800E-02,&
     & .8429600E-05,.1431700E-02,.2282600E-02,.2902200E-02,.4236800E-02,&
     & .8895000E-05,.1549500E-02,.2508400E-02,.3190100E-02,.4499200E-02,&
     & .5827500E-05,.8989500E-03,.1416100E-02,.1776000E-02,.2650800E-02,&
     & .6116600E-05,.1002500E-02,.1570600E-02,.1989300E-02,.2921900E-02,&
     & .6434000E-05,.1105500E-02,.1737500E-02,.2223700E-02,.3171800E-02,&
     & .6750100E-05,.1202800E-02,.1929100E-02,.2443300E-02,.3401500E-02,&
     & .7084300E-05,.1303400E-02,.2115200E-02,.2689200E-02,.3635000E-02,&
     & .4577600E-05,.7523300E-03,.1190600E-02,.1495700E-02,.2107600E-02,&
     & .4826200E-05,.8405100E-03,.1319200E-02,.1678000E-02,.2326700E-02,&
     & .5102400E-05,.9260300E-03,.1465300E-02,.1867100E-02,.2524200E-02,&
     & .5385600E-05,.1009300E-02,.1628800E-02,.2059900E-02,.2713400E-02,&
     & .5662500E-05,.1093900E-02,.1783400E-02,.2269300E-02,.2909200E-02,&
     & .3597800E-05,.6305500E-03,.1000100E-02,.1261800E-02,.1675900E-02,&
     & .3809000E-05,.7051900E-03,.1108000E-02,.1414200E-02,.1856300E-02,&
     & .4036500E-05,.7759300E-03,.1235200E-02,.1569200E-02,.2021500E-02,&
     & .4269200E-05,.8464800E-03,.1375300E-02,.1736100E-02,.2188200E-02,&
     & .4532500E-05,.9181500E-03,.1503100E-02,.1915600E-02,.2351100E-02,&
     & .2857700E-05,.5292300E-03,.8406000E-03,.1064100E-02,.1352400E-02,&
     & .3018900E-05,.5927500E-03,.9326400E-03,.1192500E-02,.1505100E-02,&
     & .3197100E-05,.6514000E-03,.1042200E-02,.1317400E-02,.1641300E-02,&
     & .3398100E-05,.7114400E-03,.1159400E-02,.1461300E-02,.1785100E-02,&
     & .3612700E-05,.7720000E-03,.1268500E-02,.1616400E-02,.1916600E-02,&
     & .2283200E-05,.4448900E-03,.7052300E-03,.8915000E-03,.1105400E-02,&
     & .2400500E-05,.4986000E-03,.7859600E-03,.1003300E-02,.1231000E-02,&
     & .2546500E-05,.5474600E-03,.8809600E-03,.1108400E-02,.1347900E-02,&
     & .2716700E-05,.5976000E-03,.9768500E-03,.1230500E-02,.1466800E-02,&
     & .2892700E-05,.6484400E-03,.1069500E-02,.1363200E-02,.1574600E-02/

      data   absb(176:350, 3)   /                                       &
     & .1842600E-05,.3756700E-03,.5927600E-03,.7500200E-03,.9115900E-03,&
     & .1932400E-05,.4204900E-03,.6637600E-03,.8430400E-03,.1017500E-02,&
     & .2056300E-05,.4610400E-03,.7456100E-03,.9356800E-03,.1115300E-02,&
     & .2195200E-05,.5030200E-03,.8249100E-03,.1041000E-02,.1213100E-02,&
     & .2342900E-05,.5457000E-03,.9032400E-03,.1151200E-02,.1300200E-02,&
     & .1491400E-05,.3172700E-03,.4987900E-03,.6314100E-03,.7548000E-03,&
     & .1571100E-05,.3536700E-03,.5610500E-03,.7085900E-03,.8427800E-03,&
     & .1672000E-05,.3877600E-03,.6312200E-03,.7906600E-03,.9249200E-03,&
     & .1783000E-05,.4232700E-03,.6965500E-03,.8831100E-03,.1004100E-02,&
     & .1906500E-05,.4590500E-03,.7623800E-03,.9732000E-03,.1076700E-02,&
     & .1200700E-05,.2697600E-03,.4231600E-03,.5375000E-03,.6274700E-03,&
     & .1274400E-05,.2988000E-03,.4784000E-03,.6009400E-03,.7013400E-03,&
     & .1357200E-05,.3281300E-03,.5367200E-03,.6734100E-03,.7691600E-03,&
     & .1445900E-05,.3576100E-03,.5915400E-03,.7542100E-03,.8337100E-03,&
     & .1553800E-05,.3879200E-03,.6466300E-03,.8287700E-03,.8952900E-03,&
     & .9686100E-06,.2291800E-03,.3601500E-03,.4571900E-03,.5240400E-03,&
     & .1033600E-05,.2532100E-03,.4083400E-03,.5116100E-03,.5835100E-03,&
     & .1098500E-05,.2776700E-03,.4557100E-03,.5745900E-03,.6398500E-03,&
     & .1176800E-05,.3025200E-03,.5019500E-03,.6419000E-03,.6930300E-03,&
     & .1261500E-05,.3279400E-03,.5483100E-03,.7057900E-03,.7454000E-03,&
     & .7756000E-06,.1943700E-03,.3074800E-03,.3882800E-03,.4352900E-03,&
     & .8258000E-06,.2143800E-03,.3483500E-03,.4361700E-03,.4837300E-03,&
     & .8820000E-06,.2349200E-03,.3868000E-03,.4916900E-03,.5318900E-03,&
     & .9494700E-06,.2557300E-03,.4263500E-03,.5464100E-03,.5763700E-03,&
     & .1018200E-05,.2774500E-03,.4660700E-03,.6004600E-03,.6199800E-03,&
     & .6196300E-06,.1645400E-03,.2628400E-03,.3301600E-03,.3628400E-03,&
     & .6614600E-06,.1816200E-03,.2971400E-03,.3731500E-03,.4037000E-03,&
     & .7111100E-06,.1988200E-03,.3294200E-03,.4213000E-03,.4438200E-03,&
     & .7657300E-06,.2166300E-03,.3622800E-03,.4664000E-03,.4805200E-03,&
     & .8210000E-06,.2350300E-03,.3962700E-03,.5113200E-03,.5177600E-03,&
     & .4971800E-06,.1398200E-03,.2252300E-03,.2827500E-03,.3055800E-03,&
     & .5336500E-06,.1543100E-03,.2536800E-03,.3204400E-03,.3389500E-03,&
     & .5770500E-06,.1688400E-03,.2808200E-03,.3601900E-03,.3717300E-03,&
     & .6206100E-06,.1839300E-03,.3086500E-03,.3986600E-03,.4024300E-03,&
     & .6642700E-06,.1996800E-03,.3375700E-03,.4366300E-03,.4330900E-03/

      data   absb(351:525, 3)   /                                       &
     & .3996100E-06,.1188900E-03,.1936100E-03,.2427000E-03,.2569300E-03,&
     & .4313600E-06,.1310000E-03,.2163200E-03,.2759100E-03,.2846300E-03,&
     & .4649000E-06,.1433300E-03,.2396600E-03,.3084900E-03,.3110600E-03,&
     & .5000900E-06,.1563000E-03,.2634600E-03,.3408300E-03,.3371300E-03,&
     & .5380800E-06,.1699900E-03,.2876800E-03,.3732300E-03,.3627900E-03,&
     & .3198600E-06,.1010600E-03,.1658500E-03,.2089100E-03,.2155800E-03,&
     & .3454000E-06,.1111900E-03,.1850000E-03,.2376000E-03,.2385400E-03,&
     & .3722600E-06,.1218300E-03,.2045600E-03,.2641900E-03,.2605300E-03,&
     & .4031700E-06,.1329000E-03,.2248700E-03,.2915600E-03,.2826200E-03,&
     & .4327100E-06,.1448200E-03,.2454100E-03,.3193400E-03,.3041900E-03,&
     & .2567700E-06,.8606000E-04,.1420600E-03,.1802300E-03,.1814100E-03,&
     & .2773600E-06,.9467700E-04,.1584100E-03,.2037800E-03,.2008800E-03,&
     & .3010400E-06,.1037200E-03,.1751500E-03,.2266300E-03,.2192100E-03,&
     & .3258400E-06,.1134000E-03,.1923200E-03,.2495300E-03,.2378900E-03,&
     & .3489100E-06,.1236800E-03,.2097600E-03,.2736400E-03,.2557000E-03,&
     & .2055600E-06,.7327400E-04,.1217000E-03,.1558800E-03,.1531200E-03,&
     & .2237600E-06,.8065100E-04,.1356800E-03,.1752400E-03,.1694800E-03,&
     & .2431300E-06,.8848100E-04,.1499200E-03,.1944700E-03,.1847700E-03,&
     & .2623300E-06,.9682400E-04,.1645800E-03,.2140900E-03,.2002700E-03,&
     & .2824900E-06,.1058000E-03,.1798700E-03,.2343500E-03,.2153200E-03,&
     & .1655300E-06,.6253600E-04,.1045700E-03,.1344300E-03,.1296800E-03,&
     & .1807800E-06,.6886300E-04,.1164300E-03,.1506000E-03,.1430400E-03,&
     & .1961900E-06,.7567000E-04,.1286400E-03,.1668100E-03,.1561500E-03,&
     & .2115800E-06,.8289900E-04,.1411400E-03,.1838500E-03,.1691100E-03,&
     & .2291500E-06,.9062900E-04,.1545600E-03,.2009900E-03,.1814600E-03,&
     & .1338000E-06,.5340700E-04,.8993000E-04,.1159100E-03,.1100500E-03,&
     & .1460400E-06,.5888700E-04,.1000100E-03,.1296300E-03,.1210700E-03,&
     & .1584800E-06,.6479000E-04,.1104000E-03,.1434900E-03,.1323000E-03,&
     & .1719000E-06,.7116900E-04,.1212500E-03,.1579200E-03,.1429700E-03,&
     & .1858700E-06,.7779200E-04,.1329200E-03,.1726500E-03,.1530800E-03,&
     & .1083000E-06,.4568600E-04,.7733900E-04,.9986700E-04,.9356200E-04,&
     & .1176500E-06,.5052600E-04,.8595900E-04,.1115300E-03,.1026900E-03,&
     & .1282400E-06,.5562700E-04,.9490000E-04,.1235400E-03,.1120500E-03,&
     & .1395800E-06,.6115800E-04,.1044000E-03,.1357300E-03,.1209900E-03,&
     & .1507800E-06,.6690900E-04,.1143100E-03,.1486100E-03,.1291800E-03/

      data   absb(526:700, 3)   /                                       &
     & .8753700E-07,.3907300E-04,.6633000E-04,.8586000E-04,.7929600E-04,&
     & .9527000E-07,.4328200E-04,.7374300E-04,.9573300E-04,.8708200E-04,&
     & .1042300E-06,.4770100E-04,.8144700E-04,.1060900E-03,.9485700E-04,&
     & .1134800E-06,.5248400E-04,.8962400E-04,.1163800E-03,.1021100E-03,&
     & .1228900E-06,.5743100E-04,.9812200E-04,.1276000E-03,.1086600E-03,&
     & .6989100E-07,.3312900E-04,.5632300E-04,.7303600E-04,.6656800E-04,&
     & .7646500E-07,.3672400E-04,.6262700E-04,.8152000E-04,.7317600E-04,&
     & .8383300E-07,.4061900E-04,.6929300E-04,.9016100E-04,.7954800E-04,&
     & .9143600E-07,.4467100E-04,.7636200E-04,.9907500E-04,.8549700E-04,&
     & .9926300E-07,.4888600E-04,.8355000E-04,.1084900E-03,.9072000E-04,&
     & .5508000E-07,.2779600E-04,.4726100E-04,.6129900E-04,.5523700E-04,&
     & .6056200E-07,.3085000E-04,.5263500E-04,.6851500E-04,.6079400E-04,&
     & .6644000E-07,.3420000E-04,.5829800E-04,.7584700E-04,.6609700E-04,&
     & .7291300E-07,.3765000E-04,.6431000E-04,.8353000E-04,.7103200E-04,&
     & .7954500E-07,.4121600E-04,.7039700E-04,.9130400E-04,.7519300E-04,&
     & .4364400E-07,.2299700E-04,.3908800E-04,.5071400E-04,.4537400E-04,&
     & .4818200E-07,.2558500E-04,.4366100E-04,.5681300E-04,.5005700E-04,&
     & .5306700E-07,.2842900E-04,.4846500E-04,.6302000E-04,.5449000E-04,&
     & .5834900E-07,.3136500E-04,.5360300E-04,.6951600E-04,.5858500E-04,&
     & .6372700E-07,.3439800E-04,.5871000E-04,.7603200E-04,.6204600E-04,&
     & .3466100E-07,.1900300E-04,.3229500E-04,.4189800E-04,.3720700E-04,&
     & .3841400E-07,.2120000E-04,.3617000E-04,.4704500E-04,.4115000E-04,&
     & .4237800E-07,.2360500E-04,.4024700E-04,.5230600E-04,.4487200E-04,&
     & .4659800E-07,.2611000E-04,.4462000E-04,.5780000E-04,.4829300E-04,&
     & .5099500E-07,.2868300E-04,.4893500E-04,.6325000E-04,.5115500E-04,&
     & .2754500E-07,.1570000E-04,.2666100E-04,.3460300E-04,.3048700E-04,&
     & .3055300E-07,.1756800E-04,.2994800E-04,.3893500E-04,.3381300E-04,&
     & .3377200E-07,.1960100E-04,.3341100E-04,.4340300E-04,.3692200E-04,&
     & .3716400E-07,.2172600E-04,.3711300E-04,.4803000E-04,.3977100E-04,&
     & .4079500E-07,.2390100E-04,.4076500E-04,.5260700E-04,.4215200E-04,&
     & .2185200E-07,.1281900E-04,.2175700E-04,.2822100E-04,.2474500E-04,&
     & .2420700E-07,.1439500E-04,.2452700E-04,.3186700E-04,.2750200E-04,&
     & .2680600E-07,.1609200E-04,.2746200E-04,.3564600E-04,.3014500E-04,&
     & .2961200E-07,.1789400E-04,.3057200E-04,.3952700E-04,.3254300E-04,&
     & .3257700E-07,.1974600E-04,.3367200E-04,.4339900E-04,.3455600E-04/

      data   absb(701:875, 3)   /                                       &
     & .1733600E-07,.1044100E-04,.1769500E-04,.2295400E-04,.2004200E-04,&
     & .1918700E-07,.1177100E-04,.2002500E-04,.2602900E-04,.2234600E-04,&
     & .2131100E-07,.1319000E-04,.2251600E-04,.2920100E-04,.2457700E-04,&
     & .2359700E-07,.1471200E-04,.2512200E-04,.3246600E-04,.2658400E-04,&
     & .2595800E-07,.1628000E-04,.2774400E-04,.3576200E-04,.2829500E-04,&
     & .1375200E-07,.8497000E-05,.1437900E-04,.1864000E-04,.1620500E-04,&
     & .1524000E-07,.9608200E-05,.1632500E-04,.2122500E-04,.1812400E-04,&
     & .1692400E-07,.1080200E-04,.1843000E-04,.2389600E-04,.2002400E-04,&
     & .1876700E-07,.1208700E-04,.2060900E-04,.2665100E-04,.2169800E-04,&
     & .2066600E-07,.1340900E-04,.2283700E-04,.2942700E-04,.2315300E-04,&
     & .1082700E-07,.6862400E-05,.1158600E-04,.1499900E-04,.1297500E-04,&
     & .1203000E-07,.7784700E-05,.1320500E-04,.1716400E-04,.1460500E-04,&
     & .1335200E-07,.8777900E-05,.1497400E-04,.1941000E-04,.1623100E-04,&
     & .1482900E-07,.9862000E-05,.1679500E-04,.2175100E-04,.1763400E-04,&
     & .1639600E-07,.1097500E-04,.1868200E-04,.2406600E-04,.1887900E-04,&
     & .8573100E-08,.5513200E-05,.9286200E-05,.1200000E-04,.1034700E-04,&
     & .9503300E-08,.6281600E-05,.1062900E-04,.1381700E-04,.1171200E-04,&
     & .1053700E-07,.7096400E-05,.1210800E-04,.1570000E-04,.1309500E-04,&
     & .1172200E-07,.8012000E-05,.1362100E-04,.1767900E-04,.1429600E-04,&
     & .1299100E-07,.8945700E-05,.1522000E-04,.1961400E-04,.1535000E-04,&
     & .6751500E-08,.4421200E-05,.7424400E-05,.9577600E-05,.8242900E-05,&
     & .7499900E-08,.5060300E-05,.8536000E-05,.1109500E-04,.9379100E-05,&
     & .8318200E-08,.5741100E-05,.9768600E-05,.1266700E-04,.1052800E-04,&
     & .9252600E-08,.6494200E-05,.1102800E-04,.1432800E-04,.1156600E-04,&
     & .1026900E-07,.7276900E-05,.1237500E-04,.1596400E-04,.1247200E-04,&
     & .5303100E-08,.3530100E-05,.5905000E-05,.7596500E-05,.6545200E-05,&
     & .5890700E-08,.4056100E-05,.6823300E-05,.8862100E-05,.7477100E-05,&
     & .6542200E-08,.4627000E-05,.7839300E-05,.1016900E-04,.8437000E-05,&
     & .7266900E-08,.5243300E-05,.8906100E-05,.1156300E-04,.9326800E-05,&
     & .8088000E-08,.5896700E-05,.1001600E-04,.1293000E-04,.1010600E-04,&
     & .4145900E-08,.2796500E-05,.4652900E-05,.5964800E-05,.5155200E-05,&
     & .4632400E-08,.3225300E-05,.5410000E-05,.7009400E-05,.5911000E-05,&
     & .5143300E-08,.3701400E-05,.6239500E-05,.8097600E-05,.6710000E-05,&
     & .5704500E-08,.4209100E-05,.7138100E-05,.9253700E-05,.7489600E-05,&
     & .6352600E-08,.4754100E-05,.8057300E-05,.1042800E-04,.8154000E-05/

      data   absb(876:1050, 3)   /                                      &
     & .3254900E-08,.2208900E-05,.3657100E-05,.4666300E-05,.4038700E-05,&
     & .3619200E-08,.2558100E-05,.4275200E-05,.5526500E-05,.4658200E-05,&
     & .4021500E-08,.2951500E-05,.4955300E-05,.6432700E-05,.5323600E-05,&
     & .4473600E-08,.3369000E-05,.5701500E-05,.7385900E-05,.5996000E-05,&
     & .4976000E-08,.3827600E-05,.6459600E-05,.8384400E-05,.6564700E-05,&
     & .2543200E-08,.1739000E-05,.2860700E-05,.3643800E-05,.3142500E-05,&
     & .2836400E-08,.2021900E-05,.3369800E-05,.4341200E-05,.3665400E-05,&
     & .3160100E-08,.2347200E-05,.3923200E-05,.5096400E-05,.4214200E-05,&
     & .3510100E-08,.2694800E-05,.4542900E-05,.5879700E-05,.4771500E-05,&
     & .3898900E-08,.3071800E-05,.5173100E-05,.6719000E-05,.5272000E-05,&
     & .2009300E-08,.1370400E-05,.2237300E-05,.2837500E-05,.2455000E-05,&
     & .2227000E-08,.1602000E-05,.2662600E-05,.3414400E-05,.2896300E-05,&
     & .2479700E-08,.1868600E-05,.3113100E-05,.4034800E-05,.3339800E-05,&
     & .2761400E-08,.2159300E-05,.3619900E-05,.4685600E-05,.3801900E-05,&
     & .3067400E-08,.2469900E-05,.4151700E-05,.5384400E-05,.4232200E-05,&
     & .1590700E-08,.1080300E-05,.1751800E-05,.2206400E-05,.1925400E-05,&
     & .1750900E-08,.1269100E-05,.2100000E-05,.2679400E-05,.2283200E-05,&
     & .1950900E-08,.1486100E-05,.2472600E-05,.3195700E-05,.2641800E-05,&
     & .2170300E-08,.1728400E-05,.2883800E-05,.3732500E-05,.3025700E-05,&
     & .2412400E-08,.1984400E-05,.3330500E-05,.4312800E-05,.3401500E-05,&
     & .1256400E-08,.8466900E-06,.1371700E-05,.1714700E-05,.1494700E-05,&
     & .1379400E-08,.1002400E-05,.1652300E-05,.2099300E-05,.1787000E-05,&
     & .1532000E-08,.1179000E-05,.1957700E-05,.2522000E-05,.2084500E-05,&
     & .1713000E-08,.1379100E-05,.2295200E-05,.2967700E-05,.2404000E-05,&
     & .1904100E-08,.1591400E-05,.2663500E-05,.3443900E-05,.2724700E-05,&
     & .9952300E-09,.6621900E-06,.1063200E-05,.1334100E-05,.1155000E-05,&
     & .1090300E-08,.7902200E-06,.1291400E-05,.1640100E-05,.1392900E-05,&
     & .1208200E-08,.9326600E-06,.1544500E-05,.1984100E-05,.1645200E-05,&
     & .1343700E-08,.1096900E-05,.1821200E-05,.2351700E-05,.1902900E-05,&
     & .1497200E-08,.1273400E-05,.2127600E-05,.2743100E-05,.2172200E-05,&
     & .7878200E-09,.5199300E-06,.8275200E-06,.1042100E-05,.8863200E-06,&
     & .8664200E-09,.6244000E-06,.1013800E-05,.1282000E-05,.1095800E-05,&
     & .9514100E-09,.7405100E-06,.1222100E-05,.1562600E-05,.1301400E-05,&
     & .1062200E-08,.8743800E-06,.1448100E-05,.1867500E-05,.1513800E-05,&
     & .1182900E-08,.1021700E-05,.1702500E-05,.2190000E-05,.1734900E-05/

      data   absb(1051:1175, 3)   /                                     &
     & .6273400E-09,.4084400E-06,.6467500E-06,.8116800E-06,.6801500E-06,&
     & .6867000E-09,.4943700E-06,.7982200E-06,.1003100E-05,.8595500E-06,&
     & .7555100E-09,.5881300E-06,.9664000E-06,.1230500E-05,.1029000E-05,&
     & .8389600E-09,.6976200E-06,.1152900E-05,.1484500E-05,.1202500E-05,&
     & .9372700E-09,.8199200E-06,.1361800E-05,.1748900E-05,.1385600E-05,&
     & .5021400E-09,.3202800E-06,.5052400E-06,.6314700E-06,.5190300E-06,&
     & .5477500E-09,.3884100E-06,.6258900E-06,.7844500E-06,.6709900E-06,&
     & .5997100E-09,.4660200E-06,.7635600E-06,.9697800E-06,.8091200E-06,&
     & .6646600E-09,.5550900E-06,.9153300E-06,.1176600E-05,.9528400E-06,&
     & .7393100E-09,.6562500E-06,.1086400E-05,.1394600E-05,.1105500E-05,&
     & .4017100E-09,.2500900E-06,.3921100E-06,.4925000E-06,.3933200E-06,&
     & .4337900E-09,.3047300E-06,.4893200E-06,.6134200E-06,.5207600E-06,&
     & .4772900E-09,.3687400E-06,.5997800E-06,.7624600E-06,.6355600E-06,&
     & .5245400E-09,.4407500E-06,.7248800E-06,.9307900E-06,.7544800E-06,&
     & .5854600E-09,.5238900E-06,.8656500E-06,.1110700E-05,.8785800E-06,&
     & .3222200E-09,.1953600E-06,.3048400E-06,.3812300E-06,.2979800E-06,&
     & .3455000E-09,.2400300E-06,.3820600E-06,.4803300E-06,.4019300E-06,&
     & .3793000E-09,.2917900E-06,.4722600E-06,.5993800E-06,.5005600E-06,&
     & .4167500E-09,.3503500E-06,.5750000E-06,.7364200E-06,.5983600E-06,&
     & .4632000E-09,.4185400E-06,.6904100E-06,.8854800E-06,.7000300E-06,&
     & .2616600E-09,.1586800E-06,.2469700E-06,.3089200E-06,.2396400E-06,&
     & .2809500E-09,.1955200E-06,.3106800E-06,.3903700E-06,.3246900E-06,&
     & .3073000E-09,.2385600E-06,.3855400E-06,.4890200E-06,.4073800E-06,&
     & .3378100E-09,.2874900E-06,.4715700E-06,.6028600E-06,.4880300E-06,&
     & .3752600E-09,.3443900E-06,.5678900E-06,.7274200E-06,.5723800E-06/

      data   absb(  1:175, 4)   /                                       &
     & .4903800E-04,.5078100E-02,.7825400E-02,.9648400E-02,.1061400E-01,&
     & .5422500E-04,.5490500E-02,.8571000E-02,.1046000E-01,.1120600E-01,&
     & .6113400E-04,.5877300E-02,.9318300E-02,.1134700E-01,.1172900E-01,&
     & .6879300E-04,.6279400E-02,.9976800E-02,.1224800E-01,.1225000E-01,&
     & .7742900E-04,.6689300E-02,.1062400E-01,.1306400E-01,.1273000E-01,&
     & .4070600E-04,.4304500E-02,.6624000E-02,.8122700E-02,.8710500E-02,&
     & .4578900E-04,.4640400E-02,.7254700E-02,.8850800E-02,.9218300E-02,&
     & .5135900E-04,.4967000E-02,.7864900E-02,.9592800E-02,.9659000E-02,&
     & .5844800E-04,.5311200E-02,.8416500E-02,.1035600E-01,.1008500E-01,&
     & .6658500E-04,.5659000E-02,.8977000E-02,.1102200E-01,.1054000E-01,&
     & .3316100E-04,.3637400E-02,.5600700E-02,.6840800E-02,.7039900E-02,&
     & .3731600E-04,.3914500E-02,.6131800E-02,.7481200E-02,.7453300E-02,&
     & .4218400E-04,.4194900E-02,.6627200E-02,.8109700E-02,.7839300E-02,&
     & .4824900E-04,.4486100E-02,.7093400E-02,.8733000E-02,.8226200E-02,&
     & .5491200E-04,.4782900E-02,.7568800E-02,.9283000E-02,.8605800E-02,&
     & .2659800E-04,.3064900E-02,.4732300E-02,.5774200E-02,.5664400E-02,&
     & .3009600E-04,.3295000E-02,.5177800E-02,.6314500E-02,.6019300E-02,&
     & .3408500E-04,.3535000E-02,.5580600E-02,.6844700E-02,.6370800E-02,&
     & .3891900E-04,.3783100E-02,.5971900E-02,.7346500E-02,.6708800E-02,&
     & .4449200E-04,.4039300E-02,.6375000E-02,.7812000E-02,.7024900E-02,&
     & .2144900E-04,.2575600E-02,.3993600E-02,.4866200E-02,.4620500E-02,&
     & .2410300E-04,.2770600E-02,.4361900E-02,.5316400E-02,.4920200E-02,&
     & .2745100E-04,.2976500E-02,.4695900E-02,.5763200E-02,.5212200E-02,&
     & .3138100E-04,.3188400E-02,.5022200E-02,.6173500E-02,.5479000E-02,&
     & .3595300E-04,.3409900E-02,.5364500E-02,.6565900E-02,.5741900E-02,&
     & .1746800E-04,.2164200E-02,.3364200E-02,.4097200E-02,.3792100E-02,&
     & .1950600E-04,.2330800E-02,.3668300E-02,.4469600E-02,.4038500E-02,&
     & .2232600E-04,.2507200E-02,.3945400E-02,.4850700E-02,.4277300E-02,&
     & .2555800E-04,.2690100E-02,.4224000E-02,.5185500E-02,.4500000E-02,&
     & .2951900E-04,.2881600E-02,.4513500E-02,.5516800E-02,.4725900E-02,&
     & .1416200E-04,.1816500E-02,.2831100E-02,.3449100E-02,.3126300E-02,&
     & .1588000E-04,.1959400E-02,.3079200E-02,.3757700E-02,.3327700E-02,&
     & .1825000E-04,.2112400E-02,.3311000E-02,.4071700E-02,.3525500E-02,&
     & .2094800E-04,.2271100E-02,.3551200E-02,.4354100E-02,.3711400E-02,&
     & .2427900E-04,.2436000E-02,.3797200E-02,.4633600E-02,.3906300E-02/

      data   absb(176:350, 4)   /                                       &
     & .1167700E-04,.1526800E-02,.2385400E-02,.2905400E-02,.2592700E-02,&
     & .1312700E-04,.1650200E-02,.2588400E-02,.3164200E-02,.2758400E-02,&
     & .1514500E-04,.1782700E-02,.2785400E-02,.3419800E-02,.2922800E-02,&
     & .1742600E-04,.1920800E-02,.2991900E-02,.3656300E-02,.3083600E-02,&
     & .2018700E-04,.2063800E-02,.3202700E-02,.3895500E-02,.3251700E-02,&
     & .9624800E-05,.1283900E-02,.2006900E-02,.2443900E-02,.2152100E-02,&
     & .1088400E-04,.1390500E-02,.2175300E-02,.2663400E-02,.2290100E-02,&
     & .1257600E-04,.1505700E-02,.2343000E-02,.2871400E-02,.2427100E-02,&
     & .1449200E-04,.1624500E-02,.2522900E-02,.3070900E-02,.2565300E-02,&
     & .1678500E-04,.1748700E-02,.2705300E-02,.3276900E-02,.2704100E-02,&
     & .7898700E-05,.1084100E-02,.1696300E-02,.2064600E-02,.1792000E-02,&
     & .9068800E-05,.1178300E-02,.1836400E-02,.2252300E-02,.1905400E-02,&
     & .1043600E-04,.1277600E-02,.1983200E-02,.2422000E-02,.2020000E-02,&
     & .1205100E-04,.1380000E-02,.2139100E-02,.2590400E-02,.2136400E-02,&
     & .1399200E-04,.1487700E-02,.2298300E-02,.2769000E-02,.2251900E-02,&
     & .6514300E-05,.9162900E-03,.1433700E-02,.1746800E-02,.1489000E-02,&
     & .7529500E-05,.9988000E-03,.1553200E-02,.1901700E-02,.1584400E-02,&
     & .8617200E-05,.1084300E-02,.1682800E-02,.2045400E-02,.1682200E-02,&
     & .1001700E-04,.1172900E-02,.1817600E-02,.2191200E-02,.1780000E-02,&
     & .1164000E-04,.1266800E-02,.1954900E-02,.2344700E-02,.1876500E-02,&
     & .5343200E-05,.7757100E-03,.1211400E-02,.1480500E-02,.1237200E-02,&
     & .6151500E-05,.8471800E-03,.1315300E-02,.1607800E-02,.1318100E-02,&
     & .7092300E-05,.9202400E-03,.1429300E-02,.1730000E-02,.1398900E-02,&
     & .8268400E-05,.9974000E-03,.1544700E-02,.1857300E-02,.1481100E-02,&
     & .9626800E-05,.1078900E-02,.1662700E-02,.1989200E-02,.1562100E-02,&
     & .4393200E-05,.6589800E-03,.1025900E-02,.1256800E-02,.1028700E-02,&
     & .5048300E-05,.7197300E-03,.1117400E-02,.1361000E-02,.1097300E-02,&
     & .5852700E-05,.7826900E-03,.1215700E-02,.1465800E-02,.1164800E-02,&
     & .6825800E-05,.8496800E-03,.1315800E-02,.1577400E-02,.1235400E-02,&
     & .7978000E-05,.9209300E-03,.1417100E-02,.1691700E-02,.1302900E-02,&
     & .3634900E-05,.5613900E-03,.8723200E-03,.1067100E-02,.8574700E-03,&
     & .4176300E-05,.6130700E-03,.9528400E-03,.1155900E-02,.9164000E-03,&
     & .4864300E-05,.6676900E-03,.1037700E-02,.1248000E-02,.9741700E-03,&
     & .5680600E-05,.7261800E-03,.1123800E-02,.1344300E-02,.1034100E-02,&
     & .6657400E-05,.7879100E-03,.1211800E-02,.1442600E-02,.1090800E-02/

      data   absb(351:525, 4)   /                                       &
     & .2984900E-05,.4784500E-03,.7432400E-03,.9069500E-03,.7165400E-03,&
     & .3452800E-05,.5230500E-03,.8140700E-03,.9834800E-03,.7662200E-03,&
     & .4038600E-05,.5706700E-03,.8863500E-03,.1064300E-02,.8161700E-03,&
     & .4734900E-05,.6215900E-03,.9604500E-03,.1147600E-02,.8666700E-03,&
     & .5550400E-05,.6748600E-03,.1037300E-02,.1232700E-02,.9143700E-03,&
     & .2447400E-05,.4081900E-03,.6346900E-03,.7718400E-03,.5995900E-03,&
     & .2852700E-05,.4469400E-03,.6955000E-03,.8380900E-03,.6415100E-03,&
     & .3344400E-05,.4880400E-03,.7579700E-03,.9092100E-03,.6846400E-03,&
     & .3933200E-05,.5322900E-03,.8217400E-03,.9810300E-03,.7268000E-03,&
     & .4624500E-05,.5786400E-03,.8887200E-03,.1054400E-02,.7672300E-03,&
     & .2025300E-05,.3492100E-03,.5436500E-03,.6586500E-03,.5033100E-03,&
     & .2372000E-05,.3827800E-03,.5957700E-03,.7172900E-03,.5386400E-03,&
     & .2794500E-05,.4188700E-03,.6490700E-03,.7782300E-03,.5758100E-03,&
     & .3293600E-05,.4569900E-03,.7048100E-03,.8407400E-03,.6108900E-03,&
     & .3888100E-05,.4975700E-03,.7632500E-03,.9035600E-03,.6456400E-03,&
     & .1679700E-05,.2991800E-03,.4659800E-03,.5631600E-03,.4227000E-03,&
     & .1976400E-05,.3283900E-03,.5107600E-03,.6143900E-03,.4530600E-03,&
     & .2338400E-05,.3598500E-03,.5568600E-03,.6671200E-03,.4844500E-03,&
     & .2766200E-05,.3930200E-03,.6052500E-03,.7209200E-03,.5142500E-03,&
     & .3268700E-05,.4283700E-03,.6561000E-03,.7757000E-03,.5439600E-03,&
     & .1399100E-05,.2568700E-03,.3999600E-03,.4830700E-03,.3557000E-03,&
     & .1656400E-05,.2824600E-03,.4385800E-03,.5274300E-03,.3821100E-03,&
     & .1966000E-05,.3097100E-03,.4785600E-03,.5730900E-03,.4082600E-03,&
     & .2333400E-05,.3388100E-03,.5206600E-03,.6191900E-03,.4336700E-03,&
     & .2758700E-05,.3696400E-03,.5649900E-03,.6666000E-03,.4596900E-03,&
     & .1171100E-05,.2210200E-03,.3439400E-03,.4151200E-03,.2998000E-03,&
     & .1392800E-05,.2433700E-03,.3771300E-03,.4535100E-03,.3225300E-03,&
     & .1658900E-05,.2670400E-03,.4120200E-03,.4927200E-03,.3444200E-03,&
     & .1974000E-05,.2926100E-03,.4485500E-03,.5327400E-03,.3663300E-03,&
     & .2332300E-05,.3197300E-03,.4872600E-03,.5738300E-03,.3891500E-03,&
     & .9830000E-06,.1904700E-03,.2959200E-03,.3573900E-03,.2530300E-03,&
     & .1175100E-05,.2098300E-03,.3247400E-03,.3904300E-03,.2725900E-03,&
     & .1402700E-05,.2306500E-03,.3550500E-03,.4240500E-03,.2912000E-03,&
     & .1669200E-05,.2530200E-03,.3869900E-03,.4588800E-03,.3101800E-03,&
     & .1971600E-05,.2771500E-03,.4211900E-03,.4946800E-03,.3298900E-03/

      data   absb(526:700, 4)   /                                       &
     & .8291500E-06,.1639600E-03,.2543300E-03,.3070000E-03,.2140600E-03,&
     & .9935200E-06,.1808300E-03,.2793500E-03,.3355200E-03,.2304800E-03,&
     & .1187700E-05,.1991500E-03,.3057800E-03,.3645500E-03,.2465900E-03,&
     & .1413900E-05,.2188600E-03,.3338400E-03,.3947600E-03,.2631700E-03,&
     & .1667000E-05,.2402100E-03,.3639000E-03,.4260300E-03,.2800500E-03,&
     & .6867000E-06,.1397900E-03,.2167300E-03,.2616100E-03,.1797400E-03,&
     & .8243700E-06,.1544800E-03,.2384600E-03,.2861100E-03,.1937700E-03,&
     & .9864900E-06,.1704400E-03,.2613900E-03,.3113200E-03,.2076500E-03,&
     & .1174000E-05,.1877300E-03,.2860900E-03,.3374000E-03,.2219700E-03,&
     & .1383300E-05,.2065200E-03,.3123300E-03,.3644900E-03,.2365800E-03,&
     & .5556400E-06,.1178600E-03,.1828300E-03,.2208900E-03,.1495100E-03,&
     & .6673100E-06,.1305300E-03,.2015100E-03,.2418900E-03,.1614600E-03,&
     & .8004800E-06,.1443200E-03,.2215400E-03,.2636400E-03,.1734800E-03,&
     & .9519500E-06,.1593700E-03,.2431000E-03,.2860900E-03,.1857700E-03,&
     & .1121700E-05,.1758000E-03,.2660100E-03,.3094900E-03,.1985000E-03,&
     & .4452600E-06,.9833400E-04,.1523600E-03,.1839500E-03,.1236900E-03,&
     & .5355100E-06,.1092700E-03,.1683700E-03,.2019400E-03,.1338700E-03,&
     & .6430700E-06,.1211000E-03,.1855600E-03,.2205900E-03,.1441800E-03,&
     & .7668800E-06,.1341100E-03,.2041400E-03,.2398500E-03,.1547700E-03,&
     & .9059100E-06,.1484200E-03,.2240500E-03,.2599600E-03,.1656600E-03,&
     & .3563000E-06,.8199100E-04,.1269000E-03,.1531200E-03,.1022600E-03,&
     & .4291100E-06,.9139100E-04,.1406400E-03,.1684900E-03,.1109500E-03,&
     & .5158300E-06,.1016300E-03,.1553800E-03,.1844800E-03,.1197700E-03,&
     & .6165400E-06,.1128700E-03,.1713800E-03,.2010100E-03,.1288400E-03,&
     & .7302800E-06,.1253000E-03,.1886200E-03,.2182600E-03,.1381600E-03,&
     & .2845500E-06,.6836200E-04,.1057200E-03,.1274700E-03,.8449400E-04,&
     & .3431300E-06,.7641300E-04,.1175100E-03,.1406000E-03,.9190500E-04,&
     & .4130500E-06,.8529900E-04,.1301600E-03,.1543100E-03,.9946300E-04,&
     & .4946600E-06,.9504700E-04,.1439600E-03,.1685100E-03,.1072500E-03,&
     & .5872100E-06,.1058300E-03,.1588600E-03,.1833200E-03,.1151600E-03,&
     & .2241600E-06,.5645100E-04,.8719200E-04,.1050300E-03,.6929900E-04,&
     & .2705800E-06,.6332500E-04,.9724100E-04,.1162500E-03,.7563500E-04,&
     & .3264000E-06,.7094200E-04,.1080100E-03,.1279200E-03,.8206100E-04,&
     & .3917800E-06,.7938100E-04,.1198500E-03,.1400500E-03,.8872200E-04,&
     & .4669400E-06,.8867300E-04,.1326100E-03,.1527400E-03,.9543700E-04/

      data   absb(701:875, 4)   /                                       &
     & .1762000E-06,.4649500E-04,.7182900E-04,.8635100E-04,.5670100E-04,&
     & .2129500E-06,.5238300E-04,.8038000E-04,.9593700E-04,.6210000E-04,&
     & .2572000E-06,.5890000E-04,.8954800E-04,.1059400E-03,.6756800E-04,&
     & .3095700E-06,.6613600E-04,.9964300E-04,.1162300E-03,.7328000E-04,&
     & .3700300E-06,.7420300E-04,.1106100E-03,.1270700E-03,.7897900E-04,&
     & .1384300E-06,.3823800E-04,.5912400E-04,.7098200E-04,.4633100E-04,&
     & .1672600E-06,.4326900E-04,.6642400E-04,.7912500E-04,.5095600E-04,&
     & .2023200E-06,.4887000E-04,.7424700E-04,.8766100E-04,.5556100E-04,&
     & .2440800E-06,.5511900E-04,.8282700E-04,.9645300E-04,.6044700E-04,&
     & .2927600E-06,.6202800E-04,.9222100E-04,.1057000E-03,.6528100E-04,&
     & .1079300E-06,.3119100E-04,.4830500E-04,.5794800E-04,.3766600E-04,&
     & .1301700E-06,.3546500E-04,.5448900E-04,.6484100E-04,.4155600E-04,&
     & .1577100E-06,.4023500E-04,.6117600E-04,.7208200E-04,.4545200E-04,&
     & .1906400E-06,.4559900E-04,.6848800E-04,.7957600E-04,.4959300E-04,&
     & .2294400E-06,.5155200E-04,.7643100E-04,.8746200E-04,.5371500E-04,&
     & .8387900E-07,.2532500E-04,.3929700E-04,.4711800E-04,.3051000E-04,&
     & .1008700E-06,.2892000E-04,.4453200E-04,.5292900E-04,.3377200E-04,&
     & .1223800E-06,.3297700E-04,.5018900E-04,.5907500E-04,.3706900E-04,&
     & .1481300E-06,.3755600E-04,.5645100E-04,.6544800E-04,.4056700E-04,&
     & .1787400E-06,.4267200E-04,.6319600E-04,.7215900E-04,.4408500E-04,&
     & .6526100E-07,.2051600E-04,.3189500E-04,.3827900E-04,.2465000E-04,&
     & .7814500E-07,.2354000E-04,.3631900E-04,.4316100E-04,.2739000E-04,&
     & .9480500E-07,.2696200E-04,.4113400E-04,.4835700E-04,.3019000E-04,&
     & .1149000E-06,.3085800E-04,.4642900E-04,.5378500E-04,.3313600E-04,&
     & .1390500E-06,.3521000E-04,.5221600E-04,.5948200E-04,.3612200E-04,&
     & .5066800E-07,.1652600E-04,.2575200E-04,.3095400E-04,.1981700E-04,&
     & .6036100E-07,.1905800E-04,.2947100E-04,.3506000E-04,.2212300E-04,&
     & .7310100E-07,.2193700E-04,.3356900E-04,.3946100E-04,.2449800E-04,&
     & .8874700E-07,.2522200E-04,.3804800E-04,.4403900E-04,.2697300E-04,&
     & .1076200E-06,.2893200E-04,.4296400E-04,.4887100E-04,.2949900E-04,&
     & .3918500E-07,.1320700E-04,.2062600E-04,.2483500E-04,.1582600E-04,&
     & .4633500E-07,.1530300E-04,.2372200E-04,.2827700E-04,.1776200E-04,&
     & .5589600E-07,.1770300E-04,.2716200E-04,.3197600E-04,.1978000E-04,&
     & .6796200E-07,.2045100E-04,.3094000E-04,.3586700E-04,.2183400E-04,&
     & .8253500E-07,.2358300E-04,.3512800E-04,.3993300E-04,.2397200E-04/

      data   absb(876:1050, 4)   /                                      &
     & .3043500E-07,.1052800E-04,.1645800E-04,.1987100E-04,.1263000E-04,&
     & .3565700E-07,.1225500E-04,.1903400E-04,.2275700E-04,.1423000E-04,&
     & .4277700E-07,.1424100E-04,.2191600E-04,.2585600E-04,.1591600E-04,&
     & .5197700E-07,.1653100E-04,.2509400E-04,.2914200E-04,.1763200E-04,&
     & .6324600E-07,.1916200E-04,.2863100E-04,.3258600E-04,.1944700E-04,&
     & .2372200E-07,.8365600E-05,.1308800E-04,.1583400E-04,.1006000E-04,&
     & .2750900E-07,.9786900E-05,.1522700E-04,.1825200E-04,.1136300E-04,&
     & .3276900E-07,.1142600E-04,.1762800E-04,.2085300E-04,.1276700E-04,&
     & .3971900E-07,.1332600E-04,.2028500E-04,.2363700E-04,.1422300E-04,&
     & .4841500E-07,.1552400E-04,.2326800E-04,.2653000E-04,.1573900E-04,&
     & .1862300E-07,.6662000E-05,.1042500E-04,.1263400E-04,.8013200E-05,&
     & .2139300E-07,.7826400E-05,.1218800E-04,.1464300E-04,.9075700E-05,&
     & .2528100E-07,.9178700E-05,.1418600E-04,.1682500E-04,.1024700E-04,&
     & .3053000E-07,.1075800E-04,.1641900E-04,.1917800E-04,.1148200E-04,&
     & .3721300E-07,.1259300E-04,.1892000E-04,.2164300E-04,.1274900E-04,&
     & .1468100E-07,.5304300E-05,.8295600E-05,.1006400E-04,.6366900E-05,&
     & .1671700E-07,.6250800E-05,.9750100E-05,.1173700E-04,.7247300E-05,&
     & .1958700E-07,.7364300E-05,.1140000E-04,.1355800E-04,.8229000E-05,&
     & .2355500E-07,.8674800E-05,.1326800E-04,.1553700E-04,.9263400E-05,&
     & .2867100E-07,.1020900E-04,.1537900E-04,.1763500E-04,.1031600E-04,&
     & .1162600E-07,.4214900E-05,.6574000E-05,.7983500E-05,.5059500E-05,&
     & .1310200E-07,.4988400E-05,.7775800E-05,.9384600E-05,.5785600E-05,&
     & .1520800E-07,.5895500E-05,.9139200E-05,.1089800E-04,.6593700E-05,&
     & .1817400E-07,.6975800E-05,.1069200E-04,.1255400E-04,.7456400E-05,&
     & .2205700E-07,.8250500E-05,.1246200E-04,.1433100E-04,.8330100E-05,&
     & .9213000E-08,.3337800E-05,.5199800E-05,.6300600E-05,.4009700E-05,&
     & .1029800E-07,.3970400E-05,.6183900E-05,.7477500E-05,.4607700E-05,&
     & .1183100E-07,.4708800E-05,.7306200E-05,.8734500E-05,.5262000E-05,&
     & .1403000E-07,.5594200E-05,.8592100E-05,.1011900E-04,.5983800E-05,&
     & .1697000E-07,.6644700E-05,.1006300E-04,.1162000E-04,.6720800E-05,&
     & .7344300E-08,.2649800E-05,.4122900E-05,.4978800E-05,.3192900E-05,&
     & .8145300E-08,.3168900E-05,.4932000E-05,.5970100E-05,.3670600E-05,&
     & .9285300E-08,.3774600E-05,.5856100E-05,.7013900E-05,.4210500E-05,&
     & .1091300E-07,.4498600E-05,.6919200E-05,.8173300E-05,.4808100E-05,&
     & .1315100E-07,.5368900E-05,.8146900E-05,.9437700E-05,.5432000E-05/

      data   absb(1051:1175, 4)   /                                     &
     & .5869000E-08,.2105600E-05,.3271000E-05,.3936200E-05,.2535700E-05,&
     & .6482400E-08,.2529200E-05,.3933500E-05,.4765200E-05,.2926600E-05,&
     & .7319500E-08,.3026000E-05,.4694200E-05,.5634600E-05,.3370000E-05,&
     & .8529800E-08,.3621600E-05,.5572700E-05,.6599000E-05,.3866200E-05,&
     & .1022700E-07,.4336600E-05,.6593500E-05,.7662700E-05,.4388900E-05,&
     & .4687900E-08,.1667400E-05,.2584800E-05,.3099200E-05,.2009700E-05,&
     & .5157900E-08,.2014500E-05,.3130700E-05,.3787500E-05,.2330400E-05,&
     & .5782900E-08,.2419900E-05,.3752900E-05,.4511900E-05,.2696200E-05,&
     & .6681700E-08,.2909200E-05,.4478800E-05,.5314200E-05,.3103700E-05,&
     & .7954800E-08,.3498500E-05,.5324800E-05,.6206600E-05,.3538700E-05,&
     & .3748700E-08,.1317700E-05,.2036200E-05,.2427800E-05,.1588800E-05,&
     & .4113200E-08,.1600600E-05,.2486300E-05,.3000100E-05,.1853000E-05,&
     & .4579600E-08,.1930100E-05,.2995700E-05,.3605300E-05,.2151500E-05,&
     & .5244700E-08,.2330100E-05,.3590200E-05,.4269000E-05,.2484400E-05,&
     & .6195200E-08,.2817500E-05,.4289400E-05,.5013300E-05,.2850300E-05,&
     & .3005300E-08,.1041900E-05,.1603400E-05,.1907100E-05,.1255000E-05,&
     & .3290600E-08,.1271700E-05,.1975000E-05,.2375100E-05,.1476100E-05,&
     & .3641200E-08,.1541300E-05,.2395000E-05,.2881700E-05,.1717300E-05,&
     & .4132700E-08,.1867300E-05,.2880000E-05,.3430100E-05,.1989300E-05,&
     & .4844800E-08,.2269900E-05,.3455900E-05,.4049600E-05,.2293800E-05,&
     & .2451700E-08,.8536700E-06,.1312100E-05,.1558700E-05,.1020900E-05,&
     & .2681400E-08,.1045900E-05,.1624100E-05,.1950500E-05,.1206700E-05,&
     & .2960400E-08,.1272600E-05,.1978600E-05,.2376500E-05,.1407100E-05,&
     & .3349000E-08,.1548600E-05,.2388400E-05,.2839300E-05,.1635300E-05,&
     & .3917700E-08,.1892300E-05,.2875100E-05,.3364800E-05,.1889600E-05/

      data   absb(  1:175, 5)   /                                       &
     & .1060600E-02,.1761600E-01,.2633700E-01,.3083200E-01,.2325000E-01,&
     & .1197500E-02,.1885100E-01,.2774600E-01,.3219100E-01,.2446100E-01,&
     & .1368600E-02,.2023600E-01,.2922400E-01,.3340900E-01,.2553800E-01,&
     & .1578800E-02,.2168600E-01,.3082200E-01,.3458100E-01,.2648300E-01,&
     & .1842200E-02,.2320100E-01,.3248500E-01,.3579100E-01,.2733900E-01,&
     & .9029900E-03,.1493900E-01,.2223500E-01,.2596900E-01,.1945900E-01,&
     & .1027600E-02,.1604300E-01,.2346400E-01,.2710200E-01,.2041800E-01,&
     & .1184800E-02,.1724400E-01,.2476600E-01,.2813700E-01,.2127900E-01,&
     & .1376300E-02,.1849400E-01,.2615900E-01,.2914100E-01,.2203500E-01,&
     & .1608700E-02,.1980100E-01,.2758600E-01,.3020800E-01,.2271200E-01,&
     & .7616900E-03,.1265700E-01,.1874400E-01,.2180700E-01,.1615500E-01,&
     & .8742900E-03,.1362200E-01,.1982000E-01,.2275300E-01,.1690800E-01,&
     & .1015300E-02,.1465000E-01,.2096700E-01,.2363700E-01,.1759700E-01,&
     & .1186000E-02,.1572800E-01,.2215900E-01,.2452100E-01,.1823600E-01,&
     & .1394000E-02,.1685500E-01,.2337700E-01,.2544200E-01,.1880000E-01,&
     & .6397900E-03,.1071600E-01,.1578200E-01,.1828000E-01,.1333000E-01,&
     & .7399200E-03,.1155000E-01,.1672100E-01,.1907400E-01,.1395500E-01,&
     & .8653500E-03,.1243200E-01,.1771500E-01,.1983500E-01,.1451900E-01,&
     & .1018000E-02,.1335800E-01,.1873300E-01,.2061200E-01,.1505300E-01,&
     & .1202000E-02,.1433000E-01,.1978000E-01,.2139000E-01,.1552300E-01,&
     & .5382700E-03,.9067500E-02,.1328200E-01,.1531000E-01,.1100200E-01,&
     & .6273300E-03,.9781800E-02,.1409300E-01,.1598500E-01,.1151700E-01,&
     & .7389400E-03,.1053600E-01,.1493700E-01,.1664200E-01,.1198200E-01,&
     & .8747100E-03,.1132900E-01,.1581300E-01,.1730800E-01,.1243100E-01,&
     & .1036900E-02,.1216700E-01,.1671700E-01,.1796600E-01,.1283300E-01,&
     & .4563200E-03,.7663100E-02,.1117500E-01,.1282100E-01,.9100700E-02,&
     & .5361800E-03,.8274100E-02,.1186700E-01,.1340200E-01,.9528400E-02,&
     & .6352000E-03,.8917900E-02,.1259100E-01,.1396000E-01,.9922700E-02,&
     & .7551300E-03,.9596600E-02,.1334500E-01,.1452900E-01,.1029400E-01,&
     & .8982800E-03,.1032500E-01,.1412700E-01,.1508200E-01,.1063700E-01,&
     & .3878200E-03,.6468600E-02,.9394700E-02,.1073800E-01,.7536600E-02,&
     & .4582600E-03,.6987200E-02,.9987800E-02,.1123000E-01,.7892000E-02,&
     & .5454400E-03,.7538600E-02,.1061300E-01,.1170600E-01,.8221900E-02,&
     & .6508500E-03,.8124300E-02,.1126100E-01,.1218300E-01,.8544700E-02,&
     & .7760100E-03,.8759800E-02,.1193300E-01,.1266000E-01,.8838800E-02/

      data   absb(176:350, 5)   /                                       &
     & .3330400E-03,.5462300E-02,.7902800E-02,.8996000E-02,.6263500E-02,&
     & .3955300E-03,.5905500E-02,.8415500E-02,.9411100E-02,.6561300E-02,&
     & .4718000E-03,.6377900E-02,.8950000E-02,.9817200E-02,.6845900E-02,&
     & .5645300E-03,.6886200E-02,.9504800E-02,.1022500E-01,.7124600E-02,&
     & .6748600E-03,.7441100E-02,.1008200E-01,.1063700E-01,.7378600E-02,&
     & .2869300E-03,.4618000E-02,.6650400E-02,.7537900E-02,.5207100E-02,&
     & .3418000E-03,.4994800E-02,.7089800E-02,.7886600E-02,.5464500E-02,&
     & .4081100E-03,.5400600E-02,.7547400E-02,.8233100E-02,.5712700E-02,&
     & .4887300E-03,.5843400E-02,.8021200E-02,.8584400E-02,.5947000E-02,&
     & .5859300E-03,.6326000E-02,.8518200E-02,.8940800E-02,.6166200E-02,&
     & .2497400E-03,.3924000E-02,.5620800E-02,.6334500E-02,.4343100E-02,&
     & .2974900E-03,.4252600E-02,.5999600E-02,.6626600E-02,.4562400E-02,&
     & .3560800E-03,.4604700E-02,.6391900E-02,.6925900E-02,.4773800E-02,&
     & .4269900E-03,.4992000E-02,.6799300E-02,.7227600E-02,.4971400E-02,&
     & .5119800E-03,.5416600E-02,.7230200E-02,.7538700E-02,.5166000E-02,&
     & .2171600E-03,.3337500E-02,.4752800E-02,.5323800E-02,.3625800E-02,&
     & .2588600E-03,.3624300E-02,.5079700E-02,.5574000E-02,.3810700E-02,&
     & .3102600E-03,.3935300E-02,.5416900E-02,.5830900E-02,.3987600E-02,&
     & .3723500E-03,.4275700E-02,.5771100E-02,.6091100E-02,.4158600E-02,&
     & .4456800E-03,.4646300E-02,.6147600E-02,.6363300E-02,.4327700E-02,&
     & .1873800E-03,.2841200E-02,.4025100E-02,.4475400E-02,.3023200E-02,&
     & .2242200E-03,.3093000E-02,.4306200E-02,.4692100E-02,.3178800E-02,&
     & .2694200E-03,.3369400E-02,.4598300E-02,.4912600E-02,.3328700E-02,&
     & .3231700E-03,.3670200E-02,.4909200E-02,.5139300E-02,.3477100E-02,&
     & .3863800E-03,.3995100E-02,.5238500E-02,.5381500E-02,.3623200E-02,&
     & .1622400E-03,.2423200E-02,.3414100E-02,.3766100E-02,.2526200E-02,&
     & .1946500E-03,.2646300E-02,.3657500E-02,.3954100E-02,.2657700E-02,&
     & .2339100E-03,.2893600E-02,.3913000E-02,.4144600E-02,.2784100E-02,&
     & .2804400E-03,.3159800E-02,.4185300E-02,.4344200E-02,.2909800E-02,&
     & .3348300E-03,.3443700E-02,.4473500E-02,.4562500E-02,.3035100E-02,&
     & .1413200E-03,.2073700E-02,.2903600E-02,.3176200E-02,.2115700E-02,&
     & .1697700E-03,.2274300E-02,.3115200E-02,.3338200E-02,.2225800E-02,&
     & .2040900E-03,.2494700E-02,.3340100E-02,.3504800E-02,.2333800E-02,&
     & .2447200E-03,.2729700E-02,.3579800E-02,.3683500E-02,.2442100E-02,&
     & .2913100E-03,.2980400E-02,.3830800E-02,.3879700E-02,.2550000E-02/

      data   absb(351:525, 5)   /                                       &
     & .1229300E-03,.1779700E-02,.2472500E-02,.2682200E-02,.1771800E-02,&
     & .1478900E-03,.1959500E-02,.2658400E-02,.2822200E-02,.1865500E-02,&
     & .1779000E-03,.2155200E-02,.2858400E-02,.2969500E-02,.1958900E-02,&
     & .2131500E-03,.2362400E-02,.3068800E-02,.3130900E-02,.2052000E-02,&
     & .2528500E-03,.2582500E-02,.3287600E-02,.3306800E-02,.2144500E-02,&
     & .1065700E-03,.1531900E-02,.2109600E-02,.2267800E-02,.1484400E-02,&
     & .1285600E-03,.1692000E-02,.2274300E-02,.2389900E-02,.1563900E-02,&
     & .1547100E-03,.1863600E-02,.2451100E-02,.2522100E-02,.1643600E-02,&
     & .1851200E-03,.2047200E-02,.2636400E-02,.2668100E-02,.1723800E-02,&
     & .2187200E-03,.2241100E-02,.2827000E-02,.2826000E-02,.1804800E-02,&
     & .9287600E-04,.1323700E-02,.1804900E-02,.1921400E-02,.1247100E-02,&
     & .1122600E-03,.1465900E-02,.1951800E-02,.2029500E-02,.1314300E-02,&
     & .1351900E-03,.1617100E-02,.2108900E-02,.2149800E-02,.1383300E-02,&
     & .1610200E-03,.1777800E-02,.2270300E-02,.2281000E-02,.1452800E-02,&
     & .1898100E-03,.1948400E-02,.2437400E-02,.2424900E-02,.1523600E-02,&
     & .8082800E-04,.1147100E-02,.1548600E-02,.1630700E-02,.1048200E-02,&
     & .9806300E-04,.1271800E-02,.1679700E-02,.1728700E-02,.1105300E-02,&
     & .1177200E-03,.1405000E-02,.1817700E-02,.1837900E-02,.1164800E-02,&
     & .1398100E-03,.1546900E-02,.1958900E-02,.1956500E-02,.1225800E-02,&
     & .1644600E-03,.1694700E-02,.2105500E-02,.2087400E-02,.1288500E-02,&
     & .7053900E-04,.9962100E-03,.1332500E-02,.1387900E-02,.8822000E-03,&
     & .8558900E-04,.1106300E-02,.1449200E-02,.1477600E-02,.9311300E-03,&
     & .1024400E-03,.1224000E-02,.1570100E-02,.1576400E-02,.9823700E-03,&
     & .1215100E-03,.1348800E-02,.1694200E-02,.1684600E-02,.1036400E-02,&
     & .1424500E-03,.1475300E-02,.1822900E-02,.1804200E-02,.1091000E-02,&
     & .6159800E-04,.8670800E-03,.1150600E-02,.1185700E-02,.7439600E-03,&
     & .7469800E-04,.9644300E-03,.1253300E-02,.1267300E-02,.7859000E-03,&
     & .8909200E-04,.1068400E-02,.1359000E-02,.1357100E-02,.8309000E-03,&
     & .1053700E-03,.1176600E-02,.1468200E-02,.1456000E-02,.8782700E-03,&
     & .1232700E-03,.1285000E-02,.1581300E-02,.1564300E-02,.9259900E-03,&
     & .5374800E-04,.7562900E-03,.9959500E-03,.1016300E-02,.6280900E-03,&
     & .6497200E-04,.8426500E-03,.1085600E-02,.1090800E-02,.6641600E-03,&
     & .7738600E-04,.9339700E-03,.1178700E-02,.1172700E-02,.7038700E-03,&
     & .9113800E-04,.1026600E-02,.1274900E-02,.1263000E-02,.7450700E-03,&
     & .1064000E-03,.1118900E-02,.1373800E-02,.1360800E-02,.7883000E-03/

      data   absb(526:700, 5)   /                                       &
     & .4666700E-04,.6588400E-03,.8604900E-03,.8721400E-03,.5309900E-03,&
     & .5626300E-04,.7349000E-03,.9390000E-03,.9394600E-03,.5627800E-03,&
     & .6685300E-04,.8138300E-03,.1020900E-02,.1014200E-02,.5977600E-03,&
     & .7859800E-04,.8931400E-03,.1105400E-02,.1096300E-02,.6341900E-03,&
     & .9150300E-04,.9719300E-03,.1192700E-02,.1184000E-02,.6732000E-03,&
     & .3959900E-04,.5686600E-03,.7380300E-03,.7449100E-03,.4468800E-03,&
     & .4767700E-04,.6349000E-03,.8068500E-03,.8054000E-03,.4746600E-03,&
     & .5663000E-04,.7027900E-03,.8786000E-03,.8733200E-03,.5047600E-03,&
     & .6650600E-04,.7706100E-03,.9526200E-03,.9467600E-03,.5373000E-03,&
     & .7734300E-04,.8387000E-03,.1029500E-02,.1025000E-02,.5720000E-03,&
     & .3276700E-04,.4852300E-03,.6275100E-03,.6318300E-03,.3735300E-03,&
     & .3945000E-04,.5426100E-03,.6874400E-03,.6858700E-03,.3974400E-03,&
     & .4686100E-04,.6009300E-03,.7497700E-03,.7462300E-03,.4231500E-03,&
     & .5495600E-04,.6591600E-03,.8146900E-03,.8115800E-03,.4514400E-03,&
     & .6390800E-04,.7182000E-03,.8821200E-03,.8810300E-03,.4816500E-03,&
     & .2673400E-04,.4075400E-03,.5264500E-03,.5301400E-03,.3111800E-03,&
     & .3227300E-04,.4566100E-03,.5782500E-03,.5776600E-03,.3320600E-03,&
     & .3848800E-04,.5069700E-03,.6324500E-03,.6305800E-03,.3547600E-03,&
     & .4522800E-04,.5570600E-03,.6891800E-03,.6880900E-03,.3794400E-03,&
     & .5267500E-04,.6080800E-03,.7479400E-03,.7493400E-03,.4059800E-03,&
     & .2175100E-04,.3420900E-03,.4414600E-03,.4445800E-03,.2592000E-03,&
     & .2632600E-04,.3841500E-03,.4860300E-03,.4863200E-03,.2773000E-03,&
     & .3151500E-04,.4273600E-03,.5332500E-03,.5326200E-03,.2971400E-03,&
     & .3721900E-04,.4706400E-03,.5826300E-03,.5832100E-03,.3189000E-03,&
     & .4334400E-04,.5149300E-03,.6340200E-03,.6372900E-03,.3421200E-03,&
     & .1764300E-04,.2871700E-03,.3702000E-03,.3728800E-03,.2158000E-03,&
     & .2141800E-04,.3234300E-03,.4087100E-03,.4095300E-03,.2315100E-03,&
     & .2572500E-04,.3604400E-03,.4496600E-03,.4500500E-03,.2488000E-03,&
     & .3047900E-04,.3978500E-03,.4926300E-03,.4944500E-03,.2678300E-03,&
     & .3565300E-04,.4364600E-03,.5377100E-03,.5423000E-03,.2882200E-03,&
     & .1410200E-04,.2381400E-03,.3073000E-03,.3098100E-03,.1786400E-03,&
     & .1718700E-04,.2693400E-03,.3403900E-03,.3414600E-03,.1923400E-03,&
     & .2073900E-04,.3013000E-03,.3757100E-03,.3767200E-03,.2073700E-03,&
     & .2467000E-04,.3333300E-03,.4128300E-03,.4153800E-03,.2240800E-03,&
     & .2904600E-04,.3669400E-03,.4521600E-03,.4572900E-03,.2418700E-03/

      data   absb(701:875, 5)   /                                       &
     & .1122100E-04,.1971000E-03,.2546500E-03,.2569600E-03,.1477300E-03,&
     & .1374700E-04,.2239000E-03,.2830500E-03,.2841300E-03,.1595500E-03,&
     & .1664900E-04,.2514500E-03,.3134600E-03,.3146700E-03,.1726900E-03,&
     & .1989900E-04,.2793500E-03,.3456300E-03,.3484500E-03,.1871800E-03,&
     & .2356800E-04,.3083000E-03,.3797400E-03,.3851200E-03,.2028100E-03,&
     & .8914700E-05,.1631400E-03,.2108800E-03,.2129400E-03,.1220500E-03,&
     & .1095700E-04,.1859600E-03,.2352300E-03,.2362800E-03,.1322500E-03,&
     & .1332700E-04,.2097400E-03,.2613400E-03,.2626300E-03,.1435600E-03,&
     & .1600500E-04,.2337800E-03,.2892500E-03,.2920200E-03,.1562100E-03,&
     & .1903800E-04,.2591400E-03,.3189700E-03,.3241500E-03,.1698200E-03,&
     & .6999500E-05,.1340500E-03,.1733800E-03,.1753500E-03,.1003600E-03,&
     & .8640600E-05,.1534500E-03,.1941600E-03,.1951400E-03,.1091000E-03,&
     & .1055600E-04,.1737300E-03,.2165900E-03,.2176700E-03,.1188700E-03,&
     & .1273900E-04,.1944500E-03,.2405300E-03,.2431100E-03,.1298000E-03,&
     & .1523200E-04,.2164000E-03,.2664500E-03,.2710700E-03,.1416900E-03,&
     & .5471200E-05,.1097000E-03,.1419500E-03,.1438600E-03,.8226000E-04,&
     & .6764400E-05,.1261200E-03,.1596300E-03,.1606100E-03,.8972700E-04,&
     & .8297300E-05,.1434800E-03,.1787800E-03,.1797500E-03,.9812200E-04,&
     & .1008200E-04,.1613200E-03,.1995200E-03,.2015500E-03,.1075500E-03,&
     & .1210700E-04,.1800800E-03,.2217900E-03,.2257600E-03,.1178700E-03,&
     & .4276700E-05,.8968000E-04,.1161100E-03,.1178100E-03,.6733800E-04,&
     & .5286000E-05,.1036000E-03,.1310700E-03,.1320300E-03,.7372200E-04,&
     & .6510700E-05,.1183800E-03,.1473500E-03,.1482900E-03,.8085200E-04,&
     & .7950100E-05,.1337400E-03,.1652000E-03,.1669300E-03,.8896900E-04,&
     & .9593700E-05,.1499300E-03,.1845600E-03,.1878100E-03,.9789700E-04,&
     & .3322600E-05,.7291800E-04,.9463500E-04,.9606900E-04,.5491200E-04,&
     & .4118500E-05,.8470500E-04,.1072600E-03,.1080900E-03,.6032500E-04,&
     & .5079600E-05,.9741400E-04,.1210600E-03,.1218400E-03,.6641500E-04,&
     & .6231100E-05,.1105500E-03,.1362500E-03,.1377200E-03,.7335400E-04,&
     & .7559400E-05,.1243900E-03,.1529700E-03,.1556900E-03,.8097200E-04,&
     & .2550700E-05,.5875300E-04,.7659900E-04,.7779800E-04,.4450800E-04,&
     & .3180000E-05,.6867900E-04,.8724000E-04,.8784300E-04,.4911500E-04,&
     & .3927300E-05,.7949700E-04,.9887100E-04,.9942100E-04,.5424900E-04,&
     & .4834600E-05,.9078200E-04,.1117300E-03,.1128200E-03,.6011300E-04,&
     & .5899500E-05,.1026700E-03,.1259200E-03,.1281500E-03,.6665600E-04/

      data   absb(876:1050, 5)   /                                      &
     & .1953900E-05,.4717400E-04,.6187400E-04,.6291600E-04,.3597300E-04,&
     & .2445200E-05,.5557500E-04,.7085300E-04,.7123800E-04,.3989800E-04,&
     & .3040200E-05,.6478300E-04,.8063300E-04,.8092100E-04,.4421400E-04,&
     & .3739000E-05,.7443900E-04,.9149800E-04,.9220800E-04,.4916200E-04,&
     & .4583600E-05,.8460800E-04,.1035800E-03,.1053100E-03,.5470200E-04,&
     & .1494000E-05,.3771600E-04,.4982900E-04,.5080400E-04,.2898700E-04,&
     & .1875800E-05,.4477700E-04,.5739200E-04,.5769700E-04,.3231700E-04,&
     & .2341300E-05,.5260400E-04,.6560800E-04,.6574800E-04,.3594600E-04,&
     & .2896100E-05,.6089100E-04,.7482400E-04,.7518200E-04,.4009900E-04,&
     & .3552400E-05,.6958700E-04,.8506800E-04,.8629500E-04,.4480400E-04,&
     & .1150000E-05,.3016900E-04,.4015700E-04,.4106900E-04,.2336200E-04,&
     & .1445300E-05,.3608900E-04,.4654900E-04,.4679900E-04,.2619300E-04,&
     & .1812900E-05,.4273500E-04,.5347400E-04,.5351400E-04,.2924900E-04,&
     & .2252900E-05,.4984300E-04,.6123400E-04,.6140400E-04,.3274800E-04,&
     & .2776200E-05,.5732800E-04,.7002700E-04,.7078300E-04,.3671200E-04,&
     & .8885100E-06,.2409500E-04,.3230300E-04,.3321000E-04,.1880500E-04,&
     & .1117700E-05,.2908200E-04,.3770500E-04,.3794400E-04,.2120900E-04,&
     & .1406000E-05,.3466700E-04,.4357800E-04,.4357000E-04,.2379200E-04,&
     & .1753900E-05,.4073900E-04,.5011400E-04,.5017200E-04,.2672000E-04,&
     & .2172900E-05,.4722400E-04,.5759800E-04,.5807000E-04,.3007900E-04,&
     & .6852500E-06,.1917800E-04,.2589600E-04,.2680900E-04,.1509700E-04,&
     & .8630100E-06,.2333600E-04,.3043400E-04,.3071300E-04,.1712000E-04,&
     & .1088900E-05,.2802800E-04,.3542200E-04,.3539700E-04,.1930100E-04,&
     & .1364900E-05,.3318700E-04,.4095400E-04,.4092100E-04,.2175200E-04,&
     & .1698200E-05,.3874700E-04,.4726000E-04,.4755700E-04,.2458800E-04,&
     & .5280900E-06,.1520600E-04,.2068800E-04,.2159200E-04,.1208400E-04,&
     & .6649300E-06,.1864700E-04,.2447900E-04,.2481900E-04,.1378100E-04,&
     & .8410200E-06,.2257400E-04,.2868300E-04,.2869000E-04,.1562100E-04,&
     & .1058500E-05,.2695500E-04,.3336400E-04,.3329800E-04,.1767700E-04,&
     & .1323900E-05,.3171000E-04,.3873300E-04,.3886400E-04,.2004900E-04,&
     & .4096900E-06,.1209300E-04,.1656400E-04,.1742100E-04,.9688200E-05,&
     & .5156900E-06,.1493700E-04,.1973400E-04,.2009400E-04,.1111200E-04,&
     & .6529600E-06,.1822900E-04,.2328200E-04,.2332100E-04,.1266400E-04,&
     & .8257300E-06,.2195000E-04,.2723100E-04,.2715000E-04,.1439300E-04,&
     & .1037300E-05,.2601300E-04,.3179100E-04,.3182800E-04,.1637900E-04/

      data   absb(1051:1175, 5)   /                                     &
     & .3192800E-06,.9616200E-05,.1325700E-04,.1405300E-04,.7774100E-05,&
     & .4011200E-06,.1196300E-04,.1590600E-04,.1625700E-04,.8957800E-05,&
     & .5081600E-06,.1471700E-04,.1888600E-04,.1894700E-04,.1026200E-04,&
     & .6446500E-06,.1786400E-04,.2220700E-04,.2215500E-04,.1171500E-04,&
     & .8132800E-06,.2132600E-04,.2607600E-04,.2606100E-04,.1337600E-04,&
     & .2482900E-06,.7620400E-05,.1057800E-04,.1130900E-04,.6220800E-05,&
     & .3117600E-06,.9548800E-05,.1278000E-04,.1312500E-04,.7201300E-05,&
     & .3950600E-06,.1185000E-04,.1527900E-04,.1534800E-04,.8294700E-05,&
     & .5026600E-06,.1449200E-04,.1807400E-04,.1804000E-04,.9515300E-05,&
     & .6363800E-06,.1744200E-04,.2133600E-04,.2131000E-04,.1090400E-04,&
     & .1930200E-06,.6018700E-05,.8416200E-05,.9079900E-05,.4964900E-05,&
     & .2422200E-06,.7595900E-05,.1023300E-04,.1057000E-04,.5770100E-05,&
     & .3068800E-06,.9508000E-05,.1231800E-04,.1240300E-04,.6691300E-05,&
     & .3908900E-06,.1171900E-04,.1467000E-04,.1464400E-04,.7716600E-05,&
     & .4967400E-06,.1421400E-04,.1740200E-04,.1738200E-04,.8872700E-05,&
     & .1504600E-06,.4766300E-05,.6699800E-05,.7290800E-05,.3963100E-05,&
     & .1885600E-06,.6045900E-05,.8190800E-05,.8520600E-05,.4627000E-05,&
     & .2386900E-06,.7630200E-05,.9924500E-05,.1002300E-04,.5397700E-05,&
     & .3045900E-06,.9476700E-05,.1191100E-04,.1189400E-04,.6257900E-05,&
     & .3882000E-06,.1158800E-04,.1420500E-04,.1418200E-04,.7222600E-05,&
     & .1221200E-06,.3933200E-05,.5523000E-05,.6009400E-05,.3249900E-05,&
     & .1532400E-06,.5017800E-05,.6787900E-05,.7056600E-05,.3810800E-05,&
     & .1941800E-06,.6367800E-05,.8264700E-05,.8344300E-05,.4467200E-05,&
     & .2482000E-06,.7954300E-05,.9967000E-05,.9951800E-05,.5202700E-05,&
     & .3174000E-06,.9770400E-05,.1196400E-04,.1193800E-04,.6027800E-05/

      data   absb(  1:175, 6)   /                                       &
     & .4672255E-01,.1179341E+00,.1338757E+00,.1299972E+00,.9161157E-01,&
     & .5062470E-01,.1213117E+00,.1369429E+00,.1328028E+00,.9360006E-01,&
     & .5459974E-01,.1248223E+00,.1399926E+00,.1355144E+00,.9556280E-01,&
     & .5873868E-01,.1286746E+00,.1431991E+00,.1382901E+00,.9738836E-01,&
     & .6302530E-01,.1329297E+00,.1467302E+00,.1411782E+00,.9929195E-01,&
     & .3968991E-01,.9976895E-01,.1131751E+00,.1100474E+00,.7710072E-01,&
     & .4301477E-01,.1026851E+00,.1158037E+00,.1124295E+00,.7893417E-01,&
     & .4646290E-01,.1058587E+00,.1184976E+00,.1148060E+00,.8065093E-01,&
     & .5002899E-01,.1093646E+00,.1214273E+00,.1173032E+00,.8238971E-01,&
     & .5370336E-01,.1132393E+00,.1247403E+00,.1199136E+00,.8410182E-01,&
     & .3361226E-01,.8419071E-01,.9544048E-01,.9285444E-01,.6446524E-01,&
     & .3646479E-01,.8678678E-01,.9771637E-01,.9491470E-01,.6610593E-01,&
     & .3943418E-01,.8966612E-01,.1001521E+00,.9704047E-01,.6769280E-01,&
     & .4249750E-01,.9286388E-01,.1028925E+00,.9929778E-01,.6917725E-01,&
     & .4559710E-01,.9635024E-01,.1059705E+00,.1017312E+00,.7059703E-01,&
     & .2841305E-01,.7093757E-01,.8032314E-01,.7813700E-01,.5373624E-01,&
     & .3087001E-01,.7327916E-01,.8235049E-01,.7995476E-01,.5513198E-01,&
     & .3341491E-01,.7589200E-01,.8459743E-01,.8187839E-01,.5650567E-01,&
     & .3600242E-01,.7879372E-01,.8716029E-01,.8394378E-01,.5783239E-01,&
     & .3862500E-01,.8189679E-01,.8997661E-01,.8623580E-01,.5901740E-01,&
     & .2401634E-01,.5974102E-01,.6747709E-01,.6564216E-01,.4478664E-01,&
     & .2611796E-01,.6185142E-01,.6932940E-01,.6726455E-01,.4600594E-01,&
     & .2827920E-01,.6420880E-01,.7144020E-01,.6899927E-01,.4720878E-01,&
     & .3044618E-01,.6678350E-01,.7379995E-01,.7092071E-01,.4834282E-01,&
     & .3265937E-01,.6948508E-01,.7635424E-01,.7306622E-01,.4936137E-01,&
     & .2029779E-01,.5028866E-01,.5661750E-01,.5508955E-01,.3739316E-01,&
     & .2208972E-01,.5221100E-01,.5834184E-01,.5654839E-01,.3848438E-01,&
     & .2389477E-01,.5432163E-01,.6030221E-01,.5814906E-01,.3951198E-01,&
     & .2571656E-01,.5658610E-01,.6243776E-01,.5994219E-01,.4048024E-01,&
     & .2759559E-01,.5894005E-01,.6471627E-01,.6189304E-01,.4136898E-01,&
     & .1714555E-01,.4232553E-01,.4751571E-01,.4621072E-01,.3128045E-01,&
     & .1865226E-01,.4405192E-01,.4911024E-01,.4754554E-01,.3222507E-01,&
     & .2016046E-01,.4594242E-01,.5088205E-01,.4903912E-01,.3313354E-01,&
     & .2170679E-01,.4792414E-01,.5278935E-01,.5068417E-01,.3395265E-01,&
     & .2331129E-01,.4999405E-01,.5480871E-01,.5244342E-01,.3473337E-01/

      data   absb(176:350, 6)   /                                       &
     & .1449397E-01,.3565245E-01,.3997027E-01,.3878295E-01,.2623894E-01,&
     & .1574907E-01,.3721054E-01,.4142074E-01,.4002092E-01,.2707613E-01,&
     & .1701927E-01,.3887652E-01,.4300733E-01,.4140562E-01,.2785469E-01,&
     & .1834230E-01,.4061477E-01,.4467996E-01,.4289260E-01,.2857492E-01,&
     & .1972332E-01,.4243979E-01,.4646142E-01,.4447765E-01,.2929449E-01,&
     & .1223686E-01,.3005384E-01,.3364453E-01,.3256819E-01,.2203383E-01,&
     & .1329184E-01,.3143872E-01,.3496344E-01,.3371716E-01,.2275179E-01,&
     & .1437272E-01,.3289059E-01,.3638637E-01,.3497888E-01,.2343303E-01,&
     & .1550047E-01,.3441489E-01,.3787367E-01,.3632111E-01,.2407709E-01,&
     & .1668336E-01,.3604499E-01,.3944792E-01,.3774085E-01,.2472138E-01,&
     & .1038282E-01,.2542597E-01,.2842132E-01,.2744137E-01,.1852562E-01,&
     & .1127312E-01,.2664076E-01,.2962239E-01,.2850810E-01,.1915442E-01,&
     & .1219323E-01,.2791660E-01,.3088571E-01,.2965335E-01,.1975054E-01,&
     & .1315724E-01,.2927559E-01,.3221079E-01,.3086581E-01,.2032424E-01,&
     & .1418884E-01,.3074537E-01,.3362124E-01,.3215085E-01,.2090413E-01,&
     & .8810984E-02,.2153212E-01,.2404863E-01,.2318381E-01,.1558634E-01,&
     & .9557713E-02,.2259858E-01,.2512473E-01,.2414921E-01,.1614064E-01,&
     & .1034460E-01,.2372894E-01,.2624401E-01,.2518234E-01,.1665912E-01,&
     & .1117906E-01,.2495670E-01,.2743338E-01,.2627714E-01,.1717734E-01,&
     & .1208564E-01,.2629206E-01,.2873101E-01,.2745336E-01,.1769705E-01,&
     & .7472179E-02,.1826124E-01,.2037977E-01,.1963051E-01,.1311820E-01,&
     & .8105914E-02,.1920286E-01,.2133520E-01,.2050264E-01,.1360311E-01,&
     & .8786371E-02,.2021445E-01,.2233873E-01,.2143465E-01,.1406558E-01,&
     & .9517597E-02,.2132628E-01,.2342388E-01,.2243468E-01,.1453007E-01,&
     & .1031444E-01,.2254987E-01,.2462788E-01,.2351153E-01,.1499827E-01,&
     & .6339327E-02,.1551891E-01,.1730563E-01,.1666327E-01,.1106485E-01,&
     & .6886916E-02,.1635981E-01,.1815633E-01,.1745076E-01,.1148631E-01,&
     & .7479578E-02,.1727214E-01,.1906789E-01,.1829842E-01,.1190414E-01,&
     & .8122799E-02,.1828117E-01,.2007236E-01,.1921662E-01,.1232324E-01,&
     & .8828155E-02,.1940872E-01,.2120238E-01,.2020565E-01,.1274767E-01,&
     & .5392272E-02,.1323113E-01,.1473701E-01,.1419217E-01,.9365787E-02,&
     & .5870200E-02,.1398439E-01,.1550589E-01,.1490579E-01,.9737495E-02,&
     & .6389745E-02,.1481106E-01,.1634605E-01,.1568477E-01,.1010973E-01,&
     & .6958634E-02,.1574163E-01,.1728569E-01,.1652721E-01,.1049045E-01,&
     & .7581630E-02,.1678655E-01,.1834875E-01,.1744786E-01,.1088430E-01/

      data   absb(351:525, 6)   /                                       &
     & .4592919E-02,.1130255E-01,.1258234E-01,.1211906E-01,.7936306E-02,&
     & .5013344E-02,.1198363E-01,.1328506E-01,.1277230E-01,.8268918E-02,&
     & .5471875E-02,.1274465E-01,.1406720E-01,.1349234E-01,.8605208E-02,&
     & .5975738E-02,.1360931E-01,.1495090E-01,.1427336E-01,.8951125E-02,&
     & .6524713E-02,.1458353E-01,.1594799E-01,.1514041E-01,.9313867E-02,&
     & .3921257E-02,.9679135E-02,.1077668E-01,.1037874E-01,.6732679E-02,&
     & .4289466E-02,.1030205E-01,.1142417E-01,.1098668E-01,.7034601E-02,&
     & .4696449E-02,.1101261E-01,.1215654E-01,.1165356E-01,.7339816E-02,&
     & .5140924E-02,.1181986E-01,.1298600E-01,.1238661E-01,.7655437E-02,&
     & .5627986E-02,.1273133E-01,.1391895E-01,.1320853E-01,.7987663E-02,&
     & .3358374E-02,.8318578E-02,.9266665E-02,.8925427E-02,.5731495E-02,&
     & .3681177E-02,.8898948E-02,.9870888E-02,.9491903E-02,.6003477E-02,&
     & .4042382E-02,.9566221E-02,.1056008E-01,.1011534E-01,.6284646E-02,&
     & .4437145E-02,.1032409E-01,.1134025E-01,.1081068E-01,.6574647E-02,&
     & .4869992E-02,.1118221E-01,.1221978E-01,.1159427E-01,.6878293E-02,&
     & .2880952E-02,.7177713E-02,.8000548E-02,.7709187E-02,.4888892E-02,&
     & .3168109E-02,.7724358E-02,.8567998E-02,.8237375E-02,.5136819E-02,&
     & .3487574E-02,.8352795E-02,.9218971E-02,.8825929E-02,.5394241E-02,&
     & .3839297E-02,.9067161E-02,.9957533E-02,.9491401E-02,.5660899E-02,&
     & .4223712E-02,.9875806E-02,.1079202E-01,.1023576E-01,.5939546E-02,&
     & .2478739E-02,.6226385E-02,.6940669E-02,.6691212E-02,.4181873E-02,&
     & .2734468E-02,.6743172E-02,.7478246E-02,.7187563E-02,.4408421E-02,&
     & .3017927E-02,.7337488E-02,.8096419E-02,.7749978E-02,.4643233E-02,&
     & .3331178E-02,.8012614E-02,.8797666E-02,.8384632E-02,.4887248E-02,&
     & .3674803E-02,.8776521E-02,.9606185E-02,.9089542E-02,.5147082E-02,&
     & .2139286E-02,.5432558E-02,.6053952E-02,.5837608E-02,.3588363E-02,&
     & .2366596E-02,.5923173E-02,.6566620E-02,.6309240E-02,.3794714E-02,&
     & .2619079E-02,.6485565E-02,.7155112E-02,.6847354E-02,.4009599E-02,&
     & .2899192E-02,.7125771E-02,.7834469E-02,.7451217E-02,.4235186E-02,&
     & .3206376E-02,.7850198E-02,.8612179E-02,.8123777E-02,.4476963E-02,&
     & .1850772E-02,.4769289E-02,.5310687E-02,.5121509E-02,.3086654E-02,&
     & .2053571E-02,.5233812E-02,.5801277E-02,.5573075E-02,.3276182E-02,&
     & .2279445E-02,.5767992E-02,.6371794E-02,.6087455E-02,.3472505E-02,&
     & .2530319E-02,.6378620E-02,.7028994E-02,.6663834E-02,.3681702E-02,&
     & .2802247E-02,.7072716E-02,.7774573E-02,.7310658E-02,.3905465E-02/

      data   absb(526:700, 6)   /                                       &
     & .1599613E-02,.4199571E-02,.4673794E-02,.4506393E-02,.2664028E-02,&
     & .1780622E-02,.4639345E-02,.5147271E-02,.4937749E-02,.2836730E-02,&
     & .1982209E-02,.5147946E-02,.5697272E-02,.5427726E-02,.3018582E-02,&
     & .2205837E-02,.5730040E-02,.6328408E-02,.5979873E-02,.3214093E-02,&
     & .2444633E-02,.6402438E-02,.7040553E-02,.6604978E-02,.3423730E-02,&
     & .1370767E-02,.3680434E-02,.4101387E-02,.3951642E-02,.2288026E-02,&
     & .1530214E-02,.4092955E-02,.4551071E-02,.4359160E-02,.2445148E-02,&
     & .1709076E-02,.4573254E-02,.5075448E-02,.4822350E-02,.2615383E-02,&
     & .1905572E-02,.5130202E-02,.5673701E-02,.5348975E-02,.2795090E-02,&
     & .2113656E-02,.5770683E-02,.6349792E-02,.5951205E-02,.2990809E-02,&
     & .1161344E-02,.3200182E-02,.3576104E-02,.3443194E-02,.1950452E-02,&
     & .1300611E-02,.3583463E-02,.3997263E-02,.3823311E-02,.2093149E-02,&
     & .1456618E-02,.4035349E-02,.4489225E-02,.4258002E-02,.2249394E-02,&
     & .1627537E-02,.4558261E-02,.5050496E-02,.4756449E-02,.2415485E-02,&
     & .1808199E-02,.5161226E-02,.5687359E-02,.5329814E-02,.2597049E-02,&
     & .9689423E-03,.2745553E-02,.3074976E-02,.2960730E-02,.1661770E-02,&
     & .1088391E-02,.3094206E-02,.3461113E-02,.3308475E-02,.1792192E-02,&
     & .1222284E-02,.3506981E-02,.3912501E-02,.3709778E-02,.1935066E-02,&
     & .1369638E-02,.3987964E-02,.4430278E-02,.4173459E-02,.2088437E-02,&
     & .1526024E-02,.4545546E-02,.5021863E-02,.4708792E-02,.2255640E-02,&
     & .8077252E-03,.2356556E-02,.2645654E-02,.2547112E-02,.1416723E-02,&
     & .9097668E-03,.2673661E-02,.2999603E-02,.2865604E-02,.1535808E-02,&
     & .1024815E-02,.3051199E-02,.3413691E-02,.3236220E-02,.1665481E-02,&
     & .1151477E-02,.3493356E-02,.3891808E-02,.3667727E-02,.1807129E-02,&
     & .1286842E-02,.4009109E-02,.4442138E-02,.4167563E-02,.1962379E-02,&
     & .6730887E-03,.2025445E-02,.2280105E-02,.2194379E-02,.1207918E-02,&
     & .7604151E-03,.2314345E-02,.2604175E-02,.2486551E-02,.1316161E-02,&
     & .8589949E-03,.2660424E-02,.2985221E-02,.2829802E-02,.1435082E-02,&
     & .9677787E-03,.3067804E-02,.3427622E-02,.3232026E-02,.1564953E-02,&
     & .1084841E-02,.3545518E-02,.3940911E-02,.3699314E-02,.1709431E-02,&
     & .5540015E-03,.1719029E-02,.1940291E-02,.1868227E-02,.1025773E-02,&
     & .6275728E-03,.1977116E-02,.2231569E-02,.2131986E-02,.1124006E-02,&
     & .7113040E-03,.2288889E-02,.2577000E-02,.2444792E-02,.1232036E-02,&
     & .8041626E-03,.2658785E-02,.2980992E-02,.2813689E-02,.1351842E-02,&
     & .9045868E-03,.3094678E-02,.3453278E-02,.3244781E-02,.1485313E-02/

      data   absb(701:875, 6)   /                                       &
     & .4548896E-03,.1456307E-02,.1647750E-02,.1587652E-02,.8696717E-03,&
     & .5169304E-03,.1686770E-02,.1909185E-02,.1825449E-02,.9593282E-03,&
     & .5876050E-03,.1966424E-02,.2220965E-02,.2109507E-02,.1057588E-02,&
     & .6666461E-03,.2301238E-02,.2589214E-02,.2446720E-02,.1167507E-02,&
     & .7525562E-03,.2698055E-02,.3023141E-02,.2843621E-02,.1289308E-02,&
     & .3734897E-03,.1232867E-02,.1398777E-02,.1349075E-02,.7359709E-03,&
     & .4253634E-03,.1438755E-02,.1632644E-02,.1562815E-02,.8174284E-03,&
     & .4850469E-03,.1689562E-02,.1914018E-02,.1820551E-02,.9077182E-03,&
     & .5522656E-03,.1992581E-02,.2249528E-02,.2128458E-02,.1008030E-02,&
     & .6258618E-03,.2353791E-02,.2648433E-02,.2493658E-02,.1119658E-02,&
     & .3045367E-03,.1033446E-02,.1175577E-02,.1135975E-02,.6197699E-03,&
     & .3473823E-03,.1215271E-02,.1382545E-02,.1325828E-02,.6926384E-03,&
     & .3970307E-03,.1437673E-02,.1633541E-02,.1556762E-02,.7746954E-03,&
     & .4539062E-03,.1708894E-02,.1936090E-02,.1834842E-02,.8669180E-03,&
     & .5165865E-03,.2034536E-02,.2299010E-02,.2167541E-02,.9692109E-03,&
     & .2472336E-03,.8613164E-03,.9823016E-03,.9514129E-03,.5196099E-03,&
     & .2828538E-03,.1020438E-02,.1163949E-02,.1118679E-02,.5847521E-03,&
     & .3237516E-03,.1216274E-02,.1386324E-02,.1324100E-02,.6591741E-03,&
     & .3714736E-03,.1458189E-02,.1657661E-02,.1574061E-02,.7425059E-03,&
     & .4245663E-03,.1749792E-02,.1985372E-02,.1874641E-02,.8368539E-03,&
     & .2006002E-03,.7166388E-03,.8191347E-03,.7953709E-03,.4345003E-03,&
     & .2300088E-03,.8550124E-03,.9779366E-03,.9420990E-03,.4927929E-03,&
     & .2640138E-03,.1027318E-02,.1174304E-02,.1124337E-02,.5593878E-03,&
     & .3036185E-03,.1241926E-02,.1416866E-02,.1348159E-02,.6350064E-03,&
     & .3483255E-03,.1503311E-02,.1712388E-02,.1619624E-02,.7206305E-03,&
     & .1620672E-03,.5924002E-03,.6788004E-03,.6609668E-03,.3615247E-03,&
     & .1863562E-03,.7120820E-03,.8162499E-03,.7885530E-03,.4130089E-03,&
     & .2144232E-03,.8626280E-03,.9887801E-03,.9492287E-03,.4724825E-03,&
     & .2473832E-03,.1051309E-02,.1203656E-02,.1147998E-02,.5403679E-03,&
     & .2847841E-03,.1284144E-02,.1468365E-02,.1391452E-02,.6179991E-03,&
     & .1299474E-03,.4846482E-03,.5565712E-03,.5436535E-03,.2982511E-03,&
     & .1498986E-03,.5864424E-03,.6736990E-03,.6529381E-03,.3434172E-03,&
     & .1729325E-03,.7163074E-03,.8229894E-03,.7924220E-03,.3957912E-03,&
     & .2000081E-03,.8797706E-03,.1010993E-02,.9670011E-03,.4564930E-03,&
     & .2313308E-03,.1085043E-02,.1245383E-02,.1182959E-02,.5264754E-03/

      data   absb(876:1050, 6)   /                                      &
     & .1041538E-03,.3951902E-03,.4548472E-03,.4454470E-03,.2452479E-03,&
     & .1203711E-03,.4813274E-03,.5542653E-03,.5387494E-03,.2845514E-03,&
     & .1392557E-03,.5925482E-03,.6823171E-03,.6591592E-03,.3305080E-03,&
     & .1615406E-03,.7342118E-03,.8463369E-03,.8118950E-03,.3843966E-03,&
     & .1875302E-03,.9137067E-03,.1052775E-02,.1002597E-02,.4471473E-03,&
     & .8339613E-04,.3212348E-03,.3702850E-03,.3635514E-03,.2009331E-03,&
     & .9663218E-04,.3936340E-03,.4542459E-03,.4427976E-03,.2348384E-03,&
     & .1119212E-03,.4884335E-03,.5635159E-03,.5461149E-03,.2749028E-03,&
     & .1301921E-03,.6106310E-03,.7057229E-03,.6791355E-03,.3226368E-03,&
     & .1518690E-03,.7666828E-03,.8867916E-03,.8470024E-03,.3788358E-03,&
     & .6703277E-04,.2615715E-03,.3020685E-03,.2972644E-03,.1647457E-03,&
     & .7781745E-04,.3226047E-03,.3728758E-03,.3643763E-03,.1940020E-03,&
     & .9035340E-04,.4034207E-03,.4665853E-03,.4531468E-03,.2290995E-03,&
     & .1051725E-03,.5090275E-03,.5895403E-03,.5690155E-03,.2711767E-03,&
     & .1231606E-03,.6451073E-03,.7487546E-03,.7170779E-03,.3213112E-03,&
     & .5388627E-04,.2129730E-03,.2462008E-03,.2426751E-03,.1352080E-03,&
     & .6277450E-04,.2640459E-03,.3057224E-03,.2995161E-03,.1599361E-03,&
     & .7304733E-04,.3329169E-03,.3859229E-03,.3754737E-03,.1906374E-03,&
     & .8515751E-04,.4241494E-03,.4919296E-03,.4761974E-03,.2276739E-03,&
     & .9991988E-04,.5432688E-03,.6320927E-03,.6069695E-03,.2723849E-03,&
     & .4326262E-04,.1729315E-03,.2000827E-03,.1973909E-03,.1105920E-03,&
     & .5057315E-04,.2155302E-03,.2498708E-03,.2454105E-03,.1317820E-03,&
     & .5902025E-04,.2738145E-03,.3180329E-03,.3099867E-03,.1580455E-03,&
     & .6892231E-04,.3521416E-03,.4093245E-03,.3971150E-03,.1905530E-03,&
     & .8105656E-04,.4558123E-03,.5314348E-03,.5116504E-03,.2300912E-03,&
     & .3466273E-04,.1400139E-03,.1619739E-03,.1598674E-03,.9004375E-04,&
     & .4063512E-04,.1754360E-03,.2034983E-03,.2003206E-03,.1082624E-03,&
     & .4757947E-04,.2243047E-03,.2609699E-03,.2549037E-03,.1307310E-03,&
     & .5575140E-04,.2912187E-03,.3392706E-03,.3296565E-03,.1588775E-03,&
     & .6571337E-04,.3810311E-03,.4450150E-03,.4296054E-03,.1937683E-03,&
     & .2781043E-04,.1138793E-03,.1316412E-03,.1298484E-03,.7350396E-04,&
     & .3277645E-04,.1433051E-03,.1663826E-03,.1640294E-03,.8913117E-04,&
     & .3850357E-04,.1846844E-03,.2150481E-03,.2104981E-03,.1086380E-03,&
     & .4522837E-04,.2419881E-03,.2824462E-03,.2747396E-03,.1329710E-03,&
     & .5347828E-04,.3200061E-03,.3744878E-03,.3622497E-03,.1637392E-03/

      data   absb(1051:1175, 6)   /                                     &
     & .2232236E-04,.9267720E-04,.1070487E-03,.1054862E-03,.5992194E-04,&
     & .2641653E-04,.1172861E-03,.1361391E-03,.1343624E-03,.7334096E-04,&
     & .3117369E-04,.1521353E-03,.1772789E-03,.1738983E-03,.9024944E-04,&
     & .3674804E-04,.2011031E-03,.2351698E-03,.2290276E-03,.1115235E-03,&
     & .4354793E-04,.2690881E-03,.3153806E-03,.3055401E-03,.1384935E-03,&
     & .1785498E-04,.7521527E-04,.8673443E-04,.8536647E-04,.4869020E-04,&
     & .2122523E-04,.9568767E-04,.1111031E-03,.1096456E-03,.6010776E-04,&
     & .2515713E-04,.1249154E-03,.1456399E-03,.1431298E-03,.7472628E-04,&
     & .2980978E-04,.1668004E-03,.1951256E-03,.1902764E-03,.9333750E-04,&
     & .3542498E-04,.2255442E-03,.2648513E-03,.2567580E-03,.1168654E-03,&
     & .1425709E-04,.6087198E-04,.7014411E-04,.6881006E-04,.3939206E-04,&
     & .1702198E-04,.7779210E-04,.9029022E-04,.8913052E-04,.4908012E-04,&
     & .2026100E-04,.1023006E-03,.1192918E-03,.1173391E-03,.6161544E-04,&
     & .2410865E-04,.1377899E-03,.1613227E-03,.1576056E-03,.7784401E-04,&
     & .2878722E-04,.1883410E-03,.2215045E-03,.2149526E-03,.9849260E-04,&
     & .1138303E-04,.4934557E-04,.5673238E-04,.5553184E-04,.3186568E-04,&
     & .1366146E-04,.6335560E-04,.7355595E-04,.7250807E-04,.4005918E-04,&
     & .1632332E-04,.8387900E-04,.9788807E-04,.9633384E-04,.5083176E-04,&
     & .1950274E-04,.1140523E-03,.1335383E-03,.1306879E-03,.6491699E-04,&
     & .2338070E-04,.1577480E-03,.1855710E-03,.1802136E-03,.8318615E-04,&
     & .9393833E-05,.4172083E-04,.4794329E-04,.4688225E-04,.2676670E-04,&
     & .1131203E-04,.5413446E-04,.6285587E-04,.6190797E-04,.3404469E-04,&
     & .1356891E-04,.7245624E-04,.8467100E-04,.8329182E-04,.4370019E-04,&
     & .1627959E-04,.9993465E-04,.1171212E-03,.1145831E-03,.5652037E-04,&
     & .1962119E-04,.1401179E-03,.1650762E-03,.1601642E-03,.7346404E-04/

      data   absb(  1:175, 7)   /                                       &
     & .3229900E+00,.5476000E+00,.6007900E+00,.5851700E+00,.4227600E+00,&
     & .3259400E+00,.5596300E+00,.6165300E+00,.6014700E+00,.4328100E+00,&
     & .3305600E+00,.5718200E+00,.6323600E+00,.6186300E+00,.4414900E+00,&
     & .3370600E+00,.5842600E+00,.6482200E+00,.6348900E+00,.4489300E+00,&
     & .3463500E+00,.5973900E+00,.6634300E+00,.6500300E+00,.4548600E+00,&
     & .2718500E+00,.4739600E+00,.5208700E+00,.5051800E+00,.3589800E+00,&
     & .2745500E+00,.4845400E+00,.5347300E+00,.5205800E+00,.3672500E+00,&
     & .2787800E+00,.4952300E+00,.5487600E+00,.5355600E+00,.3747700E+00,&
     & .2854000E+00,.5066000E+00,.5624400E+00,.5494800E+00,.3809500E+00,&
     & .2947400E+00,.5187600E+00,.5759200E+00,.5625500E+00,.3866000E+00,&
     & .2281800E+00,.4082500E+00,.4500300E+00,.4358200E+00,.3035000E+00,&
     & .2307400E+00,.4174900E+00,.4622000E+00,.4493000E+00,.3106400E+00,&
     & .2351000E+00,.4273700E+00,.4743400E+00,.4620200E+00,.3169600E+00,&
     & .2418600E+00,.4378300E+00,.4863700E+00,.4738700E+00,.3226500E+00,&
     & .2512100E+00,.4494300E+00,.4985500E+00,.4851900E+00,.3281600E+00,&
     & .1912000E+00,.3497500E+00,.3873200E+00,.3750900E+00,.2561700E+00,&
     & .1938500E+00,.3582600E+00,.3979900E+00,.3866600E+00,.2625900E+00,&
     & .1984600E+00,.3672700E+00,.4087200E+00,.3975000E+00,.2681500E+00,&
     & .2053700E+00,.3771200E+00,.4195400E+00,.4077700E+00,.2733000E+00,&
     & .2142700E+00,.3886800E+00,.4307100E+00,.4178400E+00,.2785300E+00,&
     & .1598800E+00,.2985400E+00,.3318000E+00,.3220100E+00,.2161100E+00,&
     & .1627900E+00,.3062800E+00,.3412900E+00,.3318800E+00,.2216200E+00,&
     & .1676300E+00,.3146300E+00,.3508500E+00,.3412200E+00,.2266400E+00,&
     & .1744400E+00,.3243200E+00,.3606800E+00,.3503200E+00,.2316100E+00,&
     & .1827400E+00,.3360900E+00,.3714000E+00,.3594400E+00,.2366500E+00,&
     & .1335200E+00,.2540400E+00,.2833000E+00,.2757700E+00,.1822500E+00,&
     & .1366700E+00,.2610300E+00,.2917100E+00,.2843100E+00,.1871200E+00,&
     & .1416400E+00,.2691000E+00,.3003000E+00,.2925400E+00,.1917600E+00,&
     & .1480900E+00,.2788600E+00,.3095900E+00,.3007300E+00,.1963700E+00,&
     & .1555100E+00,.2905000E+00,.3202000E+00,.3093000E+00,.2011500E+00,&
     & .1117700E+00,.2158000E+00,.2413600E+00,.2355200E+00,.1539200E+00,&
     & .1150400E+00,.2224300E+00,.2489000E+00,.2430400E+00,.1583300E+00,&
     & .1197500E+00,.2303100E+00,.2568700E+00,.2504800E+00,.1624400E+00,&
     & .1255300E+00,.2399400E+00,.2658800E+00,.2581200E+00,.1667100E+00,&
     & .1320400E+00,.2510700E+00,.2763900E+00,.2662900E+00,.1711000E+00/

      data   absb(176:350, 7)   /                                       &
     & .9381800E-01,.1833600E+00,.2053300E+00,.2008300E+00,.1306300E+00,&
     & .9715300E-01,.1897500E+00,.2122200E+00,.2075400E+00,.1344500E+00,&
     & .1015700E+00,.1975900E+00,.2198900E+00,.2143900E+00,.1381900E+00,&
     & .1065600E+00,.2069200E+00,.2288500E+00,.2217000E+00,.1420000E+00,&
     & .1121700E+00,.2174100E+00,.2391400E+00,.2297200E+00,.1459500E+00,&
     & .7886400E-01,.1558100E+00,.1746700E+00,.1711100E+00,.1109300E+00,&
     & .8211500E-01,.1621300E+00,.1811400E+00,.1772200E+00,.1144800E+00,&
     & .8609100E-01,.1698900E+00,.1885800E+00,.1836800E+00,.1179500E+00,&
     & .9054400E-01,.1788300E+00,.1974000E+00,.1907500E+00,.1214000E+00,&
     & .9553100E-01,.1885700E+00,.2071900E+00,.1987200E+00,.1249400E+00,&
     & .6653600E-01,.1329300E+00,.1491300E+00,.1461800E+00,.9442900E-01,&
     & .6963900E-01,.1393100E+00,.1553700E+00,.1518700E+00,.9768000E-01,&
     & .7319400E-01,.1468800E+00,.1628200E+00,.1580500E+00,.1008700E+00,&
     & .7720500E-01,.1553600E+00,.1713500E+00,.1650300E+00,.1041400E+00,&
     & .8166700E-01,.1642700E+00,.1806600E+00,.1729100E+00,.1074400E+00,&
     & .5626100E-01,.1138300E+00,.1276000E+00,.1251200E+00,.8055900E-01,&
     & .5908600E-01,.1201400E+00,.1338400E+00,.1304800E+00,.8348500E-01,&
     & .6228100E-01,.1273800E+00,.1412100E+00,.1364800E+00,.8646000E-01,&
     & .6590600E-01,.1351700E+00,.1493700E+00,.1433600E+00,.8946200E-01,&
     & .6999800E-01,.1433500E+00,.1580700E+00,.1509800E+00,.9258800E-01,&
     & .4764200E-01,.9793300E-01,.1096300E+00,.1074200E+00,.6875000E-01,&
     & .5019300E-01,.1039900E+00,.1158700E+00,.1125800E+00,.7143200E-01,&
     & .5308500E-01,.1107700E+00,.1229800E+00,.1184500E+00,.7419400E-01,&
     & .5638400E-01,.1179500E+00,.1307000E+00,.1251200E+00,.7697300E-01,&
     & .6017400E-01,.1255500E+00,.1387900E+00,.1323700E+00,.7992000E-01,&
     & .4043200E-01,.8464500E-01,.9472400E-01,.9263700E-01,.5880500E-01,&
     & .4274800E-01,.9040900E-01,.1007900E+00,.9765400E-01,.6128900E-01,&
     & .4539200E-01,.9668200E-01,.1075600E+00,.1034100E+00,.6381800E-01,&
     & .4843800E-01,.1033500E+00,.1147800E+00,.1097800E+00,.6648100E-01,&
     & .5201900E-01,.1105000E+00,.1223400E+00,.1166400E+00,.6920900E-01,&
     & .3443000E-01,.7356700E-01,.8234900E-01,.8034100E-01,.5048500E-01,&
     & .3655800E-01,.7899700E-01,.8816300E-01,.8529300E-01,.5283500E-01,&
     & .3900900E-01,.8481800E-01,.9452100E-01,.9085100E-01,.5518100E-01,&
     & .4187600E-01,.9107400E-01,.1012700E+00,.9692200E-01,.5764400E-01,&
     & .4529400E-01,.9787400E-01,.1084100E+00,.1034000E+00,.6025800E-01/

      data   absb(351:525, 7)   /                                       &
     & .2941800E-01,.6428900E-01,.7198400E-01,.7011200E-01,.4351400E-01,&
     & .3136700E-01,.6933300E-01,.7750000E-01,.7494500E-01,.4565400E-01,&
     & .3365000E-01,.7477900E-01,.8345100E-01,.8028400E-01,.4783800E-01,&
     & .3637300E-01,.8069800E-01,.8980300E-01,.8603500E-01,.5018800E-01,&
     & .3962500E-01,.8722700E-01,.9660500E-01,.9216300E-01,.5266100E-01,&
     & .2520600E-01,.5643600E-01,.6328200E-01,.6157100E-01,.3760200E-01,&
     & .2700900E-01,.6114700E-01,.6848100E-01,.6623700E-01,.3956200E-01,&
     & .2915000E-01,.6630200E-01,.7407400E-01,.7134100E-01,.4160100E-01,&
     & .3175000E-01,.7195500E-01,.8010400E-01,.7678500E-01,.4379400E-01,&
     & .3483400E-01,.7826800E-01,.8666200E-01,.8263200E-01,.4619300E-01,&
     & .2167200E-01,.4982100E-01,.5598500E-01,.5443100E-01,.3267700E-01,&
     & .2336000E-01,.5425700E-01,.6087100E-01,.5891800E-01,.3445300E-01,&
     & .2542000E-01,.5915900E-01,.6616800E-01,.6376500E-01,.3639800E-01,&
     & .2790800E-01,.6462900E-01,.7193500E-01,.6897700E-01,.3849000E-01,&
     & .3084300E-01,.7079100E-01,.7832300E-01,.7461000E-01,.4074700E-01,&
     & .1870300E-01,.4423600E-01,.4981900E-01,.4842300E-01,.2850300E-01,&
     & .2031500E-01,.4844500E-01,.5442500E-01,.5272000E-01,.3017400E-01,&
     & .2230700E-01,.5316300E-01,.5948100E-01,.5735900E-01,.3201800E-01,&
     & .2468700E-01,.5849500E-01,.6506400E-01,.6236800E-01,.3402200E-01,&
     & .2751000E-01,.6459700E-01,.7135200E-01,.6786000E-01,.3624300E-01,&
     & .1622700E-01,.3954600E-01,.4461300E-01,.4338400E-01,.2494700E-01,&
     & .1779100E-01,.4357600E-01,.4901000E-01,.4750100E-01,.2657700E-01,&
     & .1972000E-01,.4816300E-01,.5387700E-01,.5196200E-01,.2834700E-01,&
     & .2201000E-01,.5341900E-01,.5936800E-01,.5684200E-01,.3032600E-01,&
     & .2477800E-01,.5953300E-01,.6558200E-01,.6227900E-01,.3246400E-01,&
     & .1416800E-01,.3561000E-01,.4022800E-01,.3914500E-01,.2199800E-01,&
     & .1570100E-01,.3951800E-01,.4445500E-01,.4310800E-01,.2355100E-01,&
     & .1756800E-01,.4402800E-01,.4922800E-01,.4744200E-01,.2530300E-01,&
     & .1982200E-01,.4927400E-01,.5464400E-01,.5226900E-01,.2721900E-01,&
     & .2249200E-01,.5542900E-01,.6087800E-01,.5769600E-01,.2928900E-01,&
     & .1246900E-01,.3232100E-01,.3654800E-01,.3558600E-01,.1953500E-01,&
     & .1396700E-01,.3616000E-01,.4067000E-01,.3941400E-01,.2105700E-01,&
     & .1580400E-01,.4064600E-01,.4537100E-01,.4369700E-01,.2274800E-01,&
     & .1800700E-01,.4593500E-01,.5079000E-01,.4850400E-01,.2458900E-01,&
     & .2058300E-01,.5211700E-01,.5708300E-01,.5394900E-01,.2661200E-01/

      data   absb(526:700, 7)   /                                       &
     & .1103000E-01,.2946200E-01,.3333300E-01,.3245700E-01,.1744100E-01,&
     & .1248500E-01,.3325200E-01,.3737400E-01,.3620300E-01,.1890200E-01,&
     & .1428400E-01,.3773500E-01,.4203900E-01,.4044900E-01,.2053300E-01,&
     & .1640600E-01,.4305000E-01,.4747300E-01,.4525000E-01,.2231500E-01,&
     & .1888900E-01,.4922100E-01,.5383800E-01,.5070900E-01,.2429500E-01,&
     & .9704500E-02,.2676500E-01,.3029400E-01,.2949400E-01,.1552700E-01,&
     & .1111600E-01,.3047100E-01,.3423700E-01,.3315600E-01,.1693600E-01,&
     & .1283300E-01,.3490800E-01,.3882700E-01,.3733500E-01,.1849400E-01,&
     & .1485100E-01,.4016000E-01,.4422800E-01,.4209500E-01,.2023000E-01,&
     & .1722000E-01,.4627500E-01,.5060000E-01,.4752400E-01,.2213700E-01,&
     & .8478700E-02,.2413100E-01,.2733900E-01,.2661900E-01,.1375600E-01,&
     & .9803300E-02,.2770600E-01,.3114700E-01,.3016600E-01,.1510600E-01,&
     & .1140700E-01,.3201500E-01,.3561000E-01,.3423900E-01,.1659300E-01,&
     & .1330200E-01,.3714000E-01,.4090700E-01,.3890400E-01,.1825700E-01,&
     & .1553700E-01,.4312900E-01,.4718200E-01,.4423600E-01,.2007400E-01,&
     & .7268500E-02,.2139200E-01,.2430100E-01,.2367900E-01,.1208700E-01,&
     & .8473300E-02,.2475900E-01,.2789800E-01,.2704600E-01,.1334200E-01,&
     & .9935300E-02,.2884800E-01,.3215200E-01,.3094000E-01,.1474800E-01,&
     & .1167300E-01,.3373400E-01,.3722400E-01,.3542200E-01,.1631900E-01,&
     & .1374000E-01,.3948100E-01,.4325600E-01,.4056600E-01,.1804900E-01,&
     & .6232600E-02,.1899900E-01,.2164400E-01,.2110300E-01,.1062900E-01,&
     & .7326900E-02,.2217000E-01,.2504200E-01,.2429900E-01,.1180700E-01,&
     & .8658600E-02,.2604500E-01,.2909800E-01,.2801900E-01,.1313800E-01,&
     & .1025400E-01,.3069900E-01,.3395200E-01,.3231600E-01,.1462100E-01,&
     & .1216800E-01,.3621100E-01,.3974800E-01,.3728800E-01,.1626300E-01,&
     & .5352200E-02,.1692500E-01,.1933700E-01,.1886300E-01,.9369700E-02,&
     & .6343700E-02,.1991600E-01,.2255500E-01,.2190000E-01,.1047700E-01,&
     & .7560000E-02,.2359000E-01,.2642800E-01,.2545300E-01,.1173200E-01,&
     & .9027700E-02,.2802700E-01,.3108100E-01,.2958100E-01,.1314000E-01,&
     & .1080400E-01,.3332500E-01,.3665500E-01,.3439600E-01,.1471400E-01,&
     & .4518300E-02,.1486200E-01,.1704300E-01,.1664000E-01,.8190600E-02,&
     & .5396400E-02,.1763000E-01,.2004300E-01,.1948200E-01,.9233500E-02,&
     & .6483500E-02,.2104100E-01,.2367400E-01,.2282100E-01,.1040000E-01,&
     & .7810000E-02,.2519700E-01,.2805600E-01,.2673300E-01,.1171800E-01,&
     & .9431000E-02,.3020300E-01,.3332200E-01,.3132100E-01,.1320900E-01/

      data   absb(701:875, 7)   /                                       &
     & .3802000E-02,.1303500E-01,.1500700E-01,.1466500E-01,.7164500E-02,&
     & .4579600E-02,.1559200E-01,.1780200E-01,.1732000E-01,.8133200E-02,&
     & .5546400E-02,.1874900E-01,.2119100E-01,.2044800E-01,.9231200E-02,&
     & .6740800E-02,.2262900E-01,.2530900E-01,.2414300E-01,.1045800E-01,&
     & .8215500E-02,.2734700E-01,.3027700E-01,.2850600E-01,.1187100E-01,&
     & .3196200E-02,.1144400E-01,.1322900E-01,.1293700E-01,.6278800E-02,&
     & .3881500E-02,.1379800E-01,.1582200E-01,.1540700E-01,.7177400E-02,&
     & .4742200E-02,.1672200E-01,.1899000E-01,.1833900E-01,.8198600E-02,&
     & .5815400E-02,.2034200E-01,.2285700E-01,.2183000E-01,.9359400E-02,&
     & .7154600E-02,.2478700E-01,.2754500E-01,.2597300E-01,.1068400E-01,&
     & .2654200E-02,.9942100E-02,.1154500E-01,.1129700E-01,.5468100E-02,&
     & .3246700E-02,.1208500E-01,.1392600E-01,.1357500E-01,.6301700E-02,&
     & .4003700E-02,.1475800E-01,.1684800E-01,.1629600E-01,.7243400E-02,&
     & .4953400E-02,.1809400E-01,.2043800E-01,.1955600E-01,.8335400E-02,&
     & .6152500E-02,.2223000E-01,.2481700E-01,.2345000E-01,.9575100E-02,&
     & .2188100E-02,.8585700E-02,.1001900E-01,.9804300E-02,.4745600E-02,&
     & .2693700E-02,.1052500E-01,.1219200E-01,.1189800E-01,.5516800E-02,&
     & .3353100E-02,.1295200E-01,.1487000E-01,.1440800E-01,.6393800E-02,&
     & .4191600E-02,.1601600E-01,.1818700E-01,.1744100E-01,.7399500E-02,&
     & .5252900E-02,.1982500E-01,.2224500E-01,.2107400E-01,.8569300E-02,&
     & .1797400E-02,.7403600E-02,.8681900E-02,.8494900E-02,.4110000E-02,&
     & .2228300E-02,.9155600E-02,.1066400E-01,.1041400E-01,.4825700E-02,&
     & .2799300E-02,.1136000E-01,.1311500E-01,.1273100E-01,.5642300E-02,&
     & .3535800E-02,.1416400E-01,.1617000E-01,.1554300E-01,.6586700E-02,&
     & .4476200E-02,.1767100E-01,.1992800E-01,.1893100E-01,.7672500E-02,&
     & .1466600E-02,.6344300E-02,.7471000E-02,.7309900E-02,.3554000E-02,&
     & .1828600E-02,.7914000E-02,.9271900E-02,.9058800E-02,.4198100E-02,&
     & .2319600E-02,.9908900E-02,.1150800E-01,.1119200E-01,.4965600E-02,&
     & .2957500E-02,.1245100E-01,.1429600E-01,.1377900E-01,.5844100E-02,&
     & .3784200E-02,.1566200E-01,.1775800E-01,.1692100E-01,.6866000E-02,&
     & .1182200E-02,.5370000E-02,.6348300E-02,.6209000E-02,.3050000E-02,&
     & .1481700E-02,.6761600E-02,.7967800E-02,.7787400E-02,.3631700E-02,&
     & .1891800E-02,.8539200E-02,.9981800E-02,.9725400E-02,.4324800E-02,&
     & .2438200E-02,.1082000E-01,.1250300E-01,.1208800E-01,.5147700E-02,&
     & .3153300E-02,.1372500E-01,.1565600E-01,.1497000E-01,.6100000E-02/

      data   absb(876:1050, 7)   /                                      &
     & .9490600E-03,.4529800E-02,.5370600E-02,.5250600E-02,.2616200E-02,&
     & .1196500E-02,.5759700E-02,.6821300E-02,.6669800E-02,.3140300E-02,&
     & .1536900E-02,.7338000E-02,.8636800E-02,.8424600E-02,.3768700E-02,&
     & .2000800E-02,.9381400E-02,.1091100E-01,.1058000E-01,.4521300E-02,&
     & .2615900E-02,.1200200E-01,.1377400E-01,.1322100E-01,.5414200E-02,&
     & .7591300E-03,.3804300E-02,.4520300E-02,.4418100E-02,.2237900E-02,&
     & .9609700E-03,.4889900E-02,.5813800E-02,.5687800E-02,.2709800E-02,&
     & .1244700E-02,.6289500E-02,.7450100E-02,.7272700E-02,.3285900E-02,&
     & .1632200E-02,.8112200E-02,.9498900E-02,.9234900E-02,.3972000E-02,&
     & .2159700E-02,.1047000E-01,.1209200E-01,.1165000E-01,.4793800E-02,&
     & .6094900E-03,.3200900E-02,.3808000E-02,.3720400E-02,.1914800E-02,&
     & .7733200E-03,.4163700E-02,.4965800E-02,.4858800E-02,.2349800E-02,&
     & .1009900E-02,.5409500E-02,.6441600E-02,.6293800E-02,.2871300E-02,&
     & .1335300E-02,.7041700E-02,.8300400E-02,.8084400E-02,.3511400E-02,&
     & .1787300E-02,.9169100E-02,.1065800E-01,.1030300E-01,.4264100E-02,&
     & .4896000E-03,.2687300E-02,.3200800E-02,.3125900E-02,.1637800E-02,&
     & .6218700E-03,.3542300E-02,.4234900E-02,.4144300E-02,.2032500E-02,&
     & .8178800E-03,.4652900E-02,.5565600E-02,.5442400E-02,.2516400E-02,&
     & .1093500E-02,.6115500E-02,.7256500E-02,.7076500E-02,.3100700E-02,&
     & .1477600E-02,.8041600E-02,.9408700E-02,.9119700E-02,.3808500E-02,&
     & .3930200E-03,.2243500E-02,.2675400E-02,.2612200E-02,.1389800E-02,&
     & .4988900E-03,.3000600E-02,.3594300E-02,.3517200E-02,.1753200E-02,&
     & .6592200E-03,.3990600E-02,.4790100E-02,.4688000E-02,.2198000E-02,&
     & .8907500E-03,.5298000E-02,.6323700E-02,.6174800E-02,.2741200E-02,&
     & .1215000E-02,.7031200E-02,.8283700E-02,.8049700E-02,.3395600E-02,&
     & .3153300E-03,.1859800E-02,.2221600E-02,.2169000E-02,.1170500E-02,&
     & .3991900E-03,.2529000E-02,.3033800E-02,.2967900E-02,.1505100E-02,&
     & .5295800E-03,.3409600E-02,.4103000E-02,.4018900E-02,.1913700E-02,&
     & .7214700E-03,.4575600E-02,.5489400E-02,.5367800E-02,.2414700E-02,&
     & .9956300E-03,.6132100E-02,.7273900E-02,.7082600E-02,.3028600E-02,&
     & .2543400E-03,.1547700E-02,.1850200E-02,.1807500E-02,.9889100E-03,&
     & .3222100E-03,.2139800E-02,.2570800E-02,.2514800E-02,.1293500E-02,&
     & .4277500E-03,.2927200E-02,.3531500E-02,.3459900E-02,.1674100E-02,&
     & .5876200E-03,.3976100E-02,.4790700E-02,.4689600E-02,.2137600E-02,&
     & .8205300E-03,.5382400E-02,.6424400E-02,.6266400E-02,.2717300E-02/

      data   absb(1051:1175, 7)   /                                     &
     & .2061800E-03,.1286600E-02,.1540000E-02,.1505500E-02,.8357200E-03,&
     & .2605900E-03,.1809400E-02,.2178700E-02,.2130200E-02,.1109000E-02,&
     & .3465100E-03,.2514000E-02,.3039200E-02,.2978000E-02,.1462800E-02,&
     & .4788200E-03,.3460800E-02,.4183100E-02,.4100300E-02,.1899200E-02,&
     & .6767100E-03,.4733100E-02,.5681700E-02,.5550300E-02,.2436000E-02,&
     & .1668100E-03,.1062100E-02,.1272800E-02,.1246100E-02,.7016200E-03,&
     & .2106400E-03,.1521300E-02,.1835000E-02,.1794500E-02,.9482800E-03,&
     & .2802800E-03,.2150000E-02,.2604500E-02,.2552100E-02,.1272100E-02,&
     & .3890600E-03,.3002900E-02,.3638700E-02,.3571000E-02,.1683300E-02,&
     & .5556800E-03,.4153400E-02,.5011100E-02,.4902700E-02,.2184900E-02,&
     & .1348400E-03,.8716800E-03,.1044400E-02,.1024800E-02,.5851600E-03,&
     & .1700700E-03,.1271700E-02,.1536000E-02,.1502400E-02,.8080500E-03,&
     & .2260300E-03,.1829600E-02,.2221100E-02,.2175700E-02,.1100600E-02,&
     & .3156500E-03,.2596300E-02,.3153100E-02,.3095900E-02,.1485100E-02,&
     & .4541700E-03,.3636500E-02,.4405800E-02,.4315700E-02,.1962500E-02,&
     & .1093800E-03,.7144900E-03,.8565500E-03,.8424000E-03,.4877200E-03,&
     & .1375200E-03,.1062400E-02,.1284900E-02,.1258300E-02,.6868900E-03,&
     & .1830700E-03,.1557200E-02,.1895400E-02,.1856600E-02,.9534400E-03,&
     & .2567300E-03,.2248100E-02,.2736300E-02,.2687700E-02,.1307900E-02,&
     & .3723000E-03,.3192100E-02,.3881200E-02,.3807000E-02,.1757000E-02,&
     & .9215300E-04,.6279800E-03,.7537400E-03,.7424300E-03,.4316100E-03,&
     & .1167600E-03,.9501900E-03,.1152500E-02,.1129600E-02,.6195900E-03,&
     & .1568700E-03,.1416800E-02,.1728500E-02,.1693800E-02,.8746800E-03,&
     & .2227800E-03,.2074300E-02,.2530200E-02,.2486300E-02,.1216100E-02,&
     & .3272400E-03,.2979700E-02,.3631300E-02,.3566400E-02,.1657700E-02/

      data   absb(  1:175, 8)   /                                       &
     & .2147772E+01,.3506523E+01,.4007672E+01,.4063247E+01,.2937459E+01,&
     & .2178726E+01,.3529383E+01,.4034995E+01,.4103957E+01,.3003216E+01,&
     & .2233362E+01,.3568760E+01,.4070770E+01,.4144550E+01,.3061714E+01,&
     & .2307004E+01,.3623393E+01,.4115151E+01,.4185941E+01,.3115511E+01,&
     & .2388633E+01,.3691850E+01,.4168630E+01,.4227592E+01,.3167839E+01,&
     & .1860982E+01,.3098016E+01,.3558079E+01,.3627319E+01,.2627403E+01,&
     & .1905280E+01,.3134646E+01,.3595375E+01,.3671395E+01,.2687036E+01,&
     & .1969054E+01,.3186559E+01,.3641361E+01,.3716596E+01,.2741361E+01,&
     & .2041391E+01,.3252536E+01,.3696281E+01,.3763063E+01,.2795119E+01,&
     & .2116579E+01,.3332677E+01,.3758858E+01,.3812101E+01,.2847416E+01,&
     & .1617458E+01,.2741093E+01,.3152632E+01,.3221428E+01,.2339877E+01,&
     & .1670160E+01,.2788238E+01,.3198512E+01,.3269657E+01,.2396388E+01,&
     & .1732677E+01,.2849720E+01,.3253678E+01,.3319736E+01,.2449246E+01,&
     & .1797925E+01,.2926437E+01,.3317317E+01,.3372271E+01,.2501129E+01,&
     & .1862880E+01,.3012958E+01,.3388595E+01,.3427980E+01,.2552877E+01,&
     & .1410490E+01,.2430273E+01,.2789567E+01,.2851573E+01,.2070884E+01,&
     & .1462546E+01,.2486398E+01,.2843042E+01,.2903268E+01,.2125260E+01,&
     & .1517639E+01,.2557586E+01,.2906093E+01,.2957563E+01,.2178494E+01,&
     & .1573385E+01,.2638614E+01,.2977318E+01,.3015588E+01,.2231226E+01,&
     & .1630346E+01,.2723778E+01,.3055751E+01,.3077711E+01,.2281963E+01,&
     & .1230541E+01,.2156310E+01,.2468500E+01,.2517143E+01,.1827564E+01,&
     & .1276117E+01,.2220980E+01,.2528907E+01,.2572462E+01,.1880843E+01,&
     & .1322689E+01,.2296509E+01,.2597740E+01,.2631249E+01,.1932799E+01,&
     & .1370777E+01,.2376996E+01,.2674941E+01,.2694611E+01,.1983918E+01,&
     & .1421644E+01,.2457551E+01,.2758874E+01,.2763587E+01,.2034867E+01,&
     & .1070969E+01,.1915020E+01,.2186826E+01,.2218812E+01,.1611091E+01,&
     & .1109526E+01,.1984117E+01,.2252013E+01,.2276826E+01,.1661713E+01,&
     & .1149568E+01,.2059351E+01,.2326295E+01,.2340036E+01,.1710989E+01,&
     & .1192105E+01,.2135358E+01,.2407183E+01,.2409247E+01,.1761075E+01,&
     & .1238203E+01,.2211009E+01,.2491678E+01,.2482619E+01,.1811988E+01,&
     & .9283756E+00,.1702510E+01,.1938601E+01,.1955270E+01,.1419882E+01,&
     & .9616580E+00,.1771777E+01,.2008679E+01,.2016979E+01,.1468037E+01,&
     & .9971495E+00,.1842985E+01,.2085977E+01,.2084514E+01,.1515441E+01,&
     & .1035954E+01,.1914171E+01,.2167568E+01,.2156971E+01,.1563415E+01,&
     & .1079188E+01,.1986694E+01,.2252099E+01,.2233773E+01,.1612851E+01/

      data   absb(176:350, 8)   /                                       &
     & .8036031E+00,.1516186E+01,.1721627E+01,.1727213E+01,.1252674E+01,&
     & .8326793E+00,.1581949E+01,.1794758E+01,.1792292E+01,.1298932E+01,&
     & .8648349E+00,.1648188E+01,.1873088E+01,.1862805E+01,.1346221E+01,&
     & .9010472E+00,.1716012E+01,.1954493E+01,.1937771E+01,.1392209E+01,&
     & .9422352E+00,.1787156E+01,.2036790E+01,.2017682E+01,.1438445E+01,&
     & .6953625E+00,.1350659E+01,.1532717E+01,.1530267E+01,.1104610E+01,&
     & .7214726E+00,.1411984E+01,.1606492E+01,.1597505E+01,.1150564E+01,&
     & .7509696E+00,.1474817E+01,.1683945E+01,.1669864E+01,.1196729E+01,&
     & .7851546E+00,.1540757E+01,.1763201E+01,.1747500E+01,.1242539E+01,&
     & .8247478E+00,.1610749E+01,.1844223E+01,.1828234E+01,.1287882E+01,&
     & .6026917E+00,.1207364E+01,.1372521E+01,.1364577E+01,.9775652E+00,&
     & .6273178E+00,.1265727E+01,.1445916E+01,.1433870E+01,.1022600E+01,&
     & .6557573E+00,.1326639E+01,.1521041E+01,.1508141E+01,.1067263E+01,&
     & .6889575E+00,.1391759E+01,.1598467E+01,.1586453E+01,.1112837E+01,&
     & .7278457E+00,.1461987E+01,.1679430E+01,.1668167E+01,.1158214E+01,&
     & .5220814E+00,.1081272E+01,.1234021E+01,.1222809E+01,.8668184E+00,&
     & .5462161E+00,.1137925E+01,.1304901E+01,.1293790E+01,.9111319E+00,&
     & .5743245E+00,.1198422E+01,.1378329E+01,.1368895E+01,.9551067E+00,&
     & .6074624E+00,.1263493E+01,.1455169E+01,.1447720E+01,.9999031E+00,&
     & .6464214E+00,.1334732E+01,.1536800E+01,.1530362E+01,.1044710E+01,&
     & .4530676E+00,.9718752E+00,.1113625E+01,.1102459E+01,.7702607E+00,&
     & .4767827E+00,.1027819E+01,.1182580E+01,.1173836E+01,.8131121E+00,&
     & .5050006E+00,.1088357E+01,.1255221E+01,.1249573E+01,.8570364E+00,&
     & .5385629E+00,.1154487E+01,.1332423E+01,.1329078E+01,.9009956E+00,&
     & .5782279E+00,.1226660E+01,.1415277E+01,.1412212E+01,.9457995E+00,&
     & .3944211E+00,.8783220E+00,.1009831E+01,.1000358E+01,.6870193E+00,&
     & .4180031E+00,.9342002E+00,.1077916E+01,.1071914E+01,.7295172E+00,&
     & .4466039E+00,.9953298E+00,.1150465E+01,.1147983E+01,.7723351E+00,&
     & .4807452E+00,.1062397E+01,.1228665E+01,.1227974E+01,.8161121E+00,&
     & .5212184E+00,.1135828E+01,.1313016E+01,.1312663E+01,.8641655E+00,&
     & .3452462E+00,.7999293E+00,.9219804E+00,.9149658E+00,.6174465E+00,&
     & .3690573E+00,.8557897E+00,.9899741E+00,.9866927E+00,.6583574E+00,&
     & .3981461E+00,.9176898E+00,.1063178E+01,.1062821E+01,.7012026E+00,&
     & .4331021E+00,.9859104E+00,.1142768E+01,.1144162E+01,.7470311E+00,&
     & .4742396E+00,.1061101E+01,.1229416E+01,.1231158E+01,.7973609E+00/

      data   absb(351:525, 8)   /                                       &
     & .3040686E+00,.7336469E+00,.8476943E+00,.8430763E+00,.5574518E+00,&
     & .3283125E+00,.7901985E+00,.9160319E+00,.9148169E+00,.5982934E+00,&
     & .3580998E+00,.8530076E+00,.9905938E+00,.9917986E+00,.6421955E+00,&
     & .3937998E+00,.9226604E+00,.1072205E+01,.1075289E+01,.6908810E+00,&
     & .4350420E+00,.1000013E+01,.1161544E+01,.1165076E+01,.7420692E+00,&
     & .2699261E+00,.6782542E+00,.7854145E+00,.7829386E+00,.5064205E+00,&
     & .2947504E+00,.7357341E+00,.8549126E+00,.8551804E+00,.5487324E+00,&
     & .3252847E+00,.7997113E+00,.9311639E+00,.9339914E+00,.5953021E+00,&
     & .3613092E+00,.8712485E+00,.1015309E+01,.1020027E+01,.6445334E+00,&
     & .4023145E+00,.9510193E+00,.1107717E+01,.1113072E+01,.6967532E+00,&
     & .2420005E+00,.6327790E+00,.7343674E+00,.7337147E+00,.4659044E+00,&
     & .2675426E+00,.6914513E+00,.8056098E+00,.8073577E+00,.5100332E+00,&
     & .2985977E+00,.7570808E+00,.8841869E+00,.8886282E+00,.5568586E+00,&
     & .3345903E+00,.8307955E+00,.9712488E+00,.9777164E+00,.6069869E+00,&
     & .3752790E+00,.9132888E+00,.1067298E+01,.1074521E+01,.6606337E+00,&
     & .2193104E+00,.5956904E+00,.6933100E+00,.6941123E+00,.4338918E+00,&
     & .2454356E+00,.6558867E+00,.7665763E+00,.7700794E+00,.4783097E+00,&
     & .2766892E+00,.7234267E+00,.8479414E+00,.8542122E+00,.5258379E+00,&
     & .3125185E+00,.7996635E+00,.9383927E+00,.9468621E+00,.5771763E+00,&
     & .3530433E+00,.8851441E+00,.1038330E+01,.1047465E+01,.6323351E+00,&
     & .2011174E+00,.5662868E+00,.6614813E+00,.6638455E+00,.4081986E+00,&
     & .2276405E+00,.6282768E+00,.7371843E+00,.7426597E+00,.4531121E+00,&
     & .2588601E+00,.6982349E+00,.8218536E+00,.8302250E+00,.5017038E+00,&
     & .2946400E+00,.7772718E+00,.9161532E+00,.9266575E+00,.5545280E+00,&
     & .3351557E+00,.8661253E+00,.1020315E+01,.1031181E+01,.6114083E+00,&
     & .1865615E+00,.5436250E+00,.6378358E+00,.6421123E+00,.3878649E+00,&
     & .2133106E+00,.6078013E+00,.7166414E+00,.7241786E+00,.4336769E+00,&
     & .2445042E+00,.6804884E+00,.8049500E+00,.8154355E+00,.4837168E+00,&
     & .2803835E+00,.7627085E+00,.9033997E+00,.9157583E+00,.5381867E+00,&
     & .3210362E+00,.8552279E+00,.1011978E+01,.1024218E+01,.5970880E+00,&
     & .1749538E+00,.5269996E+00,.6215976E+00,.6279908E+00,.3723894E+00,&
     & .2017790E+00,.5937199E+00,.7039645E+00,.7136463E+00,.4194095E+00,&
     & .2331437E+00,.6694912E+00,.7962787E+00,.8088205E+00,.4711663E+00,&
     & .2692141E+00,.7551867E+00,.8989845E+00,.9129664E+00,.5275761E+00,&
     & .3102376E+00,.8516644E+00,.1011909E+01,.1025210E+01,.5885594E+00/

      data   absb(526:700, 8)   /                                       &
     & .1648200E+00,.5135795E+00,.6093601E+00,.6178369E+00,.3595625E+00,&
     & .1916764E+00,.5828141E+00,.6952491E+00,.7068257E+00,.4080742E+00,&
     & .2231330E+00,.6615194E+00,.7912997E+00,.8055439E+00,.4615211E+00,&
     & .2594329E+00,.7506484E+00,.8977476E+00,.9130454E+00,.5197857E+00,&
     & .3009092E+00,.8509191E+00,.1014517E+01,.1028676E+01,.5827702E+00,&
     & .1542942E+00,.4984661E+00,.5950172E+00,.6050995E+00,.3460966E+00,&
     & .1808778E+00,.5696361E+00,.6834611E+00,.6965516E+00,.3958165E+00,&
     & .2120868E+00,.6504539E+00,.7821549E+00,.7977925E+00,.4505357E+00,&
     & .2483187E+00,.7421169E+00,.8913076E+00,.9077047E+00,.5102050E+00,&
     & .2900951E+00,.8450562E+00,.1010731E+01,.1025679E+01,.5747009E+00,&
     & .1428838E+00,.4799142E+00,.5760617E+00,.5872644E+00,.3307220E+00,&
     & .1688185E+00,.5520986E+00,.6658687E+00,.6800494E+00,.3810810E+00,&
     & .1993677E+00,.6340252E+00,.7660136E+00,.7826397E+00,.4366263E+00,&
     & .2351442E+00,.7269414E+00,.8765519E+00,.8938786E+00,.4971268E+00,&
     & .2767765E+00,.8312265E+00,.9973781E+00,.1013193E+01,.5625664E+00,&
     & .1295255E+00,.4542258E+00,.5478587E+00,.5595381E+00,.3107326E+00,&
     & .1542935E+00,.5261193E+00,.6373454E+00,.6520435E+00,.3608896E+00,&
     & .1835749E+00,.6075514E+00,.7372141E+00,.7544331E+00,.4164074E+00,&
     & .2182062E+00,.6998975E+00,.8474540E+00,.8656087E+00,.4768929E+00,&
     & .2589005E+00,.8036978E+00,.9679606E+00,.9848442E+00,.5424313E+00,&
     & .1175401E+00,.4307071E+00,.5219632E+00,.5338978E+00,.2926245E+00,&
     & .1412629E+00,.5022212E+00,.6108981E+00,.6259401E+00,.3424732E+00,&
     & .1693284E+00,.5830370E+00,.7102630E+00,.7278824E+00,.3978669E+00,&
     & .2028442E+00,.6746033E+00,.8199129E+00,.8387792E+00,.4582588E+00,&
     & .2426035E+00,.7776740E+00,.9399588E+00,.9577662E+00,.5237233E+00,&
     & .1069057E+00,.4095696E+00,.4986606E+00,.5107070E+00,.2765051E+00,&
     & .1297183E+00,.4806709E+00,.5869321E+00,.6022485E+00,.3260232E+00,&
     & .1566724E+00,.5608628E+00,.6856622E+00,.7036230E+00,.3812308E+00,&
     & .1891795E+00,.6516100E+00,.7946672E+00,.8140707E+00,.4414800E+00,&
     & .2280943E+00,.7538152E+00,.9140427E+00,.9326912E+00,.5067756E+00,&
     & .9538147E-01,.3836728E+00,.4692157E+00,.4807161E+00,.2573361E+00,&
     & .1169920E+00,.4534590E+00,.5555647E+00,.5707684E+00,.3058432E+00,&
     & .1425205E+00,.5320323E+00,.6525202E+00,.6704208E+00,.3602199E+00,&
     & .1734620E+00,.6207733E+00,.7596824E+00,.7794089E+00,.4197447E+00,&
     & .2108876E+00,.7208589E+00,.8772631E+00,.8967522E+00,.4842904E+00/

      data   absb(701:875, 8)   /                                       &
     & .8483947E-01,.3588563E+00,.4408601E+00,.4516281E+00,.2391711E+00,&
     & .1052687E+00,.4273398E+00,.5251982E+00,.5400986E+00,.2865821E+00,&
     & .1294958E+00,.5040617E+00,.6199729E+00,.6376780E+00,.3398708E+00,&
     & .1588835E+00,.5906162E+00,.7250743E+00,.7449273E+00,.3985409E+00,&
     & .1946890E+00,.6883248E+00,.8405775E+00,.8607468E+00,.4621965E+00,&
     & .7543551E-01,.3358861E+00,.4145122E+00,.4243832E+00,.2224114E+00,&
     & .9465525E-01,.4026844E+00,.4964574E+00,.5108780E+00,.2686287E+00,&
     & .1177290E+00,.4776992E+00,.5890965E+00,.6064776E+00,.3207812E+00,&
     & .1456605E+00,.5620963E+00,.6920814E+00,.7118635E+00,.3784909E+00,&
     & .1798767E+00,.6573786E+00,.8054007E+00,.8260421E+00,.4411895E+00,&
     & .6608792E-01,.3109105E+00,.3855264E+00,.3941304E+00,.2044580E+00,&
     & .8394591E-01,.3756301E+00,.4645764E+00,.4780384E+00,.2491451E+00,&
     & .1057032E+00,.4484352E+00,.5543433E+00,.5711787E+00,.2997657E+00,&
     & .1320096E+00,.5302167E+00,.6546096E+00,.6740082E+00,.3560965E+00,&
     & .1642833E+00,.6224082E+00,.7650984E+00,.7859469E+00,.4175347E+00,&
     & .5738271E-01,.2860275E+00,.3565059E+00,.3636546E+00,.1867469E+00,&
     & .7381553E-01,.3484044E+00,.4324451E+00,.4446271E+00,.2297128E+00,&
     & .9417454E-01,.4188318E+00,.5189475E+00,.5351023E+00,.2785779E+00,&
     & .1189809E+00,.4980563E+00,.6163102E+00,.6351253E+00,.3333660E+00,&
     & .1491626E+00,.5867414E+00,.7235000E+00,.7442712E+00,.3933218E+00,&
     & .4966189E-01,.2627287E+00,.3290966E+00,.3348944E+00,.1702478E+00,&
     & .6472660E-01,.3227122E+00,.4021344E+00,.4129120E+00,.2114833E+00,&
     & .8369235E-01,.3907371E+00,.4852534E+00,.5005355E+00,.2585650E+00,&
     & .1070502E+00,.4673220E+00,.5794833E+00,.5976216E+00,.3116401E+00,&
     & .1353378E+00,.5528051E+00,.6835756E+00,.7040363E+00,.3701096E+00,&
     & .4256723E-01,.2397851E+00,.3016717E+00,.3062819E+00,.1540481E+00,&
     & .5622816E-01,.2970953E+00,.3718496E+00,.3810025E+00,.1934507E+00,&
     & .7377185E-01,.3626587E+00,.4515842E+00,.4655582E+00,.2385992E+00,&
     & .9559465E-01,.4363178E+00,.5420972E+00,.5594484E+00,.2897050E+00,&
     & .1220368E+00,.5185878E+00,.6428277E+00,.6627182E+00,.3464809E+00,&
     & .3589415E-01,.2161673E+00,.2729648E+00,.2765338E+00,.1375028E+00,&
     & .4805408E-01,.2703938E+00,.3401678E+00,.3474504E+00,.1747652E+00,&
     & .6397943E-01,.3329344E+00,.4160369E+00,.4283381E+00,.2177681E+00,&
     & .8411883E-01,.4035911E+00,.5024152E+00,.5187692E+00,.2666244E+00,&
     & .1087145E+00,.4824389E+00,.5992777E+00,.6183786E+00,.3213119E+00/

      data   absb(876:1050, 8)   /                                      &
     & .3011941E-01,.1941366E+00,.2458479E+00,.2487746E+00,.1221988E+00,&
     & .4084506E-01,.2453806E+00,.3101120E+00,.3157807E+00,.1572649E+00,&
     & .5518255E-01,.3048452E+00,.3825091E+00,.3929551E+00,.1981390E+00,&
     & .7368328E-01,.3724217E+00,.4646897E+00,.4797945E+00,.2447415E+00,&
     & .9652937E-01,.4480322E+00,.5576115E+00,.5757517E+00,.2972212E+00,&
     & .2515607E-01,.1736587E+00,.2203553E+00,.2229593E+00,.1080693E+00,&
     & .3453399E-01,.2219693E+00,.2815682E+00,.2859887E+00,.1409424E+00,&
     & .4732652E-01,.2782830E+00,.3508111E+00,.3593155E+00,.1796231E+00,&
     & .6419246E-01,.3427924E+00,.4290016E+00,.4424221E+00,.2240016E+00,&
     & .8533710E-01,.4152474E+00,.5177134E+00,.5349045E+00,.2742416E+00,&
     & .2108838E-01,.1557171E+00,.1978594E+00,.2004336E+00,.9577831E-01,&
     & .2927578E-01,.2012690E+00,.2560716E+00,.2596510E+00,.1265930E+00,&
     & .4069811E-01,.2546968E+00,.3225058E+00,.3293032E+00,.1632049E+00,&
     & .5604639E-01,.3162711E+00,.3971951E+00,.4089321E+00,.2055380E+00,&
     & .7561968E-01,.3858696E+00,.4819614E+00,.4980595E+00,.2536384E+00,&
     & .1767518E-01,.1394513E+00,.1774120E+00,.1800232E+00,.8475686E-01,&
     & .2479537E-01,.1824043E+00,.2325611E+00,.2356543E+00,.1136169E+00,&
     & .3495024E-01,.2330902E+00,.2962474E+00,.3017090E+00,.1481769E+00,&
     & .4886078E-01,.2917985E+00,.3678777E+00,.3778687E+00,.1885266E+00,&
     & .6700377E-01,.3587323E+00,.4490277E+00,.4637480E+00,.2345858E+00,&
     & .1476895E-01,.1243742E+00,.1583798E+00,.1609596E+00,.7459372E-01,&
     & .2091694E-01,.1647602E+00,.2103847E+00,.2132485E+00,.1015375E+00,&
     & .2988238E-01,.2127462E+00,.2712282E+00,.2756073E+00,.1340822E+00,&
     & .4238686E-01,.2685463E+00,.3400097E+00,.3482087E+00,.1724055E+00,&
     & .5904346E-01,.3326286E+00,.4175653E+00,.4306952E+00,.2164369E+00,&
     & .1230374E-01,.1104196E+00,.1406329E+00,.1430549E+00,.6528664E-01,&
     & .1757050E-01,.1482205E+00,.1894519E+00,.1922843E+00,.9031455E-01,&
     & .2540539E-01,.1934889E+00,.2472646E+00,.2509359E+00,.1208463E+00,&
     & .3658763E-01,.2464677E+00,.3133038E+00,.3198840E+00,.1571069E+00,&
     & .5175125E-01,.3076675E+00,.3875310E+00,.3989625E+00,.1991350E+00,&
     & .1033543E-01,.9851647E-01,.1254426E+00,.1276686E+00,.5744944E-01,&
     & .1487034E-01,.1339448E+00,.1713721E+00,.1742024E+00,.8071988E-01,&
     & .2174783E-01,.1768075E+00,.2263355E+00,.2295753E+00,.1094349E+00,&
     & .3178517E-01,.2272599E+00,.2898005E+00,.2951977E+00,.1438116E+00,&
     & .4561407E-01,.2857740E+00,.3612374E+00,.3710210E+00,.1839585E+00/

      data   absb(1051:1175, 8)   /                                     &
     & .8709412E-02,.8794504E-01,.1119210E+00,.1139728E+00,.5066255E-01,&
     & .1262413E-01,.1212030E+00,.1551537E+00,.1579143E+00,.7214590E-01,&
     & .1865119E-01,.1616922E+00,.2072178E+00,.2102675E+00,.9913095E-01,&
     & .2764278E-01,.2097406E+00,.2681310E+00,.2726069E+00,.1317209E+00,&
     & .4023984E-01,.2656768E+00,.3370690E+00,.3452748E+00,.1700699E+00,&
     & .7309742E-02,.7805313E-01,.9935397E-01,.1011896E+00,.4454450E-01,&
     & .1068078E-01,.1092344E+00,.1398484E+00,.1424507E+00,.6420420E-01,&
     & .1594110E-01,.1474371E+00,.1891070E+00,.1921023E+00,.8951479E-01,&
     & .2393561E-01,.1930891E+00,.2473383E+00,.2511858E+00,.1203039E+00,&
     & .3536639E-01,.2465547E+00,.3138627E+00,.3206321E+00,.1568422E+00,&
     & .6111505E-02,.6882934E-01,.8775334E-01,.8932843E-01,.3912454E-01,&
     & .9015756E-02,.9808206E-01,.1255190E+00,.1279049E+00,.5696209E-01,&
     & .1357531E-01,.1339695E+00,.1719655E+00,.1749408E+00,.8050805E-01,&
     & .2064520E-01,.1772958E+00,.2274651E+00,.2308615E+00,.1095380E+00,&
     & .3095469E-01,.2282972E+00,.2914883E+00,.2970959E+00,.1442483E+00,&
     & .5121725E-02,.6067687E-01,.7763361E-01,.7890932E-01,.3444819E-01,&
     & .7639216E-02,.8816021E-01,.1127621E+00,.1149702E+00,.5070163E-01,&
     & .1160557E-01,.1219458E+00,.1566131E+00,.1595297E+00,.7248142E-01,&
     & .1786695E-01,.1630817E+00,.2094426E+00,.2126006E+00,.9988204E-01,&
     & .2716600E-01,.2117945E+00,.2710344E+00,.2757215E+00,.1328837E+00,&
     & .4599319E-02,.5738031E-01,.7361124E-01,.7479831E-01,.3257284E-01,&
     & .6965718E-02,.8418710E-01,.1077301E+00,.1098721E+00,.4821784E-01,&
     & .1072080E-01,.1171779E+00,.1505705E+00,.1534484E+00,.6933089E-01,&
     & .1668801E-01,.1574612E+00,.2023452E+00,.2054613E+00,.9606507E-01,&
     & .2561291E-01,.2052733E+00,.2629716E+00,.2673724E+00,.1283901E+00/

      data   absb(  1:175, 9)   /                                       &
     & .1351800E+02,.1942600E+02,.2136800E+02,.2060700E+02,.1596700E+02,&
     & .1324900E+02,.1927800E+02,.2134500E+02,.2072500E+02,.1621800E+02,&
     & .1299100E+02,.1910600E+02,.2134400E+02,.2083800E+02,.1649300E+02,&
     & .1278300E+02,.1895300E+02,.2133700E+02,.2095900E+02,.1676900E+02,&
     & .1273500E+02,.1882000E+02,.2132800E+02,.2110400E+02,.1698200E+02,&
     & .1150400E+02,.1753400E+02,.1945700E+02,.1913100E+02,.1462500E+02,&
     & .1128100E+02,.1739600E+02,.1949100E+02,.1927100E+02,.1493900E+02,&
     & .1111200E+02,.1729000E+02,.1952700E+02,.1941500E+02,.1525500E+02,&
     & .1111500E+02,.1721300E+02,.1956000E+02,.1958000E+02,.1550400E+02,&
     & .1134200E+02,.1713900E+02,.1961500E+02,.1975900E+02,.1570600E+02,&
     & .9817000E+01,.1559900E+02,.1762700E+02,.1766600E+02,.1335400E+02,&
     & .9674700E+01,.1553300E+02,.1769500E+02,.1784200E+02,.1368700E+02,&
     & .9698400E+01,.1550200E+02,.1775500E+02,.1803600E+02,.1400400E+02,&
     & .9952200E+01,.1548000E+02,.1782900E+02,.1823500E+02,.1426300E+02,&
     & .1034400E+02,.1552500E+02,.1792900E+02,.1841800E+02,.1447500E+02,&
     & .8433200E+01,.1380600E+02,.1592600E+02,.1621500E+02,.1222400E+02,&
     & .8458300E+01,.1380500E+02,.1602100E+02,.1643700E+02,.1255600E+02,&
     & .8706600E+01,.1382300E+02,.1610800E+02,.1666000E+02,.1283400E+02,&
     & .9082700E+01,.1392400E+02,.1622600E+02,.1688300E+02,.1307800E+02,&
     & .9526300E+01,.1415600E+02,.1638200E+02,.1707900E+02,.1333400E+02,&
     & .7306800E+01,.1225200E+02,.1434100E+02,.1481000E+02,.1112900E+02,&
     & .7529500E+01,.1229100E+02,.1446300E+02,.1505200E+02,.1143900E+02,&
     & .7885300E+01,.1240500E+02,.1460600E+02,.1530900E+02,.1172900E+02,&
     & .8304000E+01,.1264300E+02,.1477600E+02,.1554800E+02,.1200600E+02,&
     & .8712300E+01,.1303900E+02,.1498200E+02,.1576200E+02,.1228600E+02,&
     & .6428700E+01,.1093300E+02,.1287900E+02,.1345100E+02,.1005300E+02,&
     & .6744600E+01,.1103900E+02,.1304100E+02,.1374000E+02,.1037700E+02,&
     & .7127100E+01,.1126600E+02,.1322200E+02,.1401800E+02,.1069900E+02,&
     & .7506700E+01,.1165000E+02,.1345500E+02,.1426300E+02,.1101100E+02,&
     & .7879700E+01,.1214400E+02,.1376600E+02,.1451700E+02,.1132300E+02,&
     & .5716500E+01,.9819800E+01,.1156300E+02,.1221000E+02,.9028700E+01,&
     & .6050400E+01,.1003000E+02,.1175900E+02,.1251100E+02,.9377100E+01,&
     & .6387200E+01,.1039000E+02,.1200400E+02,.1278700E+02,.9738200E+01,&
     & .6726200E+01,.1085800E+02,.1232500E+02,.1307100E+02,.1010000E+02,&
     & .7063000E+01,.1139600E+02,.1272000E+02,.1337100E+02,.1043300E+02/

      data   absb(176:350, 9)   /                                       &
     & .5107700E+01,.8868000E+01,.1041500E+02,.1105300E+02,.8096300E+01,&
     & .5401700E+01,.9210700E+01,.1066200E+02,.1136100E+02,.8475800E+01,&
     & .5701500E+01,.9669800E+01,.1098200E+02,.1166800E+02,.8848100E+01,&
     & .6004100E+01,.1019600E+02,.1138400E+02,.1200000E+02,.9237600E+01,&
     & .6310600E+01,.1073200E+02,.1188400E+02,.1234900E+02,.9625500E+01,&
     & .4541400E+01,.8084100E+01,.9433300E+01,.1000100E+02,.7298700E+01,&
     & .4806000E+01,.8516900E+01,.9739900E+01,.1032400E+02,.7679900E+01,&
     & .5075400E+01,.9031100E+01,.1013400E+02,.1067300E+02,.8069200E+01,&
     & .5348900E+01,.9566000E+01,.1062600E+02,.1104600E+02,.8467100E+01,&
     & .5622600E+01,.1010000E+02,.1118800E+02,.1146700E+02,.8887800E+01,&
     & .4051200E+01,.7453700E+01,.8611300E+01,.9076000E+01,.6640600E+01,&
     & .4286100E+01,.7945500E+01,.8998800E+01,.9432600E+01,.7029800E+01,&
     & .4528400E+01,.8472300E+01,.9480600E+01,.9828200E+01,.7436500E+01,&
     & .4776500E+01,.9002200E+01,.1003600E+02,.1026700E+02,.7853000E+01,&
     & .5027300E+01,.9537500E+01,.1064400E+02,.1076200E+02,.8292500E+01,&
     & .3627200E+01,.6934500E+01,.7929200E+01,.8282100E+01,.6086200E+01,&
     & .3832400E+01,.7441200E+01,.8405800E+01,.8677900E+01,.6481400E+01,&
     & .4049600E+01,.7955600E+01,.8949700E+01,.9127700E+01,.6903100E+01,&
     & .4273600E+01,.8488100E+01,.9551200E+01,.9635300E+01,.7342300E+01,&
     & .4500800E+01,.9023100E+01,.1019900E+02,.1019600E+02,.7810700E+01,&
     & .3261700E+01,.6492700E+01,.7381400E+01,.7620400E+01,.5635100E+01,&
     & .3445500E+01,.6986400E+01,.7914600E+01,.8066800E+01,.6048200E+01,&
     & .3637500E+01,.7498200E+01,.8512400E+01,.8568800E+01,.6479300E+01,&
     & .3836100E+01,.8025300E+01,.9153000E+01,.9138400E+01,.6948800E+01,&
     & .4043400E+01,.8577000E+01,.9824900E+01,.9770100E+01,.7439200E+01,&
     & .2950400E+01,.6099600E+01,.6937800E+01,.7093300E+01,.5269600E+01,&
     & .3113400E+01,.6585900E+01,.7519300E+01,.7587400E+01,.5690800E+01,&
     & .3285000E+01,.7099100E+01,.8158300E+01,.8145500E+01,.6150400E+01,&
     & .3463900E+01,.7635700E+01,.8828900E+01,.8776600E+01,.6639100E+01,&
     & .3657500E+01,.8203100E+01,.9533500E+01,.9472800E+01,.7118100E+01,&
     & .2678500E+01,.5759200E+01,.6594400E+01,.6679300E+01,.4965900E+01,&
     & .2830300E+01,.6251200E+01,.7218000E+01,.7229200E+01,.5415800E+01,&
     & .2984500E+01,.6770000E+01,.7885300E+01,.7851700E+01,.5892900E+01,&
     & .3149800E+01,.7321800E+01,.8589200E+01,.8540500E+01,.6377900E+01,&
     & .3336500E+01,.7909500E+01,.9326000E+01,.9287600E+01,.6859300E+01/

      data   absb(351:525, 9)   /                                       &
     & .2438100E+01,.5471900E+01,.6329600E+01,.6369900E+01,.4737800E+01,&
     & .2583300E+01,.5970600E+01,.6983900E+01,.6980900E+01,.5202400E+01,&
     & .2727600E+01,.6503200E+01,.7682300E+01,.7662300E+01,.5684900E+01,&
     & .2884300E+01,.7076800E+01,.8422300E+01,.8397300E+01,.6161300E+01,&
     & .3077600E+01,.7683700E+01,.9192900E+01,.9187100E+01,.6669700E+01,&
     & .2223800E+01,.5236600E+01,.6133500E+01,.6157400E+01,.4570500E+01,&
     & .2364800E+01,.5745100E+01,.6815100E+01,.6826600E+01,.5034700E+01,&
     & .2506700E+01,.6298000E+01,.7546000E+01,.7554800E+01,.5506400E+01,&
     & .2668000E+01,.6893000E+01,.8322300E+01,.8330800E+01,.6010300E+01,&
     & .2876300E+01,.7523400E+01,.9126900E+01,.9154700E+01,.6544300E+01,&
     & .2034300E+01,.5054000E+01,.6002100E+01,.6036000E+01,.4430600E+01,&
     & .2173200E+01,.5578500E+01,.6713400E+01,.6754400E+01,.4892400E+01,&
     & .2321700E+01,.6155400E+01,.7482800E+01,.7521700E+01,.5389900E+01,&
     & .2499100E+01,.6772700E+01,.8294300E+01,.8334100E+01,.5921100E+01,&
     & .2729800E+01,.7430900E+01,.9129400E+01,.9187700E+01,.6477600E+01,&
     & .1867800E+01,.4920300E+01,.5929100E+01,.5987500E+01,.4320400E+01,&
     & .2010200E+01,.5466700E+01,.6674700E+01,.6743000E+01,.4802600E+01,&
     & .2171900E+01,.6068200E+01,.7482700E+01,.7547700E+01,.5328100E+01,&
     & .2372000E+01,.6712600E+01,.8325200E+01,.8391000E+01,.5882600E+01,&
     & .2626400E+01,.7397700E+01,.9187000E+01,.9272300E+01,.6459800E+01,&
     & .1724400E+01,.4837000E+01,.5915600E+01,.6000100E+01,.4255500E+01,&
     & .1875500E+01,.5410700E+01,.6699900E+01,.6790400E+01,.4764400E+01,&
     & .2057300E+01,.6036300E+01,.7540200E+01,.7627400E+01,.5315500E+01,&
     & .2282900E+01,.6709400E+01,.8407900E+01,.8496200E+01,.5890100E+01,&
     & .2559000E+01,.7419500E+01,.9290900E+01,.9399400E+01,.6487100E+01,&
     & .1606900E+01,.4803000E+01,.5957900E+01,.6063900E+01,.4235600E+01,&
     & .1770500E+01,.5403700E+01,.6778800E+01,.6887400E+01,.4771200E+01,&
     & .1974200E+01,.6057400E+01,.7644800E+01,.7749100E+01,.5343400E+01,&
     & .2224700E+01,.6756000E+01,.8531000E+01,.8638500E+01,.5938100E+01,&
     & .2523800E+01,.7488000E+01,.9429200E+01,.9557000E+01,.6549700E+01,&
     & .1512800E+01,.4814000E+01,.6050100E+01,.6171400E+01,.4255000E+01,&
     & .1694600E+01,.5441700E+01,.6900000E+01,.7021200E+01,.4815500E+01,&
     & .1920600E+01,.6121300E+01,.7783900E+01,.7902000E+01,.5406900E+01,&
     & .2194600E+01,.6842200E+01,.8682700E+01,.8807500E+01,.6016900E+01,&
     & .2517300E+01,.7591300E+01,.9592400E+01,.9737300E+01,.6639800E+01/

      data   absb(526:700, 9)   /                                       &
     & .1435600E+01,.4841800E+01,.6149500E+01,.6282800E+01,.4289600E+01,&
     & .1635700E+01,.5493500E+01,.7018400E+01,.7151600E+01,.4869100E+01,&
     & .1883800E+01,.6195700E+01,.7915100E+01,.8045700E+01,.5474200E+01,&
     & .2178800E+01,.6932500E+01,.8825600E+01,.8964300E+01,.6095600E+01,&
     & .2523800E+01,.7693000E+01,.9743900E+01,.9902100E+01,.6726400E+01,&
     & .1362000E+01,.4840000E+01,.6191100E+01,.6334100E+01,.4295000E+01,&
     & .1576500E+01,.5507700E+01,.7071700E+01,.7214900E+01,.4887200E+01,&
     & .1841600E+01,.6223800E+01,.7977800E+01,.8117300E+01,.5501400E+01,&
     & .2152800E+01,.6969400E+01,.8895100E+01,.9043000E+01,.6130400E+01,&
     & .2513500E+01,.7737000E+01,.9819100E+01,.9986500E+01,.6766200E+01,&
     & .1283600E+01,.4789200E+01,.6154900E+01,.6305000E+01,.4254000E+01,&
     & .1507000E+01,.5464900E+01,.7041600E+01,.7192300E+01,.4853600E+01,&
     & .1783000E+01,.6187200E+01,.7953100E+01,.8099400E+01,.5473400E+01,&
     & .2105500E+01,.6936900E+01,.8874500E+01,.9028800E+01,.6106600E+01,&
     & .2475400E+01,.7709000E+01,.9802300E+01,.9975900E+01,.6745900E+01,&
     & .1187800E+01,.4654100E+01,.5997500E+01,.6153100E+01,.4135700E+01,&
     & .1411400E+01,.5327900E+01,.6883600E+01,.7040700E+01,.4736600E+01,&
     & .1690700E+01,.6048800E+01,.7796100E+01,.7949200E+01,.5358700E+01,&
     & .2017100E+01,.6798700E+01,.8721400E+01,.8878200E+01,.5993100E+01,&
     & .2387900E+01,.7572100E+01,.9650500E+01,.9826300E+01,.6634600E+01,&
     & .1101900E+01,.4524100E+01,.5841100E+01,.6001500E+01,.4020600E+01,&
     & .1324000E+01,.5193800E+01,.6724800E+01,.6888100E+01,.4621500E+01,&
     & .1605000E+01,.5912700E+01,.7637700E+01,.7796900E+01,.5244400E+01,&
     & .1934400E+01,.6661400E+01,.8565600E+01,.8724100E+01,.5879700E+01,&
     & .2305300E+01,.7435200E+01,.9496100E+01,.9673000E+01,.6523200E+01,&
     & .1026100E+01,.4403500E+01,.5692500E+01,.5857100E+01,.3913000E+01,&
     & .1245800E+01,.5069400E+01,.6573500E+01,.6741100E+01,.4513000E+01,&
     & .1528100E+01,.5784400E+01,.7485300E+01,.7649700E+01,.5135300E+01,&
     & .1859000E+01,.6531000E+01,.8413900E+01,.8575200E+01,.5771100E+01,&
     & .2229600E+01,.7303100E+01,.9346100E+01,.9523300E+01,.6415500E+01,&
     & .9402000E+00,.4225200E+01,.5461800E+01,.5630500E+01,.3751400E+01,&
     & .1151000E+01,.4880800E+01,.6335900E+01,.6505100E+01,.4346400E+01,&
     & .1426700E+01,.5586500E+01,.7242900E+01,.7411600E+01,.4965900E+01,&
     & .1753400E+01,.6328400E+01,.8170600E+01,.8334600E+01,.5600400E+01,&
     & .2119600E+01,.7095400E+01,.9103800E+01,.9279000E+01,.6244300E+01/

      data   absb(701:875, 9)   /                                       &
     & .8612900E+00,.4045700E+01,.5226400E+01,.5397600E+01,.3587900E+01,&
     & .1063300E+01,.4691500E+01,.6092400E+01,.6263500E+01,.4176800E+01,&
     & .1328300E+01,.5384500E+01,.6992600E+01,.7164900E+01,.4792100E+01,&
     & .1648500E+01,.6120100E+01,.7917500E+01,.8084400E+01,.5424000E+01,&
     & .2009300E+01,.6881200E+01,.8852200E+01,.9024900E+01,.6066800E+01,&
     & .7912300E+00,.3873500E+01,.4996000E+01,.5169900E+01,.3428700E+01,&
     & .9829500E+00,.4507200E+01,.5853300E+01,.6025400E+01,.4010900E+01,&
     & .1236400E+01,.5187600E+01,.6745900E+01,.6921400E+01,.4621200E+01,&
     & .1548400E+01,.5915000E+01,.7666500E+01,.7837300E+01,.5250300E+01,&
     & .1902600E+01,.6671600E+01,.8601800E+01,.8772800E+01,.5891500E+01,&
     & .7198400E+00,.3672900E+01,.4723500E+01,.4897500E+01,.3239700E+01,&
     & .8980800E+00,.4291600E+01,.5569700E+01,.5744300E+01,.3815300E+01,&
     & .1136000E+01,.4957400E+01,.6451900E+01,.6628900E+01,.4418400E+01,&
     & .1434500E+01,.5672800E+01,.7366000E+01,.7541000E+01,.5043700E+01,&
     & .1779300E+01,.6422800E+01,.8299900E+01,.8470500E+01,.5682300E+01,&
     & .6528500E+00,.3466700E+01,.4437200E+01,.4608200E+01,.3039900E+01,&
     & .8165700E+00,.4067400E+01,.5270100E+01,.5446600E+01,.3609400E+01,&
     & .1037000E+01,.4717100E+01,.6140500E+01,.6317500E+01,.4203800E+01,&
     & .1319400E+01,.5418400E+01,.7047300E+01,.7224800E+01,.4823700E+01,&
     & .1651400E+01,.6158400E+01,.7976000E+01,.8147700E+01,.5458100E+01,&
     & .5932400E+00,.3268700E+01,.4161000E+01,.4327000E+01,.2843700E+01,&
     & .7429200E+00,.3850200E+01,.4975400E+01,.5152500E+01,.3408100E+01,&
     & .9460600E+00,.4483800E+01,.5834700E+01,.6011200E+01,.3993200E+01,&
     & .1210300E+01,.5169100E+01,.6731200E+01,.6911300E+01,.4606300E+01,&
     & .1528300E+01,.5897000E+01,.7652900E+01,.7828300E+01,.5236500E+01,&
     & .5366500E+00,.3066500E+01,.3878800E+01,.4034400E+01,.2639600E+01,&
     & .6732600E+00,.3627500E+01,.4668500E+01,.4844500E+01,.3197000E+01,&
     & .8596400E+00,.4245000E+01,.5516100E+01,.5694000E+01,.3774200E+01,&
     & .1102600E+01,.4911200E+01,.6399200E+01,.6579500E+01,.4378500E+01,&
     & .1403000E+01,.5624700E+01,.7312800E+01,.7491800E+01,.5003700E+01,&
     & .4796500E+00,.2851200E+01,.3577300E+01,.3719000E+01,.2417900E+01,&
     & .6049900E+00,.3389500E+01,.4335300E+01,.4507000E+01,.2965300E+01,&
     & .7721100E+00,.3985700E+01,.5165600E+01,.5343900E+01,.3534600E+01,&
     & .9928000E+00,.4632000E+01,.6034600E+01,.6213800E+01,.4127800E+01,&
     & .1271100E+01,.5327200E+01,.6937700E+01,.7118900E+01,.4746400E+01/

      data   absb(876:1050, 9)   /                                      &
     & .4265100E+00,.2644500E+01,.3289400E+01,.3414800E+01,.2203300E+01,&
     & .5438800E+00,.3161100E+01,.4016100E+01,.4179100E+01,.2737300E+01,&
     & .6936800E+00,.3734700E+01,.4820600E+01,.4998900E+01,.3299700E+01,&
     & .8927900E+00,.4362000E+01,.5676700E+01,.5855400E+01,.3882200E+01,&
     & .1148400E+01,.5036900E+01,.6566800E+01,.6749500E+01,.4491800E+01,&
     & .3771300E+00,.2443900E+01,.3016000E+01,.3120600E+01,.1997600E+01,&
     & .4876600E+00,.2943400E+01,.3710400E+01,.3859900E+01,.2514300E+01,&
     & .6240700E+00,.3493900E+01,.4484700E+01,.4659600E+01,.3068000E+01,&
     & .8029800E+00,.4100600E+01,.5325500E+01,.5504900E+01,.3641300E+01,&
     & .1035300E+01,.4755400E+01,.6201000E+01,.6382000E+01,.4240700E+01,&
     & .3333400E+00,.2257300E+01,.2773000E+01,.2853700E+01,.1814100E+01,&
     & .4382400E+00,.2747700E+01,.3435900E+01,.3571200E+01,.2311200E+01,&
     & .5655500E+00,.3276700E+01,.4180700E+01,.4349700E+01,.2854000E+01,&
     & .7266200E+00,.3863100E+01,.5000300E+01,.5179800E+01,.3420600E+01,&
     & .9381100E+00,.4499800E+01,.5863500E+01,.6043300E+01,.4009200E+01,&
     & .2934400E+00,.2079200E+01,.2549300E+01,.2604500E+01,.1644700E+01,&
     & .3932000E+00,.2563300E+01,.3181100E+01,.3299600E+01,.2120200E+01,&
     & .5133500E+00,.3073900E+01,.3896800E+01,.4055400E+01,.2649300E+01,&
     & .6599100E+00,.3639600E+01,.4691000E+01,.4869200E+01,.3209500E+01,&
     & .8530500E+00,.4260200E+01,.5542800E+01,.5722700E+01,.3789000E+01,&
     & .2563400E+00,.1906500E+01,.2337300E+01,.2367500E+01,.1484900E+01,&
     & .3508500E+00,.2383200E+01,.2938000E+01,.3036300E+01,.1937900E+01,&
     & .4646300E+00,.2880600E+01,.3624500E+01,.3770500E+01,.2449900E+01,&
     & .6001400E+00,.3424700E+01,.4390100E+01,.4564400E+01,.3001300E+01,&
     & .7750200E+00,.4026800E+01,.5226400E+01,.5406400E+01,.3572900E+01,&
     & .2219800E+00,.1738200E+01,.2135500E+01,.2144300E+01,.1332100E+01,&
     & .3108800E+00,.2203700E+01,.2707000E+01,.2780900E+01,.1763100E+01,&
     & .4186800E+00,.2693700E+01,.3362800E+01,.3493700E+01,.2255100E+01,&
     & .5459000E+00,.3217200E+01,.4099400E+01,.4266900E+01,.2795000E+01,&
     & .7038200E+00,.3799300E+01,.4913400E+01,.5093400E+01,.3360900E+01,&
     & .1927900E+00,.1588400E+01,.1957600E+01,.1951600E+01,.1199100E+01,&
     & .2761400E+00,.2042300E+01,.2505300E+01,.2555900E+01,.1610600E+01,&
     & .3790000E+00,.2526400E+01,.3132300E+01,.3247300E+01,.2082500E+01,&
     & .4995400E+00,.3034400E+01,.3842500E+01,.3999400E+01,.2609000E+01,&
     & .6448300E+00,.3596700E+01,.4632000E+01,.4810400E+01,.3168300E+01/

      data   absb(1051:1175, 9)   /                                     &
     & .1675200E+00,.1450600E+01,.1792400E+01,.1778000E+01,.1076600E+01,&
     & .2454600E+00,.1892400E+01,.2321500E+01,.2350200E+01,.1471800E+01,&
     & .3428200E+00,.2368600E+01,.2920300E+01,.3017100E+01,.1923800E+01,&
     & .4575100E+00,.2865700E+01,.3604800E+01,.3750400E+01,.2434700E+01,&
     & .5934500E+00,.3409100E+01,.4369200E+01,.4543600E+01,.2986000E+01,&
     & .1444200E+00,.1319100E+01,.1630700E+01,.1615800E+01,.9596600E+00,&
     & .2162900E+00,.1745600E+01,.2145400E+01,.2155300E+01,.1338400E+01,&
     & .3083800E+00,.2211700E+01,.2718400E+01,.2793900E+01,.1771000E+01,&
     & .4178000E+00,.2702400E+01,.3376100E+01,.3508300E+01,.2264700E+01,&
     & .5463300E+00,.3227700E+01,.4115300E+01,.4283500E+01,.2805700E+01,&
     & .1236900E+00,.1194200E+01,.1475100E+01,.1466000E+01,.8479400E+00,&
     & .1894300E+00,.1602500E+01,.1975200E+01,.1970600E+01,.1210400E+01,&
     & .2758000E+00,.2057700E+01,.2525300E+01,.2578500E+01,.1624900E+01,&
     & .3801500E+00,.2542500E+01,.3155400E+01,.3272400E+01,.2099500E+01,&
     & .5022000E+00,.3052800E+01,.3869400E+01,.4027500E+01,.2627900E+01,&
     & .1061900E+00,.1081500E+01,.1333800E+01,.1334000E+01,.7492400E+00,&
     & .1658600E+00,.1471300E+01,.1818300E+01,.1804700E+01,.1093300E+01,&
     & .2466800E+00,.1915000E+01,.2349800E+01,.2382400E+01,.1493100E+01,&
     & .3461300E+00,.2393600E+01,.2954200E+01,.3054500E+01,.1948900E+01,&
     & .4625100E+00,.2892800E+01,.3643900E+01,.3791400E+01,.2462800E+01,&
     & .9878100E-01,.1037400E+01,.1279000E+01,.1283400E+01,.7109100E+00,&
     & .1564300E+00,.1419900E+01,.1756000E+01,.1740600E+01,.1047500E+01,&
     & .2351200E+00,.1858500E+01,.2281500E+01,.2306100E+01,.1440800E+01,&
     & .3325000E+00,.2333800E+01,.2875500E+01,.2968600E+01,.1889800E+01,&
     & .4469700E+00,.2829800E+01,.3555500E+01,.3698400E+01,.2397500E+01/

      data   absb(  1:175,10)   /                                       &
     & .5148311E+02,.5413939E+02,.5702358E+02,.5255781E+02,.4767175E+02,&
     & .5089809E+02,.5366563E+02,.5690094E+02,.5278229E+02,.4816697E+02,&
     & .5028687E+02,.5319767E+02,.5674627E+02,.5298787E+02,.4858275E+02,&
     & .4966788E+02,.5270972E+02,.5651558E+02,.5315507E+02,.4895540E+02,&
     & .4904266E+02,.5220812E+02,.5626701E+02,.5327637E+02,.4934862E+02,&
     & .4695357E+02,.5244112E+02,.5623255E+02,.5228165E+02,.4528190E+02,&
     & .4636862E+02,.5203478E+02,.5620209E+02,.5261290E+02,.4583339E+02,&
     & .4578848E+02,.5161094E+02,.5610019E+02,.5291867E+02,.4633999E+02,&
     & .4522091E+02,.5118237E+02,.5596404E+02,.5316078E+02,.4686589E+02,&
     & .4464656E+02,.5075226E+02,.5579719E+02,.5336823E+02,.4742733E+02,&
     & .4214601E+02,.5023928E+02,.5465620E+02,.5137721E+02,.4265810E+02,&
     & .4159514E+02,.4989728E+02,.5473059E+02,.5180612E+02,.4330876E+02,&
     & .4107738E+02,.4954386E+02,.5475515E+02,.5218532E+02,.4396773E+02,&
     & .4057407E+02,.4919951E+02,.5475077E+02,.5253512E+02,.4466305E+02,&
     & .4017621E+02,.4885046E+02,.5474194E+02,.5287718E+02,.4535369E+02,&
     & .3729887E+02,.4762187E+02,.5248346E+02,.4992379E+02,.3996857E+02,&
     & .3682707E+02,.4735522E+02,.5268439E+02,.5045155E+02,.4074890E+02,&
     & .3639654E+02,.4710649E+02,.5286499E+02,.5096814E+02,.4156839E+02,&
     & .3608349E+02,.4686360E+02,.5304363E+02,.5147530E+02,.4241003E+02,&
     & .3594030E+02,.4664103E+02,.5320927E+02,.5198934E+02,.4322855E+02,&
     & .3267085E+02,.4471454E+02,.4990144E+02,.4806389E+02,.3737273E+02,&
     & .3229921E+02,.4458748E+02,.5027045E+02,.4875782E+02,.3829531E+02,&
     & .3205109E+02,.4447599E+02,.5065142E+02,.4944055E+02,.3925216E+02,&
     & .3198556E+02,.4439449E+02,.5103374E+02,.5013485E+02,.4020794E+02,&
     & .3212562E+02,.4431957E+02,.5139858E+02,.5083922E+02,.4116103E+02,&
     & .2840862E+02,.4164567E+02,.4712876E+02,.4597745E+02,.3496902E+02,&
     & .2819485E+02,.4169424E+02,.4770395E+02,.4685191E+02,.3600935E+02,&
     & .2816868E+02,.4176226E+02,.4829256E+02,.4773086E+02,.3708676E+02,&
     & .2836998E+02,.4184080E+02,.4888393E+02,.4862950E+02,.3815932E+02,&
     & .2876609E+02,.4198837E+02,.4946530E+02,.4954016E+02,.3924359E+02,&
     & .2458838E+02,.3854886E+02,.4433510E+02,.4378221E+02,.3278158E+02,&
     & .2458406E+02,.3879353E+02,.4511982E+02,.4485465E+02,.3393561E+02,&
     & .2481011E+02,.3904687E+02,.4591165E+02,.4596040E+02,.3509830E+02,&
     & .2524181E+02,.3936718E+02,.4670940E+02,.4707856E+02,.3628077E+02,&
     & .2588043E+02,.3975083E+02,.4751106E+02,.4819091E+02,.3749205E+02/

      data   absb(176:350,10)   /                                       &
     & .2128943E+02,.3559835E+02,.4162163E+02,.4164333E+02,.3082318E+02,&
     & .2150537E+02,.3602723E+02,.4259734E+02,.4292976E+02,.3205684E+02,&
     & .2193801E+02,.3650531E+02,.4361086E+02,.4425346E+02,.3332155E+02,&
     & .2259310E+02,.3706210E+02,.4463171E+02,.4558654E+02,.3461692E+02,&
     & .2342964E+02,.3772964E+02,.4566117E+02,.4688864E+02,.3593171E+02,&
     & .1852062E+02,.3287436E+02,.3903572E+02,.3964314E+02,.2907608E+02,&
     & .1891782E+02,.3350332E+02,.4024543E+02,.4114241E+02,.3039091E+02,&
     & .1954091E+02,.3421519E+02,.4148282E+02,.4268352E+02,.3174986E+02,&
     & .2036981E+02,.3502995E+02,.4273744E+02,.4421574E+02,.3314916E+02,&
     & .2138764E+02,.3596408E+02,.4400706E+02,.4569955E+02,.3457774E+02,&
     & .1624714E+02,.3047658E+02,.3680690E+02,.3795944E+02,.2761997E+02,&
     & .1682346E+02,.3133476E+02,.3822752E+02,.3967654E+02,.2904446E+02,&
     & .1761414E+02,.3229856E+02,.3969010E+02,.4139802E+02,.3050220E+02,&
     & .1860843E+02,.3338503E+02,.4119523E+02,.4312030E+02,.3200324E+02,&
     & .1979142E+02,.3455683E+02,.4269211E+02,.4477201E+02,.3355101E+02,&
     & .1441593E+02,.2844847E+02,.3488725E+02,.3654168E+02,.2639799E+02,&
     & .1513627E+02,.2951628E+02,.3652960E+02,.3844842E+02,.2793035E+02,&
     & .1607257E+02,.3073671E+02,.3822881E+02,.4036376E+02,.2949013E+02,&
     & .1721158E+02,.3204955E+02,.3993969E+02,.4224228E+02,.3109975E+02,&
     & .1853506E+02,.3344658E+02,.4166331E+02,.4405921E+02,.3276077E+02,&
     & .1295279E+02,.2677411E+02,.3332381E+02,.3540024E+02,.2541395E+02,&
     & .1380491E+02,.2807574E+02,.3519465E+02,.3748808E+02,.2704929E+02,&
     & .1487470E+02,.2950308E+02,.3708718E+02,.3957763E+02,.2871542E+02,&
     & .1613680E+02,.3102968E+02,.3900635E+02,.4159900E+02,.3042625E+02,&
     & .1756668E+02,.3261608E+02,.4093414E+02,.4356179E+02,.3219685E+02,&
     & .1181308E+02,.2545547E+02,.3213568E+02,.3454579E+02,.2465802E+02,&
     & .1278415E+02,.2697292E+02,.3418991E+02,.3680560E+02,.2639551E+02,&
     & .1396046E+02,.2859206E+02,.3627615E+02,.3903520E+02,.2816396E+02,&
     & .1530966E+02,.3030034E+02,.3838513E+02,.4119274E+02,.2998441E+02,&
     & .1680246E+02,.3204287E+02,.4046635E+02,.4326902E+02,.3183828E+02,&
     & .1095745E+02,.2448401E+02,.3130388E+02,.3400597E+02,.2413910E+02,&
     & .1202575E+02,.2618548E+02,.3353135E+02,.3640460E+02,.2597275E+02,&
     & .1327161E+02,.2798353E+02,.3579506E+02,.3875055E+02,.2784255E+02,&
     & .1467686E+02,.2984441E+02,.3805097E+02,.4101420E+02,.2975816E+02,&
     & .1623812E+02,.3172488E+02,.4025111E+02,.4318326E+02,.3168296E+02/

      data   absb(351:525,10)   /                                       &
     & .1031026E+02,.2379955E+02,.3077002E+02,.3371001E+02,.2381408E+02,&
     & .1143882E+02,.2566958E+02,.3315826E+02,.3621895E+02,.2574289E+02,&
     & .1273674E+02,.2762295E+02,.3556863E+02,.3865824E+02,.2770335E+02,&
     & .1420938E+02,.2960954E+02,.3792581E+02,.4100632E+02,.2969115E+02,&
     & .1584019E+02,.3161041E+02,.4022388E+02,.4324073E+02,.3167330E+02,&
     & .9809660E+01,.2337401E+02,.3050021E+02,.3361805E+02,.2366649E+02,&
     & .1099292E+02,.2539109E+02,.3302281E+02,.3621280E+02,.2568000E+02,&
     & .1234898E+02,.2746966E+02,.3553468E+02,.3873037E+02,.2771661E+02,&
     & .1389015E+02,.2956157E+02,.3797616E+02,.4114283E+02,.2975703E+02,&
     & .1559986E+02,.3165438E+02,.4034450E+02,.4341423E+02,.3178583E+02,&
     & .9444252E+01,.2318083E+02,.3046927E+02,.3371915E+02,.2368071E+02,&
     & .1068902E+02,.2532385E+02,.3309721E+02,.3637816E+02,.2576903E+02,&
     & .1210865E+02,.2749920E+02,.3567757E+02,.3895396E+02,.2786288E+02,&
     & .1372527E+02,.2967804E+02,.3817975E+02,.4140126E+02,.2994365E+02,&
     & .1551148E+02,.3183065E+02,.4058662E+02,.4368854E+02,.3200806E+02,&
     & .9197704E+01,.2317475E+02,.3061830E+02,.3396112E+02,.2382472E+02,&
     & .1050749E+02,.2541270E+02,.3332073E+02,.3667202E+02,.2597119E+02,&
     & .1200173E+02,.2766533E+02,.3594813E+02,.3928170E+02,.2810254E+02,&
     & .1369150E+02,.2990848E+02,.3849061E+02,.4174119E+02,.3021392E+02,&
     & .1554967E+02,.3210113E+02,.4091548E+02,.4403004E+02,.3230534E+02,&
     & .9070152E+01,.2333358E+02,.3092333E+02,.3432849E+02,.2408369E+02,&
     & .1044943E+02,.2564057E+02,.3366988E+02,.3707300E+02,.2627089E+02,&
     & .1201766E+02,.2795335E+02,.3632847E+02,.3969279E+02,.2842654E+02,&
     & .1377594E+02,.3023853E+02,.3888979E+02,.4215285E+02,.3056150E+02,&
     & .1570981E+02,.3246011E+02,.4132215E+02,.4443246E+02,.3266580E+02,&
     & .9050881E+01,.2361502E+02,.3134148E+02,.3479389E+02,.2443447E+02,&
     & .1049856E+02,.2597874E+02,.3411303E+02,.3755332E+02,.2664478E+02,&
     & .1213853E+02,.2833372E+02,.3679073E+02,.4016895E+02,.2881882E+02,&
     & .1396914E+02,.3064406E+02,.3935900E+02,.4261289E+02,.3096497E+02,&
     & .1597202E+02,.3287736E+02,.4178044E+02,.4487584E+02,.3307507E+02,&
     & .9121485E+01,.2399340E+02,.3184110E+02,.3533107E+02,.2484905E+02,&
     & .1063898E+02,.2639662E+02,.3462510E+02,.3808849E+02,.2707110E+02,&
     & .1235080E+02,.2877918E+02,.3730856E+02,.4068584E+02,.2925606E+02,&
     & .1425215E+02,.3109959E+02,.3987030E+02,.4310628E+02,.3140551E+02,&
     & .1631751E+02,.3333996E+02,.4227596E+02,.4534660E+02,.3351854E+02/

      data   absb(526:700,10)   /                                       &
     & .9221530E+01,.2435543E+02,.3229891E+02,.3581316E+02,.2522887E+02,&
     & .1079853E+02,.2679042E+02,.3509314E+02,.3856585E+02,.2745675E+02,&
     & .1257368E+02,.2918513E+02,.3777390E+02,.4114200E+02,.2964790E+02,&
     & .1453244E+02,.3151030E+02,.4032236E+02,.4354185E+02,.3180140E+02,&
     & .1664739E+02,.3375272E+02,.4271105E+02,.4575694E+02,.3391150E+02,&
     & .9234921E+01,.2453098E+02,.3252381E+02,.3605734E+02,.2542131E+02,&
     & .1085339E+02,.2698432E+02,.3532561E+02,.3880579E+02,.2765412E+02,&
     & .1266775E+02,.2939222E+02,.3801034E+02,.4137794E+02,.2985073E+02,&
     & .1466294E+02,.3172547E+02,.4055737E+02,.4377010E+02,.3200723E+02,&
     & .1680944E+02,.3397153E+02,.4293979E+02,.4597565E+02,.3411921E+02,&
     & .9122231E+01,.2446628E+02,.3246758E+02,.3601819E+02,.2538766E+02,&
     & .1075904E+02,.2694105E+02,.3528642E+02,.3878018E+02,.2762932E+02,&
     & .1258935E+02,.2936114E+02,.3798364E+02,.4136006E+02,.2983193E+02,&
     & .1459875E+02,.3170547E+02,.4053737E+02,.4375979E+02,.3199342E+02,&
     & .1675964E+02,.3396515E+02,.4293329E+02,.4597404E+02,.3411195E+02,&
     & .8803526E+01,.2404740E+02,.3199538E+02,.3556585E+02,.2502152E+02,&
     & .1042899E+02,.2653898E+02,.3484146E+02,.3835958E+02,.2727937E+02,&
     & .1224422E+02,.2898189E+02,.3757031E+02,.4097542E+02,.2949406E+02,&
     & .1424127E+02,.3135230E+02,.4015581E+02,.4341193E+02,.3166878E+02,&
     & .1639118E+02,.3362956E+02,.4258239E+02,.4565679E+02,.3379358E+02,&
     & .8495800E+01,.2362002E+02,.3150999E+02,.3509681E+02,.2464648E+02,&
     & .1010745E+02,.2612626E+02,.3438400E+02,.3792281E+02,.2691822E+02,&
     & .1190476E+02,.2858932E+02,.3713908E+02,.4057067E+02,.2914314E+02,&
     & .1388319E+02,.3098024E+02,.3975563E+02,.4304137E+02,.3132741E+02,&
     & .1601937E+02,.3327847E+02,.4221299E+02,.4532208E+02,.3346428E+02,&
     & .8209035E+01,.2320772E+02,.3103726E+02,.3463708E+02,.2427972E+02,&
     & .9807919E+01,.2572338E+02,.3393496E+02,.3749067E+02,.2656635E+02,&
     & .1158465E+02,.2820687E+02,.3671655E+02,.4017333E+02,.2879950E+02,&
     & .1354242E+02,.3061364E+02,.3936092E+02,.4267581E+02,.3099415E+02,&
     & .1566241E+02,.3293272E+02,.4184735E+02,.4498926E+02,.3314168E+02,&
     & .7784536E+01,.2254512E+02,.3026965E+02,.3388321E+02,.2367971E+02,&
     & .9354919E+01,.2507029E+02,.3319736E+02,.3677442E+02,.2598726E+02,&
     & .1108920E+02,.2757544E+02,.3601743E+02,.3950998E+02,.2823588E+02,&
     & .1300857E+02,.3000630E+02,.3870211E+02,.4206119E+02,.3044264E+02,&
     & .1509523E+02,.3235435E+02,.4123355E+02,.4442692E+02,.3260463E+02/

      data   absb(701:875,10)   /                                       &
     & .7359863E+01,.2185814E+02,.2946589E+02,.3308961E+02,.2305342E+02,&
     & .8903128E+01,.2438939E+02,.3241822E+02,.3601163E+02,.2537863E+02,&
     & .1058971E+02,.2690897E+02,.3527629E+02,.3879993E+02,.2764129E+02,&
     & .1246583E+02,.2936503E+02,.3800506E+02,.4140154E+02,.2986144E+02,&
     & .1451513E+02,.3174374E+02,.4058041E+02,.4382112E+02,.3203789E+02,&
     & .6956382E+01,.2118043E+02,.2866324E+02,.3229115E+02,.2242833E+02,&
     & .8461548E+01,.2370968E+02,.3163712E+02,.3524703E+02,.2476677E+02,&
     & .1010795E+02,.2624061E+02,.3452870E+02,.3808268E+02,.2704785E+02,&
     & .1193898E+02,.2872275E+02,.3729875E+02,.4073465E+02,.2928035E+02,&
     & .1394781E+02,.3112498E+02,.3991541E+02,.4320304E+02,.3147062E+02,&
     & .6487545E+01,.2036428E+02,.2769057E+02,.3132006E+02,.2167153E+02,&
     & .7946649E+01,.2289019E+02,.3068619E+02,.3430975E+02,.2402051E+02,&
     & .9549264E+01,.2542789E+02,.3361405E+02,.3719415E+02,.2632349E+02,&
     & .1132268E+02,.2793770E+02,.3642692E+02,.3990874E+02,.2857172E+02,&
     & .1327981E+02,.3036678E+02,.3909744E+02,.4243907E+02,.3077710E+02,&
     & .6005571E+01,.1948807E+02,.2664131E+02,.3026331E+02,.2085818E+02,&
     & .7413456E+01,.2201077E+02,.2965560E+02,.3329082E+02,.2321444E+02,&
     & .8970318E+01,.2455040E+02,.3261620E+02,.3621899E+02,.2554069E+02,&
     & .1068617E+02,.2708083E+02,.3547019E+02,.3899468E+02,.2780284E+02,&
     & .1258054E+02,.2953954E+02,.3819719E+02,.4159109E+02,.3002584E+02,&
     & .5548432E+01,.1861826E+02,.2559278E+02,.2919402E+02,.2004363E+02,&
     & .6900092E+01,.2113681E+02,.2862219E+02,.3226196E+02,.2240744E+02,&
     & .8410066E+01,.2367383E+02,.3160742E+02,.3522918E+02,.2474973E+02,&
     & .1007047E+02,.2621742E+02,.3450362E+02,.3806349E+02,.2703382E+02,&
     & .1190409E+02,.2870722E+02,.3728217E+02,.4072426E+02,.2927143E+02,&
     & .5089620E+01,.1770102E+02,.2447913E+02,.2805560E+02,.1917787E+02,&
     & .6378669E+01,.2021146E+02,.2751878E+02,.3115717E+02,.2154621E+02,&
     & .7837964E+01,.2274529E+02,.3052386E+02,.3415832E+02,.2390212E+02,&
     & .9443196E+01,.2529235E+02,.3346307E+02,.3705110E+02,.2620879E+02,&
     & .1120944E+02,.2781188E+02,.3628879E+02,.3978127E+02,.2846286E+02,&
     & .4610896E+01,.1669349E+02,.2324369E+02,.2677647E+02,.1820576E+02,&
     & .5824606E+01,.1918277E+02,.2628270E+02,.2990605E+02,.2058572E+02,&
     & .7220182E+01,.2171074E+02,.2930791E+02,.3295106E+02,.2294896E+02,&
     & .8769986E+01,.2425684E+02,.3228052E+02,.3589409E+02,.2528213E+02,&
     & .1046384E+02,.2679654E+02,.3515901E+02,.3869877E+02,.2755405E+02/

      data   absb(876:1050,10)   /                                      &
     & .4168305E+01,.1570562E+02,.2201962E+02,.2548007E+02,.1723083E+02,&
     & .5302277E+01,.1816020E+02,.2504538E+02,.2864169E+02,.1962267E+02,&
     & .6628824E+01,.2068096E+02,.2808483E+02,.3173022E+02,.2199260E+02,&
     & .8120143E+01,.2322187E+02,.3108353E+02,.3471491E+02,.2434405E+02,&
     & .9751318E+01,.2577042E+02,.3400727E+02,.3758542E+02,.2664002E+02,&
     & .3760997E+01,.1474503E+02,.2081030E+02,.2417490E+02,.1625573E+02,&
     & .4812522E+01,.1715003E+02,.2381102E+02,.2737145E+02,.1865721E+02,&
     & .6065394E+01,.1965574E+02,.2685618E+02,.3049364E+02,.2103561E+02,&
     & .7493897E+01,.2218813E+02,.2987458E+02,.3351954E+02,.2339558E+02,&
     & .9069722E+01,.2473888E+02,.3283783E+02,.3644425E+02,.2572001E+02,&
     & .3407008E+01,.1387327E+02,.1968672E+02,.2295130E+02,.1534229E+02,&
     & .4384559E+01,.1621715E+02,.2266121E+02,.2616930E+02,.1774766E+02,&
     & .5564493E+01,.1869821E+02,.2569991E+02,.2931565E+02,.2013618E+02,&
     & .6930044E+01,.2122343E+02,.2873366E+02,.3238405E+02,.2250272E+02,&
     & .8451961E+01,.2376957E+02,.3172435E+02,.3535077E+02,.2484828E+02,&
     & .3086932E+01,.1305420E+02,.1860890E+02,.2176702E+02,.1445866E+02,&
     & .3998495E+01,.1533239E+02,.2155796E+02,.2498933E+02,.1686364E+02,&
     & .5105023E+01,.1777463E+02,.2457856E+02,.2816630E+02,.1926213E+02,&
     & .6405638E+01,.2029281E+02,.2762298E+02,.3127387E+02,.2163511E+02,&
     & .7876512E+01,.2283418E+02,.3063295E+02,.3427110E+02,.2399217E+02,&
     & .2791590E+01,.1226154E+02,.1753948E+02,.2058143E+02,.1357966E+02,&
     & .3641206E+01,.1447202E+02,.2046653E+02,.2380567E+02,.1597990E+02,&
     & .4672585E+01,.1686375E+02,.2346271E+02,.2701461E+02,.1838557E+02,&
     & .5905294E+01,.1936440E+02,.2650933E+02,.3014596E+02,.2076820E+02,&
     & .7318334E+01,.2189961E+02,.2953666E+02,.3318455E+02,.2313340E+02,&
     & .2518774E+01,.1149827E+02,.1648112E+02,.1939248E+02,.1269754E+02,&
     & .3307338E+01,.1363661E+02,.1938353E+02,.2262314E+02,.1509794E+02,&
     & .4268652E+01,.1596732E+02,.2235360E+02,.2584534E+02,.1750525E+02,&
     & .5428748E+01,.1843909E+02,.2539018E+02,.2900296E+02,.1989679E+02,&
     & .6779426E+01,.2096706E+02,.2843121E+02,.3208295E+02,.2226801E+02,&
     & .2289464E+01,.1082893E+02,.1552622E+02,.1830474E+02,.1189008E+02,&
     & .3020620E+01,.1289378E+02,.1840077E+02,.2154161E+02,.1429147E+02,&
     & .3922374E+01,.1516305E+02,.2134768E+02,.2476500E+02,.1669686E+02,&
     & .5015015E+01,.1759724E+02,.2436604E+02,.2795197E+02,.1909808E+02,&
     & .6305470E+01,.2011705E+02,.2741465E+02,.3106251E+02,.2147332E+02/

      data   absb(1051:1175,10)   /                                     &
     & .2089789E+01,.1021602E+02,.1463344E+02,.1726758E+02,.1112336E+02,&
     & .2767245E+01,.1220573E+02,.1746541E+02,.2050030E+02,.1351996E+02,&
     & .3612314E+01,.1441119E+02,.2039079E+02,.2372523E+02,.1592114E+02,&
     & .4639139E+01,.1679873E+02,.2338586E+02,.2693757E+02,.1832633E+02,&
     & .5869463E+01,.1930265E+02,.2643475E+02,.3007091E+02,.2071169E+02,&
     & .1902771E+01,.9617449E+01,.1375735E+02,.1623578E+02,.1036934E+02,&
     & .2529334E+01,.1153875E+02,.1653836E+02,.1945888E+02,.1274803E+02,&
     & .3320906E+01,.1368161E+02,.1944359E+02,.2269151E+02,.1514962E+02,&
     & .4286297E+01,.1601548E+02,.2241651E+02,.2591604E+02,.1755694E+02,&
     & .5452967E+01,.1849373E+02,.2545654E+02,.2907074E+02,.1994983E+02,&
     & .1725440E+01,.9031381E+01,.1289432E+02,.1520765E+02,.9626329E+01,&
     & .2309750E+01,.1089872E+02,.1562812E+02,.1842270E+02,.1197707E+02,&
     & .3046266E+01,.1297125E+02,.1850543E+02,.2165803E+02,.1437884E+02,&
     & .3954352E+01,.1524573E+02,.2145498E+02,.2488428E+02,.1678481E+02,&
     & .5056606E+01,.1768754E+02,.2447686E+02,.2806579E+02,.1918626E+02,&
     & .1564823E+01,.8489334E+01,.1209006E+02,.1424280E+02,.8931475E+01,&
     & .2117660E+01,.1031347E+02,.1477913E+02,.1744092E+02,.1125085E+02,&
     & .2802981E+01,.1231464E+02,.1761845E+02,.2067534E+02,.1364984E+02,&
     & .3659632E+01,.1453293E+02,.2055035E+02,.2390260E+02,.1605194E+02,&
     & .4698820E+01,.1693318E+02,.2355149E+02,.2710920E+02,.1845815E+02,&
     & .1502352E+01,.8276308E+01,.1177375E+02,.1385967E+02,.8655479E+01,&
     & .2044007E+01,.1008294E+02,.1444326E+02,.1704897E+02,.1096188E+02,&
     & .2709296E+01,.1205664E+02,.1726418E+02,.2027874E+02,.1335631E+02,&
     & .3545610E+01,.1425196E+02,.2018817E+02,.2350704E+02,.1575791E+02,&
     & .4560655E+01,.1663292E+02,.2318230E+02,.2672443E+02,.1816710E+02/

      data   absb(  1:175,11)   /                                       &
     & .2056300E+03,.1639500E+03,.1513100E+03,.1392800E+03,.1733600E+03,&
     & .2058300E+03,.1634600E+03,.1510200E+03,.1386700E+03,.1720900E+03,&
     & .2057900E+03,.1628200E+03,.1505100E+03,.1380300E+03,.1711900E+03,&
     & .2053800E+03,.1619800E+03,.1500500E+03,.1374600E+03,.1706200E+03,&
     & .2047100E+03,.1610000E+03,.1494200E+03,.1368500E+03,.1699800E+03,&
     & .2138700E+03,.1748600E+03,.1641000E+03,.1489600E+03,.1792600E+03,&
     & .2139800E+03,.1740700E+03,.1635400E+03,.1484900E+03,.1787200E+03,&
     & .2137100E+03,.1730600E+03,.1630000E+03,.1479600E+03,.1783800E+03,&
     & .2132000E+03,.1719300E+03,.1623800E+03,.1474800E+03,.1780700E+03,&
     & .2123900E+03,.1706200E+03,.1615800E+03,.1466100E+03,.1772600E+03,&
     & .2186800E+03,.1839200E+03,.1759700E+03,.1587700E+03,.1840900E+03,&
     & .2186100E+03,.1828500E+03,.1753300E+03,.1584500E+03,.1841300E+03,&
     & .2183800E+03,.1816800E+03,.1746900E+03,.1581400E+03,.1842300E+03,&
     & .2178500E+03,.1802800E+03,.1739200E+03,.1574700E+03,.1839100E+03,&
     & .2171400E+03,.1788100E+03,.1728100E+03,.1564400E+03,.1833700E+03,&
     & .2198900E+03,.1909000E+03,.1866600E+03,.1682300E+03,.1871700E+03,&
     & .2200000E+03,.1898900E+03,.1860400E+03,.1682200E+03,.1879700E+03,&
     & .2198200E+03,.1885600E+03,.1853300E+03,.1679400E+03,.1885100E+03,&
     & .2195400E+03,.1870900E+03,.1843200E+03,.1671500E+03,.1885400E+03,&
     & .2189800E+03,.1854200E+03,.1830300E+03,.1660000E+03,.1883900E+03,&
     & .2179300E+03,.1959600E+03,.1960300E+03,.1770300E+03,.1885700E+03,&
     & .2182100E+03,.1949200E+03,.1954800E+03,.1772500E+03,.1902300E+03,&
     & .2185000E+03,.1936900E+03,.1946200E+03,.1770100E+03,.1913300E+03,&
     & .2185500E+03,.1922100E+03,.1934300E+03,.1761600E+03,.1919400E+03,&
     & .2183900E+03,.1905700E+03,.1920600E+03,.1749000E+03,.1922500E+03,&
     & .2131100E+03,.1993500E+03,.2038500E+03,.1849200E+03,.1886400E+03,&
     & .2139900E+03,.1983800E+03,.2033900E+03,.1853600E+03,.1911700E+03,&
     & .2147700E+03,.1972700E+03,.2025100E+03,.1851700E+03,.1929600E+03,&
     & .2153700E+03,.1960200E+03,.2013100E+03,.1843300E+03,.1942600E+03,&
     & .2159400E+03,.1946000E+03,.1998900E+03,.1829900E+03,.1951100E+03,&
     & .2062800E+03,.2013100E+03,.2101800E+03,.1918100E+03,.1878000E+03,&
     & .2078600E+03,.2006100E+03,.2098500E+03,.1924900E+03,.1911700E+03,&
     & .2093900E+03,.1997800E+03,.2091500E+03,.1924500E+03,.1937900E+03,&
     & .2107800E+03,.1987600E+03,.2080800E+03,.1915900E+03,.1957800E+03,&
     & .2119600E+03,.1976200E+03,.2066600E+03,.1902000E+03,.1972600E+03/

      data   absb(176:350,11)   /                                       &
     & .1980600E+03,.2021500E+03,.2152500E+03,.1978100E+03,.1864600E+03,&
     & .2005400E+03,.2018000E+03,.2152700E+03,.1987400E+03,.1907200E+03,&
     & .2029800E+03,.2013200E+03,.2147800E+03,.1987700E+03,.1941300E+03,&
     & .2052200E+03,.2007000E+03,.2138400E+03,.1979300E+03,.1968100E+03,&
     & .2072500E+03,.1997500E+03,.2124900E+03,.1965900E+03,.1989900E+03,&
     & .1891900E+03,.2021700E+03,.2194200E+03,.2029100E+03,.1849900E+03,&
     & .1926800E+03,.2023900E+03,.2197400E+03,.2041500E+03,.1901200E+03,&
     & .1960900E+03,.2023100E+03,.2194700E+03,.2042500E+03,.1943200E+03,&
     & .1992600E+03,.2020100E+03,.2186600E+03,.2034600E+03,.1977000E+03,&
     & .2022400E+03,.2013600E+03,.2174200E+03,.2021200E+03,.2004400E+03,&
     & .1806400E+03,.2018200E+03,.2227600E+03,.2074000E+03,.1840900E+03,&
     & .1851300E+03,.2025300E+03,.2234100E+03,.2087600E+03,.1898800E+03,&
     & .1895100E+03,.2028700E+03,.2232900E+03,.2089000E+03,.1947400E+03,&
     & .1936800E+03,.2028300E+03,.2225900E+03,.2081200E+03,.1986900E+03,&
     & .1975200E+03,.2024800E+03,.2214300E+03,.2067600E+03,.2018500E+03,&
     & .1726100E+03,.2012900E+03,.2256000E+03,.2112600E+03,.1836000E+03,&
     & .1781300E+03,.2025500E+03,.2264500E+03,.2127700E+03,.1899300E+03,&
     & .1834700E+03,.2032200E+03,.2264900E+03,.2128800E+03,.1953400E+03,&
     & .1885800E+03,.2034500E+03,.2259300E+03,.2121100E+03,.1997200E+03,&
     & .1932500E+03,.2033000E+03,.2247000E+03,.2106600E+03,.2031400E+03,&
     & .1655000E+03,.2007900E+03,.2279700E+03,.2145900E+03,.1835200E+03,&
     & .1719900E+03,.2024700E+03,.2289900E+03,.2161400E+03,.1903100E+03,&
     & .1782900E+03,.2034900E+03,.2292000E+03,.2162100E+03,.1961000E+03,&
     & .1842400E+03,.2039500E+03,.2286500E+03,.2154400E+03,.2008000E+03,&
     & .1896600E+03,.2039300E+03,.2273500E+03,.2138800E+03,.2043700E+03,&
     & .1594900E+03,.2004800E+03,.2299900E+03,.2175800E+03,.1839500E+03,&
     & .1669800E+03,.2024700E+03,.2311900E+03,.2190300E+03,.1910800E+03,&
     & .1741100E+03,.2037000E+03,.2314400E+03,.2190400E+03,.1971100E+03,&
     & .1808800E+03,.2043500E+03,.2308200E+03,.2181700E+03,.2019200E+03,&
     & .1871000E+03,.2045000E+03,.2295000E+03,.2164800E+03,.2055900E+03,&
     & .1547900E+03,.2003800E+03,.2318100E+03,.2201600E+03,.1849000E+03,&
     & .1632200E+03,.2025800E+03,.2330700E+03,.2214500E+03,.1922200E+03,&
     & .1711900E+03,.2040100E+03,.2332600E+03,.2214000E+03,.1983100E+03,&
     & .1786600E+03,.2047800E+03,.2325900E+03,.2203800E+03,.2031500E+03,&
     & .1855300E+03,.2049800E+03,.2312200E+03,.2185400E+03,.2068500E+03/

      data   absb(351:525,11)   /                                       &
     & .1514300E+03,.2004900E+03,.2334400E+03,.2224100E+03,.1861500E+03,&
     & .1607000E+03,.2027900E+03,.2346500E+03,.2235200E+03,.1935300E+03,&
     & .1693200E+03,.2043000E+03,.2347300E+03,.2233100E+03,.1996000E+03,&
     & .1773700E+03,.2051800E+03,.2340200E+03,.2221200E+03,.2044000E+03,&
     & .1847200E+03,.2053700E+03,.2325700E+03,.2201500E+03,.2080700E+03,&
     & .1493500E+03,.2007800E+03,.2348700E+03,.2243600E+03,.1876200E+03,&
     & .1592600E+03,.2031200E+03,.2359700E+03,.2252300E+03,.1949100E+03,&
     & .1683900E+03,.2046700E+03,.2359700E+03,.2248400E+03,.2009200E+03,&
     & .1768700E+03,.2055400E+03,.2351600E+03,.2234500E+03,.2056200E+03,&
     & .1845300E+03,.2057000E+03,.2335700E+03,.2213500E+03,.2092100E+03,&
     & .1484400E+03,.2011700E+03,.2361400E+03,.2260100E+03,.1892700E+03,&
     & .1587800E+03,.2035000E+03,.2370700E+03,.2266400E+03,.1964200E+03,&
     & .1682900E+03,.2050500E+03,.2369600E+03,.2260100E+03,.2022900E+03,&
     & .1770200E+03,.2058500E+03,.2360200E+03,.2244500E+03,.2068500E+03,&
     & .1848700E+03,.2059900E+03,.2343000E+03,.2222000E+03,.2103100E+03,&
     & .1484100E+03,.2016500E+03,.2372400E+03,.2274200E+03,.1910100E+03,&
     & .1590300E+03,.2039500E+03,.2379800E+03,.2277400E+03,.1979600E+03,&
     & .1687800E+03,.2054300E+03,.2377400E+03,.2268900E+03,.2036700E+03,&
     & .1776500E+03,.2061400E+03,.2366200E+03,.2251500E+03,.2080400E+03,&
     & .1855600E+03,.2062600E+03,.2347800E+03,.2227400E+03,.2113300E+03,&
     & .1491000E+03,.2021900E+03,.2381600E+03,.2285700E+03,.1928000E+03,&
     & .1598700E+03,.2044200E+03,.2387300E+03,.2286000E+03,.1995500E+03,&
     & .1697400E+03,.2057800E+03,.2383000E+03,.2275100E+03,.2050300E+03,&
     & .1787000E+03,.2064000E+03,.2370300E+03,.2255900E+03,.2091700E+03,&
     & .1865600E+03,.2064700E+03,.2350400E+03,.2230300E+03,.2123200E+03,&
     & .1503100E+03,.2027900E+03,.2389700E+03,.2294900E+03,.1945900E+03,&
     & .1611900E+03,.2048900E+03,.2393100E+03,.2292200E+03,.2011300E+03,&
     & .1711000E+03,.2061000E+03,.2386900E+03,.2279100E+03,.2063400E+03,&
     & .1800200E+03,.2066200E+03,.2372300E+03,.2258200E+03,.2102900E+03,&
     & .1877700E+03,.2065900E+03,.2351000E+03,.2230800E+03,.2132200E+03,&
     & .1519700E+03,.2034100E+03,.2396500E+03,.2301900E+03,.1963900E+03,&
     & .1628500E+03,.2053400E+03,.2397600E+03,.2296400E+03,.2026700E+03,&
     & .1727000E+03,.2063800E+03,.2389300E+03,.2281100E+03,.2075800E+03,&
     & .1815100E+03,.2068000E+03,.2372700E+03,.2258400E+03,.2113200E+03,&
     & .1891200E+03,.2066500E+03,.2350000E+03,.2229600E+03,.2140600E+03/

      data   absb(526:700,11)   /                                       &
     & .1535200E+03,.2039100E+03,.2401400E+03,.2306900E+03,.1979000E+03,&
     & .1643700E+03,.2056600E+03,.2400600E+03,.2299200E+03,.2039600E+03,&
     & .1741600E+03,.2066200E+03,.2390700E+03,.2282300E+03,.2086300E+03,&
     & .1828500E+03,.2069500E+03,.2372800E+03,.2258000E+03,.2121800E+03,&
     & .1903300E+03,.2066700E+03,.2348700E+03,.2228000E+03,.2147400E+03,&
     & .1542100E+03,.2041800E+03,.2405200E+03,.2311000E+03,.1987600E+03,&
     & .1650700E+03,.2058600E+03,.2403300E+03,.2302000E+03,.2046800E+03,&
     & .1748600E+03,.2067700E+03,.2392400E+03,.2284100E+03,.2092300E+03,&
     & .1835000E+03,.2070500E+03,.2373700E+03,.2258900E+03,.2126900E+03,&
     & .1909400E+03,.2066900E+03,.2348900E+03,.2228100E+03,.2151400E+03,&
     & .1538100E+03,.2042400E+03,.2408000E+03,.2314500E+03,.1988800E+03,&
     & .1647800E+03,.2059200E+03,.2406000E+03,.2305500E+03,.2048400E+03,&
     & .1746400E+03,.2068400E+03,.2395000E+03,.2287400E+03,.2093800E+03,&
     & .1833600E+03,.2071500E+03,.2376200E+03,.2261800E+03,.2128400E+03,&
     & .1908600E+03,.2067800E+03,.2351400E+03,.2231100E+03,.2153200E+03,&
     & .1517700E+03,.2039400E+03,.2409600E+03,.2317800E+03,.1980200E+03,&
     & .1629800E+03,.2057800E+03,.2409100E+03,.2310200E+03,.2041600E+03,&
     & .1730700E+03,.2068100E+03,.2399400E+03,.2293400E+03,.2089300E+03,&
     & .1820400E+03,.2072100E+03,.2381800E+03,.2268800E+03,.2125500E+03,&
     & .1897300E+03,.2069500E+03,.2357300E+03,.2238300E+03,.2151600E+03,&
     & .1496800E+03,.2035900E+03,.2410400E+03,.2320200E+03,.1970300E+03,&
     & .1611300E+03,.2056000E+03,.2411600E+03,.2314300E+03,.2034100E+03,&
     & .1714400E+03,.2067400E+03,.2403300E+03,.2298800E+03,.2083800E+03,&
     & .1806300E+03,.2072400E+03,.2386600E+03,.2275200E+03,.2121600E+03,&
     & .1885500E+03,.2071000E+03,.2363200E+03,.2245300E+03,.2149300E+03,&
     & .1476400E+03,.2032200E+03,.2410600E+03,.2322000E+03,.1960400E+03,&
     & .1593200E+03,.2054000E+03,.2413700E+03,.2318000E+03,.2026200E+03,&
     & .1698400E+03,.2066500E+03,.2406500E+03,.2303600E+03,.2078100E+03,&
     & .1792300E+03,.2072500E+03,.2391000E+03,.2281000E+03,.2117500E+03,&
     & .1873700E+03,.2072200E+03,.2368400E+03,.2251700E+03,.2146700E+03,&
     & .1443800E+03,.2025400E+03,.2409200E+03,.2322500E+03,.1942500E+03,&
     & .1563300E+03,.2049800E+03,.2415200E+03,.2321700E+03,.2011500E+03,&
     & .1671800E+03,.2064200E+03,.2410500E+03,.2309600E+03,.2067200E+03,&
     & .1769000E+03,.2072000E+03,.2396900E+03,.2288700E+03,.2109300E+03,&
     & .1853900E+03,.2073300E+03,.2375800E+03,.2260900E+03,.2141100E+03/

      data   absb(701:875,11)   /                                       &
     & .1409300E+03,.2017000E+03,.2406200E+03,.2321400E+03,.1922600E+03,&
     & .1530900E+03,.2044100E+03,.2415500E+03,.2324300E+03,.1995100E+03,&
     & .1643400E+03,.2061300E+03,.2413600E+03,.2314900E+03,.2054500E+03,&
     & .1743900E+03,.2070900E+03,.2402200E+03,.2296300E+03,.2100000E+03,&
     & .1832500E+03,.2073900E+03,.2382900E+03,.2270000E+03,.2134400E+03,&
     & .1374200E+03,.2007200E+03,.2401800E+03,.2318800E+03,.1901700E+03,&
     & .1498700E+03,.2038100E+03,.2415100E+03,.2326000E+03,.1977800E+03,&
     & .1614400E+03,.2058000E+03,.2415900E+03,.2319400E+03,.2040800E+03,&
     & .1718200E+03,.2069300E+03,.2406800E+03,.2303000E+03,.2089700E+03,&
     & .1810100E+03,.2073900E+03,.2389400E+03,.2278500E+03,.2126900E+03,&
     & .1331600E+03,.1994600E+03,.2395200E+03,.2313400E+03,.1874600E+03,&
     & .1459000E+03,.2029900E+03,.2413100E+03,.2326500E+03,.1955500E+03,&
     & .1578200E+03,.2053100E+03,.2417600E+03,.2323700E+03,.2022800E+03,&
     & .1686000E+03,.2066600E+03,.2411300E+03,.2309900E+03,.2076100E+03,&
     & .1781800E+03,.2073300E+03,.2396400E+03,.2287800E+03,.2116600E+03,&
     & .1285100E+03,.1979300E+03,.2386000E+03,.2305100E+03,.1844000E+03,&
     & .1415600E+03,.2019500E+03,.2409200E+03,.2324900E+03,.1929800E+03,&
     & .1538100E+03,.2046800E+03,.2418200E+03,.2326800E+03,.2001500E+03,&
     & .1649400E+03,.2062500E+03,.2414800E+03,.2316300E+03,.2059700E+03,&
     & .1749900E+03,.2071900E+03,.2402900E+03,.2296900E+03,.2104300E+03,&
     & .1237700E+03,.1962200E+03,.2374500E+03,.2293800E+03,.1811400E+03,&
     & .1371300E+03,.2007400E+03,.2403500E+03,.2321000E+03,.1902400E+03,&
     & .1496900E+03,.2038900E+03,.2417100E+03,.2328300E+03,.1978700E+03,&
     & .1612200E+03,.2058000E+03,.2417200E+03,.2321500E+03,.2041700E+03,&
     & .1716900E+03,.2069700E+03,.2408500E+03,.2305100E+03,.2090700E+03,&
     & .1187100E+03,.1941800E+03,.2360000E+03,.2279000E+03,.1774700E+03,&
     & .1323100E+03,.1992900E+03,.2395400E+03,.2314300E+03,.1871200E+03,&
     & .1451400E+03,.2028800E+03,.2413800E+03,.2328000E+03,.1952600E+03,&
     & .1571100E+03,.2052300E+03,.2418700E+03,.2325700E+03,.2020700E+03,&
     & .1680200E+03,.2066400E+03,.2413200E+03,.2312600E+03,.2074900E+03,&
     & .1129500E+03,.1915900E+03,.2339800E+03,.2258800E+03,.1731300E+03,&
     & .1268300E+03,.1974100E+03,.2383500E+03,.2303100E+03,.1834200E+03,&
     & .1399900E+03,.2015900E+03,.2408500E+03,.2325300E+03,.1921700E+03,&
     & .1523400E+03,.2044200E+03,.2418600E+03,.2328600E+03,.1994900E+03,&
     & .1637300E+03,.2061800E+03,.2416900E+03,.2319400E+03,.2054900E+03/

      data   absb(876:1050,11)   /                                      &
     & .1071200E+03,.1887100E+03,.2315900E+03,.2234700E+03,.1685000E+03,&
     & .1212400E+03,.1952900E+03,.2368700E+03,.2288200E+03,.1794600E+03,&
     & .1347300E+03,.2000700E+03,.2400500E+03,.2319100E+03,.1888300E+03,&
     & .1474200E+03,.2034200E+03,.2416400E+03,.2329400E+03,.1967000E+03,&
     & .1592800E+03,.2056200E+03,.2419200E+03,.2324700E+03,.2032500E+03,&
     & .1012000E+03,.1855000E+03,.2288000E+03,.2207100E+03,.1636300E+03,&
     & .1155600E+03,.1928500E+03,.2350400E+03,.2269700E+03,.1752400E+03,&
     & .1293200E+03,.1983200E+03,.2390100E+03,.2309700E+03,.1852400E+03,&
     & .1423800E+03,.2022600E+03,.2412000E+03,.2327800E+03,.1937000E+03,&
     & .1546000E+03,.2048800E+03,.2419900E+03,.2328300E+03,.2007800E+03,&
     & .9560300E+02,.1821600E+03,.2257600E+03,.2176800E+03,.1587800E+03,&
     & .1101500E+03,.1903100E+03,.2329800E+03,.2249000E+03,.1710500E+03,&
     & .1241500E+03,.1964600E+03,.2377500E+03,.2297500E+03,.1816600E+03,&
     & .1375200E+03,.2009400E+03,.2405800E+03,.2323600E+03,.1906900E+03,&
     & .1500700E+03,.2040300E+03,.2418700E+03,.2329900E+03,.1982600E+03,&
     & .9013700E+02,.1786800E+03,.2224000E+03,.2144200E+03,.1538400E+03,&
     & .1048400E+03,.1875700E+03,.2306600E+03,.2225800E+03,.1667700E+03,&
     & .1190700E+03,.1944200E+03,.2362800E+03,.2282400E+03,.1779600E+03,&
     & .1326900E+03,.1994800E+03,.2397600E+03,.2316700E+03,.1875600E+03,&
     & .1455200E+03,.2030200E+03,.2415400E+03,.2329600E+03,.1956500E+03,&
     & .8469000E+02,.1749600E+03,.2186400E+03,.2108000E+03,.1486200E+03,&
     & .9947100E+02,.1845300E+03,.2279800E+03,.2199200E+03,.1622500E+03,&
     & .1139100E+03,.1921300E+03,.2345100E+03,.2264400E+03,.1740600E+03,&
     & .1277700E+03,.1978100E+03,.2387200E+03,.2307000E+03,.1842400E+03,&
     & .1409100E+03,.2018900E+03,.2410700E+03,.2327300E+03,.1928600E+03,&
     & .7926200E+02,.1709500E+03,.2143600E+03,.2068100E+03,.1431700E+03,&
     & .9404700E+02,.1812600E+03,.2249400E+03,.2169000E+03,.1574800E+03,&
     & .1086500E+03,.1895800E+03,.2324100E+03,.2243400E+03,.1699400E+03,&
     & .1227400E+03,.1959500E+03,.2374200E+03,.2294100E+03,.1807000E+03,&
     & .1361800E+03,.2005600E+03,.2404000E+03,.2322400E+03,.1898800E+03,&
     & .7431200E+02,.1670100E+03,.2100000E+03,.2028300E+03,.1380100E+03,&
     & .8905900E+02,.1780200E+03,.2217700E+03,.2138100E+03,.1528900E+03,&
     & .1037900E+03,.1870100E+03,.2302100E+03,.2221400E+03,.1659500E+03,&
     & .1180800E+03,.1940300E+03,.2360200E+03,.2279700E+03,.1772800E+03,&
     & .1317400E+03,.1991800E+03,.2396100E+03,.2315500E+03,.1869700E+03/

      data   absb(1051:1175,11)   /                                     &
     & .6961500E+02,.1629600E+03,.2054000E+03,.1986600E+03,.1328800E+03,&
     & .8427600E+02,.1746600E+03,.2183500E+03,.2105600E+03,.1482800E+03,&
     & .9906600E+02,.1843300E+03,.2278200E+03,.2197700E+03,.1619700E+03,&
     & .1135400E+03,.1919900E+03,.2344200E+03,.2263600E+03,.1738300E+03,&
     & .1274100E+03,.1977000E+03,.2386600E+03,.2306500E+03,.1840300E+03,&
     & .6494400E+02,.1587700E+03,.2004400E+03,.1942100E+03,.1276200E+03,&
     & .7952400E+02,.1711400E+03,.2146100E+03,.2070700E+03,.1435100E+03,&
     & .9432300E+02,.1814400E+03,.2251400E+03,.2171100E+03,.1577800E+03,&
     & .1089400E+03,.1897600E+03,.2325700E+03,.2245200E+03,.1702200E+03,&
     & .1230100E+03,.1960500E+03,.2375200E+03,.2295400E+03,.1809400E+03,&
     & .6035300E+02,.1543100E+03,.1951400E+03,.1894500E+03,.1222500E+03,&
     & .7481500E+02,.1674200E+03,.2104900E+03,.2032900E+03,.1386000E+03,&
     & .8956700E+02,.1783700E+03,.2221400E+03,.2141900E+03,.1534300E+03,&
     & .1043100E+03,.1873300E+03,.2305000E+03,.2224400E+03,.1664400E+03,&
     & .1185600E+03,.1942300E+03,.2361800E+03,.2281600E+03,.1776700E+03,&
     & .5609800E+02,.1499100E+03,.1898000E+03,.1846500E+03,.1170600E+03,&
     & .7037700E+02,.1636700E+03,.2062200E+03,.1994200E+03,.1337800E+03,&
     & .8504900E+02,.1752800E+03,.2190200E+03,.2111900E+03,.1491000E+03,&
     & .9985700E+02,.1848200E+03,.2282700E+03,.2202200E+03,.1626800E+03,&
     & .1142800E+03,.1923300E+03,.2347000E+03,.2266600E+03,.1744400E+03,&
     & .5441800E+02,.1480900E+03,.1875400E+03,.1826100E+03,.1149500E+03,&
     & .6859800E+02,.1621000E+03,.2044100E+03,.1977800E+03,.1318000E+03,&
     & .8324600E+02,.1739800E+03,.2176800E+03,.2099300E+03,.1473300E+03,&
     & .9805500E+02,.1837500E+03,.2273200E+03,.2192700E+03,.1611300E+03,&
     & .1125500E+03,.1915300E+03,.2340600E+03,.2260100E+03,.1731000E+03/

      data   absb(  1:175,12)   /                                       &
     & .4338110E+03,.3258785E+03,.2765841E+03,.3551700E+03,.4738213E+03,&
     & .4358591E+03,.3274735E+03,.2732330E+03,.3477157E+03,.4639305E+03,&
     & .4367994E+03,.3282050E+03,.2698072E+03,.3403736E+03,.4541170E+03,&
     & .4367673E+03,.3281913E+03,.2662171E+03,.3330191E+03,.4442737E+03,&
     & .4356257E+03,.3273304E+03,.2624520E+03,.3260525E+03,.4349606E+03,&
     & .4965751E+03,.3730543E+03,.3165865E+03,.3940821E+03,.5257811E+03,&
     & .4977615E+03,.3739641E+03,.3123416E+03,.3853185E+03,.5140967E+03,&
     & .4978529E+03,.3740600E+03,.3078301E+03,.3766318E+03,.5024939E+03,&
     & .4965358E+03,.3731122E+03,.3030947E+03,.3681551E+03,.4911078E+03,&
     & .4942501E+03,.3714089E+03,.2982254E+03,.3606307E+03,.4809116E+03,&
     & .5631880E+03,.4233224E+03,.3594901E+03,.4335570E+03,.5784608E+03,&
     & .5634438E+03,.4233039E+03,.3539288E+03,.4233711E+03,.5648410E+03,&
     & .5619418E+03,.4221964E+03,.3480452E+03,.4132459E+03,.5512312E+03,&
     & .5592483E+03,.4202048E+03,.3418750E+03,.4039045E+03,.5386286E+03,&
     & .5551436E+03,.4171466E+03,.3356718E+03,.3953284E+03,.5268417E+03,&
     & .6327464E+03,.4759753E+03,.4043694E+03,.4738466E+03,.6318088E+03,&
     & .6309974E+03,.4742402E+03,.3971672E+03,.4618354E+03,.6156456E+03,&
     & .6278582E+03,.4716973E+03,.3895188E+03,.4502024E+03,.5999656E+03,&
     & .6230476E+03,.4681132E+03,.3817116E+03,.4394718E+03,.5854793E+03,&
     & .6168383E+03,.4634762E+03,.3737303E+03,.4293418E+03,.5716261E+03,&
     & .7031362E+03,.5296686E+03,.4502274E+03,.5144971E+03,.6844628E+03,&
     & .6994439E+03,.5260700E+03,.4409153E+03,.5003459E+03,.6654934E+03,&
     & .6938124E+03,.5214150E+03,.4313297E+03,.4869200E+03,.6474992E+03,&
     & .6865591E+03,.5157531E+03,.4215792E+03,.4743711E+03,.6306843E+03,&
     & .6778828E+03,.5092809E+03,.4115306E+03,.4623577E+03,.6144981E+03,&
     & .7731573E+03,.5833614E+03,.4960552E+03,.5546751E+03,.7351434E+03,&
     & .7667584E+03,.5774695E+03,.4843214E+03,.5381637E+03,.7133093E+03,&
     & .7584006E+03,.5703374E+03,.4724758E+03,.5225411E+03,.6927570E+03,&
     & .7484491E+03,.5623616E+03,.4604054E+03,.5079339E+03,.6733220E+03,&
     & .7367135E+03,.5534249E+03,.4480895E+03,.4939401E+03,.6546934E+03,&
     & .8409349E+03,.6358707E+03,.5406929E+03,.5933755E+03,.7826997E+03,&
     & .8314773E+03,.6270808E+03,.5263991E+03,.5744137E+03,.7580256E+03,&
     & .8200103E+03,.6173497E+03,.5119158E+03,.5563407E+03,.7348534E+03,&
     & .8068534E+03,.6066626E+03,.4972618E+03,.5395008E+03,.7127611E+03,&
     & .7922845E+03,.5951793E+03,.4826400E+03,.5234260E+03,.6916658E+03/

      data   absb(176:350,12)   /                                       &
     & .9048913E+03,.6856063E+03,.5829003E+03,.6293251E+03,.8259779E+03,&
     & .8920622E+03,.6739392E+03,.5656996E+03,.6076975E+03,.7986291E+03,&
     & .8773064E+03,.6613225E+03,.5484044E+03,.5872679E+03,.7727058E+03,&
     & .8609036E+03,.6477146E+03,.5311640E+03,.5681427E+03,.7481234E+03,&
     & .8431150E+03,.6336171E+03,.5140156E+03,.5498732E+03,.7244630E+03,&
     & .9636991E+03,.7316951E+03,.6218536E+03,.6620342E+03,.8647359E+03,&
     & .9475066E+03,.7168594E+03,.6016241E+03,.6376903E+03,.8346844E+03,&
     & .9293968E+03,.7013269E+03,.5815752E+03,.6149608E+03,.8061544E+03,&
     & .9098003E+03,.6849951E+03,.5617381E+03,.5935925E+03,.7790911E+03,&
     & .8886450E+03,.6681002E+03,.5420821E+03,.5733205E+03,.7530675E+03,&
     & .1015405E+04,.7723224E+03,.6554984E+03,.6893842E+03,.8965322E+03,&
     & .9957317E+03,.7543125E+03,.6322716E+03,.6625732E+03,.8641101E+03,&
     & .9742889E+03,.7357574E+03,.6095984E+03,.6377464E+03,.8332099E+03,&
     & .9513341E+03,.7167192E+03,.5872099E+03,.6143318E+03,.8038674E+03,&
     & .9271699E+03,.6973008E+03,.5653278E+03,.5922105E+03,.7757920E+03,&
     & .1060462E+04,.8075265E+03,.6843897E+03,.7124747E+03,.9230005E+03,&
     & .1037459E+04,.7864953E+03,.6584383E+03,.6833274E+03,.8883530E+03,&
     & .1012792E+04,.7651897E+03,.6331576E+03,.6565949E+03,.8552652E+03,&
     & .9866354E+03,.7436769E+03,.6084848E+03,.6313485E+03,.8239323E+03,&
     & .9595800E+03,.7219245E+03,.5846257E+03,.6076220E+03,.7941595E+03,&
     & .1098608E+04,.8371963E+03,.7083084E+03,.7311988E+03,.9441461E+03,&
     & .1072494E+04,.8134318E+03,.6798957E+03,.7001704E+03,.9076022E+03,&
     & .1044667E+04,.7896182E+03,.6523292E+03,.6716308E+03,.8727223E+03,&
     & .1015777E+04,.7658591E+03,.6257102E+03,.6448513E+03,.8396486E+03,&
     & .9862086E+03,.7420869E+03,.6001289E+03,.6196814E+03,.8085246E+03,&
     & .1130080E+04,.8613842E+03,.7273980E+03,.7455565E+03,.9603143E+03,&
     & .1100795E+04,.8351571E+03,.6967578E+03,.7129616E+03,.9220699E+03,&
     & .1070329E+04,.8092621E+03,.6672876E+03,.6829948E+03,.8856533E+03,&
     & .1038932E+04,.7834641E+03,.6390022E+03,.6549212E+03,.8512901E+03,&
     & .1007073E+04,.7578586E+03,.6119071E+03,.6285978E+03,.8190226E+03,&
     & .1154817E+04,.8801654E+03,.7415039E+03,.7556111E+03,.9714143E+03,&
     & .1122664E+04,.8518781E+03,.7090277E+03,.7218209E+03,.9318180E+03,&
     & .1089916E+04,.8241127E+03,.6780777E+03,.6907287E+03,.8943125E+03,&
     & .1056461E+04,.7965849E+03,.6484079E+03,.6616194E+03,.8589272E+03,&
     & .1022590E+04,.7695324E+03,.6201114E+03,.6344623E+03,.8258624E+03/

      data   absb(351:525,12)   /                                       &
     & .1173826E+04,.8944232E+03,.7516288E+03,.7622958E+03,.9786164E+03,&
     & .1139213E+04,.8644602E+03,.7176974E+03,.7275753E+03,.9379617E+03,&
     & .1104499E+04,.8350215E+03,.6855100E+03,.6956267E+03,.8996608E+03,&
     & .1069228E+04,.8060327E+03,.6547517E+03,.6658171E+03,.8636644E+03,&
     & .1033679E+04,.7777792E+03,.6254959E+03,.6380265E+03,.8299256E+03,&
     & .1187612E+04,.9045966E+03,.7582339E+03,.7660705E+03,.9825372E+03,&
     & .1151104E+04,.8732077E+03,.7232198E+03,.7307711E+03,.9412613E+03,&
     & .1114812E+04,.8425100E+03,.6900707E+03,.6982050E+03,.9022920E+03,&
     & .1077922E+04,.8124079E+03,.6584780E+03,.6679077E+03,.8658499E+03,&
     & .1040927E+04,.7831204E+03,.6285352E+03,.6396645E+03,.8317461E+03,&
     & .1196774E+04,.9111403E+03,.7616196E+03,.7671722E+03,.9834539E+03,&
     & .1158870E+04,.8786102E+03,.7258194E+03,.7315277E+03,.9417190E+03,&
     & .1121068E+04,.8468836E+03,.6919844E+03,.6985953E+03,.9023781E+03,&
     & .1082920E+04,.8159387E+03,.6597813E+03,.6679809E+03,.8657188E+03,&
     & .1044783E+04,.7858772E+03,.6294298E+03,.6395237E+03,.8313784E+03,&
     & .1202260E+04,.9148111E+03,.7624882E+03,.7661952E+03,.9820032E+03,&
     & .1163158E+04,.8813333E+03,.7261740E+03,.7304250E+03,.9399991E+03,&
     & .1124180E+04,.8488606E+03,.6918461E+03,.6972877E+03,.9005744E+03,&
     & .1084954E+04,.8172592E+03,.6592986E+03,.6665878E+03,.8638719E+03,&
     & .1045950E+04,.7865464E+03,.6286194E+03,.6380504E+03,.8294321E+03,&
     & .1204456E+04,.9157270E+03,.7610818E+03,.7633951E+03,.9784831E+03,&
     & .1164367E+04,.8816366E+03,.7244819E+03,.7276259E+03,.9364715E+03,&
     & .1124324E+04,.8486129E+03,.6898842E+03,.6945597E+03,.8971270E+03,&
     & .1084324E+04,.8165570E+03,.6571659E+03,.6638568E+03,.8604884E+03,&
     & .1044694E+04,.7853529E+03,.6263972E+03,.6354464E+03,.8261775E+03,&
     & .1204084E+04,.9146184E+03,.7579482E+03,.7592747E+03,.9734683E+03,&
     & .1162995E+04,.8799770E+03,.7211610E+03,.7236223E+03,.9315108E+03,&
     & .1122270E+04,.8466270E+03,.6864778E+03,.6906781E+03,.8923413E+03,&
     & .1081628E+04,.8142783E+03,.6537748E+03,.6601480E+03,.8559108E+03,&
     & .1041535E+04,.7827690E+03,.6229791E+03,.6318521E+03,.8217287E+03,&
     & .1201431E+04,.9117433E+03,.7533822E+03,.7540042E+03,.9671155E+03,&
     & .1159753E+04,.8768687E+03,.7166394E+03,.7186372E+03,.9254210E+03,&
     & .1118355E+04,.8432577E+03,.6819604E+03,.6859134E+03,.8865933E+03,&
     & .1077205E+04,.8106161E+03,.6493437E+03,.6556209E+03,.8503919E+03,&
     & .1036866E+04,.7790721E+03,.6187296E+03,.6275874E+03,.8165163E+03/

      data   absb(526:700,12)   /                                       &
     & .1198563E+04,.9088441E+03,.7491554E+03,.7492438E+03,.9613627E+03,&
     & .1156284E+04,.8737803E+03,.7124077E+03,.7140741E+03,.9199132E+03,&
     & .1114455E+04,.8399712E+03,.6778293E+03,.6816338E+03,.8814481E+03,&
     & .1073022E+04,.8072052E+03,.6453367E+03,.6515614E+03,.8454321E+03,&
     & .1032356E+04,.7755667E+03,.6148054E+03,.6237372E+03,.8118204E+03,&
     & .1198333E+04,.9083513E+03,.7477168E+03,.7473560E+03,.9590262E+03,&
     & .1155753E+04,.8731138E+03,.7109009E+03,.7122360E+03,.9176511E+03,&
     & .1113496E+04,.8390696E+03,.6762632E+03,.6798170E+03,.8792408E+03,&
     & .1071776E+04,.8061247E+03,.6437713E+03,.6498449E+03,.8432884E+03,&
     & .1030817E+04,.7743687E+03,.6132096E+03,.6220561E+03,.8097579E+03,&
     & .1201812E+04,.9109743E+03,.7497301E+03,.7489114E+03,.9607489E+03,&
     & .1158859E+04,.8754611E+03,.7126895E+03,.7135931E+03,.9191682E+03,&
     & .1116249E+04,.8411676E+03,.6778498E+03,.6810248E+03,.8806473E+03,&
     & .1074182E+04,.8079523E+03,.6451507E+03,.6509316E+03,.8445579E+03,&
     & .1033029E+04,.7760505E+03,.6144744E+03,.6230304E+03,.8108939E+03,&
     & .1211130E+04,.9186048E+03,.7571510E+03,.7557283E+03,.9686341E+03,&
     & .1167769E+04,.8825784E+03,.7195377E+03,.7197860E+03,.9264272E+03,&
     & .1124863E+04,.8479369E+03,.6841799E+03,.6866852E+03,.8873138E+03,&
     & .1082412E+04,.8143560E+03,.6510060E+03,.6560895E+03,.8507530E+03,&
     & .1040908E+04,.7820519E+03,.6199358E+03,.6277981E+03,.8166320E+03,&
     & .1220285E+04,.9261353E+03,.7646141E+03,.7626244E+03,.9766109E+03,&
     & .1176440E+04,.8896203E+03,.7264230E+03,.7260792E+03,.9337937E+03,&
     & .1133375E+04,.8546623E+03,.6905960E+03,.6924389E+03,.8940982E+03,&
     & .1090677E+04,.8207879E+03,.6569885E+03,.6613756E+03,.8570661E+03,&
     & .1048813E+04,.7881007E+03,.6254696E+03,.6327012E+03,.8225054E+03,&
     & .1228866E+04,.9332604E+03,.7717386E+03,.7692485E+03,.9842332E+03,&
     & .1184837E+04,.8964165E+03,.7330783E+03,.7321425E+03,.9408794E+03,&
     & .1141425E+04,.8610630E+03,.6967646E+03,.6979977E+03,.9006456E+03,&
     & .1098473E+04,.8268792E+03,.6626986E+03,.6664793E+03,.8631791E+03,&
     & .1056323E+04,.7938687E+03,.6308135E+03,.6374322E+03,.8281892E+03,&
     & .1241753E+04,.9440723E+03,.7828610E+03,.7798359E+03,.9964019E+03,&
     & .1197417E+04,.9066817E+03,.7434365E+03,.7418632E+03,.9522524E+03,&
     & .1153661E+04,.8708987E+03,.7064391E+03,.7069149E+03,.9111337E+03,&
     & .1110381E+04,.8362458E+03,.6717052E+03,.6746952E+03,.8729740E+03,&
     & .1067895E+04,.8028379E+03,.6392420E+03,.6449926E+03,.8372651E+03/

      data   absb(701:875,12)   /                                       &
     & .1254711E+04,.9552218E+03,.7944804E+03,.7910279E+03,.1009186E+04,&
     & .1210280E+04,.9173371E+03,.7543339E+03,.7521533E+03,.9642251E+03,&
     & .1166182E+04,.8810139E+03,.7165982E+03,.7163206E+03,.9222282E+03,&
     & .1122776E+04,.8459987E+03,.6811757E+03,.6833364E+03,.8832595E+03,&
     & .1079901E+04,.8122269E+03,.6480909E+03,.6529289E+03,.8468137E+03,&
     & .1267704E+04,.9665631E+03,.8063197E+03,.8024662E+03,.1022026E+04,&
     & .1222982E+04,.9279730E+03,.7652524E+03,.7624963E+03,.9762665E+03,&
     & .1178689E+04,.8911031E+03,.7268015E+03,.7258004E+03,.9333389E+03,&
     & .1134938E+04,.8556974E+03,.6907044E+03,.6920393E+03,.8935613E+03,&
     & .1091744E+04,.8215054E+03,.6569768E+03,.6609780E+03,.8565207E+03,&
     & .1282885E+04,.9799862E+03,.8205225E+03,.8164768E+03,.1037659E+04,&
     & .1238160E+04,.9407046E+03,.7785501E+03,.7751742E+03,.9909163E+03,&
     & .1193461E+04,.9032104E+03,.7391944E+03,.7374586E+03,.9470329E+03,&
     & .1149466E+04,.8673209E+03,.7023116E+03,.7027833E+03,.9062086E+03,&
     & .1106044E+04,.8327209E+03,.6678128E+03,.6708431E+03,.8683165E+03,&
     & .1299221E+04,.9946371E+03,.8360635E+03,.8320461E+03,.1054652E+04,&
     & .1254105E+04,.9544361E+03,.7930175E+03,.7891647E+03,.1006958E+04,&
     & .1209314E+04,.9163157E+03,.7527076E+03,.7503398E+03,.9620928E+03,&
     & .1165150E+04,.8800178E+03,.7151041E+03,.7146356E+03,.9201686E+03,&
     & .1121407E+04,.8448306E+03,.6796764E+03,.6817394E+03,.8813213E+03,&
     & .1315203E+04,.1009411E+04,.8519211E+03,.8481780E+03,.1072024E+04,&
     & .1270148E+04,.9683874E+03,.8077450E+03,.8035704E+03,.1023229E+04,&
     & .1225178E+04,.9296125E+03,.7664897E+03,.7634566E+03,.9773360E+03,&
     & .1180684E+04,.8926430E+03,.7280109E+03,.7267010E+03,.9343522E+03,&
     & .1136746E+04,.8570580E+03,.6917330E+03,.6928240E+03,.8944515E+03,&
     & .1332014E+04,.1025376E+04,.8690997E+03,.8658147E+03,.1090797E+04,&
     & .1287083E+04,.9833984E+03,.8237334E+03,.8194241E+03,.1040908E+04,&
     & .1241959E+04,.9438288E+03,.7814714E+03,.7777789E+03,.9938697E+03,&
     & .1197174E+04,.9061941E+03,.7419541E+03,.7398805E+03,.9497877E+03,&
     & .1152918E+04,.8700812E+03,.7048184E+03,.7049556E+03,.9087731E+03,&
     & .1350647E+04,.1043564E+04,.8888305E+03,.8860745E+03,.1112065E+04,&
     & .1305779E+04,.1000456E+04,.8419893E+03,.8378345E+03,.1060921E+04,&
     & .1260705E+04,.9599420E+03,.7984917E+03,.7943099E+03,.1012757E+04,&
     & .1215666E+04,.9215564E+03,.7578604E+03,.7550366E+03,.9674587E+03,&
     & .1171119E+04,.8847872E+03,.7197368E+03,.7188825E+03,.9251420E+03/

      data   absb(876:1050,12)   /                                      &
     & .1369025E+04,.1061988E+04,.9090576E+03,.9070507E+03,.1133822E+04,&
     & .1324283E+04,.1017792E+04,.8607146E+03,.8569957E+03,.1081438E+04,&
     & .1279221E+04,.9763247E+03,.8159408E+03,.8115038E+03,.1032081E+04,&
     & .1234106E+04,.9370840E+03,.7741603E+03,.7706394E+03,.9855998E+03,&
     & .1189255E+04,.8996323E+03,.7349893E+03,.7332176E+03,.9419966E+03,&
     & .1387296E+04,.1080897E+04,.9299250E+03,.9288068E+03,.1156040E+04,&
     & .1342857E+04,.1035650E+04,.8800130E+03,.8768348E+03,.1102421E+04,&
     & .1297820E+04,.9930678E+03,.8338259E+03,.8293976E+03,.1051801E+04,&
     & .1252579E+04,.9528599E+03,.7908416E+03,.7867574E+03,.1004167E+04,&
     & .1207574E+04,.9146924E+03,.7505619E+03,.7480415E+03,.9593335E+03,&
     & .1404104E+04,.1098935E+04,.9500783E+03,.9498735E+03,.1177271E+04,&
     & .1360008E+04,.1052670E+04,.8986275E+03,.8960550E+03,.1122511E+04,&
     & .1315196E+04,.1009091E+04,.8510987E+03,.8469719E+03,.1070745E+04,&
     & .1269965E+04,.9680021E+03,.8068947E+03,.8024578E+03,.1021945E+04,&
     & .1224670E+04,.9290678E+03,.7655748E+03,.7624022E+03,.9760562E+03,&
     & .1420396E+04,.1116667E+04,.9703151E+03,.9709136E+03,.1198267E+04,&
     & .1376519E+04,.1069496E+04,.9171148E+03,.9153343E+03,.1142380E+04,&
     & .1331895E+04,.1024943E+04,.8682037E+03,.8645664E+03,.1089458E+04,&
     & .1286713E+04,.9828640E+03,.8228042E+03,.8182606E+03,.1039599E+04,&
     & .1241514E+04,.9432594E+03,.7805968E+03,.7767631E+03,.9926584E+03,&
     & .1436295E+04,.1134602E+04,.9911820E+03,.9925055E+03,.1219679E+04,&
     & .1392861E+04,.1086682E+04,.9361507E+03,.9351807E+03,.1162507E+04,&
     & .1348464E+04,.1041116E+04,.8857605E+03,.8826630E+03,.1108509E+04,&
     & .1303445E+04,.9981438E+03,.8391267E+03,.8346717E+03,.1057512E+04,&
     & .1258155E+04,.9576321E+03,.7957736E+03,.7914987E+03,.1009530E+04,&
     & .1451947E+04,.1152803E+04,.1012912E+04,.1014725E+04,.1241375E+04,&
     & .1409171E+04,.1104184E+04,.9558739E+03,.9558293E+03,.1183310E+04,&
     & .1365042E+04,.1057601E+04,.9039098E+03,.9014495E+03,.1128065E+04,&
     & .1320122E+04,.1013646E+04,.8559028E+03,.8518298E+03,.1075944E+04,&
     & .1274918E+04,.9723353E+03,.8114299E+03,.8068514E+03,.1026857E+04,&
     & .1466066E+04,.1169745E+04,.1033600E+04,.1035678E+04,.1261517E+04,&
     & .1423885E+04,.1120363E+04,.9744091E+03,.9750701E+03,.1202414E+04,&
     & .1380086E+04,.1073062E+04,.9208922E+03,.9191581E+03,.1146254E+04,&
     & .1335390E+04,.1028184E+04,.8716013E+03,.8680259E+03,.1093132E+04,&
     & .1290197E+04,.9859694E+03,.8260174E+03,.8214000E+03,.1043057E+04/

      data   absb(1051:1175,12)   /                                     &
     & .1479052E+04,.1186138E+04,.1053992E+04,.1056277E+04,.1281061E+04,&
     & .1437721E+04,.1136174E+04,.9928663E+03,.9941031E+03,.1221167E+04,&
     & .1394340E+04,.1088094E+04,.9376415E+03,.9367024E+03,.1164059E+04,&
     & .1349933E+04,.1042434E+04,.8871475E+03,.8840194E+03,.1109960E+04,&
     & .1304838E+04,.9993418E+03,.8403647E+03,.8358995E+03,.1058827E+04,&
     & .1492071E+04,.1202807E+04,.1075082E+04,.1077418E+04,.1300751E+04,&
     & .1451378E+04,.1152078E+04,.1011888E+04,.1013598E+04,.1240207E+04,&
     & .1408590E+04,.1103420E+04,.9548463E+03,.9546446E+03,.1182081E+04,&
     & .1364310E+04,.1056805E+04,.9029595E+03,.9004074E+03,.1127003E+04,&
     & .1319414E+04,.1012901E+04,.8550197E+03,.8508617E+03,.1074911E+04,&
     & .1504643E+04,.1219744E+04,.1096579E+04,.1098934E+04,.1320376E+04,&
     & .1464791E+04,.1168191E+04,.1031579E+04,.1033563E+04,.1259497E+04,&
     & .1422597E+04,.1118810E+04,.9725173E+03,.9730206E+03,.1200352E+04,&
     & .1378559E+04,.1071486E+04,.9191235E+03,.9172829E+03,.1144372E+04,&
     & .1333898E+04,.1026715E+04,.8699929E+03,.8663091E+03,.1091312E+04,&
     & .1516339E+04,.1236030E+04,.1117522E+04,.1119831E+04,.1339023E+04,&
     & .1477239E+04,.1183650E+04,.1050741E+04,.1052932E+04,.1277904E+04,&
     & .1435679E+04,.1133650E+04,.9897694E+03,.9909248E+03,.1218096E+04,&
     & .1392107E+04,.1085680E+04,.9348858E+03,.9337357E+03,.1161095E+04,&
     & .1347658E+04,.1040131E+04,.8845784E+03,.8813290E+03,.1107099E+04,&
     & .1520825E+04,.1242563E+04,.1126048E+04,.1128336E+04,.1346460E+04,&
     & .1482173E+04,.1189885E+04,.1058578E+04,.1060811E+04,.1285327E+04,&
     & .1440948E+04,.1139665E+04,.9968499E+03,.9981671E+03,.1225222E+04,&
     & .1397470E+04,.1091462E+04,.9413190E+03,.9405053E+03,.1167923E+04,&
     & .1353151E+04,.1045578E+04,.8905298E+03,.8874634E+03,.1113520E+04/

!  ---
      data   forref(1:4,1:12) /  .5532580E-03,.5554860E-03,.6013390E-03,&
     & .7082800E-03,.1585580E-02,.1629570E-02,.2049910E-02,.4758810E-02,&
     & .7725420E-02,.7845620E-02,.1119790E-01,.2290160E-01,.2550970E-01,&
     & .2562720E-01,.2706910E-01,.2595050E-01,.3232630E-01,.3244950E-01,&
     & .3055350E-01,.2639930E-01,.3557280E-01,.3564191E-01,.3329601E-01,&
     & .2956313E-01,.3784510E-01,.3753410E-01,.3743690E-01,.3203340E-01,&
     & .4098704E-01,.3991769E-01,.3953723E-01,.3197576E-01,.4282540E-01,&
     & .4411510E-01,.4088870E-01,.3270770E-01,.4529070E-01,.4526648E-01,&
     & .4032414E-01,.3589280E-01,.4839280E-01,.4772840E-01,.3806840E-01,&
     & .3879400E-01,.5074618E-01,.4940565E-01,.4711472E-01,.3831108E-01/

      data   selfref(1:10,1:12)  /                                      &
     & .1605370E-01,.1490380E-01,.1383630E-01,.1284520E-01,.1192510E-01,&
     & .1107090E-01,.1027790E-01,.9541750E-02,.8858290E-02,.8223790E-02,&
     & .3657530E-01,.3422670E-01,.3202880E-01,.2997200E-01,.2804740E-01,&
     & .2624630E-01,.2456090E-01,.2298370E-01,.2150780E-01,.2012670E-01,&
     & .1274190E+00,.1185530E+00,.1103040E+00,.1026290E+00,.9548830E-01,&
     & .8884420E-01,.8266240E-01,.7691070E-01,.7155930E-01,.6658020E-01,&
     & .3786870E+00,.3489610E+00,.3215680E+00,.2963250E+00,.2730640E+00,&
     & .2516290E+00,.2318760E+00,.2136740E+00,.1969010E+00,.1814440E+00,&
     & .4728220E+00,.4350180E+00,.4002360E+00,.3682360E+00,.3387940E+00,&
     & .3117060E+00,.2867830E+00,.2638540E+00,.2427570E+00,.2233480E+00,&
     & .5168017E+00,.4753661E+00,.4372531E+00,.4021962E+00,.3699493E+00,&
     & .3402876E+00,.3130049E+00,.2879097E+00,.2648259E+00,.2435933E+00,&
     & .5402220E+00,.4977460E+00,.4586100E+00,.4225510E+00,.3893270E+00,&
     & .3587160E+00,.3305110E+00,.3045240E+00,.2805800E+00,.2585190E+00,&
     & .5700865E+00,.5267717E+00,.4867481E+00,.4497659E+00,.4155940E+00,&
     & .3840184E+00,.3548416E+00,.3278825E+00,.3029720E+00,.2799538E+00,&
     & .6451760E+00,.5889570E+00,.5376360E+00,.4907880E+00,.4480220E+00,&
     & .4089820E+00,.3733440E+00,.3408120E+00,.3111140E+00,.2840040E+00,&
     & .6556933E+00,.6013742E+00,.5515585E+00,.5058718E+00,.4639719E+00,&
     & .4255448E+00,.3903026E+00,.3579808E+00,.3283371E+00,.3011502E+00,&
     & .6925540E+00,.6355740E+00,.5832820E+00,.5352930E+00,.4912510E+00,&
     & .4508340E+00,.4137410E+00,.3797010E+00,.3484610E+00,.3197910E+00,&
     & .7229700E+00,.6637789E+00,.6094446E+00,.5595658E+00,.5137786E+00,&
     & .4717450E+00,.4331576E+00,.3977321E+00,.3652099E+00,.3353523E+00/

!........................................!
      end module module_radsw_kgb17      !
!========================================!


!> This module sets up absorption coeffients for band 18: 4000-4650
!! cm-1 (low - h2o,ch4; high - ch4)
!========================================!
      module module_radsw_kgb18          !
!........................................!
!
! *********        the original program descriptions        ********* !
!                                                                     !
!     originally by j.delamere, atmospheric & environmental research. !
!     revision: 2.4                                                   !
!     band 18:  4000-4650 cm-1 (low - h2o,ch4; high - ch4)            !
!     reformatted for f90 by jjmorcrette, ecmwf                       !
!                                                                     !
!     this table has been re-generated for reduced number of g-point  !
!     by y.t.hou, ncep                                                !
!                                                                     !
! *********    *********    end description    *********    ********* !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NG18

!
      implicit none
!
      private
!
!> msa18=585
      integer,  public :: MSA18
!> msb18=235
      integer,  public :: MSB18
!> msf18=10
      integer,  public :: MSF18
!> mfr18=3
      integer,  public :: MFR18
      parameter (MSA18=585, MSB18=235, MSF18=10, MFR18=3)

!>    the array selfref contains the coefficient of the water vapor
!!    self-continuum (including the energy term).  the first index
!!    refers to temperature in 7.2 degree increments.  for instance,
!!    jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!!    etc.  the second index runs over the g-channel (1 to 8).
      real (kind=kind_phys), public :: selfref(MSF18,NG18)

!>    the array absa(585,NG18) (ka(9,5,13,NG18)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels> ~100mb,
!!    temperatures, and binary species parameters (see taumol.f for definition).
!!    the first index in the array, js, runs from 1 to 9, and corresponds
!!    to different values of the binary species parameter.  for instance,
!!    js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!!    js = 3 corresponds to the parameter value 2/8, etc.  the second index
!!    in the array, jt, which runs from 1 to 5, corresponds to different
!!    temperatures.  more specifically, Jt = 3 means that the data are for
!!    the reference temperature tref for this  pressure level, jt = 2 refers
!!    to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!!    is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!!    to the jpth reference pressure level (see taumol.f for these levels
!!    in mb).  the fourth index, ig, goes from 1 to 8, and indicates
!!    which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absa(MSA18,NG18)

!>    the array absb(235,8) (kb(5,13:59,8)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels < ~100mb and
!!    temperatures. the first index in the array, jt, which runs from 1 to 5,
!!    corresponds to different temperatures.  more specifically, jt = 3 means
!!    that the data are for the reference temperature tref for this pressure
!!    level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!!    tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!!    the second index, jp, runs from 13 to 59 and refers to the jpth
!!    reference pressure level (see taumol.f for the value of these
!!    pressure levels in mb).  the third index, ig, goes from 1 to 8,
!!    and tells us which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absb(MSB18,NG18)

      real (kind=kind_phys), public :: forref(MFR18,NG18)

!> rayleigh extinction coefficient at \f$v=4325 cm^{-1}\f$
      real (kind=kind_phys), parameter, public :: rayl = 1.39e-09


      data   absa(  1:180, 1)   /                                       &
     & .1481300E-04,.3984200E-04,.4336200E-04,.4309500E-04,.3981100E-04,&
     & .3499400E-04,.2898400E-04,.2060900E-04,.1884500E-05,.1443200E-04,&
     & .3874900E-04,.4211900E-04,.4183400E-04,.3890000E-04,.3436500E-04,&
     & .2857100E-04,.2050300E-04,.1471500E-05,.1396900E-04,.3766600E-04,&
     & .4094200E-04,.4070300E-04,.3798200E-04,.3373100E-04,.2815800E-04,&
     & .2035100E-04,.1084200E-05,.1336700E-04,.3660500E-04,.3972900E-04,&
     & .3952900E-04,.3703300E-04,.3309600E-04,.2769000E-04,.2015400E-04,&
     & .7870200E-06,.1276500E-04,.3558400E-04,.3853600E-04,.3836100E-04,&
     & .3607300E-04,.3238400E-04,.2724200E-04,.1993200E-04,.5829600E-06,&
     & .1229900E-04,.3417900E-04,.3740500E-04,.3753500E-04,.3471400E-04,&
     & .3033500E-04,.2489200E-04,.1752400E-04,.1330100E-05,.1206400E-04,&
     & .3322500E-04,.3635700E-04,.3651200E-04,.3387000E-04,.2978900E-04,&
     & .2457800E-04,.1743500E-04,.1021700E-05,.1169700E-04,.3230100E-04,&
     & .3531600E-04,.3548100E-04,.3304900E-04,.2922600E-04,.2423300E-04,&
     & .1730400E-04,.7215200E-06,.1120900E-04,.3139500E-04,.3426300E-04,&
     & .3443600E-04,.3217300E-04,.2861000E-04,.2384700E-04,.1716100E-04,&
     & .5178700E-06,.1070600E-04,.3049600E-04,.3331100E-04,.3339600E-04,&
     & .3129500E-04,.2800200E-04,.2344500E-04,.1697100E-04,.4023000E-06,&
     & .1027100E-04,.2927300E-04,.3204900E-04,.3236500E-04,.2985700E-04,&
     & .2617100E-04,.2135200E-04,.1481400E-04,.8047000E-06,.1005800E-04,&
     & .2845800E-04,.3119600E-04,.3144000E-04,.2913300E-04,.2564000E-04,&
     & .2105600E-04,.1475600E-04,.5301000E-06,.9817200E-05,.2765100E-04,&
     & .3035200E-04,.3060900E-04,.2839600E-04,.2511000E-04,.2073000E-04,&
     & .1465500E-04,.3865200E-06,.9457700E-05,.2687400E-04,.2947500E-04,&
     & .2974900E-04,.2766900E-04,.2455300E-04,.2040000E-04,.1453600E-04,&
     & .3323600E-06,.9050800E-05,.2612800E-04,.2863100E-04,.2883400E-04,&
     & .2695200E-04,.2401500E-04,.2004500E-04,.1438200E-04,.3065900E-06,&
     & .8608100E-05,.2489700E-04,.2731700E-04,.2763200E-04,.2546500E-04,&
     & .2232800E-04,.1811200E-04,.1246800E-04,.3928600E-06,.8379300E-05,&
     & .2427300E-04,.2659000E-04,.2686500E-04,.2487600E-04,.2186700E-04,&
     & .1784900E-04,.1242000E-04,.3226600E-06,.8229600E-05,.2358400E-04,&
     & .2586900E-04,.2615500E-04,.2425500E-04,.2140300E-04,.1757900E-04,&
     & .1233800E-04,.3077500E-06,.7973100E-05,.2291500E-04,.2516600E-04,&
     & .2543500E-04,.2362700E-04,.2091700E-04,.1728200E-04,.1224200E-04,&
     & .3041800E-06,.7655300E-05,.2225900E-04,.2444000E-04,.2467300E-04,&
     & .2300300E-04,.2043700E-04,.1698400E-04,.1211200E-04,.3063800E-06/

      data   absa(181:315, 1)   /                                       &
     & .7225900E-05,.2108900E-04,.2317700E-04,.2341900E-04,.2158400E-04,&
     & .1892600E-04,.1525100E-04,.1041700E-04,.2517000E-06,.7021500E-05,&
     & .2056700E-04,.2254100E-04,.2278800E-04,.2110200E-04,.1852600E-04,&
     & .1504900E-04,.1037900E-04,.2595800E-06,.6849000E-05,.2001800E-04,&
     & .2193800E-04,.2220600E-04,.2057900E-04,.1813900E-04,.1480600E-04,&
     & .1031300E-04,.2666600E-06,.6697300E-05,.1944400E-04,.2134000E-04,&
     & .2161500E-04,.2007000E-04,.1772800E-04,.1455400E-04,.1023900E-04,&
     & .2858900E-06,.6480800E-05,.1888300E-04,.2073800E-04,.2099400E-04,&
     & .1954600E-04,.1730400E-04,.1430600E-04,.1014800E-04,.3088600E-06,&
     & .6072700E-05,.1779400E-04,.1960900E-04,.1976900E-04,.1822800E-04,&
     & .1594600E-04,.1279300E-04,.8655900E-05,.1855900E-06,.5894900E-05,&
     & .1739000E-04,.1905900E-04,.1924400E-04,.1782000E-04,.1564700E-04,&
     & .1262600E-04,.8635400E-05,.1954400E-06,.5726800E-05,.1693500E-04,&
     & .1856400E-04,.1872400E-04,.1740200E-04,.1530400E-04,.1243300E-04,&
     & .8588200E-05,.2203300E-06,.5624300E-05,.1645900E-04,.1805400E-04,&
     & .1826300E-04,.1698400E-04,.1497100E-04,.1221500E-04,.8526800E-05,&
     & .2507200E-06,.5458400E-05,.1598400E-04,.1754800E-04,.1777300E-04,&
     & .1655100E-04,.1460100E-04,.1200400E-04,.8456800E-05,.2811000E-06,&
     & .5062400E-05,.1496300E-04,.1648200E-04,.1661100E-04,.1533900E-04,&
     & .1333300E-04,.1065600E-04,.7157000E-05,.1345300E-06,.4942800E-05,&
     & .1463600E-04,.1604000E-04,.1618100E-04,.1499200E-04,.1314400E-04,&
     & .1053300E-04,.7153200E-05,.1562000E-06,.4795500E-05,.1425600E-04,&
     & .1561100E-04,.1573500E-04,.1464300E-04,.1286500E-04,.1039000E-04,&
     & .7124800E-05,.1857100E-06,.4674300E-05,.1387600E-04,.1521300E-04,&
     & .1534200E-04,.1429300E-04,.1257600E-04,.1021500E-04,.7073500E-05,&
     & .2138200E-06,.4570800E-05,.1347300E-04,.1478200E-04,.1494700E-04,&
     & .1393300E-04,.1227400E-04,.1003600E-04,.7017700E-05,.2441900E-06/

      data   absa(316:450, 1)   /                                       &
     & .4228900E-05,.1252800E-04,.1381700E-04,.1392200E-04,.1286700E-04,&
     & .1112600E-04,.8846800E-05,.5901000E-05,.1060000E-06,.4131900E-05,&
     & .1228600E-04,.1348000E-04,.1358200E-04,.1258400E-04,.1097300E-04,&
     & .8764400E-05,.5904900E-05,.1295100E-06,.4031700E-05,.1199600E-04,&
     & .1311600E-04,.1320600E-04,.1229500E-04,.1078600E-04,.8653700E-05,&
     & .5889800E-05,.1573100E-06,.3910700E-05,.1166600E-04,.1277100E-04,&
     & .1287000E-04,.1201000E-04,.1054600E-04,.8524400E-05,.5853600E-05,&
     & .1871600E-06,.3838400E-05,.1134600E-04,.1243500E-04,.1254900E-04,&
     & .1170600E-04,.1028900E-04,.8375600E-05,.5810400E-05,.2167900E-06,&
     & .3524000E-05,.1041800E-04,.1155800E-04,.1168700E-04,.1074800E-04,&
     & .9230500E-05,.7319800E-05,.4851400E-05,.1017600E-06,.3447800E-05,&
     & .1029600E-04,.1129600E-04,.1136900E-04,.1053200E-04,.9137500E-05,&
     & .7273700E-05,.4863900E-05,.1269500E-06,.3371800E-05,.1006800E-04,&
     & .1099600E-04,.1106300E-04,.1029000E-04,.8995700E-05,.7184400E-05,&
     & .4858300E-05,.1548200E-06,.3280100E-05,.9799000E-05,.1071200E-04,&
     & .1076200E-04,.1005600E-04,.8825500E-05,.7090700E-05,.4836400E-05,&
     & .1812300E-06,.3196000E-05,.9535000E-05,.1042700E-04,.1050100E-04,&
     & .9805600E-05,.8615700E-05,.6972300E-05,.4801000E-05,.2073800E-06,&
     & .2919700E-05,.8618300E-05,.9610000E-05,.9744800E-05,.8918700E-05,&
     & .7635400E-05,.6031800E-05,.3973300E-05,.1031000E-06,.2867700E-05,&
     & .8556500E-05,.9419100E-05,.9468200E-05,.8765300E-05,.7572500E-05,&
     & .6008900E-05,.3995600E-05,.1295100E-06,.2803600E-05,.8412100E-05,&
     & .9190200E-05,.9225100E-05,.8581100E-05,.7465300E-05,.5945200E-05,&
     & .3996700E-05,.1568400E-06,.2742000E-05,.8203500E-05,.8944000E-05,&
     & .8974800E-05,.8381700E-05,.7342200E-05,.5872500E-05,.3984400E-05,&
     & .1856400E-06,.2659600E-05,.7973900E-05,.8707400E-05,.8750800E-05,&
     & .8178400E-05,.7178100E-05,.5781500E-05,.3956900E-05,.2147600E-06/

      data   absa(451:585, 1)   /                                       &
     & .2398800E-05,.7113500E-05,.7912100E-05,.8011800E-05,.7337000E-05,&
     & .6284400E-05,.4961000E-05,.3264100E-05,.9169700E-07,.2355200E-05,&
     & .7058700E-05,.7751900E-05,.7782100E-05,.7211300E-05,.6229700E-05,&
     & .4940700E-05,.3281500E-05,.1130700E-06,.2304600E-05,.6936700E-05,&
     & .7566200E-05,.7579000E-05,.7063100E-05,.6140900E-05,.4886700E-05,&
     & .3281200E-05,.1371500E-06,.2255700E-05,.6763200E-05,.7364000E-05,&
     & .7377200E-05,.6897200E-05,.6038100E-05,.4825400E-05,.3270500E-05,&
     & .1643600E-06,.2189900E-05,.6574800E-05,.7169500E-05,.7194600E-05,&
     & .6729500E-05,.5902200E-05,.4748800E-05,.3247600E-05,.1902500E-06,&
     & .1969900E-05,.5867000E-05,.6507500E-05,.6580800E-05,.6030000E-05,&
     & .5166500E-05,.4074900E-05,.2678800E-05,.7710000E-07,.1934800E-05,&
     & .5814600E-05,.6373000E-05,.6389100E-05,.5926000E-05,.5119600E-05,&
     & .4057700E-05,.2692600E-05,.9569300E-07,.1894700E-05,.5713400E-05,&
     & .6220700E-05,.6223800E-05,.5809000E-05,.5045900E-05,.4011300E-05,&
     & .2691600E-05,.1157200E-06,.1855200E-05,.5570900E-05,.6052200E-05,&
     & .6061100E-05,.5669300E-05,.4959400E-05,.3960600E-05,.2682400E-05,&
     & .1379700E-06,.1804000E-05,.5413700E-05,.5893500E-05,.5909600E-05,&
     & .5529400E-05,.4844300E-05,.3897700E-05,.2663300E-05,.1588400E-06,&
     & .1616700E-05,.4827200E-05,.5347500E-05,.5398300E-05,.4950300E-05,&
     & .4241300E-05,.3343100E-05,.2195600E-05,.6138100E-07,.1589500E-05,&
     & .4780900E-05,.5234200E-05,.5240900E-05,.4865200E-05,.4202100E-05,&
     & .3326300E-05,.2207200E-05,.7638200E-07,.1558200E-05,.4698200E-05,&
     & .5105500E-05,.5105800E-05,.4771600E-05,.4140600E-05,.3288600E-05,&
     & .2205800E-05,.9308300E-07,.1524200E-05,.4583100E-05,.4969600E-05,&
     & .4973500E-05,.4653400E-05,.4067800E-05,.3246900E-05,.2196900E-05,&
     & .1121100E-06,.1485900E-05,.4449200E-05,.4839200E-05,.4847900E-05,&
     & .4537800E-05,.3968900E-05,.3193300E-05,.2181300E-05,.1297000E-06/

      data   absa(  1:180, 2)   /                                       &
     & .1013800E-03,.2045600E-03,.2211700E-03,.2194800E-03,.2080500E-03,&
     & .1837300E-03,.1499900E-03,.9538200E-04,.3321300E-05,.1014100E-03,&
     & .2088900E-03,.2245600E-03,.2222300E-03,.2091200E-03,.1846700E-03,&
     & .1506000E-03,.9571300E-04,.2847900E-05,.1012500E-03,.2122300E-03,&
     & .2271200E-03,.2238800E-03,.2098300E-03,.1851300E-03,.1510000E-03,&
     & .9588100E-04,.2633800E-05,.1010500E-03,.2144200E-03,.2286300E-03,&
     & .2242100E-03,.2104600E-03,.1854000E-03,.1506000E-03,.9579800E-04,&
     & .2763800E-05,.1004000E-03,.2161200E-03,.2289900E-03,.2241600E-03,&
     & .2104600E-03,.1857000E-03,.1500100E-03,.9556900E-04,.2931300E-05,&
     & .8963900E-04,.1779700E-03,.1928400E-03,.1898600E-03,.1782400E-03,&
     & .1578900E-03,.1290600E-03,.8352400E-04,.2392600E-05,.8949900E-04,&
     & .1814400E-03,.1956300E-03,.1922700E-03,.1794100E-03,.1584400E-03,&
     & .1296100E-03,.8363200E-04,.2196900E-05,.8941100E-04,.1840100E-03,&
     & .1975300E-03,.1935000E-03,.1799900E-03,.1588300E-03,.1298900E-03,&
     & .8357700E-04,.2315800E-05,.8884100E-04,.1854100E-03,.1987200E-03,&
     & .1939800E-03,.1805300E-03,.1590700E-03,.1297200E-03,.8339600E-04,&
     & .2376000E-05,.8794900E-04,.1865900E-03,.1990200E-03,.1940700E-03,&
     & .1807300E-03,.1591000E-03,.1293700E-03,.8312000E-04,.2524700E-05,&
     & .7746600E-04,.1526400E-03,.1670400E-03,.1634200E-03,.1530000E-03,&
     & .1349900E-03,.1100700E-03,.7188100E-04,.1783900E-05,.7748300E-04,&
     & .1561100E-03,.1695000E-03,.1655100E-03,.1540800E-03,.1358000E-03,&
     & .1105500E-03,.7202000E-04,.1856400E-05,.7714200E-04,.1586700E-03,&
     & .1711200E-03,.1667200E-03,.1548100E-03,.1361600E-03,.1107700E-03,&
     & .7192200E-04,.2022800E-05,.7646000E-04,.1605800E-03,.1718600E-03,&
     & .1671700E-03,.1551900E-03,.1364100E-03,.1106000E-03,.7177500E-04,&
     & .2171700E-05,.7551000E-04,.1615800E-03,.1721000E-03,.1673000E-03,&
     & .1552100E-03,.1364900E-03,.1102200E-03,.7151600E-04,.2314000E-05,&
     & .6593100E-04,.1294000E-03,.1429500E-03,.1400500E-03,.1309500E-03,&
     & .1149400E-03,.9365600E-04,.6116700E-04,.1709400E-05,.6609000E-04,&
     & .1326900E-03,.1454200E-03,.1423300E-03,.1320100E-03,.1155800E-03,&
     & .9397500E-04,.6143600E-04,.1852700E-05,.6563000E-04,.1353200E-03,&
     & .1471700E-03,.1436400E-03,.1326600E-03,.1160400E-03,.9412600E-04,&
     & .6146700E-04,.1957600E-05,.6497100E-04,.1370800E-03,.1481800E-03,&
     & .1442700E-03,.1330400E-03,.1161600E-03,.9415200E-04,.6133400E-04,&
     & .2061400E-05,.6412200E-04,.1383400E-03,.1486400E-03,.1445100E-03,&
     & .1331200E-03,.1163800E-03,.9395500E-04,.6116200E-04,.2164000E-05/

      data   absa(181:315, 2)   /                                       &
     & .5559400E-04,.1083000E-03,.1206800E-03,.1186500E-03,.1112700E-03,&
     & .9792500E-04,.7920800E-04,.5184200E-04,.1595400E-05,.5566900E-04,&
     & .1111700E-03,.1229900E-03,.1209100E-03,.1124800E-03,.9864300E-04,&
     & .7943300E-04,.5212700E-04,.1696600E-05,.5543000E-04,.1134500E-03,&
     & .1247800E-03,.1223500E-03,.1133200E-03,.9911700E-04,.7982300E-04,&
     & .5228600E-04,.1830400E-05,.5480000E-04,.1154300E-03,.1260100E-03,&
     & .1230400E-03,.1138900E-03,.9934600E-04,.7993500E-04,.5227500E-04,&
     & .1947900E-05,.5402700E-04,.1166500E-03,.1266000E-03,.1233900E-03,&
     & .1141000E-03,.9953500E-04,.7984000E-04,.5216800E-04,.2058300E-05,&
     & .4650000E-04,.8988700E-04,.1004800E-03,.9916100E-04,.9313600E-04,&
     & .8227900E-04,.6687200E-04,.4361700E-04,.1473000E-05,.4663200E-04,&
     & .9244400E-04,.1027100E-03,.1012300E-03,.9455900E-04,.8313500E-04,&
     & .6720400E-04,.4404600E-04,.1621500E-05,.4649000E-04,.9441500E-04,&
     & .1044600E-03,.1026900E-03,.9538700E-04,.8367600E-04,.6757900E-04,&
     & .4427900E-04,.1735800E-05,.4600400E-04,.9619000E-04,.1057000E-03,&
     & .1034600E-03,.9588300E-04,.8403800E-04,.6775300E-04,.4434800E-04,&
     & .1809600E-05,.4536200E-04,.9751100E-04,.1064500E-03,.1038400E-03,&
     & .9619700E-04,.8434700E-04,.6775800E-04,.4430500E-04,.1888200E-05,&
     & .3859700E-04,.7401900E-04,.8288200E-04,.8180700E-04,.7693600E-04,&
     & .6827800E-04,.5574400E-04,.3650700E-04,.1340000E-05,.3872800E-04,&
     & .7635300E-04,.8503000E-04,.8382200E-04,.7852300E-04,.6922100E-04,&
     & .5619900E-04,.3693200E-04,.1476600E-05,.3868800E-04,.7816400E-04,&
     & .8669100E-04,.8526900E-04,.7942400E-04,.6984900E-04,.5650700E-04,&
     & .3721100E-04,.1607700E-05,.3839900E-04,.7965400E-04,.8788200E-04,&
     & .8614500E-04,.7991700E-04,.7021400E-04,.5669800E-04,.3730200E-04,&
     & .1736500E-05,.3789900E-04,.8101200E-04,.8870600E-04,.8658900E-04,&
     & .8025900E-04,.7042900E-04,.5681000E-04,.3728700E-04,.1859500E-05/

      data   absa(316:450, 2)   /                                       &
     & .3195900E-04,.6067100E-04,.6796300E-04,.6705300E-04,.6305700E-04,&
     & .5628700E-04,.4618700E-04,.3029800E-04,.1207100E-05,.3206200E-04,&
     & .6275300E-04,.6989200E-04,.6894100E-04,.6462400E-04,.5718700E-04,&
     & .4656600E-04,.3068700E-04,.1358200E-05,.3209800E-04,.6443500E-04,&
     & .7142300E-04,.7035900E-04,.6564600E-04,.5788200E-04,.4689700E-04,&
     & .3099300E-04,.1521700E-05,.3195600E-04,.6575800E-04,.7265000E-04,&
     & .7127500E-04,.6627500E-04,.5829800E-04,.4710000E-04,.3107900E-04,&
     & .1679400E-05,.3158400E-04,.6691600E-04,.7350200E-04,.7181900E-04,&
     & .6661100E-04,.5850500E-04,.4724300E-04,.3108800E-04,.1825000E-05,&
     & .2635500E-04,.4947200E-04,.5541500E-04,.5458400E-04,.5136100E-04,&
     & .4613500E-04,.3802700E-04,.2498300E-04,.1354900E-05,.2647200E-04,&
     & .5130200E-04,.5716500E-04,.5641700E-04,.5293600E-04,.4704800E-04,&
     & .3841700E-04,.2534700E-04,.1494100E-05,.2654500E-04,.5286600E-04,&
     & .5856300E-04,.5777600E-04,.5398800E-04,.4770600E-04,.3873900E-04,&
     & .2563800E-04,.1656300E-05,.2646900E-04,.5410500E-04,.5974900E-04,&
     & .5874600E-04,.5466800E-04,.4816100E-04,.3897000E-04,.2574400E-04,&
     & .1826200E-05,.2624800E-04,.5507900E-04,.6061900E-04,.5931700E-04,&
     & .5509900E-04,.4843500E-04,.3911500E-04,.2577800E-04,.2023800E-05,&
     & .2163100E-04,.4029700E-04,.4517000E-04,.4429800E-04,.4187600E-04,&
     & .3769000E-04,.3121200E-04,.2055200E-04,.1632800E-05,.2181400E-04,&
     & .4189600E-04,.4668100E-04,.4606500E-04,.4322700E-04,.3860800E-04,&
     & .3161400E-04,.2085700E-04,.1861000E-05,.2186400E-04,.4324700E-04,&
     & .4794500E-04,.4732700E-04,.4427300E-04,.3920100E-04,.3189400E-04,&
     & .2111900E-04,.2077500E-05,.2183500E-04,.4438600E-04,.4898600E-04,&
     & .4822800E-04,.4491600E-04,.3963700E-04,.3209300E-04,.2122800E-04,&
     & .2254100E-05,.2172800E-04,.4526300E-04,.4980400E-04,.4881800E-04,&
     & .4540900E-04,.3995000E-04,.3226900E-04,.2129000E-04,.2452500E-05/

      data   absa(451:585, 2)   /                                       &
     & .1782300E-04,.3337800E-04,.3733700E-04,.3661900E-04,.3459800E-04,&
     & .3111300E-04,.2576100E-04,.1695600E-04,.1504200E-05,.1797000E-04,&
     & .3467900E-04,.3858900E-04,.3806500E-04,.3569500E-04,.3186900E-04,&
     & .2604400E-04,.1721400E-04,.1684700E-05,.1799700E-04,.3579100E-04,&
     & .3961900E-04,.3909700E-04,.3653900E-04,.3237400E-04,.2629800E-04,&
     & .1742800E-04,.1888700E-05,.1796700E-04,.3671700E-04,.4046300E-04,&
     & .3982100E-04,.3707100E-04,.3271600E-04,.2646100E-04,.1749600E-04,&
     & .2054600E-05,.1786800E-04,.3742300E-04,.4112400E-04,.4030400E-04,&
     & .3747300E-04,.3297100E-04,.2661200E-04,.1753200E-04,.2224000E-05,&
     & .1466500E-04,.2760500E-04,.3081100E-04,.3025000E-04,.2852500E-04,&
     & .2564400E-04,.2118600E-04,.1395500E-04,.1258800E-05,.1477500E-04,&
     & .2865500E-04,.3185100E-04,.3140800E-04,.2941800E-04,.2626100E-04,&
     & .2143000E-04,.1417200E-04,.1418100E-05,.1479900E-04,.2957100E-04,&
     & .3268700E-04,.3224200E-04,.3009300E-04,.2668300E-04,.2163600E-04,&
     & .1433800E-04,.1561900E-05,.1477700E-04,.3032500E-04,.3337100E-04,&
     & .3282600E-04,.3055300E-04,.2694500E-04,.2177500E-04,.1438000E-04,&
     & .1691700E-05,.1468500E-04,.3090400E-04,.3390700E-04,.3321400E-04,&
     & .3087500E-04,.2715800E-04,.2188300E-04,.1441400E-04,.1842300E-05,&
     & .1204400E-04,.2277200E-04,.2537500E-04,.2492600E-04,.2345800E-04,&
     & .2105900E-04,.1738700E-04,.1145700E-04,.9912200E-06,.1212800E-04,&
     & .2363700E-04,.2624900E-04,.2584200E-04,.2418700E-04,.2158200E-04,&
     & .1759000E-04,.1163600E-04,.1103700E-05,.1215200E-04,.2438600E-04,&
     & .2691000E-04,.2653100E-04,.2473100E-04,.2192300E-04,.1776000E-04,&
     & .1175500E-04,.1202600E-05,.1213000E-04,.2499600E-04,.2746400E-04,&
     & .2700600E-04,.2511300E-04,.2213800E-04,.1787400E-04,.1179700E-04,&
     & .1302800E-05,.1205000E-04,.2547400E-04,.2789500E-04,.2729700E-04,&
     & .2537300E-04,.2230200E-04,.1794800E-04,.1181700E-04,.1401000E-05/

      data   absa(  1:180, 3)   /                                       &
     & .3682300E-03,.4961300E-03,.5181700E-03,.5053800E-03,.4571100E-03,&
     & .3960500E-03,.3148600E-03,.2147400E-03,.1093200E-04,.3700800E-03,&
     & .5077800E-03,.5336100E-03,.5177800E-03,.4699500E-03,.4062900E-03,&
     & .3230900E-03,.2187800E-03,.1142100E-04,.3698900E-03,.5181200E-03,&
     & .5469900E-03,.5292200E-03,.4803600E-03,.4155900E-03,.3296500E-03,&
     & .2222600E-03,.1239500E-04,.3672100E-03,.5276300E-03,.5574600E-03,&
     & .5395700E-03,.4889100E-03,.4231300E-03,.3354500E-03,.2258800E-03,&
     & .1351500E-04,.3631300E-03,.5365200E-03,.5668000E-03,.5479100E-03,&
     & .4966400E-03,.4290700E-03,.3404300E-03,.2291600E-03,.1473400E-04,&
     & .3241100E-03,.4281500E-03,.4472500E-03,.4381300E-03,.4015000E-03,&
     & .3487100E-03,.2791200E-03,.1892900E-03,.8074100E-05,.3256700E-03,&
     & .4378600E-03,.4606900E-03,.4502700E-03,.4131900E-03,.3581900E-03,&
     & .2863500E-03,.1932500E-03,.8734000E-05,.3242900E-03,.4463800E-03,&
     & .4722800E-03,.4607400E-03,.4227200E-03,.3660200E-03,.2920100E-03,&
     & .1965300E-03,.9611000E-05,.3213700E-03,.4553100E-03,.4814200E-03,&
     & .4695100E-03,.4299600E-03,.3725000E-03,.2970900E-03,.1997400E-03,&
     & .1069700E-04,.3177600E-03,.4620700E-03,.4888000E-03,.4765900E-03,&
     & .4355100E-03,.3777600E-03,.3013800E-03,.2027400E-03,.1170400E-04,&
     & .2848800E-03,.3676700E-03,.3808100E-03,.3743800E-03,.3452500E-03,&
     & .3015000E-03,.2433200E-03,.1641500E-03,.6397000E-05,.2853900E-03,&
     & .3750900E-03,.3919600E-03,.3851000E-03,.3556600E-03,.3099800E-03,&
     & .2496300E-03,.1676600E-03,.7200600E-05,.2839700E-03,.3821500E-03,&
     & .4015600E-03,.3942800E-03,.3641800E-03,.3172700E-03,.2544600E-03,&
     & .1710700E-03,.7957400E-05,.2813400E-03,.3886700E-03,.4095900E-03,&
     & .4023000E-03,.3709300E-03,.3230100E-03,.2589800E-03,.1744700E-03,&
     & .8713100E-05,.2788600E-03,.3936000E-03,.4164200E-03,.4085200E-03,&
     & .3760900E-03,.3274000E-03,.2631200E-03,.1773700E-03,.9470400E-05,&
     & .2492300E-03,.3155500E-03,.3242300E-03,.3173100E-03,.2941000E-03,&
     & .2567100E-03,.2076200E-03,.1407400E-03,.5587700E-05,.2497700E-03,&
     & .3212000E-03,.3337200E-03,.3265700E-03,.3029500E-03,.2647200E-03,&
     & .2130300E-03,.1439600E-03,.6157500E-05,.2486500E-03,.3266800E-03,&
     & .3418000E-03,.3344600E-03,.3103300E-03,.2714400E-03,.2177700E-03,&
     & .1471400E-03,.6791600E-05,.2470900E-03,.3313300E-03,.3484100E-03,&
     & .3413400E-03,.3160000E-03,.2764700E-03,.2217900E-03,.1501600E-03,&
     & .7465300E-05,.2445500E-03,.3349100E-03,.3538800E-03,.3470800E-03,&
     & .3206900E-03,.2800500E-03,.2252200E-03,.1524200E-03,.8161900E-05/

      data   absa(181:315, 3)   /                                       &
     & .2152600E-03,.2711500E-03,.2750500E-03,.2680200E-03,.2496800E-03,&
     & .2169000E-03,.1748100E-03,.1190600E-03,.5085000E-05,.2161100E-03,&
     & .2762100E-03,.2829800E-03,.2762000E-03,.2572200E-03,.2241100E-03,&
     & .1800400E-03,.1219700E-03,.5573800E-05,.2158300E-03,.2806000E-03,&
     & .2895500E-03,.2828200E-03,.2634600E-03,.2300500E-03,.1841500E-03,&
     & .1245100E-03,.6086400E-05,.2144600E-03,.2836300E-03,.2946900E-03,&
     & .2889600E-03,.2686000E-03,.2343500E-03,.1876800E-03,.1269000E-03,&
     & .6641700E-05,.2121700E-03,.2862600E-03,.2992900E-03,.2937000E-03,&
     & .2725000E-03,.2376300E-03,.1907300E-03,.1290900E-03,.7212200E-05,&
     & .1826200E-03,.2311200E-03,.2338800E-03,.2268100E-03,.2106200E-03,&
     & .1819600E-03,.1463400E-03,.9970700E-04,.4807800E-05,.1839300E-03,&
     & .2359100E-03,.2409400E-03,.2336400E-03,.2166300E-03,.1881400E-03,&
     & .1511400E-03,.1020500E-03,.5225200E-05,.1837900E-03,.2398400E-03,&
     & .2467600E-03,.2395600E-03,.2222100E-03,.1935600E-03,.1548900E-03,&
     & .1043400E-03,.5640500E-05,.1827000E-03,.2425800E-03,.2510200E-03,&
     & .2445800E-03,.2268500E-03,.1977100E-03,.1580500E-03,.1065500E-03,&
     & .6072800E-05,.1806700E-03,.2444600E-03,.2546700E-03,.2489400E-03,&
     & .2305100E-03,.2005600E-03,.1608800E-03,.1084400E-03,.6516000E-05,&
     & .1526900E-03,.1940700E-03,.1964000E-03,.1907300E-03,.1771500E-03,&
     & .1522400E-03,.1218100E-03,.8330200E-04,.4518500E-05,.1545000E-03,&
     & .1983400E-03,.2024600E-03,.1967500E-03,.1824200E-03,.1577600E-03,&
     & .1261000E-03,.8530600E-04,.4931500E-05,.1545900E-03,.2017800E-03,&
     & .2075900E-03,.2017600E-03,.1873700E-03,.1623300E-03,.1295100E-03,&
     & .8741700E-04,.5349100E-05,.1536800E-03,.2041000E-03,.2116000E-03,&
     & .2062300E-03,.1916100E-03,.1660300E-03,.1324600E-03,.8923700E-04,&
     & .5756200E-05,.1520700E-03,.2056700E-03,.2145800E-03,.2099100E-03,&
     & .1952500E-03,.1690600E-03,.1348100E-03,.9093200E-04,.6165900E-05/

      data   absa(316:450, 3)   /                                       &
     & .1267200E-03,.1615900E-03,.1632900E-03,.1582100E-03,.1474600E-03,&
     & .1262400E-03,.1011600E-03,.6930600E-04,.4665200E-05,.1285300E-03,&
     & .1653900E-03,.1685900E-03,.1635900E-03,.1520800E-03,.1312000E-03,&
     & .1051300E-03,.7100500E-04,.5139200E-05,.1290100E-03,.1684100E-03,&
     & .1730100E-03,.1681300E-03,.1561400E-03,.1353200E-03,.1082900E-03,&
     & .7286300E-04,.5524400E-05,.1283300E-03,.1705800E-03,.1765200E-03,&
     & .1718400E-03,.1597100E-03,.1386700E-03,.1108700E-03,.7447500E-04,&
     & .5819300E-05,.1271500E-03,.1719700E-03,.1791800E-03,.1751300E-03,&
     & .1629200E-03,.1414400E-03,.1130100E-03,.7596500E-04,.6170500E-05,&
     & .1042800E-03,.1334800E-03,.1346200E-03,.1299500E-03,.1215800E-03,&
     & .1037900E-03,.8300500E-04,.5737100E-04,.5596500E-05,.1061900E-03,&
     & .1371000E-03,.1393200E-03,.1346900E-03,.1253800E-03,.1080000E-03,&
     & .8663200E-04,.5895400E-04,.6196100E-05,.1069000E-03,.1397800E-03,&
     & .1431500E-03,.1387700E-03,.1288500E-03,.1116900E-03,.8964300E-04,&
     & .6056800E-04,.6852600E-05,.1066800E-03,.1417800E-03,.1462600E-03,&
     & .1421300E-03,.1320700E-03,.1146800E-03,.9198400E-04,.6207000E-04,&
     & .7512200E-05,.1058200E-03,.1431600E-03,.1487300E-03,.1450100E-03,&
     & .1347100E-03,.1170400E-03,.9387600E-04,.6338800E-04,.8076100E-05,&
     & .8547600E-04,.1098200E-03,.1103400E-03,.1064300E-03,.9970200E-04,&
     & .8496700E-04,.6800800E-04,.4722900E-04,.6614500E-05,.8724200E-04,&
     & .1132200E-03,.1146300E-03,.1104500E-03,.1029300E-03,.8858000E-04,&
     & .7111300E-04,.4863100E-04,.7545200E-05,.8811100E-04,.1156300E-03,&
     & .1179200E-03,.1140400E-03,.1058700E-03,.9187700E-04,.7379000E-04,&
     & .5004400E-04,.8488100E-05,.8820000E-04,.1174300E-03,.1206500E-03,&
     & .1170800E-03,.1087000E-03,.9448000E-04,.7589700E-04,.5138400E-04,&
     & .9513600E-05,.8758300E-04,.1187100E-03,.1229200E-03,.1195400E-03,&
     & .1110300E-03,.9652300E-04,.7752600E-04,.5251800E-04,.1053200E-04/

      data   absa(451:585, 3)   /                                       &
     & .7055600E-04,.9142800E-04,.9186700E-04,.8846300E-04,.8284600E-04,&
     & .7077400E-04,.5663200E-04,.3916700E-04,.6187200E-05,.7193300E-04,&
     & .9418200E-04,.9526900E-04,.9164600E-04,.8538300E-04,.7373800E-04,&
     & .5921900E-04,.4047400E-04,.6996500E-05,.7256700E-04,.9609800E-04,&
     & .9792700E-04,.9460600E-04,.8783700E-04,.7637000E-04,.6132600E-04,&
     & .4163300E-04,.7823500E-05,.7252900E-04,.9753200E-04,.1001100E-03,&
     & .9707600E-04,.9013400E-04,.7842800E-04,.6301600E-04,.4275400E-04,&
     & .8790400E-05,.7199000E-04,.9855200E-04,.1019300E-03,.9908000E-04,&
     & .9209100E-04,.8006800E-04,.6434900E-04,.4367400E-04,.9801100E-05,&
     & .5807200E-04,.7581500E-04,.7616800E-04,.7325200E-04,.6852100E-04,&
     & .5874800E-04,.4702900E-04,.3243000E-04,.5183800E-05,.5914900E-04,&
     & .7800500E-04,.7888900E-04,.7585300E-04,.7069100E-04,.6116000E-04,&
     & .4910500E-04,.3357200E-04,.5866900E-05,.5961500E-04,.7957200E-04,&
     & .8104300E-04,.7830100E-04,.7271600E-04,.6326500E-04,.5077700E-04,&
     & .3450600E-04,.6646400E-05,.5950600E-04,.8071200E-04,.8281100E-04,&
     & .8032100E-04,.7457100E-04,.6493500E-04,.5216500E-04,.3544700E-04,&
     & .7441200E-05,.5906700E-04,.8152000E-04,.8428800E-04,.8197700E-04,&
     & .7617200E-04,.6625900E-04,.5328100E-04,.3617800E-04,.8246500E-05,&
     & .4764000E-04,.6258300E-04,.6288700E-04,.6048500E-04,.5655900E-04,&
     & .4862400E-04,.3890400E-04,.2677300E-04,.4058200E-05,.4849800E-04,&
     & .6437900E-04,.6508700E-04,.6264100E-04,.5837700E-04,.5057200E-04,&
     & .4056600E-04,.2771700E-04,.4661600E-05,.4884400E-04,.6565600E-04,&
     & .6685100E-04,.6462800E-04,.6004200E-04,.5225400E-04,.4191700E-04,&
     & .2851000E-04,.5320400E-05,.4872200E-04,.6657400E-04,.6830300E-04,&
     & .6630000E-04,.6154700E-04,.5361400E-04,.4305800E-04,.2925400E-04,&
     & .5944200E-05,.4835500E-04,.6721700E-04,.6951100E-04,.6765100E-04,&
     & .6286400E-04,.5471200E-04,.4399800E-04,.2987500E-04,.6643600E-05/

      data   absa(  1:180, 4)   /                                       &
     & .8538700E-03,.9726800E-03,.9520100E-03,.8922600E-03,.8146500E-03,&
     & .6970700E-03,.5482900E-03,.3588200E-03,.3792000E-04,.8549600E-03,&
     & .9874600E-03,.9718800E-03,.9177500E-03,.8352900E-03,.7164900E-03,&
     & .5637600E-03,.3706100E-03,.4315500E-04,.8515400E-03,.9990800E-03,&
     & .9896900E-03,.9401400E-03,.8560400E-03,.7337900E-03,.5783200E-03,&
     & .3818300E-03,.4870200E-04,.8460600E-03,.1009000E-02,.1006800E-02,&
     & .9594400E-03,.8743400E-03,.7497500E-03,.5924300E-03,.3918800E-03,&
     & .5402200E-04,.8406100E-03,.1017100E-02,.1021800E-02,.9768400E-03,&
     & .8890600E-03,.7634500E-03,.6047100E-03,.4018000E-03,.5977800E-04,&
     & .7610600E-03,.8676300E-03,.8462000E-03,.7933700E-03,.7236800E-03,&
     & .6263300E-03,.4916800E-03,.3234600E-03,.3132200E-04,.7615300E-03,&
     & .8814800E-03,.8634900E-03,.8155900E-03,.7448000E-03,.6444800E-03,&
     & .5063500E-03,.3343000E-03,.3556600E-04,.7597700E-03,.8919500E-03,&
     & .8787100E-03,.8360500E-03,.7628600E-03,.6612700E-03,.5195900E-03,&
     & .3447200E-03,.3970500E-04,.7577100E-03,.8998700E-03,.8940600E-03,&
     & .8553000E-03,.7812300E-03,.6761200E-03,.5322100E-03,.3544700E-03,&
     & .4426600E-04,.7528900E-03,.9065700E-03,.9077100E-03,.8720000E-03,&
     & .7971500E-03,.6892600E-03,.5438200E-03,.3637200E-03,.4870800E-04,&
     & .6689300E-03,.7624600E-03,.7433400E-03,.6975200E-03,.6350100E-03,&
     & .5536200E-03,.4345500E-03,.2884300E-03,.2426000E-04,.6711800E-03,&
     & .7758800E-03,.7594400E-03,.7171000E-03,.6546100E-03,.5712200E-03,&
     & .4499100E-03,.2990600E-03,.2743400E-04,.6721800E-03,.7850500E-03,&
     & .7732800E-03,.7349700E-03,.6728600E-03,.5871800E-03,.4637500E-03,&
     & .3093300E-03,.3080800E-04,.6705800E-03,.7924200E-03,.7862700E-03,&
     & .7518100E-03,.6894600E-03,.6018300E-03,.4753200E-03,.3174300E-03,&
     & .3432400E-04,.6646700E-03,.7982200E-03,.7977600E-03,.7677900E-03,&
     & .7048100E-03,.6151000E-03,.4855300E-03,.3252600E-03,.3790500E-04,&
     & .5829700E-03,.6602000E-03,.6433700E-03,.6061000E-03,.5522200E-03,&
     & .4833800E-03,.3815400E-03,.2553700E-03,.2005400E-04,.5861600E-03,&
     & .6731100E-03,.6576400E-03,.6236200E-03,.5699600E-03,.4993900E-03,&
     & .3965700E-03,.2641800E-03,.2245700E-04,.5871900E-03,.6829100E-03,&
     & .6701400E-03,.6392600E-03,.5870300E-03,.5145300E-03,.4097600E-03,&
     & .2728000E-03,.2497400E-04,.5845100E-03,.6892500E-03,.6814400E-03,&
     & .6541000E-03,.6031200E-03,.5290300E-03,.4214700E-03,.2809600E-03,&
     & .2761300E-04,.5788500E-03,.6936900E-03,.6915300E-03,.6672500E-03,&
     & .6166600E-03,.5411400E-03,.4312300E-03,.2890700E-03,.3024300E-04/

      data   absa(181:315, 4)   /                                       &
     & .5063900E-03,.5689700E-03,.5541200E-03,.5212700E-03,.4734100E-03,&
     & .4166900E-03,.3329300E-03,.2221500E-03,.1705700E-04,.5106500E-03,&
     & .5811100E-03,.5674000E-03,.5364200E-03,.4897000E-03,.4316600E-03,&
     & .3467100E-03,.2311100E-03,.1919100E-04,.5113100E-03,.5892300E-03,&
     & .5780200E-03,.5498900E-03,.5051900E-03,.4452000E-03,.3586700E-03,&
     & .2395500E-03,.2123800E-04,.5088800E-03,.5943200E-03,.5870000E-03,&
     & .5623800E-03,.5189900E-03,.4578700E-03,.3696000E-03,.2473300E-03,&
     & .2344200E-04,.5041100E-03,.5980400E-03,.5949700E-03,.5734200E-03,&
     & .5315800E-03,.4692400E-03,.3788600E-03,.2545000E-03,.2575600E-04,&
     & .4367300E-03,.4878200E-03,.4749900E-03,.4462900E-03,.4035400E-03,&
     & .3545700E-03,.2860500E-03,.1908600E-03,.1470200E-04,.4410200E-03,&
     & .4983100E-03,.4863900E-03,.4594300E-03,.4187900E-03,.3680800E-03,&
     & .2980200E-03,.1998500E-03,.1647100E-04,.4422100E-03,.5051800E-03,&
     & .4953800E-03,.4707500E-03,.4318300E-03,.3799500E-03,.3088200E-03,&
     & .2076300E-03,.1842000E-04,.4401700E-03,.5099400E-03,.5032300E-03,&
     & .4814100E-03,.4436000E-03,.3911600E-03,.3184400E-03,.2146000E-03,&
     & .2047300E-04,.4365600E-03,.5124900E-03,.5100000E-03,.4906800E-03,&
     & .4542500E-03,.4010700E-03,.3265300E-03,.2210900E-03,.2260000E-04,&
     & .3762700E-03,.4152500E-03,.4036500E-03,.3792300E-03,.3425700E-03,&
     & .3014700E-03,.2432400E-03,.1609400E-03,.1340000E-04,.3805100E-03,&
     & .4248900E-03,.4140000E-03,.3906700E-03,.3555500E-03,.3125700E-03,&
     & .2535300E-03,.1694700E-03,.1490500E-04,.3819600E-03,.4315400E-03,&
     & .4216900E-03,.4003500E-03,.3670300E-03,.3231100E-03,.2629100E-03,&
     & .1767200E-03,.1656600E-04,.3810800E-03,.4358700E-03,.4280900E-03,&
     & .4089000E-03,.3769400E-03,.3324600E-03,.2709400E-03,.1830500E-03,&
     & .1835200E-04,.3785400E-03,.4388900E-03,.4340200E-03,.4169700E-03,&
     & .3853200E-03,.3407400E-03,.2781600E-03,.1887300E-03,.2022000E-04/

      data   absa(316:450, 4)   /                                       &
     & .3196800E-03,.3530100E-03,.3431000E-03,.3220600E-03,.2890400E-03,&
     & .2535600E-03,.2052300E-03,.1343200E-03,.1354800E-04,.3238600E-03,&
     & .3622900E-03,.3528800E-03,.3321400E-03,.3003300E-03,.2634500E-03,&
     & .2137400E-03,.1419800E-03,.1496600E-04,.3252100E-03,.3684800E-03,&
     & .3598600E-03,.3405600E-03,.3104200E-03,.2722200E-03,.2211500E-03,&
     & .1484100E-03,.1627800E-04,.3252700E-03,.3725800E-03,.3655300E-03,&
     & .3481100E-03,.3191600E-03,.2803400E-03,.2283700E-03,.1544300E-03,&
     & .1761400E-04,.3227200E-03,.3753300E-03,.3702500E-03,.3543800E-03,&
     & .3263200E-03,.2874000E-03,.2348400E-03,.1595200E-03,.1907500E-04,&
     & .2679400E-03,.2965500E-03,.2885100E-03,.2715500E-03,.2429000E-03,&
     & .2118200E-03,.1718700E-03,.1115800E-03,.1689300E-04,.2721500E-03,&
     & .3047700E-03,.2976500E-03,.2807500E-03,.2533600E-03,.2212000E-03,&
     & .1785800E-03,.1179400E-03,.1857600E-04,.2740000E-03,.3106100E-03,&
     & .3044700E-03,.2883600E-03,.2622400E-03,.2288600E-03,.1847400E-03,&
     & .1237200E-03,.2005600E-04,.2738500E-03,.3142400E-03,.3092300E-03,&
     & .2947700E-03,.2697700E-03,.2355400E-03,.1909300E-03,.1288400E-03,&
     & .2132100E-04,.2719600E-03,.3164700E-03,.3130800E-03,.3000600E-03,&
     & .2760400E-03,.2419100E-03,.1964900E-03,.1335200E-03,.2261100E-04,&
     & .2225500E-03,.2471600E-03,.2408700E-03,.2267200E-03,.2021600E-03,&
     & .1758200E-03,.1433900E-03,.9252800E-04,.2247100E-04,.2266300E-03,&
     & .2542900E-03,.2490800E-03,.2350600E-03,.2116700E-03,.1837700E-03,&
     & .1491000E-03,.9797200E-04,.2453600E-04,.2286700E-03,.2596300E-03,&
     & .2553200E-03,.2416900E-03,.2192300E-03,.1906700E-03,.1545800E-03,&
     & .1029300E-03,.2645300E-04,.2285300E-03,.2627300E-03,.2596000E-03,&
     & .2470400E-03,.2254900E-03,.1967900E-03,.1596800E-03,.1072800E-03,&
     & .2838300E-04,.2270400E-03,.2646900E-03,.2625900E-03,.2514100E-03,&
     & .2308000E-03,.2021600E-03,.1643300E-03,.1111600E-03,.3011100E-04/

      data   absa(451:585, 4)   /                                       &
     & .1857000E-03,.2079100E-03,.2031000E-03,.1910600E-03,.1706600E-03,&
     & .1479700E-03,.1209700E-03,.7870400E-04,.2072000E-04,.1886800E-03,&
     & .2135600E-03,.2096900E-03,.1980700E-03,.1782100E-03,.1543400E-03,&
     & .1258400E-03,.8317900E-04,.2326800E-04,.1897200E-03,.2173200E-03,&
     & .2144700E-03,.2030500E-03,.1841600E-03,.1599300E-03,.1300600E-03,&
     & .8713400E-04,.2578100E-04,.1892900E-03,.2195500E-03,.2177100E-03,&
     & .2071100E-03,.1890700E-03,.1649300E-03,.1342000E-03,.9069900E-04,&
     & .2818100E-04,.1876600E-03,.2209500E-03,.2198700E-03,.2104500E-03,&
     & .1931200E-03,.1693400E-03,.1381200E-03,.9384000E-04,.3048900E-04,&
     & .1540900E-03,.1739900E-03,.1701000E-03,.1600200E-03,.1431100E-03,&
     & .1237600E-03,.1012100E-03,.6642800E-04,.1781300E-04,.1559900E-03,&
     & .1783100E-03,.1752100E-03,.1655400E-03,.1489000E-03,.1288100E-03,&
     & .1052700E-03,.6996400E-04,.1991800E-04,.1566100E-03,.1811900E-03,&
     & .1789200E-03,.1693400E-03,.1536500E-03,.1333800E-03,.1086400E-03,&
     & .7314200E-04,.2212100E-04,.1560800E-03,.1829300E-03,.1814500E-03,&
     & .1725400E-03,.1574300E-03,.1374700E-03,.1120900E-03,.7597100E-04,&
     & .2442500E-04,.1545300E-03,.1838300E-03,.1830900E-03,.1751000E-03,&
     & .1606700E-03,.1410200E-03,.1152900E-03,.7860700E-04,.2679100E-04,&
     & .1270500E-03,.1448500E-03,.1415500E-03,.1331200E-03,.1191400E-03,&
     & .1029900E-03,.8441600E-04,.5559800E-04,.1442900E-04,.1284300E-03,&
     & .1482200E-03,.1455500E-03,.1374800E-03,.1237700E-03,.1070200E-03,&
     & .8752000E-04,.5840500E-04,.1614700E-04,.1287500E-03,.1504300E-03,&
     & .1485000E-03,.1405300E-03,.1275200E-03,.1108100E-03,.9034800E-04,&
     & .6093400E-04,.1799200E-04,.1282200E-03,.1517500E-03,.1504900E-03,&
     & .1430100E-03,.1305900E-03,.1141300E-03,.9317100E-04,.6329600E-04,&
     & .1988800E-04,.1268400E-03,.1523900E-03,.1517900E-03,.1451100E-03,&
     & .1332000E-03,.1170200E-03,.9578600E-04,.6546800E-04,.2170600E-04/

      data   absa(  1:180, 5)   /                                       &
     & .2275875E-02,.2332064E-02,.2216796E-02,.2037339E-02,.1810754E-02,&
     & .1542017E-02,.1221835E-02,.8225168E-03,.2378785E-03,.2274525E-02,&
     & .2355192E-02,.2253198E-02,.2079119E-02,.1858235E-02,.1590203E-02,&
     & .1266759E-02,.8576253E-03,.2590612E-03,.2266023E-02,.2375024E-02,&
     & .2283817E-02,.2117878E-02,.1902253E-02,.1637433E-02,.1310064E-02,&
     & .8925666E-03,.2839408E-03,.2251924E-02,.2389473E-02,.2311430E-02,&
     & .2154739E-02,.1945283E-02,.1682775E-02,.1352510E-02,.9268463E-03,&
     & .3108163E-03,.2232455E-02,.2399259E-02,.2336395E-02,.2189553E-02,&
     & .1986811E-02,.1727976E-02,.1393776E-02,.9592790E-03,.3382077E-03,&
     & .2106993E-02,.2154787E-02,.2043104E-02,.1868719E-02,.1654669E-02,&
     & .1404007E-02,.1111065E-02,.7507219E-03,.1949377E-03,.2109908E-02,&
     & .2178927E-02,.2078863E-02,.1908781E-02,.1697578E-02,.1448872E-02,&
     & .1153419E-02,.7832801E-03,.2139563E-03,.2105465E-02,.2199441E-02,&
     & .2109463E-02,.1944753E-02,.1739524E-02,.1492018E-02,.1195120E-02,&
     & .8155632E-03,.2349725E-03,.2093796E-02,.2214377E-02,.2136026E-02,&
     & .1977275E-02,.1778542E-02,.1533694E-02,.1236220E-02,.8467658E-03,&
     & .2569866E-03,.2076724E-02,.2225712E-02,.2159173E-02,.2008267E-02,&
     & .1816462E-02,.1574183E-02,.1275209E-02,.8771670E-03,.2791186E-03,&
     & .1940537E-02,.1975605E-02,.1864311E-02,.1695911E-02,.1496137E-02,&
     & .1263628E-02,.9971610E-03,.6738674E-03,.1565544E-03,.1948245E-02,&
     & .2001986E-02,.1900553E-02,.1736295E-02,.1537745E-02,.1305077E-02,&
     & .1036131E-02,.7035141E-03,.1728753E-03,.1946880E-02,.2024379E-02,&
     & .1930764E-02,.1771154E-02,.1575770E-02,.1344673E-02,.1074374E-02,&
     & .7332422E-03,.1899479E-03,.1938868E-02,.2041140E-02,.1955874E-02,&
     & .1802287E-02,.1611440E-02,.1382812E-02,.1112406E-02,.7633322E-03,&
     & .2072614E-03,.1926362E-02,.2054363E-02,.1977970E-02,.1829838E-02,&
     & .1645759E-02,.1419736E-02,.1149822E-02,.7922080E-03,.2267149E-03,&
     & .1777321E-02,.1802735E-02,.1693881E-02,.1532687E-02,.1345353E-02,&
     & .1130261E-02,.8895753E-03,.5968647E-03,.1247261E-03,.1788584E-02,&
     & .1831387E-02,.1729859E-02,.1572174E-02,.1384900E-02,.1169670E-02,&
     & .9252263E-03,.6260595E-03,.1385262E-03,.1791618E-02,.1854218E-02,&
     & .1759929E-02,.1606995E-02,.1420262E-02,.1206226E-02,.9598075E-03,&
     & .6546109E-03,.1526122E-03,.1788339E-02,.1872013E-02,.1786084E-02,&
     & .1636442E-02,.1452167E-02,.1240664E-02,.9939639E-03,.6826539E-03,&
     & .1672628E-03,.1778773E-02,.1885271E-02,.1807813E-02,.1662029E-02,&
     & .1483280E-02,.1274547E-02,.1027899E-02,.7097707E-03,.1836638E-03/

      data   absa(181:315, 5)   /                                       &
     & .1611039E-02,.1628826E-02,.1528886E-02,.1381805E-02,.1208375E-02,&
     & .1011477E-02,.7896015E-03,.5279159E-03,.9908775E-04,.1625251E-02,&
     & .1659059E-02,.1564750E-02,.1419820E-02,.1247291E-02,.1047725E-02,&
     & .8222728E-03,.5541873E-03,.1104539E-03,.1631934E-02,.1683911E-02,&
     & .1595157E-02,.1453710E-02,.1281311E-02,.1080902E-02,.8540548E-03,&
     & .5806197E-03,.1219897E-03,.1631403E-02,.1703448E-02,.1621164E-02,&
     & .1482847E-02,.1310904E-02,.1111385E-02,.8847073E-03,.6065309E-03,&
     & .1346152E-03,.1624079E-02,.1717008E-02,.1643088E-02,.1507761E-02,&
     & .1337964E-02,.1141306E-02,.9151343E-03,.6325088E-03,.1481383E-03,&
     & .1446415E-02,.1455765E-02,.1361590E-02,.1229351E-02,.1074912E-02,&
     & .9006186E-03,.7006813E-03,.4647234E-03,.8189242E-04,.1463779E-02,&
     & .1487367E-02,.1398211E-02,.1267666E-02,.1112416E-02,.9358187E-03,&
     & .7313361E-03,.4886301E-03,.9135341E-04,.1472351E-02,.1513138E-02,&
     & .1428682E-02,.1300953E-02,.1145840E-02,.9675370E-03,.7596020E-03,&
     & .5122126E-03,.1005443E-03,.1474221E-02,.1532600E-02,.1454339E-02,&
     & .1329459E-02,.1174906E-02,.9956373E-03,.7867852E-03,.5359551E-03,&
     & .1109812E-03,.1468517E-02,.1546507E-02,.1474790E-02,.1352688E-02,&
     & .1200227E-02,.1022681E-02,.8131587E-03,.5591171E-03,.1219361E-03,&
     & .1282142E-02,.1287927E-02,.1202141E-02,.1083733E-02,.9460514E-03,&
     & .7915341E-03,.6168183E-03,.4084185E-03,.6913010E-04,.1301052E-02,&
     & .1319819E-02,.1238509E-02,.1121340E-02,.9823064E-03,.8254094E-03,&
     & .6455635E-03,.4300976E-03,.7651588E-04,.1311903E-02,.1345238E-02,&
     & .1268573E-02,.1153718E-02,.1014080E-02,.8553954E-03,.6719282E-03,&
     & .4514177E-03,.8484352E-04,.1314609E-02,.1365010E-02,.1292722E-02,&
     & .1180653E-02,.1041774E-02,.8822963E-03,.6970864E-03,.4722480E-03,&
     & .9377363E-04,.1310264E-02,.1377648E-02,.1311780E-02,.1202781E-02,&
     & .1065878E-02,.9070933E-03,.7215140E-03,.4928747E-03,.1032168E-03/

      data   absa(316:450, 5)   /                                       &
     & .1123722E-02,.1125509E-02,.1049077E-02,.9463051E-03,.8264962E-03,&
     & .6910472E-03,.5381327E-03,.3566196E-03,.6074714E-04,.1144528E-02,&
     & .1157108E-02,.1084145E-02,.9824251E-03,.8612451E-03,.7227531E-03,&
     & .5650534E-03,.3769954E-03,.6752101E-04,.1156115E-02,.1182481E-02,&
     & .1113388E-02,.1013135E-02,.8909498E-03,.7510056E-03,.5899576E-03,&
     & .3965626E-03,.7509877E-04,.1159636E-02,.1200610E-02,.1136628E-02,&
     & .1038419E-02,.9163190E-03,.7757266E-03,.6129711E-03,.4156689E-03,&
     & .8319483E-04,.1156675E-02,.1212668E-02,.1153943E-02,.1059223E-02,&
     & .9387272E-03,.7983980E-03,.6348116E-03,.4345877E-03,.9149048E-04,&
     & .9788217E-03,.9762902E-03,.9076191E-03,.8180805E-03,.7140719E-03,&
     & .5979311E-03,.4657057E-03,.3080365E-03,.6376560E-04,.9995801E-03,&
     & .1006767E-02,.9409303E-03,.8518710E-03,.7465719E-03,.6272074E-03,&
     & .4915396E-03,.3271476E-03,.7043059E-04,.1011283E-02,.1030289E-02,&
     & .9679162E-03,.8806454E-03,.7743638E-03,.6533475E-03,.5145414E-03,&
     & .3451157E-03,.7748348E-04,.1015523E-02,.1047024E-02,.9891231E-03,&
     & .9038676E-03,.7980900E-03,.6765444E-03,.5353367E-03,.3623073E-03,&
     & .8485843E-04,.1014685E-02,.1058164E-02,.1005055E-02,.9228604E-03,&
     & .8183767E-03,.6971991E-03,.5549716E-03,.3793243E-03,.9265390E-04,&
     & .8467257E-03,.8439047E-03,.7839737E-03,.7069600E-03,.6168442E-03,&
     & .5161363E-03,.4002993E-03,.2648075E-03,.8538662E-04,.8663801E-03,&
     & .8726556E-03,.8143622E-03,.7377253E-03,.6460394E-03,.5429897E-03,&
     & .4242444E-03,.2822342E-03,.9232499E-04,.8777160E-03,.8939799E-03,&
     & .8388502E-03,.7633777E-03,.6710202E-03,.5661949E-03,.4453570E-03,&
     & .2983921E-03,.9968327E-04,.8822976E-03,.9090305E-03,.8576617E-03,&
     & .7836215E-03,.6918973E-03,.5865524E-03,.4643056E-03,.3139878E-03,&
     & .1075001E-03,.8816821E-03,.9194329E-03,.8721358E-03,.8002626E-03,&
     & .7096839E-03,.6047828E-03,.4818629E-03,.3291128E-03,.1153966E-03/

      data   absa(451:585, 5)   /                                       &
     & .7345259E-03,.7353571E-03,.6844124E-03,.6191407E-03,.5419865E-03,&
     & .4543535E-03,.3524790E-03,.2326534E-03,.8831490E-04,.7490617E-03,&
     & .7579739E-03,.7093615E-03,.6447658E-03,.5665622E-03,.4774427E-03,&
     & .3729073E-03,.2477509E-03,.9439904E-04,.7564292E-03,.7744541E-03,&
     & .7287630E-03,.6652977E-03,.5871248E-03,.4970065E-03,.3911667E-03,&
     & .2619945E-03,.9990772E-04,.7586881E-03,.7858289E-03,.7437324E-03,&
     & .6818515E-03,.6043581E-03,.5141276E-03,.4072910E-03,.2755984E-03,&
     & .1066252E-03,.7563639E-03,.7931595E-03,.7548375E-03,.6950726E-03,&
     & .6192222E-03,.5294685E-03,.4222590E-03,.2889243E-03,.1133924E-03,&
     & .6314226E-03,.6349627E-03,.5926562E-03,.5376114E-03,.4719251E-03,&
     & .3962308E-03,.3074879E-03,.2032569E-03,.8003266E-04,.6418725E-03,&
     & .6525898E-03,.6125061E-03,.5580757E-03,.4921735E-03,.4155327E-03,&
     & .3248794E-03,.2162533E-03,.8576530E-04,.6471266E-03,.6653221E-03,&
     & .6276452E-03,.5745768E-03,.5090609E-03,.4319430E-03,.3403545E-03,&
     & .2285539E-03,.9041149E-04,.6478227E-03,.6738338E-03,.6394020E-03,&
     & .5876220E-03,.5231812E-03,.4462724E-03,.3541410E-03,.2403090E-03,&
     & .9584348E-04,.6442291E-03,.6788200E-03,.6475947E-03,.5980228E-03,&
     & .5350971E-03,.4588656E-03,.3667841E-03,.2515799E-03,.1017699E-03,&
     & .5366413E-03,.5433012E-03,.5091779E-03,.4634801E-03,.4076173E-03,&
     & .3421937E-03,.2655508E-03,.1759732E-03,.6609799E-04,.5446297E-03,&
     & .5573704E-03,.5249762E-03,.4800512E-03,.4241853E-03,.3581272E-03,&
     & .2800981E-03,.1871731E-03,.7047861E-04,.5481742E-03,.5672989E-03,&
     & .5373039E-03,.4932643E-03,.4379422E-03,.3716071E-03,.2930461E-03,&
     & .1977626E-03,.7433299E-04,.5476171E-03,.5738401E-03,.5461812E-03,&
     & .5035685E-03,.4492907E-03,.3832995E-03,.3046021E-03,.2077508E-03,&
     & .7880238E-04,.5438946E-03,.5774436E-03,.5525251E-03,.5119901E-03,&
     & .4590263E-03,.3938170E-03,.3152705E-03,.2173888E-03,.8388184E-04/

      data   absa(  1:180, 6)   /                                       &
     & .7453129E-02,.7117405E-02,.6661341E-02,.6078589E-02,.5376308E-02,&
     & .4581666E-02,.3692876E-02,.2668674E-02,.1857553E-02,.7414650E-02,&
     & .7132841E-02,.6723850E-02,.6175876E-02,.5501579E-02,.4730655E-02,&
     & .3847363E-02,.2834084E-02,.2101417E-02,.7367697E-02,.7142450E-02,&
     & .6787572E-02,.6271236E-02,.5628812E-02,.4876081E-02,.4005906E-02,&
     & .3014246E-02,.2361444E-02,.7309381E-02,.7154158E-02,.6847198E-02,&
     & .6364336E-02,.5757194E-02,.5023915E-02,.4174265E-02,.3211257E-02,&
     & .2644919E-02,.7240917E-02,.7165375E-02,.6901545E-02,.6460272E-02,&
     & .5886974E-02,.5176789E-02,.4355003E-02,.3427381E-02,.2951988E-02,&
     & .7302417E-02,.6986029E-02,.6506078E-02,.5915241E-02,.5213714E-02,&
     & .4409454E-02,.3509969E-02,.2477433E-02,.1531293E-02,.7272446E-02,&
     & .7005207E-02,.6569159E-02,.6009028E-02,.5333024E-02,.4547186E-02,&
     & .3651478E-02,.2623452E-02,.1730635E-02,.7229818E-02,.7019119E-02,&
     & .6631639E-02,.6101575E-02,.5452906E-02,.4682444E-02,.3795174E-02,&
     & .2779605E-02,.1947925E-02,.7174295E-02,.7033174E-02,.6689925E-02,&
     & .6193261E-02,.5571003E-02,.4818428E-02,.3945240E-02,.2947728E-02,&
     & .2183570E-02,.7110805E-02,.7045732E-02,.6744602E-02,.6284811E-02,&
     & .5687916E-02,.4958393E-02,.4105561E-02,.3129372E-02,.2439844E-02,&
     & .7107732E-02,.6797516E-02,.6299565E-02,.5698070E-02,.4995391E-02,&
     & .4198481E-02,.3302300E-02,.2275195E-02,.1230993E-02,.7086152E-02,&
     & .6823070E-02,.6363343E-02,.5788270E-02,.5108424E-02,.4328271E-02,&
     & .3431427E-02,.2406277E-02,.1392637E-02,.7052294E-02,.6841431E-02,&
     & .6426576E-02,.5878051E-02,.5223227E-02,.4454560E-02,.3563303E-02,&
     & .2543558E-02,.1569089E-02,.7006893E-02,.6857640E-02,.6486440E-02,&
     & .5967434E-02,.5334883E-02,.4578334E-02,.3699649E-02,.2688020E-02,&
     & .1760440E-02,.6949333E-02,.6870773E-02,.6541587E-02,.6057335E-02,&
     & .5442169E-02,.4706411E-02,.3839426E-02,.2841732E-02,.1964963E-02,&
     & .6855130E-02,.6551944E-02,.6044314E-02,.5438518E-02,.4743060E-02,&
     & .3966419E-02,.3087108E-02,.2087789E-02,.9879948E-03,.6846892E-02,&
     & .6585398E-02,.6112347E-02,.5528377E-02,.4852610E-02,.4086708E-02,&
     & .3208051E-02,.2203856E-02,.1119742E-02,.6824941E-02,.6610372E-02,&
     & .6177682E-02,.5615844E-02,.4962743E-02,.4204184E-02,.3329715E-02,&
     & .2324960E-02,.1263158E-02,.6787743E-02,.6632257E-02,.6236180E-02,&
     & .5702032E-02,.5069502E-02,.4319861E-02,.3453836E-02,.2450583E-02,&
     & .1417217E-02,.6740085E-02,.6651560E-02,.6290312E-02,.5788524E-02,&
     & .5171266E-02,.4436812E-02,.3580405E-02,.2581552E-02,.1582148E-02/

      data   absa(181:315, 6)   /                                       &
     & .6558083E-02,.6261779E-02,.5751378E-02,.5145860E-02,.4467228E-02,&
     & .3714081E-02,.2872846E-02,.1912747E-02,.7974292E-03,.6563849E-02,&
     & .6305787E-02,.5824503E-02,.5238051E-02,.4571578E-02,.3828273E-02,&
     & .2985436E-02,.2017845E-02,.9044707E-03,.6552571E-02,.6339245E-02,&
     & .5892376E-02,.5324090E-02,.4676050E-02,.3939449E-02,.3097458E-02,&
     & .2125514E-02,.1021181E-02,.6528895E-02,.6367176E-02,.5952402E-02,&
     & .5408542E-02,.4777701E-02,.4049784E-02,.3211983E-02,.2236558E-02,&
     & .1145140E-02,.6494313E-02,.6390805E-02,.6006376E-02,.5490888E-02,&
     & .4876710E-02,.4160137E-02,.3328414E-02,.2349602E-02,.1279011E-02,&
     & .6217297E-02,.5928758E-02,.5426959E-02,.4833520E-02,.4174155E-02,&
     & .3449793E-02,.2650701E-02,.1741615E-02,.6370094E-03,.6238859E-02,&
     & .5985757E-02,.5505992E-02,.4926333E-02,.4277869E-02,.3557040E-02,&
     & .2755926E-02,.1837721E-02,.7241979E-03,.6246176E-02,.6029777E-02,&
     & .5577993E-02,.5013764E-02,.4377842E-02,.3662657E-02,.2861509E-02,&
     & .1935493E-02,.8193352E-03,.6236692E-02,.6066266E-02,.5641135E-02,&
     & .5097431E-02,.4474642E-02,.3767073E-02,.2967217E-02,.2035588E-02,&
     & .9198253E-03,.6214342E-02,.6096770E-02,.5698970E-02,.5179034E-02,&
     & .4569107E-02,.3869801E-02,.3075666E-02,.2136680E-02,.1028239E-02,&
     & .5843663E-02,.5561987E-02,.5073558E-02,.4502171E-02,.3871558E-02,&
     & .3184826E-02,.2432337E-02,.1579438E-02,.5066815E-03,.5883549E-02,&
     & .5632733E-02,.5159626E-02,.4598553E-02,.3975656E-02,.3289560E-02,&
     & .2531458E-02,.1667525E-02,.5778308E-03,.5906315E-02,.5689718E-02,&
     & .5238108E-02,.4688943E-02,.4074005E-02,.3390408E-02,.2630796E-02,&
     & .1755973E-02,.6537602E-03,.5911760E-02,.5735215E-02,.5308819E-02,&
     & .4775175E-02,.4168666E-02,.3489857E-02,.2728910E-02,.1847257E-02,&
     & .7347999E-03,.5905012E-02,.5775036E-02,.5371993E-02,.4856819E-02,&
     & .4260538E-02,.3587351E-02,.2827507E-02,.1938584E-02,.8225668E-03/

      data   absa(316:450, 6)   /                                       &
     & .5448799E-02,.5174237E-02,.4702978E-02,.4159096E-02,.3562636E-02,&
     & .2919042E-02,.2216707E-02,.1428474E-02,.4372080E-03,.5503786E-02,&
     & .5258826E-02,.4798801E-02,.4260222E-02,.3666742E-02,.3022053E-02,&
     & .2311790E-02,.1509380E-02,.4923566E-03,.5542089E-02,.5327495E-02,&
     & .4884619E-02,.4354537E-02,.3765908E-02,.3119877E-02,.2406348E-02,&
     & .1590203E-02,.5508055E-03,.5564630E-02,.5383808E-02,.4961019E-02,&
     & .4443926E-02,.3860688E-02,.3216549E-02,.2498792E-02,.1671379E-02,&
     & .6126355E-03,.5570361E-02,.5432506E-02,.5031105E-02,.4526985E-02,&
     & .3951398E-02,.3309750E-02,.2590269E-02,.1752856E-02,.6794564E-03,&
     & .5025183E-02,.4765836E-02,.4321116E-02,.3810537E-02,.3254592E-02,&
     & .2655950E-02,.2006115E-02,.1286407E-02,.4311014E-03,.5098265E-02,&
     & .4863995E-02,.4425250E-02,.3917786E-02,.3360874E-02,.2757245E-02,&
     & .2097126E-02,.1360829E-02,.4799419E-03,.5154179E-02,.4945399E-02,&
     & .4519161E-02,.4017193E-02,.3461670E-02,.2853712E-02,.2187196E-02,&
     & .1435268E-02,.5332265E-03,.5190969E-02,.5013790E-02,.4604355E-02,&
     & .4110643E-02,.3555937E-02,.2947656E-02,.2275382E-02,.1509403E-02,&
     & .5888012E-03,.5208138E-02,.5072165E-02,.4680561E-02,.4195990E-02,&
     & .3646519E-02,.3037397E-02,.2361829E-02,.1583627E-02,.6478322E-03,&
     & .4623391E-02,.4366790E-02,.3943485E-02,.3465227E-02,.2955259E-02,&
     & .2405643E-02,.1812616E-02,.1157161E-02,.4945874E-03,.4710125E-02,&
     & .4474615E-02,.4054953E-02,.3577026E-02,.3062985E-02,.2506113E-02,&
     & .1900613E-02,.1226503E-02,.5465245E-03,.4776716E-02,.4564405E-02,&
     & .4153594E-02,.3680995E-02,.3166029E-02,.2601953E-02,.1986110E-02,&
     & .1295516E-02,.6032750E-03,.4823514E-02,.4640328E-02,.4243823E-02,&
     & .3778649E-02,.3262225E-02,.2694009E-02,.2069881E-02,.1363711E-02,&
     & .6621313E-03,.4852467E-02,.4703395E-02,.4324910E-02,.3866669E-02,&
     & .3352109E-02,.2781681E-02,.2151024E-02,.1432036E-02,.7228720E-03/

      data   absa(451:585, 6)   /                                       &
     & .4268691E-02,.4040183E-02,.3646880E-02,.3203178E-02,.2726389E-02,&
     & .2213905E-02,.1666317E-02,.1066325E-02,.4983873E-03,.4358586E-02,&
     & .4148745E-02,.3757876E-02,.3313137E-02,.2831232E-02,.2309276E-02,&
     & .1749842E-02,.1130782E-02,.5479266E-03,.4428469E-02,.4241143E-02,&
     & .3857210E-02,.3414820E-02,.2929730E-02,.2400315E-02,.1831540E-02,&
     & .1194311E-02,.6008082E-03,.4479685E-02,.4318791E-02,.3947347E-02,&
     & .3507844E-02,.3019422E-02,.2487256E-02,.1910638E-02,.1257207E-02,&
     & .6538267E-03,.4513413E-02,.4384415E-02,.4026009E-02,.3591207E-02,&
     & .3103605E-02,.2569184E-02,.1986548E-02,.1320050E-02,.7081602E-03,&
     & .3915100E-02,.3711716E-02,.3349816E-02,.2944438E-02,.2506601E-02,&
     & .2037038E-02,.1533857E-02,.9781961E-03,.4622673E-03,.4005434E-02,&
     & .3819687E-02,.3461208E-02,.3053733E-02,.2609936E-02,.2129368E-02,&
     & .1612065E-02,.1038339E-02,.5051377E-03,.4076956E-02,.3913286E-02,&
     & .3561928E-02,.3154739E-02,.2705010E-02,.2215707E-02,.1687464E-02,&
     & .1097449E-02,.5512414E-03,.4130247E-02,.3993297E-02,.3650863E-02,&
     & .3245814E-02,.2792524E-02,.2297376E-02,.1759369E-02,.1155953E-02,&
     & .5980184E-03,.4167218E-02,.4060665E-02,.3729251E-02,.3327359E-02,&
     & .2873454E-02,.2374433E-02,.1829761E-02,.1214279E-02,.6444273E-03,&
     & .3565599E-02,.3385516E-02,.3057279E-02,.2688351E-02,.2290666E-02,&
     & .1864426E-02,.1405266E-02,.8966862E-03,.3986965E-03,.3656897E-02,&
     & .3493575E-02,.3167640E-02,.2795196E-02,.2390148E-02,.1952514E-02,&
     & .1480086E-02,.9522316E-03,.4352189E-03,.3729366E-02,.3587046E-02,&
     & .3266101E-02,.2893554E-02,.2481255E-02,.2035902E-02,.1551310E-02,&
     & .1006380E-02,.4737097E-03,.3784570E-02,.3667287E-02,.3354159E-02,&
     & .2981192E-02,.2566707E-02,.2114949E-02,.1619376E-02,.1060223E-02,&
     & .5128669E-03,.3823908E-02,.3735653E-02,.3431813E-02,.3061934E-02,&
     & .2645587E-02,.2189124E-02,.1685116E-02,.1114442E-02,.5524785E-03/

      data   absa(  1:180, 7)   /                                       &
     & .2337626E-01,.2175176E-01,.2063054E-01,.1953289E-01,.1869334E-01,&
     & .1836575E-01,.1866188E-01,.1986732E-01,.2252135E-01,.2315624E-01,&
     & .2164265E-01,.2067161E-01,.1978356E-01,.1921682E-01,.1922562E-01,&
     & .1992930E-01,.2155800E-01,.2450575E-01,.2293107E-01,.2154457E-01,&
     & .2074715E-01,.2011748E-01,.1985588E-01,.2024390E-01,.2137333E-01,&
     & .2342559E-01,.2668727E-01,.2270210E-01,.2145089E-01,.2088033E-01,&
     & .2054255E-01,.2062482E-01,.2143098E-01,.2299821E-01,.2549251E-01,&
     & .2907234E-01,.2247095E-01,.2137582E-01,.2108331E-01,.2107230E-01,&
     & .2155716E-01,.2281018E-01,.2484353E-01,.2780611E-01,.3172795E-01,&
     & .2458189E-01,.2285200E-01,.2156813E-01,.2024632E-01,.1900708E-01,&
     & .1819156E-01,.1796768E-01,.1857224E-01,.2076937E-01,.2435250E-01,&
     & .2275178E-01,.2160965E-01,.2048103E-01,.1947652E-01,.1898219E-01,&
     & .1913925E-01,.2015560E-01,.2268231E-01,.2412387E-01,.2265897E-01,&
     & .2168600E-01,.2079019E-01,.2005803E-01,.1992705E-01,.2048310E-01,&
     & .2192140E-01,.2477362E-01,.2389129E-01,.2256825E-01,.2181441E-01,&
     & .2118241E-01,.2077464E-01,.2103762E-01,.2201020E-01,.2389372E-01,&
     & .2708335E-01,.2365223E-01,.2248782E-01,.2200444E-01,.2167167E-01,&
     & .2165150E-01,.2233835E-01,.2375741E-01,.2611574E-01,.2966311E-01,&
     & .2579688E-01,.2397618E-01,.2249064E-01,.2092442E-01,.1929406E-01,&
     & .1790138E-01,.1706811E-01,.1698922E-01,.1852242E-01,.2557176E-01,&
     & .2388374E-01,.2253992E-01,.2114151E-01,.1969695E-01,.1860179E-01,&
     & .1812445E-01,.1843215E-01,.2030655E-01,.2534781E-01,.2379879E-01,&
     & .2261460E-01,.2141516E-01,.2020271E-01,.1944733E-01,.1934135E-01,&
     & .2004948E-01,.2227174E-01,.2511276E-01,.2371341E-01,.2273562E-01,&
     & .2176102E-01,.2083763E-01,.2045509E-01,.2073270E-01,.2187030E-01,&
     & .2445363E-01,.2487422E-01,.2363274E-01,.2290622E-01,.2219407E-01,&
     & .2162169E-01,.2162715E-01,.2232993E-01,.2392248E-01,.2688023E-01,&
     & .2698586E-01,.2506791E-01,.2338609E-01,.2157417E-01,.1958193E-01,&
     & .1763778E-01,.1616964E-01,.1538518E-01,.1621144E-01,.2676194E-01,&
     & .2498910E-01,.2344897E-01,.2177479E-01,.1993962E-01,.1825433E-01,&
     & .1710937E-01,.1668719E-01,.1786207E-01,.2654257E-01,.2491521E-01,&
     & .2352676E-01,.2201454E-01,.2038036E-01,.1899998E-01,.1819673E-01,&
     & .1814869E-01,.1968096E-01,.2631697E-01,.2483856E-01,.2364114E-01,&
     & .2232785E-01,.2093620E-01,.1989270E-01,.1944697E-01,.1980212E-01,&
     & .2170612E-01,.2608471E-01,.2475687E-01,.2379589E-01,.2271583E-01,&
     & .2161901E-01,.2093691E-01,.2088135E-01,.2166610E-01,.2395288E-01/

      data   absa(181:315, 7)   /                                       &
     & .2810778E-01,.2610373E-01,.2423479E-01,.2218757E-01,.1986177E-01,&
     & .1746306E-01,.1537001E-01,.1389630E-01,.1400744E-01,.2789745E-01,&
     & .2604100E-01,.2431619E-01,.2238153E-01,.2019516E-01,.1800500E-01,&
     & .1621009E-01,.1507015E-01,.1553290E-01,.2769341E-01,.2597967E-01,&
     & .2440064E-01,.2260000E-01,.2059359E-01,.1865773E-01,.1717719E-01,&
     & .1638502E-01,.1720845E-01,.2747581E-01,.2591687E-01,.2451639E-01,&
     & .2287475E-01,.2108787E-01,.1943977E-01,.1828677E-01,.1786928E-01,&
     & .1906661E-01,.2724525E-01,.2584524E-01,.2465979E-01,.2322623E-01,&
     & .2169174E-01,.2036043E-01,.1956477E-01,.1955161E-01,.2112946E-01,&
     & .2916121E-01,.2706131E-01,.2501249E-01,.2273351E-01,.2011979E-01,&
     & .1734949E-01,.1468202E-01,.1254397E-01,.1192697E-01,.2896641E-01,&
     & .2702014E-01,.2511575E-01,.2293374E-01,.2042776E-01,.1783904E-01,&
     & .1542709E-01,.1359860E-01,.1333064E-01,.2876765E-01,.2697723E-01,&
     & .2521010E-01,.2314222E-01,.2079486E-01,.1840512E-01,.1626966E-01,&
     & .1476028E-01,.1484637E-01,.2856294E-01,.2693274E-01,.2532642E-01,&
     & .2339329E-01,.2123982E-01,.1907661E-01,.1724154E-01,.1606921E-01,&
     & .1652405E-01,.2834370E-01,.2687419E-01,.2545751E-01,.2371200E-01,&
     & .2178226E-01,.1987947E-01,.1836893E-01,.1756873E-01,.1840173E-01,&
     & .3011114E-01,.2792230E-01,.2568609E-01,.2319257E-01,.2034293E-01,&
     & .1726900E-01,.1415286E-01,.1138485E-01,.1005692E-01,.2992968E-01,&
     & .2790487E-01,.2581306E-01,.2340902E-01,.2064353E-01,.1772380E-01,&
     & .1480077E-01,.1231510E-01,.1132697E-01,.2974561E-01,.2787980E-01,&
     & .2592483E-01,.2361776E-01,.2099074E-01,.1823014E-01,.1552253E-01,&
     & .1333086E-01,.1268422E-01,.2955745E-01,.2785343E-01,.2604522E-01,&
     & .2385432E-01,.2139366E-01,.1881776E-01,.1636294E-01,.1447960E-01,&
     & .1418800E-01,.2935104E-01,.2781242E-01,.2617843E-01,.2414962E-01,&
     & .2187815E-01,.1952002E-01,.1734737E-01,.1580301E-01,.1587593E-01/

      data   absa(316:450, 7)   /                                       &
     & .3090210E-01,.2863464E-01,.2623745E-01,.2353758E-01,.2049216E-01,&
     & .1719200E-01,.1374636E-01,.1041842E-01,.8306976E-02,.3074160E-01,&
     & .2864184E-01,.2639435E-01,.2377357E-01,.2080408E-01,.1762587E-01,&
     & .1431865E-01,.1122298E-01,.9441247E-02,.3058186E-01,.2864960E-01,&
     & .2652689E-01,.2399238E-01,.2114556E-01,.1809030E-01,.1494024E-01,&
     & .1210474E-01,.1065458E-01,.3041236E-01,.2865545E-01,.2666041E-01,&
     & .2422874E-01,.2152521E-01,.1861362E-01,.1566308E-01,.1310960E-01,&
     & .1200349E-01,.3023431E-01,.2863951E-01,.2679661E-01,.2451409E-01,&
     & .2196729E-01,.1923952E-01,.1651206E-01,.1426729E-01,.1351982E-01,&
     & .3150955E-01,.2917466E-01,.2663128E-01,.2375509E-01,.2054034E-01,&
     & .1706533E-01,.1339031E-01,.9619698E-02,.6589618E-02,.3137548E-01,&
     & .2921391E-01,.2682829E-01,.2401415E-01,.2086776E-01,.1749458E-01,&
     & .1391017E-01,.1031015E-01,.7573865E-02,.3124577E-01,.2926104E-01,&
     & .2699356E-01,.2424726E-01,.2121023E-01,.1794016E-01,.1446463E-01,&
     & .1107269E-01,.8630532E-02,.3111437E-01,.2930266E-01,.2714520E-01,&
     & .2449152E-01,.2158301E-01,.1842118E-01,.1509518E-01,.1193837E-01,&
     & .9801078E-02,.3097127E-01,.2932558E-01,.2729444E-01,.2477809E-01,&
     & .2199750E-01,.1898438E-01,.1582945E-01,.1293776E-01,.1111997E-01,&
     & .3182644E-01,.2948280E-01,.2685783E-01,.2385751E-01,.2050472E-01,&
     & .1690419E-01,.1307973E-01,.9016818E-02,.5889262E-02,.3174219E-01,&
     & .2957180E-01,.2709658E-01,.2415027E-01,.2085202E-01,.1733699E-01,&
     & .1357172E-01,.9617188E-02,.6575397E-02,.3166606E-01,.2967426E-01,&
     & .2730658E-01,.2441152E-01,.2120507E-01,.1777789E-01,.1408374E-01,&
     & .1027324E-01,.7358277E-02,.3158325E-01,.2976719E-01,.2749330E-01,&
     & .2467389E-01,.2157919E-01,.1824104E-01,.1464950E-01,.1102019E-01,&
     & .8229230E-02,.3148314E-01,.2984161E-01,.2766625E-01,.2497260E-01,&
     & .2198691E-01,.1876667E-01,.1530472E-01,.1188221E-01,.9182963E-02/

      data   absa(451:585, 7)   /                                       &
     & .3191734E-01,.2961741E-01,.2697787E-01,.2391284E-01,.2049970E-01,&
     & .1685955E-01,.1296507E-01,.8750098E-02,.6274631E-02,.3190439E-01,&
     & .2977990E-01,.2725815E-01,.2422984E-01,.2087607E-01,.1730510E-01,&
     & .1344617E-01,.9295005E-02,.7007386E-02,.3188423E-01,.2993640E-01,&
     & .2750670E-01,.2452542E-01,.2126330E-01,.1774973E-01,.1394711E-01,&
     & .9900539E-02,.7782927E-02,.3184397E-01,.3007747E-01,.2772343E-01,&
     & .2483444E-01,.2166507E-01,.1822702E-01,.1449986E-01,.1058694E-01,&
     & .8671897E-02,.3178483E-01,.3018389E-01,.2793133E-01,.2517191E-01,&
     & .2210215E-01,.1876481E-01,.1512017E-01,.1136237E-01,.9662908E-02,&
     & .3187853E-01,.2963766E-01,.2698548E-01,.2385689E-01,.2040357E-01,&
     & .1672870E-01,.1278988E-01,.8497981E-02,.6174837E-02,.3194590E-01,&
     & .2988270E-01,.2731430E-01,.2420369E-01,.2080895E-01,.1718277E-01,&
     & .1326475E-01,.9006730E-02,.6900214E-02,.3198870E-01,.3010121E-01,&
     & .2760049E-01,.2453900E-01,.2122452E-01,.1764036E-01,.1376646E-01,&
     & .9568560E-02,.7665374E-02,.3200932E-01,.3028722E-01,.2785313E-01,&
     & .2489082E-01,.2164826E-01,.1813119E-01,.1432073E-01,.1019444E-01,&
     & .8520390E-02,.3200022E-01,.3043673E-01,.2810016E-01,.2525761E-01,&
     & .2211496E-01,.1867947E-01,.1492276E-01,.1090133E-01,.9520500E-02,&
     & .3173815E-01,.2955655E-01,.2688573E-01,.2371393E-01,.2023390E-01,&
     & .1654567E-01,.1257985E-01,.8246315E-02,.5647340E-02,.3187913E-01,&
     & .2987574E-01,.2727046E-01,.2410527E-01,.2067778E-01,.1701522E-01,&
     & .1304871E-01,.8729674E-02,.6295753E-02,.3199367E-01,.3016037E-01,&
     & .2760193E-01,.2448927E-01,.2112455E-01,.1748698E-01,.1354982E-01,&
     & .9254652E-02,.7027536E-02,.3208158E-01,.3040579E-01,.2790109E-01,&
     & .2488587E-01,.2157632E-01,.1798983E-01,.1409732E-01,.9833255E-02,&
     & .7857223E-02,.3212706E-01,.3060928E-01,.2819618E-01,.2528322E-01,&
     & .2206831E-01,.1854255E-01,.1468812E-01,.1048268E-01,.8776360E-02/

      data   absa(  1:180, 8)   /                                       &
     & .6013431E-01,.5707436E-01,.7233980E-01,.9815382E-01,.1294324E+00,&
     & .1617466E+00,.1940373E+00,.2262581E+00,.2513644E+00,.5875631E-01,&
     & .5691751E-01,.7437061E-01,.1026345E+00,.1360233E+00,.1699724E+00,&
     & .2039042E+00,.2377536E+00,.2641912E+00,.5766566E-01,.5691471E-01,&
     & .7651533E-01,.1069690E+00,.1422529E+00,.1777591E+00,.2132449E+00,&
     & .2486476E+00,.2761623E+00,.5690082E-01,.5712785E-01,.7881629E-01,&
     & .1114369E+00,.1484198E+00,.1854598E+00,.2224755E+00,.2594138E+00,&
     & .2880248E+00,.5637766E-01,.5755994E-01,.8139096E-01,.1160220E+00,&
     & .1545869E+00,.1931597E+00,.2317210E+00,.2701851E+00,.2998888E+00,&
     & .6618075E-01,.6337862E-01,.7888769E-01,.1046010E+00,.1369770E+00,&
     & .1711778E+00,.2053496E+00,.2394477E+00,.2688156E+00,.6469675E-01,&
     & .6319075E-01,.8109464E-01,.1094837E+00,.1444540E+00,.1805151E+00,&
     & .2165594E+00,.2525366E+00,.2834257E+00,.6352136E-01,.6312417E-01,&
     & .8332160E-01,.1142143E+00,.1514669E+00,.1892772E+00,.2270740E+00,&
     & .2647762E+00,.2971560E+00,.6267746E-01,.6327579E-01,.8565769E-01,&
     & .1189657E+00,.1582452E+00,.1977516E+00,.2372313E+00,.2766038E+00,&
     & .3104162E+00,.6209896E-01,.6369481E-01,.8824714E-01,.1238764E+00,&
     & .1649199E+00,.2060757E+00,.2472064E+00,.2882573E+00,.3234086E+00,&
     & .7395912E-01,.7063428E-01,.8565208E-01,.1104255E+00,.1427485E+00,&
     & .1783683E+00,.2139880E+00,.2495178E+00,.2828639E+00,.7228712E-01,&
     & .7036097E-01,.8793923E-01,.1157395E+00,.1512413E+00,.1890002E+00,&
     & .2267276E+00,.2644122E+00,.2997266E+00,.7086841E-01,.7018107E-01,&
     & .9023900E-01,.1209004E+00,.1591022E+00,.1988304E+00,.2385408E+00,&
     & .2781769E+00,.3153056E+00,.6982927E-01,.7021452E-01,.9257013E-01,&
     & .1259826E+00,.1665897E+00,.2081673E+00,.2497368E+00,.2911947E+00,&
     & .3300931E+00,.6907982E-01,.7054544E-01,.9512295E-01,.1311315E+00,&
     & .1739073E+00,.2173068E+00,.2606881E+00,.3039811E+00,.3445288E+00,&
     & .8336981E-01,.7909588E-01,.9293217E-01,.1163289E+00,.1478222E+00,&
     & .1841644E+00,.2209570E+00,.2576676E+00,.2934903E+00,.8148502E-01,&
     & .7868615E-01,.9524447E-01,.1221028E+00,.1571946E+00,.1961923E+00,&
     & .2353851E+00,.2744893E+00,.3126755E+00,.7980558E-01,.7839855E-01,&
     & .9757525E-01,.1277440E+00,.1659210E+00,.2072782E+00,.2486777E+00,&
     & .2899782E+00,.3303130E+00,.7850723E-01,.7829446E-01,.9994420E-01,&
     & .1331287E+00,.1741803E+00,.2176536E+00,.2611345E+00,.3044935E+00,&
     & .3468364E+00,.7749949E-01,.7851124E-01,.1025029E+00,.1385272E+00,&
     & .1822356E+00,.2277238E+00,.2731983E+00,.3185592E+00,.3628488E+00/

      data   absa(181:315, 8)   /                                       &
     & .9446343E-01,.8887098E-01,.1010405E+00,.1226819E+00,.1528690E+00,&
     & .1887604E+00,.2264637E+00,.2640916E+00,.3014227E+00,.9223331E-01,&
     & .8827980E-01,.1033453E+00,.1288736E+00,.1629549E+00,.2021939E+00,&
     & .2425871E+00,.2829079E+00,.3229151E+00,.9024155E-01,.8783436E-01,&
     & .1057387E+00,.1349856E+00,.1724292E+00,.2146241E+00,.2574756E+00,&
     & .3002489E+00,.3427212E+00,.8862647E-01,.8754527E-01,.1081502E+00,&
     & .1408020E+00,.1814043E+00,.2262377E+00,.2714152E+00,.3165045E+00,&
     & .3612330E+00,.8735280E-01,.8760876E-01,.1107602E+00,.1464752E+00,&
     & .1900954E+00,.2373504E+00,.2847413E+00,.3320395E+00,.3789626E+00,&
     & .1071919E+00,.1001003E+00,.1101040E+00,.1294657E+00,.1574729E+00,&
     & .1915463E+00,.2292568E+00,.2673709E+00,.3054402E+00,.1045794E+00,&
     & .9926080E-01,.1123922E+00,.1359833E+00,.1683351E+00,.2062283E+00,&
     & .2471625E+00,.2882394E+00,.3292971E+00,.1022687E+00,.9862247E-01,&
     & .1148716E+00,.1424605E+00,.1785172E+00,.2200001E+00,.2638803E+00,&
     & .3077436E+00,.3515504E+00,.1003101E+00,.9811233E-01,.1173419E+00,&
     & .1486525E+00,.1881373E+00,.2329575E+00,.2794837E+00,.3259493E+00,&
     & .3723585E+00,.9873698E-01,.9795941E-01,.1200135E+00,.1545733E+00,&
     & .1973376E+00,.2452232E+00,.2942202E+00,.3431039E+00,.3919463E+00,&
     & .1214673E+00,.1126359E+00,.1204125E+00,.1370664E+00,.1621239E+00,&
     & .1935142E+00,.2296249E+00,.2677842E+00,.3060416E+00,.1185186E+00,&
     & .1115740E+00,.1227071E+00,.1438053E+00,.1736162E+00,.2092232E+00,&
     & .2494502E+00,.2909400E+00,.3324970E+00,.1158775E+00,.1107513E+00,&
     & .1251835E+00,.1505785E+00,.1844456E+00,.2241943E+00,.2681063E+00,&
     & .3126858E+00,.3573659E+00,.1135645E+00,.1100550E+00,.1276724E+00,&
     & .1570942E+00,.1947551E+00,.2383233E+00,.2854861E+00,.3329504E+00,&
     & .3805180E+00,.1116111E+00,.1096678E+00,.1303082E+00,.1632798E+00,&
     & .2045649E+00,.2516490E+00,.3017320E+00,.3519240E+00,.4021839E+00/

      data   absa(316:450, 8)   /                                       &
     & .1374520E+00,.1266748E+00,.1318516E+00,.1456457E+00,.1672523E+00,&
     & .1950666E+00,.2283281E+00,.2656918E+00,.3037056E+00,.1341802E+00,&
     & .1254071E+00,.1340698E+00,.1526186E+00,.1792338E+00,.2117540E+00,&
     & .2498928E+00,.2912047E+00,.3328745E+00,.1311544E+00,.1242569E+00,&
     & .1365313E+00,.1596294E+00,.1905813E+00,.2277714E+00,.2703124E+00,&
     & .3152132E+00,.3603201E+00,.1284296E+00,.1232723E+00,.1390451E+00,&
     & .1663760E+00,.2014538E+00,.2429694E+00,.2894262E+00,.3375633E+00,&
     & .3858525E+00,.1260194E+00,.1225869E+00,.1416527E+00,.1727500E+00,&
     & .2118169E+00,.2572538E+00,.3073512E+00,.3584426E+00,.4097428E+00,&
     & .1553761E+00,.1422412E+00,.1444711E+00,.1550385E+00,.1728286E+00,&
     & .1965109E+00,.2258295E+00,.2610663E+00,.2984504E+00,.1516669E+00,&
     & .1407304E+00,.1465588E+00,.1622080E+00,.1852388E+00,.2140019E+00,&
     & .2488334E+00,.2889075E+00,.3302659E+00,.1481522E+00,.1392528E+00,&
     & .1489452E+00,.1694320E+00,.1970843E+00,.2308530E+00,.2707638E+00,&
     & .3150768E+00,.3601714E+00,.1448954E+00,.1379226E+00,.1514075E+00,&
     & .1763709E+00,.2084024E+00,.2469639E+00,.2914326E+00,.3395756E+00,&
     & .3881786E+00,.1419554E+00,.1368141E+00,.1539816E+00,.1829006E+00,&
     & .2192687E+00,.2621639E+00,.3109195E+00,.3624990E+00,.4144086E+00,&
     & .1748198E+00,.1592727E+00,.1584450E+00,.1657904E+00,.1799080E+00,&
     & .1994836E+00,.2245418E+00,.2566689E+00,.2810339E+00,.1705294E+00,&
     & .1573886E+00,.1604084E+00,.1730449E+00,.1927005E+00,.2176246E+00,&
     & .2486329E+00,.2864375E+00,.3169857E+00,.1664134E+00,.1555110E+00,&
     & .1626512E+00,.1803894E+00,.2049406E+00,.2351258E+00,.2717671E+00,&
     & .3146346E+00,.3505565E+00,.1625805E+00,.1537499E+00,.1650044E+00,&
     & .1874575E+00,.2166562E+00,.2519400E+00,.2937443E+00,.3411186E+00,&
     & .3823263E+00,.1590426E+00,.1521966E+00,.1674556E+00,.1940812E+00,&
     & .2278853E+00,.2678905E+00,.3145120E+00,.3659844E+00,.4129266E+00/

      data   absa(451:585, 8)   /                                       &
     & .1935762E+00,.1766297E+00,.1744420E+00,.1806968E+00,.1935473E+00,&
     & .2114104E+00,.2347348E+00,.2655809E+00,.2757601E+00,.1885866E+00,&
     & .1741931E+00,.1762668E+00,.1880588E+00,.2064070E+00,.2298344E+00,&
     & .2593418E+00,.2963843E+00,.3112482E+00,.1838929E+00,.1718429E+00,&
     & .1783334E+00,.1953918E+00,.2186897E+00,.2477198E+00,.2829811E+00,&
     & .3255600E+00,.3451455E+00,.1794971E+00,.1696478E+00,.1805638E+00,&
     & .2022902E+00,.2305237E+00,.2647488E+00,.3054649E+00,.3531405E+00,&
     & .3769471E+00,.1753728E+00,.1677953E+00,.1828746E+00,.2087886E+00,&
     & .2417893E+00,.2809426E+00,.3270047E+00,.3794515E+00,.4073030E+00,&
     & .2131992E+00,.1947813E+00,.1912304E+00,.1963678E+00,.2076432E+00,&
     & .2236360E+00,.2450570E+00,.2741551E+00,.2789175E+00,.2074072E+00,&
     & .1916172E+00,.1927506E+00,.2036992E+00,.2204764E+00,.2423112E+00,&
     & .2701300E+00,.3057154E+00,.3147569E+00,.2019948E+00,.1886976E+00,&
     & .1945776E+00,.2108178E+00,.2327675E+00,.2604245E+00,.2941651E+00,&
     & .3357507E+00,.3491138E+00,.1968339E+00,.1860228E+00,.1965911E+00,&
     & .2174692E+00,.2445952E+00,.2776478E+00,.3170335E+00,.3643697E+00,&
     & .3816200E+00,.1920176E+00,.1837096E+00,.1986031E+00,.2238583E+00,&
     & .2557920E+00,.2940556E+00,.3391509E+00,.3918311E+00,.4121885E+00,&
     & .2332827E+00,.2134102E+00,.2085161E+00,.2124079E+00,.2219989E+00,&
     & .2359574E+00,.2553647E+00,.2822947E+00,.2887838E+00,.2266775E+00,&
     & .2095424E+00,.2096502E+00,.2195248E+00,.2347351E+00,.2548294E+00,&
     & .2808212E+00,.3144477E+00,.3259151E+00,.2203906E+00,.2059404E+00,&
     & .2111339E+00,.2263053E+00,.2469443E+00,.2730233E+00,.3051566E+00,&
     & .3452396E+00,.3610131E+00,.2143742E+00,.2026221E+00,.2127678E+00,&
     & .2326886E+00,.2586790E+00,.2903679E+00,.3283839E+00,.3747332E+00,&
     & .3941719E+00,.2087774E+00,.1996808E+00,.2143237E+00,.2388436E+00,&
     & .2697785E+00,.3069660E+00,.3509603E+00,.4031790E+00,.4260779E+00/

      data   absb(  1:120, 1)  /                                        &
     & .1281000E-03,.1289100E-03,.1289000E-03,.1271300E-03,.1247700E-03,&
     & .1065300E-03,.1066700E-03,.1064800E-03,.1047300E-03,.1029300E-03,&
     & .9017600E-04,.9007700E-04,.8928600E-04,.8787900E-04,.8598700E-04,&
     & .7770400E-04,.7730400E-04,.7587900E-04,.7424600E-04,.7207000E-04,&
     & .6718300E-04,.6590600E-04,.6461900E-04,.6300000E-04,.6105600E-04,&
     & .5874100E-04,.5764900E-04,.5602800E-04,.5429300E-04,.5250600E-04,&
     & .5183900E-04,.5071500E-04,.4905300E-04,.4719500E-04,.4536800E-04,&
     & .4432100E-04,.4329700E-04,.4192300E-04,.4029800E-04,.3874400E-04,&
     & .3752700E-04,.3651600E-04,.3538000E-04,.3403700E-04,.3295100E-04,&
     & .3175200E-04,.3070300E-04,.2958600E-04,.2849900E-04,.2751900E-04,&
     & .2663800E-04,.2564300E-04,.2469300E-04,.2383200E-04,.2291300E-04,&
     & .2232500E-04,.2148000E-04,.2062100E-04,.1986700E-04,.1906600E-04,&
     & .1876400E-04,.1799800E-04,.1726900E-04,.1652700E-04,.1582100E-04,&
     & .1565000E-04,.1499200E-04,.1435600E-04,.1371000E-04,.1311900E-04,&
     & .1305400E-04,.1247700E-04,.1192900E-04,.1139300E-04,.1094800E-04,&
     & .1085200E-04,.1039000E-04,.9919500E-05,.9502000E-05,.9199300E-05,&
     & .9099000E-05,.8720900E-05,.8338400E-05,.8002900E-05,.7724500E-05,&
     & .7716900E-05,.7366200E-05,.7055200E-05,.6802900E-05,.6574900E-05,&
     & .6525000E-05,.6236400E-05,.6017200E-05,.5780400E-05,.5550400E-05,&
     & .5516700E-05,.5302100E-05,.5108800E-05,.4911300E-05,.4715000E-05,&
     & .4726100E-05,.4553000E-05,.4364100E-05,.4172500E-05,.4007700E-05,&
     & .4064600E-05,.3906200E-05,.3747200E-05,.3600900E-05,.3454900E-05,&
     & .3547900E-05,.3405700E-05,.3270400E-05,.3142000E-05,.3021900E-05,&
     & .3135300E-05,.3027000E-05,.2902400E-05,.2777600E-05,.2660000E-05/

      data   absb(121:235, 1)   /                                       &
     & .2659500E-05,.2562100E-05,.2464600E-05,.2358200E-05,.2256300E-05,&
     & .2246800E-05,.2165100E-05,.2081200E-05,.2004300E-05,.1922100E-05,&
     & .1900900E-05,.1833500E-05,.1768500E-05,.1703900E-05,.1640600E-05,&
     & .1574600E-05,.1518800E-05,.1466900E-05,.1414800E-05,.1366800E-05,&
     & .1300100E-05,.1256100E-05,.1212200E-05,.1171800E-05,.1130600E-05,&
     & .1071600E-05,.1038000E-05,.1001800E-05,.9674500E-06,.9347500E-06,&
     & .8765100E-06,.8517200E-06,.8245300E-06,.7956000E-06,.7701100E-06,&
     & .7156500E-06,.6969900E-06,.6769900E-06,.6540300E-06,.6320200E-06,&
     & .5839100E-06,.5694600E-06,.5531800E-06,.5357500E-06,.5177400E-06,&
     & .4749000E-06,.4653100E-06,.4517500E-06,.4368600E-06,.4226000E-06,&
     & .3871800E-06,.3761000E-06,.3665000E-06,.3558000E-06,.3441700E-06,&
     & .3146400E-06,.3049400E-06,.2987200E-06,.2909600E-06,.2823000E-06,&
     & .2554200E-06,.2488500E-06,.2431400E-06,.2374500E-06,.2309200E-06,&
     & .2074100E-06,.2017500E-06,.1969900E-06,.1930500E-06,.1879100E-06,&
     & .1676000E-06,.1640600E-06,.1599300E-06,.1563100E-06,.1532700E-06,&
     & .1357900E-06,.1328700E-06,.1298500E-06,.1272500E-06,.1247000E-06,&
     & .1100600E-06,.1075700E-06,.1056800E-06,.1031400E-06,.1010100E-06,&
     & .8871100E-07,.8713100E-07,.8516200E-07,.8340500E-07,.8158500E-07,&
     & .7088900E-07,.6991700E-07,.6869200E-07,.6746300E-07,.6604700E-07,&
     & .5665900E-07,.5607700E-07,.5551800E-07,.5446200E-07,.5356300E-07,&
     & .4538000E-07,.4515900E-07,.4471800E-07,.4406300E-07,.4311600E-07,&
     & .3596200E-07,.3617700E-07,.3599700E-07,.3560000E-07,.3434000E-07,&
     & .2857800E-07,.2892400E-07,.2890100E-07,.2817600E-07,.2724600E-07/

      data   absb(  1:120, 2)   /                                       &
     & .6321500E-03,.6475600E-03,.6573400E-03,.6651500E-03,.6707700E-03,&
     & .5237800E-03,.5363100E-03,.5448500E-03,.5507100E-03,.5560500E-03,&
     & .4377700E-03,.4481400E-03,.4552100E-03,.4612400E-03,.4670000E-03,&
     & .3702300E-03,.3786900E-03,.3856800E-03,.3907000E-03,.3976000E-03,&
     & .3175300E-03,.3263200E-03,.3325400E-03,.3381700E-03,.3431200E-03,&
     & .2758000E-03,.2831000E-03,.2891500E-03,.2948800E-03,.2985500E-03,&
     & .2412900E-03,.2469700E-03,.2530200E-03,.2578500E-03,.2613700E-03,&
     & .2047000E-03,.2104000E-03,.2155200E-03,.2196300E-03,.2226000E-03,&
     & .1729000E-03,.1778200E-03,.1818800E-03,.1851300E-03,.1879400E-03,&
     & .1459900E-03,.1506300E-03,.1538900E-03,.1568200E-03,.1596500E-03,&
     & .1239000E-03,.1279700E-03,.1307100E-03,.1334700E-03,.1357100E-03,&
     & .1055300E-03,.1087500E-03,.1113400E-03,.1135100E-03,.1151300E-03,&
     & .9026300E-04,.9256000E-04,.9466700E-04,.9622500E-04,.9704100E-04,&
     & .7654200E-04,.7822700E-04,.7975500E-04,.8076800E-04,.8141200E-04,&
     & .6456200E-04,.6602400E-04,.6712100E-04,.6787800E-04,.6799200E-04,&
     & .5458700E-04,.5566800E-04,.5643900E-04,.5677800E-04,.5664900E-04,&
     & .4624800E-04,.4688500E-04,.4746900E-04,.4747600E-04,.4746400E-04,&
     & .3908400E-04,.3970100E-04,.3983600E-04,.3991900E-04,.3976200E-04,&
     & .3329700E-04,.3367000E-04,.3378100E-04,.3357100E-04,.3341800E-04,&
     & .2847500E-04,.2847900E-04,.2835200E-04,.2820200E-04,.2810700E-04,&
     & .2402300E-04,.2397700E-04,.2387400E-04,.2385800E-04,.2372900E-04,&
     & .2027600E-04,.2031400E-04,.2026200E-04,.2020500E-04,.2001300E-04,&
     & .1717700E-04,.1717900E-04,.1718100E-04,.1704800E-04,.1691500E-04,&
     & .1458600E-04,.1463200E-04,.1456900E-04,.1446500E-04,.1440400E-04/

      data   absb(121:235, 2)   /                                       &
     & .1216500E-04,.1222000E-04,.1216400E-04,.1209900E-04,.1203600E-04,&
     & .1017800E-04,.1022400E-04,.1020100E-04,.1013400E-04,.1006000E-04,&
     & .8562200E-05,.8582600E-05,.8552000E-05,.8510900E-05,.8437800E-05,&
     & .7026300E-05,.7054900E-05,.7033400E-05,.7000000E-05,.6950000E-05,&
     & .5751000E-05,.5774900E-05,.5769200E-05,.5738400E-05,.5712100E-05,&
     & .4703800E-05,.4726100E-05,.4727800E-05,.4702300E-05,.4686700E-05,&
     & .3824900E-05,.3833900E-05,.3849800E-05,.3836400E-05,.3812500E-05,&
     & .3097300E-05,.3112000E-05,.3116600E-05,.3120300E-05,.3102200E-05,&
     & .2508500E-05,.2526300E-05,.2529300E-05,.2532800E-05,.2524900E-05,&
     & .2026100E-05,.2047900E-05,.2056500E-05,.2057500E-05,.2058800E-05,&
     & .1631900E-05,.1663900E-05,.1674500E-05,.1674000E-05,.1675900E-05,&
     & .1309700E-05,.1344900E-05,.1361100E-05,.1362400E-05,.1358900E-05,&
     & .1052100E-05,.1084700E-05,.1105200E-05,.1110200E-05,.1108500E-05,&
     & .8466800E-06,.8723500E-06,.8954400E-06,.9060700E-06,.9061600E-06,&
     & .6806000E-06,.7018000E-06,.7218700E-06,.7356200E-06,.7380000E-06,&
     & .5463400E-06,.5648100E-06,.5825600E-06,.5954300E-06,.6012200E-06,&
     & .4386600E-06,.4557700E-06,.4695700E-06,.4817600E-06,.4894000E-06,&
     & .3506200E-06,.3664700E-06,.3793500E-06,.3897800E-06,.3969700E-06,&
     & .2788300E-06,.2951400E-06,.3039700E-06,.3125100E-06,.3200400E-06,&
     & .2209800E-06,.2355000E-06,.2428900E-06,.2501800E-06,.2570800E-06,&
     & .1729500E-06,.1862400E-06,.1933200E-06,.1993400E-06,.2058700E-06,&
     & .1362300E-06,.1468000E-06,.1540300E-06,.1591200E-06,.1649900E-06,&
     & .1086400E-06,.1168000E-06,.1235300E-06,.1284000E-06,.1335200E-06/

      data   absb(  1:120, 3)   /                                       &
     & .2106900E-02,.2157100E-02,.2194300E-02,.2206100E-02,.2212400E-02,&
     & .1741100E-02,.1783300E-02,.1809700E-02,.1822500E-02,.1825100E-02,&
     & .1444000E-02,.1479000E-02,.1496300E-02,.1506600E-02,.1506600E-02,&
     & .1199800E-02,.1226100E-02,.1239800E-02,.1250000E-02,.1248700E-02,&
     & .9998000E-03,.1020200E-02,.1033200E-02,.1039900E-02,.1039200E-02,&
     & .8380500E-03,.8562900E-03,.8682700E-03,.8730300E-03,.8754000E-03,&
     & .7074300E-03,.7238100E-03,.7325100E-03,.7355000E-03,.7393100E-03,&
     & .5934800E-03,.6057200E-03,.6113700E-03,.6149600E-03,.6176100E-03,&
     & .4955600E-03,.5048800E-03,.5098700E-03,.5142100E-03,.5158100E-03,&
     & .4146100E-03,.4214500E-03,.4258400E-03,.4296900E-03,.4305500E-03,&
     & .3468300E-03,.3518100E-03,.3566000E-03,.3591100E-03,.3595800E-03,&
     & .2900400E-03,.2938300E-03,.2978700E-03,.2996400E-03,.3004200E-03,&
     & .2414600E-03,.2456100E-03,.2487900E-03,.2505300E-03,.2513400E-03,&
     & .2016000E-03,.2053700E-03,.2077400E-03,.2092800E-03,.2102300E-03,&
     & .1691600E-03,.1722600E-03,.1738600E-03,.1752900E-03,.1763400E-03,&
     & .1418300E-03,.1441400E-03,.1460400E-03,.1474100E-03,.1484400E-03,&
     & .1195400E-03,.1216100E-03,.1228600E-03,.1238600E-03,.1244800E-03,&
     & .1006700E-03,.1020500E-03,.1033100E-03,.1040900E-03,.1047600E-03,&
     & .8501400E-04,.8624800E-04,.8702400E-04,.8795800E-04,.8857900E-04,&
     & .7193000E-04,.7307400E-04,.7398500E-04,.7493400E-04,.7484800E-04,&
     & .6120500E-04,.6225900E-04,.6320900E-04,.6338300E-04,.6346600E-04,&
     & .5204600E-04,.5289600E-04,.5342000E-04,.5351600E-04,.5375500E-04,&
     & .4438000E-04,.4512100E-04,.4527700E-04,.4560400E-04,.4576100E-04,&
     & .3805800E-04,.3840900E-04,.3865700E-04,.3881300E-04,.3862800E-04/

      data   absb(121:235, 3)   /                                       &
     & .3173500E-04,.3199000E-04,.3219900E-04,.3228800E-04,.3217900E-04,&
     & .2637800E-04,.2666700E-04,.2677800E-04,.2684000E-04,.2676000E-04,&
     & .2194900E-04,.2219500E-04,.2234000E-04,.2233000E-04,.2228100E-04,&
     & .1800100E-04,.1820900E-04,.1833800E-04,.1837600E-04,.1830900E-04,&
     & .1471500E-04,.1490600E-04,.1502500E-04,.1509800E-04,.1502600E-04,&
     & .1201900E-04,.1219700E-04,.1231200E-04,.1238000E-04,.1234600E-04,&
     & .9756600E-05,.9931800E-05,.1003600E-04,.1009600E-04,.1011800E-04,&
     & .7881000E-05,.8071600E-05,.8163700E-05,.8220200E-05,.8260800E-05,&
     & .6354100E-05,.6517100E-05,.6633000E-05,.6697700E-05,.6723900E-05,&
     & .5107700E-05,.5242800E-05,.5363600E-05,.5436300E-05,.5469300E-05,&
     & .4088300E-05,.4219600E-05,.4319900E-05,.4407400E-05,.4446200E-05,&
     & .3290900E-05,.3394100E-05,.3474600E-05,.3554600E-05,.3608900E-05,&
     & .2640100E-05,.2726900E-05,.2800300E-05,.2861600E-05,.2918800E-05,&
     & .2122200E-05,.2203200E-05,.2263800E-05,.2312700E-05,.2360300E-05,&
     & .1705600E-05,.1783800E-05,.1838600E-05,.1880400E-05,.1913400E-05,&
     & .1366200E-05,.1433300E-05,.1485500E-05,.1523800E-05,.1552100E-05,&
     & .1088900E-05,.1144500E-05,.1195500E-05,.1230000E-05,.1256000E-05,&
     & .8625100E-06,.9155600E-06,.9599900E-06,.9945400E-06,.1017100E-05,&
     & .6864000E-06,.7286800E-06,.7738900E-06,.8062700E-06,.8269600E-06,&
     & .5433400E-06,.5833900E-06,.6195300E-06,.6519000E-06,.6728500E-06,&
     & .4333600E-06,.4651100E-06,.4956100E-06,.5233700E-06,.5449100E-06,&
     & .3442800E-06,.3723600E-06,.3968700E-06,.4201700E-06,.4379000E-06,&
     & .2788300E-06,.3023700E-06,.3209600E-06,.3397800E-06,.3548500E-06/

      data   absb(  1:120, 4)   /                                       &
     & .5280600E-02,.5355100E-02,.5383600E-02,.5381000E-02,.5341300E-02,&
     & .4359400E-02,.4411400E-02,.4426800E-02,.4418300E-02,.4387700E-02,&
     & .3608000E-02,.3642100E-02,.3660300E-02,.3654000E-02,.3630700E-02,&
     & .3000700E-02,.3032400E-02,.3047900E-02,.3037700E-02,.3017500E-02,&
     & .2511300E-02,.2541500E-02,.2548900E-02,.2542800E-02,.2528100E-02,&
     & .2111900E-02,.2134000E-02,.2139600E-02,.2138100E-02,.2123200E-02,&
     & .1770500E-02,.1788400E-02,.1794200E-02,.1793100E-02,.1781900E-02,&
     & .1471400E-02,.1485400E-02,.1492100E-02,.1492000E-02,.1487400E-02,&
     & .1222400E-02,.1234900E-02,.1243200E-02,.1241100E-02,.1239500E-02,&
     & .1017000E-02,.1028100E-02,.1031900E-02,.1031000E-02,.1028600E-02,&
     & .8468100E-03,.8548400E-03,.8551700E-03,.8554500E-03,.8557100E-03,&
     & .7033000E-03,.7087500E-03,.7107800E-03,.7126100E-03,.7111200E-03,&
     & .5848100E-03,.5887000E-03,.5915700E-03,.5930500E-03,.5923400E-03,&
     & .4864000E-03,.4904400E-03,.4939500E-03,.4946800E-03,.4946600E-03,&
     & .4047200E-03,.4091500E-03,.4123500E-03,.4127800E-03,.4134500E-03,&
     & .3382600E-03,.3420500E-03,.3436400E-03,.3448000E-03,.3452800E-03,&
     & .2830800E-03,.2862700E-03,.2879200E-03,.2891000E-03,.2895800E-03,&
     & .2375900E-03,.2399400E-03,.2419600E-03,.2429500E-03,.2427000E-03,&
     & .1999300E-03,.2020200E-03,.2039000E-03,.2045600E-03,.2041100E-03,&
     & .1685100E-03,.1706000E-03,.1715100E-03,.1716900E-03,.1713300E-03,&
     & .1427500E-03,.1439200E-03,.1444500E-03,.1445200E-03,.1439900E-03,&
     & .1206000E-03,.1214200E-03,.1218100E-03,.1216500E-03,.1210500E-03,&
     & .1016800E-03,.1024000E-03,.1027600E-03,.1026100E-03,.1020200E-03,&
     & .8578000E-04,.8663600E-04,.8681500E-04,.8678000E-04,.8672200E-04/

      data   absb(121:235, 4)   /                                       &
     & .7154000E-04,.7220100E-04,.7242200E-04,.7250100E-04,.7239800E-04,&
     & .5960900E-04,.6015200E-04,.6044300E-04,.6054000E-04,.6053400E-04,&
     & .4968100E-04,.5019400E-04,.5048100E-04,.5064900E-04,.5064300E-04,&
     & .4081500E-04,.4133500E-04,.4156500E-04,.4172900E-04,.4177600E-04,&
     & .3347600E-04,.3393200E-04,.3417100E-04,.3430100E-04,.3439700E-04,&
     & .2741900E-04,.2782400E-04,.2808300E-04,.2822100E-04,.2829600E-04,&
     & .2234900E-04,.2275300E-04,.2297500E-04,.2310000E-04,.2316400E-04,&
     & .1819700E-04,.1848800E-04,.1874700E-04,.1887800E-04,.1894300E-04,&
     & .1481500E-04,.1508100E-04,.1528400E-04,.1540800E-04,.1549300E-04,&
     & .1204600E-04,.1230400E-04,.1247600E-04,.1259000E-04,.1265400E-04,&
     & .9783400E-05,.1001800E-04,.1018700E-04,.1029500E-04,.1036500E-04,&
     & .7905800E-05,.8158300E-05,.8315800E-05,.8425900E-05,.8479100E-05,&
     & .6384700E-05,.6621200E-05,.6773500E-05,.6881800E-05,.6938200E-05,&
     & .5161900E-05,.5369800E-05,.5523900E-05,.5623000E-05,.5693100E-05,&
     & .4176400E-05,.4354600E-05,.4501800E-05,.4592700E-05,.4660900E-05,&
     & .3369600E-05,.3530200E-05,.3658500E-05,.3754800E-05,.3809100E-05,&
     & .2714200E-05,.2858500E-05,.2970000E-05,.3063200E-05,.3116300E-05,&
     & .2197400E-05,.2317600E-05,.2416900E-05,.2498000E-05,.2555500E-05,&
     & .1782000E-05,.1882700E-05,.1964700E-05,.2034700E-05,.2089500E-05,&
     & .1444500E-05,.1527400E-05,.1599900E-05,.1655800E-05,.1700400E-05,&
     & .1161200E-05,.1239000E-05,.1302000E-05,.1350800E-05,.1388100E-05,&
     & .9360600E-06,.1001700E-05,.1059800E-05,.1101600E-05,.1136700E-05,&
     & .7668200E-06,.8231800E-06,.8728100E-06,.9120800E-06,.9403700E-06/

      data   absb(  1:120, 5)   /                                       &
     & .2162906E-01,.2201877E-01,.2224013E-01,.2232097E-01,.2226088E-01,&
     & .1828684E-01,.1857354E-01,.1872091E-01,.1873615E-01,.1863965E-01,&
     & .1535086E-01,.1556848E-01,.1564923E-01,.1561925E-01,.1551236E-01,&
     & .1285052E-01,.1299486E-01,.1303075E-01,.1299836E-01,.1289615E-01,&
     & .1075316E-01,.1084397E-01,.1086973E-01,.1082689E-01,.1072798E-01,&
     & .9002505E-02,.9072913E-02,.9081241E-02,.9034900E-02,.8955849E-02,&
     & .7562305E-02,.7609626E-02,.7609405E-02,.7571232E-02,.7507851E-02,&
     & .6329221E-02,.6362683E-02,.6359991E-02,.6325694E-02,.6271378E-02,&
     & .5284028E-02,.5311691E-02,.5305973E-02,.5285178E-02,.5242950E-02,&
     & .4403121E-02,.4424122E-02,.4421960E-02,.4403212E-02,.4371719E-02,&
     & .3670406E-02,.3685830E-02,.3685814E-02,.3671437E-02,.3647935E-02,&
     & .3056358E-02,.3068927E-02,.3067363E-02,.3057520E-02,.3043098E-02,&
     & .2542912E-02,.2553558E-02,.2555470E-02,.2549363E-02,.2539078E-02,&
     & .2116477E-02,.2124848E-02,.2126732E-02,.2124859E-02,.2121688E-02,&
     & .1763370E-02,.1771671E-02,.1775722E-02,.1777888E-02,.1775372E-02,&
     & .1472312E-02,.1480631E-02,.1485906E-02,.1489295E-02,.1490270E-02,&
     & .1235246E-02,.1243142E-02,.1250718E-02,.1254010E-02,.1256275E-02,&
     & .1040850E-02,.1049997E-02,.1055542E-02,.1060339E-02,.1065075E-02,&
     & .8811911E-03,.8895566E-03,.8956959E-03,.9025351E-03,.9097381E-03,&
     & .7497834E-03,.7578993E-03,.7659613E-03,.7738154E-03,.7827692E-03,&
     & .6406593E-03,.6501363E-03,.6581906E-03,.6679566E-03,.6771392E-03,&
     & .5512107E-03,.5601135E-03,.5694184E-03,.5792748E-03,.5886259E-03,&
     & .4749345E-03,.4844659E-03,.4947239E-03,.5042924E-03,.5143445E-03,&
     & .4104977E-03,.4207339E-03,.4309147E-03,.4409904E-03,.4509584E-03/

      data   absb(121:235, 5)   /                                       &
     & .3490867E-03,.3588313E-03,.3687089E-03,.3789368E-03,.3886881E-03,&
     & .2966250E-03,.3063019E-03,.3155889E-03,.3255266E-03,.3349869E-03,&
     & .2526321E-03,.2616358E-03,.2706359E-03,.2802750E-03,.2890766E-03,&
     & .2118746E-03,.2200243E-03,.2284269E-03,.2373129E-03,.2453378E-03,&
     & .1772518E-03,.1845848E-03,.1923345E-03,.2004084E-03,.2080501E-03,&
     & .1480145E-03,.1547136E-03,.1617671E-03,.1689561E-03,.1761243E-03,&
     & .1228565E-03,.1288933E-03,.1351904E-03,.1417161E-03,.1481713E-03,&
     & .1015563E-03,.1069125E-03,.1124515E-03,.1181336E-03,.1240966E-03,&
     & .8391796E-04,.8846774E-04,.9331366E-04,.9841094E-04,.1037701E-03,&
     & .6907800E-04,.7301768E-04,.7725329E-04,.8170854E-04,.8659904E-04,&
     & .5663202E-04,.6003002E-04,.6370177E-04,.6762785E-04,.7189491E-04,&
     & .4624238E-04,.4927219E-04,.5239730E-04,.5580819E-04,.5956741E-04,&
     & .3765713E-04,.4030391E-04,.4301446E-04,.4597554E-04,.4926441E-04,&
     & .3073045E-04,.3306204E-04,.3540699E-04,.3792633E-04,.4077919E-04,&
     & .2508246E-04,.2707468E-04,.2909111E-04,.3125371E-04,.3371262E-04,&
     & .2044678E-04,.2211599E-04,.2384933E-04,.2568808E-04,.2780633E-04,&
     & .1657839E-04,.1800707E-04,.1948311E-04,.2105650E-04,.2284205E-04,&
     & .1347071E-04,.1469642E-04,.1596158E-04,.1729776E-04,.1878962E-04,&
     & .1095998E-04,.1197888E-04,.1307499E-04,.1421130E-04,.1546214E-04,&
     & .8897744E-05,.9746282E-05,.1066740E-04,.1163972E-04,.1268761E-04,&
     & .7209215E-05,.7906994E-05,.8676190E-05,.9504972E-05,.1037954E-04,&
     & .5835869E-05,.6420526E-05,.7058145E-05,.7757162E-05,.8507066E-05,&
     & .4829548E-05,.5320639E-05,.5859975E-05,.6447762E-05,.7091888E-05/

      data   absb(  1:120, 6)   /                                       &
     & .1413456E+00,.1451118E+00,.1482471E+00,.1506848E+00,.1524351E+00,&
     & .1284824E+00,.1323235E+00,.1353715E+00,.1377662E+00,.1396511E+00,&
     & .1165087E+00,.1201098E+00,.1230983E+00,.1255529E+00,.1275427E+00,&
     & .1056253E+00,.1090989E+00,.1120377E+00,.1145066E+00,.1165271E+00,&
     & .9552257E-01,.9900700E-01,.1020287E+00,.1046013E+00,.1068915E+00,&
     & .8651031E-01,.9000117E-01,.9310752E-01,.9595637E-01,.9839325E-01,&
     & .7836450E-01,.8201222E-01,.8531971E-01,.8822345E-01,.9085876E-01,&
     & .7092238E-01,.7459607E-01,.7796568E-01,.8111548E-01,.8403243E-01,&
     & .6424556E-01,.6792949E-01,.7138543E-01,.7475690E-01,.7782387E-01,&
     & .5852648E-01,.6226349E-01,.6593117E-01,.6939257E-01,.7271605E-01,&
     & .5365315E-01,.5746839E-01,.6121888E-01,.6482400E-01,.6833360E-01,&
     & .4952470E-01,.5342082E-01,.5723031E-01,.6098038E-01,.6470372E-01,&
     & .4609148E-01,.5006244E-01,.5398056E-01,.5794359E-01,.6187053E-01,&
     & .4334627E-01,.4739302E-01,.5147575E-01,.5560970E-01,.5968009E-01,&
     & .4118624E-01,.4532777E-01,.4955345E-01,.5391898E-01,.5813420E-01,&
     & .3952662E-01,.4385566E-01,.4826024E-01,.5268550E-01,.5705181E-01,&
     & .3841724E-01,.4291105E-01,.4755308E-01,.5207099E-01,.5646855E-01,&
     & .3779958E-01,.4252491E-01,.4729707E-01,.5191028E-01,.5641217E-01,&
     & .3769831E-01,.4260967E-01,.4736941E-01,.5212121E-01,.5677123E-01,&
     & .3802439E-01,.4300132E-01,.4793508E-01,.5274887E-01,.5740482E-01,&
     & .3864580E-01,.4371453E-01,.4869077E-01,.5366066E-01,.5849903E-01,&
     & .3933702E-01,.4447547E-01,.4962428E-01,.5473829E-01,.5982359E-01,&
     & .3971900E-01,.4497057E-01,.5034745E-01,.5572486E-01,.6081806E-01,&
     & .3964192E-01,.4514283E-01,.5071851E-01,.5624884E-01,.6141856E-01/

      data   absb(121:235, 6)   /                                       &
     & .3866583E-01,.4430320E-01,.4992214E-01,.5555938E-01,.6080523E-01,&
     & .3776931E-01,.4345093E-01,.4914913E-01,.5483878E-01,.6021309E-01,&
     & .3697431E-01,.4267835E-01,.4846432E-01,.5419260E-01,.5971603E-01,&
     & .3540090E-01,.4109817E-01,.4690264E-01,.5270132E-01,.5824775E-01,&
     & .3376861E-01,.3944516E-01,.4521743E-01,.5105174E-01,.5670807E-01,&
     & .3217884E-01,.3777623E-01,.4359346E-01,.4943539E-01,.5515047E-01,&
     & .3023451E-01,.3578942E-01,.4155720E-01,.4739287E-01,.5322236E-01,&
     & .2814056E-01,.3368562E-01,.3938021E-01,.4520123E-01,.5100155E-01,&
     & .2613012E-01,.3159507E-01,.3723484E-01,.4301429E-01,.4881413E-01,&
     & .2404068E-01,.2939351E-01,.3504512E-01,.4079501E-01,.4656480E-01,&
     & .2179194E-01,.2706163E-01,.3260199E-01,.3830559E-01,.4403615E-01,&
     & .1964345E-01,.2472402E-01,.3015571E-01,.3586422E-01,.4157514E-01,&
     & .1761158E-01,.2244787E-01,.2779047E-01,.3341890E-01,.3908741E-01,&
     & .1578946E-01,.2041434E-01,.2560918E-01,.3115439E-01,.3682998E-01,&
     & .1410355E-01,.1854242E-01,.2353769E-01,.2895272E-01,.3464369E-01,&
     & .1247313E-01,.1674870E-01,.2151305E-01,.2681285E-01,.3241759E-01,&
     & .1093422E-01,.1501782E-01,.1957117E-01,.2466990E-01,.3022010E-01,&
     & .9611967E-02,.1347400E-01,.1786246E-01,.2277546E-01,.2821315E-01,&
     & .8428789E-02,.1205812E-01,.1629395E-01,.2103849E-01,.2627794E-01,&
     & .7314617E-02,.1072816E-01,.1475277E-01,.1935069E-01,.2443117E-01,&
     & .6277076E-02,.9462864E-02,.1326850E-01,.1769409E-01,.2260642E-01,&
     & .5370648E-02,.8332516E-02,.1192724E-01,.1617813E-01,.2094606E-01,&
     & .5000457E-02,.7850552E-02,.1134835E-01,.1553696E-01,.2024820E-01/

      data   absb(  1:120, 7)   /                                       &
     & .1233797E+01,.1240309E+01,.1245906E+01,.1249038E+01,.1250667E+01,&
     & .1227509E+01,.1237527E+01,.1245130E+01,.1250890E+01,.1254582E+01,&
     & .1220370E+01,.1233242E+01,.1243916E+01,.1252002E+01,.1257208E+01,&
     & .1209755E+01,.1226066E+01,.1239648E+01,.1251101E+01,.1262465E+01,&
     & .1198271E+01,.1218221E+01,.1236549E+01,.1253105E+01,.1264757E+01,&
     & .1187954E+01,.1214287E+01,.1237014E+01,.1254616E+01,.1268426E+01,&
     & .1184097E+01,.1214093E+01,.1238496E+01,.1259993E+01,.1276453E+01,&
     & .1180879E+01,.1213827E+01,.1241956E+01,.1265044E+01,.1281987E+01,&
     & .1178298E+01,.1215263E+01,.1245209E+01,.1269051E+01,.1287376E+01,&
     & .1182695E+01,.1219813E+01,.1250266E+01,.1274554E+01,.1293058E+01,&
     & .1187264E+01,.1225491E+01,.1256328E+01,.1280895E+01,.1299927E+01,&
     & .1193354E+01,.1231549E+01,.1262983E+01,.1287976E+01,.1307592E+01,&
     & .1201358E+01,.1239017E+01,.1270766E+01,.1295922E+01,.1315914E+01,&
     & .1210113E+01,.1248264E+01,.1279581E+01,.1304944E+01,.1322887E+01,&
     & .1220261E+01,.1258227E+01,.1289322E+01,.1313044E+01,.1333034E+01,&
     & .1231439E+01,.1268725E+01,.1298471E+01,.1322363E+01,.1344202E+01,&
     & .1243861E+01,.1279684E+01,.1309193E+01,.1334435E+01,.1354444E+01,&
     & .1256537E+01,.1292561E+01,.1320904E+01,.1346216E+01,.1368544E+01,&
     & .1270814E+01,.1305822E+01,.1336261E+01,.1360944E+01,.1384844E+01,&
     & .1288177E+01,.1321480E+01,.1350393E+01,.1377826E+01,.1400827E+01,&
     & .1302859E+01,.1337128E+01,.1367775E+01,.1395784E+01,.1419468E+01,&
     & .1319163E+01,.1354279E+01,.1387311E+01,.1414162E+01,.1436650E+01,&
     & .1332975E+01,.1371023E+01,.1402616E+01,.1431177E+01,.1453004E+01,&
     & .1345134E+01,.1383758E+01,.1415866E+01,.1445650E+01,.1471428E+01/

      data   absb(121:235, 7)   /                                       &
     & .1345082E+01,.1383743E+01,.1417311E+01,.1448976E+01,.1475650E+01,&
     & .1344235E+01,.1383125E+01,.1418137E+01,.1452392E+01,.1479460E+01,&
     & .1345148E+01,.1385709E+01,.1420152E+01,.1456438E+01,.1484592E+01,&
     & .1336140E+01,.1378290E+01,.1414883E+01,.1450315E+01,.1479273E+01,&
     & .1326907E+01,.1369209E+01,.1406929E+01,.1443061E+01,.1473182E+01,&
     & .1317970E+01,.1361334E+01,.1398756E+01,.1434897E+01,.1468648E+01,&
     & .1307925E+01,.1351090E+01,.1391766E+01,.1425847E+01,.1460105E+01,&
     & .1292135E+01,.1338462E+01,.1379193E+01,.1415171E+01,.1450080E+01,&
     & .1275926E+01,.1325616E+01,.1367315E+01,.1406267E+01,.1439728E+01,&
     & .1262168E+01,.1313046E+01,.1356870E+01,.1395704E+01,.1429802E+01,&
     & .1243844E+01,.1297984E+01,.1345253E+01,.1384776E+01,.1423560E+01,&
     & .1221932E+01,.1280865E+01,.1330001E+01,.1373395E+01,.1410813E+01,&
     & .1200347E+01,.1260855E+01,.1314395E+01,.1359376E+01,.1397674E+01,&
     & .1181546E+01,.1245642E+01,.1304804E+01,.1351665E+01,.1391744E+01,&
     & .1161296E+01,.1229915E+01,.1291786E+01,.1343999E+01,.1385754E+01,&
     & .1137476E+01,.1211401E+01,.1274402E+01,.1331651E+01,.1376935E+01,&
     & .1111820E+01,.1188448E+01,.1255900E+01,.1313802E+01,.1364916E+01,&
     & .1087055E+01,.1170159E+01,.1237638E+01,.1301159E+01,.1353229E+01,&
     & .1062808E+01,.1149118E+01,.1221983E+01,.1285669E+01,.1342137E+01,&
     & .1033591E+01,.1125023E+01,.1202761E+01,.1269609E+01,.1328922E+01,&
     & .1001992E+01,.1099491E+01,.1180679E+01,.1251796E+01,.1314844E+01,&
     & .9714260E+00,.1073483E+01,.1159480E+01,.1233092E+01,.1298532E+01,&
     & .9605430E+00,.1062574E+01,.1153238E+01,.1229435E+01,.1295998E+01/

      data   absb(  1:120, 8)   /                                       &
     & .8974716E+01,.8691653E+01,.8415802E+01,.8168444E+01,.7938743E+01,&
     & .9715620E+01,.9387697E+01,.9091159E+01,.8812007E+01,.8550103E+01,&
     & .1040539E+02,.1005124E+02,.9716540E+01,.9405217E+01,.9117647E+01,&
     & .1107237E+02,.1067879E+02,.1030786E+02,.9949266E+01,.9577784E+01,&
     & .1169578E+02,.1125483E+02,.1081775E+02,.1038726E+02,.9999236E+01,&
     & .1224439E+02,.1171912E+02,.1121989E+02,.1076358E+02,.1033881E+02,&
     & .1266035E+02,.1208097E+02,.1155062E+02,.1103907E+02,.1056995E+02,&
     & .1303081E+02,.1240926E+02,.1182594E+02,.1128091E+02,.1079129E+02,&
     & .1335386E+02,.1267661E+02,.1206291E+02,.1149646E+02,.1098015E+02,&
     & .1353946E+02,.1285236E+02,.1221793E+02,.1163647E+02,.1110309E+02,&
     & .1368295E+02,.1297221E+02,.1232355E+02,.1172711E+02,.1117398E+02,&
     & .1377130E+02,.1305174E+02,.1238638E+02,.1177241E+02,.1119857E+02,&
     & .1380231E+02,.1307928E+02,.1239887E+02,.1176960E+02,.1117753E+02,&
     & .1379128E+02,.1305118E+02,.1236306E+02,.1171797E+02,.1114040E+02,&
     & .1373516E+02,.1298613E+02,.1228873E+02,.1164833E+02,.1103314E+02,&
     & .1364257E+02,.1288771E+02,.1219339E+02,.1154194E+02,.1089134E+02,&
     & .1350900E+02,.1275874E+02,.1205091E+02,.1137192E+02,.1073702E+02,&
     & .1335003E+02,.1258061E+02,.1187564E+02,.1118588E+02,.1051162E+02,&
     & .1314747E+02,.1237631E+02,.1163728E+02,.1094357E+02,.1023812E+02,&
     & .1288648E+02,.1212623E+02,.1139312E+02,.1065542E+02,.9954987E+01,&
     & .1264609E+02,.1186159E+02,.1109721E+02,.1034029E+02,.9619395E+01,&
     & .1238368E+02,.1157958E+02,.1077263E+02,.1001950E+02,.9300261E+01,&
     & .1217819E+02,.1132895E+02,.1052829E+02,.9741813E+01,.9027899E+01,&
     & .1202662E+02,.1116103E+02,.1034657E+02,.9538267E+01,.8771268E+01/

      data   absb(121:235, 8)   /                                       &
     & .1208641E+02,.1121978E+02,.1038937E+02,.9556683E+01,.8777908E+01,&
     & .1215415E+02,.1128733E+02,.1043945E+02,.9575262E+01,.8790855E+01,&
     & .1219484E+02,.1131019E+02,.1047069E+02,.9582740E+01,.8782235E+01,&
     & .1240319E+02,.1150569E+02,.1064514E+02,.9772131E+01,.8965666E+01,&
     & .1261524E+02,.1172536E+02,.1085921E+02,.9982767E+01,.9163514E+01,&
     & .1282029E+02,.1192789E+02,.1107155E+02,.1020133E+02,.9340664E+01,&
     & .1305607E+02,.1217814E+02,.1129130E+02,.1045671E+02,.9593408E+01,&
     & .1336869E+02,.1246260E+02,.1158864E+02,.1074016E+02,.9877648E+01,&
     & .1367966E+02,.1274532E+02,.1187204E+02,.1099717E+02,.1016259E+02,&
     & .1395988E+02,.1302676E+02,.1213900E+02,.1127751E+02,.1044562E+02,&
     & .1430124E+02,.1334454E+02,.1243000E+02,.1157468E+02,.1069646E+02,&
     & .1467926E+02,.1368264E+02,.1276261E+02,.1187105E+02,.1102298E+02,&
     & .1504278E+02,.1404941E+02,.1309066E+02,.1219578E+02,.1135077E+02,&
     & .1535675E+02,.1433897E+02,.1332735E+02,.1242447E+02,.1156923E+02,&
     & .1567971E+02,.1462376E+02,.1359863E+02,.1264514E+02,.1177949E+02,&
     & .1604096E+02,.1493692E+02,.1391760E+02,.1291724E+02,.1202255E+02,&
     & .1641711E+02,.1529760E+02,.1424404E+02,.1325595E+02,.1230184E+02,&
     & .1677154E+02,.1558947E+02,.1455293E+02,.1351142E+02,.1256144E+02,&
     & .1711050E+02,.1590689E+02,.1481896E+02,.1379270E+02,.1280515E+02,&
     & .1750874E+02,.1625687E+02,.1512545E+02,.1407639E+02,.1306970E+02,&
     & .1793109E+02,.1661941E+02,.1546240E+02,.1437682E+02,.1334037E+02,&
     & .1833366E+02,.1698072E+02,.1578011E+02,.1467905E+02,.1362875E+02,&
     & .1847759E+02,.1713162E+02,.1587877E+02,.1475220E+02,.1369335E+02/

!  ---
      data   forref(1:3,1: 8)    /                                      &
     & .8605600E-06,.1304390E-05,.3823780E-05,.8179260E-06,.1585990E-05,&
     & .6587710E-04,.1293690E-05,.8244060E-05,.9527780E-04,.4389180E-05,&
     & .3753560E-04,.1191110E-03,.5773510E-04,.7309656E-04,.8352278E-04,&
     & .2517591E-03,.2051344E-03,.5589648E-04,.3341543E-03,.2978433E-03,&
     & .5905554E-04,.4375873E-03,.4018119E-03,.2342177E-03/


      data   selfref(1:10,1: 8)  /                                      &
     & .7503700E-03,.6449380E-03,.5543210E-03,.4764360E-03,.4094940E-03,&
     & .3519570E-03,.3025050E-03,.2600020E-03,.2234700E-03,.1920710E-03,&
     & .1361350E-02,.1131870E-02,.9410760E-03,.7824400E-03,.6505460E-03,&
     & .5408850E-03,.4497090E-03,.3739020E-03,.3108740E-03,.2584710E-03,&
     & .3339500E-02,.2563910E-02,.1968450E-02,.1511290E-02,.1160300E-02,&
     & .8908240E-03,.6839340E-03,.5250930E-03,.4031430E-03,.3095150E-03,&
     & .7933920E-02,.5898650E-02,.4385480E-02,.3260480E-02,.2424080E-02,&
     & .1802230E-02,.1339910E-02,.9961860E-03,.7406360E-03,.5506420E-03,&
     & .8309571E-02,.7388342E-02,.6584621E-02,.5882080E-02,.5266743E-02,&
     & .4726659E-02,.4251616E-02,.3832861E-02,.3462890E-02,.3135269E-02,&
     & .1297838E-01,.1251031E-01,.1206401E-01,.1163845E-01,.1123247E-01,&
     & .1084516E-01,.1047551E-01,.1012259E-01,.9785517E-02,.9463545E-02,&
     & .1916794E-01,.1802151E-01,.1694407E-01,.1593151E-01,.1497992E-01,&
     & .1408552E-01,.1324501E-01,.1245490E-01,.1171229E-01,.1101426E-01,&
     & .2508831E-01,.2343262E-01,.2189176E-01,.2045740E-01,.1912177E-01,&
     & .1787778E-01,.1671886E-01,.1563887E-01,.1463222E-01,.1369363E-01/

!........................................!
      end module module_radsw_kgb18      !
!========================================!

!> This module sets up absorption coeffients for band 19: 4650-5150
!! cm-1 (low - h2o,co2; high - co2)
!========================================!
      module module_radsw_kgb19          !
!........................................!
!
! *********        the original program descriptions        ********* !
!                                                                     !
!     originally by j.delamere, atmospheric & environmental research. !
!     revision: 2.4                                                   !
!     band 19:  4650-5150 cm-1 (low - h2o,co2; high - co2)            !
!     reformatted for f90 by jjmorcrette, ecmwf                       !
!                                                                     !
!     this table has been re-generated for reduced number of g-point  !
!     by y.t.hou, ncep                                                !
!                                                                     !
! *********    *********    end description    *********    ********* !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NG19

!
      implicit none
!
      private
!
!> MSA19=585
      integer,  public :: MSA19
!> MSB19=235
      integer,  public :: MSB19
!> MSF19=10
      integer,  public :: MSF19
!> MFR19=3
      integer,  public :: MFR19
      parameter (MSA19=585, MSB19=235, MSF19=10, MFR19=3)

!>    the array selfref contains the coefficient of the water vapor
!!    self-continuum (including the energy term).  the first index
!!    refers to temperature in 7.2 degree increments.  for instance,
!!    jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!!    etc.  the second index runs over the g-channel (1 to 8).
      real (kind=kind_phys), public :: selfref(MSF19,NG19)

!>    the array absa(585,NG19) (ka(9,5,13,NG19)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels> ~100mb,
!!    temperatures, and binary species parameters (see taumol.f for definition).
!!    the first index in the array, js, runs from 1 to 9, and corresponds to
!!    different values of the binary species parameter.  for instance,
!!    js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!!    js = 3 corresponds to the parameter value 2/8, etc.  the second index
!!    in the array, jt, which runs from 1 to 5, corresponds to different
!!    temperatures.  more specifically, jt = 3 means that the data are for
!!    the reference temperature tref for this  pressure level, jt = 2 refers
!!    to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!!    is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!!    to the jpth reference pressure level (see taumol.f for these levels
!!    in mb).  the fourth index, ig, goes from 1 to 8, and indicates
!!    which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absa(MSA19,NG19)

!>    the array absb(235,8) (kb(5,13:59,8)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels < ~100mb and
!!    temperatures. the first index in the array, jt, which runs from 1 to 5,
!!    corresponds to different temperatures.  more specifically, jt = 3 means
!!    that the data are for the reference temperature tref for this pressure
!!    level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!!    tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!!    the second index, jp, runs from 13 to 59 and refers to the jpth
!!    reference pressure level (see taumol.f for the value of these
!!    pressure levels in mb).  the third index, ig, goes from 1 to 8,
!!    and tells us which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absb(MSB19,NG19)

      real (kind=kind_phys), public :: forref(MFR19,NG19)

!> rayleigh extinction coefficient at \f$v=4900cm^{-1}\f$
      real (kind=kind_phys), parameter, public :: rayl = 2.29e-09

      data   absa(  1:180, 1)   /                                       &
     & .1498100E-05,.2665900E-05,.3187400E-05,.3550900E-05,.3759300E-05,&
     & .3851400E-05,.3736900E-05,.3501100E-05,.2389400E-05,.1510300E-05,&
     & .2734800E-05,.3269200E-05,.3655800E-05,.3892600E-05,.3973700E-05,&
     & .3879800E-05,.3656500E-05,.2407200E-05,.1523300E-05,.2809000E-05,&
     & .3374400E-05,.3772900E-05,.4025800E-05,.4110700E-05,.4038900E-05,&
     & .3822600E-05,.2470400E-05,.1521900E-05,.2897800E-05,.3501500E-05,&
     & .3898400E-05,.4156900E-05,.4277100E-05,.4211000E-05,.3996300E-05,&
     & .2550400E-05,.1525400E-05,.2963300E-05,.3622400E-05,.4043500E-05,&
     & .4297500E-05,.4441000E-05,.4384900E-05,.4184700E-05,.2642000E-05,&
     & .1302400E-05,.2369900E-05,.2837000E-05,.3144300E-05,.3332600E-05,&
     & .3379800E-05,.3279700E-05,.3007800E-05,.1881900E-05,.1324900E-05,&
     & .2429900E-05,.2910300E-05,.3233600E-05,.3430800E-05,.3501900E-05,&
     & .3404600E-05,.3132300E-05,.1951100E-05,.1324100E-05,.2505900E-05,&
     & .3001900E-05,.3331700E-05,.3551600E-05,.3633300E-05,.3537200E-05,&
     & .3261500E-05,.2025400E-05,.1332500E-05,.2584000E-05,.3100900E-05,&
     & .3439900E-05,.3669900E-05,.3772700E-05,.3681100E-05,.3401600E-05,&
     & .2061100E-05,.1332100E-05,.2660100E-05,.3218900E-05,.3564000E-05,&
     & .3801200E-05,.3911300E-05,.3827100E-05,.3549900E-05,.2136800E-05,&
     & .1107900E-05,.2075700E-05,.2484600E-05,.2751100E-05,.2888300E-05,&
     & .2914200E-05,.2820300E-05,.2570600E-05,.1512400E-05,.1129800E-05,&
     & .2127800E-05,.2555100E-05,.2830600E-05,.2981400E-05,.3012600E-05,&
     & .2932800E-05,.2672100E-05,.1578400E-05,.1140500E-05,.2181200E-05,&
     & .2623700E-05,.2913000E-05,.3081600E-05,.3126500E-05,.3048000E-05,&
     & .2778400E-05,.1646800E-05,.1134700E-05,.2247200E-05,.2709400E-05,&
     & .3005700E-05,.3182100E-05,.3253600E-05,.3171400E-05,.2883700E-05,&
     & .1718900E-05,.1138300E-05,.2315200E-05,.2801600E-05,.3112600E-05,&
     & .3305100E-05,.3375900E-05,.3296100E-05,.2999300E-05,.1791900E-05,&
     & .9310400E-06,.1800500E-05,.2169300E-05,.2381900E-05,.2482900E-05,&
     & .2486000E-05,.2386000E-05,.2158600E-05,.1219800E-05,.9459100E-06,&
     & .1830900E-05,.2215300E-05,.2448700E-05,.2557400E-05,.2572100E-05,&
     & .2480200E-05,.2245800E-05,.1310500E-05,.9563800E-06,.1875300E-05,&
     & .2278800E-05,.2514900E-05,.2649900E-05,.2667100E-05,.2579700E-05,&
     & .2333500E-05,.1401700E-05,.9576300E-06,.1937900E-05,.2327100E-05,&
     & .2585800E-05,.2743700E-05,.2779200E-05,.2679000E-05,.2426800E-05,&
     & .1487200E-05,.9566800E-06,.1987700E-05,.2399700E-05,.2672100E-05,&
     & .2838200E-05,.2880600E-05,.2785700E-05,.2522500E-05,.1569800E-05/

      data   absa(181:315, 1)   /                                       &
     & .7732900E-06,.1535400E-05,.1872300E-05,.2041100E-05,.2109500E-05,&
     & .2101700E-05,.1999800E-05,.1801700E-05,.9762900E-06,.7824100E-06,&
     & .1566000E-05,.1909100E-05,.2097000E-05,.2178600E-05,.2173000E-05,&
     & .2079400E-05,.1873100E-05,.1064500E-05,.7946300E-06,.1594800E-05,&
     & .1945800E-05,.2152900E-05,.2255700E-05,.2257600E-05,.2163100E-05,&
     & .1946200E-05,.1150700E-05,.7977900E-06,.1638900E-05,.1987700E-05,&
     & .2208700E-05,.2327400E-05,.2346200E-05,.2250100E-05,.2020400E-05,&
     & .1236600E-05,.7987500E-06,.1686500E-05,.2041000E-05,.2278900E-05,&
     & .2404900E-05,.2428300E-05,.2338800E-05,.2100600E-05,.1315500E-05,&
     & .6379900E-06,.1307900E-05,.1587300E-05,.1718700E-05,.1768900E-05,&
     & .1750900E-05,.1662500E-05,.1495000E-05,.7762400E-06,.6442300E-06,&
     & .1318000E-05,.1617300E-05,.1764800E-05,.1825300E-05,.1815200E-05,&
     & .1729900E-05,.1548200E-05,.8545700E-06,.6521400E-06,.1348100E-05,&
     & .1647600E-05,.1815800E-05,.1886700E-05,.1885900E-05,.1799900E-05,&
     & .1608000E-05,.9396800E-06,.6594900E-06,.1372800E-05,.1679500E-05,&
     & .1861800E-05,.1950700E-05,.1956800E-05,.1869300E-05,.1671000E-05,&
     & .1025600E-05,.6601200E-06,.1405100E-05,.1717400E-05,.1915400E-05,&
     & .2011100E-05,.2023000E-05,.1942100E-05,.1736100E-05,.1107200E-05,&
     & .5265200E-06,.1099500E-05,.1322100E-05,.1428200E-05,.1462700E-05,&
     & .1449400E-05,.1375400E-05,.1235900E-05,.6239900E-06,.5263100E-06,&
     & .1104700E-05,.1349800E-05,.1464600E-05,.1509400E-05,.1496100E-05,&
     & .1427200E-05,.1279100E-05,.6969200E-06,.5326300E-06,.1123100E-05,&
     & .1376300E-05,.1505900E-05,.1560200E-05,.1553300E-05,.1482700E-05,&
     & .1325700E-05,.7695900E-06,.5378800E-06,.1144800E-05,.1402600E-05,&
     & .1547800E-05,.1612200E-05,.1613700E-05,.1539800E-05,.1374800E-05,&
     & .8448800E-06,.5424300E-06,.1166000E-05,.1428800E-05,.1589100E-05,&
     & .1661900E-05,.1667200E-05,.1597800E-05,.1426500E-05,.9243900E-06/

      data   absa(316:450, 1)   /                                       &
     & .4286000E-06,.9120800E-06,.1088700E-05,.1173200E-05,.1200000E-05,&
     & .1190100E-05,.1130000E-05,.1018100E-05,.5136000E-06,.4260500E-06,&
     & .9135500E-06,.1110500E-05,.1200400E-05,.1235900E-05,.1223800E-05,&
     & .1167400E-05,.1049600E-05,.5859300E-06,.4291000E-06,.9245800E-06,&
     & .1134100E-05,.1235500E-05,.1275400E-05,.1266700E-05,.1210000E-05,&
     & .1087700E-05,.6520900E-06,.4334500E-06,.9402400E-06,.1154000E-05,&
     & .1266900E-05,.1318600E-05,.1316300E-05,.1256700E-05,.1128000E-05,&
     & .7205100E-06,.4374400E-06,.9584500E-06,.1176400E-05,.1302200E-05,&
     & .1359500E-05,.1361800E-05,.1303400E-05,.1167100E-05,.7937300E-06,&
     & .3445800E-06,.7484400E-06,.8921700E-06,.9587500E-06,.9824300E-06,&
     & .9756600E-06,.9293600E-06,.8361100E-06,.4418100E-06,.3479700E-06,&
     & .7530400E-06,.9081600E-06,.9826400E-06,.1008000E-05,.9988900E-06,&
     & .9531800E-06,.8604100E-06,.5101300E-06,.3459300E-06,.7602000E-06,&
     & .9270700E-06,.1006200E-05,.1039100E-05,.1029600E-05,.9853000E-06,&
     & .8901100E-06,.5807000E-06,.3497100E-06,.7708800E-06,.9462300E-06,&
     & .1034200E-05,.1072600E-05,.1068900E-05,.1021000E-05,.9202000E-06,&
     & .6569700E-06,.3528200E-06,.7851700E-06,.9624500E-06,.1060900E-05,&
     & .1107100E-05,.1107300E-05,.1058300E-05,.9505400E-06,.7295600E-06,&
     & .2784000E-06,.6103500E-06,.7293000E-06,.7841500E-06,.8045100E-06,&
     & .7992400E-06,.7623100E-06,.6872700E-06,.4020800E-06,.2817100E-06,&
     & .6179800E-06,.7419700E-06,.8011500E-06,.8218300E-06,.8162100E-06,&
     & .7803000E-06,.7042500E-06,.4670000E-06,.2802300E-06,.6226900E-06,&
     & .7568500E-06,.8210100E-06,.8453800E-06,.8382100E-06,.8030100E-06,&
     & .7266300E-06,.5378100E-06,.2816300E-06,.6315600E-06,.7721400E-06,&
     & .8424400E-06,.8718400E-06,.8675200E-06,.8302400E-06,.7500000E-06,&
     & .6047800E-06,.2851800E-06,.6407000E-06,.7862800E-06,.8643900E-06,&
     & .8996900E-06,.8991000E-06,.8593300E-06,.7737600E-06,.6701600E-06/

      data   absa(451:585, 1)   /                                       &
     & .2255500E-06,.5004200E-06,.5996000E-06,.6453700E-06,.6632400E-06,&
     & .6592400E-06,.6300300E-06,.5689000E-06,.3446600E-06,.2285800E-06,&
     & .5070500E-06,.6099700E-06,.6594800E-06,.6768800E-06,.6731100E-06,&
     & .6446200E-06,.5833300E-06,.4030000E-06,.2267700E-06,.5117800E-06,&
     & .6229900E-06,.6757200E-06,.6962400E-06,.6912600E-06,.6628500E-06,&
     & .6009000E-06,.4606600E-06,.2282500E-06,.5190700E-06,.6350000E-06,&
     & .6931100E-06,.7179600E-06,.7148400E-06,.6841600E-06,.6185900E-06,&
     & .5196600E-06,.2312200E-06,.5259800E-06,.6457400E-06,.7111600E-06,&
     & .7404600E-06,.7397900E-06,.7067200E-06,.6370800E-06,.5819600E-06,&
     & .1814900E-06,.4082900E-06,.4902200E-06,.5291000E-06,.5440800E-06,&
     & .5416300E-06,.5186700E-06,.4693300E-06,.2849600E-06,.1843300E-06,&
     & .4137800E-06,.4990800E-06,.5402700E-06,.5552200E-06,.5528600E-06,&
     & .5303800E-06,.4810300E-06,.3333600E-06,.1824600E-06,.4179900E-06,&
     & .5096500E-06,.5534700E-06,.5713400E-06,.5678700E-06,.5447700E-06,&
     & .4946200E-06,.3802300E-06,.1840900E-06,.4234700E-06,.5193600E-06,&
     & .5674900E-06,.5887700E-06,.5864800E-06,.5614900E-06,.5082500E-06,&
     & .4308100E-06,.1858200E-06,.4288800E-06,.5283000E-06,.5824000E-06,&
     & .6064700E-06,.6059800E-06,.5793300E-06,.5227700E-06,.4856500E-06,&
     & .1434500E-06,.3292700E-06,.3975000E-06,.4300000E-06,.4431800E-06,&
     & .4420100E-06,.4246900E-06,.3853800E-06,.2337700E-06,.1457100E-06,&
     & .3335500E-06,.4043000E-06,.4395400E-06,.4525600E-06,.4515900E-06,&
     & .4339900E-06,.3946700E-06,.2733800E-06,.1444200E-06,.3371900E-06,&
     & .4131600E-06,.4499600E-06,.4657500E-06,.4636400E-06,.4452800E-06,&
     & .4051000E-06,.3115400E-06,.1460000E-06,.3414100E-06,.4213600E-06,&
     & .4609600E-06,.4793700E-06,.4781700E-06,.4585000E-06,.4159800E-06,&
     & .3528900E-06,.1466700E-06,.3460800E-06,.4285300E-06,.4729600E-06,&
     & .4930000E-06,.4935200E-06,.4728000E-06,.4275700E-06,.3975200E-06/

      data   absa(  1:180, 2)   /                                       &
     & .1628800E-04,.2581700E-04,.2959700E-04,.3131900E-04,.3122000E-04,&
     & .2953900E-04,.2562700E-04,.1975800E-04,.7294400E-05,.1812300E-04,&
     & .2856500E-04,.3263300E-04,.3442700E-04,.3421900E-04,.3222300E-04,&
     & .2798700E-04,.2148500E-04,.7955400E-05,.1995700E-04,.3160700E-04,&
     & .3603300E-04,.3788700E-04,.3769400E-04,.3539900E-04,.3071800E-04,&
     & .2341200E-04,.8529300E-05,.2200500E-04,.3506400E-04,.3976300E-04,&
     & .4169400E-04,.4149000E-04,.3889100E-04,.3371500E-04,.2551200E-04,&
     & .9123200E-05,.2424000E-04,.3890600E-04,.4389900E-04,.4587000E-04,&
     & .4558000E-04,.4261600E-04,.3691200E-04,.2772800E-04,.9779200E-05,&
     & .1494400E-04,.2322300E-04,.2579200E-04,.2701900E-04,.2696300E-04,&
     & .2543100E-04,.2206900E-04,.1693300E-04,.5907100E-05,.1661500E-04,&
     & .2545200E-04,.2832200E-04,.2980000E-04,.2959300E-04,.2777600E-04,&
     & .2412300E-04,.1844400E-04,.6314200E-05,.1835500E-04,.2799600E-04,&
     & .3133800E-04,.3289200E-04,.3257500E-04,.3047800E-04,.2650400E-04,&
     & .2013200E-04,.6789800E-05,.2009200E-04,.3093800E-04,.3468000E-04,&
     & .3625500E-04,.3583600E-04,.3350000E-04,.2910600E-04,.2195000E-04,&
     & .7334300E-05,.2189700E-04,.3415000E-04,.3827400E-04,.3993100E-04,&
     & .3933700E-04,.3675200E-04,.3183100E-04,.2388200E-04,.7870500E-05,&
     & .1313000E-04,.2036300E-04,.2237700E-04,.2300300E-04,.2268300E-04,&
     & .2142900E-04,.1862100E-04,.1417700E-04,.4789200E-05,.1461000E-04,&
     & .2229900E-04,.2448600E-04,.2514000E-04,.2484200E-04,.2341300E-04,&
     & .2036100E-04,.1546900E-04,.5225000E-05,.1613300E-04,.2448200E-04,&
     & .2687600E-04,.2765300E-04,.2735700E-04,.2566900E-04,.2234100E-04,&
     & .1689700E-04,.5664700E-05,.1770800E-04,.2688700E-04,.2952200E-04,&
     & .3057700E-04,.3015100E-04,.2822100E-04,.2454500E-04,.1845500E-04,&
     & .6073500E-05,.1930500E-04,.2936400E-04,.3252000E-04,.3374100E-04,&
     & .3316200E-04,.3097600E-04,.2680000E-04,.2010300E-04,.6486400E-05,&
     & .1116300E-04,.1749800E-04,.1906900E-04,.1956100E-04,.1915400E-04,&
     & .1790600E-04,.1552200E-04,.1177800E-04,.4046100E-05,.1250500E-04,&
     & .1923800E-04,.2091700E-04,.2136500E-04,.2091300E-04,.1947100E-04,&
     & .1698600E-04,.1285600E-04,.4393500E-05,.1384800E-04,.2113200E-04,&
     & .2297500E-04,.2343300E-04,.2289300E-04,.2132700E-04,.1866200E-04,&
     & .1405600E-04,.4701500E-05,.1521900E-04,.2315900E-04,.2525000E-04,&
     & .2575300E-04,.2508800E-04,.2343700E-04,.2045500E-04,.1536600E-04,&
     & .5023900E-05,.1663100E-04,.2531100E-04,.2765100E-04,.2824300E-04,&
     & .2756600E-04,.2577500E-04,.2234400E-04,.1675800E-04,.5359400E-05/

      data   absa(181:315, 2)   /                                       &
     & .9268800E-05,.1482900E-04,.1603600E-04,.1638800E-04,.1608700E-04,&
     & .1504900E-04,.1293900E-04,.9724300E-05,.3506400E-05,.1042800E-04,&
     & .1628900E-04,.1759600E-04,.1792800E-04,.1754700E-04,.1634700E-04,&
     & .1410800E-04,.1062100E-04,.3789300E-05,.1160000E-04,.1792200E-04,&
     & .1940300E-04,.1970100E-04,.1922700E-04,.1783500E-04,.1543900E-04,&
     & .1162700E-04,.4042600E-05,.1279500E-04,.1967200E-04,.2136000E-04,&
     & .2167700E-04,.2107800E-04,.1953500E-04,.1691900E-04,.1274600E-04,&
     & .4301500E-05,.1402900E-04,.2155100E-04,.2341000E-04,.2374600E-04,&
     & .2310600E-04,.2139700E-04,.1850000E-04,.1391800E-04,.4566300E-05,&
     & .7535800E-05,.1228400E-04,.1332100E-04,.1358000E-04,.1332200E-04,&
     & .1245900E-04,.1075700E-04,.8001200E-05,.2859500E-05,.8480100E-05,&
     & .1354100E-04,.1460000E-04,.1485700E-04,.1451900E-04,.1352800E-04,&
     & .1174700E-04,.8719900E-05,.3125000E-05,.9479700E-05,.1489800E-04,&
     & .1608800E-04,.1632900E-04,.1592200E-04,.1478600E-04,.1283900E-04,&
     & .9540600E-05,.3392500E-05,.1051000E-04,.1638700E-04,.1770900E-04,&
     & .1796800E-04,.1746500E-04,.1620500E-04,.1399900E-04,.1046500E-04,&
     & .3678600E-05,.1155400E-04,.1797300E-04,.1944200E-04,.1973600E-04,&
     & .1917000E-04,.1775100E-04,.1528800E-04,.1144100E-04,.3961800E-05,&
     & .5999400E-05,.1006800E-04,.1090300E-04,.1110000E-04,.1091000E-04,&
     & .1020700E-04,.8830100E-05,.6575500E-05,.2353700E-05,.6791100E-05,&
     & .1109800E-04,.1196400E-04,.1216100E-04,.1191000E-04,.1111300E-04,&
     & .9647500E-05,.7173300E-05,.2602100E-05,.7601200E-05,.1220500E-04,&
     & .1316700E-04,.1336800E-04,.1305800E-04,.1213600E-04,.1057100E-04,&
     & .7845200E-05,.2841600E-05,.8458200E-05,.1341200E-04,.1451800E-04,&
     & .1472100E-04,.1432800E-04,.1332300E-04,.1152700E-04,.8599500E-05,&
     & .3092500E-05,.9334600E-05,.1474900E-04,.1597300E-04,.1618000E-04,&
     & .1573000E-04,.1459700E-04,.1260100E-04,.9397300E-05,.3328500E-05/

      data   absa(316:450, 2)   /                                       &
     & .4717400E-05,.8136700E-05,.8818400E-05,.9000700E-05,.8836300E-05,&
     & .8305900E-05,.7210100E-05,.5349600E-05,.1886200E-05,.5354300E-05,&
     & .8985200E-05,.9690400E-05,.9866600E-05,.9650900E-05,.9036800E-05,&
     & .7870000E-05,.5834100E-05,.2114700E-05,.6008800E-05,.9885500E-05,&
     & .1066600E-04,.1084900E-04,.1059700E-04,.9882600E-05,.8624300E-05,&
     & .6394900E-05,.2367200E-05,.6695800E-05,.1087300E-04,.1175100E-04,&
     & .1194500E-04,.1163600E-04,.1085200E-04,.9410100E-05,.7026200E-05,&
     & .2632900E-05,.7419100E-05,.1196000E-04,.1294300E-04,.1315200E-04,&
     & .1278800E-04,.1188900E-04,.1029700E-04,.7697700E-05,.2901900E-05,&
     & .3688600E-05,.6514000E-05,.7068400E-05,.7237700E-05,.7108000E-05,&
     & .6707600E-05,.5835300E-05,.4329700E-05,.1641900E-05,.4177200E-05,&
     & .7210400E-05,.7779300E-05,.7933500E-05,.7768200E-05,.7290300E-05,&
     & .6364000E-05,.4721000E-05,.1872100E-05,.4712300E-05,.7947100E-05,&
     & .8566200E-05,.8739600E-05,.8531900E-05,.7976900E-05,.6969400E-05,&
     & .5179300E-05,.2101500E-05,.5261200E-05,.8731900E-05,.9436500E-05,&
     & .9619100E-05,.9378300E-05,.8758100E-05,.7627700E-05,.5697400E-05,&
     & .2350500E-05,.5840100E-05,.9610600E-05,.1040400E-04,.1059400E-04,&
     & .1030800E-04,.9603000E-05,.8340200E-05,.6262700E-05,.2617000E-05,&
     & .2902300E-05,.5223000E-05,.5666900E-05,.5815800E-05,.5713700E-05,&
     & .5424100E-05,.4741600E-05,.3509800E-05,.1422300E-05,.3275200E-05,&
     & .5790300E-05,.6254800E-05,.6385900E-05,.6252800E-05,.5888100E-05,&
     & .5154200E-05,.3825700E-05,.1655900E-05,.3704700E-05,.6389200E-05,&
     & .6892800E-05,.7031200E-05,.6873400E-05,.6435700E-05,.5636100E-05,&
     & .4200300E-05,.1924800E-05,.4147100E-05,.7017900E-05,.7592000E-05,&
     & .7749600E-05,.7550800E-05,.7062400E-05,.6177800E-05,.4627100E-05,&
     & .2207400E-05,.4606100E-05,.7721500E-05,.8369900E-05,.8530000E-05,&
     & .8312100E-05,.7749300E-05,.6755100E-05,.5093100E-05,.2504300E-05/

      data   absa(451:585, 2)   /                                       &
     & .2406000E-05,.4375200E-05,.4737200E-05,.4859800E-05,.4768900E-05,&
     & .4529000E-05,.3965300E-05,.2944400E-05,.1225500E-05,.2712300E-05,&
     & .4847900E-05,.5233900E-05,.5348200E-05,.5234200E-05,.4927500E-05,&
     & .4323900E-05,.3219600E-05,.1428200E-05,.3063200E-05,.5340200E-05,&
     & .5766000E-05,.5893600E-05,.5751500E-05,.5387300E-05,.4734400E-05,&
     & .3541000E-05,.1663600E-05,.3424500E-05,.5865300E-05,.6355400E-05,&
     & .6487800E-05,.6324400E-05,.5914700E-05,.5185500E-05,.3906800E-05,&
     & .1885800E-05,.3798400E-05,.6455700E-05,.7003200E-05,.7143200E-05,&
     & .6959100E-05,.6493600E-05,.5674800E-05,.4288900E-05,.2152900E-05,&
     & .1982200E-05,.3643800E-05,.3943900E-05,.4042800E-05,.3965600E-05,&
     & .3762700E-05,.3304800E-05,.2462000E-05,.1034900E-05,.2234700E-05,&
     & .4032900E-05,.4355900E-05,.4453500E-05,.4355600E-05,.4101800E-05,&
     & .3615600E-05,.2697600E-05,.1207800E-05,.2519600E-05,.4440500E-05,&
     & .4796500E-05,.4902500E-05,.4785000E-05,.4485400E-05,.3954200E-05,&
     & .2973400E-05,.1390800E-05,.2814200E-05,.4879200E-05,.5288200E-05,&
     & .5396200E-05,.5264600E-05,.4920200E-05,.4322900E-05,.3281700E-05,&
     & .1589500E-05,.3118200E-05,.5368400E-05,.5825600E-05,.5941300E-05,&
     & .5785500E-05,.5404300E-05,.4729800E-05,.3592500E-05,.1805000E-05,&
     & .1620600E-05,.3012200E-05,.3261200E-05,.3345100E-05,.3279400E-05,&
     & .3114400E-05,.2743000E-05,.2048800E-05,.8526700E-06,.1826700E-05,&
     & .3333100E-05,.3600200E-05,.3681600E-05,.3605400E-05,.3395000E-05,&
     & .3002100E-05,.2248700E-05,.9949000E-06,.2058700E-05,.3668100E-05,&
     & .3965100E-05,.4053100E-05,.3956400E-05,.3712700E-05,.3281500E-05,&
     & .2479500E-05,.1142200E-05,.2297300E-05,.4032500E-05,.4372800E-05,&
     & .4460000E-05,.4354000E-05,.4071800E-05,.3584800E-05,.2736600E-05,&
     & .1305100E-05,.2545200E-05,.4437900E-05,.4816400E-05,.4910000E-05,&
     & .4783300E-05,.4467400E-05,.3919400E-05,.2990400E-05,.1481700E-05/

      data   absa(  1:180, 3)   /                                       &
     & .6664100E-04,.9299000E-04,.9627700E-04,.9518200E-04,.9142400E-04,&
     & .8544500E-04,.7701400E-04,.6209100E-04,.1828400E-04,.7535000E-04,&
     & .1039100E-03,.1075100E-03,.1060600E-03,.1017400E-03,.9475600E-04,&
     & .8448700E-04,.6726700E-04,.2003700E-04,.8528600E-04,.1156300E-03,&
     & .1196600E-03,.1177700E-03,.1125900E-03,.1043500E-03,.9207600E-04,&
     & .7249200E-04,.2222900E-04,.9613700E-04,.1283600E-03,.1326500E-03,&
     & .1303100E-03,.1240400E-03,.1140900E-03,.9975600E-04,.7789700E-04,&
     & .2465600E-04,.1073900E-03,.1420900E-03,.1465200E-03,.1436300E-03,&
     & .1357900E-03,.1240800E-03,.1074900E-03,.8327300E-04,.2699800E-04,&
     & .5700500E-04,.8068800E-04,.8419900E-04,.8306700E-04,.7948000E-04,&
     & .7409600E-04,.6632800E-04,.5353200E-04,.1453400E-04,.6451800E-04,&
     & .9027400E-04,.9414900E-04,.9249600E-04,.8837400E-04,.8204800E-04,&
     & .7269800E-04,.5787100E-04,.1620200E-04,.7301400E-04,.1006500E-03,&
     & .1046600E-03,.1027100E-03,.9788000E-04,.9037200E-04,.7936100E-04,&
     & .6235700E-04,.1805600E-04,.8229700E-04,.1119500E-03,.1158600E-03,&
     & .1136500E-03,.1079900E-03,.9882900E-04,.8592300E-04,.6689100E-04,&
     & .1989000E-04,.9217500E-04,.1239900E-03,.1279400E-03,.1253100E-03,&
     & .1182500E-03,.1076400E-03,.9274400E-04,.7160800E-04,.2196800E-04,&
     & .4732400E-04,.6816700E-04,.7106900E-04,.7029300E-04,.6731300E-04,&
     & .6245600E-04,.5564600E-04,.4493300E-04,.1137400E-04,.5365800E-04,&
     & .7634200E-04,.7956000E-04,.7854700E-04,.7499700E-04,.6926400E-04,&
     & .6111900E-04,.4858700E-04,.1263900E-04,.6090500E-04,.8524700E-04,&
     & .8873800E-04,.8738700E-04,.8309800E-04,.7643200E-04,.6673300E-04,&
     & .5227500E-04,.1387100E-04,.6883100E-04,.9496000E-04,.9863800E-04,&
     & .9677800E-04,.9181600E-04,.8374700E-04,.7240300E-04,.5616500E-04,&
     & .1542300E-04,.7725900E-04,.1054900E-03,.1090900E-03,.1067200E-03,&
     & .1006500E-03,.9132000E-04,.7840600E-04,.6015700E-04,.1714100E-04,&
     & .4010200E-04,.5699000E-04,.5944400E-04,.5873200E-04,.5609100E-04,&
     & .5190200E-04,.4624700E-04,.3744200E-04,.9402200E-05,.4493000E-04,&
     & .6389300E-04,.6662900E-04,.6569400E-04,.6262700E-04,.5782200E-04,&
     & .5081900E-04,.4042100E-04,.1047500E-04,.5053200E-04,.7142400E-04,&
     & .7435200E-04,.7322100E-04,.6965100E-04,.6403400E-04,.5543000E-04,&
     & .4352900E-04,.1172800E-04,.5695300E-04,.7974000E-04,.8276900E-04,&
     & .8131400E-04,.7727800E-04,.7032400E-04,.6039600E-04,.4674700E-04,&
     & .1306300E-04,.6401900E-04,.8864700E-04,.9176800E-04,.8992600E-04,&
     & .8495800E-04,.7674000E-04,.6556100E-04,.5006600E-04,.1443500E-04/

      data   absa(181:315, 3)   /                                       &
     & .3380600E-04,.4782600E-04,.4944300E-04,.4877500E-04,.4647700E-04,&
     & .4279100E-04,.3815500E-04,.3117500E-04,.7788600E-05,.3797300E-04,&
     & .5339200E-04,.5552600E-04,.5471900E-04,.5206200E-04,.4776300E-04,&
     & .4202100E-04,.3351000E-04,.8740400E-05,.4275800E-04,.5966600E-04,&
     & .6205800E-04,.6112700E-04,.5801200E-04,.5307100E-04,.4597100E-04,&
     & .3611400E-04,.9849900E-05,.4800600E-04,.6667900E-04,.6923000E-04,&
     & .6798900E-04,.6440000E-04,.5852100E-04,.5019600E-04,.3871000E-04,&
     & .1102200E-04,.5365700E-04,.7424400E-04,.7681300E-04,.7530300E-04,&
     & .7101100E-04,.6409200E-04,.5459900E-04,.4148100E-04,.1222500E-04,&
     & .2795300E-04,.4022300E-04,.4133100E-04,.4033200E-04,.3811400E-04,&
     & .3496700E-04,.3115800E-04,.2570100E-04,.6760400E-05,.3159500E-04,&
     & .4497500E-04,.4637400E-04,.4522900E-04,.4277500E-04,.3916200E-04,&
     & .3437300E-04,.2761600E-04,.7555800E-05,.3564900E-04,.5026300E-04,&
     & .5171300E-04,.5051600E-04,.4785500E-04,.4360500E-04,.3770500E-04,&
     & .2980000E-04,.8406300E-05,.4008600E-04,.5590500E-04,.5758400E-04,&
     & .5634800E-04,.5320000E-04,.4828300E-04,.4133100E-04,.3193600E-04,&
     & .9268600E-05,.4493100E-04,.6202500E-04,.6394200E-04,.6249800E-04,&
     & .5878800E-04,.5303200E-04,.4508500E-04,.3422900E-04,.1017500E-04,&
     & .2282900E-04,.3327800E-04,.3419600E-04,.3338100E-04,.3144400E-04,&
     & .2861500E-04,.2533100E-04,.2109600E-04,.6073200E-05,.2591700E-04,&
     & .3742400E-04,.3852700E-04,.3758000E-04,.3530200E-04,.3202000E-04,&
     & .2795700E-04,.2266000E-04,.6818000E-05,.2938600E-04,.4193300E-04,&
     & .4321300E-04,.4203900E-04,.3944500E-04,.3565900E-04,.3078000E-04,&
     & .2443200E-04,.7629600E-05,.3318600E-04,.4681100E-04,.4821800E-04,&
     & .4684900E-04,.4384800E-04,.3956700E-04,.3385800E-04,.2625700E-04,&
     & .8263400E-05,.3722700E-04,.5206000E-04,.5359200E-04,.5194800E-04,&
     & .4847100E-04,.4359500E-04,.3702200E-04,.2813000E-04,.9005300E-05/

      data   absa(316:450, 3)   /                                       &
     & .1833300E-04,.2725900E-04,.2792900E-04,.2723000E-04,.2567200E-04,&
     & .2337400E-04,.2057100E-04,.1733300E-04,.5415700E-05,.2094800E-04,&
     & .3077400E-04,.3160400E-04,.3082300E-04,.2900200E-04,.2629200E-04,&
     & .2273400E-04,.1859300E-04,.6166900E-05,.2388300E-04,.3464200E-04,&
     & .3565600E-04,.3467000E-04,.3250700E-04,.2930300E-04,.2511100E-04,&
     & .1998500E-04,.6904800E-05,.2709800E-04,.3880200E-04,.4001300E-04,&
     & .3877700E-04,.3621400E-04,.3254200E-04,.2767400E-04,.2152600E-04,&
     & .7659100E-05,.3046200E-04,.4330700E-04,.4459600E-04,.4312700E-04,&
     & .4014200E-04,.3589000E-04,.3027600E-04,.2306500E-04,.8448000E-05,&
     & .1451000E-04,.2204900E-04,.2259600E-04,.2202100E-04,.2075000E-04,&
     & .1889500E-04,.1668500E-04,.1415800E-04,.5028400E-05,.1671000E-04,&
     & .2501200E-04,.2568600E-04,.2501700E-04,.2353700E-04,.2135000E-04,&
     & .1851700E-04,.1517000E-04,.5901600E-05,.1913800E-04,.2827100E-04,&
     & .2910400E-04,.2827200E-04,.2649300E-04,.2391600E-04,.2053000E-04,&
     & .1629700E-04,.6841300E-05,.2179800E-04,.3185900E-04,.3280200E-04,&
     & .3176200E-04,.2960300E-04,.2664300E-04,.2265000E-04,.1755900E-04,&
     & .7809300E-05,.2459900E-04,.3561800E-04,.3673600E-04,.3545600E-04,&
     & .3298100E-04,.2945900E-04,.2487700E-04,.1882100E-04,.8814900E-05,&
     & .1145500E-04,.1779600E-04,.1828200E-04,.1783300E-04,.1681400E-04,&
     & .1530500E-04,.1352600E-04,.1162800E-04,.4763700E-05,.1329500E-04,&
     & .2029000E-04,.2086200E-04,.2031600E-04,.1912500E-04,.1734800E-04,&
     & .1504200E-04,.1245500E-04,.5649500E-05,.1529700E-04,.2299400E-04,&
     & .2374000E-04,.2307600E-04,.2162600E-04,.1952900E-04,.1676600E-04,&
     & .1339900E-04,.6598800E-05,.1748800E-04,.2598600E-04,.2688200E-04,&
     & .2602100E-04,.2424600E-04,.2182000E-04,.1855400E-04,.1442700E-04,&
     & .7464400E-05,.1980000E-04,.2916000E-04,.3020800E-04,.2916100E-04,&
     & .2708600E-04,.2418900E-04,.2046200E-04,.1547400E-04,.8439500E-05/

      data   absa(451:585, 3)   /                                       &
     & .9636600E-05,.1516800E-04,.1561100E-04,.1523000E-04,.1438100E-04,&
     & .1305800E-04,.1145700E-04,.9819500E-05,.4321000E-05,.1122100E-04,&
     & .1730400E-04,.1785900E-04,.1739000E-04,.1638300E-04,.1483700E-04,&
     & .1282000E-04,.1054100E-04,.5091500E-05,.1292200E-04,.1961100E-04,&
     & .2037600E-04,.1976900E-04,.1851000E-04,.1673300E-04,.1429700E-04,&
     & .1137500E-04,.5934900E-05,.1475900E-04,.2222600E-04,.2308700E-04,&
     & .2231900E-04,.2078900E-04,.1867700E-04,.1585400E-04,.1223300E-04,&
     & .6864600E-05,.1668700E-04,.2490100E-04,.2589400E-04,.2502000E-04,&
     & .2323400E-04,.2074600E-04,.1745000E-04,.1314300E-04,.7794500E-05,&
     & .8072800E-05,.1283300E-04,.1324100E-04,.1293000E-04,.1220900E-04,&
     & .1107700E-04,.9687400E-05,.8251800E-05,.3698700E-05,.9409600E-05,&
     & .1464700E-04,.1517900E-04,.1478900E-04,.1392900E-04,.1261400E-04,&
     & .1086300E-04,.8883900E-05,.4364200E-05,.1082700E-04,.1662500E-04,&
     & .1733600E-04,.1682500E-04,.1575700E-04,.1421500E-04,.1213500E-04,&
     & .9570200E-05,.5127600E-05,.1234600E-04,.1882300E-04,.1962600E-04,&
     & .1901100E-04,.1771500E-04,.1589100E-04,.1347200E-04,.1029700E-04,&
     & .5881800E-05,.1395400E-04,.2107100E-04,.2198500E-04,.2127800E-04,&
     & .1977300E-04,.1764200E-04,.1481300E-04,.1110900E-04,.6648500E-05,&
     & .6719800E-05,.1077700E-04,.1115300E-04,.1090000E-04,.1030100E-04,&
     & .9334500E-05,.8143300E-05,.6915600E-05,.3049800E-05,.7828300E-05,&
     & .1228300E-04,.1280500E-04,.1249200E-04,.1175700E-04,.1064900E-04,&
     & .9151900E-05,.7449500E-05,.3602800E-05,.8996600E-05,.1398300E-04,&
     & .1462900E-04,.1421800E-04,.1331500E-04,.1199000E-04,.1023000E-04,&
     & .8023700E-05,.4238200E-05,.1025000E-04,.1579400E-04,.1653600E-04,&
     & .1605500E-04,.1497900E-04,.1342100E-04,.1135500E-04,.8637100E-05,&
     & .4851100E-05,.1157500E-04,.1768900E-04,.1850200E-04,.1795000E-04,&
     & .1668900E-04,.1488300E-04,.1247600E-04,.9353700E-05,.5479900E-05/

      data   absa(  1:180, 4)   /                                       &
     & .2031300E-03,.2641000E-03,.2705400E-03,.2584200E-03,.2379400E-03,&
     & .2103300E-03,.1773000E-03,.1383100E-03,.5299500E-04,.2272200E-03,&
     & .2905800E-03,.2957200E-03,.2820000E-03,.2589800E-03,.2281400E-03,&
     & .1920900E-03,.1495700E-03,.5909600E-04,.2505400E-03,.3158000E-03,&
     & .3199100E-03,.3048600E-03,.2795500E-03,.2463400E-03,.2078700E-03,&
     & .1617900E-03,.6511800E-04,.2719700E-03,.3397100E-03,.3431000E-03,&
     & .3267700E-03,.3000200E-03,.2657400E-03,.2250200E-03,.1752000E-03,&
     & .7171100E-04,.2917800E-03,.3621300E-03,.3650300E-03,.3481500E-03,&
     & .3211400E-03,.2859900E-03,.2432800E-03,.1895800E-03,.7852300E-04,&
     & .1754700E-03,.2301200E-03,.2363900E-03,.2260600E-03,.2079900E-03,&
     & .1832600E-03,.1538500E-03,.1188800E-03,.4363300E-04,.1960100E-03,&
     & .2526900E-03,.2575100E-03,.2462900E-03,.2262300E-03,.1992600E-03,&
     & .1671200E-03,.1287700E-03,.4838200E-04,.2159800E-03,.2742300E-03,&
     & .2781200E-03,.2657600E-03,.2440300E-03,.2155200E-03,.1814100E-03,&
     & .1398200E-03,.5356000E-04,.2344900E-03,.2948200E-03,.2982000E-03,&
     & .2845400E-03,.2617100E-03,.2325400E-03,.1968100E-03,.1519200E-03,&
     & .5905400E-04,.2519800E-03,.3140800E-03,.3172300E-03,.3034100E-03,&
     & .2804000E-03,.2502200E-03,.2133300E-03,.1648700E-03,.6477900E-04,&
     & .1473600E-03,.1948300E-03,.2006500E-03,.1924200E-03,.1772800E-03,&
     & .1562700E-03,.1307800E-03,.1003700E-03,.3421700E-04,.1649200E-03,&
     & .2144100E-03,.2189200E-03,.2098000E-03,.1928900E-03,.1700000E-03,&
     & .1424900E-03,.1090200E-03,.3835200E-04,.1820000E-03,.2329400E-03,&
     & .2365800E-03,.2265800E-03,.2082200E-03,.1839100E-03,.1548400E-03,&
     & .1188800E-03,.4270000E-04,.1981100E-03,.2509400E-03,.2538500E-03,&
     & .2427600E-03,.2233800E-03,.1984100E-03,.1682200E-03,.1294300E-03,&
     & .4725700E-04,.2135200E-03,.2678400E-03,.2702700E-03,.2591700E-03,&
     & .2396600E-03,.2139900E-03,.1823900E-03,.1405800E-03,.5199700E-04,&
     & .1207000E-03,.1628200E-03,.1687000E-03,.1621500E-03,.1496600E-03,&
     & .1319500E-03,.1100300E-03,.8374300E-04,.2684100E-04,.1361600E-03,&
     & .1797900E-03,.1842000E-03,.1769200E-03,.1628800E-03,.1435100E-03,&
     & .1200000E-03,.9132400E-04,.2995600E-04,.1514200E-03,.1960600E-03,&
     & .1993300E-03,.1913200E-03,.1758200E-03,.1550600E-03,.1306100E-03,&
     & .9988100E-04,.3317500E-04,.1659100E-03,.2117000E-03,.2142100E-03,&
     & .2052500E-03,.1888800E-03,.1674700E-03,.1419200E-03,.1090700E-03,&
     & .3662500E-04,.1795600E-03,.2265500E-03,.2283300E-03,.2194900E-03,&
     & .2028300E-03,.1808600E-03,.1539300E-03,.1186500E-03,.4051300E-04/

      data   absa(181:315, 4)   /                                       &
     & .9862500E-04,.1348000E-03,.1409900E-03,.1365300E-03,.1260000E-03,&
     & .1110700E-03,.9210200E-04,.6941400E-04,.2259700E-04,.1116500E-03,&
     & .1497500E-03,.1542900E-03,.1488600E-03,.1371000E-03,.1207600E-03,&
     & .1005100E-03,.7604600E-04,.2526500E-04,.1246100E-03,.1639100E-03,&
     & .1672500E-03,.1609100E-03,.1479600E-03,.1304700E-03,.1094100E-03,&
     & .8319300E-04,.2796200E-04,.1373800E-03,.1776100E-03,.1798200E-03,&
     & .1728100E-03,.1591600E-03,.1408700E-03,.1189200E-03,.9097200E-04,&
     & .3085500E-04,.1496700E-03,.1906600E-03,.1923100E-03,.1850900E-03,&
     & .1710900E-03,.1522700E-03,.1290600E-03,.9913600E-04,.3401700E-04,&
     & .7986300E-04,.1103400E-03,.1162200E-03,.1140000E-03,.1053200E-03,&
     & .9297800E-04,.7666100E-04,.5728700E-04,.1866400E-04,.9075300E-04,&
     & .1231800E-03,.1276600E-03,.1244000E-03,.1148200E-03,.1011000E-03,&
     & .8369700E-04,.6266300E-04,.2092800E-04,.1018300E-03,.1353200E-03,&
     & .1390700E-03,.1346400E-03,.1239600E-03,.1092500E-03,.9098400E-04,&
     & .6851300E-04,.2326400E-04,.1129300E-03,.1474200E-03,.1499500E-03,&
     & .1448100E-03,.1335400E-03,.1179700E-03,.9884500E-04,.7496500E-04,&
     & .2585000E-04,.1236000E-03,.1591400E-03,.1609000E-03,.1552100E-03,&
     & .1436100E-03,.1275500E-03,.1073900E-03,.8191500E-04,.2863500E-04,&
     & .6567900E-04,.9008400E-04,.9568100E-04,.9456700E-04,.8733800E-04,&
     & .7727500E-04,.6364200E-04,.4709700E-04,.1550800E-04,.7455100E-04,&
     & .1008900E-03,.1053100E-03,.1032700E-03,.9536400E-04,.8415400E-04,&
     & .6956900E-04,.5142600E-04,.1723400E-04,.8358400E-04,.1113700E-03,&
     & .1146900E-03,.1119300E-03,.1032500E-03,.9107500E-04,.7547900E-04,&
     & .5622800E-04,.1912700E-04,.9276200E-04,.1217100E-03,.1242000E-03,&
     & .1206000E-03,.1114700E-03,.9846500E-04,.8196900E-04,.6152900E-04,&
     & .2145200E-04,.1018900E-03,.1318200E-03,.1336400E-03,.1296000E-03,&
     & .1200800E-03,.1065200E-03,.8904500E-04,.6733400E-04,.2371100E-04/

      data   absa(316:450, 4)   /                                       &
     & .5365600E-04,.7410900E-04,.7862900E-04,.7824500E-04,.7232200E-04,&
     & .6391500E-04,.5262500E-04,.3859100E-04,.1441600E-04,.6116300E-04,&
     & .8310400E-04,.8683200E-04,.8556200E-04,.7899200E-04,.6966600E-04,&
     & .5757400E-04,.4209800E-04,.1592100E-04,.6891400E-04,.9190900E-04,&
     & .9459400E-04,.9281600E-04,.8559000E-04,.7549100E-04,.6244500E-04,&
     & .4610900E-04,.1720900E-04,.7684700E-04,.1005900E-03,.1025600E-03,&
     & .1001900E-03,.9249600E-04,.8172900E-04,.6776300E-04,.5044500E-04,&
     & .1858900E-04,.8484700E-04,.1090200E-03,.1107100E-03,.1078400E-03,&
     & .9981200E-04,.8851600E-04,.7371800E-04,.5525000E-04,.2019400E-04,&
     & .4334600E-04,.6060800E-04,.6481800E-04,.6468100E-04,.5965500E-04,&
     & .5261700E-04,.4320600E-04,.3157300E-04,.1469600E-04,.4966900E-04,&
     & .6838000E-04,.7180000E-04,.7086000E-04,.6526000E-04,.5749600E-04,&
     & .4731300E-04,.3440500E-04,.1616400E-04,.5627400E-04,.7604000E-04,&
     & .7845400E-04,.7694400E-04,.7066500E-04,.6234700E-04,.5138300E-04,&
     & .3766500E-04,.1784100E-04,.6312700E-04,.8345100E-04,.8517800E-04,&
     & .8308200E-04,.7643700E-04,.6753800E-04,.5579700E-04,.4126400E-04,&
     & .1950400E-04,.7012400E-04,.9080700E-04,.9196000E-04,.8943200E-04,&
     & .8252700E-04,.7317600E-04,.6066800E-04,.4517900E-04,.2099200E-04,&
     & .3505900E-04,.4953100E-04,.5337600E-04,.5366500E-04,.4964400E-04,&
     & .4359200E-04,.3560900E-04,.2584500E-04,.1323100E-04,.4037600E-04,&
     & .5617400E-04,.5950200E-04,.5904000E-04,.5439900E-04,.4768700E-04,&
     & .3900700E-04,.2818900E-04,.1522300E-04,.4600600E-04,.6289900E-04,&
     & .6521400E-04,.6417800E-04,.5897600E-04,.5172600E-04,.4238600E-04,&
     & .3084300E-04,.1724100E-04,.5190500E-04,.6939500E-04,.7098800E-04,&
     & .6946100E-04,.6380100E-04,.5604000E-04,.4605900E-04,.3383100E-04,&
     & .1971400E-04,.5799000E-04,.7573700E-04,.7682600E-04,.7471800E-04,&
     & .6883900E-04,.6066400E-04,.5010500E-04,.3704900E-04,.2230800E-04/

      data   absa(451:585, 4)   /                                       &
     & .3015000E-04,.4273800E-04,.4601100E-04,.4633000E-04,.4290700E-04,&
     & .3760500E-04,.3070200E-04,.2196200E-04,.1179100E-04,.3471400E-04,&
     & .4851600E-04,.5112000E-04,.5072600E-04,.4691200E-04,.4103300E-04,&
     & .3350500E-04,.2395200E-04,.1324200E-04,.3964900E-04,.5432500E-04,&
     & .5614600E-04,.5527000E-04,.5090200E-04,.4455900E-04,.3635100E-04,&
     & .2626600E-04,.1507500E-04,.4477900E-04,.5989300E-04,.6116600E-04,&
     & .5976600E-04,.5507000E-04,.4830300E-04,.3949100E-04,.2884700E-04,&
     & .1716400E-04,.4994600E-04,.6541100E-04,.6625600E-04,.6434500E-04,&
     & .5947400E-04,.5228700E-04,.4300300E-04,.3161200E-04,.1954200E-04,&
     & .2568100E-04,.3662200E-04,.3942800E-04,.3958200E-04,.3678600E-04,&
     & .3221000E-04,.2627000E-04,.1862900E-04,.1003500E-04,.2963600E-04,&
     & .4162000E-04,.4370100E-04,.4342500E-04,.4015000E-04,.3510500E-04,&
     & .2863700E-04,.2036500E-04,.1141600E-04,.3392100E-04,.4662800E-04,&
     & .4806200E-04,.4730100E-04,.4363300E-04,.3818900E-04,.3113300E-04,&
     & .2240500E-04,.1294600E-04,.3834100E-04,.5140100E-04,.5245700E-04,&
     & .5112900E-04,.4723800E-04,.4142900E-04,.3390500E-04,.2454600E-04,&
     & .1477000E-04,.4257600E-04,.5615300E-04,.5686300E-04,.5518200E-04,&
     & .5111300E-04,.4494000E-04,.3698700E-04,.2689800E-04,.1686500E-04,&
     & .2170700E-04,.3114200E-04,.3348700E-04,.3365000E-04,.3130100E-04,&
     & .2739600E-04,.2235200E-04,.1574400E-04,.8300500E-05,.2511200E-04,&
     & .3547400E-04,.3715800E-04,.3697000E-04,.3417000E-04,.2987100E-04,&
     & .2436100E-04,.1731200E-04,.9355900E-05,.2876200E-04,.3971200E-04,&
     & .4092400E-04,.4019900E-04,.3716900E-04,.3256500E-04,.2656600E-04,&
     & .1907400E-04,.1067000E-04,.3244800E-04,.4380800E-04,.4473400E-04,&
     & .4353400E-04,.4032300E-04,.3536700E-04,.2901100E-04,.2089700E-04,&
     & .1224000E-04,.3590500E-04,.4777100E-04,.4848200E-04,.4709400E-04,&
     & .4370300E-04,.3847400E-04,.3171500E-04,.2289900E-04,.1395800E-04/

      data   absa(  1:180, 5)   /                                       &
     & .7617673E-03,.8971104E-03,.9278378E-03,.9076002E-03,.8545237E-03,&
     & .7758218E-03,.6640038E-03,.5009860E-03,.2100718E-03,.7925828E-03,&
     & .9442955E-03,.9755110E-03,.9542425E-03,.9014791E-03,.8188875E-03,&
     & .7022179E-03,.5337239E-03,.2374824E-03,.8249630E-03,.9924062E-03,&
     & .1024145E-02,.1003594E-02,.9497958E-03,.8633171E-03,.7423153E-03,&
     & .5703062E-03,.2679147E-03,.8583524E-03,.1040661E-02,.1074578E-02,&
     & .1055443E-02,.9993194E-03,.9103749E-03,.7856031E-03,.6103198E-03,&
     & .3005890E-03,.8914210E-03,.1089831E-02,.1127450E-02,.1108978E-02,&
     & .1051150E-02,.9598170E-03,.8317411E-03,.6540146E-03,.3372384E-03,&
     & .6627779E-03,.7779969E-03,.8008718E-03,.7838334E-03,.7398886E-03,&
     & .6733388E-03,.5758040E-03,.4394197E-03,.1791672E-03,.6931434E-03,&
     & .8216003E-03,.8453718E-03,.8264838E-03,.7819403E-03,.7110987E-03,&
     & .6107270E-03,.4698192E-03,.2021868E-03,.7240368E-03,.8657540E-03,&
     & .8908257E-03,.8720081E-03,.8257095E-03,.7512371E-03,.6480081E-03,&
     & .5029834E-03,.2276890E-03,.7549662E-03,.9109098E-03,.9378553E-03,&
     & .9196391E-03,.8709512E-03,.7935808E-03,.6875784E-03,.5392493E-03,&
     & .2558951E-03,.7854858E-03,.9570998E-03,.9870226E-03,.9682145E-03,&
     & .9174446E-03,.8393444E-03,.7305754E-03,.5787675E-03,.2867225E-03,&
     & .5698218E-03,.6646945E-03,.6811412E-03,.6675254E-03,.6306449E-03,&
     & .5735521E-03,.4911267E-03,.3756753E-03,.1468372E-03,.5979398E-03,&
     & .7039422E-03,.7219077E-03,.7061823E-03,.6682445E-03,.6070832E-03,&
     & .5222789E-03,.4034172E-03,.1660478E-03,.6260630E-03,.7441821E-03,&
     & .7640366E-03,.7472492E-03,.7070503E-03,.6422790E-03,.5549202E-03,&
     & .4333711E-03,.1875462E-03,.6541096E-03,.7851727E-03,.8072018E-03,&
     & .7903360E-03,.7475103E-03,.6804148E-03,.5905641E-03,.4657632E-03,&
     & .2111058E-03,.6816820E-03,.8277472E-03,.8519631E-03,.8340047E-03,&
     & .7892681E-03,.7212366E-03,.6290190E-03,.5008742E-03,.2367354E-03,&
     & .4861342E-03,.5638913E-03,.5756770E-03,.5644357E-03,.5327681E-03,&
     & .4835474E-03,.4138863E-03,.3179456E-03,.1189626E-03,.5113753E-03,&
     & .5987845E-03,.6124007E-03,.5992699E-03,.5662793E-03,.5134428E-03,&
     & .4408138E-03,.3421165E-03,.1351360E-03,.5363872E-03,.6346295E-03,&
     & .6504470E-03,.6359055E-03,.6008677E-03,.5448237E-03,.4696582E-03,&
     & .3682585E-03,.1527590E-03,.5608616E-03,.6715761E-03,.6895903E-03,&
     & .6743859E-03,.6365185E-03,.5786033E-03,.5010073E-03,.3967389E-03,&
     & .1724827E-03,.5849723E-03,.7098200E-03,.7303204E-03,.7137716E-03,&
     & .6742570E-03,.6146787E-03,.5348228E-03,.4271745E-03,.1936713E-03/

      data   absa(181:315, 5)   /                                       &
     & .4124715E-03,.4761738E-03,.4853020E-03,.4745565E-03,.4479021E-03,&
     & .4057915E-03,.3474465E-03,.2667416E-03,.9516165E-04,.4344911E-03,&
     & .5069050E-03,.5177953E-03,.5062569E-03,.4775319E-03,.4319312E-03,&
     & .3702139E-03,.2876629E-03,.1085617E-03,.4559645E-03,.5386154E-03,&
     & .5518018E-03,.5389479E-03,.5081875E-03,.4599769E-03,.3953892E-03,&
     & .3104428E-03,.1229340E-03,.4769668E-03,.5709955E-03,.5869421E-03,&
     & .5732638E-03,.5400999E-03,.4895927E-03,.4226193E-03,.3347530E-03,&
     & .1391112E-03,.4977864E-03,.6042702E-03,.6230627E-03,.6084073E-03,&
     & .5737828E-03,.5212317E-03,.4520350E-03,.3599815E-03,.1565328E-03,&
     & .3481462E-03,.3997927E-03,.4064302E-03,.3958879E-03,.3740189E-03,&
     & .3386598E-03,.2894388E-03,.2220176E-03,.7721520E-04,.3667478E-03,&
     & .4264134E-03,.4347817E-03,.4238152E-03,.3999290E-03,.3615359E-03,&
     & .3088615E-03,.2396985E-03,.8791758E-04,.3848509E-03,.4536800E-03,&
     & .4643234E-03,.4533341E-03,.4271237E-03,.3859069E-03,.3305190E-03,&
     & .2589672E-03,.9917704E-04,.4025904E-03,.4814459E-03,.4948753E-03,&
     & .4836881E-03,.4553829E-03,.4119222E-03,.3539441E-03,.2788595E-03,&
     & .1117935E-03,.4204129E-03,.5099243E-03,.5262683E-03,.5145801E-03,&
     & .4847600E-03,.4394867E-03,.3794179E-03,.3009917E-03,.1256044E-03,&
     & .2923436E-03,.3350198E-03,.3390235E-03,.3289566E-03,.3102593E-03,&
     & .2807257E-03,.2400716E-03,.1841474E-03,.6514967E-04,.3079436E-03,&
     & .3575834E-03,.3635135E-03,.3531948E-03,.3332576E-03,.3010207E-03,&
     & .2570614E-03,.1988560E-03,.7422428E-04,.3232435E-03,.3805545E-03,&
     & .3886434E-03,.3787663E-03,.3570394E-03,.3221858E-03,.2755102E-03,&
     & .2150719E-03,.8445731E-04,.3383325E-03,.4040182E-03,.4144616E-03,&
     & .4051543E-03,.3816543E-03,.3448001E-03,.2956857E-03,.2316464E-03,&
     & .9529395E-04,.3535369E-03,.4281879E-03,.4413816E-03,.4318028E-03,&
     & .4070455E-03,.3690091E-03,.3175244E-03,.2504699E-03,.1070304E-03/

      data   absa(316:450, 5)   /                                       &
     & .2453512E-03,.2800882E-03,.2825252E-03,.2730919E-03,.2566169E-03,&
     & .2318802E-03,.1983126E-03,.1522973E-03,.5516642E-04,.2581699E-03,&
     & .2989905E-03,.3031376E-03,.2937127E-03,.2763027E-03,.2495537E-03,&
     & .2126952E-03,.1646454E-03,.6300898E-04,.2707693E-03,.3182237E-03,&
     & .3242581E-03,.3153146E-03,.2972863E-03,.2680977E-03,.2289101E-03,&
     & .1781625E-03,.7205544E-04,.2834359E-03,.3379200E-03,.3459773E-03,&
     & .3376659E-03,.3181431E-03,.2874562E-03,.2463233E-03,.1919211E-03,&
     & .8179890E-04,.2961968E-03,.3584988E-03,.3687767E-03,.3605384E-03,&
     & .3397168E-03,.3080263E-03,.2651369E-03,.2080999E-03,.9227521E-04,&
     & .2054310E-03,.2336020E-03,.2346874E-03,.2262676E-03,.2118556E-03,&
     & .1911126E-03,.1628566E-03,.1251317E-03,.5265615E-04,.2158983E-03,&
     & .2491904E-03,.2518957E-03,.2435142E-03,.2286421E-03,.2061588E-03,&
     & .1755758E-03,.1356394E-03,.5986972E-04,.2262897E-03,.2650278E-03,&
     & .2694531E-03,.2615131E-03,.2463445E-03,.2220951E-03,.1892217E-03,&
     & .1469564E-03,.6762150E-04,.2368500E-03,.2815163E-03,.2876091E-03,&
     & .2802849E-03,.2643048E-03,.2385402E-03,.2043353E-03,.1585617E-03,&
     & .7608500E-04,.2475469E-03,.2987922E-03,.3069398E-03,.2998091E-03,&
     & .2825176E-03,.2560367E-03,.2200683E-03,.1724086E-03,.8542444E-04,&
     & .1724065E-03,.1956039E-03,.1956892E-03,.1881402E-03,.1756197E-03,&
     & .1579428E-03,.1342625E-03,.1031020E-03,.6401734E-04,.1809875E-03,&
     & .2084355E-03,.2098304E-03,.2024866E-03,.1896267E-03,.1710181E-03,&
     & .1452408E-03,.1119637E-03,.7219467E-04,.1895575E-03,.2214531E-03,&
     & .2244155E-03,.2174338E-03,.2045039E-03,.1844632E-03,.1570651E-03,&
     & .1214344E-03,.8097348E-04,.1983280E-03,.2351430E-03,.2396300E-03,&
     & .2331386E-03,.2197000E-03,.1984522E-03,.1696161E-03,.1313673E-03,&
     & .9012464E-04,.2073235E-03,.2496450E-03,.2559261E-03,.2497870E-03,&
     & .2354805E-03,.2133323E-03,.1830084E-03,.1431258E-03,.9946132E-04/

      data   absa(451:585, 5)   /                                       &
     & .1474531E-03,.1682762E-03,.1683793E-03,.1616613E-03,.1506254E-03,&
     & .1352532E-03,.1147895E-03,.8797863E-04,.6244135E-04,.1544785E-03,&
     & .1789686E-03,.1802145E-03,.1739110E-03,.1625411E-03,.1464475E-03,&
     & .1243412E-03,.9584180E-04,.7111930E-04,.1616375E-03,.1899537E-03,&
     & .1924972E-03,.1864886E-03,.1751900E-03,.1580145E-03,.1346723E-03,&
     & .1036729E-03,.7851804E-04,.1690889E-03,.2016073E-03,.2056052E-03,&
     & .2000516E-03,.1882465E-03,.1701850E-03,.1455146E-03,.1127735E-03,&
     & .8728573E-04,.1768424E-03,.2142601E-03,.2196970E-03,.2143761E-03,&
     & .2020742E-03,.1832716E-03,.1571724E-03,.1229344E-03,.9627397E-04,&
     & .1259909E-03,.1440828E-03,.1441655E-03,.1384718E-03,.1288147E-03,&
     & .1155429E-03,.9784118E-04,.7511447E-04,.5542102E-04,.1317506E-03,&
     & .1530088E-03,.1541902E-03,.1487423E-03,.1390011E-03,.1250725E-03,&
     & .1060741E-03,.8202844E-04,.6380787E-04,.1377366E-03,.1623281E-03,&
     & .1647109E-03,.1596366E-03,.1497584E-03,.1350689E-03,.1149756E-03,&
     & .8862399E-04,.7101181E-04,.1440326E-03,.1724497E-03,.1760537E-03,&
     & .1714078E-03,.1610900E-03,.1456787E-03,.1244204E-03,.9651500E-04,&
     & .7800418E-04,.1507641E-03,.1835052E-03,.1883108E-03,.1838011E-03,&
     & .1732376E-03,.1570042E-03,.1346136E-03,.1051140E-03,.8602649E-04,&
     & .1072501E-03,.1233653E-03,.1231303E-03,.1180489E-03,.1097712E-03,&
     & .9835244E-04,.8317681E-04,.6399482E-04,.4621221E-04,.1121185E-03,&
     & .1307584E-03,.1315510E-03,.1267530E-03,.1184196E-03,.1064469E-03,&
     & .9028401E-04,.6947188E-04,.5321306E-04,.1172881E-03,.1387038E-03,&
     & .1405630E-03,.1362742E-03,.1276910E-03,.1151082E-03,.9795080E-04,&
     & .7537889E-04,.5873871E-04,.1228106E-03,.1473997E-03,.1504434E-03,&
     & .1464443E-03,.1375481E-03,.1243899E-03,.1061458E-03,.8236870E-04,&
     & .6477734E-04,.1289014E-03,.1570159E-03,.1610714E-03,.1572623E-03,&
     & .1482077E-03,.1342864E-03,.1150946E-03,.8978401E-04,.7158089E-04/

      data   absa(  1:180, 6)   /                                       &
     & .3618426E-02,.3641770E-02,.3630477E-02,.3560259E-02,.3416323E-02,&
     & .3176047E-02,.2830864E-02,.2373983E-02,.1742217E-02,.3598081E-02,&
     & .3703258E-02,.3743793E-02,.3712218E-02,.3589778E-02,.3370737E-02,&
     & .3037900E-02,.2596162E-02,.1991398E-02,.3584083E-02,.3774324E-02,&
     & .3866788E-02,.3867624E-02,.3769761E-02,.3570448E-02,.3255037E-02,&
     & .2830185E-02,.2261505E-02,.3577298E-02,.3852926E-02,.3993744E-02,&
     & .4026505E-02,.3954894E-02,.3775808E-02,.3481871E-02,.3078695E-02,&
     & .2554303E-02,.3579835E-02,.3939097E-02,.4125092E-02,.4189431E-02,&
     & .4145315E-02,.3990234E-02,.3720434E-02,.3340287E-02,.2865995E-02,&
     & .3248394E-02,.3297351E-02,.3295356E-02,.3225177E-02,.3084542E-02,&
     & .2861202E-02,.2553948E-02,.2123300E-02,.1461572E-02,.3235583E-02,&
     & .3361291E-02,.3404532E-02,.3369041E-02,.3247031E-02,.3043618E-02,&
     & .2745319E-02,.2323230E-02,.1677107E-02,.3231488E-02,.3436144E-02,&
     & .3523056E-02,.3516391E-02,.3416868E-02,.3231131E-02,.2944369E-02,&
     & .2533343E-02,.1912294E-02,.3237872E-02,.3517820E-02,.3646256E-02,&
     & .3668618E-02,.3593949E-02,.3426895E-02,.3151603E-02,.2756947E-02,&
     & .2166557E-02,.3253132E-02,.3606661E-02,.3773434E-02,.3826500E-02,&
     & .3776601E-02,.3630583E-02,.3369967E-02,.2994163E-02,.2437904E-02,&
     & .2886205E-02,.2942447E-02,.2941032E-02,.2863367E-02,.2725094E-02,&
     & .2518760E-02,.2241939E-02,.1855734E-02,.1188420E-02,.2880736E-02,&
     & .3006642E-02,.3042538E-02,.2997000E-02,.2874870E-02,.2685370E-02,&
     & .2415558E-02,.2030078E-02,.1369150E-02,.2886508E-02,.3082007E-02,&
     & .3154756E-02,.3134344E-02,.3033232E-02,.2858755E-02,.2598268E-02,&
     & .2216396E-02,.1566621E-02,.2902943E-02,.3166339E-02,.3272849E-02,&
     & .3277933E-02,.3197754E-02,.3039090E-02,.2789791E-02,.2414716E-02,&
     & .1779400E-02,.2928623E-02,.3256730E-02,.3396972E-02,.3427808E-02,&
     & .3370428E-02,.3228051E-02,.2990498E-02,.2624579E-02,.2010439E-02,&
     & .2540902E-02,.2597254E-02,.2589360E-02,.2508382E-02,.2375438E-02,&
     & .2188928E-02,.1938803E-02,.1600312E-02,.9604968E-03,.2542011E-02,&
     & .2659586E-02,.2684388E-02,.2628952E-02,.2511349E-02,.2338234E-02,&
     & .2094432E-02,.1754453E-02,.1111332E-02,.2555832E-02,.2734832E-02,&
     & .2789039E-02,.2755882E-02,.2656417E-02,.2495319E-02,.2258255E-02,&
     & .1919151E-02,.1276371E-02,.2581259E-02,.2819185E-02,.2901637E-02,&
     & .2889786E-02,.2808889E-02,.2659442E-02,.2431576E-02,.2093659E-02,&
     & .1455003E-02,.2614546E-02,.2911442E-02,.3020621E-02,.3031206E-02,&
     & .2967469E-02,.2832291E-02,.2614781E-02,.2279966E-02,.1647719E-02/

      data   absa(181:315, 6)   /                                       &
     & .2219972E-02,.2275081E-02,.2258189E-02,.2178017E-02,.2052724E-02,&
     & .1883431E-02,.1661334E-02,.1366404E-02,.7744816E-03,.2227260E-02,&
     & .2335504E-02,.2345620E-02,.2284987E-02,.2174042E-02,.2016070E-02,&
     & .1799386E-02,.1503145E-02,.9005459E-03,.2248029E-02,.2409026E-02,&
     & .2443618E-02,.2400559E-02,.2304822E-02,.2156227E-02,.1945494E-02,&
     & .1648525E-02,.1039558E-02,.2279277E-02,.2492598E-02,.2549202E-02,&
     & .2524778E-02,.2443050E-02,.2304879E-02,.2101498E-02,.1803888E-02,&
     & .1189595E-02,.2314653E-02,.2584274E-02,.2663118E-02,.2656729E-02,&
     & .2589557E-02,.2462949E-02,.2266082E-02,.1970673E-02,.1351586E-02,&
     & .1926874E-02,.1977274E-02,.1952607E-02,.1874742E-02,.1757624E-02,&
     & .1604309E-02,.1409103E-02,.1152190E-02,.6162171E-03,.1939310E-02,&
     & .2034841E-02,.2032252E-02,.1970439E-02,.1864280E-02,.1719652E-02,&
     & .1530483E-02,.1272154E-02,.7211166E-03,.1964743E-02,.2105464E-02,&
     & .2122862E-02,.2074082E-02,.1979741E-02,.1844073E-02,.1659546E-02,&
     & .1399922E-02,.8374058E-03,.1996506E-02,.2186938E-02,.2222376E-02,&
     & .2187288E-02,.2103966E-02,.1977177E-02,.1797677E-02,.1537669E-02,&
     & .9633722E-03,.2032972E-02,.2273283E-02,.2330136E-02,.2309382E-02,&
     & .2237782E-02,.2120007E-02,.1943914E-02,.1683301E-02,.1099973E-02,&
     & .1663865E-02,.1707960E-02,.1678656E-02,.1605070E-02,.1496908E-02,&
     & .1359728E-02,.1186597E-02,.9637506E-03,.4837199E-03,.1679462E-02,&
     & .1762213E-02,.1749710E-02,.1688780E-02,.1589709E-02,.1459287E-02,&
     & .1291495E-02,.1067463E-02,.5692182E-03,.1705626E-02,.1829062E-02,&
     & .1833331E-02,.1782521E-02,.1691254E-02,.1568704E-02,.1405411E-02,&
     & .1178417E-02,.6634981E-03,.1737149E-02,.1904246E-02,.1926414E-02,&
     & .1885283E-02,.1802747E-02,.1686750E-02,.1526571E-02,.1298759E-02,&
     & .7673774E-03,.1774180E-02,.1984243E-02,.2025542E-02,.1997561E-02,&
     & .1924977E-02,.1813742E-02,.1655518E-02,.1426328E-02,.8813749E-03/

      data   absa(316:450, 6)   /                                       &
     & .1431715E-02,.1468745E-02,.1437010E-02,.1368212E-02,.1270165E-02,&
     & .1147038E-02,.9947349E-03,.8008574E-03,.3951453E-03,.1448558E-02,&
     & .1518890E-02,.1500478E-02,.1441411E-02,.1351002E-02,.1233436E-02,&
     & .1085630E-02,.8898114E-03,.4614759E-03,.1473293E-02,.1580156E-02,&
     & .1576676E-02,.1525332E-02,.1440556E-02,.1328775E-02,.1183815E-02,&
     & .9858304E-03,.5346545E-03,.1503819E-02,.1648007E-02,.1660678E-02,&
     & .1618866E-02,.1540977E-02,.1433665E-02,.1289444E-02,.1090469E-02,&
     & .6144704E-03,.1540870E-02,.1721704E-02,.1750156E-02,.1719303E-02,&
     & .1651341E-02,.1546906E-02,.1402428E-02,.1200202E-02,.7022918E-03,&
     & .1229566E-02,.1259757E-02,.1226783E-02,.1162063E-02,.1073623E-02,&
     & .9638267E-03,.8304962E-03,.6619654E-03,.3708964E-03,.1245677E-02,&
     & .1304848E-02,.1282510E-02,.1226105E-02,.1143293E-02,.1038470E-02,&
     & .9077754E-03,.7376031E-03,.4310410E-03,.1268666E-02,.1359022E-02,&
     & .1349687E-02,.1300919E-02,.1222896E-02,.1121551E-02,.9925926E-03,&
     & .8196689E-03,.4969197E-03,.1297981E-02,.1419642E-02,.1423650E-02,&
     & .1383567E-02,.1312384E-02,.1213886E-02,.1083936E-02,.9092041E-03,&
     & .5687180E-03,.1333382E-02,.1486555E-02,.1503607E-02,.1472905E-02,&
     & .1409828E-02,.1314124E-02,.1183141E-02,.1003566E-02,.6475735E-03,&
     & .1053427E-02,.1080875E-02,.1049457E-02,.9889791E-03,.9098076E-03,&
     & .8127675E-03,.6957123E-03,.5493509E-03,.3851070E-03,.1068299E-02,&
     & .1121177E-02,.1098614E-02,.1045290E-02,.9707095E-03,.8766135E-03,&
     & .7618622E-03,.6136871E-03,.4400535E-03,.1090018E-02,.1169156E-02,&
     & .1157141E-02,.1111356E-02,.1041184E-02,.9497327E-03,.8348360E-03,&
     & .6839361E-03,.4994168E-03,.1118137E-02,.1223841E-02,.1222357E-02,&
     & .1184095E-02,.1119621E-02,.1031067E-02,.9147253E-03,.7606201E-03,&
     & .5637416E-03,.1150470E-02,.1284547E-02,.1293867E-02,.1263779E-02,&
     & .1204832E-02,.1118950E-02,.1001368E-02,.8420516E-03,.6348671E-03/

      data   absa(451:585, 6)   /                                       &
     & .9046486E-03,.9397347E-03,.9133460E-03,.8617363E-03,.7934190E-03,&
     & .7090706E-03,.6066044E-03,.4794255E-03,.3743553E-03,.9212914E-03,&
     & .9782734E-03,.9602049E-03,.9151820E-03,.8507753E-03,.7678173E-03,&
     & .6660920E-03,.5356152E-03,.4226059E-03,.9442639E-03,.1023620E-02,&
     & .1014658E-02,.9758938E-03,.9151114E-03,.8345845E-03,.7318442E-03,&
     & .5984128E-03,.4774443E-03,.9713828E-03,.1075284E-02,.1075663E-02,&
     & .1042999E-02,.9862816E-03,.9076184E-03,.8041084E-03,.6658818E-03,&
     & .5353482E-03,.1002245E-02,.1131170E-02,.1142136E-02,.1116667E-02,&
     & .1063668E-02,.9860813E-03,.8820903E-03,.7392501E-03,.5986885E-03,&
     & .7757001E-03,.8156004E-03,.7935608E-03,.7501868E-03,.6919875E-03,&
     & .6187418E-03,.5293689E-03,.4171913E-03,.3400983E-03,.7936331E-03,&
     & .8519754E-03,.8374063E-03,.7995658E-03,.7443953E-03,.6725458E-03,&
     & .5825365E-03,.4667964E-03,.3809920E-03,.8158134E-03,.8946178E-03,&
     & .8881924E-03,.8553581E-03,.8027948E-03,.7321369E-03,.6414490E-03,&
     & .5225120E-03,.4287453E-03,.8413662E-03,.9418592E-03,.9443588E-03,&
     & .9171257E-03,.8676347E-03,.7974095E-03,.7054434E-03,.5826005E-03,&
     & .4804212E-03,.8686293E-03,.9928428E-03,.1005484E-02,.9842193E-03,&
     & .9373691E-03,.8680623E-03,.7750971E-03,.6486142E-03,.5361541E-03,&
     & .6656671E-03,.7063220E-03,.6882893E-03,.6520290E-03,.6021008E-03,&
     & .5393669E-03,.4616943E-03,.3624013E-03,.2876110E-03,.6827709E-03,&
     & .7403802E-03,.7293362E-03,.6972791E-03,.6496824E-03,.5872587E-03,&
     & .5091135E-03,.4069669E-03,.3235500E-03,.7030563E-03,.7795754E-03,&
     & .7759040E-03,.7481102E-03,.7027456E-03,.6406035E-03,.5612231E-03,&
     & .4560596E-03,.3644783E-03,.7249585E-03,.8222986E-03,.8269765E-03,&
     & .8039916E-03,.7610904E-03,.6992438E-03,.6179241E-03,.5093407E-03,&
     & .4084015E-03,.7487037E-03,.8678454E-03,.8821100E-03,.8646953E-03,&
     & .8237630E-03,.7629048E-03,.6800190E-03,.5680363E-03,.4564484E-03/

      data   absa(  1:180, 7)   /                                       &
     & .1944086E-01,.1794384E-01,.1665104E-01,.1551773E-01,.1445828E-01,&
     & .1338923E-01,.1239141E-01,.1193783E-01,.1267749E-01,.1929939E-01,&
     & .1802401E-01,.1696529E-01,.1609074E-01,.1524109E-01,.1436017E-01,&
     & .1365895E-01,.1355010E-01,.1460849E-01,.1914240E-01,.1812275E-01,&
     & .1733603E-01,.1672488E-01,.1609343E-01,.1543656E-01,.1506976E-01,&
     & .1532896E-01,.1671755E-01,.1896990E-01,.1824026E-01,.1775719E-01,&
     & .1740823E-01,.1702225E-01,.1660281E-01,.1659697E-01,.1723872E-01,&
     & .1897711E-01,.1878230E-01,.1836854E-01,.1822350E-01,.1813959E-01,&
     & .1798727E-01,.1786818E-01,.1824983E-01,.1930509E-01,.2141343E-01,&
     & .2002304E-01,.1846205E-01,.1712703E-01,.1593023E-01,.1477768E-01,&
     & .1360259E-01,.1233974E-01,.1147779E-01,.1184062E-01,.1987095E-01,&
     & .1854946E-01,.1743597E-01,.1649501E-01,.1555867E-01,.1455749E-01,&
     & .1351955E-01,.1298854E-01,.1367056E-01,.1970539E-01,.1864562E-01,&
     & .1779988E-01,.1712573E-01,.1640112E-01,.1560634E-01,.1483953E-01,&
     & .1466071E-01,.1566922E-01,.1952211E-01,.1876024E-01,.1821996E-01,&
     & .1780079E-01,.1732241E-01,.1670897E-01,.1627296E-01,.1646775E-01,&
     & .1781807E-01,.1932576E-01,.1888733E-01,.1868624E-01,.1852437E-01,&
     & .1828787E-01,.1789646E-01,.1782466E-01,.1840912E-01,.2012300E-01,&
     & .2046316E-01,.1881445E-01,.1742883E-01,.1614035E-01,.1486365E-01,&
     & .1356768E-01,.1209778E-01,.1077331E-01,.1065343E-01,.2030717E-01,&
     & .1890498E-01,.1772206E-01,.1668034E-01,.1562712E-01,.1448379E-01,&
     & .1318643E-01,.1216891E-01,.1235738E-01,.2013274E-01,.1900626E-01,&
     & .1806511E-01,.1729725E-01,.1644190E-01,.1549571E-01,.1438909E-01,&
     & .1370013E-01,.1420354E-01,.1994378E-01,.1911368E-01,.1847290E-01,&
     & .1795539E-01,.1733637E-01,.1657610E-01,.1569548E-01,.1535991E-01,&
     & .1619433E-01,.1974273E-01,.1923315E-01,.1892487E-01,.1866106E-01,&
     & .1828690E-01,.1769725E-01,.1711524E-01,.1715953E-01,.1834492E-01,&
     & .2071926E-01,.1899869E-01,.1758640E-01,.1621993E-01,.1481733E-01,&
     & .1340442E-01,.1178457E-01,.1005441E-01,.9415201E-02,.2056070E-01,&
     & .1908468E-01,.1786585E-01,.1672760E-01,.1555529E-01,.1427366E-01,&
     & .1280976E-01,.1132568E-01,.1098244E-01,.2038159E-01,.1918328E-01,&
     & .1818981E-01,.1732353E-01,.1633951E-01,.1523974E-01,.1393712E-01,&
     & .1273530E-01,.1270208E-01,.2019166E-01,.1930154E-01,.1857522E-01,&
     & .1795884E-01,.1719737E-01,.1628902E-01,.1513446E-01,.1425698E-01,&
     & .1454259E-01,.1999244E-01,.1941900E-01,.1900880E-01,.1863941E-01,&
     & .1812618E-01,.1738164E-01,.1642114E-01,.1589227E-01,.1651351E-01/

      data   absa(181:315, 7)   /                                       &
     & .2078596E-01,.1902221E-01,.1759185E-01,.1618479E-01,.1468286E-01,&
     & .1316356E-01,.1141915E-01,.9423027E-02,.8269424E-02,.2061989E-01,&
     & .1909317E-01,.1787631E-01,.1666093E-01,.1538845E-01,.1399494E-01,&
     & .1238597E-01,.1057539E-01,.9705846E-02,.2044247E-01,.1918749E-01,&
     & .1817839E-01,.1723098E-01,.1614137E-01,.1490895E-01,.1346159E-01,&
     & .1185395E-01,.1127991E-01,.2025356E-01,.1930375E-01,.1854460E-01,&
     & .1784554E-01,.1696147E-01,.1591307E-01,.1460469E-01,.1324520E-01,&
     & .1297427E-01,.2007410E-01,.1943195E-01,.1896380E-01,.1849843E-01,&
     & .1785367E-01,.1697392E-01,.1579435E-01,.1474526E-01,.1480343E-01,&
     & .2067653E-01,.1889520E-01,.1744869E-01,.1600835E-01,.1443542E-01,&
     & .1280811E-01,.1096768E-01,.8811489E-02,.7161785E-02,.2051191E-01,&
     & .1895371E-01,.1772735E-01,.1645848E-01,.1509553E-01,.1360134E-01,&
     & .1187179E-01,.9849619E-02,.8463973E-02,.2033498E-01,.1904213E-01,&
     & .1803233E-01,.1698898E-01,.1581982E-01,.1446262E-01,.1288709E-01,&
     & .1101445E-01,.9901719E-02,.2016531E-01,.1915657E-01,.1837300E-01,&
     & .1758375E-01,.1659315E-01,.1541743E-01,.1398290E-01,.1227676E-01,&
     & .1145224E-01,.2000011E-01,.1929718E-01,.1877422E-01,.1820930E-01,&
     & .1743917E-01,.1643489E-01,.1512599E-01,.1364249E-01,.1312614E-01,&
     & .2038344E-01,.1861656E-01,.1717360E-01,.1569740E-01,.1410542E-01,&
     & .1238154E-01,.1046720E-01,.8234826E-02,.6157987E-02,.2021852E-01,&
     & .1866243E-01,.1743952E-01,.1614631E-01,.1471423E-01,.1312828E-01,&
     & .1131255E-01,.9181663E-02,.7325752E-02,.2004782E-01,.1874347E-01,&
     & .1774105E-01,.1664594E-01,.1539580E-01,.1393665E-01,.1225989E-01,&
     & .1024134E-01,.8627463E-02,.1988902E-01,.1886277E-01,.1808084E-01,&
     & .1721183E-01,.1612179E-01,.1483288E-01,.1330233E-01,.1139021E-01,&
     & .1003993E-01,.1973638E-01,.1901550E-01,.1847625E-01,.1780825E-01,&
     & .1691910E-01,.1580549E-01,.1439570E-01,.1263517E-01,.1157007E-01/

      data   absa(316:450, 7)   /                                       &
     & .1987279E-01,.1815516E-01,.1673357E-01,.1526588E-01,.1367570E-01,&
     & .1190873E-01,.9931491E-02,.7668521E-02,.5208898E-02,.1970849E-01,&
     & .1819142E-01,.1699015E-01,.1570273E-01,.1425525E-01,.1259893E-01,&
     & .1071895E-01,.8538062E-02,.6255037E-02,.1954825E-01,.1826821E-01,&
     & .1728456E-01,.1619427E-01,.1489632E-01,.1335233E-01,.1159816E-01,&
     & .9521153E-02,.7436933E-02,.1939994E-01,.1839394E-01,.1763047E-01,&
     & .1674380E-01,.1557766E-01,.1418956E-01,.1258113E-01,.1057697E-01,&
     & .8728022E-02,.1926214E-01,.1855724E-01,.1803115E-01,.1732234E-01,&
     & .1632534E-01,.1511381E-01,.1362321E-01,.1171409E-01,.1013851E-01,&
     & .1916880E-01,.1752174E-01,.1613409E-01,.1469612E-01,.1312851E-01,&
     & .1136007E-01,.9364186E-02,.7102147E-02,.4213687E-02,.1900554E-01,&
     & .1754861E-01,.1637886E-01,.1511063E-01,.1368165E-01,.1201315E-01,&
     & .1009580E-01,.7893303E-02,.5110717E-02,.1885316E-01,.1762136E-01,&
     & .1666679E-01,.1558750E-01,.1430162E-01,.1272233E-01,.1090848E-01,&
     & .8800048E-02,.6141449E-02,.1871378E-01,.1775166E-01,.1701481E-01,&
     & .1613266E-01,.1495234E-01,.1350195E-01,.1182791E-01,.9791825E-02,&
     & .7281500E-02,.1858911E-01,.1792329E-01,.1741640E-01,.1670595E-01,&
     & .1566347E-01,.1437003E-01,.1281383E-01,.1084750E-01,.8535050E-02,&
     & .1826944E-01,.1672464E-01,.1541276E-01,.1403848E-01,.1252666E-01,&
     & .1078514E-01,.8823245E-02,.6591625E-02,.4177613E-02,.1810805E-01,&
     & .1674909E-01,.1564423E-01,.1443382E-01,.1304370E-01,.1140669E-01,&
     & .9515283E-02,.7318294E-02,.4841751E-02,.1796423E-01,.1682298E-01,&
     & .1593052E-01,.1489393E-01,.1363432E-01,.1209430E-01,.1027133E-01,&
     & .8153051E-02,.5674979E-02,.1783274E-01,.1695370E-01,.1627489E-01,&
     & .1542379E-01,.1427771E-01,.1283692E-01,.1112848E-01,.9081882E-02,&
     & .6511275E-02,.1772600E-01,.1713577E-01,.1668031E-01,.1599272E-01,&
     & .1497596E-01,.1366160E-01,.1205854E-01,.1007221E-01,.7398941E-02/

      data   absa(451:585, 7)   /                                       &
     & .1712738E-01,.1578625E-01,.1465104E-01,.1343124E-01,.1204735E-01,&
     & .1040471E-01,.8547650E-02,.6380408E-02,.4682263E-02,.1697749E-01,&
     & .1582950E-01,.1490046E-01,.1383672E-01,.1256494E-01,.1102596E-01,&
     & .9230857E-02,.7091836E-02,.5424321E-02,.1684963E-01,.1592973E-01,&
     & .1520871E-01,.1431627E-01,.1316095E-01,.1170953E-01,.9981803E-02,&
     & .7904548E-02,.6291977E-02,.1674859E-01,.1609441E-01,.1557937E-01,&
     & .1485696E-01,.1380508E-01,.1246505E-01,.1081864E-01,.8796469E-02,&
     & .7234547E-02,.1667150E-01,.1630711E-01,.1601588E-01,.1543527E-01,&
     & .1451902E-01,.1330051E-01,.1171274E-01,.9750545E-02,.8163171E-02,&
     & .1585812E-01,.1473647E-01,.1378249E-01,.1272207E-01,.1146005E-01,&
     & .9946309E-02,.8216259E-02,.6139489E-02,.4795601E-02,.1573042E-01,&
     & .1480849E-01,.1405564E-01,.1313767E-01,.1198402E-01,.1056976E-01,&
     & .8883654E-02,.6832310E-02,.5569244E-02,.1562948E-01,.1493848E-01,&
     & .1438406E-01,.1362526E-01,.1258929E-01,.1125212E-01,.9629528E-02,&
     & .7616188E-02,.6452525E-02,.1556093E-01,.1513902E-01,.1478476E-01,&
     & .1418150E-01,.1324034E-01,.1201861E-01,.1045886E-01,.8467010E-02,&
     & .7448495E-02,.1552707E-01,.1538861E-01,.1525660E-01,.1477573E-01,&
     & .1397485E-01,.1285192E-01,.1132600E-01,.9383049E-02,.8380165E-02,&
     & .1451815E-01,.1362323E-01,.1284326E-01,.1193288E-01,.1079141E-01,&
     & .9422285E-02,.7828749E-02,.5876279E-02,.4564233E-02,.1441454E-01,&
     & .1372037E-01,.1313620E-01,.1235490E-01,.1132640E-01,.1004968E-01,&
     & .8482915E-02,.6542489E-02,.5277165E-02,.1434814E-01,.1388758E-01,&
     & .1349254E-01,.1286003E-01,.1193607E-01,.1073461E-01,.9223888E-02,&
     & .7293736E-02,.6181241E-02,.1432181E-01,.1412637E-01,.1392568E-01,&
     & .1342872E-01,.1260305E-01,.1150145E-01,.1004198E-01,.8104002E-02,&
     & .7114973E-02,.1432961E-01,.1441453E-01,.1442859E-01,.1404464E-01,&
     & .1335348E-01,.1232881E-01,.1089297E-01,.8984991E-02,.8052181E-02/

      data   absa(  1:180, 8)   /                                       &
     & .7758517E-01,.6971744E-01,.6856234E-01,.7085104E-01,.7686968E-01,&
     & .8798079E-01,.1038826E+00,.1198350E+00,.1359548E+00,.7760111E-01,&
     & .7121281E-01,.7343034E-01,.7934822E-01,.9017738E-01,.1066736E+00,&
     & .1269235E+00,.1465907E+00,.1664556E+00,.7752743E-01,.7302330E-01,&
     & .7902576E-01,.8954413E-01,.1059545E+00,.1282979E+00,.1531287E+00,&
     & .1770168E+00,.2010905E+00,.7732969E-01,.7518701E-01,.8549302E-01,&
     & .1015046E+00,.1240275E+00,.1528286E+00,.1826322E+00,.2112286E+00,&
     & .2400654E+00,.7704248E-01,.7773694E-01,.9279924E-01,.1151544E+00,&
     & .1447190E+00,.1798906E+00,.2150779E+00,.2488116E+00,.2828065E+00,&
     & .8996337E-01,.8019925E-01,.7649483E-01,.7619075E-01,.7950947E-01,&
     & .8744378E-01,.1013048E+00,.1167488E+00,.1333495E+00,.9002593E-01,&
     & .8150228E-01,.8115314E-01,.8427387E-01,.9208357E-01,.1052256E+00,&
     & .1239442E+00,.1430385E+00,.1634839E+00,.8993043E-01,.8319207E-01,&
     & .8655274E-01,.9399363E-01,.1071792E+00,.1260258E+00,.1498926E+00,&
     & .1732190E+00,.1980122E+00,.8971818E-01,.8524906E-01,.9277644E-01,&
     & .1055693E+00,.1245408E+00,.1502414E+00,.1793335E+00,.2072943E+00,&
     & .2370216E+00,.8937589E-01,.8770760E-01,.9989309E-01,.1188472E+00,&
     & .1444723E+00,.1773019E+00,.2118836E+00,.2451113E+00,.2803148E+00,&
     & .1035251E+00,.9170912E-01,.8469424E-01,.8124545E-01,.8127330E-01,&
     & .8520134E-01,.9536063E-01,.1094892E+00,.1259212E+00,.1036314E+00,&
     & .9276163E-01,.8901665E-01,.8866608E-01,.9260241E-01,.1015125E+00,&
     & .1166863E+00,.1344219E+00,.1546172E+00,.1035650E+00,.9415806E-01,&
     & .9411336E-01,.9760915E-01,.1066039E+00,.1208833E+00,.1416737E+00,&
     & .1635528E+00,.1881599E+00,.1033569E+00,.9604690E-01,.9993466E-01,&
     & .1084464E+00,.1229492E+00,.1435592E+00,.1703234E+00,.1968385E+00,&
     & .2265370E+00,.1029798E+00,.9835979E-01,.1066697E+00,.1210700E+00,&
     & .1417416E+00,.1696560E+00,.2023878E+00,.2340132E+00,.2693038E+00,&
     & .1182256E+00,.1042918E+00,.9385612E-01,.8708706E-01,.8392123E-01,&
     & .8386827E-01,.8993733E-01,.1021139E+00,.1178700E+00,.1183624E+00,&
     & .1051802E+00,.9770525E-01,.9387938E-01,.9397963E-01,.9866810E-01,&
     & .1093764E+00,.1253639E+00,.1447078E+00,.1183448E+00,.1063559E+00,&
     & .1024326E+00,.1020095E+00,.1067911E+00,.1165316E+00,.1326917E+00,&
     & .1528199E+00,.1764343E+00,.1181015E+00,.1078331E+00,.1078798E+00,&
     & .1120521E+00,.1219908E+00,.1374912E+00,.1601454E+00,.1848559E+00,&
     & .2134132E+00,.1177256E+00,.1098961E+00,.1142273E+00,.1239094E+00,&
     & .1395842E+00,.1620158E+00,.1913771E+00,.2212322E+00,.2554318E+00/

      data   absa(181:315, 8)   /                                       &
     & .1337645E+00,.1176975E+00,.1042174E+00,.9408829E-01,.8780506E-01,&
     & .8428291E-01,.8648117E-01,.9578625E-01,.1107310E+00,.1340095E+00,&
     & .1185612E+00,.1073992E+00,.1002667E+00,.9676787E-01,.9754447E-01,&
     & .1041226E+00,.1173533E+00,.1356505E+00,.1339933E+00,.1195717E+00,&
     & .1117737E+00,.1076768E+00,.1084293E+00,.1139755E+00,.1254459E+00,&
     & .1433024E+00,.1656569E+00,.1337898E+00,.1208507E+00,.1168604E+00,&
     & .1168941E+00,.1225882E+00,.1334602E+00,.1509971E+00,.1738575E+00,&
     & .2010332E+00,.1333402E+00,.1225783E+00,.1227702E+00,.1280534E+00,&
     & .1390939E+00,.1563754E+00,.1810270E+00,.2089873E+00,.2415778E+00,&
     & .1499294E+00,.1316511E+00,.1153149E+00,.1019770E+00,.9253012E-01,&
     & .8589064E-01,.8430058E-01,.8998453E-01,.1039889E+00,.1502123E+00,&
     & .1324781E+00,.1179942E+00,.1074361E+00,.1005474E+00,.9758469E-01,&
     & .1000820E+00,.1097714E+00,.1269049E+00,.1502509E+00,.1333737E+00,&
     & .1216948E+00,.1142169E+00,.1109667E+00,.1125131E+00,.1194607E+00,&
     & .1338501E+00,.1548471E+00,.1500164E+00,.1344800E+00,.1264420E+00,&
     & .1225692E+00,.1240810E+00,.1304661E+00,.1429117E+00,.1627629E+00,&
     & .1883115E+00,.1495747E+00,.1359816E+00,.1319394E+00,.1329504E+00,&
     & .1395630E+00,.1517775E+00,.1709091E+00,.1964203E+00,.2271878E+00,&
     & .1667524E+00,.1461966E+00,.1270922E+00,.1109576E+00,.9817555E-01,&
     & .8876777E-01,.8374701E-01,.8543513E-01,.9818294E-01,.1671368E+00,&
     & .1470313E+00,.1293919E+00,.1154650E+00,.1053506E+00,.9907389E-01,&
     & .9777459E-01,.1033433E+00,.1192365E+00,.1672667E+00,.1478601E+00,&
     & .1325480E+00,.1215247E+00,.1147604E+00,.1125732E+00,.1153357E+00,&
     & .1255724E+00,.1452073E+00,.1670804E+00,.1488123E+00,.1366626E+00,&
     & .1291483E+00,.1268401E+00,.1291153E+00,.1368015E+00,.1526760E+00,&
     & .1766400E+00,.1666531E+00,.1500830E+00,.1416654E+00,.1387987E+00,&
     & .1413161E+00,.1488921E+00,.1627763E+00,.1846442E+00,.2135966E+00/

      data   absa(316:450, 8)   /                                       &
     & .1846850E+00,.1617527E+00,.1399797E+00,.1208986E+00,.1049700E+00,&
     & .9268890E-01,.8468015E-01,.8250239E-01,.9346009E-01,.1852004E+00,&
     & .1626022E+00,.1419177E+00,.1246679E+00,.1111616E+00,.1018673E+00,&
     & .9711144E-01,.9858998E-01,.1128165E+00,.1854002E+00,.1633815E+00,&
     & .1446073E+00,.1298548E+00,.1195851E+00,.1140911E+00,.1130781E+00,&
     & .1187616E+00,.1368729E+00,.1852584E+00,.1641980E+00,.1481385E+00,&
     & .1365970E+00,.1306723E+00,.1293064E+00,.1327425E+00,.1439504E+00,&
     & .1663864E+00,.1848096E+00,.1652487E+00,.1525559E+00,.1454722E+00,&
     & .1442002E+00,.1476584E+00,.1567932E+00,.1741988E+00,.2014170E+00,&
     & .2035660E+00,.1782138E+00,.1538641E+00,.1319292E+00,.1129603E+00,&
     & .9769919E-01,.8663861E-01,.8077073E-01,.8932903E-01,.2042261E+00,&
     & .1790867E+00,.1555132E+00,.1351430E+00,.1182459E+00,.1056510E+00,&
     & .9764716E-01,.9524493E-01,.1071831E+00,.2045296E+00,.1798786E+00,&
     & .1578247E+00,.1395652E+00,.1254969E+00,.1165399E+00,.1121007E+00,&
     & .1135498E+00,.1294885E+00,.2044541E+00,.1805616E+00,.1608548E+00,&
     & .1453947E+00,.1354455E+00,.1305073E+00,.1300663E+00,.1365302E+00,&
     & .1570562E+00,.2040215E+00,.1814068E+00,.1647421E+00,.1533244E+00,&
     & .1479662E+00,.1475364E+00,.1523061E+00,.1647332E+00,.1901427E+00,&
     & .2235492E+00,.1956768E+00,.1686716E+00,.1440048E+00,.1221516E+00,&
     & .1041925E+00,.9012153E-01,.8095605E-01,.7543929E-01,.2243439E+00,&
     & .1965703E+00,.1701921E+00,.1468360E+00,.1269493E+00,.1112681E+00,&
     & .9999583E-01,.9424336E-01,.9351393E-01,.2246860E+00,.1973265E+00,&
     & .1722057E+00,.1507546E+00,.1334737E+00,.1209496E+00,.1133134E+00,&
     & .1112100E+00,.1153114E+00,.2246584E+00,.1979469E+00,.1749030E+00,&
     & .1560431E+00,.1423719E+00,.1337802E+00,.1300644E+00,.1325688E+00,&
     & .1435519E+00,.2242020E+00,.1986013E+00,.1783455E+00,.1632177E+00,&
     & .1538706E+00,.1497467E+00,.1509387E+00,.1590812E+00,.1779531E+00/

      data   absa(451:585, 8)   /                                       &
     & .2450204E+00,.2144676E+00,.1850038E+00,.1579688E+00,.1339651E+00,&
     & .1141008E+00,.9793249E-01,.8695958E-01,.6939988E-01,.2456964E+00,&
     & .2153041E+00,.1865121E+00,.1608968E+00,.1389868E+00,.1214503E+00,&
     & .1082108E+00,.1007628E+00,.8668012E-01,.2459004E+00,.2159604E+00,&
     & .1885239E+00,.1648976E+00,.1457743E+00,.1314989E+00,.1219175E+00,&
     & .1183238E+00,.1082310E+00,.2456185E+00,.2163907E+00,.1911660E+00,&
     & .1703594E+00,.1550454E+00,.1445551E+00,.1392768E+00,.1404833E+00,&
     & .1353141E+00,.2449307E+00,.2169326E+00,.1945158E+00,.1777645E+00,&
     & .1667152E+00,.1608041E+00,.1609850E+00,.1678148E+00,.1694632E+00,&
     & .2671603E+00,.2338498E+00,.2019245E+00,.1725008E+00,.1464684E+00,&
     & .1244209E+00,.1059693E+00,.9304783E-01,.6959000E-01,.2676057E+00,&
     & .2345222E+00,.2033654E+00,.1755662E+00,.1517012E+00,.1320205E+00,&
     & .1167556E+00,.1073595E+00,.8673211E-01,.2675630E+00,.2350038E+00,&
     & .2053552E+00,.1797362E+00,.1586943E+00,.1424342E+00,.1308446E+00,&
     & .1256134E+00,.1083518E+00,.2670071E+00,.2352413E+00,.2078987E+00,&
     & .1853108E+00,.1681964E+00,.1557131E+00,.1486669E+00,.1486318E+00,&
     & .1350793E+00,.2659779E+00,.2356017E+00,.2111172E+00,.1928578E+00,&
     & .1799895E+00,.1723719E+00,.1711124E+00,.1767808E+00,.1695167E+00,&
     & .2893519E+00,.2532903E+00,.2190193E+00,.1873805E+00,.1594254E+00,&
     & .1350214E+00,.1142355E+00,.9903594E-01,.7490192E-01,.2895758E+00,&
     & .2538124E+00,.2204052E+00,.1906072E+00,.1647749E+00,.1428610E+00,&
     & .1254558E+00,.1139403E+00,.9307717E-01,.2892263E+00,.2540433E+00,&
     & .2223015E+00,.1948310E+00,.1719923E+00,.1535461E+00,.1399201E+00,&
     & .1328863E+00,.1147608E+00,.2883124E+00,.2540521E+00,.2247025E+00,&
     & .2005493E+00,.1816141E+00,.1671436E+00,.1582248E+00,.1567001E+00,&
     & .1426012E+00,.2869377E+00,.2542235E+00,.2278499E+00,.2081519E+00,&
     & .1935477E+00,.1842067E+00,.1812086E+00,.1856280E+00,.1773106E+00/


      data   absb(  1:120, 1)   /                                       &
     & .8084900E-06,.8136200E-06,.8139100E-06,.8183200E-06,.8100000E-06,&
     & .6259300E-06,.6309000E-06,.6315800E-06,.6282100E-06,.6250200E-06,&
     & .4782500E-06,.4785200E-06,.4771300E-06,.4768800E-06,.4798900E-06,&
     & .3603900E-06,.3590500E-06,.3606400E-06,.3629800E-06,.3686000E-06,&
     & .2767900E-06,.2760800E-06,.2774200E-06,.2811200E-06,.2875900E-06,&
     & .2177100E-06,.2182600E-06,.2201900E-06,.2242900E-06,.2299600E-06,&
     & .1733400E-06,.1744100E-06,.1768100E-06,.1805500E-06,.1853700E-06,&
     & .1397400E-06,.1407600E-06,.1432600E-06,.1469400E-06,.1507000E-06,&
     & .1128900E-06,.1139400E-06,.1163100E-06,.1196300E-06,.1228400E-06,&
     & .9057400E-07,.9191400E-07,.9404600E-07,.9699800E-07,.9984100E-07,&
     & .7248300E-07,.7406900E-07,.7618600E-07,.7848200E-07,.8096400E-07,&
     & .5746300E-07,.5913000E-07,.6080200E-07,.6290300E-07,.6521800E-07,&
     & .4576700E-07,.4718400E-07,.4869600E-07,.5049400E-07,.5252700E-07,&
     & .3663600E-07,.3779400E-07,.3921700E-07,.4076800E-07,.4246500E-07,&
     & .2917300E-07,.3018300E-07,.3146900E-07,.3283400E-07,.3421300E-07,&
     & .2322500E-07,.2413700E-07,.2516700E-07,.2637800E-07,.2753400E-07,&
     & .1867500E-07,.1947200E-07,.2034000E-07,.2134000E-07,.2228600E-07,&
     & .1502000E-07,.1569000E-07,.1646800E-07,.1725800E-07,.1806000E-07,&
     & .1214900E-07,.1272800E-07,.1337500E-07,.1403500E-07,.1471100E-07,&
     & .9859600E-08,.1034400E-07,.1087900E-07,.1142700E-07,.1200300E-07,&
     & .8012700E-08,.8419900E-08,.8861000E-08,.9318800E-08,.9810200E-08,&
     & .6547300E-08,.6885000E-08,.7247000E-08,.7637900E-08,.8032900E-08,&
     & .5310700E-08,.5593000E-08,.5898400E-08,.6216100E-08,.6551100E-08,&
     & .4274900E-08,.4507200E-08,.4759200E-08,.5024400E-08,.5303400E-08/

      data   absb(121:235, 1)   /                                       &
     & .3459300E-08,.3647900E-08,.3855700E-08,.4071400E-08,.4303100E-08,&
     & .2798100E-08,.2952500E-08,.3121200E-08,.3298400E-08,.3489000E-08,&
     & .2261600E-08,.2387900E-08,.2526600E-08,.2671600E-08,.2829500E-08,&
     & .1823200E-08,.1924500E-08,.2036200E-08,.2155400E-08,.2283100E-08,&
     & .1467900E-08,.1551100E-08,.1640200E-08,.1737600E-08,.1841300E-08,&
     & .1182800E-08,.1248800E-08,.1320000E-08,.1400800E-08,.1484100E-08,&
     & .9497000E-09,.1002800E-08,.1061000E-08,.1125400E-08,.1193300E-08,&
     & .7621000E-09,.8040100E-09,.8509800E-09,.9020300E-09,.9579400E-09,&
     & .6124200E-09,.6447100E-09,.6820500E-09,.7233600E-09,.7684600E-09,&
     & .4921400E-09,.5164200E-09,.5463300E-09,.5791600E-09,.6150600E-09,&
     & .3955200E-09,.4134300E-09,.4365800E-09,.4626600E-09,.4912100E-09,&
     & .3177300E-09,.3316100E-09,.3487800E-09,.3697000E-09,.3923100E-09,&
     & .2552600E-09,.2661300E-09,.2790600E-09,.2951000E-09,.3134600E-09,&
     & .2057100E-09,.2141900E-09,.2237900E-09,.2362200E-09,.2507600E-09,&
     & .1657300E-09,.1725100E-09,.1800200E-09,.1892500E-09,.2007400E-09,&
     & .1331200E-09,.1388800E-09,.1448000E-09,.1517000E-09,.1606700E-09,&
     & .1071700E-09,.1120000E-09,.1165300E-09,.1218700E-09,.1286400E-09,&
     & .8539500E-10,.9019000E-10,.9397800E-10,.9811500E-10,.1032400E-09,&
     & .6793300E-10,.7268200E-10,.7585700E-10,.7907900E-10,.8294200E-10,&
     & .5357200E-10,.5809500E-10,.6123100E-10,.6373800E-10,.6673800E-10,&
     & .4172900E-10,.4638000E-10,.4935700E-10,.5140900E-10,.5374300E-10,&
     & .3231900E-10,.3673700E-10,.3957200E-10,.4148700E-10,.4332700E-10,&
     & .2582000E-10,.2954100E-10,.3210700E-10,.3376200E-10,.3525000E-10/

      data   absb(  1:120, 2)   /                                       &
     & .9380900E-05,.1053600E-04,.1177700E-04,.1309800E-04,.1448600E-04,&
     & .7705000E-05,.8652000E-05,.9674500E-05,.1075900E-04,.1189100E-04,&
     & .6311200E-05,.7091400E-05,.7932000E-05,.8817900E-05,.9734400E-05,&
     & .5167200E-05,.5817200E-05,.6496100E-05,.7213300E-05,.7959500E-05,&
     & .4247000E-05,.4781700E-05,.5339600E-05,.5922900E-05,.6538500E-05,&
     & .3510500E-05,.3951300E-05,.4404200E-05,.4886500E-05,.5398300E-05,&
     & .2905700E-05,.3268000E-05,.3640900E-05,.4041900E-05,.4464600E-05,&
     & .2413200E-05,.2711100E-05,.3019200E-05,.3349800E-05,.3698500E-05,&
     & .2004100E-05,.2248400E-05,.2505200E-05,.2777000E-05,.3066100E-05,&
     & .1676200E-05,.1877500E-05,.2090000E-05,.2315800E-05,.2554300E-05,&
     & .1401300E-05,.1566500E-05,.1743300E-05,.1930800E-05,.2129000E-05,&
     & .1168000E-05,.1303900E-05,.1451500E-05,.1608500E-05,.1773300E-05,&
     & .9749100E-06,.1088600E-05,.1211400E-05,.1342800E-05,.1480600E-05,&
     & .8160800E-06,.9119800E-06,.1014800E-05,.1124400E-05,.1239200E-05,&
     & .6840700E-06,.7643200E-06,.8504800E-06,.9421000E-06,.1039300E-05,&
     & .5734000E-06,.6407100E-06,.7131100E-06,.7898800E-06,.8730400E-06,&
     & .4823400E-06,.5388600E-06,.5999200E-06,.6655700E-06,.7356500E-06,&
     & .4062800E-06,.4538400E-06,.5063800E-06,.5617500E-06,.6208600E-06,&
     & .3427600E-06,.3840900E-06,.4281700E-06,.4755100E-06,.5250800E-06,&
     & .2902400E-06,.3254800E-06,.3629100E-06,.4026300E-06,.4442300E-06,&
     & .2462300E-06,.2759600E-06,.3074800E-06,.3410400E-06,.3757100E-06,&
     & .2081400E-06,.2331800E-06,.2597500E-06,.2874600E-06,.3167400E-06,&
     & .1739900E-06,.1947600E-06,.2169800E-06,.2402100E-06,.2646300E-06,&
     & .1435200E-06,.1607900E-06,.1792000E-06,.1983800E-06,.2185200E-06/

      data   absb(121:235, 2)   /                                       &
     & .1164400E-06,.1306100E-06,.1456900E-06,.1614800E-06,.1779400E-06,&
     & .9432700E-07,.1059700E-06,.1182700E-06,.1312100E-06,.1447100E-06,&
     & .7638600E-07,.8593200E-07,.9600400E-07,.1067500E-06,.1178500E-06,&
     & .6109300E-07,.6885600E-07,.7708800E-07,.8583100E-07,.9487800E-07,&
     & .4878100E-07,.5505800E-07,.6172500E-07,.6885200E-07,.7623400E-07,&
     & .3890900E-07,.4398700E-07,.4939100E-07,.5516100E-07,.6119700E-07,&
     & .3077900E-07,.3486600E-07,.3922200E-07,.4387000E-07,.4878500E-07,&
     & .2424900E-07,.2751400E-07,.3101400E-07,.3477400E-07,.3877200E-07,&
     & .1907600E-07,.2169300E-07,.2451400E-07,.2752200E-07,.3075600E-07,&
     & .1494900E-07,.1704600E-07,.1930300E-07,.2172100E-07,.2432000E-07,&
     & .1162700E-07,.1329400E-07,.1510200E-07,.1704000E-07,.1911500E-07,&
     & .9020800E-08,.1035900E-07,.1180300E-07,.1334900E-07,.1500300E-07,&
     & .6997900E-08,.8062500E-08,.9207500E-08,.1044500E-07,.1176800E-07,&
     & .5450700E-08,.6277700E-08,.7200400E-08,.8189900E-08,.9248800E-08,&
     & .4254100E-08,.4893300E-08,.5630400E-08,.6421400E-08,.7272500E-08,&
     & .3316100E-08,.3812100E-08,.4397600E-08,.5030100E-08,.5714000E-08,&
     & .2581700E-08,.2969100E-08,.3429100E-08,.3936800E-08,.4483600E-08,&
     & .2020300E-08,.2322800E-08,.2682300E-08,.3090000E-08,.3528500E-08,&
     & .1583300E-08,.1819600E-08,.2102300E-08,.2427600E-08,.2778700E-08,&
     & .1240300E-08,.1425000E-08,.1646100E-08,.1904500E-08,.2185500E-08,&
     & .9724900E-09,.1116400E-08,.1288100E-08,.1492300E-08,.1717500E-08,&
     & .7643100E-09,.8769900E-09,.1010800E-08,.1170900E-08,.1351600E-08,&
     & .6155300E-09,.7064900E-09,.8146300E-09,.9431500E-09,.1090700E-08/

      data   absb(  1:120, 3)   /                                       &
     & .3770800E-04,.4394200E-04,.5058600E-04,.5765600E-04,.6513900E-04,&
     & .3169400E-04,.3686200E-04,.4235000E-04,.4819700E-04,.5434100E-04,&
     & .2656400E-04,.3082400E-04,.3537200E-04,.4018000E-04,.4523100E-04,&
     & .2223200E-04,.2574000E-04,.2951200E-04,.3344700E-04,.3758800E-04,&
     & .1857800E-04,.2146700E-04,.2458200E-04,.2783200E-04,.3122100E-04,&
     & .1553000E-04,.1792200E-04,.2049800E-04,.2317000E-04,.2597300E-04,&
     & .1297200E-04,.1495200E-04,.1705900E-04,.1927000E-04,.2158000E-04,&
     & .1085800E-04,.1249800E-04,.1422800E-04,.1606100E-04,.1795800E-04,&
     & .9081900E-05,.1044600E-04,.1188000E-04,.1339400E-04,.1495100E-04,&
     & .7664800E-05,.8797900E-05,.9993100E-05,.1125500E-04,.1254200E-04,&
     & .6463400E-05,.7414000E-05,.8408600E-05,.9454600E-05,.1052700E-04,&
     & .5452300E-05,.6246400E-05,.7080900E-05,.7950800E-05,.8844600E-05,&
     & .4611000E-05,.5271600E-05,.5969300E-05,.6690400E-05,.7432200E-05,&
     & .3913400E-05,.4462000E-05,.5042500E-05,.5643900E-05,.6257700E-05,&
     & .3320800E-05,.3777700E-05,.4262900E-05,.4760700E-05,.5271400E-05,&
     & .2817700E-05,.3200800E-05,.3603100E-05,.4018700E-05,.4435800E-05,&
     & .2393200E-05,.2715800E-05,.3051600E-05,.3394500E-05,.3741600E-05,&
     & .2034000E-05,.2303500E-05,.2582700E-05,.2867200E-05,.3155900E-05,&
     & .1730800E-05,.1954700E-05,.2186900E-05,.2424300E-05,.2664900E-05,&
     & .1473400E-05,.1659300E-05,.1853600E-05,.2051800E-05,.2251400E-05,&
     & .1253800E-05,.1409700E-05,.1571800E-05,.1736900E-05,.1901300E-05,&
     & .1062600E-05,.1193200E-05,.1329000E-05,.1466200E-05,.1602500E-05,&
     & .8899000E-06,.9987800E-06,.1112200E-05,.1226600E-05,.1339500E-05,&
     & .7353500E-06,.8269000E-06,.9209900E-06,.1015900E-05,.1109900E-05/

      data   absb(121:235, 3)   /                                       &
     & .5972800E-06,.6730100E-06,.7512700E-06,.8302100E-06,.9084500E-06,&
     & .4848400E-06,.5473700E-06,.6122900E-06,.6776200E-06,.7425800E-06,&
     & .3934700E-06,.4448500E-06,.4988600E-06,.5529000E-06,.6067400E-06,&
     & .3148800E-06,.3570300E-06,.4014800E-06,.4464000E-06,.4909100E-06,&
     & .2513000E-06,.2856100E-06,.3222400E-06,.3595300E-06,.3961800E-06,&
     & .2003000E-06,.2284000E-06,.2583400E-06,.2893000E-06,.3196500E-06,&
     & .1580700E-06,.1810900E-06,.2054200E-06,.2309400E-06,.2563500E-06,&
     & .1241800E-06,.1428200E-06,.1626100E-06,.1836400E-06,.2046500E-06,&
     & .9738300E-07,.1124500E-06,.1285500E-06,.1457200E-06,.1630700E-06,&
     & .7601500E-07,.8806700E-07,.1012000E-06,.1151000E-06,.1293500E-06,&
     & .5881300E-07,.6849300E-07,.7910200E-07,.9027300E-07,.1020000E-06,&
     & .4544000E-07,.5314400E-07,.6159700E-07,.7063900E-07,.8025500E-07,&
     & .3501500E-07,.4113600E-07,.4786900E-07,.5521600E-07,.6299400E-07,&
     & .2705600E-07,.3191000E-07,.3726700E-07,.4321000E-07,.4952200E-07,&
     & .2090800E-07,.2475600E-07,.2904400E-07,.3379100E-07,.3890400E-07,&
     & .1613100E-07,.1918100E-07,.2258900E-07,.2637800E-07,.3050400E-07,&
     & .1241800E-07,.1483500E-07,.1753700E-07,.2054700E-07,.2386000E-07,&
     & .9599400E-08,.1151400E-07,.1366200E-07,.1606500E-07,.1871700E-07,&
     & .7423900E-08,.8945300E-08,.1065600E-07,.1256900E-07,.1469300E-07,&
     & .5730000E-08,.6944100E-08,.8303000E-08,.9819300E-08,.1152300E-07,&
     & .4415200E-08,.5381100E-08,.6458100E-08,.7661000E-08,.9020100E-08,&
     & .3407200E-08,.4170800E-08,.5029700E-08,.5987600E-08,.7068400E-08,&
     & .2730200E-08,.3349200E-08,.4048100E-08,.4826600E-08,.5704600E-08/

      data   absb(  1:120, 4)   /                                       &
     & .1201500E-03,.1392000E-03,.1593700E-03,.1800100E-03,.1992900E-03,&
     & .1025600E-03,.1188300E-03,.1358700E-03,.1525900E-03,.1681200E-03,&
     & .8720300E-04,.1009900E-03,.1149800E-03,.1284700E-03,.1411400E-03,&
     & .7385900E-04,.8530400E-04,.9681600E-04,.1077400E-03,.1180500E-03,&
     & .6233400E-04,.7178600E-04,.8122500E-04,.9009100E-04,.9845400E-04,&
     & .5258100E-04,.6034400E-04,.6801800E-04,.7523700E-04,.8190300E-04,&
     & .4419700E-04,.5057300E-04,.5684000E-04,.6265600E-04,.6796800E-04,&
     & .3715100E-04,.4238400E-04,.4749400E-04,.5219300E-04,.5646700E-04,&
     & .3119300E-04,.3548000E-04,.3960000E-04,.4342700E-04,.4687000E-04,&
     & .2637400E-04,.2992500E-04,.3322400E-04,.3626100E-04,.3904000E-04,&
     & .2231500E-04,.2519800E-04,.2783300E-04,.3029000E-04,.3250900E-04,&
     & .1888300E-04,.2118400E-04,.2329100E-04,.2526800E-04,.2704600E-04,&
     & .1597700E-04,.1780800E-04,.1952900E-04,.2110600E-04,.2253300E-04,&
     & .1352400E-04,.1499600E-04,.1639100E-04,.1765800E-04,.1879600E-04,&
     & .1143300E-04,.1263100E-04,.1375700E-04,.1476900E-04,.1567700E-04,&
     & .9655800E-05,.1064000E-04,.1153700E-04,.1235700E-04,.1308600E-04,&
     & .8163200E-05,.8956000E-05,.9680100E-05,.1033900E-04,.1091900E-04,&
     & .6897200E-05,.7532100E-05,.8118900E-05,.8641800E-05,.9112000E-05,&
     & .5825500E-05,.6342700E-05,.6811500E-05,.7229100E-05,.7608800E-05,&
     & .4923500E-05,.5340400E-05,.5712800E-05,.6045000E-05,.6354000E-05,&
     & .4160200E-05,.4490400E-05,.4787300E-05,.5056700E-05,.5308500E-05,&
     & .3499300E-05,.3764000E-05,.4003300E-05,.4221700E-05,.4432100E-05,&
     & .2916800E-05,.3133100E-05,.3326400E-05,.3507200E-05,.3680800E-05,&
     & .2410500E-05,.2586300E-05,.2745900E-05,.2896900E-05,.3043300E-05/

      data   absb(121:235, 4)   /                                       &
     & .1969900E-05,.2116700E-05,.2249900E-05,.2375100E-05,.2498400E-05,&
     & .1608500E-05,.1731600E-05,.1843100E-05,.1948200E-05,.2052000E-05,&
     & .1314000E-05,.1417200E-05,.1510200E-05,.1598900E-05,.1686200E-05,&
     & .1064300E-05,.1151800E-05,.1230000E-05,.1305200E-05,.1378900E-05,&
     & .8610300E-06,.9349400E-06,.1001300E-05,.1064800E-05,.1127500E-05,&
     & .6956600E-06,.7580300E-06,.8146800E-06,.8681700E-06,.9211300E-06,&
     & .5585600E-06,.6113500E-06,.6599200E-06,.7050100E-06,.7491500E-06,&
     & .4464400E-06,.4911300E-06,.5328400E-06,.5706600E-06,.6077600E-06,&
     & .3563800E-06,.3938600E-06,.4294700E-06,.4619200E-06,.4929500E-06,&
     & .2831700E-06,.3149000E-06,.3450300E-06,.3727600E-06,.3989400E-06,&
     & .2234100E-06,.2500200E-06,.2756000E-06,.2995700E-06,.3214400E-06,&
     & .1755400E-06,.1979800E-06,.2196200E-06,.2401000E-06,.2587600E-06,&
     & .1373700E-06,.1564200E-06,.1745100E-06,.1919200E-06,.2079500E-06,&
     & .1075400E-06,.1235700E-06,.1387900E-06,.1534300E-06,.1671200E-06,&
     & .8412100E-07,.9744100E-07,.1101800E-06,.1225600E-06,.1341900E-06,&
     & .6560400E-07,.7656400E-07,.8735600E-07,.9770300E-07,.1074500E-06,&
     & .5094300E-07,.5998800E-07,.6904800E-07,.7762900E-07,.8593300E-07,&
     & .3967900E-07,.4712100E-07,.5461400E-07,.6176600E-07,.6878300E-07,&
     & .3093400E-07,.3698000E-07,.4311300E-07,.4914200E-07,.5500500E-07,&
     & .2405000E-07,.2892300E-07,.3399300E-07,.3904700E-07,.4392800E-07,&
     & .1866000E-07,.2255700E-07,.2674800E-07,.3095100E-07,.3500000E-07,&
     & .1448800E-07,.1762500E-07,.2104300E-07,.2451100E-07,.2788400E-07,&
     & .1167400E-07,.1423400E-07,.1704400E-07,.1990200E-07,.2270200E-07/

      data   absb(  1:120, 5)   /                                       &
     & .5907500E-03,.6176547E-03,.6462139E-03,.6766147E-03,.7105858E-03,&
     & .5021733E-03,.5254919E-03,.5505928E-03,.5781750E-03,.6090718E-03,&
     & .4251753E-03,.4458740E-03,.4682385E-03,.4935183E-03,.5206768E-03,&
     & .3591813E-03,.3774559E-03,.3977574E-03,.4201300E-03,.4431469E-03,&
     & .3027495E-03,.3191476E-03,.3371282E-03,.3563107E-03,.3755935E-03,&
     & .2550227E-03,.2694860E-03,.2850784E-03,.3010196E-03,.3174204E-03,&
     & .2145715E-03,.2270374E-03,.2400302E-03,.2534708E-03,.2670980E-03,&
     & .1804017E-03,.1910788E-03,.2019239E-03,.2130155E-03,.2241997E-03,&
     & .1515116E-03,.1603628E-03,.1693253E-03,.1785842E-03,.1878199E-03,&
     & .1274750E-03,.1347358E-03,.1422851E-03,.1500079E-03,.1574823E-03,&
     & .1070376E-03,.1130951E-03,.1194785E-03,.1258411E-03,.1318753E-03,&
     & .8978466E-04,.9494123E-04,.1002546E-03,.1054132E-03,.1102246E-03,&
     & .7533356E-04,.7966324E-04,.8405144E-04,.8817180E-04,.9207527E-04,&
     & .6331538E-04,.6693246E-04,.7046596E-04,.7375138E-04,.7692004E-04,&
     & .5321123E-04,.5620271E-04,.5902917E-04,.6164644E-04,.6430450E-04,&
     & .4466516E-04,.4710753E-04,.4935907E-04,.5155918E-04,.5379983E-04,&
     & .3751313E-04,.3945958E-04,.4130911E-04,.4316167E-04,.4502731E-04,&
     & .3148456E-04,.3306301E-04,.3457534E-04,.3614273E-04,.3767792E-04,&
     & .2643306E-04,.2770164E-04,.2898435E-04,.3028820E-04,.3158349E-04,&
     & .2217901E-04,.2324093E-04,.2432947E-04,.2540558E-04,.2651303E-04,&
     & .1860493E-04,.1950781E-04,.2040165E-04,.2133522E-04,.2227687E-04,&
     & .1560186E-04,.1635369E-04,.1712128E-04,.1791364E-04,.1869262E-04,&
     & .1303087E-04,.1366605E-04,.1431103E-04,.1497004E-04,.1563111E-04,&
     & .1082591E-04,.1136497E-04,.1190838E-04,.1246591E-04,.1301791E-04/

      data   absb(121:235, 5)   /                                       &
     & .8949161E-05,.9402237E-05,.9857226E-05,.1033892E-04,.1080399E-04,&
     & .7398494E-05,.7779511E-05,.8166568E-05,.8566423E-05,.8964480E-05,&
     & .6113971E-05,.6437029E-05,.6769198E-05,.7108135E-05,.7445280E-05,&
     & .5026369E-05,.5297079E-05,.5578033E-05,.5865750E-05,.6156368E-05,&
     & .4126175E-05,.4355500E-05,.4592219E-05,.4836238E-05,.5085641E-05,&
     & .3390696E-05,.3580871E-05,.3780710E-05,.3987163E-05,.4198069E-05,&
     & .2776281E-05,.2936183E-05,.3106181E-05,.3277157E-05,.3458786E-05,&
     & .2266975E-05,.2401731E-05,.2543341E-05,.2688159E-05,.2843911E-05,&
     & .1851175E-05,.1964040E-05,.2082434E-05,.2205585E-05,.2334866E-05,&
     & .1508390E-05,.1603576E-05,.1702375E-05,.1806116E-05,.1913393E-05,&
     & .1224515E-05,.1304492E-05,.1387169E-05,.1474342E-05,.1565346E-05,&
     & .9927416E-06,.1060100E-05,.1128545E-05,.1203015E-05,.1278997E-05,&
     & .8040118E-06,.8600684E-06,.9178680E-06,.9793761E-06,.1043884E-05,&
     & .6511957E-06,.6983405E-06,.7470839E-06,.7982212E-06,.8522692E-06,&
     & .5274519E-06,.5672397E-06,.6079811E-06,.6509056E-06,.6959894E-06,&
     & .4266819E-06,.4601105E-06,.4940630E-06,.5297410E-06,.5677419E-06,&
     & .3444399E-06,.3727641E-06,.4009432E-06,.4307494E-06,.4625489E-06,&
     & .2784617E-06,.3023078E-06,.3258711E-06,.3507791E-06,.3773299E-06,&
     & .2249763E-06,.2450479E-06,.2650681E-06,.2856452E-06,.3078613E-06,&
     & .1813509E-06,.1984089E-06,.2151828E-06,.2323405E-06,.2508149E-06,&
     & .1459558E-06,.1603926E-06,.1745826E-06,.1886604E-06,.2039625E-06,&
     & .1175460E-06,.1296812E-06,.1415814E-06,.1532665E-06,.1658461E-06,&
     & .9640155E-07,.1066253E-06,.1166034E-06,.1263881E-06,.1369759E-06/

      data   absb(  1:120, 6)   /                                       &
     & .3651810E-02,.3746595E-02,.3858789E-02,.3979164E-02,.4109370E-02,&
     & .3141811E-02,.3230210E-02,.3325595E-02,.3429577E-02,.3547976E-02,&
     & .2701370E-02,.2775610E-02,.2858077E-02,.2953528E-02,.3061412E-02,&
     & .2315393E-02,.2378831E-02,.2454205E-02,.2541014E-02,.2642100E-02,&
     & .1977791E-02,.2036050E-02,.2104700E-02,.2186085E-02,.2283190E-02,&
     & .1685324E-02,.1740054E-02,.1805015E-02,.1884094E-02,.1978199E-02,&
     & .1437021E-02,.1487251E-02,.1550029E-02,.1626786E-02,.1715576E-02,&
     & .1225237E-02,.1273610E-02,.1334615E-02,.1407663E-02,.1492701E-02,&
     & .1044735E-02,.1092958E-02,.1152692E-02,.1223437E-02,.1303990E-02,&
     & .8942887E-03,.9428463E-03,.1001793E-02,.1070477E-02,.1149176E-02,&
     & .7694049E-03,.8171499E-03,.8745104E-03,.9407106E-03,.1016851E-02,&
     & .6645252E-03,.7113905E-03,.7673264E-03,.8320640E-03,.9036498E-03,&
     & .5770389E-03,.6234065E-03,.6779963E-03,.7399304E-03,.8071342E-03,&
     & .5049575E-03,.5502834E-03,.6030293E-03,.6613349E-03,.7246181E-03,&
     & .4447202E-03,.4886958E-03,.5389704E-03,.5940660E-03,.6530676E-03,&
     & .3944259E-03,.4369189E-03,.4843540E-03,.5362101E-03,.5917015E-03,&
     & .3520821E-03,.3928555E-03,.4377636E-03,.4866798E-03,.5390983E-03,&
     & .3162045E-03,.3551094E-03,.3978195E-03,.4440662E-03,.4940346E-03,&
     & .2859841E-03,.3231854E-03,.3639308E-03,.4079919E-03,.4558740E-03,&
     & .2600638E-03,.2959725E-03,.3350934E-03,.3776361E-03,.4237286E-03,&
     & .2380660E-03,.2725920E-03,.3104257E-03,.3517378E-03,.3964521E-03,&
     & .2181478E-03,.2514965E-03,.2881695E-03,.3284820E-03,.3717262E-03,&
     & .1983067E-03,.2303075E-03,.2658491E-03,.3047978E-03,.3465710E-03,&
     & .1780704E-03,.2085530E-03,.2427653E-03,.2802542E-03,.3204639E-03/

      data   absb(121:235, 6)   /                                       &
     & .1565144E-03,.1849367E-03,.2172813E-03,.2529291E-03,.2913913E-03,&
     & .1374906E-03,.1639820E-03,.1944800E-03,.2286994E-03,.2656361E-03,&
     & .1209235E-03,.1455643E-03,.1744351E-03,.2072121E-03,.2429831E-03,&
     & .1042521E-03,.1269284E-03,.1536021E-03,.1846991E-03,.2190618E-03,&
     & .8956332E-04,.1101797E-03,.1349815E-03,.1642251E-03,.1972149E-03,&
     & .7682060E-04,.9563184E-04,.1183845E-03,.1457632E-03,.1774806E-03,&
     & .6483539E-04,.8163448E-04,.1023214E-03,.1276117E-03,.1579098E-03,&
     & .5421096E-04,.6906822E-04,.8771080E-04,.1108467E-03,.1392070E-03,&
     & .4516018E-04,.5817822E-04,.7490186E-04,.9595379E-04,.1222179E-03,&
     & .3720506E-04,.4849290E-04,.6322139E-04,.8217544E-04,.1061847E-03,&
     & .3014760E-04,.3978783E-04,.5256468E-04,.6924860E-04,.9085729E-04,&
     & .2427491E-04,.3242037E-04,.4333198E-04,.5801079E-04,.7732452E-04,&
     & .1942608E-04,.2622750E-04,.3554119E-04,.4824771E-04,.6532610E-04,&
     & .1560503E-04,.2127119E-04,.2922144E-04,.4016492E-04,.5526442E-04,&
     & .1246448E-04,.1717739E-04,.2388696E-04,.3336674E-04,.4667005E-04,&
     & .9928517E-05,.1381768E-04,.1943442E-04,.2753949E-04,.3915061E-04,&
     & .7845842E-05,.1104661E-04,.1568945E-04,.2259744E-04,.3259294E-04,&
     & .6241865E-05,.8874763E-05,.1274389E-04,.1859414E-04,.2728085E-04,&
     & .4962728E-05,.7110134E-05,.1034291E-04,.1527272E-04,.2280240E-04,&
     & .3921680E-05,.5664841E-05,.8348994E-05,.1247617E-04,.1894519E-04,&
     & .3092625E-05,.4493204E-05,.6711569E-05,.1013101E-04,.1563245E-04,&
     & .2446444E-05,.3564130E-05,.5373137E-05,.8228205E-05,.1289787E-04,&
     & .2061392E-05,.3034672E-05,.4641777E-05,.7229029E-05,.1154580E-04/

      data   absb(  1:120, 7)   /                                       &
     & .7974511E-01,.7917223E-01,.7880571E-01,.7865688E-01,.7869742E-01,&
     & .7216761E-01,.7180196E-01,.7168710E-01,.7178858E-01,.7210934E-01,&
     & .6471287E-01,.6460046E-01,.6473991E-01,.6511587E-01,.6570290E-01,&
     & .5758865E-01,.5773359E-01,.5813143E-01,.5877105E-01,.5965013E-01,&
     & .5098748E-01,.5136835E-01,.5201772E-01,.5291936E-01,.5405095E-01,&
     & .4498346E-01,.4559189E-01,.4647453E-01,.4761481E-01,.4899661E-01,&
     & .3962956E-01,.4044591E-01,.4154636E-01,.4290895E-01,.4451971E-01,&
     & .3494623E-01,.3595631E-01,.3725553E-01,.3882539E-01,.4065371E-01,&
     & .3090317E-01,.3209598E-01,.3356918E-01,.3532088E-01,.3734086E-01,&
     & .2753735E-01,.2889680E-01,.3055328E-01,.3249469E-01,.3469930E-01,&
     & .2472194E-01,.2624583E-01,.2807624E-01,.3018719E-01,.3256894E-01,&
     & .2241246E-01,.2409593E-01,.2607770E-01,.2835897E-01,.3091024E-01,&
     & .2056186E-01,.2238553E-01,.2452455E-01,.2695966E-01,.2967202E-01,&
     & .1912250E-01,.2109409E-01,.2338231E-01,.2597361E-01,.2884315E-01,&
     & .1803069E-01,.2014547E-01,.2257560E-01,.2531768E-01,.2833188E-01,&
     & .1723757E-01,.1948465E-01,.2205955E-01,.2494239E-01,.2808185E-01,&
     & .1671584E-01,.1909967E-01,.2181124E-01,.2481820E-01,.2807426E-01,&
     & .1642188E-01,.1893762E-01,.2177522E-01,.2489493E-01,.2825602E-01,&
     & .1633447E-01,.1897666E-01,.2192891E-01,.2515422E-01,.2861454E-01,&
     & .1642117E-01,.1918126E-01,.2224051E-01,.2556670E-01,.2911713E-01,&
     & .1665212E-01,.1952573E-01,.2268617E-01,.2610433E-01,.2973637E-01,&
     & .1690445E-01,.1987409E-01,.2312114E-01,.2661658E-01,.3031656E-01,&
     & .1697413E-01,.2000260E-01,.2330589E-01,.2685241E-01,.3059806E-01,&
     & .1679058E-01,.1984390E-01,.2316985E-01,.2673928E-01,.3050650E-01/

      data   absb(121:235, 7)   /                                       &
     & .1621314E-01,.1924283E-01,.2254989E-01,.2610500E-01,.2986511E-01,&
     & .1565005E-01,.1865129E-01,.2193291E-01,.2546985E-01,.2921685E-01,&
     & .1512354E-01,.1809306E-01,.2134739E-01,.2486405E-01,.2859408E-01,&
     & .1434388E-01,.1724600E-01,.2044363E-01,.2391201E-01,.2760136E-01,&
     & .1356639E-01,.1639374E-01,.1952578E-01,.2293718E-01,.2658026E-01,&
     & .1282426E-01,.1557142E-01,.1863464E-01,.2198740E-01,.2557922E-01,&
     & .1197741E-01,.1462053E-01,.1760067E-01,.2086625E-01,.2439863E-01,&
     & .1111477E-01,.1364234E-01,.1651789E-01,.1969313E-01,.2315038E-01,&
     & .1030179E-01,.1271031E-01,.1547643E-01,.1856508E-01,.2193139E-01,&
     & .9491085E-02,.1177021E-01,.1441481E-01,.1739504E-01,.2066837E-01,&
     & .8639429E-02,.1078206E-01,.1328441E-01,.1613831E-01,.1930011E-01,&
     & .7839892E-02,.9857468E-02,.1221482E-01,.1493348E-01,.1797877E-01,&
     & .7092072E-02,.8991607E-02,.1120568E-01,.1378380E-01,.1670474E-01,&
     & .6432463E-02,.8224187E-02,.1031515E-01,.1275809E-01,.1555586E-01,&
     & .5832717E-02,.7522561E-02,.9503860E-02,.1181296E-01,.1448506E-01,&
     & .5280145E-02,.6862909E-02,.8736900E-02,.1091927E-01,.1346213E-01,&
     & .4760950E-02,.6239258E-02,.8009269E-02,.1007450E-01,.1248495E-01,&
     & .4319927E-02,.5706660E-02,.7378235E-02,.9343778E-02,.1163217E-01,&
     & .3927425E-02,.5223469E-02,.6805075E-02,.8675949E-02,.1085346E-01,&
     & .3566312E-02,.4771326E-02,.6260588E-02,.8040438E-02,.1011504E-01,&
     & .3235298E-02,.4349610E-02,.5751274E-02,.7436457E-02,.9414970E-02,&
     & .2948268E-02,.3977194E-02,.5291955E-02,.6891886E-02,.8780537E-02,&
     & .2838304E-02,.3833360E-02,.5113920E-02,.6679829E-02,.8533277E-02/

      data   absb(  1:120, 8)   /                                       &
     & .1589380E+01,.1590617E+01,.1588657E+01,.1583627E+01,.1576103E+01,&
     & .1709558E+01,.1708607E+01,.1704305E+01,.1697075E+01,.1686632E+01,&
     & .1824948E+01,.1821482E+01,.1814617E+01,.1804445E+01,.1791378E+01,&
     & .1933407E+01,.1927272E+01,.1917585E+01,.1904625E+01,.1888292E+01,&
     & .2032788E+01,.2024036E+01,.2011680E+01,.1995779E+01,.1976689E+01,&
     & .2122514E+01,.2111187E+01,.2096103E+01,.2077495E+01,.2055661E+01,&
     & .2201894E+01,.2188149E+01,.2170688E+01,.2149637E+01,.2125223E+01,&
     & .2271095E+01,.2255113E+01,.2235383E+01,.2211893E+01,.2185029E+01,&
     & .2330637E+01,.2312532E+01,.2290760E+01,.2265233E+01,.2236248E+01,&
     & .2380235E+01,.2360023E+01,.2336137E+01,.2308453E+01,.2277303E+01,&
     & .2421520E+01,.2399497E+01,.2373596E+01,.2343962E+01,.2310850E+01,&
     & .2455548E+01,.2431616E+01,.2403985E+01,.2372410E+01,.2337341E+01,&
     & .2483000E+01,.2457533E+01,.2427896E+01,.2394579E+01,.2357794E+01,&
     & .2504497E+01,.2477271E+01,.2446062E+01,.2410900E+01,.2372279E+01,&
     & .2521140E+01,.2492125E+01,.2459283E+01,.2422484E+01,.2382142E+01,&
     & .2533448E+01,.2502926E+01,.2468416E+01,.2429941E+01,.2388304E+01,&
     & .2541996E+01,.2509801E+01,.2473737E+01,.2433957E+01,.2391015E+01,&
     & .2547277E+01,.2513671E+01,.2476134E+01,.2434937E+01,.2390872E+01,&
     & .2549698E+01,.2514609E+01,.2475824E+01,.2433436E+01,.2388147E+01,&
     & .2549776E+01,.2513277E+01,.2473300E+01,.2429667E+01,.2383220E+01,&
     & .2547817E+01,.2510019E+01,.2468803E+01,.2424113E+01,.2376687E+01,&
     & .2545438E+01,.2506563E+01,.2464177E+01,.2418709E+01,.2370529E+01,&
     & .2545437E+01,.2505887E+01,.2462992E+01,.2416876E+01,.2368167E+01,&
     & .2548620E+01,.2508763E+01,.2465654E+01,.2419269E+01,.2370306E+01/

      data   absb(121:235, 8)   /                                       &
     & .2556882E+01,.2517384E+01,.2474533E+01,.2428347E+01,.2379571E+01,&
     & .2564751E+01,.2525708E+01,.2483281E+01,.2437321E+01,.2388806E+01,&
     & .2572195E+01,.2533600E+01,.2491357E+01,.2445820E+01,.2397529E+01,&
     & .2582733E+01,.2544997E+01,.2503670E+01,.2458754E+01,.2411027E+01,&
     & .2593208E+01,.2556450E+01,.2515941E+01,.2471723E+01,.2424698E+01,&
     & .2603291E+01,.2567534E+01,.2527827E+01,.2484500E+01,.2438025E+01,&
     & .2614485E+01,.2580122E+01,.2541480E+01,.2499317E+01,.2453682E+01,&
     & .2626088E+01,.2593101E+01,.2555788E+01,.2514770E+01,.2470119E+01,&
     & .2636975E+01,.2605452E+01,.2569448E+01,.2529581E+01,.2486026E+01,&
     & .2647757E+01,.2617959E+01,.2583514E+01,.2544917E+01,.2502605E+01,&
     & .2659180E+01,.2630964E+01,.2598269E+01,.2561345E+01,.2520508E+01,&
     & .2669920E+01,.2643199E+01,.2612358E+01,.2577040E+01,.2537623E+01,&
     & .2680036E+01,.2654848E+01,.2625604E+01,.2592088E+01,.2554283E+01,&
     & .2688888E+01,.2664859E+01,.2637463E+01,.2605558E+01,.2569275E+01,&
     & .2696948E+01,.2674379E+01,.2648210E+01,.2617842E+01,.2583127E+01,&
     & .2704362E+01,.2683131E+01,.2658263E+01,.2629633E+01,.2596479E+01,&
     & .2711382E+01,.2691630E+01,.2667973E+01,.2640779E+01,.2609393E+01,&
     & .2717487E+01,.2698684E+01,.2676350E+01,.2650456E+01,.2620490E+01,&
     & .2722676E+01,.2705192E+01,.2684099E+01,.2659244E+01,.2630640E+01,&
     & .2727533E+01,.2711288E+01,.2691406E+01,.2667688E+01,.2640330E+01,&
     & .2731955E+01,.2717083E+01,.2698177E+01,.2675744E+01,.2649670E+01,&
     & .2735798E+01,.2722099E+01,.2704431E+01,.2682979E+01,.2657966E+01,&
     & .2737188E+01,.2723903E+01,.2706774E+01,.2685895E+01,.2661240E+01/

!  ---

      data   forref(1:3,1: 8)    /                                      &
     & .1062750E-05,.1041850E-05,.4201540E-05,.1543430E-05,.6531930E-05,&
     & .1745960E-04,.3489170E-05,.1084200E-04,.5408490E-04,.1458220E-04,&
     & .1560270E-04,.8812630E-04,.3256123E-04,.1001071E-03,.8710486E-04,&
     & .1958190E-03,.1469074E-03,.7616317E-04,.2991387E-03,.1425125E-03,&
     & .6636773E-04,.4406950E-03,.1586718E-03,.3817266E-04/


      data   selfref(1:10,1: 8)  /                                      &
     & .3317280E-03,.2874800E-03,.2491350E-03,.2159040E-03,.1871060E-03,&
     & .1621490E-03,.1405200E-03,.1217770E-03,.1055340E-03,.9145730E-04,&
     & .8826280E-03,.6989140E-03,.5534390E-03,.4382440E-03,.3470260E-03,&
     & .2747950E-03,.2175980E-03,.1723060E-03,.1364420E-03,.1080420E-03,&
     & .1154610E-02,.9372030E-03,.7607300E-03,.6174860E-03,.5012150E-03,&
     & .4068370E-03,.3302310E-03,.2680490E-03,.2175760E-03,.1766070E-03,&
     & .1034500E-02,.9602680E-03,.8913600E-03,.8273970E-03,.7680240E-03,&
     & .7129110E-03,.6617540E-03,.6142670E-03,.5701880E-03,.5292720E-03,&
     & .3227190E-02,.2709139E-02,.2274479E-02,.1909745E-02,.1603665E-02,&
     & .1346766E-02,.1131137E-02,.9501252E-03,.7981584E-03,.6705633E-03,&
     & .3142835E-02,.3104435E-02,.3068990E-02,.3036443E-02,.3006728E-02,&
     & .2979763E-02,.2955519E-02,.2933917E-02,.2914907E-02,.2898444E-02,&
     & .2729233E-02,.2892026E-02,.3065501E-02,.3250452E-02,.3447726E-02,&
     & .3658264E-02,.3883086E-02,.4123283E-02,.4380054E-02,.4654719E-02,&
     & .2594476E-02,.2804964E-02,.3065165E-02,.3386332E-02,.3782724E-02,&
     & .4272483E-02,.4878732E-02,.5631073E-02,.6567477E-02,.7736768E-02/

!........................................!
      end module module_radsw_kgb19      !
!========================================!


!> This module sets up absorption coeffients for band 20: 5150-6150
!! cm-1 (low - h2o; high - h2o)
!========================================!
      module module_radsw_kgb20          !
!........................................!
!
! *********        the original program descriptions        ********* !
!                                                                     !
!     originally by j.delamere, atmospheric & environmental research. !
!     revision: 2.4                                                   !
!     band 20:  5150-6150 cm-1 (low - h2o; high - h2o)                !
!     reformatted for f90 by jjmorcrette, ecmwf                       !
!                                                                     !
!     this table has been re-generated for reduced number of g-point  !
!     by y.t.hou, ncep                                                !
!                                                                     !
! *********    *********    end description    *********    ********* !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NG20

!
      implicit none
!
      private
!
!> msa20=65
      integer,  public :: MSA20
!> msb20=235
      integer,  public :: MSB20
!> msf20=10
      integer,  public :: MSF20
!> mfr20=4
      integer,  public :: MFR20
      parameter (MSA20=65, MSB20=235, MSF20=10, MFR20=4)
      
      real (kind=kind_phys), public :: forref(MFR20,NG20)

!> ch4
      real (kind=kind_phys), public :: absch4(NG20)

!>    the array absa(65,NG20) (ka(5,13,NG20)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels> ~100mb,
!!    temperatures, and binary species parameters (see taumol.f for definition).
!!    the first index in the array, js, runs from 1 to 9, and corresponds to
!!    different values of the binary species parameter.  for instance,
!!    js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!!    js = 3 corresponds to the parameter value 2/8, etc.  the second index
!!    in the array, jt, which runs from 1 to 5, corresponds to different
!!    temperatures.  more specifically, jt = 3 means that the data are for
!!    the reference temperature tref for this  pressure level, jt = 2 refers
!!    to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!!    is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!!    to the jpth reference pressure level (see taumol.f for these levels
!!    in mb).  the fourth index, ig, goes from 1 to 10, and indicates
!!    which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absa(MSA20,NG20)

!>    the array absb(235,10) (kb(5,13:59,10)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels < ~100mb and
!!    temperatures. the first index in the array, jt, which runs from 1 to 5,
!!    corresponds to different temperatures.  more specifically, jt = 3 means
!!    that the data are for the reference temperature tref for this pressure
!!    level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!!    tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!!    the second index, jp, runs from 13 to 59 and refers to the jpth
!!    reference pressure level (see taumol.f for the value of these
!!    pressure levels in mb).  the third index, ig, goes from 1 to 10,
!!    and tells us which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absb(MSB20,NG20)

!>    the array selfref contains the coefficient of the water vapor
!!    self-continuum (including the energy term).  the first index
!!    refers to temperature in 7.2 degree increments.  For instance,
!!    jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!!    etc.  the second index runs over the g-channel (1 to 10).
      real (kind=kind_phys), public :: selfref(MSF20,NG20)

!> rayleigh extinction coefficient at \f$v=5670cm^{-1}\f$
      real (kind=kind_phys), parameter, public :: rayl = 4.12e-09

      data   absa(  1: 65, 1)   /                                       &
     & .7838300E-06,.8622000E-06,.9535900E-06,.1059000E-05,.1178200E-05,&
     & .6504000E-06,.7251000E-06,.8131800E-06,.9005900E-06,.9978600E-06,&
     & .5807200E-06,.6588800E-06,.7426500E-06,.8185400E-06,.9064500E-06,&
     & .5360100E-06,.6076500E-06,.6808800E-06,.7574100E-06,.8380100E-06,&
     & .5014200E-06,.5695100E-06,.6402800E-06,.7194400E-06,.7981300E-06,&
     & .4716400E-06,.5400800E-06,.6104000E-06,.6800600E-06,.7503400E-06,&
     & .4933700E-06,.5617800E-06,.6221500E-06,.6912700E-06,.7606000E-06,&
     & .6158100E-06,.7011700E-06,.7894200E-06,.8714500E-06,.9464700E-06,&
     & .1415400E-05,.1575400E-05,.1726100E-05,.1873900E-05,.1977400E-05,&
     & .3418600E-05,.3701200E-05,.3960200E-05,.4223500E-05,.4425600E-05,&
     & .3857400E-05,.4208000E-05,.4470100E-05,.4745400E-05,.5000200E-05,&
     & .3579400E-05,.3868600E-05,.4144300E-05,.4393900E-05,.4650000E-05,&
     & .2958600E-05,.3193800E-05,.3416700E-05,.3625300E-05,.3825000E-05/

      data   absa(  1: 65, 2)   /                                       &
     & .5709800E-05,.6463000E-05,.7311700E-05,.8243600E-05,.9194700E-05,&
     & .4638400E-05,.5312500E-05,.6030100E-05,.6802800E-05,.7582400E-05,&
     & .4065700E-05,.4648600E-05,.5245900E-05,.5895500E-05,.6554600E-05,&
     & .3889500E-05,.4425800E-05,.4975900E-05,.5566300E-05,.6170200E-05,&
     & .3897100E-05,.4435900E-05,.4993300E-05,.5552300E-05,.6123400E-05,&
     & .3953200E-05,.4464400E-05,.4973400E-05,.5487500E-05,.6010200E-05,&
     & .4106800E-05,.4583200E-05,.5069800E-05,.5552100E-05,.6017500E-05,&
     & .4792200E-05,.5215600E-05,.5681700E-05,.6157600E-05,.6631300E-05,&
     & .8319900E-05,.8831700E-05,.9368800E-05,.9975400E-05,.1062000E-04,&
     & .1683600E-04,.1852600E-04,.1988700E-04,.2116800E-04,.2210400E-04,&
     & .1888200E-04,.2100500E-04,.2289600E-04,.2477700E-04,.2611500E-04,&
     & .1774400E-04,.1978000E-04,.2160000E-04,.2352300E-04,.2512800E-04,&
     & .1473600E-04,.1635600E-04,.1795500E-04,.1953300E-04,.2086100E-04/

      data   absa(  1: 65, 3)   /                                       &
     & .4177600E-04,.4815000E-04,.5509700E-04,.6266100E-04,.7092000E-04,&
     & .3390900E-04,.3917600E-04,.4462200E-04,.5075400E-04,.5733600E-04,&
     & .2743800E-04,.3147300E-04,.3612200E-04,.4113400E-04,.4645600E-04,&
     & .2322200E-04,.2653500E-04,.3014100E-04,.3409900E-04,.3841600E-04,&
     & .2111400E-04,.2388800E-04,.2688300E-04,.3034000E-04,.3378500E-04,&
     & .2075000E-04,.2292600E-04,.2553600E-04,.2849200E-04,.3167600E-04,&
     & .2158400E-04,.2411200E-04,.2687200E-04,.2979400E-04,.3294000E-04,&
     & .2419400E-04,.2698100E-04,.3013700E-04,.3354600E-04,.3718200E-04,&
     & .3746100E-04,.4215800E-04,.4671800E-04,.5104800E-04,.5515400E-04,&
     & .7239100E-04,.7716400E-04,.8401600E-04,.8965800E-04,.9551100E-04,&
     & .9173600E-04,.9910700E-04,.1046300E-03,.1095200E-03,.1154900E-03,&
     & .9120000E-04,.9881200E-04,.1043200E-03,.1089300E-03,.1134900E-03,&
     & .7621700E-04,.8270200E-04,.8750000E-04,.9134900E-04,.9534900E-04/

      data   absa(  1: 65, 4)   /                                       &
     & .7170500E-03,.8274300E-03,.9470000E-03,.1067000E-02,.1190200E-02,&
     & .5790900E-03,.6709600E-03,.7672400E-03,.8658200E-03,.9716700E-03,&
     & .4477100E-03,.5199700E-03,.5986100E-03,.6796700E-03,.7667600E-03,&
     & .3408500E-03,.3983300E-03,.4622600E-03,.5288900E-03,.6000800E-03,&
     & .2667800E-03,.3123400E-03,.3634100E-03,.4166300E-03,.4741800E-03,&
     & .2090300E-03,.2471800E-03,.2892200E-03,.3335300E-03,.3817600E-03,&
     & .1717300E-03,.2033000E-03,.2371100E-03,.2740100E-03,.3144500E-03,&
     & .1672200E-03,.1939600E-03,.2230700E-03,.2534700E-03,.2856100E-03,&
     & .2504300E-03,.2882400E-03,.3254700E-03,.3619300E-03,.3980600E-03,&
     & .4332300E-03,.4866700E-03,.5346200E-03,.5787900E-03,.6204000E-03,&
     & .4771900E-03,.5227100E-03,.5640200E-03,.6038800E-03,.6352000E-03,&
     & .4506100E-03,.4901400E-03,.5239100E-03,.5527600E-03,.5776000E-03,&
     & .3766400E-03,.4095200E-03,.4369000E-03,.4605500E-03,.4801800E-03/

      data   absa(  1: 65, 5)   /                                       &
     & .6646600E-02,.6868600E-02,.7089700E-02,.7301900E-02,.7491400E-02,&
     & .5402300E-02,.5601000E-02,.5800800E-02,.5998700E-02,.6168500E-02,&
     & .4321800E-02,.4512700E-02,.4684800E-02,.4864700E-02,.5026700E-02,&
     & .3463900E-02,.3647300E-02,.3795100E-02,.3957000E-02,.4104900E-02,&
     & .2774500E-02,.2937000E-02,.3075400E-02,.3214800E-02,.3349200E-02,&
     & .2202000E-02,.2352000E-02,.2483600E-02,.2596500E-02,.2711900E-02,&
     & .1762400E-02,.1896300E-02,.2016100E-02,.2115300E-02,.2211600E-02,&
     & .1416900E-02,.1533200E-02,.1635000E-02,.1723100E-02,.1812100E-02,&
     & .1303000E-02,.1390400E-02,.1469000E-02,.1550000E-02,.1627500E-02,&
     & .1715500E-02,.1816900E-02,.1894300E-02,.1962000E-02,.2021500E-02,&
     & .1715600E-02,.1780800E-02,.1841900E-02,.1892500E-02,.1931300E-02,&
     & .1526900E-02,.1581100E-02,.1628300E-02,.1668600E-02,.1706100E-02,&
     & .1276800E-02,.1320700E-02,.1355500E-02,.1392400E-02,.1423400E-02/

      data   absa(  1: 65, 6)   /                                       &
     & .1919100E-01,.1946300E-01,.1969200E-01,.1990000E-01,.2011200E-01,&
     & .1596700E-01,.1621900E-01,.1642600E-01,.1661300E-01,.1680500E-01,&
     & .1318600E-01,.1341100E-01,.1361400E-01,.1378500E-01,.1394400E-01,&
     & .1093500E-01,.1113100E-01,.1131800E-01,.1147500E-01,.1162000E-01,&
     & .9054100E-02,.9234400E-02,.9403500E-02,.9553700E-02,.9686200E-02,&
     & .7472400E-02,.7632000E-02,.7779400E-02,.7929400E-02,.8052800E-02,&
     & .6118300E-02,.6264400E-02,.6396100E-02,.6530300E-02,.6647800E-02,&
     & .5063800E-02,.5198400E-02,.5322200E-02,.5441200E-02,.5546500E-02,&
     & .4287000E-02,.4400900E-02,.4510900E-02,.4597600E-02,.4679600E-02,&
     & .4523900E-02,.4607300E-02,.4696800E-02,.4797900E-02,.4895600E-02,&
     & .4408900E-02,.4500600E-02,.4588100E-02,.4672100E-02,.4801500E-02,&
     & .3959800E-02,.4029000E-02,.4110800E-02,.4189900E-02,.4267200E-02,&
     & .3342500E-02,.3414800E-02,.3480800E-02,.3540500E-02,.3598300E-02/

      data   absa(  1: 65, 7)   /                                       &
     & .5016500E-01,.5034300E-01,.5045200E-01,.5042400E-01,.5032900E-01,&
     & .4272300E-01,.4288000E-01,.4293900E-01,.4291000E-01,.4286100E-01,&
     & .3611700E-01,.3625500E-01,.3630100E-01,.3629400E-01,.3628800E-01,&
     & .3058500E-01,.3072000E-01,.3078700E-01,.3081600E-01,.3084200E-01,&
     & .2587900E-01,.2602900E-01,.2611600E-01,.2617900E-01,.2622500E-01,&
     & .2182200E-01,.2197800E-01,.2209400E-01,.2217800E-01,.2224400E-01,&
     & .1830400E-01,.1847600E-01,.1860600E-01,.1870500E-01,.1878000E-01,&
     & .1522400E-01,.1539400E-01,.1551900E-01,.1561900E-01,.1569000E-01,&
     & .1283500E-01,.1302900E-01,.1316900E-01,.1328100E-01,.1336900E-01,&
     & .1163200E-01,.1171400E-01,.1176000E-01,.1177600E-01,.1180600E-01,&
     & .1117000E-01,.1125200E-01,.1128800E-01,.1135700E-01,.1139700E-01,&
     & .1040400E-01,.1053300E-01,.1070500E-01,.1083300E-01,.1094700E-01,&
     & .9104100E-02,.9267400E-02,.9406800E-02,.9541200E-02,.9676900E-02/

      data   absa(  1: 65, 8)   /                                       &
     & .1452700E+00,.1448300E+00,.1443200E+00,.1438900E+00,.1434900E+00,&
     & .1273900E+00,.1270200E+00,.1266100E+00,.1262400E+00,.1258400E+00,&
     & .1109100E+00,.1105800E+00,.1102400E+00,.1099000E+00,.1095300E+00,&
     & .9659000E-01,.9633500E-01,.9610500E-01,.9581800E-01,.9553000E-01,&
     & .8379900E-01,.8370800E-01,.8357200E-01,.8334700E-01,.8312600E-01,&
     & .7236900E-01,.7238800E-01,.7228900E-01,.7214300E-01,.7198100E-01,&
     & .6215800E-01,.6224700E-01,.6218500E-01,.6211100E-01,.6200700E-01,&
     & .5299800E-01,.5314200E-01,.5316200E-01,.5315400E-01,.5310100E-01,&
     & .4477600E-01,.4487300E-01,.4492000E-01,.4494600E-01,.4488500E-01,&
     & .3823700E-01,.3851400E-01,.3871600E-01,.3887700E-01,.3896600E-01,&
     & .3351100E-01,.3372100E-01,.3384300E-01,.3384000E-01,.3383900E-01,&
     & .2970400E-01,.2982500E-01,.2980800E-01,.2978700E-01,.2979400E-01,&
     & .2597300E-01,.2600400E-01,.2603200E-01,.2602300E-01,.2610700E-01/

      data   absa(  1: 65, 9)   /                                       &
     & .6574795E+00,.6554552E+00,.6533155E+00,.6509209E+00,.6484065E+00,&
     & .6255494E+00,.6246438E+00,.6232566E+00,.6216554E+00,.6197426E+00,&
     & .5862362E+00,.5862909E+00,.5857885E+00,.5847743E+00,.5835312E+00,&
     & .5431781E+00,.5440621E+00,.5441314E+00,.5437951E+00,.5431448E+00,&
     & .4982859E+00,.4996650E+00,.5003975E+00,.5006596E+00,.5007368E+00,&
     & .4522079E+00,.4542740E+00,.4556478E+00,.4567195E+00,.4575025E+00,&
     & .4066660E+00,.4093121E+00,.4113888E+00,.4131704E+00,.4142502E+00,&
     & .3626680E+00,.3658157E+00,.3685601E+00,.3707145E+00,.3720343E+00,&
     & .3201627E+00,.3237696E+00,.3268366E+00,.3290751E+00,.3306950E+00,&
     & .2769076E+00,.2802971E+00,.2831727E+00,.2851341E+00,.2866169E+00,&
     & .2450947E+00,.2482141E+00,.2507025E+00,.2525044E+00,.2537402E+00,&
     & .2182159E+00,.2209177E+00,.2229032E+00,.2242398E+00,.2253965E+00,&
     & .1913957E+00,.1940531E+00,.1961425E+00,.1981273E+00,.1993420E+00/

      data   absa(  1: 65,10)   /                                       &
     & .3268900E+01,.3251984E+01,.3235834E+01,.3221079E+01,.3207393E+01,&
     & .3622351E+01,.3601154E+01,.3582181E+01,.3564354E+01,.3547975E+01,&
     & .3989098E+01,.3964764E+01,.3942540E+01,.3922453E+01,.3903201E+01,&
     & .4343054E+01,.4315865E+01,.4291453E+01,.4268631E+01,.4246808E+01,&
     & .4682358E+01,.4653365E+01,.4626500E+01,.4601251E+01,.4576531E+01,&
     & .5008731E+01,.4977636E+01,.4948849E+01,.4920625E+01,.4892993E+01,&
     & .5317015E+01,.5284091E+01,.5253355E+01,.5222597E+01,.5194092E+01,&
     & .5603209E+01,.5569632E+01,.5536865E+01,.5505362E+01,.5476476E+01,&
     & .5865108E+01,.5831464E+01,.5798284E+01,.5767336E+01,.5737919E+01,&
     & .6093085E+01,.6061308E+01,.6030013E+01,.6000723E+01,.5972079E+01,&
     & .6260183E+01,.6230003E+01,.6201202E+01,.6173485E+01,.6146273E+01,&
     & .6409424E+01,.6381513E+01,.6354613E+01,.6328684E+01,.6301983E+01,&
     & .6564707E+01,.6537322E+01,.6510214E+01,.6481775E+01,.6454592E+01/

      data   absb(  1:120, 1)   /                                       &
     & .2959700E-05,.3192900E-05,.3415900E-05,.3625600E-05,.3826400E-05,&
     & .2423300E-05,.2615100E-05,.2786600E-05,.2950200E-05,.3117000E-05,&
     & .1920600E-05,.2078900E-05,.2209900E-05,.2338300E-05,.2450500E-05,&
     & .1493300E-05,.1607100E-05,.1703900E-05,.1799400E-05,.1881600E-05,&
     & .1142900E-05,.1214400E-05,.1289500E-05,.1350300E-05,.1417400E-05,&
     & .8413400E-06,.8926400E-06,.9426000E-06,.9900300E-06,.1039100E-05,&
     & .6068300E-06,.6414200E-06,.6801800E-06,.7204200E-06,.7547900E-06,&
     & .4580900E-06,.4881300E-06,.5193400E-06,.5482300E-06,.5726100E-06,&
     & .3521900E-06,.3783500E-06,.4032200E-06,.4277200E-06,.4473600E-06,&
     & .2741900E-06,.2954200E-06,.3161700E-06,.3339800E-06,.3493200E-06,&
     & .2138500E-06,.2312700E-06,.2467000E-06,.2595100E-06,.2695200E-06,&
     & .1676700E-06,.1817900E-06,.1928200E-06,.2013600E-06,.2111300E-06,&
     & .1336100E-06,.1430100E-06,.1498000E-06,.1577800E-06,.1661500E-06,&
     & .1061500E-06,.1121800E-06,.1188900E-06,.1257400E-06,.1311100E-06,&
     & .8366000E-07,.8929600E-07,.9501900E-07,.1000700E-06,.1039700E-06,&
     & .6666700E-07,.7121700E-07,.7561500E-07,.7923700E-07,.8223900E-07,&
     & .5277600E-07,.5608600E-07,.5982800E-07,.6251300E-07,.6421800E-07,&
     & .4142900E-07,.4437000E-07,.4673100E-07,.4855300E-07,.5014300E-07,&
     & .3243700E-07,.3452300E-07,.3620800E-07,.3750000E-07,.3857100E-07,&
     & .2523700E-07,.2673000E-07,.2786800E-07,.2882900E-07,.2994300E-07,&
     & .1954500E-07,.2061400E-07,.2146700E-07,.2228000E-07,.2314100E-07,&
     & .1514600E-07,.1586900E-07,.1650900E-07,.1717900E-07,.1798000E-07,&
     & .1148200E-07,.1203900E-07,.1254700E-07,.1325500E-07,.1387400E-07,&
     & .8617200E-08,.9002300E-08,.9560700E-08,.1010600E-07,.1059500E-07/

      data   absb(121:235, 1)   /                                       &
     & .6741100E-08,.7078100E-08,.7555300E-08,.7973900E-08,.8401100E-08,&
     & .5260400E-08,.5574300E-08,.5941000E-08,.6303100E-08,.6651600E-08,&
     & .4112900E-08,.4385400E-08,.4685100E-08,.4977100E-08,.5246300E-08,&
     & .3308900E-08,.3528900E-08,.3770200E-08,.4012500E-08,.4241500E-08,&
     & .2679400E-08,.2848800E-08,.3042800E-08,.3243700E-08,.3442000E-08,&
     & .2167000E-08,.2303000E-08,.2458900E-08,.2625200E-08,.2786700E-08,&
     & .1769900E-08,.1885300E-08,.2000100E-08,.2133000E-08,.2265700E-08,&
     & .1444600E-08,.1547100E-08,.1639700E-08,.1742300E-08,.1852600E-08,&
     & .1173400E-08,.1275400E-08,.1354200E-08,.1424200E-08,.1519300E-08,&
     & .9574300E-09,.1047000E-08,.1118300E-08,.1172600E-08,.1238900E-08,&
     & .7734500E-09,.8519700E-09,.9218400E-09,.9736800E-09,.1014900E-08,&
     & .6250000E-09,.6928800E-09,.7554800E-09,.8044600E-09,.8442800E-09,&
     & .5009500E-09,.5596700E-09,.6190400E-09,.6644100E-09,.7033900E-09,&
     & .4080600E-09,.4571200E-09,.5053000E-09,.5497400E-09,.5824600E-09,&
     & .3310200E-09,.3713000E-09,.4112100E-09,.4502400E-09,.4843600E-09,&
     & .2690600E-09,.3073900E-09,.3384500E-09,.3688400E-09,.4002400E-09,&
     & .2220200E-09,.2510800E-09,.2793700E-09,.3059200E-09,.3301200E-09,&
     & .1866200E-09,.2076100E-09,.2307200E-09,.2534200E-09,.2744500E-09,&
     & .1574100E-09,.1714700E-09,.1914600E-09,.2102900E-09,.2285700E-09,&
     & .1315100E-09,.1463700E-09,.1597700E-09,.1754700E-09,.1897400E-09,&
     & .1087400E-09,.1244700E-09,.1363400E-09,.1475200E-09,.1602200E-09,&
     & .9020600E-10,.1048400E-09,.1161700E-09,.1254500E-09,.1347400E-09,&
     & .7675900E-10,.8886600E-10,.9934000E-10,.1070600E-09,.1144400E-09/

      data   absb(  1:120, 2)   /                                       &
     & .1472700E-04,.1635700E-04,.1796100E-04,.1953000E-04,.2085200E-04,&
     & .1214800E-04,.1347800E-04,.1481800E-04,.1604900E-04,.1710000E-04,&
     & .9690200E-05,.1078700E-04,.1183500E-04,.1277000E-04,.1351500E-04,&
     & .7623500E-05,.8457500E-05,.9221700E-05,.9852600E-05,.1044900E-04,&
     & .5804900E-05,.6423900E-05,.6973300E-05,.7413100E-05,.7808200E-05,&
     & .4320400E-05,.4751000E-05,.5116500E-05,.5457900E-05,.5743400E-05,&
     & .3221100E-05,.3532700E-05,.3817500E-05,.4053500E-05,.4224500E-05,&
     & .2526500E-05,.2765200E-05,.2964400E-05,.3133100E-05,.3273000E-05,&
     & .1992100E-05,.2174900E-05,.2334300E-05,.2437700E-05,.2545800E-05,&
     & .1585900E-05,.1725900E-05,.1823600E-05,.1912400E-05,.1994200E-05,&
     & .1259100E-05,.1356500E-05,.1430900E-05,.1503200E-05,.1564200E-05,&
     & .1001200E-05,.1070100E-05,.1124600E-05,.1181900E-05,.1227900E-05,&
     & .7898500E-06,.8407400E-06,.8902900E-06,.9314500E-06,.9708500E-06,&
     & .6299600E-06,.6684700E-06,.7063700E-06,.7351300E-06,.7713500E-06,&
     & .5028300E-06,.5311100E-06,.5565300E-06,.5830600E-06,.6179900E-06,&
     & .3996200E-06,.4232000E-06,.4426500E-06,.4671500E-06,.4951600E-06,&
     & .3181000E-06,.3359700E-06,.3519100E-06,.3744800E-06,.3976800E-06,&
     & .2524200E-06,.2644700E-06,.2810900E-06,.3008900E-06,.3154300E-06,&
     & .1983700E-06,.2095400E-06,.2244200E-06,.2376200E-06,.2518300E-06,&
     & .1559900E-06,.1670100E-06,.1778900E-06,.1888000E-06,.2003000E-06,&
     & .1230700E-06,.1321100E-06,.1404600E-06,.1511100E-06,.1598800E-06,&
     & .9842900E-07,.1049800E-06,.1132600E-06,.1207400E-06,.1277900E-06,&
     & .7774300E-07,.8395100E-07,.9047300E-07,.9627700E-07,.1020400E-06,&
     & .6122000E-07,.6639900E-07,.7110200E-07,.7594600E-07,.8080300E-07/

      data   absb(121:235, 2)   /                                       &
     & .4881900E-07,.5296200E-07,.5666900E-07,.6081700E-07,.6470900E-07,&
     & .3887100E-07,.4222600E-07,.4537600E-07,.4864100E-07,.5185500E-07,&
     & .3086900E-07,.3348500E-07,.3623300E-07,.3886800E-07,.4152300E-07,&
     & .2479800E-07,.2698300E-07,.2918000E-07,.3143700E-07,.3360000E-07,&
     & .1995100E-07,.2178400E-07,.2351800E-07,.2546000E-07,.2721900E-07,&
     & .1609800E-07,.1754200E-07,.1897900E-07,.2054700E-07,.2201600E-07,&
     & .1302000E-07,.1414400E-07,.1541400E-07,.1661000E-07,.1787000E-07,&
     & .1054800E-07,.1150000E-07,.1249200E-07,.1352000E-07,.1452400E-07,&
     & .8560200E-08,.9324500E-08,.1009600E-07,.1097800E-07,.1181900E-07,&
     & .6914900E-08,.7565300E-08,.8215400E-08,.8882200E-08,.9600200E-08,&
     & .5570800E-08,.6116400E-08,.6667800E-08,.7192200E-08,.7763300E-08,&
     & .4480400E-08,.4931200E-08,.5396700E-08,.5866200E-08,.6278000E-08,&
     & .3646700E-08,.3986600E-08,.4362800E-08,.4758900E-08,.5120800E-08,&
     & .2952400E-08,.3254000E-08,.3547100E-08,.3863800E-08,.4188700E-08,&
     & .2388700E-08,.2652600E-08,.2895900E-08,.3150400E-08,.3411200E-08,&
     & .1932100E-08,.2164800E-08,.2362500E-08,.2583900E-08,.2817300E-08,&
     & .1568400E-08,.1772500E-08,.1947600E-08,.2124000E-08,.2311800E-08,&
     & .1279900E-08,.1446700E-08,.1607300E-08,.1749400E-08,.1916000E-08,&
     & .1044200E-08,.1191100E-08,.1321600E-08,.1453700E-08,.1577100E-08,&
     & .8574100E-09,.9793700E-09,.1093600E-08,.1205600E-08,.1316700E-08,&
     & .7035800E-09,.8033400E-09,.9082400E-09,.1005600E-08,.1101000E-08,&
     & .5742800E-09,.6644100E-09,.7542000E-09,.8362700E-09,.9248500E-09,&
     & .4792200E-09,.5559200E-09,.6349900E-09,.7073100E-09,.7867800E-09/

      data   absb(  1:120, 3)   /                                       &
     & .7620100E-04,.8269900E-04,.8749900E-04,.9135200E-04,.9534800E-04,&
     & .6282100E-04,.6786100E-04,.7186800E-04,.7510800E-04,.7855700E-04,&
     & .4977500E-04,.5336700E-04,.5663900E-04,.5900100E-04,.6230300E-04,&
     & .3800500E-04,.4088300E-04,.4326900E-04,.4540300E-04,.4813900E-04,&
     & .2855000E-04,.3075000E-04,.3233100E-04,.3448900E-04,.3668700E-04,&
     & .2097600E-04,.2232200E-04,.2374300E-04,.2555400E-04,.2722300E-04,&
     & .1538900E-04,.1642200E-04,.1779200E-04,.1912700E-04,.2029500E-04,&
     & .1188300E-04,.1281600E-04,.1396200E-04,.1487200E-04,.1572500E-04,&
     & .9294800E-05,.1018700E-04,.1098600E-04,.1171300E-04,.1242600E-04,&
     & .7396900E-05,.8084000E-05,.8726300E-05,.9385600E-05,.9953300E-05,&
     & .5896100E-05,.6459100E-05,.6964800E-05,.7449400E-05,.7938500E-05,&
     & .4744500E-05,.5174200E-05,.5603600E-05,.5975800E-05,.6372400E-05,&
     & .3844200E-05,.4179100E-05,.4483700E-05,.4807800E-05,.5129400E-05,&
     & .3133500E-05,.3392300E-05,.3640000E-05,.3896100E-05,.4148700E-05,&
     & .2549900E-05,.2743200E-05,.2956000E-05,.3154500E-05,.3343300E-05,&
     & .2063500E-05,.2223100E-05,.2389700E-05,.2551600E-05,.2693800E-05,&
     & .1663500E-05,.1802400E-05,.1927900E-05,.2049800E-05,.2162700E-05,&
     & .1335900E-05,.1447400E-05,.1547000E-05,.1634600E-05,.1745700E-05,&
     & .1069700E-05,.1147700E-05,.1225400E-05,.1309700E-05,.1391400E-05,&
     & .8495800E-06,.9146400E-06,.9818900E-06,.1048900E-05,.1115000E-05,&
     & .6817800E-06,.7335600E-06,.7890500E-06,.8422600E-06,.8947600E-06,&
     & .5473700E-06,.5899500E-06,.6337400E-06,.6773200E-06,.7218200E-06,&
     & .4368000E-06,.4708600E-06,.5077100E-06,.5420900E-06,.5768700E-06,&
     & .3461200E-06,.3746600E-06,.4037000E-06,.4307300E-06,.4597900E-06/

      data   absb(121:235, 3)   /                                       &
     & .2772500E-06,.3004800E-06,.3245100E-06,.3463800E-06,.3710500E-06,&
     & .2222800E-06,.2411400E-06,.2604900E-06,.2787100E-06,.2993500E-06,&
     & .1779700E-06,.1931700E-06,.2087800E-06,.2241100E-06,.2410200E-06,&
     & .1434600E-06,.1560100E-06,.1691300E-06,.1819700E-06,.1956600E-06,&
     & .1155900E-06,.1261500E-06,.1369700E-06,.1478500E-06,.1587400E-06,&
     & .9335800E-07,.1021400E-06,.1108900E-06,.1199800E-06,.1290400E-06,&
     & .7559900E-07,.8269000E-07,.8980800E-07,.9752400E-07,.1051100E-06,&
     & .6128500E-07,.6678600E-07,.7293700E-07,.7908700E-07,.8552500E-07,&
     & .4972500E-07,.5429400E-07,.5926300E-07,.6421300E-07,.6958300E-07,&
     & .4033600E-07,.4418600E-07,.4803400E-07,.5218400E-07,.5653400E-07,&
     & .3290400E-07,.3563300E-07,.3895900E-07,.4240200E-07,.4588300E-07,&
     & .2672500E-07,.2910400E-07,.3169300E-07,.3431200E-07,.3730400E-07,&
     & .2175100E-07,.2384900E-07,.2580300E-07,.2802400E-07,.3035600E-07,&
     & .1758700E-07,.1960900E-07,.2121600E-07,.2302200E-07,.2494200E-07,&
     & .1437600E-07,.1599000E-07,.1749300E-07,.1887200E-07,.2046300E-07,&
     & .1181100E-07,.1310600E-07,.1448900E-07,.1561500E-07,.1683200E-07,&
     & .9600100E-08,.1078800E-07,.1192100E-07,.1298500E-07,.1390400E-07,&
     & .7824300E-08,.8917800E-08,.9843000E-08,.1084000E-07,.1161900E-07,&
     & .6363800E-08,.7367800E-08,.8221400E-08,.8999700E-08,.9756500E-08,&
     & .5175300E-08,.6049700E-08,.6841900E-08,.7456000E-08,.8236200E-08,&
     & .4216400E-08,.4991600E-08,.5706600E-08,.6330800E-08,.6895000E-08,&
     & .3461800E-08,.4142400E-08,.4805400E-08,.5375700E-08,.5809500E-08,&
     & .2908300E-08,.3529100E-08,.4093200E-08,.4628500E-08,.4986500E-08/

      data   absb(  1:120, 4)   /                                       &
     & .3767400E-03,.4095300E-03,.4369000E-03,.4605900E-03,.4800700E-03,&
     & .3120200E-03,.3390600E-03,.3606600E-03,.3800300E-03,.3962700E-03,&
     & .2513500E-03,.2727100E-03,.2906900E-03,.3066100E-03,.3200500E-03,&
     & .1979700E-03,.2152700E-03,.2295500E-03,.2432800E-03,.2545600E-03,&
     & .1535400E-03,.1667900E-03,.1793300E-03,.1910100E-03,.2004100E-03,&
     & .1174200E-03,.1279700E-03,.1392300E-03,.1483600E-03,.1565400E-03,&
     & .9051900E-04,.9938600E-04,.1081900E-03,.1160100E-03,.1231900E-03,&
     & .7257900E-04,.7991000E-04,.8707500E-04,.9372800E-04,.9965600E-04,&
     & .5886600E-04,.6480500E-04,.7070500E-04,.7615300E-04,.8099600E-04,&
     & .4815700E-04,.5288300E-04,.5774000E-04,.6213000E-04,.6610000E-04,&
     & .3933700E-04,.4328900E-04,.4727500E-04,.5090600E-04,.5416100E-04,&
     & .3228900E-04,.3557700E-04,.3880800E-04,.4181400E-04,.4443300E-04,&
     & .2661500E-04,.2930100E-04,.3198400E-04,.3439600E-04,.3650900E-04,&
     & .2206000E-04,.2427000E-04,.2646000E-04,.2845000E-04,.3020300E-04,&
     & .1834400E-04,.2017700E-04,.2194800E-04,.2359500E-04,.2500100E-04,&
     & .1525600E-04,.1678700E-04,.1824200E-04,.1956100E-04,.2069600E-04,&
     & .1267900E-04,.1395200E-04,.1517000E-04,.1621100E-04,.1716300E-04,&
     & .1055000E-04,.1160500E-04,.1257900E-04,.1345100E-04,.1421000E-04,&
     & .8751200E-05,.9632200E-05,.1043300E-04,.1113300E-04,.1176200E-04,&
     & .7300100E-05,.8020300E-05,.8674500E-05,.9229200E-05,.9729500E-05,&
     & .6095800E-05,.6686200E-05,.7200600E-05,.7635900E-05,.8057600E-05,&
     & .5085500E-05,.5567100E-05,.5967000E-05,.6339200E-05,.6686900E-05,&
     & .4203300E-05,.4590500E-05,.4925000E-05,.5238400E-05,.5536900E-05,&
     & .3438400E-05,.3761800E-05,.4043000E-05,.4306000E-05,.4557700E-05/

      data   absb(121:235, 4)   /                                       &
     & .2803900E-05,.3074900E-05,.3313500E-05,.3534400E-05,.3745800E-05,&
     & .2284200E-05,.2510400E-05,.2712600E-05,.2896600E-05,.3076800E-05,&
     & .1858500E-05,.2051000E-05,.2219400E-05,.2373100E-05,.2524900E-05,&
     & .1504100E-05,.1667600E-05,.1809700E-05,.1940200E-05,.2068100E-05,&
     & .1215400E-05,.1354100E-05,.1474900E-05,.1585200E-05,.1693400E-05,&
     & .9803300E-06,.1097500E-05,.1201100E-05,.1294500E-05,.1385900E-05,&
     & .7872400E-06,.8862500E-06,.9748600E-06,.1055400E-05,.1130700E-05,&
     & .6305900E-06,.7138200E-06,.7894800E-06,.8575500E-06,.9204400E-06,&
     & .5042600E-06,.5739100E-06,.6380800E-06,.6971800E-06,.7497200E-06,&
     & .4013300E-06,.4585800E-06,.5141400E-06,.5640000E-06,.6096400E-06,&
     & .3158000E-06,.3644000E-06,.4112000E-06,.4543300E-06,.4934800E-06,&
     & .2484700E-06,.2890700E-06,.3287900E-06,.3653700E-06,.3990300E-06,&
     & .1953000E-06,.2285800E-06,.2624700E-06,.2933100E-06,.3221700E-06,&
     & .1536400E-06,.1811300E-06,.2091100E-06,.2358900E-06,.2599400E-06,&
     & .1207200E-06,.1441300E-06,.1665600E-06,.1895200E-06,.2098300E-06,&
     & .9492500E-07,.1139700E-06,.1325500E-06,.1519300E-06,.1696000E-06,&
     & .7447300E-07,.9019700E-07,.1057400E-06,.1215300E-06,.1366100E-06,&
     & .5900900E-07,.7159400E-07,.8478100E-07,.9775800E-07,.1101800E-06,&
     & .4700800E-07,.5699700E-07,.6783800E-07,.7875300E-07,.8911600E-07,&
     & .3741500E-07,.4564200E-07,.5442000E-07,.6346600E-07,.7191000E-07,&
     & .2989600E-07,.3663500E-07,.4369700E-07,.5103600E-07,.5826700E-07,&
     & .2394600E-07,.2952300E-07,.3531600E-07,.4118300E-07,.4739100E-07,&
     & .1988100E-07,.2464200E-07,.2939000E-07,.3409900E-07,.3923400E-07/

      data   absb(  1:120, 5)   /                                       &
     & .1277100E-02,.1320200E-02,.1355700E-02,.1392700E-02,.1423300E-02,&
     & .1060800E-02,.1095100E-02,.1124900E-02,.1154900E-02,.1178800E-02,&
     & .8675000E-03,.8967900E-03,.9198600E-03,.9439900E-03,.9640500E-03,&
     & .7007300E-03,.7250400E-03,.7463000E-03,.7649500E-03,.7816500E-03,&
     & .5628000E-03,.5832100E-03,.6015300E-03,.6154400E-03,.6288500E-03,&
     & .4494000E-03,.4659800E-03,.4788700E-03,.4905800E-03,.5022300E-03,&
     & .3575300E-03,.3706400E-03,.3829000E-03,.3939000E-03,.4038700E-03,&
     & .2901300E-03,.3009900E-03,.3117900E-03,.3211700E-03,.3301100E-03,&
     & .2365700E-03,.2460200E-03,.2550400E-03,.2634000E-03,.2714500E-03,&
     & .1942300E-03,.2022400E-03,.2098700E-03,.2171100E-03,.2243900E-03,&
     & .1596300E-03,.1664300E-03,.1730800E-03,.1794900E-03,.1858000E-03,&
     & .1316600E-03,.1374500E-03,.1433200E-03,.1487100E-03,.1542800E-03,&
     & .1088500E-03,.1139400E-03,.1189600E-03,.1237100E-03,.1284600E-03,&
     & .9040300E-04,.9491400E-04,.9915700E-04,.1031000E-03,.1072900E-03,&
     & .7526500E-04,.7916600E-04,.8264800E-04,.8620900E-04,.8988200E-04,&
     & .6287000E-04,.6605000E-04,.6911100E-04,.7228900E-04,.7545100E-04,&
     & .5246500E-04,.5519200E-04,.5788700E-04,.6066000E-04,.6336700E-04,&
     & .4388700E-04,.4619100E-04,.4858800E-04,.5091000E-04,.5327800E-04,&
     & .3676000E-04,.3876600E-04,.4078300E-04,.4278300E-04,.4478000E-04,&
     & .3083100E-04,.3256100E-04,.3426700E-04,.3600000E-04,.3759400E-04,&
     & .2590800E-04,.2738900E-04,.2883200E-04,.3029000E-04,.3158100E-04,&
     & .2177600E-04,.2302600E-04,.2426600E-04,.2544400E-04,.2651000E-04,&
     & .1823300E-04,.1928000E-04,.2032400E-04,.2126600E-04,.2214100E-04,&
     & .1516000E-04,.1604200E-04,.1691800E-04,.1768600E-04,.1840600E-04/

      data   absb(121:235, 5)   /                                       &
     & .1253800E-04,.1328500E-04,.1401500E-04,.1465700E-04,.1526900E-04,&
     & .1035700E-04,.1099000E-04,.1159500E-04,.1214800E-04,.1265100E-04,&
     & .8556600E-05,.9085600E-05,.9598600E-05,.1005700E-04,.1047800E-04,&
     & .7039000E-05,.7482600E-05,.7920000E-05,.8306100E-05,.8663400E-05,&
     & .5782700E-05,.6157200E-05,.6526900E-05,.6852800E-05,.7159500E-05,&
     & .4747700E-05,.5065600E-05,.5370500E-05,.5648600E-05,.5910000E-05,&
     & .3880200E-05,.4153300E-05,.4408900E-05,.4651500E-05,.4871900E-05,&
     & .3165800E-05,.3397100E-05,.3617700E-05,.3820500E-05,.4008900E-05,&
     & .2579600E-05,.2774000E-05,.2963500E-05,.3133600E-05,.3298600E-05,&
     & .2094500E-05,.2263600E-05,.2421400E-05,.2569800E-05,.2707700E-05,&
     & .1693100E-05,.1838300E-05,.1972900E-05,.2099600E-05,.2217200E-05,&
     & .1365000E-05,.1488900E-05,.1604300E-05,.1713900E-05,.1813600E-05,&
     & .1098100E-05,.1204200E-05,.1302800E-05,.1396300E-05,.1482400E-05,&
     & .8840700E-06,.9737000E-06,.1058900E-05,.1138300E-05,.1212400E-05,&
     & .7115000E-06,.7868200E-06,.8604800E-06,.9271100E-06,.9912800E-06,&
     & .5714000E-06,.6351300E-06,.6979000E-06,.7546600E-06,.8094600E-06,&
     & .4587100E-06,.5118300E-06,.5649400E-06,.6137600E-06,.6602200E-06,&
     & .3688900E-06,.4131900E-06,.4575900E-06,.4997300E-06,.5393300E-06,&
     & .2968300E-06,.3339600E-06,.3711000E-06,.4067100E-06,.4402600E-06,&
     & .2382800E-06,.2697300E-06,.3004800E-06,.3309900E-06,.3594900E-06,&
     & .1908300E-06,.2178200E-06,.2435000E-06,.2688300E-06,.2930900E-06,&
     & .1533400E-06,.1759200E-06,.1975300E-06,.2186900E-06,.2390300E-06,&
     & .1263500E-06,.1450400E-06,.1632100E-06,.1807400E-06,.1975900E-06/

      data   absb(  1:120, 6)   /                                       &
     & .3343300E-02,.3415900E-02,.3479800E-02,.3539700E-02,.3599500E-02,&
     & .2803900E-02,.2858900E-02,.2909000E-02,.2959500E-02,.3010800E-02,&
     & .2313500E-02,.2357900E-02,.2401200E-02,.2447000E-02,.2490000E-02,&
     & .1892500E-02,.1928100E-02,.1967100E-02,.2009000E-02,.2044500E-02,&
     & .1536500E-02,.1570300E-02,.1603900E-02,.1642400E-02,.1676000E-02,&
     & .1242400E-02,.1274600E-02,.1306700E-02,.1339700E-02,.1368700E-02,&
     & .1011300E-02,.1041700E-02,.1066400E-02,.1095800E-02,.1123100E-02,&
     & .8337500E-03,.8586300E-03,.8809800E-03,.9068000E-03,.9309300E-03,&
     & .6884900E-03,.7105600E-03,.7316100E-03,.7540200E-03,.7743200E-03,&
     & .5707600E-03,.5900100E-03,.6092500E-03,.6281600E-03,.6453100E-03,&
     & .4746200E-03,.4913800E-03,.5081800E-03,.5242100E-03,.5393000E-03,&
     & .3955200E-03,.4101100E-03,.4246000E-03,.4386500E-03,.4520600E-03,&
     & .3304100E-03,.3431700E-03,.3557300E-03,.3680000E-03,.3800500E-03,&
     & .2769900E-03,.2880600E-03,.2989200E-03,.3098300E-03,.3203800E-03,&
     & .2325300E-03,.2422900E-03,.2520800E-03,.2614400E-03,.2703900E-03,&
     & .1957300E-03,.2044200E-03,.2127600E-03,.2208700E-03,.2286900E-03,&
     & .1652800E-03,.1727100E-03,.1797400E-03,.1868100E-03,.1937300E-03,&
     & .1394900E-03,.1459900E-03,.1521400E-03,.1583400E-03,.1644800E-03,&
     & .1179300E-03,.1234500E-03,.1289100E-03,.1343200E-03,.1397500E-03,&
     & .9984500E-04,.1046400E-03,.1094100E-03,.1142500E-03,.1189600E-03,&
     & .8472300E-04,.8880800E-04,.9305400E-04,.9725700E-04,.1014400E-03,&
     & .7185700E-04,.7547800E-04,.7924500E-04,.8287800E-04,.8652700E-04,&
     & .6074100E-04,.6398700E-04,.6719000E-04,.7044500E-04,.7363400E-04,&
     & .5109400E-04,.5395300E-04,.5678700E-04,.5964400E-04,.6242100E-04/

      data   absb(121:235, 6)   /                                       &
     & .4280900E-04,.4532400E-04,.4780500E-04,.5031400E-04,.5279700E-04,&
     & .3586200E-04,.3808200E-04,.4024900E-04,.4245900E-04,.4469100E-04,&
     & .3004700E-04,.3197000E-04,.3387700E-04,.3584100E-04,.3779200E-04,&
     & .2505100E-04,.2673400E-04,.2842900E-04,.3017700E-04,.3185900E-04,&
     & .2085800E-04,.2233700E-04,.2383700E-04,.2534800E-04,.2684500E-04,&
     & .1735800E-04,.1865300E-04,.1994700E-04,.2127900E-04,.2261400E-04,&
     & .1438200E-04,.1550700E-04,.1663700E-04,.1780800E-04,.1897500E-04,&
     & .1187500E-04,.1285100E-04,.1384700E-04,.1486500E-04,.1588600E-04,&
     & .9782400E-05,.1064100E-04,.1149700E-04,.1238600E-04,.1328800E-04,&
     & .8030500E-05,.8771200E-05,.9517000E-05,.1028200E-04,.1107300E-04,&
     & .6555400E-05,.7188100E-05,.7833100E-05,.8502200E-05,.9179900E-05,&
     & .5339900E-05,.5881800E-05,.6436400E-05,.7009300E-05,.7599200E-05,&
     & .4339600E-05,.4799000E-05,.5272700E-05,.5763300E-05,.6278500E-05,&
     & .3527200E-05,.3913900E-05,.4323000E-05,.4742300E-05,.5181900E-05,&
     & .2862400E-05,.3191300E-05,.3537900E-05,.3897900E-05,.4274400E-05,&
     & .2318900E-05,.2596100E-05,.2890700E-05,.3197600E-05,.3520000E-05,&
     & .1873600E-05,.2107200E-05,.2357600E-05,.2617900E-05,.2894300E-05,&
     & .1515500E-05,.1713500E-05,.1924500E-05,.2145100E-05,.2380800E-05,&
     & .1225000E-05,.1392300E-05,.1571400E-05,.1757800E-05,.1959700E-05,&
     & .9901200E-06,.1129600E-05,.1280100E-05,.1439500E-05,.1610700E-05,&
     & .7987500E-06,.9141800E-06,.1040800E-05,.1177700E-05,.1321200E-05,&
     & .6447400E-06,.7410900E-06,.8466400E-06,.9632800E-06,.1084900E-05,&
     & .5329600E-06,.6142700E-06,.7038500E-06,.8037500E-06,.9086300E-06/

      data   absb(  1:120, 7)   /                                       &
     & .9106300E-02,.9268100E-02,.9404500E-02,.9538800E-02,.9673500E-02,&
     & .7848300E-02,.7980600E-02,.8123800E-02,.8252800E-02,.8360300E-02,&
     & .6626900E-02,.6744500E-02,.6884500E-02,.6991100E-02,.7085400E-02,&
     & .5518300E-02,.5641400E-02,.5757400E-02,.5848700E-02,.5940700E-02,&
     & .4569300E-02,.4678000E-02,.4774300E-02,.4862500E-02,.4946700E-02,&
     & .3761100E-02,.3853500E-02,.3940800E-02,.4023400E-02,.4100700E-02,&
     & .3099100E-02,.3180900E-02,.3264700E-02,.3338800E-02,.3404200E-02,&
     & .2585800E-02,.2658500E-02,.2734700E-02,.2799000E-02,.2857900E-02,&
     & .2167200E-02,.2232600E-02,.2297300E-02,.2354700E-02,.2405800E-02,&
     & .1821100E-02,.1880900E-02,.1936400E-02,.1985900E-02,.2035200E-02,&
     & .1531400E-02,.1584400E-02,.1633200E-02,.1676000E-02,.1724300E-02,&
     & .1292400E-02,.1338500E-02,.1379300E-02,.1420900E-02,.1465900E-02,&
     & .1092600E-02,.1132200E-02,.1168900E-02,.1208100E-02,.1249300E-02,&
     & .9275600E-03,.9613600E-03,.9960100E-03,.1032400E-02,.1069500E-02,&
     & .7883200E-03,.8189300E-03,.8513700E-03,.8846500E-03,.9188500E-03,&
     & .6709700E-03,.6997900E-03,.7298500E-03,.7603800E-03,.7921300E-03,&
     & .5723100E-03,.5993400E-03,.6272700E-03,.6547100E-03,.6838900E-03,&
     & .4899600E-03,.5144100E-03,.5394500E-03,.5656400E-03,.5916400E-03,&
     & .4199800E-03,.4423700E-03,.4655100E-03,.4892400E-03,.5132200E-03,&
     & .3613300E-03,.3816800E-03,.4029700E-03,.4244100E-03,.4472200E-03,&
     & .3115300E-03,.3305500E-03,.3498000E-03,.3697200E-03,.3908600E-03,&
     & .2694100E-03,.2865700E-03,.3043600E-03,.3229500E-03,.3425600E-03,&
     & .2321000E-03,.2478200E-03,.2641100E-03,.2812100E-03,.2993000E-03,&
     & .1990400E-03,.2131100E-03,.2279700E-03,.2438000E-03,.2603000E-03/

      data   absb(121:235, 7)   /                                       &
     & .1700000E-03,.1830000E-03,.1966600E-03,.2112800E-03,.2264100E-03,&
     & .1452800E-03,.1570400E-03,.1697500E-03,.1832000E-03,.1973500E-03,&
     & .1241400E-03,.1349600E-03,.1465400E-03,.1589900E-03,.1721100E-03,&
     & .1056700E-03,.1154700E-03,.1261800E-03,.1376500E-03,.1499400E-03,&
     & .8977400E-04,.9875100E-04,.1085200E-03,.1191800E-03,.1305700E-03,&
     & .7623200E-04,.8438300E-04,.9333600E-04,.1031400E-03,.1138100E-03,&
     & .6437000E-04,.7168400E-04,.7979700E-04,.8885000E-04,.9873300E-04,&
     & .5412500E-04,.6068000E-04,.6804200E-04,.7631300E-04,.8543200E-04,&
     & .4537400E-04,.5128000E-04,.5793200E-04,.6542900E-04,.7385200E-04,&
     & .3786100E-04,.4307800E-04,.4901600E-04,.5584400E-04,.6353800E-04,&
     & .3134200E-04,.3587900E-04,.4117500E-04,.4725200E-04,.5428600E-04,&
     & .2582100E-04,.2977700E-04,.3445800E-04,.3988000E-04,.4621800E-04,&
     & .2118000E-04,.2461700E-04,.2871900E-04,.3350700E-04,.3918400E-04,&
     & .1737700E-04,.2034500E-04,.2391600E-04,.2817700E-04,.3323700E-04,&
     & .1422300E-04,.1679000E-04,.1986800E-04,.2365600E-04,.2816300E-04,&
     & .1159300E-04,.1378800E-04,.1647900E-04,.1977300E-04,.2376500E-04,&
     & .9414100E-05,.1127900E-04,.1359100E-04,.1644700E-04,.1996900E-04,&
     & .7662200E-05,.9241700E-05,.1123300E-04,.1370900E-04,.1681900E-04,&
     & .6231500E-05,.7567100E-05,.9269900E-05,.1142900E-04,.1418600E-04,&
     & .5047300E-05,.6176600E-05,.7631600E-05,.9497900E-05,.1189000E-04,&
     & .4077300E-05,.5024200E-05,.6254000E-05,.7859400E-05,.9937800E-05,&
     & .3293700E-05,.4086200E-05,.5122800E-05,.6499200E-05,.8311200E-05,&
     & .2759300E-05,.3450500E-05,.4371600E-05,.5603500E-05,.7260900E-05/

      data   absb(  1:120, 8)   /                                       &
     & .2599000E-01,.2599700E-01,.2602400E-01,.2603800E-01,.2613200E-01,&
     & .2264800E-01,.2273100E-01,.2280300E-01,.2286100E-01,.2302000E-01,&
     & .1964100E-01,.1981200E-01,.1988000E-01,.1999800E-01,.2014100E-01,&
     & .1690900E-01,.1703000E-01,.1714800E-01,.1729600E-01,.1750200E-01,&
     & .1436500E-01,.1450400E-01,.1464300E-01,.1480800E-01,.1499600E-01,&
     & .1207200E-01,.1220700E-01,.1234600E-01,.1251300E-01,.1274200E-01,&
     & .1013500E-01,.1026300E-01,.1040200E-01,.1060200E-01,.1079400E-01,&
     & .8650800E-02,.8795100E-02,.8976500E-02,.9153000E-02,.9344700E-02,&
     & .7422800E-02,.7596400E-02,.7780500E-02,.7944100E-02,.8126600E-02,&
     & .6394900E-02,.6592000E-02,.6758400E-02,.6929800E-02,.7096800E-02,&
     & .5538900E-02,.5716500E-02,.5883400E-02,.6049000E-02,.6216300E-02,&
     & .4798000E-02,.4969600E-02,.5133300E-02,.5305500E-02,.5469500E-02,&
     & .4167600E-02,.4335700E-02,.4499900E-02,.4674900E-02,.4817900E-02,&
     & .3639100E-02,.3807900E-02,.3983400E-02,.4137400E-02,.4275400E-02,&
     & .3191700E-02,.3358000E-02,.3523700E-02,.3673700E-02,.3798200E-02,&
     & .2809600E-02,.2970900E-02,.3124200E-02,.3261300E-02,.3382900E-02,&
     & .2478600E-02,.2627200E-02,.2770600E-02,.2898600E-02,.3013000E-02,&
     & .2190700E-02,.2328200E-02,.2457400E-02,.2577800E-02,.2680600E-02,&
     & .1934600E-02,.2056900E-02,.2175700E-02,.2280500E-02,.2374200E-02,&
     & .1710100E-02,.1824700E-02,.1930800E-02,.2025600E-02,.2117600E-02,&
     & .1517100E-02,.1616800E-02,.1712100E-02,.1803500E-02,.1891100E-02,&
     & .1346600E-02,.1439000E-02,.1524600E-02,.1608300E-02,.1692100E-02,&
     & .1187100E-02,.1269800E-02,.1353400E-02,.1432000E-02,.1507100E-02,&
     & .1038800E-02,.1117900E-02,.1193400E-02,.1264700E-02,.1339100E-02/

      data   absb(121:235, 8)   /                                       &
     & .9223300E-03,.9985400E-03,.1072200E-02,.1142600E-02,.1211800E-02,&
     & .8184300E-03,.8923600E-03,.9660500E-03,.1034000E-02,.1100300E-02,&
     & .7273800E-03,.7977700E-03,.8699700E-03,.9349600E-03,.9985200E-03,&
     & .6514900E-03,.7209800E-03,.7923500E-03,.8611600E-03,.9247400E-03,&
     & .5831800E-03,.6515400E-03,.7216000E-03,.7940000E-03,.8597300E-03,&
     & .5212700E-03,.5877000E-03,.6578100E-03,.7311000E-03,.8016500E-03,&
     & .4640400E-03,.5296000E-03,.5988600E-03,.6736600E-03,.7472700E-03,&
     & .4122900E-03,.4760900E-03,.5442400E-03,.6191800E-03,.6936600E-03,&
     & .3653400E-03,.4262400E-03,.4937400E-03,.5666800E-03,.6436400E-03,&
     & .3213400E-03,.3798700E-03,.4459700E-03,.5166900E-03,.5946900E-03,&
     & .2785500E-03,.3353100E-03,.3985500E-03,.4683200E-03,.5456200E-03,&
     & .2409400E-03,.2944000E-03,.3547700E-03,.4228400E-03,.4976700E-03,&
     & .2071000E-03,.2571200E-03,.3142500E-03,.3809800E-03,.4546500E-03,&
     & .1775700E-03,.2242200E-03,.2790600E-03,.3423100E-03,.4141900E-03,&
     & .1519800E-03,.1953800E-03,.2475800E-03,.3079600E-03,.3776500E-03,&
     & .1294300E-03,.1690700E-03,.2181200E-03,.2758500E-03,.3428400E-03,&
     & .1094600E-03,.1456000E-03,.1910300E-03,.2463300E-03,.3104900E-03,&
     & .9295000E-04,.1259900E-03,.1684800E-03,.2210900E-03,.2826600E-03,&
     & .7872200E-04,.1088300E-03,.1483100E-03,.1979000E-03,.2577000E-03,&
     & .6632300E-04,.9365100E-04,.1298300E-03,.1763800E-03,.2339300E-03,&
     & .5564400E-04,.8008500E-04,.1134900E-03,.1574100E-03,.2127400E-03,&
     & .4653700E-04,.6844200E-04,.9919000E-04,.1404500E-03,.1932700E-03,&
     & .4155800E-04,.6250300E-04,.9237700E-04,.1325800E-03,.1852800E-03/

      data   absb(  1:120, 9)   /                                       &
     & .1913712E+00,.1940307E+00,.1961583E+00,.1981395E+00,.1992614E+00,&
     & .1688802E+00,.1712855E+00,.1734248E+00,.1752061E+00,.1762731E+00,&
     & .1486065E+00,.1508840E+00,.1528820E+00,.1544423E+00,.1558272E+00,&
     & .1299204E+00,.1323469E+00,.1341015E+00,.1357898E+00,.1371436E+00,&
     & .1134080E+00,.1156327E+00,.1174880E+00,.1190475E+00,.1205541E+00,&
     & .9849405E-01,.1005410E+00,.1023631E+00,.1040376E+00,.1053612E+00,&
     & .8553764E-01,.8752614E-01,.8929406E-01,.9077432E-01,.9251699E-01,&
     & .7472428E-01,.7670302E-01,.7818691E-01,.7985497E-01,.8155359E-01,&
     & .6533778E-01,.6717463E-01,.6889333E-01,.7050656E-01,.7229544E-01,&
     & .5740823E-01,.5919933E-01,.6095743E-01,.6261078E-01,.6444982E-01,&
     & .5056393E-01,.5229127E-01,.5411951E-01,.5593955E-01,.5779249E-01,&
     & .4478827E-01,.4654763E-01,.4835743E-01,.5018840E-01,.5221163E-01,&
     & .3995582E-01,.4170732E-01,.4352502E-01,.4537017E-01,.4745831E-01,&
     & .3584681E-01,.3763708E-01,.3940778E-01,.4140061E-01,.4366003E-01,&
     & .3241328E-01,.3414397E-01,.3597457E-01,.3798590E-01,.4030971E-01,&
     & .2947190E-01,.3126308E-01,.3314326E-01,.3529197E-01,.3770177E-01,&
     & .2705105E-01,.2882380E-01,.3077801E-01,.3304753E-01,.3559583E-01,&
     & .2495554E-01,.2671043E-01,.2883204E-01,.3115120E-01,.3377072E-01,&
     & .2314399E-01,.2503929E-01,.2721446E-01,.2967647E-01,.3240495E-01,&
     & .2168424E-01,.2363375E-01,.2593698E-01,.2846534E-01,.3128868E-01,&
     & .2052923E-01,.2253442E-01,.2489053E-01,.2751815E-01,.3055501E-01,&
     & .1954963E-01,.2161307E-01,.2412847E-01,.2681518E-01,.3006419E-01,&
     & .1858710E-01,.2078212E-01,.2330820E-01,.2617072E-01,.2946078E-01,&
     & .1751615E-01,.1979783E-01,.2246608E-01,.2526516E-01,.2885677E-01/

      data   absb(121:235, 9)   /                                       &
     & .1645401E-01,.1873164E-01,.2142687E-01,.2424787E-01,.2779912E-01,&
     & .1549653E-01,.1781459E-01,.2042772E-01,.2337789E-01,.2680098E-01,&
     & .1464498E-01,.1698778E-01,.1958951E-01,.2255141E-01,.2592896E-01,&
     & .1374930E-01,.1598681E-01,.1858564E-01,.2145636E-01,.2483064E-01,&
     & .1297376E-01,.1509265E-01,.1767753E-01,.2043917E-01,.2378184E-01,&
     & .1225680E-01,.1431909E-01,.1681124E-01,.1957355E-01,.2273600E-01,&
     & .1154426E-01,.1346806E-01,.1585020E-01,.1858172E-01,.2161718E-01,&
     & .1073729E-01,.1273625E-01,.1500546E-01,.1763766E-01,.2052634E-01,&
     & .1003661E-01,.1202377E-01,.1415118E-01,.1671879E-01,.1956038E-01,&
     & .9280320E-02,.1124463E-01,.1334817E-01,.1580509E-01,.1857998E-01,&
     & .8562651E-02,.1045470E-01,.1255035E-01,.1487256E-01,.1749111E-01,&
     & .7862421E-02,.9650932E-02,.1175305E-01,.1399227E-01,.1656834E-01,&
     & .7176180E-02,.8968905E-02,.1094204E-01,.1318820E-01,.1566044E-01,&
     & .6553611E-02,.8323164E-02,.1026176E-01,.1243120E-01,.1484228E-01,&
     & .6016277E-02,.7685907E-02,.9617875E-02,.1169791E-01,.1410813E-01,&
     & .5488131E-02,.7123060E-02,.9006212E-02,.1103169E-01,.1339416E-01,&
     & .5039153E-02,.6572314E-02,.8368057E-02,.1042322E-01,.1261228E-01,&
     & .4678553E-02,.6104245E-02,.7858028E-02,.9862751E-02,.1203004E-01,&
     & .4397629E-02,.5698528E-02,.7389611E-02,.9348370E-02,.1154387E-01,&
     & .4125069E-02,.5371631E-02,.6928103E-02,.8831411E-02,.1098275E-01,&
     & .3904687E-02,.5099382E-02,.6546356E-02,.8396888E-02,.1047269E-01,&
     & .3677700E-02,.4862089E-02,.6220196E-02,.7951650E-02,.1006273E-01,&
     & .3646869E-02,.4855547E-02,.6218200E-02,.7940987E-02,.1000532E-01/

      data   absb(  1:120,10)   /                                       &
     & .6563434E+01,.6536173E+01,.6509225E+01,.6480775E+01,.6453855E+01,&
     & .6694993E+01,.6668788E+01,.6641684E+01,.6613791E+01,.6586857E+01,&
     & .6814067E+01,.6788196E+01,.6761389E+01,.6734157E+01,.6706224E+01,&
     & .6922709E+01,.6896320E+01,.6870266E+01,.6842505E+01,.6814432E+01,&
     & .7018842E+01,.6992813E+01,.6966297E+01,.6939182E+01,.6910339E+01,&
     & .7104680E+01,.7079554E+01,.7053120E+01,.7025060E+01,.6997053E+01,&
     & .7178134E+01,.7153173E+01,.7126816E+01,.7099394E+01,.7069733E+01,&
     & .7237652E+01,.7212460E+01,.7187023E+01,.7158873E+01,.7129293E+01,&
     & .7288276E+01,.7263276E+01,.7236743E+01,.7208943E+01,.7178798E+01,&
     & .7329577E+01,.7304627E+01,.7277602E+01,.7249474E+01,.7218919E+01,&
     & .7364634E+01,.7339549E+01,.7311831E+01,.7282930E+01,.7252217E+01,&
     & .7393382E+01,.7367435E+01,.7340013E+01,.7310737E+01,.7279210E+01,&
     & .7416720E+01,.7390784E+01,.7362566E+01,.7332879E+01,.7300963E+01,&
     & .7435429E+01,.7408936E+01,.7380806E+01,.7350034E+01,.7317321E+01,&
     & .7450559E+01,.7423883E+01,.7395076E+01,.7363982E+01,.7330904E+01,&
     & .7462543E+01,.7435279E+01,.7405991E+01,.7374365E+01,.7340536E+01,&
     & .7471315E+01,.7443887E+01,.7414078E+01,.7381879E+01,.7347478E+01,&
     & .7478800E+01,.7450609E+01,.7419846E+01,.7387182E+01,.7352811E+01,&
     & .7483666E+01,.7454950E+01,.7423611E+01,.7390401E+01,.7355051E+01,&
     & .7487253E+01,.7457765E+01,.7425717E+01,.7391895E+01,.7356050E+01,&
     & .7488506E+01,.7458765E+01,.7426508E+01,.7392054E+01,.7355156E+01,&
     & .7489850E+01,.7459250E+01,.7426035E+01,.7391305E+01,.7353645E+01,&
     & .7492443E+01,.7461624E+01,.7428244E+01,.7392628E+01,.7354878E+01,&
     & .7497726E+01,.7466429E+01,.7432747E+01,.7397279E+01,.7358181E+01/

      data   absb(121:235,10)   /                                       &
     & .7505737E+01,.7474800E+01,.7440929E+01,.7405596E+01,.7366801E+01,&
     & .7513970E+01,.7482707E+01,.7449119E+01,.7413672E+01,.7375439E+01,&
     & .7520741E+01,.7489941E+01,.7456502E+01,.7420983E+01,.7383218E+01,&
     & .7530440E+01,.7499850E+01,.7466874E+01,.7431912E+01,.7393870E+01,&
     & .7538723E+01,.7509468E+01,.7476827E+01,.7442592E+01,.7405041E+01,&
     & .7547204E+01,.7518633E+01,.7486628E+01,.7452286E+01,.7415815E+01,&
     & .7556213E+01,.7528691E+01,.7497855E+01,.7463955E+01,.7428134E+01,&
     & .7565953E+01,.7538872E+01,.7508898E+01,.7475761E+01,.7440938E+01,&
     & .7575062E+01,.7548586E+01,.7519907E+01,.7487622E+01,.7453098E+01,&
     & .7584131E+01,.7558667E+01,.7530653E+01,.7499618E+01,.7465741E+01,&
     & .7593779E+01,.7569486E+01,.7542361E+01,.7512145E+01,.7479280E+01,&
     & .7602657E+01,.7580065E+01,.7553432E+01,.7524179E+01,.7492239E+01,&
     & .7610916E+01,.7589199E+01,.7564289E+01,.7535843E+01,.7505070E+01,&
     & .7618668E+01,.7597891E+01,.7574135E+01,.7546664E+01,.7516426E+01,&
     & .7625181E+01,.7605859E+01,.7582588E+01,.7556608E+01,.7527157E+01,&
     & .7630829E+01,.7612911E+01,.7590839E+01,.7566252E+01,.7537672E+01,&
     & .7636310E+01,.7619845E+01,.7598957E+01,.7575011E+01,.7547925E+01,&
     & .7640632E+01,.7625839E+01,.7606042E+01,.7583011E+01,.7556788E+01,&
     & .7644312E+01,.7630300E+01,.7612179E+01,.7589767E+01,.7564669E+01,&
     & .7647263E+01,.7634539E+01,.7617801E+01,.7596900E+01,.7572581E+01,&
     & .7650044E+01,.7638295E+01,.7623237E+01,.7603393E+01,.7580008E+01,&
     & .7651936E+01,.7641531E+01,.7627671E+01,.7609561E+01,.7586309E+01,&
     & .7652501E+01,.7642169E+01,.7629030E+01,.7611100E+01,.7588572E+01/

!  ---
      data   forref(1:4,1:10)    /                                      &
     & .2145040E-06,.4604180E-06,.3576080E-05,.1920370E-05,.1425760E-05,&
     & .3644630E-05,.1170330E-04,.1120850E-04,.1015360E-04,.1240960E-04,&
     & .5091900E-04,.5652820E-04,.1433940E-03,.1547000E-03,.4664980E-03,&
     & .9188290E-03,.2516310E-02,.2417290E-02,.2400570E-02,.3504080E-02,&
     & .4103090E-02,.4168510E-02,.3909250E-02,.3836940E-02,.4453870E-02,&
     & .4486570E-02,.4323100E-02,.3707390E-02,.4581500E-02,.4600140E-02,&
     & .4502450E-02,.3367180E-02,.4682248E-02,.4680093E-02,.4684726E-02,&
     & .3680135E-02,.5243173E-02,.5159096E-02,.5132896E-02,.3931064E-02/


      data   selfref(1:10,1:10)  /                                      &
     & .2170580E-03,.1763910E-03,.1433420E-03,.1164860E-03,.9466140E-04,&
     & .7692570E-04,.6251310E-04,.5080070E-04,.4128280E-04,.3354810E-04,&
     & .5980550E-03,.4848050E-03,.3930000E-03,.3185800E-03,.2582520E-03,&
     & .2093480E-03,.1697050E-03,.1375690E-03,.1115180E-03,.9040080E-04,&
     & .1026910E-02,.9302810E-03,.8427400E-03,.7634370E-03,.6915960E-03,&
     & .6265160E-03,.5675600E-03,.5141520E-03,.4657690E-03,.4219400E-03,&
     & .3885690E-02,.3650980E-02,.3430450E-02,.3223240E-02,.3028540E-02,&
     & .2845610E-02,.2673720E-02,.2512220E-02,.2360470E-02,.2217890E-02,&
     & .3498450E-01,.3266780E-01,.3050450E-01,.2848450E-01,.2659820E-01,&
     & .2483690E-01,.2319210E-01,.2165630E-01,.2022220E-01,.1888310E-01,&
     & .6137050E-01,.5626760E-01,.5158900E-01,.4729940E-01,.4336650E-01,&
     & .3976060E-01,.3645450E-01,.3342330E-01,.3064420E-01,.2809610E-01,&
     & .6569810E-01,.6026600E-01,.5528300E-01,.5071200E-01,.4651900E-01,&
     & .4267260E-01,.3914430E-01,.3590770E-01,.3293870E-01,.3021530E-01,&
     & .6717820E-01,.6164610E-01,.5656950E-01,.5191100E-01,.4763610E-01,&
     & .4371320E-01,.4011340E-01,.3681000E-01,.3377870E-01,.3099700E-01,&
     & .6790842E-01,.6238861E-01,.5731740E-01,.5265845E-01,.4837825E-01,&
     & .4444589E-01,.4083320E-01,.3751416E-01,.3446487E-01,.3166347E-01,&
     & .7454175E-01,.6857684E-01,.6309064E-01,.5804463E-01,.5340338E-01,&
     & .4913438E-01,.4520762E-01,.4159562E-01,.3827306E-01,.3521673E-01/

!  ---  ch4
      data   absch4(1:10)        /                                      &
     & .1013810E-02,.6336920E-02,.1941850E-01,.4832100E-01,.2365740E-02,&
     & .6619730E-03,.5645520E-03,.2831830E-03,.6714756E-04,.2647642E-06/

!........................................!
      end module module_radsw_kgb20      !
!========================================!


!> This module sets up absorption coeffients for band 21: 6150-7700
!! cm-1 (low - h2o,co2; high - h2o,co2)
!========================================!
      module module_radsw_kgb21          !
!........................................!
!
! *********        the original program descriptions        ********* !
!                                                                     !
!     originally by j.delamere, atmospheric & environmental research. !
!     revision: 2.4                                                   !
!     band 21:  6150-7700 cm-1 (low - j2o,co2; high - h2o,co2)        !
!     reformatted for f90 by jjmorcrette, ecmwf                       !
!                                                                     !
!     this table has been re-generated for reduced number of g-point  !
!     by y.t.hou, ncep                                                !
!                                                                     !
! *********    *********    end description    *********    ********* !
!
      use physparam,               only : kind_phys
      use module_radsw_parameters, only : NG21

!
      implicit none
!
      private
!
!> msa21=585
      integer,  public :: MSA21
!> msb21=1175
      integer,  public :: MSB21
!> msf21=10
      integer,  public :: MSF21
!> mfr21=4
      integer,  public :: MFR21
      parameter (MSA21=585, MSB21=1175, MSF21=10, MFR21=4)

      real (kind=kind_phys), public :: forref(MFR21,NG21)

!>    the array absa(585,NG21) (ka((9,5,13,NG21)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels> ~100mb,
!!    temperatures, and binary species parameters (see taumol.f for definition).
!!    the first index in the array, js, runs from 1 to 9, and corresponds to
!!    different values of the binary species parameter.  for instance,
!!    js=1 refers to dry air, js = 2 corresponds to the paramter value 1/8,
!!    js = 3 corresponds to the parameter value 2/8, etc.  the second index
!!    in the array, jt, which runs from 1 to 5, corresponds to different
!!    temperatures.  more specifically, jt = 3 means that the data are for
!!    the reference temperature tref for this  pressure level, jt = 2 refers
!!    to tref-15, jt = 1 is for tref-30, jt = 4 is for tref+15, and jt = 5
!!    is for tref+30.  the third index, jp, runs from 1 to 13 and refers
!!    to the jpth reference pressure level (see taumol.f for these levels
!!    in mb).  the fourth index, ig, goes from 1 to 10, and indicates
!!    which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absa(MSA21,NG21)

!>    the array absb(1175,10) (kb(5,5,13:59,10)) contains absorption coefs at
!!    the 16 chosen g-values for a range of pressure levels < ~100mb and
!!    temperatures. the first index in the array, jt, which runs from 1 to 5,
!!    corresponds to different temperatures.  more specifically, jt = 3 means
!!    that the data are for the reference temperature tref for this pressure
!!    level, jt = 2 refers to the temperature tref-15, jt = 1 is for
!!    tref-30, jt = 4 is for tref+15, and jt = 5 is for tref+30.
!!    the second index, jp, runs from 13 to 59 and refers to the jpth
!!    reference pressure level (see taumol.f for the value of these
!!    pressure levels in mb).  the third index, ig, goes from 1 to 10,
!!    and tells us which g-interval the absorption coefficients are for.
      real (kind=kind_phys), public :: absb(MSB21,NG21)

!>    the array selfref contains the coefficient of the water vapor
!!    self-continuum (including the energy term).  the first index
!!    refers to temperature in 7.2 degree increments.  for instance,
!!    jt = 1 refers to a temperature of 245.6, jt = 2 refers to 252.8,
!!    etc.  the second index runs over the g-channel (1 to 10).
      real (kind=kind_phys), public :: selfref(MSF21,NG21)

!> rayleigh extinction coefficient at \f$v=6925cm^{-1}\f$
      real (kind=kind_phys), parameter, public :: rayl = 9.41e-09


      data   absa(  1:180, 1)   /                                       &
     & .3148200E-07,.6401000E-05,.9501700E-05,.1173500E-04,.1356100E-04,&
     & .1507700E-04,.1612100E-04,.1613800E-04,.4168700E-06,.2979100E-07,&
     & .6820600E-05,.1026000E-04,.1276400E-04,.1488000E-04,.1666500E-04,&
     & .1794400E-04,.1817000E-04,.4424600E-06,.2827200E-07,.7244800E-05,&
     & .1108300E-04,.1386600E-04,.1625600E-04,.1832300E-04,.1988600E-04,&
     & .2022900E-04,.4747200E-06,.2690000E-07,.7693500E-05,.1193400E-04,&
     & .1501500E-04,.1765700E-04,.2003800E-04,.2190200E-04,.2236100E-04,&
     & .5077000E-06,.2565600E-07,.8166100E-05,.1276900E-04,.1618100E-04,&
     & .1912700E-04,.2175500E-04,.2396300E-04,.2442100E-04,.5420500E-06,&
     & .2639600E-07,.5222500E-05,.7722100E-05,.9515000E-05,.1100400E-04,&
     & .1219500E-04,.1303700E-04,.1298500E-04,.3209700E-06,.2494500E-07,&
     & .5573700E-05,.8361200E-05,.1037500E-04,.1211600E-04,.1356900E-04,&
     & .1456100E-04,.1471100E-04,.3457900E-06,.2364600E-07,.5933300E-05,&
     & .9053100E-05,.1130100E-04,.1324700E-04,.1495600E-04,.1619300E-04,&
     & .1639600E-04,.3720000E-06,.2247600E-07,.6314400E-05,.9773200E-05,&
     & .1227400E-04,.1443600E-04,.1637700E-04,.1787900E-04,.1822300E-04,&
     & .4005100E-06,.2141500E-07,.6713300E-05,.1047100E-04,.1326400E-04,&
     & .1568100E-04,.1784300E-04,.1962700E-04,.1995400E-04,.4292900E-06,&
     & .2239700E-07,.4193000E-05,.6155900E-05,.7554400E-05,.8690200E-05,&
     & .9589600E-05,.1026900E-04,.1011400E-04,.2414800E-06,.2112400E-07,&
     & .4483700E-05,.6678600E-05,.8247500E-05,.9635300E-05,.1074200E-04,&
     & .1146900E-04,.1155600E-04,.2630400E-06,.1998800E-07,.4783900E-05,&
     & .7249900E-05,.9022900E-05,.1056000E-04,.1190000E-04,.1285900E-04,&
     & .1293500E-04,.2861800E-06,.1896800E-07,.5104000E-05,.7858600E-05,&
     & .9830800E-05,.1155100E-04,.1309400E-04,.1424000E-04,.1445500E-04,&
     & .3098300E-06,.1804700E-07,.5436600E-05,.8446200E-05,.1067000E-04,&
     & .1258500E-04,.1431300E-04,.1569500E-04,.1593400E-04,.3325500E-06,&
     & .1908000E-07,.3340900E-05,.4873800E-05,.5935400E-05,.6787700E-05,&
     & .7490000E-05,.7914200E-05,.7720000E-05,.1846600E-06,.1795400E-07,&
     & .3591000E-05,.5296000E-05,.6503700E-05,.7574400E-05,.8408200E-05,&
     & .8942200E-05,.8930200E-05,.2029200E-06,.1695400E-07,.3831700E-05,&
     & .5753600E-05,.7132200E-05,.8342400E-05,.9378800E-05,.1006300E-04,&
     & .1009800E-04,.2217300E-06,.1605900E-07,.4093200E-05,.6258600E-05,&
     & .7796100E-05,.9147500E-05,.1034700E-04,.1122200E-04,.1131200E-04,&
     & .2400500E-06,.1525400E-07,.4367800E-05,.6752500E-05,.8487300E-05,&
     & .9999500E-05,.1135300E-04,.1241600E-04,.1258600E-04,.2593900E-06/

      data   absa(181:315, 1)   /                                       &
     & .1626100E-07,.2654000E-05,.3854100E-05,.4644300E-05,.5296100E-05,&
     & .5805400E-05,.6071800E-05,.5854000E-05,.1427400E-06,.1526500E-07,&
     & .2859800E-05,.4183100E-05,.5122100E-05,.5911000E-05,.6520700E-05,&
     & .6976100E-05,.6868500E-05,.1572500E-06,.1438400E-07,.3062300E-05,&
     & .4552800E-05,.5612400E-05,.6565000E-05,.7343600E-05,.7820800E-05,&
     & .7852300E-05,.1727100E-06,.1359900E-07,.3272000E-05,.4962200E-05,&
     & .6160500E-05,.7224400E-05,.8158200E-05,.8797100E-05,.8809600E-05,&
     & .1876600E-06,.1289500E-07,.3498500E-05,.5383300E-05,.6724000E-05,&
     & .7909100E-05,.8968800E-05,.9769800E-05,.9863900E-05,.2037500E-06,&
     & .1389800E-07,.2098100E-05,.3038400E-05,.3604100E-05,.4104000E-05,&
     & .4433400E-05,.4576100E-05,.4395000E-05,.1102500E-06,.1301100E-07,&
     & .2264500E-05,.3289000E-05,.4004200E-05,.4585100E-05,.5047300E-05,&
     & .5325000E-05,.5185400E-05,.1222700E-06,.1223100E-07,.2432200E-05,&
     & .3584200E-05,.4391500E-05,.5125900E-05,.5690300E-05,.6034300E-05,&
     & .6016600E-05,.1350600E-06,.1153900E-07,.2603200E-05,.3910700E-05,&
     & .4832100E-05,.5652500E-05,.6362500E-05,.6807800E-05,.6802400E-05,&
     & .1471900E-06,.1092100E-07,.2786600E-05,.4257800E-05,.5291900E-05,&
     & .6218100E-05,.7036200E-05,.7614000E-05,.7651900E-05,.1605200E-06,&
     & .1188600E-07,.1656300E-05,.2386000E-05,.2827600E-05,.3144900E-05,&
     & .3377100E-05,.3430500E-05,.3282400E-05,.8494300E-07,.1109600E-07,&
     & .1787400E-05,.2586100E-05,.3110000E-05,.3542500E-05,.3873700E-05,&
     & .4046800E-05,.3893300E-05,.9473400E-07,.1040500E-07,.1928600E-05,&
     & .2816600E-05,.3441700E-05,.3978600E-05,.4380400E-05,.4675900E-05,&
     & .4583500E-05,.1053100E-06,.9794800E-08,.2068000E-05,.3074100E-05,&
     & .3780300E-05,.4416300E-05,.4942000E-05,.5247700E-05,.5250900E-05,&
     & .1151700E-06,.9252200E-08,.2214000E-05,.3355800E-05,.4154400E-05,&
     & .4871900E-05,.5496800E-05,.5922500E-05,.5902700E-05,.1261800E-06/

      data   absa(316:450, 1)   /                                       &
     & .1016400E-07,.1305600E-05,.1869000E-05,.2194000E-05,.2421000E-05,&
     & .2550400E-05,.2567600E-05,.2444400E-05,.6602100E-07,.9461200E-08,&
     & .1410600E-05,.2039800E-05,.2413600E-05,.2742400E-05,.2957800E-05,&
     & .3044600E-05,.2918200E-05,.7398500E-07,.8848900E-08,.1525600E-05,&
     & .2212300E-05,.2689400E-05,.3070400E-05,.3380800E-05,.3552100E-05,&
     & .3452600E-05,.8266600E-07,.8311100E-08,.1639600E-05,.2415900E-05,&
     & .2956300E-05,.3450400E-05,.3819600E-05,.4042500E-05,.4009300E-05,&
     & .9094000E-07,.7834900E-08,.1757000E-05,.2639800E-05,.3254700E-05,&
     & .3806000E-05,.4281300E-05,.4573200E-05,.4546500E-05,.1002200E-06,&
     & .8698300E-08,.1028400E-05,.1473400E-05,.1697300E-05,.1855100E-05,&
     & .1922300E-05,.1922200E-05,.1828900E-05,.5346000E-07,.8071200E-08,&
     & .1114100E-05,.1601600E-05,.1890200E-05,.2102700E-05,.2250400E-05,&
     & .2280000E-05,.2174900E-05,.6018000E-07,.7528400E-08,.1203300E-05,&
     & .1737600E-05,.2084900E-05,.2371900E-05,.2590600E-05,.2697700E-05,&
     & .2591000E-05,.6688600E-07,.7054100E-08,.1297900E-05,.1895900E-05,&
     & .2309000E-05,.2669500E-05,.2936700E-05,.3123900E-05,.3057400E-05,&
     & .7397800E-07,.6635900E-08,.1393700E-05,.2072000E-05,.2544800E-05,&
     & .2968000E-05,.3320800E-05,.3519100E-05,.3509700E-05,.8172400E-07,&
     & .7400600E-08,.8170500E-06,.1161000E-05,.1332500E-05,.1430500E-05,&
     & .1466800E-05,.1464200E-05,.1376500E-05,.4472400E-07,.6847700E-08,&
     & .8830600E-06,.1264000E-05,.1476800E-05,.1632900E-05,.1720900E-05,&
     & .1730600E-05,.1644100E-05,.5068800E-07,.6371700E-08,.9552200E-06,&
     & .1378300E-05,.1630300E-05,.1854600E-05,.1996300E-05,.2057000E-05,&
     & .1967800E-05,.5605200E-07,.5957600E-08,.1033100E-05,.1497500E-05,&
     & .1818000E-05,.2077000E-05,.2281300E-05,.2402100E-05,.2335000E-05,&
     & .6241200E-07,.5594000E-08,.1110200E-05,.1637000E-05,.1999900E-05,&
     & .2334900E-05,.2590000E-05,.2732800E-05,.2705700E-05,.6946600E-07/

      data   absa(451:585, 1)   /                                       &
     & .6061700E-08,.6716700E-06,.9559400E-06,.1093900E-05,.1176700E-05,&
     & .1205900E-05,.1203200E-05,.1131400E-05,.3709300E-07,.5608700E-08,&
     & .7270400E-06,.1040600E-05,.1213700E-05,.1343400E-05,.1417100E-05,&
     & .1424600E-05,.1352100E-05,.4191800E-07,.5218700E-08,.7868100E-06,&
     & .1133800E-05,.1341900E-05,.1525400E-05,.1642600E-05,.1695000E-05,&
     & .1620300E-05,.4633900E-07,.4879300E-08,.8501300E-06,.1232700E-05,&
     & .1497100E-05,.1710900E-05,.1877600E-05,.1978400E-05,.1924000E-05,&
     & .5172500E-07,.4581500E-08,.9142000E-06,.1342700E-05,.1645800E-05,&
     & .1922100E-05,.2132300E-05,.2249000E-05,.2226300E-05,.5755500E-07,&
     & .4962900E-08,.5522300E-06,.7867700E-06,.8978900E-06,.9677800E-06,&
     & .9914800E-06,.9888300E-06,.9302800E-06,.3061000E-07,.4592000E-08,&
     & .5983200E-06,.8563200E-06,.9973000E-06,.1105200E-05,.1166500E-05,&
     & .1172900E-05,.1112500E-05,.3455300E-07,.4272700E-08,.6476000E-06,&
     & .9331500E-06,.1104100E-05,.1255200E-05,.1352100E-05,.1395600E-05,&
     & .1333300E-05,.3817600E-07,.3994900E-08,.6992600E-06,.1011000E-05,&
     & .1230400E-05,.1408300E-05,.1544200E-05,.1628300E-05,.1583700E-05,&
     & .4264800E-07,.3751000E-08,.7523900E-06,.1100600E-05,.1353500E-05,&
     & .1578500E-05,.1754700E-05,.1850100E-05,.1829800E-05,.4745200E-07,&
     & .4063300E-08,.4537800E-06,.6471900E-06,.7363700E-06,.7957100E-06,&
     & .8150300E-06,.8126400E-06,.7648800E-06,.2508000E-07,.3759600E-08,&
     & .4920800E-06,.7043000E-06,.8189400E-06,.9085600E-06,.9591900E-06,&
     & .9643500E-06,.9141100E-06,.2829300E-07,.3498200E-08,.5325900E-06,&
     & .7637000E-06,.9076100E-06,.1031600E-05,.1111600E-05,.1147200E-05,&
     & .1095600E-05,.3126700E-07,.3270700E-08,.5748800E-06,.8283300E-06,&
     & .1009600E-05,.1157700E-05,.1268800E-05,.1338800E-05,.1301500E-05,&
     & .3493000E-07,.3071000E-08,.6183200E-06,.9025000E-06,.1112000E-05,&
     & .1296200E-05,.1442400E-05,.1520600E-05,.1503200E-05,.3886100E-07/

      data   absa(  1:180, 2)   /                                       &
     & .1421500E-06,.6572600E-04,.1132700E-03,.1532400E-03,.1847300E-03,&
     & .2065800E-03,.2160300E-03,.2061700E-03,.1325200E-04,.1345200E-06,&
     & .6952100E-04,.1205900E-03,.1634200E-03,.1961300E-03,.2198700E-03,&
     & .2294600E-03,.2187900E-03,.1481600E-04,.1276600E-06,.7331300E-04,&
     & .1278600E-03,.1730300E-03,.2075400E-03,.2332600E-03,.2432400E-03,&
     & .2295600E-03,.1653600E-04,.1214700E-06,.7708700E-04,.1352100E-03,&
     & .1826300E-03,.2190400E-03,.2461400E-03,.2563700E-03,.2406000E-03,&
     & .1809200E-04,.1158400E-06,.8094600E-04,.1423300E-03,.1923000E-03,&
     & .2304700E-03,.2586900E-03,.2690400E-03,.2513900E-03,.1976800E-04,&
     & .1191800E-06,.5516700E-04,.9455700E-04,.1276000E-03,.1532200E-03,&
     & .1714800E-03,.1789400E-03,.1704900E-03,.1071000E-04,.1126400E-06,&
     & .5858100E-04,.1011300E-03,.1364800E-03,.1633500E-03,.1828800E-03,&
     & .1905900E-03,.1813100E-03,.1206700E-04,.1067800E-06,.6193500E-04,&
     & .1075600E-03,.1449700E-03,.1733500E-03,.1943700E-03,.2024500E-03,&
     & .1907000E-03,.1355300E-04,.1014900E-06,.6526100E-04,.1140400E-03,&
     & .1533100E-03,.1832300E-03,.2055900E-03,.2138500E-03,.2002400E-03,&
     & .1480100E-04,.9669900E-07,.6869800E-04,.1203800E-03,.1616700E-03,&
     & .1930800E-03,.2163500E-03,.2247900E-03,.2096400E-03,.1624500E-04,&
     & .1011200E-06,.4539900E-04,.7722600E-04,.1039700E-03,.1246200E-03,&
     & .1395600E-03,.1453100E-03,.1381600E-03,.8363500E-05,.9538100E-07,&
     & .4835700E-04,.8298600E-04,.1117300E-03,.1333800E-03,.1489800E-03,&
     & .1553400E-03,.1478600E-03,.9491500E-05,.9025700E-07,.5135500E-04,&
     & .8872700E-04,.1192000E-03,.1421500E-03,.1585900E-03,.1654100E-03,&
     & .1562200E-03,.1068300E-04,.8565300E-07,.5432200E-04,.9436900E-04,&
     & .1264600E-03,.1507200E-03,.1683800E-03,.1753600E-03,.1643300E-03,&
     & .1176400E-04,.8149200E-07,.5730000E-04,.9994000E-04,.1335500E-03,&
     & .1592900E-03,.1780000E-03,.1849600E-03,.1723400E-03,.1297900E-04,&
     & .8613200E-07,.3691000E-04,.6234600E-04,.8362800E-04,.1002500E-03,&
     & .1122400E-03,.1170600E-03,.1110900E-03,.6471500E-05,.8106200E-07,&
     & .3949000E-04,.6727400E-04,.9038400E-04,.1078500E-03,.1202600E-03,&
     & .1254300E-03,.1194400E-03,.7399100E-05,.7655300E-07,.4205500E-04,&
     & .7226900E-04,.9695600E-04,.1152800E-03,.1283300E-03,.1339600E-03,&
     & .1269700E-03,.8389900E-05,.7251600E-07,.4469600E-04,.7723500E-04,&
     & .1032100E-03,.1227600E-03,.1367400E-03,.1426000E-03,.1338400E-03,&
     & .9305200E-05,.6888100E-07,.4727300E-04,.8211800E-04,.1093500E-03,&
     & .1301600E-03,.1450600E-03,.1508000E-03,.1406200E-03,.1031400E-04/

      data   absa(181:315, 2)   /                                       &
     & .7340100E-07,.2977200E-04,.4995700E-04,.6672000E-04,.8010400E-04,&
     & .8976100E-04,.9385300E-04,.8895300E-04,.5017100E-05,.6891600E-07,&
     & .3197600E-04,.5421700E-04,.7257000E-04,.8667100E-04,.9665300E-04,&
     & .1007300E-03,.9587500E-04,.5774600E-05,.6494500E-07,.3421900E-04,&
     & .5841700E-04,.7832700E-04,.9295000E-04,.1035800E-03,.1080300E-03,&
     & .1027400E-03,.6619700E-05,.6140500E-07,.3645700E-04,.6269400E-04,&
     & .8375600E-04,.9935600E-04,.1105800E-03,.1154200E-03,.1085800E-03,&
     & .7372900E-05,.5822900E-07,.3871300E-04,.6705900E-04,.8903300E-04,&
     & .1057300E-03,.1176400E-03,.1224300E-03,.1144000E-03,.8214900E-05,&
     & .6272300E-07,.2375000E-04,.3964300E-04,.5279600E-04,.6347800E-04,&
     & .7115100E-04,.7464000E-04,.7019300E-04,.3889900E-05,.5873500E-07,&
     & .2565700E-04,.4327400E-04,.5771900E-04,.6912400E-04,.7737500E-04,&
     & .8049800E-04,.7654300E-04,.4483300E-05,.5522200E-07,.2756300E-04,&
     & .4686600E-04,.6269900E-04,.7449400E-04,.8294600E-04,.8665300E-04,&
     & .8250900E-04,.5163500E-05,.5210300E-07,.2952800E-04,.5047000E-04,&
     & .6747300E-04,.7989900E-04,.8900400E-04,.9278600E-04,.8762800E-04,&
     & .5810400E-05,.4931700E-07,.3144400E-04,.5417500E-04,.7200000E-04,&
     & .8532200E-04,.9488500E-04,.9882600E-04,.9264800E-04,.6499700E-05,&
     & .5363400E-07,.1886200E-04,.3132500E-04,.4157500E-04,.5014100E-04,&
     & .5619000E-04,.5888200E-04,.5502600E-04,.2979500E-05,.5008400E-07,&
     & .2047400E-04,.3434200E-04,.4573200E-04,.5488400E-04,.6149400E-04,&
     & .6418700E-04,.6085700E-04,.3458900E-05,.4697300E-07,.2206300E-04,&
     & .3741100E-04,.4989300E-04,.5947500E-04,.6627900E-04,.6912900E-04,&
     & .6575800E-04,.3998300E-05,.4422400E-07,.2376400E-04,.4048100E-04,&
     & .5410700E-04,.6402600E-04,.7135000E-04,.7432800E-04,.7051300E-04,&
     & .4552600E-05,.4177700E-07,.2543400E-04,.4355500E-04,.5795800E-04,&
     & .6859900E-04,.7630400E-04,.7945600E-04,.7464900E-04,.5104600E-05/

      data   absa(316:450, 2)   /                                       &
     & .4585700E-07,.1494500E-04,.2459400E-04,.3269600E-04,.3957100E-04,&
     & .4434800E-04,.4609000E-04,.4303400E-04,.2283300E-05,.4269700E-07,&
     & .1626500E-04,.2719800E-04,.3610800E-04,.4338100E-04,.4864200E-04,&
     & .5094300E-04,.4784800E-04,.2666900E-05,.3994400E-07,.1761600E-04,&
     & .2977500E-04,.3957700E-04,.4736600E-04,.5279100E-04,.5499200E-04,&
     & .5223600E-04,.3093200E-05,.3752200E-07,.1904200E-04,.3235100E-04,&
     & .4312600E-04,.5115300E-04,.5698700E-04,.5931600E-04,.5643800E-04,&
     & .3566600E-05,.3537600E-07,.2047600E-04,.3494600E-04,.4652100E-04,&
     & .5500900E-04,.6122100E-04,.6371400E-04,.6003500E-04,.4003700E-05,&
     & .3923300E-07,.1181700E-04,.1925400E-04,.2559600E-04,.3084300E-04,&
     & .3457400E-04,.3603500E-04,.3342500E-04,.1767300E-05,.3641800E-07,&
     & .1291400E-04,.2140400E-04,.2837200E-04,.3416100E-04,.3832100E-04,&
     & .4005300E-04,.3743500E-04,.2075100E-05,.3397900E-07,.1402600E-04,&
     & .2357400E-04,.3131600E-04,.3751400E-04,.4197700E-04,.4374700E-04,&
     & .4134600E-04,.2415100E-05,.3184400E-07,.1520500E-04,.2577000E-04,&
     & .3423100E-04,.4072300E-04,.4531500E-04,.4720200E-04,.4486900E-04,&
     & .2815100E-05,.2996100E-07,.1642000E-04,.2795500E-04,.3718700E-04,&
     & .4394100E-04,.4887300E-04,.5085600E-04,.4817000E-04,.3173300E-05,&
     & .3337300E-07,.9350500E-05,.1521600E-04,.2019800E-04,.2424400E-04,&
     & .2707700E-04,.2818600E-04,.2599600E-04,.1394100E-05,.3089300E-07,&
     & .1028400E-04,.1690200E-04,.2246800E-04,.2707200E-04,.3038200E-04,&
     & .3160300E-04,.2945600E-04,.1635200E-05,.2875400E-07,.1120700E-04,&
     & .1871500E-04,.2484900E-04,.2981500E-04,.3337900E-04,.3490100E-04,&
     & .3283600E-04,.1918800E-05,.2689200E-07,.1217200E-04,.2057800E-04,&
     & .2726900E-04,.3253600E-04,.3618200E-04,.3770300E-04,.3575000E-04,&
     & .2246700E-05,.2525500E-07,.1319500E-04,.2242200E-04,.2978100E-04,&
     & .3522000E-04,.3915700E-04,.4071900E-04,.3872900E-04,.2553500E-05/

      data   absa(451:585, 2)   /                                       &
     & .2733500E-07,.7749100E-05,.1260700E-04,.1669500E-04,.2009300E-04,&
     & .2243100E-04,.2334500E-04,.2152200E-04,.1156900E-05,.2530300E-07,&
     & .8532200E-05,.1402200E-04,.1859300E-04,.2237800E-04,.2513300E-04,&
     & .2616100E-04,.2437900E-04,.1370900E-05,.2355100E-07,.9291200E-05,&
     & .1553600E-04,.2061800E-04,.2472200E-04,.2764700E-04,.2887400E-04,&
     & .2719600E-04,.1609900E-05,.2202500E-07,.1011700E-04,.1711500E-04,&
     & .2264500E-04,.2697400E-04,.2994700E-04,.3121000E-04,.2958100E-04,&
     & .1884500E-05,.2068400E-07,.1097700E-04,.1864500E-04,.2473400E-04,&
     & .2920300E-04,.3245300E-04,.3373300E-04,.3205600E-04,.2132600E-05,&
     & .2238000E-07,.6414400E-05,.1043600E-04,.1380200E-04,.1662200E-04,&
     & .1855700E-04,.1931400E-04,.1779300E-04,.9590600E-06,.2071600E-07,&
     & .7060300E-05,.1162500E-04,.1538500E-04,.1850300E-04,.2077600E-04,&
     & .2163000E-04,.2015200E-04,.1136300E-05,.1928200E-07,.7696000E-05,&
     & .1288600E-04,.1708700E-04,.2047500E-04,.2286700E-04,.2385600E-04,&
     & .2250600E-04,.1340000E-05,.1803200E-07,.8398300E-05,.1420600E-04,&
     & .1878300E-04,.2233400E-04,.2476400E-04,.2581500E-04,.2446100E-04,&
     & .1563700E-05,.1693400E-07,.9118600E-05,.1549100E-04,.2051300E-04,&
     & .2420100E-04,.2686400E-04,.2791700E-04,.2649100E-04,.1770700E-05,&
     & .1832300E-07,.5302200E-05,.8627300E-05,.1138500E-04,.1372900E-04,&
     & .1531800E-04,.1595200E-04,.1469300E-04,.7892700E-06,.1696100E-07,&
     & .5831600E-05,.9620400E-05,.1271700E-04,.1527500E-04,.1714100E-04,&
     & .1786300E-04,.1663900E-04,.9365900E-06,.1578700E-07,.6371000E-05,&
     & .1067500E-04,.1414000E-04,.1692500E-04,.1888100E-04,.1967500E-04,&
     & .1856400E-04,.1105000E-05,.1476400E-07,.6960700E-05,.1177900E-04,&
     & .1555400E-04,.1847500E-04,.2046100E-04,.2132000E-04,.2020700E-04,&
     & .1286800E-05,.1386500E-07,.7561100E-05,.1284300E-04,.1699100E-04,&
     & .2002900E-04,.2220800E-04,.2307900E-04,.2183800E-04,.1456500E-05/

      data   absa(  1:180, 3)   /                                       &
     & .5594800E-06,.4347200E-03,.6568900E-03,.8023700E-03,.9032600E-03,&
     & .9599500E-03,.9424200E-03,.8180100E-03,.1838000E-03,.5317500E-06,&
     & .4539500E-03,.6877900E-03,.8404000E-03,.9459800E-03,.1001100E-02,&
     & .9848100E-03,.8574800E-03,.1956700E-03,.5064800E-06,.4683500E-03,&
     & .7140300E-03,.8768400E-03,.9852000E-03,.1039300E-02,.1023300E-02,&
     & .8972900E-03,.2072500E-03,.4833000E-06,.4814300E-03,.7397900E-03,&
     & .9106300E-03,.1021200E-02,.1076200E-02,.1060000E-02,.9350700E-03,&
     & .2181400E-03,.4619500E-06,.4941300E-03,.7647000E-03,.9420800E-03,&
     & .1054700E-02,.1109900E-02,.1096000E-02,.9716400E-03,.2305600E-03,&
     & .4681300E-06,.3671100E-03,.5574300E-03,.6807000E-03,.7661400E-03,&
     & .8144200E-03,.7994000E-03,.6969100E-03,.1498800E-03,.4445000E-06,&
     & .3841300E-03,.5850200E-03,.7150600E-03,.8040900E-03,.8519100E-03,&
     & .8372200E-03,.7323100E-03,.1598500E-03,.4230100E-06,.3974400E-03,&
     & .6085200E-03,.7474100E-03,.8392000E-03,.8858800E-03,.8720600E-03,&
     & .7667700E-03,.1696800E-03,.4033500E-06,.4092700E-03,.6320000E-03,&
     & .7783500E-03,.8719000E-03,.9184300E-03,.9046700E-03,.7985900E-03,&
     & .1794800E-03,.3852900E-06,.4203400E-03,.6535300E-03,.8075800E-03,&
     & .9024800E-03,.9492900E-03,.9371500E-03,.8313600E-03,.1901800E-03,&
     & .3959100E-06,.3045900E-03,.4636400E-03,.5659000E-03,.6368000E-03,&
     & .6768100E-03,.6651700E-03,.5810600E-03,.1193600E-03,.3753800E-06,&
     & .3203600E-03,.4887400E-03,.5964900E-03,.6717000E-03,.7116100E-03,&
     & .6994300E-03,.6115900E-03,.1280400E-03,.3567500E-06,.3332100E-03,&
     & .5099900E-03,.6259100E-03,.7031900E-03,.7433300E-03,.7301500E-03,&
     & .6407000E-03,.1364200E-03,.3397800E-06,.3438200E-03,.5310200E-03,&
     & .6539100E-03,.7327800E-03,.7730300E-03,.7599000E-03,.6697900E-03,&
     & .1447700E-03,.3242200E-06,.3541600E-03,.5507100E-03,.6804200E-03,&
     & .7604100E-03,.8011800E-03,.7891400E-03,.6992300E-03,.1536500E-03,&
     & .3359800E-06,.2497000E-03,.3805300E-03,.4646400E-03,.5223400E-03,&
     & .5548800E-03,.5467500E-03,.4779400E-03,.9527500E-04,.3180000E-06,&
     & .2641500E-03,.4031300E-03,.4921000E-03,.5535900E-03,.5866100E-03,&
     & .5769400E-03,.5039300E-03,.1029200E-03,.3017700E-06,.2763800E-03,&
     & .4226800E-03,.5182300E-03,.5834800E-03,.6159100E-03,.6049700E-03,&
     & .5294700E-03,.1101600E-03,.2870300E-06,.2861200E-03,.4416500E-03,&
     & .5432600E-03,.6101000E-03,.6429900E-03,.6315100E-03,.5555800E-03,&
     & .1170800E-03,.2735600E-06,.2955000E-03,.4590300E-03,.5670300E-03,&
     & .6352200E-03,.6683900E-03,.6581200E-03,.5814200E-03,.1244400E-03/

      data   absa(181:315, 3)   /                                       &
     & .2851700E-06,.2030500E-03,.3094000E-03,.3785400E-03,.4251600E-03,&
     & .4516700E-03,.4461700E-03,.3900600E-03,.7639400E-04,.2694300E-06,&
     & .2163100E-03,.3294200E-03,.4027900E-03,.4528100E-03,.4795900E-03,&
     & .4727400E-03,.4121900E-03,.8309300E-04,.2552600E-06,.2276000E-03,&
     & .3473600E-03,.4260300E-03,.4792800E-03,.5057300E-03,.4975300E-03,&
     & .4342300E-03,.8913700E-04,.2424500E-06,.2366100E-03,.3642300E-03,&
     & .4483200E-03,.5037300E-03,.5302100E-03,.5209200E-03,.4571500E-03,&
     & .9502600E-04,.2307900E-06,.2449000E-03,.3797900E-03,.4695800E-03,&
     & .5267100E-03,.5536000E-03,.5442200E-03,.4798200E-03,.1010000E-03,&
     & .2425800E-06,.1637400E-03,.2488900E-03,.3052600E-03,.3425600E-03,&
     & .3643700E-03,.3605400E-03,.3164300E-03,.6098300E-04,.2287300E-06,&
     & .1755700E-03,.2667400E-03,.3266200E-03,.3669800E-03,.3886200E-03,&
     & .3839500E-03,.3348100E-03,.6671700E-04,.2163200E-06,.1859600E-03,&
     & .2831500E-03,.3471000E-03,.3903200E-03,.4118600E-03,.4059700E-03,&
     & .3534700E-03,.7180500E-04,.2051500E-06,.1941200E-03,.2982500E-03,&
     & .3667200E-03,.4122300E-03,.4337000E-03,.4265400E-03,.3730800E-03,&
     & .7695000E-04,.1950000E-06,.2016300E-03,.3117700E-03,.3857300E-03,&
     & .4329800E-03,.4547400E-03,.4465900E-03,.3927300E-03,.8185400E-04,&
     & .2064400E-06,.1308800E-03,.1988300E-03,.2445300E-03,.2737900E-03,&
     & .2913300E-03,.2897200E-03,.2552700E-03,.4848300E-04,.1942200E-06,&
     & .1412300E-03,.2146400E-03,.2630700E-03,.2952000E-03,.3126200E-03,&
     & .3097600E-03,.2705700E-03,.5318900E-04,.1833400E-06,.1505600E-03,&
     & .2294500E-03,.2810200E-03,.3157000E-03,.3333600E-03,.3292100E-03,&
     & .2865700E-03,.5766400E-04,.1735900E-06,.1584500E-03,.2425400E-03,&
     & .2981200E-03,.3353300E-03,.3527300E-03,.3473700E-03,.3029000E-03,&
     & .6208000E-04,.1647600E-06,.1651500E-03,.2547200E-03,.3149200E-03,&
     & .3536800E-03,.3713400E-03,.3646600E-03,.3197100E-03,.6628300E-04/

      data   absa(316:450, 3)   /                                       &
     & .1756400E-06,.1037700E-03,.1579900E-03,.1944300E-03,.2177500E-03,&
     & .2322100E-03,.2317800E-03,.2051800E-03,.3861300E-04,.1648600E-06,&
     & .1129100E-03,.1717700E-03,.2106000E-03,.2360500E-03,.2502300E-03,&
     & .2486000E-03,.2184000E-03,.4257400E-04,.1553200E-06,.1216000E-03,&
     & .1847200E-03,.2261300E-03,.2538300E-03,.2681300E-03,.2656900E-03,&
     & .2314900E-03,.4643800E-04,.1467900E-06,.1287100E-03,.1962900E-03,&
     & .2411700E-03,.2710100E-03,.2854600E-03,.2814800E-03,.2451100E-03,&
     & .5013200E-04,.1391300E-06,.1347000E-03,.2071100E-03,.2558300E-03,&
     & .2873500E-03,.3017100E-03,.2965300E-03,.2593100E-03,.5389200E-04,&
     & .1495000E-06,.8194100E-04,.1247300E-03,.1536300E-03,.1725700E-03,&
     & .1840900E-03,.1842400E-03,.1643900E-03,.3077200E-04,.1399800E-06,&
     & .8970200E-04,.1365500E-03,.1676300E-03,.1877900E-03,.1991900E-03,&
     & .1987700E-03,.1758100E-03,.3402200E-04,.1316000E-06,.9726300E-04,&
     & .1477600E-03,.1808900E-03,.2029200E-03,.2145100E-03,.2131700E-03,&
     & .1864000E-03,.3738100E-04,.1241500E-06,.1038900E-03,.1581700E-03,&
     & .1940400E-03,.2179400E-03,.2298000E-03,.2271500E-03,.1978400E-03,&
     & .4045700E-04,.1174900E-06,.1093400E-03,.1676800E-03,.2068000E-03,&
     & .2323400E-03,.2441400E-03,.2402600E-03,.2095900E-03,.4358600E-04,&
     & .1265800E-06,.6476300E-04,.9869000E-04,.1217000E-03,.1370900E-03,&
     & .1464400E-03,.1470900E-03,.1321500E-03,.2473700E-04,.1182600E-06,&
     & .7140500E-04,.1087500E-03,.1336800E-03,.1497800E-03,.1590100E-03,&
     & .1593700E-03,.1415300E-03,.2748700E-04,.1109700E-06,.7785700E-04,&
     & .1184200E-03,.1451000E-03,.1626100E-03,.1720400E-03,.1712900E-03,&
     & .1504100E-03,.3025000E-04,.1045100E-06,.8376400E-04,.1276200E-03,&
     & .1562500E-03,.1754400E-03,.1851600E-03,.1834500E-03,.1601000E-03,&
     & .3289100E-04,.9876700E-07,.8883300E-04,.1359500E-03,.1672300E-03,&
     & .1879800E-03,.1976600E-03,.1947700E-03,.1697900E-03,.3554700E-04/

      data   absa(451:585, 3)   /                                       &
     & .1036800E-06,.5371400E-04,.8174800E-04,.1008500E-03,.1134900E-03,&
     & .1212600E-03,.1218000E-03,.1095000E-03,.2078700E-04,.9685400E-07,&
     & .5917500E-04,.9013100E-04,.1108500E-03,.1242500E-03,.1316600E-03,&
     & .1320700E-03,.1173300E-03,.2302800E-04,.9088200E-07,.6459300E-04,&
     & .9827500E-04,.1204100E-03,.1349700E-03,.1427300E-03,.1421500E-03,&
     & .1247900E-03,.2536200E-04,.8560200E-07,.6954500E-04,.1059200E-03,&
     & .1298200E-03,.1458300E-03,.1538800E-03,.1522800E-03,.1329700E-03,&
     & .2758500E-04,.8088900E-07,.7373100E-04,.1129000E-03,.1391400E-03,&
     & .1563700E-03,.1643100E-03,.1618900E-03,.1411600E-03,.2982800E-04,&
     & .8488600E-07,.4441800E-04,.6764100E-04,.8349000E-04,.9390600E-04,&
     & .1003400E-03,.1007500E-03,.9061400E-04,.1735000E-04,.7929700E-07,&
     & .4898100E-04,.7465600E-04,.9182100E-04,.1029900E-03,.1089100E-03,&
     & .1093300E-03,.9722100E-04,.1923700E-04,.7440800E-07,.5355300E-04,&
     & .8150300E-04,.9987200E-04,.1119600E-03,.1183200E-03,.1178200E-03,&
     & .1034100E-03,.2118900E-04,.7008500E-07,.5769800E-04,.8788700E-04,&
     & .1078100E-03,.1210700E-03,.1277500E-03,.1262400E-03,.1103400E-03,&
     & .2303800E-04,.6622600E-07,.6117700E-04,.9369400E-04,.1156700E-03,&
     & .1299500E-03,.1364500E-03,.1343600E-03,.1172600E-03,.2491300E-04,&
     & .6949900E-07,.3671600E-04,.5590200E-04,.6906700E-04,.7763500E-04,&
     & .8293400E-04,.8330200E-04,.7490500E-04,.1441900E-04,.6492300E-07,&
     & .4051100E-04,.6177900E-04,.7598400E-04,.8523900E-04,.9009000E-04,&
     & .9045600E-04,.8026400E-04,.1599600E-04,.6092000E-07,.4435600E-04,&
     & .6753300E-04,.8274600E-04,.9281600E-04,.9801700E-04,.9759000E-04,&
     & .8562100E-04,.1762300E-04,.5738100E-07,.4782600E-04,.7287400E-04,&
     & .8944300E-04,.1004000E-03,.1058800E-03,.1045600E-03,.9142900E-04,&
     & .1914400E-04,.5422100E-07,.5075000E-04,.7769800E-04,.9600500E-04,&
     & .1078400E-03,.1131800E-03,.1113900E-03,.9725800E-04,.2067100E-04/

      data   absa(  1:180, 4)   /                                       &
     & .3665400E-05,.1368900E-02,.2110800E-02,.2631400E-02,.2945800E-02,&
     & .3083100E-02,.3020700E-02,.2600300E-02,.9368900E-03,.3530400E-05,&
     & .1402500E-02,.2171600E-02,.2707800E-02,.3055100E-02,.3194800E-02,&
     & .3164900E-02,.2725800E-02,.9798300E-03,.3405900E-05,.1434000E-02,&
     & .2221400E-02,.2777300E-02,.3134800E-02,.3304900E-02,.3283500E-02,&
     & .2846100E-02,.1025100E-02,.3289500E-05,.1461000E-02,.2259600E-02,&
     & .2831400E-02,.3211300E-02,.3413100E-02,.3392100E-02,.2950800E-02,&
     & .1073000E-02,.3181400E-05,.1481400E-02,.2290500E-02,.2878900E-02,&
     & .3278500E-02,.3499600E-02,.3495200E-02,.3052800E-02,.1120500E-02,&
     & .3029300E-05,.1160000E-02,.1794600E-02,.2237000E-02,.2520500E-02,&
     & .2643200E-02,.2617200E-02,.2269100E-02,.7846600E-03,.2917300E-05,&
     & .1191600E-02,.1846800E-02,.2307200E-02,.2609400E-02,.2747700E-02,&
     & .2738800E-02,.2380900E-02,.8235600E-03,.2815200E-05,.1220300E-02,&
     & .1893900E-02,.2369100E-02,.2681400E-02,.2850600E-02,.2841400E-02,&
     & .2482500E-02,.8616100E-03,.2720700E-05,.1245400E-02,.1929100E-02,&
     & .2418000E-02,.2748900E-02,.2938000E-02,.2934500E-02,.2580800E-02,&
     & .9032800E-03,.2631600E-05,.1264400E-02,.1958600E-02,.2462400E-02,&
     & .2814600E-02,.3012500E-02,.3023700E-02,.2669500E-02,.9467800E-03,&
     & .2528800E-05,.9734800E-03,.1507000E-02,.1879500E-02,.2120600E-02,&
     & .2236200E-02,.2221000E-02,.1936900E-02,.6430800E-03,.2435300E-05,&
     & .1003100E-02,.1555200E-02,.1944100E-02,.2201300E-02,.2325700E-02,&
     & .2328600E-02,.2036200E-02,.6752900E-03,.2349500E-05,.1029500E-02,&
     & .1598600E-02,.2001900E-02,.2266300E-02,.2416000E-02,.2420400E-02,&
     & .2127600E-02,.7099100E-03,.2269900E-05,.1053300E-02,.1633700E-02,&
     & .2048900E-02,.2328800E-02,.2491100E-02,.2501600E-02,.2214800E-02,&
     & .7463500E-03,.2195100E-05,.1072400E-02,.1662500E-02,.2089100E-02,&
     & .2389600E-02,.2558200E-02,.2581200E-02,.2292500E-02,.7843200E-03,&
     & .2116900E-05,.8105300E-03,.1256800E-02,.1564700E-02,.1765400E-02,&
     & .1866200E-02,.1856800E-02,.1626900E-02,.5260800E-03,.2040700E-05,&
     & .8387500E-03,.1301900E-02,.1625200E-02,.1838300E-02,.1944100E-02,&
     & .1953800E-02,.1713900E-02,.5532500E-03,.1967400E-05,.8632900E-03,&
     & .1341700E-02,.1678800E-02,.1898500E-02,.2024600E-02,.2031500E-02,&
     & .1797800E-02,.5825300E-03,.1899400E-05,.8855100E-03,.1375300E-02,&
     & .1723800E-02,.1956600E-02,.2092300E-02,.2103500E-02,.1872400E-02,&
     & .6143800E-03,.1835800E-05,.9044200E-03,.1404200E-02,.1761900E-02,&
     & .2011900E-02,.2154200E-02,.2174900E-02,.1942500E-02,.6476200E-03/

      data   absa(181:315, 4)   /                                       &
     & .1772500E-05,.6714600E-03,.1039600E-02,.1293500E-02,.1457600E-02,&
     & .1543700E-02,.1534100E-02,.1353200E-02,.4298100E-03,.1709500E-05,&
     & .6977400E-03,.1082300E-02,.1347700E-02,.1522700E-02,.1611700E-02,&
     & .1622200E-02,.1430200E-02,.4528600E-03,.1647800E-05,.7204400E-03,&
     & .1119000E-02,.1396700E-02,.1578300E-02,.1681700E-02,.1689700E-02,&
     & .1504000E-02,.4778200E-03,.1589600E-05,.7413600E-03,.1151200E-02,&
     & .1439400E-02,.1630700E-02,.1743500E-02,.1753600E-02,.1568700E-02,&
     & .5046800E-03,.1535300E-05,.7597700E-03,.1179000E-02,.1475500E-02,&
     & .1681100E-02,.1799000E-02,.1817500E-02,.1627900E-02,.5335300E-03,&
     & .1485600E-05,.5516300E-03,.8518200E-03,.1058300E-02,.1193300E-02,&
     & .1263300E-02,.1251900E-02,.1112600E-02,.3496900E-03,.1432300E-05,&
     & .5753400E-03,.8921100E-03,.1108400E-02,.1251000E-02,.1324800E-02,&
     & .1330200E-02,.1181500E-02,.3689300E-03,.1381800E-05,.5970500E-03,&
     & .9255200E-03,.1153000E-02,.1301800E-02,.1386000E-02,.1394200E-02,&
     & .1243700E-02,.3899100E-03,.1331700E-05,.6167100E-03,.9556800E-03,&
     & .1192200E-02,.1348100E-02,.1442600E-02,.1450800E-02,.1300100E-02,&
     & .4124800E-03,.1285100E-05,.6341400E-03,.9822700E-03,.1226500E-02,&
     & .1393600E-02,.1491300E-02,.1506400E-02,.1351200E-02,.4369600E-03,&
     & .1245300E-05,.4498200E-03,.6928600E-03,.8585700E-03,.9711500E-03,&
     & .1026800E-02,.1013200E-02,.9066100E-03,.2833300E-03,.1199700E-05,&
     & .4716300E-03,.7298400E-03,.9052500E-03,.1020900E-02,.1083000E-02,&
     & .1081500E-02,.9682000E-03,.2999100E-03,.1157100E-05,.4917000E-03,&
     & .7606400E-03,.9459700E-03,.1068300E-02,.1133700E-02,.1141800E-02,&
     & .1021000E-02,.3173800E-03,.1115700E-05,.5096400E-03,.7885900E-03,&
     & .9823600E-03,.1108800E-02,.1185900E-02,.1192000E-02,.1069000E-02,&
     & .3360000E-03,.1075700E-05,.5257800E-03,.8133000E-03,.1014100E-02,&
     & .1149300E-02,.1228500E-02,.1239900E-02,.1115100E-02,.3563100E-03/

      data   absa(316:450, 4)   /                                       &
     & .1043600E-05,.3647500E-03,.5594800E-03,.6939900E-03,.7869900E-03,&
     & .8273300E-03,.8154600E-03,.7344300E-03,.2287400E-03,.1004300E-05,&
     & .3846100E-03,.5936500E-03,.7363100E-03,.8290200E-03,.8799600E-03,&
     & .8758900E-03,.7874500E-03,.2428100E-03,.9678500E-06,.4022700E-03,&
     & .6219100E-03,.7718300E-03,.8706700E-03,.9233800E-03,.9311600E-03,&
     & .8333400E-03,.2573500E-03,.9336700E-06,.4188100E-03,.6469700E-03,&
     & .8047400E-03,.9075400E-03,.9680600E-03,.9743400E-03,.8743900E-03,&
     & .2728600E-03,.9001700E-06,.4337700E-03,.6702600E-03,.8339800E-03,&
     & .9426900E-03,.1008000E-02,.1015400E-02,.9147900E-03,.2894800E-03,&
     & .8746900E-06,.2934500E-03,.4485700E-03,.5565300E-03,.6337100E-03,&
     & .6625300E-03,.6522900E-03,.5900900E-03,.1847500E-03,.8406200E-06,&
     & .3115200E-03,.4797500E-03,.5942200E-03,.6707700E-03,.7116800E-03,&
     & .7055000E-03,.6367400E-03,.1966900E-03,.8094700E-06,.3277900E-03,&
     & .5058200E-03,.6270700E-03,.7070500E-03,.7493600E-03,.7525200E-03,&
     & .6768200E-03,.2089600E-03,.7805200E-06,.3426200E-03,.5282000E-03,&
     & .6563100E-03,.7400200E-03,.7873100E-03,.7925700E-03,.7115500E-03,&
     & .2219700E-03,.7530000E-06,.3561800E-03,.5494000E-03,.6823600E-03,&
     & .7697800E-03,.8227100E-03,.8280600E-03,.7465800E-03,.2358800E-03,&
     & .7308000E-06,.2361900E-03,.3600200E-03,.4476900E-03,.5100100E-03,&
     & .5329400E-03,.5231900E-03,.4749200E-03,.1500500E-03,.7014200E-06,&
     & .2522500E-03,.3874200E-03,.4803900E-03,.5428500E-03,.5740700E-03,&
     & .5676500E-03,.5144400E-03,.1600000E-03,.6748400E-06,.2668700E-03,&
     & .4111900E-03,.5092400E-03,.5737200E-03,.6091200E-03,.6081300E-03,&
     & .5494500E-03,.1703600E-03,.6503200E-06,.2800200E-03,.4310400E-03,&
     & .5351300E-03,.6029400E-03,.6399800E-03,.6448500E-03,.5788800E-03,&
     & .1812000E-03,.6275200E-06,.2918200E-03,.4495000E-03,.5583600E-03,&
     & .6284900E-03,.6714800E-03,.6757000E-03,.6091800E-03,.1927100E-03/

      data   absa(451:585, 4)   /                                       &
     & .5984600E-06,.1960300E-03,.2990200E-03,.3715900E-03,.4236700E-03,&
     & .4424100E-03,.4355000E-03,.3961800E-03,.1249800E-03,.5744700E-06,&
     & .2095300E-03,.3218700E-03,.3988200E-03,.4502100E-03,.4771700E-03,&
     & .4724000E-03,.4289400E-03,.1336200E-03,.5526500E-06,.2218500E-03,&
     & .3413300E-03,.4228300E-03,.4761600E-03,.5048100E-03,.5055300E-03,&
     & .4565400E-03,.1424600E-03,.5325900E-06,.2330100E-03,.3581600E-03,&
     & .4444800E-03,.5004800E-03,.5314700E-03,.5357400E-03,.4816900E-03,&
     & .1516800E-03,.5137900E-06,.2430200E-03,.3738300E-03,.4635200E-03,&
     & .5220300E-03,.5578300E-03,.5617500E-03,.5073700E-03,.1615300E-03,&
     & .4900600E-06,.1625900E-03,.2480200E-03,.3080400E-03,.3511100E-03,&
     & .3667200E-03,.3616600E-03,.3296800E-03,.1040100E-03,.4703300E-06,&
     & .1737900E-03,.2669700E-03,.3306900E-03,.3729200E-03,.3958700E-03,&
     & .3923900E-03,.3567200E-03,.1113800E-03,.4524700E-06,.1841700E-03,&
     & .2829500E-03,.3504500E-03,.3946500E-03,.4182500E-03,.4198400E-03,&
     & .3786700E-03,.1188200E-03,.4360500E-06,.1935100E-03,.2971600E-03,&
     & .3685000E-03,.4148900E-03,.4407400E-03,.4446000E-03,.4003100E-03,&
     & .1266700E-03,.4206600E-06,.2020600E-03,.3103800E-03,.3846400E-03,&
     & .4333700E-03,.4629700E-03,.4666700E-03,.4220200E-03,.1349800E-03,&
     & .4012200E-06,.1346500E-03,.2054100E-03,.2548800E-03,.2900600E-03,&
     & .3035800E-03,.2998400E-03,.2737500E-03,.8640600E-04,.3850200E-06,&
     & .1439800E-03,.2208600E-03,.2737600E-03,.3083200E-03,.3279400E-03,&
     & .3251700E-03,.2956200E-03,.9261400E-04,.3704500E-06,.1526600E-03,&
     & .2341600E-03,.2901300E-03,.3267500E-03,.3462200E-03,.3481600E-03,&
     & .3140100E-03,.9888000E-04,.3570200E-06,.1605000E-03,.2460800E-03,&
     & .3052100E-03,.3433600E-03,.3652200E-03,.3685700E-03,.3324600E-03,&
     & .1055900E-03,.3444000E-06,.1677100E-03,.2574200E-03,.3190400E-03,&
     & .3594300E-03,.3840500E-03,.3871500E-03,.3508700E-03,.1126200E-03/

      data   absa(  1:180, 5)   /                                       &
     & .1558300E-03,.3754600E-02,.5353600E-02,.6416100E-02,.7122300E-02,&
     & .7498200E-02,.7359600E-02,.6513500E-02,.3517500E-02,.1614800E-03,&
     & .3856900E-02,.5525600E-02,.6646100E-02,.7375700E-02,.7753000E-02,&
     & .7566500E-02,.6649600E-02,.3684000E-02,.1648400E-03,.3935300E-02,&
     & .5674500E-02,.6836400E-02,.7616600E-02,.7971100E-02,.7771700E-02,&
     & .6784600E-02,.3848500E-02,.1662100E-03,.3994600E-02,.5794700E-02,&
     & .7007100E-02,.7819300E-02,.8158100E-02,.7959200E-02,.6913200E-02,&
     & .3976900E-02,.1651800E-03,.4038600E-02,.5892500E-02,.7153000E-02,&
     & .7994600E-02,.8325800E-02,.8110700E-02,.7022600E-02,.4113400E-02,&
     & .1248200E-03,.3220400E-02,.4610200E-02,.5547500E-02,.6173000E-02,&
     & .6500000E-02,.6394200E-02,.5639400E-02,.3000200E-02,.1302700E-03,&
     & .3309900E-02,.4761700E-02,.5749700E-02,.6403100E-02,.6718400E-02,&
     & .6582900E-02,.5772100E-02,.3144200E-02,.1337000E-03,.3380800E-02,&
     & .4884500E-02,.5923500E-02,.6607400E-02,.6904800E-02,.6766000E-02,&
     & .5901800E-02,.3278800E-02,.1354900E-03,.3434000E-02,.4988500E-02,&
     & .6074500E-02,.6787700E-02,.7078000E-02,.6920200E-02,.6013100E-02,&
     & .3394600E-02,.1354300E-03,.3477800E-02,.5074500E-02,.6198100E-02,&
     & .6928200E-02,.7222900E-02,.7052100E-02,.6121800E-02,.3509700E-02,&
     & .9843700E-04,.2720900E-02,.3903500E-02,.4718500E-02,.5270600E-02,&
     & .5557900E-02,.5484600E-02,.4846900E-02,.2507700E-02,.1039600E-03,&
     & .2803100E-02,.4038400E-02,.4899300E-02,.5475300E-02,.5757500E-02,&
     & .5652500E-02,.4968400E-02,.2636600E-02,.1077000E-03,.2870200E-02,&
     & .4151300E-02,.5050400E-02,.5664000E-02,.5924500E-02,.5812500E-02,&
     & .5092200E-02,.2747800E-02,.1099600E-03,.2923200E-02,.4245100E-02,&
     & .5183800E-02,.5825700E-02,.6079300E-02,.5949800E-02,.5194900E-02,&
     & .2849000E-02,.1108800E-03,.2964500E-02,.4325100E-02,.5299200E-02,&
     & .5946800E-02,.6209600E-02,.6064800E-02,.5290400E-02,.2949500E-02,&
     & .7658700E-04,.2279200E-02,.3273100E-02,.3972400E-02,.4456100E-02,&
     & .4710900E-02,.4670600E-02,.4144400E-02,.2081000E-02,.8222500E-04,&
     & .2354800E-02,.3394300E-02,.4132500E-02,.4641500E-02,.4893900E-02,&
     & .4820000E-02,.4259100E-02,.2192500E-02,.8620300E-04,.2418000E-02,&
     & .3499100E-02,.4268400E-02,.4807400E-02,.5041600E-02,.4965100E-02,&
     & .4364300E-02,.2293800E-02,.8876600E-04,.2469400E-02,.3587400E-02,&
     & .4387500E-02,.4946900E-02,.5181000E-02,.5085000E-02,.4460600E-02,&
     & .2379100E-02,.9021800E-04,.2511000E-02,.3661300E-02,.4493700E-02,&
     & .5054800E-02,.5298200E-02,.5188300E-02,.4549800E-02,.2465800E-02/

      data   absa(181:315, 5)   /                                       &
     & .5908000E-04,.1897200E-02,.2729400E-02,.3319500E-02,.3739500E-02,&
     & .3968300E-02,.3952300E-02,.3516100E-02,.1719500E-02,.6450800E-04,&
     & .1967300E-02,.2837600E-02,.3463200E-02,.3903300E-02,.4134300E-02,&
     & .4082600E-02,.3619500E-02,.1817300E-02,.6853800E-04,.2026400E-02,&
     & .2934200E-02,.3585600E-02,.4048800E-02,.4266400E-02,.4214100E-02,&
     & .3714400E-02,.1908100E-02,.7132800E-04,.2075000E-02,.3016500E-02,&
     & .3693300E-02,.4171900E-02,.4388000E-02,.4322100E-02,.3800600E-02,&
     & .1981800E-02,.7307300E-04,.2115500E-02,.3086100E-02,.3789200E-02,&
     & .4269000E-02,.4490100E-02,.4416100E-02,.3886400E-02,.2058300E-02,&
     & .4506900E-04,.1567300E-02,.2259600E-02,.2756600E-02,.3111700E-02,&
     & .3316200E-02,.3313500E-02,.2957400E-02,.1408900E-02,.4995900E-04,&
     & .1634200E-02,.2357700E-02,.2882800E-02,.3257200E-02,.3462600E-02,&
     & .3430200E-02,.3047600E-02,.1495500E-02,.5401300E-04,.1689100E-02,&
     & .2445700E-02,.2992600E-02,.3386500E-02,.3580900E-02,.3542700E-02,&
     & .3135800E-02,.1576300E-02,.5691000E-04,.1735600E-02,.2522000E-02,&
     & .3090300E-02,.3496800E-02,.3685300E-02,.3644300E-02,.3214000E-02,&
     & .1642800E-02,.5885200E-04,.1774300E-02,.2586800E-02,.3177100E-02,&
     & .3585700E-02,.3778200E-02,.3728400E-02,.3290500E-02,.1707900E-02,&
     & .3413100E-04,.1287300E-02,.1861900E-02,.2274500E-02,.2568800E-02,&
     & .2749400E-02,.2757400E-02,.2465800E-02,.1144400E-02,.3837900E-04,&
     & .1350300E-02,.1949000E-02,.2386900E-02,.2698500E-02,.2876100E-02,&
     & .2860700E-02,.2544300E-02,.1224200E-02,.4214600E-04,.1401800E-02,&
     & .2028700E-02,.2484300E-02,.2811900E-02,.2984800E-02,.2956900E-02,&
     & .2625800E-02,.1295600E-02,.4511600E-04,.1445400E-02,.2098900E-02,&
     & .2571800E-02,.2911500E-02,.3074000E-02,.3048200E-02,.2696700E-02,&
     & .1353700E-02,.4717800E-04,.1481800E-02,.2159100E-02,.2650900E-02,&
     & .2994500E-02,.3159000E-02,.3123500E-02,.2764100E-02,.1410000E-02/

      data   absa(316:450, 5)   /                                       &
     & .2565600E-04,.1052500E-02,.1526600E-02,.1864800E-02,.2106000E-02,&
     & .2266000E-02,.2280700E-02,.2041500E-02,.9248200E-03,.2931300E-04,&
     & .1109600E-02,.1602700E-02,.1962800E-02,.2222400E-02,.2373000E-02,&
     & .2368300E-02,.2111100E-02,.9936600E-03,.3264300E-04,.1157900E-02,&
     & .1673200E-02,.2050300E-02,.2323100E-02,.2472100E-02,.2451200E-02,&
     & .2184200E-02,.1057000E-02,.3550900E-04,.1198500E-02,.1737900E-02,&
     & .2128400E-02,.2411000E-02,.2551900E-02,.2533900E-02,.2249600E-02,&
     & .1111600E-02,.3761600E-04,.1232700E-02,.1793300E-02,.2199600E-02,&
     & .2488000E-02,.2625800E-02,.2602900E-02,.2308800E-02,.1159500E-02,&
     & .1916600E-04,.8555900E-03,.1244600E-02,.1521300E-02,.1716000E-02,&
     & .1852700E-02,.1872300E-02,.1679400E-02,.7431600E-03,.2225100E-04,&
     & .9070100E-03,.1310700E-02,.1606100E-02,.1817700E-02,.1945200E-02,&
     & .1949600E-02,.1742000E-02,.8016800E-03,.2512300E-04,.9510600E-03,&
     & .1372400E-02,.1682600E-02,.1906400E-02,.2035200E-02,.2024100E-02,&
     & .1806400E-02,.8565900E-03,.2771300E-04,.9890300E-03,.1430100E-02,&
     & .1752300E-02,.1986200E-02,.2107900E-02,.2094400E-02,.1865600E-02,&
     & .9065500E-03,.2982100E-04,.1020700E-02,.1481300E-02,.1816300E-02,&
     & .2057000E-02,.2173700E-02,.2157400E-02,.1915300E-02,.9488200E-03,&
     & .1445500E-04,.6950800E-03,.1014300E-02,.1238200E-02,.1397700E-02,&
     & .1510100E-02,.1533900E-02,.1379400E-02,.6006400E-03,.1697500E-04,&
     & .7405500E-03,.1071500E-02,.1311700E-02,.1485100E-02,.1593800E-02,&
     & .1602600E-02,.1435400E-02,.6503400E-03,.1942100E-04,.7799600E-03,&
     & .1124300E-02,.1379100E-02,.1563600E-02,.1670700E-02,.1667800E-02,&
     & .1490800E-02,.6987500E-03,.2165900E-04,.8136900E-03,.1174600E-02,&
     & .1440800E-02,.1634000E-02,.1737600E-02,.1726700E-02,.1541800E-02,&
     & .7433600E-03,.2361600E-04,.8431600E-03,.1221500E-02,.1497200E-02,&
     & .1697100E-02,.1794500E-02,.1783400E-02,.1584400E-02,.7804500E-03/

      data   absa(451:585, 5)   /                                       &
     & .1181200E-04,.5806700E-03,.8465800E-03,.1034900E-02,.1168100E-02,&
     & .1262200E-02,.1280600E-02,.1151800E-02,.5019700E-03,.1387000E-04,&
     & .6187000E-03,.8936400E-03,.1095700E-02,.1241100E-02,.1330700E-02,&
     & .1336600E-02,.1199000E-02,.5441900E-03,.1587200E-04,.6513100E-03,&
     & .9384300E-03,.1152000E-02,.1306800E-02,.1396600E-02,.1392700E-02,&
     & .1245900E-02,.5841200E-03,.1770600E-04,.6795700E-03,.9812800E-03,&
     & .1203700E-02,.1365300E-02,.1451000E-02,.1442300E-02,.1287300E-02,&
     & .6208600E-03,.1930700E-04,.7047700E-03,.1020800E-02,.1252200E-02,&
     & .1418700E-02,.1500000E-02,.1488200E-02,.1322300E-02,.6527300E-03,&
     & .9663300E-05,.4838400E-03,.7046400E-03,.8621400E-03,.9732500E-03,&
     & .1052500E-02,.1066100E-02,.9581800E-03,.4183800E-03,.1134600E-04,&
     & .5154400E-03,.7437500E-03,.9125500E-03,.1034600E-02,.1108800E-02,&
     & .1112400E-02,.9978700E-03,.4543600E-03,.1298300E-04,.5425800E-03,&
     & .7819100E-03,.9599400E-03,.1089300E-02,.1163600E-02,.1159500E-02,&
     & .1037900E-02,.4876700E-03,.1448300E-04,.5666200E-03,.8179500E-03,&
     & .1003600E-02,.1138500E-02,.1209100E-02,.1201200E-02,.1070700E-02,&
     & .5181000E-03,.1579000E-04,.5880500E-03,.8516500E-03,.1044700E-02,&
     & .1183000E-02,.1250000E-02,.1238900E-02,.1101200E-02,.5450800E-03,&
     & .7905700E-05,.4021500E-03,.5851600E-03,.7166600E-03,.8096900E-03,&
     & .8751800E-03,.8858100E-03,.7956900E-03,.3482500E-03,.9280800E-05,&
     & .4285300E-03,.6181800E-03,.7584700E-03,.8608200E-03,.9216000E-03,&
     & .9243600E-03,.8289400E-03,.3787100E-03,.1062100E-04,.4513400E-03,&
     & .6502800E-03,.7981500E-03,.9062200E-03,.9675200E-03,.9629100E-03,&
     & .8617700E-03,.4064200E-03,.1184900E-04,.4716100E-03,.6809000E-03,&
     & .8353000E-03,.9477000E-03,.1005400E-02,.9981500E-03,.8890800E-03,&
     & .4314900E-03,.1289700E-04,.4900000E-03,.7094400E-03,.8698200E-03,&
     & .9843000E-03,.1039800E-02,.1029700E-02,.9148300E-03,.4545900E-03/

      data   absa(  1:180, 6)   /                                       &
     & .2357500E-02,.9645700E-02,.1312800E-01,.1536100E-01,.1666200E-01,&
     & .1699900E-01,.1630300E-01,.1436800E-01,.1018500E-01,.2450800E-02,&
     & .9976800E-02,.1356200E-01,.1578600E-01,.1704600E-01,.1731600E-01,&
     & .1653900E-01,.1462700E-01,.1044700E-01,.2517600E-02,.1026200E-01,&
     & .1394700E-01,.1618200E-01,.1739000E-01,.1758800E-01,.1675200E-01,&
     & .1485700E-01,.1069900E-01,.2562700E-02,.1051000E-01,.1427100E-01,&
     & .1651700E-01,.1767800E-01,.1781500E-01,.1694100E-01,.1507300E-01,&
     & .1097000E-01,.2590400E-02,.1071200E-01,.1452800E-01,.1678800E-01,&
     & .1791200E-01,.1801300E-01,.1712600E-01,.1528200E-01,.1121500E-01,&
     & .1942200E-02,.8517900E-02,.1160300E-01,.1356100E-01,.1469400E-01,&
     & .1495200E-01,.1427400E-01,.1261600E-01,.8691000E-02,.2023100E-02,&
     & .8820800E-02,.1199200E-01,.1394200E-01,.1504300E-01,.1524100E-01,&
     & .1450300E-01,.1284900E-01,.8920300E-02,.2082400E-02,.9089200E-02,&
     & .1233300E-01,.1428300E-01,.1534300E-01,.1549300E-01,.1470400E-01,&
     & .1306500E-01,.9141200E-02,.2123500E-02,.9319300E-02,.1261300E-01,&
     & .1457300E-01,.1558600E-01,.1569600E-01,.1490000E-01,.1326900E-01,&
     & .9373100E-02,.2150100E-02,.9506400E-02,.1284400E-01,.1480200E-01,&
     & .1578900E-01,.1587200E-01,.1507900E-01,.1345900E-01,.9592200E-02,&
     & .1571000E-02,.7373900E-02,.1008400E-01,.1179500E-01,.1279100E-01,&
     & .1300200E-01,.1240800E-01,.1094800E-01,.7325600E-02,.1644500E-02,&
     & .7660300E-02,.1044100E-01,.1213600E-01,.1309800E-01,.1327200E-01,&
     & .1262900E-01,.1116900E-01,.7527700E-02,.1701600E-02,.7912000E-02,&
     & .1073800E-01,.1244400E-01,.1336500E-01,.1350200E-01,.1282700E-01,&
     & .1136500E-01,.7727500E-02,.1741900E-02,.8120400E-02,.1099200E-01,&
     & .1270400E-01,.1357900E-01,.1368700E-01,.1302200E-01,.1156500E-01,&
     & .7932700E-02,.1769300E-02,.8289600E-02,.1120100E-01,.1290300E-01,&
     & .1376100E-01,.1385400E-01,.1319300E-01,.1176000E-01,.8132400E-02,&
     & .1256800E-02,.6292500E-02,.8657000E-02,.1015600E-01,.1102000E-01,&
     & .1122000E-01,.1071000E-01,.9432900E-02,.6181200E-02,.1324200E-02,&
     & .6553200E-02,.8975600E-02,.1046000E-01,.1129200E-01,.1146300E-01,&
     & .1092400E-01,.9644900E-02,.6362000E-02,.1378200E-02,.6780700E-02,&
     & .9244700E-02,.1073900E-01,.1153100E-01,.1167600E-01,.1111500E-01,&
     & .9838400E-02,.6541700E-02,.1418500E-02,.6972800E-02,.9472400E-02,&
     & .1096700E-01,.1173200E-01,.1185300E-01,.1129900E-01,.1002900E-01,&
     & .6731400E-02,.1446600E-02,.7129400E-02,.9655100E-02,.1115000E-01,&
     & .1189700E-01,.1200300E-01,.1145700E-01,.1021200E-01,.6913400E-02/

      data   absa(181:315, 6)   /                                       &
     & .9981500E-03,.5318400E-02,.7366800E-02,.8666100E-02,.9418900E-02,&
     & .9608800E-02,.9186000E-02,.8085400E-02,.5214700E-02,.1059400E-02,&
     & .5550200E-02,.7648000E-02,.8938200E-02,.9667500E-02,.9828100E-02,&
     & .9390800E-02,.8290400E-02,.5380500E-02,.1109300E-02,.5753400E-02,&
     & .7882000E-02,.9187300E-02,.9886000E-02,.1002700E-01,.9573900E-02,&
     & .8477400E-02,.5539800E-02,.1148700E-02,.5926400E-02,.8074600E-02,&
     & .9395200E-02,.1007000E-01,.1019600E-01,.9746000E-02,.8658600E-02,&
     & .5711800E-02,.1177000E-02,.6071500E-02,.8232200E-02,.9559000E-02,&
     & .1022500E-01,.1033700E-01,.9889100E-02,.8828200E-02,.5875600E-02,&
     & .7848400E-03,.4455700E-02,.6208800E-02,.7323800E-02,.7975800E-02,&
     & .8156500E-02,.7823400E-02,.6894900E-02,.4372100E-02,.8410400E-03,&
     & .4658400E-02,.6454800E-02,.7572000E-02,.8204900E-02,.8360300E-02,&
     & .8013900E-02,.7091200E-02,.4520800E-02,.8870300E-03,.4838600E-02,&
     & .6658500E-02,.7793900E-02,.8407700E-02,.8549400E-02,.8187800E-02,&
     & .7267300E-02,.4666400E-02,.9242900E-03,.4993900E-02,.6825500E-02,&
     & .7979300E-02,.8581100E-02,.8712300E-02,.8342300E-02,.7432400E-02,&
     & .4819300E-02,.9528700E-03,.5126900E-02,.6966400E-02,.8121600E-02,&
     & .8725600E-02,.8840400E-02,.8479300E-02,.7588500E-02,.4968400E-02,&
     & .6117600E-03,.3710500E-02,.5192100E-02,.6149500E-02,.6708600E-02,&
     & .6873700E-02,.6619600E-02,.5849900E-02,.3639800E-02,.6635500E-03,&
     & .3884500E-02,.5409200E-02,.6368400E-02,.6919100E-02,.7063800E-02,&
     & .6796700E-02,.6032900E-02,.3772500E-02,.7058400E-03,.4042900E-02,&
     & .5589100E-02,.6566500E-02,.7106100E-02,.7239200E-02,.6956600E-02,&
     & .6195800E-02,.3904000E-02,.7401600E-03,.4180400E-02,.5736200E-02,&
     & .6730600E-02,.7265600E-02,.7393900E-02,.7100200E-02,.6345800E-02,&
     & .4042200E-02,.7679300E-03,.4301300E-02,.5863400E-02,.6860500E-02,&
     & .7391700E-02,.7516000E-02,.7229200E-02,.6486700E-02,.4175300E-02/

      data   absa(316:450, 6)   /                                       &
     & .4738300E-03,.3071900E-02,.4315700E-02,.5129900E-02,.5611300E-02,&
     & .5760400E-02,.5558400E-02,.4926300E-02,.3009900E-02,.5203000E-03,&
     & .3222400E-02,.4508800E-02,.5325600E-02,.5804900E-02,.5936900E-02,&
     & .5724900E-02,.5095000E-02,.3133500E-02,.5589700E-03,.3360100E-02,&
     & .4669000E-02,.5500800E-02,.5973300E-02,.6098200E-02,.5870300E-02,&
     & .5241600E-02,.3250500E-02,.5905100E-03,.3482300E-02,.4800300E-02,&
     & .5649200E-02,.6114700E-02,.6240700E-02,.6001100E-02,.5377600E-02,&
     & .3369100E-02,.6163800E-03,.3590800E-02,.4915100E-02,.5768200E-02,&
     & .6226900E-02,.6350500E-02,.6122600E-02,.5506100E-02,.3489300E-02,&
     & .3645200E-03,.2530100E-02,.3565500E-02,.4255800E-02,.4666800E-02,&
     & .4799500E-02,.4639100E-02,.4117600E-02,.2472300E-02,.4052100E-03,&
     & .2660900E-02,.3736800E-02,.4431300E-02,.4842200E-02,.4960500E-02,&
     & .4787600E-02,.4268200E-02,.2585400E-02,.4404200E-03,.2781100E-02,&
     & .3882200E-02,.4587100E-02,.4995100E-02,.5104800E-02,.4918900E-02,&
     & .4398800E-02,.2692200E-02,.4693400E-03,.2889000E-02,.4001500E-02,&
     & .4720400E-02,.5119500E-02,.5232900E-02,.5039800E-02,.4521100E-02,&
     & .2794700E-02,.4929600E-03,.2986300E-02,.4105200E-02,.4829600E-02,&
     & .5221200E-02,.5332000E-02,.5151600E-02,.4641000E-02,.2901400E-02,&
     & .2814100E-03,.2083900E-02,.2943400E-02,.3525000E-02,.3870500E-02,&
     & .3990700E-02,.3857400E-02,.3428000E-02,.2027600E-02,.3164200E-03,&
     & .2196200E-02,.3094700E-02,.3679800E-02,.4027900E-02,.4131700E-02,&
     & .3990500E-02,.3559100E-02,.2130400E-02,.3474300E-03,.2300200E-02,&
     & .3225100E-02,.3816700E-02,.4163200E-02,.4258800E-02,.4107700E-02,&
     & .3673700E-02,.2224600E-02,.3733700E-03,.2395800E-02,.3332600E-02,&
     & .3935100E-02,.4272700E-02,.4369900E-02,.4216700E-02,.3785000E-02,&
     & .2317100E-02,.3946400E-03,.2481800E-02,.3424500E-02,.4034100E-02,&
     & .4363500E-02,.4459200E-02,.4313300E-02,.3893100E-02,.2412300E-02/

      data   absa(451:585, 6)   /                                       &
     & .2304600E-03,.1756100E-02,.2482100E-02,.2969700E-02,.3260300E-02,&
     & .3359000E-02,.3246500E-02,.2891100E-02,.1700000E-02,.2590000E-03,&
     & .1849900E-02,.2607900E-02,.3098300E-02,.3391400E-02,.3479000E-02,&
     & .3359800E-02,.3000000E-02,.1787900E-02,.2843100E-03,.1937100E-02,&
     & .2715400E-02,.3212300E-02,.3500900E-02,.3581000E-02,.3458200E-02,&
     & .3098900E-02,.1870900E-02,.3055400E-03,.2017500E-02,.2804700E-02,&
     & .3309400E-02,.3589700E-02,.3670800E-02,.3550000E-02,.3196400E-02,&
     & .1952400E-02,.3229000E-03,.2090600E-02,.2882700E-02,.3390700E-02,&
     & .3663800E-02,.3744800E-02,.3632200E-02,.3292100E-02,.2034200E-02,&
     & .1887300E-03,.1475300E-02,.2085900E-02,.2493200E-02,.2736200E-02,&
     & .2816300E-02,.2723100E-02,.2428700E-02,.1422600E-02,.2120600E-03,&
     & .1554200E-02,.2190500E-02,.2599700E-02,.2843600E-02,.2914600E-02,&
     & .2818200E-02,.2519600E-02,.1496400E-02,.2327100E-03,.1628200E-02,&
     & .2278900E-02,.2693700E-02,.2932000E-02,.2999500E-02,.2900400E-02,&
     & .2606300E-02,.1568400E-02,.2500500E-03,.1695900E-02,.2353900E-02,&
     & .2773900E-02,.3005200E-02,.3073800E-02,.2978100E-02,.2692400E-02,&
     & .1638800E-02,.2642400E-03,.1757900E-02,.2419800E-02,.2841800E-02,&
     & .3068800E-02,.3137000E-02,.3047800E-02,.2773400E-02,.1709000E-02,&
     & .1545100E-03,.1236200E-02,.1747100E-02,.2085600E-02,.2287000E-02,&
     & .2353000E-02,.2275800E-02,.2031800E-02,.1186800E-02,.1735700E-03,&
     & .1302700E-02,.1833700E-02,.2173600E-02,.2375200E-02,.2433900E-02,&
     & .2354900E-02,.2110400E-02,.1249200E-02,.1904500E-03,.1365100E-02,&
     & .1907100E-02,.2252000E-02,.2447900E-02,.2505200E-02,.2425500E-02,&
     & .2186200E-02,.1311400E-02,.2045900E-03,.1422500E-02,.1970200E-02,&
     & .2319500E-02,.2510900E-02,.2568000E-02,.2490600E-02,.2259200E-02,&
     & .1371700E-02,.2162100E-03,.1475100E-02,.2026700E-02,.2377500E-02,&
     & .2565400E-02,.2622100E-02,.2551300E-02,.2328300E-02,.1432100E-02/

      data   absa(  1:180, 7)   /                                       &
     & .1455100E-01,.2628200E-01,.3205300E-01,.3518900E-01,.3660000E-01,&
     & .3664600E-01,.3555200E-01,.3321500E-01,.2730900E-01,.1536000E-01,&
     & .2684700E-01,.3241100E-01,.3542600E-01,.3680300E-01,.3699400E-01,&
     & .3603700E-01,.3367400E-01,.2788100E-01,.1605200E-01,.2735800E-01,&
     & .3271100E-01,.3560200E-01,.3700200E-01,.3732200E-01,.3649500E-01,&
     & .3409200E-01,.2836500E-01,.1663400E-01,.2776500E-01,.3297500E-01,&
     & .3575700E-01,.3719800E-01,.3762900E-01,.3687100E-01,.3448900E-01,&
     & .2881700E-01,.1709200E-01,.2808800E-01,.3317300E-01,.3589900E-01,&
     & .3736900E-01,.3789400E-01,.3720600E-01,.3485600E-01,.2923600E-01,&
     & .1267500E-01,.2375700E-01,.2882700E-01,.3147500E-01,.3264700E-01,&
     & .3277900E-01,.3187400E-01,.2965900E-01,.2373500E-01,.1336300E-01,&
     & .2426100E-01,.2916500E-01,.3173400E-01,.3290500E-01,.3313700E-01,&
     & .3235700E-01,.3011400E-01,.2424100E-01,.1395200E-01,.2467300E-01,&
     & .2945800E-01,.3194000E-01,.3316700E-01,.3348600E-01,.3279400E-01,&
     & .3054000E-01,.2470400E-01,.1444000E-01,.2501000E-01,.2969200E-01,&
     & .3213100E-01,.3340200E-01,.3381600E-01,.3318500E-01,.3094400E-01,&
     & .2512300E-01,.1481000E-01,.2524600E-01,.2986500E-01,.3232300E-01,&
     & .3362400E-01,.3412500E-01,.3353000E-01,.3129900E-01,.2550100E-01,&
     & .1076400E-01,.2118000E-01,.2564500E-01,.2793400E-01,.2889100E-01,&
     & .2897800E-01,.2821500E-01,.2625000E-01,.2034000E-01,.1137100E-01,&
     & .2161100E-01,.2597600E-01,.2822100E-01,.2920400E-01,.2934700E-01,&
     & .2868900E-01,.2668700E-01,.2080300E-01,.1188200E-01,.2197700E-01,&
     & .2625800E-01,.2846700E-01,.2949800E-01,.2973000E-01,.2911300E-01,&
     & .2711500E-01,.2124100E-01,.1230200E-01,.2227300E-01,.2647700E-01,&
     & .2869600E-01,.2976600E-01,.3009900E-01,.2950700E-01,.2750300E-01,&
     & .2162000E-01,.1262400E-01,.2248100E-01,.2663100E-01,.2890000E-01,&
     & .3003200E-01,.3042500E-01,.2985900E-01,.2784200E-01,.2196600E-01,&
     & .8986000E-02,.1865900E-01,.2260500E-01,.2458000E-01,.2538300E-01,&
     & .2543500E-01,.2476100E-01,.2303700E-01,.1741400E-01,.9520700E-02,&
     & .1904800E-01,.2292400E-01,.2488800E-01,.2573900E-01,.2582300E-01,&
     & .2520000E-01,.2345000E-01,.1784700E-01,.9972300E-02,.1938000E-01,&
     & .2318200E-01,.2515500E-01,.2606100E-01,.2620900E-01,.2561100E-01,&
     & .2386200E-01,.1825100E-01,.1035400E-01,.1964400E-01,.2339700E-01,&
     & .2540100E-01,.2634500E-01,.2658000E-01,.2600000E-01,.2423300E-01,&
     & .1860000E-01,.1063700E-01,.1982200E-01,.2355600E-01,.2560500E-01,&
     & .2661900E-01,.2691300E-01,.2635900E-01,.2456200E-01,.1892500E-01/

      data   absa(181:315, 7)   /                                       &
     & .7410400E-02,.1626400E-01,.1971700E-01,.2145200E-01,.2217200E-01,&
     & .2220700E-01,.2159800E-01,.2007500E-01,.1488000E-01,.7877800E-02,&
     & .1662400E-01,.2002200E-01,.2176500E-01,.2253100E-01,.2259600E-01,&
     & .2199900E-01,.2045800E-01,.1527900E-01,.8269400E-02,.1692800E-01,&
     & .2027700E-01,.2203600E-01,.2285900E-01,.2297800E-01,.2238400E-01,&
     & .2083400E-01,.1565000E-01,.8591900E-02,.1716800E-01,.2049800E-01,&
     & .2227400E-01,.2314300E-01,.2332600E-01,.2275400E-01,.2118900E-01,&
     & .1598000E-01,.8833700E-02,.1733200E-01,.2066400E-01,.2247700E-01,&
     & .2340700E-01,.2364800E-01,.2311100E-01,.2150900E-01,.1628300E-01,&
     & .6030500E-02,.1400800E-01,.1702800E-01,.1857400E-01,.1923600E-01,&
     & .1926900E-01,.1872000E-01,.1734700E-01,.1265600E-01,.6435600E-02,&
     & .1433900E-01,.1731800E-01,.1887200E-01,.1958000E-01,.1964500E-01,&
     & .1909500E-01,.1770500E-01,.1302500E-01,.6770800E-02,.1462600E-01,&
     & .1757500E-01,.1913500E-01,.1988300E-01,.2000400E-01,.1945400E-01,&
     & .1805500E-01,.1336600E-01,.7044900E-02,.1485300E-01,.1779500E-01,&
     & .1936300E-01,.2015600E-01,.2032400E-01,.1981100E-01,.1839200E-01,&
     & .1367900E-01,.7258600E-02,.1501400E-01,.1795400E-01,.1956900E-01,&
     & .2040100E-01,.2062200E-01,.2015500E-01,.1870600E-01,.1396300E-01,&
     & .4861500E-02,.1193600E-01,.1457400E-01,.1596300E-01,.1657800E-01,&
     & .1661500E-01,.1612500E-01,.1490400E-01,.1071800E-01,.5209300E-02,&
     & .1224700E-01,.1484500E-01,.1623900E-01,.1689000E-01,.1696500E-01,&
     & .1647700E-01,.1523800E-01,.1105000E-01,.5498100E-02,.1252000E-01,&
     & .1508900E-01,.1648500E-01,.1716300E-01,.1729600E-01,.1682300E-01,&
     & .1556100E-01,.1136700E-01,.5734100E-02,.1273200E-01,.1530200E-01,&
     & .1670900E-01,.1741200E-01,.1758700E-01,.1715600E-01,.1588200E-01,&
     & .1165700E-01,.5924500E-02,.1287900E-01,.1546000E-01,.1690000E-01,&
     & .1764600E-01,.1786200E-01,.1747500E-01,.1618900E-01,.1192400E-01/

      data   absa(316:450, 7)   /                                       &
     & .3890300E-02,.1008600E-01,.1239300E-01,.1361700E-01,.1417900E-01,&
     & .1423000E-01,.1381000E-01,.1274100E-01,.9040200E-02,.4188300E-02,&
     & .1037900E-01,.1264200E-01,.1386600E-01,.1445400E-01,.1454900E-01,&
     & .1414000E-01,.1305000E-01,.9340700E-02,.4437400E-02,.1062800E-01,&
     & .1286600E-01,.1409300E-01,.1470200E-01,.1484700E-01,.1446600E-01,&
     & .1335400E-01,.9628900E-02,.4641700E-02,.1082300E-01,.1306700E-01,&
     & .1429700E-01,.1493600E-01,.1511300E-01,.1477700E-01,.1365900E-01,&
     & .9894600E-02,.4810600E-02,.1095800E-01,.1321200E-01,.1447700E-01,&
     & .1515500E-01,.1536900E-01,.1506500E-01,.1395400E-01,.1013700E-01,&
     & .3090600E-02,.8471600E-02,.1047500E-01,.1153300E-01,.1203700E-01,&
     & .1210200E-01,.1175700E-01,.1084000E-01,.7579000E-02,.3345600E-02,&
     & .8738000E-02,.1069800E-01,.1175900E-01,.1228100E-01,.1238400E-01,&
     & .1206300E-01,.1112400E-01,.7852400E-02,.3562400E-02,.8963600E-02,&
     & .1090800E-01,.1196400E-01,.1250300E-01,.1265100E-01,.1236600E-01,&
     & .1141300E-01,.8111600E-02,.3740900E-02,.9139400E-02,.1108500E-01,&
     & .1215600E-01,.1271700E-01,.1289500E-01,.1264600E-01,.1170300E-01,&
     & .8354300E-02,.3889900E-02,.9266300E-02,.1121000E-01,.1232300E-01,&
     & .1292300E-01,.1313500E-01,.1290900E-01,.1197900E-01,.8574800E-02,&
     & .2458800E-02,.7099700E-02,.8815600E-02,.9728000E-02,.1017500E-01,&
     & .1024600E-01,.9971200E-02,.9195100E-02,.6332400E-02,.2674300E-02,&
     & .7339300E-02,.9018300E-02,.9928900E-02,.1038700E-01,.1049600E-01,&
     & .1024900E-01,.9456600E-02,.6577000E-02,.2861000E-02,.7535200E-02,&
     & .9202800E-02,.1011700E-01,.1058600E-01,.1073000E-01,.1052400E-01,&
     & .9730700E-02,.6807700E-02,.3014700E-02,.7689900E-02,.9355000E-02,&
     & .1029100E-01,.1078100E-01,.1095400E-01,.1077800E-01,.1000100E-01,&
     & .7020100E-02,.3144400E-02,.7805000E-02,.9466800E-02,.1043300E-01,&
     & .1097400E-01,.1117300E-01,.1101900E-01,.1025300E-01,.7218900E-02/

      data   absa(451:585, 7)   /                                       &
     & .2034500E-02,.6024900E-02,.7471600E-02,.8249600E-02,.8645800E-02,&
     & .8734000E-02,.8522200E-02,.7865700E-02,.5371400E-02,.2207100E-02,&
     & .6217600E-02,.7644000E-02,.8425100E-02,.8827300E-02,.8946100E-02,&
     & .8768900E-02,.8112500E-02,.5583500E-02,.2355400E-02,.6372900E-02,&
     & .7793200E-02,.8586800E-02,.9005800E-02,.9154800E-02,.9007500E-02,&
     & .8366000E-02,.5782700E-02,.2478000E-02,.6494400E-02,.7910600E-02,&
     & .8729100E-02,.9185700E-02,.9356500E-02,.9230200E-02,.8603600E-02,&
     & .5969500E-02,.2582000E-02,.6580200E-02,.7997400E-02,.8842700E-02,&
     & .9348200E-02,.9553900E-02,.9444300E-02,.8818000E-02,.6151200E-02,&
     & .1678900E-02,.5090600E-02,.6307800E-02,.6975000E-02,.7317200E-02,&
     & .7407800E-02,.7250400E-02,.6701500E-02,.4539400E-02,.1817000E-02,&
     & .5243700E-02,.6451200E-02,.7125500E-02,.7479000E-02,.7595900E-02,&
     & .7465500E-02,.6930700E-02,.4723700E-02,.1935600E-02,.5368900E-02,&
     & .6570100E-02,.7261000E-02,.7641500E-02,.7781100E-02,.7670300E-02,&
     & .7150800E-02,.4895300E-02,.2034200E-02,.5464400E-02,.6664000E-02,&
     & .7379200E-02,.7792900E-02,.7963500E-02,.7866000E-02,.7353200E-02,&
     & .5060700E-02,.2118200E-02,.5532800E-02,.6735200E-02,.7473600E-02,&
     & .7928200E-02,.8130700E-02,.8054200E-02,.7540700E-02,.5223400E-02,&
     & .1381900E-02,.4282300E-02,.5309600E-02,.5877100E-02,.6174500E-02,&
     & .6260100E-02,.6141100E-02,.5689300E-02,.3820100E-02,.1492700E-02,&
     & .4407800E-02,.5426100E-02,.6005500E-02,.6316700E-02,.6427800E-02,&
     & .6326800E-02,.5889100E-02,.3978900E-02,.1588400E-02,.4509000E-02,&
     & .5524600E-02,.6121700E-02,.6457700E-02,.6590300E-02,.6504400E-02,&
     & .6075800E-02,.4128800E-02,.1668300E-02,.4585800E-02,.5602900E-02,&
     & .6219300E-02,.6585900E-02,.6745400E-02,.6676800E-02,.6250600E-02,&
     & .4276500E-02,.1736400E-02,.4641900E-02,.5664400E-02,.6299600E-02,&
     & .6699800E-02,.6888800E-02,.6840400E-02,.6418800E-02,.4419800E-02/

      data   absa(  1:180, 8)   /                                       &
     & .5776000E-01,.6943200E-01,.7640200E-01,.8151700E-01,.8512500E-01,&
     & .8728900E-01,.8740800E-01,.8331700E-01,.7472000E-01,.5825600E-01,&
     & .6966700E-01,.7672300E-01,.8191100E-01,.8536200E-01,.8728900E-01,&
     & .8730600E-01,.8335700E-01,.7517700E-01,.5871000E-01,.6985900E-01,&
     & .7697300E-01,.8232800E-01,.8569000E-01,.8742000E-01,.8726400E-01,&
     & .8343500E-01,.7570100E-01,.5905200E-01,.7005000E-01,.7717600E-01,&
     & .8264200E-01,.8607100E-01,.8766000E-01,.8734100E-01,.8354400E-01,&
     & .7623000E-01,.5929800E-01,.7019800E-01,.7737400E-01,.8288300E-01,&
     & .8646900E-01,.8797700E-01,.8747300E-01,.8369100E-01,.7676900E-01,&
     & .5090800E-01,.6284800E-01,.7000300E-01,.7526300E-01,.7867000E-01,&
     & .8059400E-01,.8069400E-01,.7704800E-01,.6704800E-01,.5146900E-01,&
     & .6317800E-01,.7039500E-01,.7576300E-01,.7903300E-01,.8077500E-01,&
     & .8068600E-01,.7719900E-01,.6760500E-01,.5195600E-01,.6351900E-01,&
     & .7074800E-01,.7622100E-01,.7951200E-01,.8106700E-01,.8079700E-01,&
     & .7737000E-01,.6818200E-01,.5232900E-01,.6383100E-01,.7111500E-01,&
     & .7659100E-01,.8003200E-01,.8144500E-01,.8100100E-01,.7756800E-01,&
     & .6876800E-01,.5264800E-01,.6410800E-01,.7144100E-01,.7694100E-01,&
     & .8046400E-01,.8186600E-01,.8128200E-01,.7779600E-01,.6937700E-01,&
     & .4463600E-01,.5657500E-01,.6366700E-01,.6880400E-01,.7214600E-01,&
     & .7394000E-01,.7375600E-01,.7035900E-01,.5944600E-01,.4521600E-01,&
     & .5700500E-01,.6411100E-01,.6932800E-01,.7262000E-01,.7422800E-01,&
     & .7385400E-01,.7060000E-01,.6008600E-01,.4571600E-01,.5744800E-01,&
     & .6459100E-01,.6981600E-01,.7317800E-01,.7461600E-01,.7409300E-01,&
     & .7084300E-01,.6073800E-01,.4613700E-01,.5786200E-01,.6506000E-01,&
     & .7028300E-01,.7372400E-01,.7506700E-01,.7440800E-01,.7113000E-01,&
     & .6138500E-01,.4649000E-01,.5824000E-01,.6553700E-01,.7075200E-01,&
     & .7418700E-01,.7554500E-01,.7481700E-01,.7147900E-01,.6199800E-01,&
     & .3891000E-01,.5082300E-01,.5750500E-01,.6236400E-01,.6555900E-01,&
     & .6715400E-01,.6679800E-01,.6353100E-01,.5240100E-01,.3948800E-01,&
     & .5130700E-01,.5802200E-01,.6291200E-01,.6607400E-01,.6751800E-01,&
     & .6702500E-01,.6384000E-01,.5309300E-01,.3999100E-01,.5181200E-01,&
     & .5856600E-01,.6345100E-01,.6663200E-01,.6801000E-01,.6734800E-01,&
     & .6416900E-01,.5378200E-01,.4038700E-01,.5229100E-01,.5911700E-01,&
     & .6400100E-01,.6719600E-01,.6851400E-01,.6778900E-01,.6455500E-01,&
     & .5447000E-01,.4071900E-01,.5273900E-01,.5969100E-01,.6457200E-01,&
     & .6772500E-01,.6903100E-01,.6830700E-01,.6498500E-01,.5509100E-01/

      data   absa(181:315, 8)   /                                       &
     & .3369200E-01,.4545900E-01,.5170500E-01,.5611100E-01,.5898400E-01,&
     & .6038400E-01,.5995100E-01,.5677000E-01,.4593900E-01,.3425400E-01,&
     & .4597900E-01,.5225200E-01,.5666400E-01,.5951500E-01,.6081600E-01,&
     & .6027300E-01,.5714600E-01,.4664700E-01,.3473800E-01,.4653200E-01,&
     & .5284600E-01,.5723800E-01,.6007700E-01,.6135200E-01,.6071400E-01,&
     & .5756600E-01,.4735900E-01,.3511700E-01,.4704700E-01,.5345900E-01,&
     & .5784900E-01,.6066800E-01,.6190200E-01,.6125900E-01,.5803000E-01,&
     & .4803600E-01,.3542800E-01,.4750800E-01,.5408300E-01,.5849300E-01,&
     & .6125300E-01,.6248100E-01,.6185100E-01,.5852300E-01,.4864800E-01,&
     & .2891000E-01,.4042900E-01,.4617600E-01,.5010800E-01,.5260900E-01,&
     & .5375000E-01,.5328900E-01,.5028600E-01,.4006100E-01,.2943300E-01,&
     & .4096700E-01,.4673100E-01,.5065300E-01,.5314900E-01,.5423200E-01,&
     & .5368700E-01,.5071400E-01,.4075600E-01,.2988300E-01,.4152000E-01,&
     & .4734900E-01,.5125100E-01,.5373100E-01,.5479600E-01,.5420700E-01,&
     & .5118800E-01,.4144300E-01,.3025200E-01,.4202400E-01,.4799800E-01,&
     & .5191800E-01,.5433900E-01,.5538700E-01,.5482000E-01,.5171100E-01,&
     & .4208500E-01,.3054200E-01,.4244700E-01,.4863500E-01,.5259800E-01,&
     & .5498600E-01,.5603800E-01,.5543000E-01,.5223600E-01,.4267500E-01,&
     & .2460000E-01,.3572500E-01,.4095200E-01,.4439800E-01,.4655800E-01,&
     & .4746200E-01,.4695000E-01,.4422700E-01,.3475100E-01,.2506000E-01,&
     & .3625000E-01,.4149800E-01,.4495000E-01,.4709700E-01,.4797700E-01,&
     & .4739900E-01,.4469000E-01,.3541000E-01,.2546600E-01,.3676900E-01,&
     & .4211900E-01,.4557200E-01,.4769600E-01,.4854800E-01,.4796500E-01,&
     & .4519800E-01,.3605000E-01,.2580400E-01,.3723200E-01,.4276400E-01,&
     & .4624900E-01,.4834900E-01,.4917300E-01,.4860300E-01,.4573700E-01,&
     & .3664900E-01,.2607500E-01,.3764500E-01,.4337100E-01,.4695000E-01,&
     & .4904500E-01,.4985600E-01,.4922300E-01,.4627900E-01,.3719800E-01/

      data   absa(316:450, 8)   /                                       &
     & .2074700E-01,.3132500E-01,.3601600E-01,.3904700E-01,.4092100E-01,&
     & .4165600E-01,.4109700E-01,.3868200E-01,.2995500E-01,.2114500E-01,&
     & .3181100E-01,.3654100E-01,.3959700E-01,.4145500E-01,.4216900E-01,&
     & .4157500E-01,.3914500E-01,.3057700E-01,.2149700E-01,.3229000E-01,&
     & .3715500E-01,.4023800E-01,.4206700E-01,.4274400E-01,.4215400E-01,&
     & .3964800E-01,.3117800E-01,.2179600E-01,.3271700E-01,.3776600E-01,&
     & .4092300E-01,.4273800E-01,.4339400E-01,.4276800E-01,.4018500E-01,&
     & .3173300E-01,.2204400E-01,.3311200E-01,.3833800E-01,.4159700E-01,&
     & .4346200E-01,.4408700E-01,.4338500E-01,.4074000E-01,.3225400E-01,&
     & .1736000E-01,.2721800E-01,.3140100E-01,.3408400E-01,.3574100E-01,&
     & .3637300E-01,.3579800E-01,.3359400E-01,.2566400E-01,.1769300E-01,&
     & .2766200E-01,.3190800E-01,.3462200E-01,.3626400E-01,.3687700E-01,&
     & .3628500E-01,.3404900E-01,.2625100E-01,.1799200E-01,.2808900E-01,&
     & .3247700E-01,.3525500E-01,.3687400E-01,.3744400E-01,.3683100E-01,&
     & .3454400E-01,.2681400E-01,.1825800E-01,.2848100E-01,.3305000E-01,&
     & .3590900E-01,.3755200E-01,.3807800E-01,.3742100E-01,.3507300E-01,&
     & .2733500E-01,.1848400E-01,.2884900E-01,.3360100E-01,.3653600E-01,&
     & .3825400E-01,.3875600E-01,.3802600E-01,.3561600E-01,.2782900E-01,&
     & .1446200E-01,.2346300E-01,.2718600E-01,.2958800E-01,.3107900E-01,&
     & .3162500E-01,.3105300E-01,.2901200E-01,.2191600E-01,.1473700E-01,&
     & .2385300E-01,.2766200E-01,.3010900E-01,.3158100E-01,.3211600E-01,&
     & .3152700E-01,.2946100E-01,.2246000E-01,.1498300E-01,.2423800E-01,&
     & .2818700E-01,.3071000E-01,.3217600E-01,.3266400E-01,.3204400E-01,&
     & .2994800E-01,.2298700E-01,.1522800E-01,.2459300E-01,.2871800E-01,&
     & .3131200E-01,.3282600E-01,.3328200E-01,.3261500E-01,.3046700E-01,&
     & .2347800E-01,.1541100E-01,.2493200E-01,.2922800E-01,.3190300E-01,&
     & .3345500E-01,.3394300E-01,.3320700E-01,.3099400E-01,.2393900E-01/

      data   absa(451:585, 8)   /                                       &
     & .1209700E-01,.2023000E-01,.2355400E-01,.2573700E-01,.2707000E-01,&
     & .2752600E-01,.2698700E-01,.2512500E-01,.1884300E-01,.1231600E-01,&
     & .2056400E-01,.2401400E-01,.2626500E-01,.2758900E-01,.2802100E-01,&
     & .2746200E-01,.2558000E-01,.1934100E-01,.1252500E-01,.2089300E-01,&
     & .2449400E-01,.2681700E-01,.2817200E-01,.2857800E-01,.2798500E-01,&
     & .2607300E-01,.1981100E-01,.1269800E-01,.2121200E-01,.2497100E-01,&
     & .2735800E-01,.2875200E-01,.2919600E-01,.2855400E-01,.2657500E-01,&
     & .2025400E-01,.1281300E-01,.2152700E-01,.2542800E-01,.2788600E-01,&
     & .2931600E-01,.2979400E-01,.2914400E-01,.2709500E-01,.2066900E-01,&
     & .1007900E-01,.1733200E-01,.2032600E-01,.2228300E-01,.2345200E-01,&
     & .2384000E-01,.2335100E-01,.2170100E-01,.1615000E-01,.1025300E-01,&
     & .1762400E-01,.2073800E-01,.2277200E-01,.2396300E-01,.2434300E-01,&
     & .2383000E-01,.2215900E-01,.1660500E-01,.1041200E-01,.1791400E-01,&
     & .2116500E-01,.2325700E-01,.2449100E-01,.2490700E-01,.2436400E-01,&
     & .2263300E-01,.1703100E-01,.1052800E-01,.1820300E-01,.2158500E-01,&
     & .2373200E-01,.2501900E-01,.2546600E-01,.2492500E-01,.2312300E-01,&
     & .1744000E-01,.1059500E-01,.1849100E-01,.2198700E-01,.2420500E-01,&
     & .2552600E-01,.2601300E-01,.2549700E-01,.2363300E-01,.1782500E-01,&
     & .8367500E-02,.1477800E-01,.1745500E-01,.1920100E-01,.2021400E-01,&
     & .2054500E-01,.2013300E-01,.1870300E-01,.1381000E-01,.8503700E-02,&
     & .1503300E-01,.1782400E-01,.1962100E-01,.2068400E-01,.2105200E-01,&
     & .2061600E-01,.1914500E-01,.1421800E-01,.8612900E-02,.1530300E-01,&
     & .1819700E-01,.2004300E-01,.2116300E-01,.2157700E-01,.2114300E-01,&
     & .1960000E-01,.1461700E-01,.8691200E-02,.1556800E-01,.1856500E-01,&
     & .2047200E-01,.2163600E-01,.2209100E-01,.2167400E-01,.2009000E-01,&
     & .1499100E-01,.8730200E-02,.1580000E-01,.1891700E-01,.2088700E-01,&
     & .2210300E-01,.2259400E-01,.2219800E-01,.2058600E-01,.1535100E-01/

      data   absa(  1:180, 9)   /                                       &
     & .2875600E+00,.2846608E+00,.3007336E+00,.3116711E+00,.3128429E+00,&
     & .3061657E+00,.2955131E+00,.2798330E+00,.2885174E+00,.2846791E+00,&
     & .2818630E+00,.2979418E+00,.3088891E+00,.3105850E+00,.3045729E+00,&
     & .2947321E+00,.2798508E+00,.2884665E+00,.2823420E+00,.2795426E+00,&
     & .2956237E+00,.3062904E+00,.3083329E+00,.3027922E+00,.2937895E+00,&
     & .2797232E+00,.2884416E+00,.2804602E+00,.2775936E+00,.2936396E+00,&
     & .3040736E+00,.3061042E+00,.3009290E+00,.2927341E+00,.2793614E+00,&
     & .2882076E+00,.2789708E+00,.2760060E+00,.2919045E+00,.3021032E+00,&
     & .3039633E+00,.2991521E+00,.2914947E+00,.2788660E+00,.2878374E+00,&
     & .2691839E+00,.2738289E+00,.2925806E+00,.3047303E+00,.3076965E+00,&
     & .3021735E+00,.2903704E+00,.2740484E+00,.2753391E+00,.2667388E+00,&
     & .2713965E+00,.2901694E+00,.3021752E+00,.3055593E+00,.3006242E+00,&
     & .2897230E+00,.2740378E+00,.2755566E+00,.2649469E+00,.2695060E+00,&
     & .2881919E+00,.2999513E+00,.3034581E+00,.2989484E+00,.2889087E+00,&
     & .2738977E+00,.2755970E+00,.2636807E+00,.2680315E+00,.2865238E+00,&
     & .2980684E+00,.3014061E+00,.2973517E+00,.2879069E+00,.2735423E+00,&
     & .2754315E+00,.2627764E+00,.2669319E+00,.2851547E+00,.2963620E+00,&
     & .2995768E+00,.2957673E+00,.2866862E+00,.2730982E+00,.2752015E+00,&
     & .2490195E+00,.2608954E+00,.2819694E+00,.2948392E+00,.2994181E+00,&
     & .2960253E+00,.2839092E+00,.2667913E+00,.2610712E+00,.2470483E+00,&
     & .2588078E+00,.2799097E+00,.2926407E+00,.2974527E+00,.2945959E+00,&
     & .2834853E+00,.2668456E+00,.2614061E+00,.2457732E+00,.2572386E+00,&
     & .2782150E+00,.2907979E+00,.2955550E+00,.2931132E+00,.2827947E+00,&
     & .2667547E+00,.2614625E+00,.2450220E+00,.2561881E+00,.2768616E+00,&
     & .2892386E+00,.2937867E+00,.2916947E+00,.2819208E+00,.2664583E+00,&
     & .2614477E+00,.2447576E+00,.2555647E+00,.2758063E+00,.2878616E+00,&
     & .2922788E+00,.2903473E+00,.2808432E+00,.2661024E+00,.2614835E+00,&
     & .2278433E+00,.2459460E+00,.2687748E+00,.2823420E+00,.2883606E+00,&
     & .2867773E+00,.2761135E+00,.2583114E+00,.2469847E+00,.2262821E+00,&
     & .2442492E+00,.2669664E+00,.2804797E+00,.2866383E+00,.2856021E+00,&
     & .2758521E+00,.2584755E+00,.2474068E+00,.2254367E+00,.2430836E+00,&
     & .2656062E+00,.2789372E+00,.2850299E+00,.2843218E+00,.2753572E+00,&
     & .2584303E+00,.2476169E+00,.2252826E+00,.2424728E+00,.2646308E+00,&
     & .2777122E+00,.2836401E+00,.2831157E+00,.2746060E+00,.2582669E+00,&
     & .2478333E+00,.2256551E+00,.2423529E+00,.2639731E+00,.2766779E+00,&
     & .2825116E+00,.2820352E+00,.2736912E+00,.2580734E+00,.2481106E+00/

      data   absa(181:315, 9)   /                                       &
     & .2062751E+00,.2298044E+00,.2532990E+00,.2676292E+00,.2748784E+00,&
     & .2747996E+00,.2666545E+00,.2487411E+00,.2325541E+00,.2051043E+00,&
     & .2284554E+00,.2518239E+00,.2660870E+00,.2734853E+00,.2738569E+00,&
     & .2665350E+00,.2490789E+00,.2332102E+00,.2047919E+00,.2277211E+00,&
     & .2507972E+00,.2649403E+00,.2721940E+00,.2728699E+00,.2661103E+00,&
     & .2492251E+00,.2337220E+00,.2051961E+00,.2275744E+00,.2501902E+00,&
     & .2640418E+00,.2711870E+00,.2720181E+00,.2654582E+00,.2492504E+00,&
     & .2342238E+00,.2061900E+00,.2279645E+00,.2499630E+00,.2633833E+00,&
     & .2704230E+00,.2712077E+00,.2647461E+00,.2492510E+00,.2347478E+00,&
     & .1853388E+00,.2130833E+00,.2365145E+00,.2513589E+00,.2593134E+00,&
     & .2609466E+00,.2549939E+00,.2376988E+00,.2170867E+00,.1845601E+00,&
     & .2121024E+00,.2353552E+00,.2502019E+00,.2582784E+00,.2602288E+00,&
     & .2550486E+00,.2382771E+00,.2180680E+00,.1846602E+00,.2117219E+00,&
     & .2346887E+00,.2493886E+00,.2574166E+00,.2595177E+00,.2547983E+00,&
     & .2386926E+00,.2189256E+00,.1855458E+00,.2119966E+00,.2344799E+00,&
     & .2488166E+00,.2567948E+00,.2589356E+00,.2543373E+00,.2389459E+00,&
     & .2197901E+00,.1870772E+00,.2128598E+00,.2346986E+00,.2485625E+00,&
     & .2564000E+00,.2584353E+00,.2538996E+00,.2391912E+00,.2205482E+00,&
     & .1654511E+00,.1961135E+00,.2190853E+00,.2338982E+00,.2425818E+00,&
     & .2456184E+00,.2412218E+00,.2252744E+00,.2008814E+00,.1650292E+00,&
     & .1954131E+00,.2182427E+00,.2330965E+00,.2418982E+00,.2451802E+00,&
     & .2414790E+00,.2260855E+00,.2021875E+00,.1654771E+00,.1953910E+00,&
     & .2179136E+00,.2326080E+00,.2414414E+00,.2447571E+00,.2414612E+00,&
     & .2267328E+00,.2033987E+00,.1667613E+00,.1960735E+00,.2180521E+00,&
     & .2324354E+00,.2411480E+00,.2444710E+00,.2413228E+00,.2272824E+00,&
     & .2045050E+00,.1686834E+00,.1973572E+00,.2187006E+00,.2325944E+00,&
     & .2410697E+00,.2443064E+00,.2412853E+00,.2277903E+00,.2054612E+00/

      data   absa(316:450, 9)   /                                       &
     & .1466987E+00,.1792921E+00,.2014461E+00,.2160765E+00,.2251959E+00,&
     & .2288858E+00,.2256137E+00,.2115906E+00,.1843005E+00,.1465091E+00,&
     & .1788078E+00,.2009639E+00,.2155790E+00,.2248838E+00,.2288389E+00,&
     & .2260797E+00,.2126564E+00,.1858920E+00,.1472457E+00,.1790577E+00,&
     & .2009403E+00,.2154072E+00,.2247365E+00,.2287839E+00,.2263437E+00,&
     & .2135828E+00,.1874072E+00,.1487936E+00,.1800885E+00,.2014536E+00,&
     & .2156062E+00,.2248091E+00,.2288471E+00,.2265932E+00,.2144085E+00,&
     & .1887703E+00,.1510004E+00,.1817540E+00,.2025068E+00,.2162005E+00,&
     & .2250829E+00,.2290843E+00,.2269522E+00,.2151187E+00,.1899588E+00,&
     & .1290166E+00,.1628134E+00,.1841393E+00,.1984523E+00,.2074091E+00,&
     & .2111610E+00,.2086918E+00,.1966022E+00,.1675581E+00,.1290151E+00,&
     & .1624988E+00,.1838874E+00,.1982834E+00,.2074512E+00,.2114622E+00,&
     & .2094660E+00,.1978766E+00,.1694333E+00,.1299542E+00,.1629858E+00,&
     & .1841354E+00,.1984427E+00,.2076315E+00,.2117841E+00,.2100747E+00,&
     & .1990529E+00,.1712079E+00,.1317221E+00,.1642880E+00,.1849872E+00,&
     & .1989586E+00,.2080569E+00,.2122421E+00,.2107272E+00,.2001600E+00,&
     & .1728028E+00,.1340336E+00,.1662556E+00,.1864180E+00,.1999647E+00,&
     & .2087569E+00,.2128849E+00,.2114836E+00,.2011527E+00,.1742183E+00,&
     & .1126804E+00,.1466556E+00,.1672889E+00,.1811021E+00,.1894716E+00,&
     & .1930711E+00,.1911967E+00,.1807203E+00,.1511468E+00,.1128902E+00,&
     & .1466324E+00,.1672756E+00,.1812574E+00,.1898492E+00,.1936951E+00,&
     & .1922455E+00,.1822647E+00,.1532592E+00,.1140256E+00,.1474289E+00,&
     & .1678506E+00,.1817516E+00,.1903705E+00,.1944050E+00,.1932330E+00,&
     & .1836902E+00,.1552103E+00,.1159334E+00,.1490190E+00,.1690437E+00,&
     & .1826551E+00,.1911836E+00,.1952461E+00,.1942671E+00,.1850452E+00,&
     & .1569828E+00,.1181595E+00,.1512541E+00,.1708193E+00,.1840599E+00,&
     & .1923574E+00,.1963008E+00,.1954232E+00,.1863167E+00,.1585924E+00/

      data   absa(451:585, 9)   /                                       &
     & .9798189E-01,.1313265E+00,.1510348E+00,.1641642E+00,.1721410E+00,&
     & .1755759E+00,.1741644E+00,.1652309E+00,.1361983E+00,.9876591E-01,&
     & .1319241E+00,.1515858E+00,.1647606E+00,.1728744E+00,.1765573E+00,&
     & .1754929E+00,.1669557E+00,.1384236E+00,.1003841E+00,.1333210E+00,&
     & .1527464E+00,.1657657E+00,.1738505E+00,.1776840E+00,.1768404E+00,&
     & .1686021E+00,.1404407E+00,.1023815E+00,.1354349E+00,.1545144E+00,&
     & .1673004E+00,.1752277E+00,.1790049E+00,.1782869E+00,.1701818E+00,&
     & .1423084E+00,.1045182E+00,.1378387E+00,.1568227E+00,.1693189E+00,&
     & .1770176E+00,.1806205E+00,.1797803E+00,.1716817E+00,.1441206E+00,&
     & .8513105E-01,.1174272E+00,.1357845E+00,.1479723E+00,.1555369E+00,&
     & .1588474E+00,.1576920E+00,.1498764E+00,.1219425E+00,.8639013E-01,&
     & .1185546E+00,.1369085E+00,.1490184E+00,.1566903E+00,.1601862E+00,&
     & .1592810E+00,.1518099E+00,.1242064E+00,.8811019E-01,.1204518E+00,&
     & .1385898E+00,.1506015E+00,.1581843E+00,.1617403E+00,.1609764E+00,&
     & .1536670E+00,.1263021E+00,.9004252E-01,.1227432E+00,.1408853E+00,&
     & .1527101E+00,.1601143E+00,.1635634E+00,.1627369E+00,.1554561E+00,&
     & .1283211E+00,.9212719E-01,.1251177E+00,.1434973E+00,.1553078E+00,&
     & .1624820E+00,.1656416E+00,.1645751E+00,.1572251E+00,.1303753E+00,&
     & .7386256E-01,.1049065E+00,.1217281E+00,.1327938E+00,.1397977E+00,&
     & .1430454E+00,.1419673E+00,.1350204E+00,.1084738E+00,.7529440E-01,&
     & .1065145E+00,.1232849E+00,.1343471E+00,.1414069E+00,.1447364E+00,&
     & .1438645E+00,.1371204E+00,.1108041E+00,.7697705E-01,.1086621E+00,&
     & .1254864E+00,.1364483E+00,.1434325E+00,.1467317E+00,.1458386E+00,&
     & .1392068E+00,.1129989E+00,.7887272E-01,.1109542E+00,.1281014E+00,&
     & .1390876E+00,.1459050E+00,.1489849E+00,.1479204E+00,.1412350E+00,&
     & .1152445E+00,.8092106E-01,.1133621E+00,.1308019E+00,.1420681E+00,&
     & .1487342E+00,.1514606E+00,.1501482E+00,.1433560E+00,.1175338E+00/

      data   absa(  1:180,10)   /                                       &
     & .1892134E+01,.1658490E+01,.1482317E+01,.1379089E+01,.1354260E+01,&
     & .1396373E+01,.1493971E+01,.1670802E+01,.1862417E+01,.1891097E+01,&
     & .1657654E+01,.1481328E+01,.1378847E+01,.1353670E+01,.1394913E+01,&
     & .1490545E+01,.1665288E+01,.1856307E+01,.1886928E+01,.1654075E+01,&
     & .1478255E+01,.1377182E+01,.1351932E+01,.1393126E+01,.1486947E+01,&
     & .1659721E+01,.1849481E+01,.1880181E+01,.1648189E+01,.1473312E+01,&
     & .1373916E+01,.1349297E+01,.1390619E+01,.1483307E+01,.1654665E+01,&
     & .1843360E+01,.1870673E+01,.1639925E+01,.1466606E+01,.1369235E+01,&
     & .1345693E+01,.1387158E+01,.1479894E+01,.1649791E+01,.1837634E+01,&
     & .2033645E+01,.1786076E+01,.1609696E+01,.1508649E+01,.1481540E+01,&
     & .1520376E+01,.1619882E+01,.1790721E+01,.2011948E+01,.2032041E+01,&
     & .1784578E+01,.1608196E+01,.1508007E+01,.1480683E+01,.1518803E+01,&
     & .1616344E+01,.1785487E+01,.2005460E+01,.2027050E+01,.1780171E+01,&
     & .1604546E+01,.1505776E+01,.1478425E+01,.1516595E+01,.1612478E+01,&
     & .1780218E+01,.1999063E+01,.2019259E+01,.1773344E+01,.1599018E+01,&
     & .1501972E+01,.1475347E+01,.1513286E+01,.1608726E+01,.1775367E+01,&
     & .1993239E+01,.2008510E+01,.1763923E+01,.1591459E+01,.1496491E+01,&
     & .1471124E+01,.1509095E+01,.1605098E+01,.1770446E+01,.1987403E+01,&
     & .2178944E+01,.1920080E+01,.1745231E+01,.1648443E+01,.1618386E+01,&
     & .1649487E+01,.1748723E+01,.1914679E+01,.2160515E+01,.2176816E+01,&
     & .1918278E+01,.1743380E+01,.1647385E+01,.1617218E+01,.1647942E+01,&
     & .1744763E+01,.1909779E+01,.2154360E+01,.2171431E+01,.1913602E+01,&
     & .1739533E+01,.1644632E+01,.1614855E+01,.1645433E+01,.1740900E+01,&
     & .1904856E+01,.2148715E+01,.2162892E+01,.1906049E+01,.1733600E+01,&
     & .1640053E+01,.1611602E+01,.1641860E+01,.1736869E+01,.1900085E+01,&
     & .2142847E+01,.2151134E+01,.1895842E+01,.1725453E+01,.1633876E+01,&
     & .1606822E+01,.1637313E+01,.1732726E+01,.1894852E+01,.2136670E+01,&
     & .2323030E+01,.2056823E+01,.1886477E+01,.1793946E+01,.1761503E+01,&
     & .1786099E+01,.1877376E+01,.2039031E+01,.2297025E+01,.2320669E+01,&
     & .2054539E+01,.1884669E+01,.1792680E+01,.1760212E+01,.1784371E+01,&
     & .1873523E+01,.2034631E+01,.2291545E+01,.2314935E+01,.2049383E+01,&
     & .1880494E+01,.1789605E+01,.1757829E+01,.1781704E+01,.1869518E+01,&
     & .2030102E+01,.2286263E+01,.2305612E+01,.2041188E+01,.1873956E+01,&
     & .1784458E+01,.1754036E+01,.1777972E+01,.1865353E+01,.2025199E+01,&
     & .2280190E+01,.2293056E+01,.2030193E+01,.1865150E+01,.1777683E+01,&
     & .1748548E+01,.1773179E+01,.1860971E+01,.2019743E+01,.2273813E+01/

      data   absa(181:315,10)   /                                       &
     & .2463042E+01,.2192642E+01,.2030621E+01,.1942197E+01,.1908343E+01,&
     & .1927535E+01,.2006081E+01,.2161944E+01,.2425159E+01,.2460420E+01,&
     & .2190039E+01,.2028628E+01,.1940843E+01,.1906973E+01,.1925788E+01,&
     & .2002593E+01,.2157807E+01,.2420144E+01,.2453905E+01,.2184091E+01,&
     & .2024115E+01,.1937152E+01,.1904425E+01,.1922817E+01,.1998960E+01,&
     & .2153199E+01,.2414552E+01,.2443881E+01,.2175206E+01,.2017059E+01,&
     & .1931637E+01,.1900107E+01,.1918693E+01,.1994945E+01,.2148178E+01,&
     & .2408426E+01,.2430382E+01,.2163388E+01,.2007456E+01,.1924180E+01,&
     & .1893992E+01,.1913520E+01,.1990266E+01,.2142434E+01,.2401933E+01,&
     & .2595361E+01,.2325394E+01,.2174062E+01,.2090012E+01,.2056871E+01,&
     & .2069942E+01,.2136930E+01,.2284007E+01,.2549107E+01,.2592280E+01,&
     & .2322210E+01,.2171892E+01,.2088399E+01,.2055272E+01,.2068312E+01,&
     & .2133783E+01,.2280033E+01,.2544255E+01,.2585394E+01,.2315981E+01,&
     & .2166893E+01,.2084474E+01,.2052146E+01,.2065367E+01,.2130298E+01,&
     & .2275321E+01,.2538577E+01,.2574759E+01,.2306564E+01,.2159165E+01,&
     & .2078513E+01,.2047302E+01,.2061084E+01,.2126258E+01,.2270085E+01,&
     & .2532268E+01,.2560534E+01,.2294295E+01,.2148978E+01,.2070421E+01,&
     & .2040584E+01,.2055565E+01,.2121352E+01,.2264122E+01,.2525794E+01,&
     & .2717730E+01,.2453250E+01,.2313374E+01,.2235402E+01,.2202356E+01,&
     & .2210413E+01,.2268848E+01,.2403823E+01,.2668217E+01,.2714408E+01,&
     & .2449934E+01,.2311069E+01,.2233513E+01,.2200700E+01,.2208877E+01,&
     & .2266063E+01,.2400211E+01,.2663785E+01,.2707312E+01,.2443349E+01,&
     & .2305816E+01,.2229419E+01,.2197247E+01,.2206137E+01,.2262718E+01,&
     & .2395786E+01,.2658274E+01,.2696315E+01,.2433710E+01,.2297781E+01,&
     & .2222901E+01,.2191947E+01,.2201655E+01,.2258338E+01,.2390352E+01,&
     & .2652236E+01,.2681738E+01,.2420849E+01,.2287046E+01,.2214166E+01,&
     & .2184829E+01,.2195666E+01,.2252886E+01,.2384216E+01,.2646214E+01/

      data   absa(316:450,10)   /                                       &
     & .2830476E+01,.2574810E+01,.2446984E+01,.2374964E+01,.2342639E+01,&
     & .2348427E+01,.2400182E+01,.2520854E+01,.2781859E+01,.2827137E+01,&
     & .2571641E+01,.2444361E+01,.2373065E+01,.2340938E+01,.2346711E+01,&
     & .2397962E+01,.2517827E+01,.2778017E+01,.2819904E+01,.2565126E+01,&
     & .2438846E+01,.2368655E+01,.2337305E+01,.2343718E+01,.2394597E+01,&
     & .2513695E+01,.2772893E+01,.2808916E+01,.2555136E+01,.2430441E+01,&
     & .2361733E+01,.2331640E+01,.2338952E+01,.2389996E+01,.2508304E+01,&
     & .2767177E+01,.2794379E+01,.2541925E+01,.2419332E+01,.2352428E+01,&
     & .2324012E+01,.2332369E+01,.2384042E+01,.2502259E+01,.2761425E+01,&
     & .2933800E+01,.2689436E+01,.2572674E+01,.2506396E+01,.2476838E+01,&
     & .2482312E+01,.2528761E+01,.2636078E+01,.2889382E+01,.2930816E+01,&
     & .2686703E+01,.2570265E+01,.2504494E+01,.2475090E+01,.2480774E+01,&
     & .2526766E+01,.2633891E+01,.2886327E+01,.2923537E+01,.2680192E+01,&
     & .2564715E+01,.2499735E+01,.2471291E+01,.2477512E+01,.2523484E+01,&
     & .2630022E+01,.2881726E+01,.2912511E+01,.2670142E+01,.2556117E+01,&
     & .2492614E+01,.2465215E+01,.2472392E+01,.2518562E+01,.2624696E+01,&
     & .2876581E+01,.2898553E+01,.2656790E+01,.2544582E+01,.2482931E+01,&
     & .2457049E+01,.2465309E+01,.2512229E+01,.2618696E+01,.2871034E+01,&
     & .3027058E+01,.2797212E+01,.2690074E+01,.2629551E+01,.2604067E+01,&
     & .2610197E+01,.2652331E+01,.2748363E+01,.2990031E+01,.3023852E+01,&
     & .2794164E+01,.2687619E+01,.2627303E+01,.2602246E+01,.2608704E+01,&
     & .2650569E+01,.2746538E+01,.2987720E+01,.3016507E+01,.2787386E+01,&
     & .2681752E+01,.2622344E+01,.2598218E+01,.2605186E+01,.2647180E+01,&
     & .2742958E+01,.2983973E+01,.3005577E+01,.2777090E+01,.2672822E+01,&
     & .2614852E+01,.2591693E+01,.2599702E+01,.2642041E+01,.2737917E+01,&
     & .2979411E+01,.2992905E+01,.2763507E+01,.2660993E+01,.2604652E+01,&
     & .2582864E+01,.2591914E+01,.2635114E+01,.2731878E+01,.2974194E+01/

      data   absa(451:585,10)   /                                       &
     & .3108634E+01,.2895420E+01,.2798425E+01,.2744368E+01,.2722653E+01,&
     & .2730122E+01,.2769336E+01,.2855617E+01,.3083249E+01,.3103552E+01,&
     & .2890636E+01,.2794237E+01,.2740840E+01,.2719834E+01,.2727815E+01,&
     & .2767022E+01,.2853516E+01,.3081043E+01,.3094635E+01,.2882058E+01,&
     & .2786673E+01,.2734481E+01,.2714488E+01,.2723042E+01,.2762618E+01,&
     & .2849365E+01,.3077793E+01,.3083887E+01,.2870120E+01,.2776133E+01,&
     & .2725197E+01,.2706459E+01,.2715971E+01,.2756337E+01,.2843995E+01,&
     & .3073502E+01,.3072208E+01,.2856377E+01,.2762710E+01,.2713207E+01,&
     & .2695854E+01,.2706674E+01,.2748406E+01,.2837477E+01,.3067985E+01,&
     & .3179487E+01,.2983632E+01,.2897614E+01,.2850758E+01,.2832739E+01,&
     & .2841435E+01,.2878587E+01,.2957109E+01,.3169588E+01,.3172853E+01,&
     & .2977291E+01,.2891777E+01,.2845999E+01,.2828739E+01,.2838054E+01,&
     & .2875642E+01,.2954402E+01,.3167982E+01,.3164067E+01,.2967272E+01,&
     & .2882837E+01,.2838175E+01,.2822030E+01,.2832002E+01,.2870360E+01,&
     & .2950082E+01,.3164988E+01,.3154001E+01,.2955025E+01,.2870648E+01,&
     & .2827230E+01,.2812357E+01,.2823496E+01,.2863128E+01,.2944286E+01,&
     & .3160501E+01,.3142799E+01,.2941633E+01,.2856439E+01,.2813383E+01,&
     & .2800017E+01,.2812937E+01,.2854322E+01,.2937121E+01,.3154451E+01,&
     & .3241097E+01,.3062217E+01,.2987360E+01,.2948157E+01,.2934292E+01,&
     & .2943847E+01,.2979640E+01,.3051660E+01,.3249168E+01,.3234006E+01,&
     & .3054284E+01,.2980255E+01,.2942054E+01,.2929013E+01,.2939386E+01,&
     & .2975962E+01,.3048876E+01,.3247803E+01,.3225888E+01,.3043668E+01,&
     & .2969796E+01,.2932743E+01,.2920751E+01,.2932040E+01,.2969899E+01,&
     & .3044051E+01,.3244864E+01,.3216320E+01,.3031794E+01,.2956859E+01,&
     & .2920139E+01,.2909436E+01,.2922425E+01,.2961874E+01,.3037648E+01,&
     & .3239868E+01,.3205483E+01,.3018784E+01,.2942726E+01,.2905212E+01,&
     & .2895638E+01,.2910719E+01,.2952008E+01,.3029378E+01,.3233244E+01/


      data   absb(  1:175, 1)   /                                       &
     & .4063300E-08,.6471800E-06,.7956700E-06,.8126400E-06,.2508000E-07,&
     & .3759600E-08,.7042900E-06,.9085500E-06,.9643200E-06,.2829200E-07,&
     & .3498200E-08,.7637000E-06,.1031600E-05,.1147200E-05,.3126500E-07,&
     & .3270700E-08,.8283400E-06,.1157700E-05,.1338700E-05,.3492800E-07,&
     & .3071000E-08,.9025200E-06,.1296200E-05,.1520600E-05,.3885900E-07,&
     & .3302600E-08,.5359200E-06,.6617100E-06,.6763300E-06,.2076700E-07,&
     & .3057400E-08,.5795900E-06,.7538600E-06,.8039500E-06,.2333400E-07,&
     & .2846100E-08,.6289700E-06,.8548600E-06,.9574500E-06,.2584100E-07,&
     & .2662100E-08,.6835200E-06,.9613200E-06,.1114800E-05,.2886200E-07,&
     & .2500500E-08,.7451700E-06,.1073900E-05,.1263200E-05,.3209300E-07,&
     & .2682100E-08,.4408100E-06,.5509300E-06,.5646500E-06,.1718600E-07,&
     & .2484500E-08,.4777300E-06,.6263600E-06,.6718000E-06,.1923400E-07,&
     & .2314000E-08,.5181100E-06,.7079100E-06,.7965800E-06,.2134800E-07,&
     & .2165400E-08,.5641600E-06,.7989300E-06,.9287300E-06,.2383200E-07,&
     & .2034700E-08,.6153500E-06,.8898500E-06,.1049900E-05,.2648500E-07,&
     & .2177500E-08,.3625400E-06,.4587000E-06,.4712500E-06,.1421100E-07,&
     & .2018300E-08,.3937600E-06,.5205300E-06,.5616300E-06,.1586900E-07,&
     & .1880800E-08,.4268700E-06,.5869900E-06,.6638600E-06,.1762700E-07,&
     & .1760800E-08,.4655900E-06,.6638800E-06,.7706300E-06,.1966400E-07,&
     & .1655200E-08,.5074700E-06,.7369900E-06,.8724900E-06,.2184400E-07,&
     & .1768200E-08,.2983600E-06,.3810400E-06,.3931400E-06,.1173500E-07,&
     & .1639900E-08,.3249300E-06,.4321500E-06,.4689700E-06,.1307700E-07,&
     & .1529000E-08,.3520300E-06,.4866500E-06,.5532800E-06,.1453500E-07,&
     & .1432100E-08,.3838300E-06,.5503300E-06,.6384400E-06,.1620800E-07,&
     & .1346800E-08,.4182200E-06,.6095000E-06,.7237600E-06,.1799500E-07,&
     & .1436200E-08,.2456000E-06,.3166700E-06,.3279700E-06,.9683100E-08,&
     & .1332700E-08,.2678400E-06,.3587200E-06,.3913400E-06,.1077300E-07,&
     & .1243200E-08,.2900800E-06,.4033400E-06,.4602600E-06,.1198000E-07,&
     & .1165000E-08,.3163700E-06,.4558300E-06,.5288000E-06,.1335000E-07,&
     & .1096000E-08,.3450300E-06,.5036300E-06,.5996700E-06,.1481500E-07,&
     & .1166700E-08,.2023100E-06,.2631300E-06,.2735900E-06,.7993400E-08,&
     & .1083300E-08,.2208500E-06,.2976000E-06,.3262000E-06,.8880700E-08,&
     & .1011000E-08,.2391200E-06,.3342300E-06,.3827100E-06,.9877100E-08,&
     & .9477800E-09,.2607300E-06,.3768500E-06,.4379100E-06,.1100200E-07,&
     & .8919800E-09,.2842500E-06,.4158900E-06,.4968200E-06,.1220400E-07/

      data   absb(176:350, 1)   /                                       &
     & .9466800E-09,.1673500E-06,.2182200E-06,.2291400E-06,.6623700E-08,&
     & .8795600E-09,.1825500E-06,.2477000E-06,.2729000E-06,.7347000E-08,&
     & .8213200E-09,.1977600E-06,.2778600E-06,.3191300E-06,.8173200E-08,&
     & .7703200E-09,.2155700E-06,.3119700E-06,.3638000E-06,.9101900E-08,&
     & .7252800E-09,.2345800E-06,.3444900E-06,.4125700E-06,.1008400E-07,&
     & .7680900E-09,.1385800E-06,.1813800E-06,.1918800E-06,.5492900E-08,&
     & .7140800E-09,.1508800E-06,.2061400E-06,.2283500E-06,.6082500E-08,&
     & .6671700E-09,.1636600E-06,.2311100E-06,.2662200E-06,.6768000E-08,&
     & .6260500E-09,.1781600E-06,.2583400E-06,.3024200E-06,.7534400E-08,&
     & .5897000E-09,.1937400E-06,.2854200E-06,.3422200E-06,.8339400E-08,&
     & .6201300E-09,.1154700E-06,.1521200E-06,.1622700E-06,.4592600E-08,&
     & .5770900E-09,.1255200E-06,.1721100E-06,.1926500E-06,.5072800E-08,&
     & .5396400E-09,.1361900E-06,.1936300E-06,.2240200E-06,.5645400E-08,&
     & .5067500E-09,.1480500E-06,.2155700E-06,.2532000E-06,.6280500E-08,&
     & .4776400E-09,.1609700E-06,.2379900E-06,.2857400E-06,.6941400E-08,&
     & .5005500E-09,.9616500E-07,.1277600E-06,.1375200E-06,.3793300E-08,&
     & .4662700E-09,.1042100E-06,.1438700E-06,.1622600E-06,.4215400E-08,&
     & .4363800E-09,.1132000E-06,.1624000E-06,.1880600E-06,.4676100E-08,&
     & .4100900E-09,.1231200E-06,.1800200E-06,.2121400E-06,.5172400E-08,&
     & .3867900E-09,.1338000E-06,.1983500E-06,.2383600E-06,.5686300E-08,&
     & .4038400E-09,.8017400E-07,.1074200E-06,.1166500E-06,.3143500E-08,&
     & .3765600E-09,.8661100E-07,.1206200E-06,.1369400E-06,.3474600E-08,&
     & .3527300E-09,.9420300E-07,.1361000E-06,.1572900E-06,.3858600E-08,&
     & .3317400E-09,.1024600E-06,.1503500E-06,.1777500E-06,.4270700E-08,&
     & .3131000E-09,.1112200E-06,.1654300E-06,.1991700E-06,.4702000E-08,&
     & .3256200E-09,.6678500E-07,.9044700E-07,.9905900E-07,.2613400E-08,&
     & .3039300E-09,.7213300E-07,.1013900E-06,.1157100E-06,.2887200E-08,&
     & .2849500E-09,.7844800E-07,.1138500E-06,.1317800E-06,.3206100E-08,&
     & .2682100E-09,.8533900E-07,.1256300E-06,.1491400E-06,.3548900E-08,&
     & .2533200E-09,.9244500E-07,.1381400E-06,.1667400E-06,.3906000E-08,&
     & .2622400E-09,.5568000E-07,.7636900E-07,.8440000E-07,.2175900E-08,&
     & .2450400E-09,.6020400E-07,.8542300E-07,.9788500E-07,.2411800E-08,&
     & .2299600E-09,.6547700E-07,.9532100E-07,.1109300E-06,.2677400E-08,&
     & .2166300E-09,.7118800E-07,.1052900E-06,.1254400E-06,.2961500E-08,&
     & .2047500E-09,.7694000E-07,.1155600E-06,.1397700E-06,.3256900E-08/

      data   absb(351:525, 1)   /                                       &
     & .2112000E-09,.4638200E-07,.6389400E-07,.7137900E-07,.1817800E-08,&
     & .1975600E-09,.5027800E-07,.7203300E-07,.8290900E-07,.2018100E-08,&
     & .1855700E-09,.5467000E-07,.7995300E-07,.9350500E-07,.2239600E-08,&
     & .1749600E-09,.5940800E-07,.8817900E-07,.1055300E-06,.2473700E-08,&
     & .1654900E-09,.6402500E-07,.9669600E-07,.1170400E-06,.2709200E-08,&
     & .1701000E-09,.3863100E-07,.5377300E-07,.6057900E-07,.1522000E-08,&
     & .1592800E-09,.4200100E-07,.6066000E-07,.6972700E-07,.1690100E-08,&
     & .1497600E-09,.4566100E-07,.6706500E-07,.7884500E-07,.1869600E-08,&
     & .1413100E-09,.4957100E-07,.7380900E-07,.8848100E-07,.2061100E-08,&
     & .1337600E-09,.5330400E-07,.8091000E-07,.9801700E-07,.2256500E-08,&
     & .1369600E-09,.3227000E-07,.4539200E-07,.5143800E-07,.1272100E-08,&
     & .1283900E-09,.3507900E-07,.5094600E-07,.5872600E-07,.1411800E-08,&
     & .1208200E-09,.3815000E-07,.5625800E-07,.6649300E-07,.1562800E-08,&
     & .1141000E-09,.4132400E-07,.6184300E-07,.7438700E-07,.1720500E-08,&
     & .1080800E-09,.4438700E-07,.6773900E-07,.8221600E-07,.1874500E-08,&
     & .1102900E-09,.2697000E-07,.3833000E-07,.4369800E-07,.1065500E-08,&
     & .1034900E-09,.2931900E-07,.4273900E-07,.4955200E-07,.1182300E-08,&
     & .9748300E-10,.3186900E-07,.4721000E-07,.5605700E-07,.1303800E-08,&
     & .9213300E-10,.3443600E-07,.5181200E-07,.6248500E-07,.1428500E-08,&
     & .8734000E-10,.3695300E-07,.5670900E-07,.6892100E-07,.1556300E-08,&
     & .8880600E-10,.2255500E-07,.3238600E-07,.3712900E-07,.8902900E-09,&
     & .8341700E-10,.2451900E-07,.3592100E-07,.4187000E-07,.9854800E-09,&
     & .7864400E-10,.2663800E-07,.3960900E-07,.4727300E-07,.1085400E-08,&
     & .7438800E-10,.2869400E-07,.4342300E-07,.5243500E-07,.1188000E-08,&
     & .7056800E-10,.3077600E-07,.4741100E-07,.5775000E-07,.1293100E-08,&
     & .7150700E-10,.1887000E-07,.2733200E-07,.3130900E-07,.7443600E-09,&
     & .6723500E-10,.2051200E-07,.3018900E-07,.3540700E-07,.8229400E-09,&
     & .6344400E-10,.2225900E-07,.3321800E-07,.3973000E-07,.9048300E-09,&
     & .6005800E-10,.2392300E-07,.3640500E-07,.4397600E-07,.9890800E-09,&
     & .5701500E-10,.2563400E-07,.3954800E-07,.4834600E-07,.1075000E-08,&
     & .5758600E-10,.1577900E-07,.2295000E-07,.2641500E-07,.6225600E-09,&
     & .5419900E-10,.1715300E-07,.2535400E-07,.2990400E-07,.6873400E-09,&
     & .5118800E-10,.1856900E-07,.2786900E-07,.3340400E-07,.7546700E-09,&
     & .4849300E-10,.1993500E-07,.3043700E-07,.3685600E-07,.8236900E-09,&
     & .4606900E-10,.2134500E-07,.3298800E-07,.4047800E-07,.8940200E-09/

      data   absb(526:700, 1)   /                                       &
     & .4648700E-10,.1316100E-07,.1921600E-07,.2221200E-07,.5191000E-09,&
     & .4378900E-10,.1430100E-07,.2118200E-07,.2509000E-07,.5724700E-09,&
     & .4138600E-10,.1544700E-07,.2320800E-07,.2793800E-07,.6276700E-09,&
     & .3923400E-10,.1657100E-07,.2534700E-07,.3079500E-07,.6842600E-09,&
     & .3729400E-10,.1772500E-07,.2739200E-07,.3378100E-07,.7417900E-09,&
     & .3775600E-10,.1089000E-07,.1588200E-07,.1844100E-07,.4287100E-09,&
     & .3558100E-10,.1183000E-07,.1751400E-07,.2081000E-07,.4725100E-09,&
     & .3364300E-10,.1275700E-07,.1918400E-07,.2314800E-07,.5176900E-09,&
     & .3190500E-10,.1368200E-07,.2094100E-07,.2547700E-07,.5639400E-09,&
     & .3033800E-10,.1462800E-07,.2261300E-07,.2794800E-07,.6110000E-09,&
     & .3087200E-10,.8929000E-08,.1300700E-07,.1510800E-07,.3503400E-09,&
     & .2909600E-10,.9699500E-08,.1434300E-07,.1705100E-07,.3860900E-09,&
     & .2751300E-10,.1045600E-07,.1571300E-07,.1896800E-07,.4229600E-09,&
     & .2609300E-10,.1121300E-07,.1715300E-07,.2087500E-07,.4607300E-09,&
     & .2481200E-10,.1198700E-07,.1852200E-07,.2289900E-07,.4991600E-09,&
     & .2547900E-10,.7233800E-08,.1050800E-07,.1216500E-07,.2826800E-09,&
     & .2400200E-10,.7859300E-08,.1159400E-07,.1374700E-07,.3117300E-09,&
     & .2268700E-10,.8484500E-08,.1271200E-07,.1531500E-07,.3417800E-09,&
     & .2150800E-10,.9099700E-08,.1389200E-07,.1688600E-07,.3726000E-09,&
     & .2044600E-10,.9732200E-08,.1501200E-07,.1852600E-07,.4039800E-09,&
     & .2104100E-10,.5856500E-08,.8483100E-08,.9789200E-08,.2279100E-09,&
     & .1981100E-10,.6363600E-08,.9361900E-08,.1107200E-07,.2514800E-09,&
     & .1871700E-10,.6878400E-08,.1027600E-07,.1236000E-07,.2759500E-09,&
     & .1773800E-10,.7379200E-08,.1123400E-07,.1363800E-07,.3010900E-09,&
     & .1685600E-10,.7897100E-08,.1216600E-07,.1497700E-07,.3267100E-09,&
     & .1737700E-10,.4741800E-08,.6856100E-08,.7879500E-08,.1837300E-09,&
     & .1635300E-10,.5153200E-08,.7562000E-08,.8912800E-08,.2028400E-09,&
     & .1544300E-10,.5575200E-08,.8306900E-08,.9966600E-08,.2227900E-09,&
     & .1462900E-10,.5985700E-08,.9087200E-08,.1101100E-07,.2432700E-09,&
     & .1389600E-10,.6408600E-08,.9861600E-08,.1210400E-07,.2642000E-09,&
     & .1444500E-10,.3809100E-08,.5479900E-08,.6275400E-08,.1467900E-09,&
     & .1358100E-10,.4139100E-08,.6053000E-08,.7090900E-08,.1622000E-09,&
     & .1281500E-10,.4488800E-08,.6655500E-08,.7953200E-08,.1784200E-09,&
     & .1213100E-10,.4820700E-08,.7285500E-08,.8807100E-08,.1950900E-09,&
     & .1151600E-10,.5164600E-08,.7926800E-08,.9693200E-08,.2121400E-09/

      data   absb(701:875, 1)   /                                       &
     & .1202100E-10,.3056600E-08,.4360600E-08,.4992100E-08,.1171300E-09,&
     & .1129100E-10,.3320500E-08,.4834000E-08,.5625600E-08,.1295400E-09,&
     & .1064500E-10,.3605300E-08,.5326100E-08,.6347300E-08,.1426900E-09,&
     & .1006900E-10,.3878200E-08,.5834700E-08,.7038600E-08,.1562400E-09,&
     & .9552000E-11,.4158000E-08,.6364700E-08,.7751500E-08,.1701300E-09,&
     & .1000700E-10,.2451600E-08,.3465300E-08,.3939800E-08,.9347500E-10,&
     & .9390200E-11,.2663100E-08,.3858800E-08,.4463500E-08,.1034200E-09,&
     & .8844900E-11,.2892600E-08,.4260000E-08,.5046600E-08,.1140600E-09,&
     & .8359400E-11,.3119800E-08,.4671900E-08,.5621800E-08,.1250700E-09,&
     & .7924400E-11,.3346500E-08,.5106500E-08,.6199900E-08,.1364000E-09,&
     & .8364700E-11,.1957800E-08,.2734100E-08,.3082700E-08,.7427700E-10,&
     & .7838900E-11,.2126300E-08,.3069000E-08,.3521200E-08,.8211800E-10,&
     & .7375200E-11,.2310300E-08,.3386500E-08,.3983600E-08,.9067200E-10,&
     & .6963300E-11,.2498400E-08,.3719800E-08,.4456900E-08,.9962800E-10,&
     & .6595000E-11,.2683300E-08,.4070900E-08,.4926500E-08,.1088300E-09,&
     & .7007000E-11,.1562800E-08,.2152100E-08,.2401700E-08,.5911600E-10,&
     & .6557000E-11,.1695400E-08,.2425400E-08,.2777200E-08,.6502700E-10,&
     & .6161200E-11,.1841300E-08,.2684500E-08,.3131000E-08,.7189200E-10,&
     & .5810600E-11,.1997700E-08,.2954200E-08,.3521100E-08,.7914400E-10,&
     & .5497600E-11,.2146400E-08,.3236000E-08,.3904500E-08,.8662900E-10,&
     & .5872800E-11,.1250000E-08,.1707900E-08,.1874300E-08,.4664600E-10,&
     & .5487300E-11,.1350500E-08,.1910500E-08,.2171900E-08,.5149900E-10,&
     & .5149200E-11,.1467000E-08,.2125000E-08,.2459000E-08,.5698200E-10,&
     & .4850400E-11,.1593100E-08,.2346400E-08,.2780200E-08,.6283900E-10,&
     & .4584400E-11,.1716700E-08,.2573100E-08,.3095400E-08,.6890800E-10,&
     & .4932800E-11,.9981100E-09,.1339700E-08,.1445700E-08,.3669400E-10,&
     & .4601200E-11,.1074200E-08,.1498600E-08,.1687500E-08,.4074200E-10,&
     & .4311300E-11,.1166500E-08,.1681300E-08,.1928100E-08,.4504900E-10,&
     & .4055800E-11,.1267300E-08,.1856700E-08,.2181700E-08,.4975100E-10,&
     & .3828800E-11,.1370200E-08,.2039900E-08,.2441900E-08,.5467900E-10,&
     & .4159200E-11,.7920500E-09,.1043800E-08,.1105500E-08,.2870900E-10,&
     & .3871700E-11,.8535700E-09,.1171000E-08,.1301500E-08,.3219800E-10,&
     & .3621500E-11,.9251500E-09,.1319600E-08,.1510300E-08,.3544800E-10,&
     & .3401600E-11,.1004600E-08,.1461900E-08,.1701000E-08,.3920300E-10,&
     & .3206900E-11,.1090400E-08,.1610000E-08,.1916800E-08,.4318700E-10/

      data   absb(876:1050, 1)   /                                      &
     & .3510000E-11,.6284900E-09,.8131900E-09,.8456700E-09,.2245600E-10,&
     & .3260500E-11,.6802100E-09,.9216900E-09,.1004000E-08,.2522900E-10,&
     & .3044100E-11,.7329400E-09,.1031400E-08,.1166500E-08,.2790300E-10,&
     & .2854700E-11,.7961500E-09,.1149500E-08,.1325100E-08,.3087900E-10,&
     & .2687400E-11,.8646300E-09,.1270300E-08,.1499400E-08,.3408200E-10,&
     & .2964900E-11,.4986700E-09,.6357200E-09,.6455700E-09,.1755400E-10,&
     & .2748000E-11,.5412400E-09,.7187100E-09,.7684000E-09,.1974900E-10,&
     & .2560600E-11,.5814000E-09,.8041300E-09,.9005400E-09,.2201200E-10,&
     & .2397200E-11,.6309700E-09,.9063700E-09,.1034900E-08,.2430800E-10,&
     & .2253400E-11,.6854500E-09,.1000600E-08,.1170400E-08,.2687200E-10,&
     & .2501900E-11,.3985300E-09,.4937400E-09,.4961500E-09,.1376200E-10,&
     & .2313600E-11,.4295200E-09,.5614100E-09,.5898000E-09,.1550000E-10,&
     & .2151700E-11,.4633400E-09,.6328400E-09,.6978800E-09,.1739000E-10,&
     & .2011000E-11,.5013000E-09,.7112400E-09,.8090300E-09,.1918400E-10,&
     & .1887500E-11,.5444600E-09,.7890800E-09,.9143500E-09,.2123000E-10,&
     & .2111100E-11,.3153400E-09,.3855200E-09,.3832400E-09,.1085100E-10,&
     & .1947800E-11,.3417900E-09,.4398600E-09,.4535500E-09,.1217600E-10,&
     & .1807900E-11,.3701600E-09,.4983500E-09,.5391900E-09,.1368300E-10,&
     & .1686800E-11,.3984600E-09,.5574600E-09,.6280500E-09,.1516400E-10,&
     & .1580900E-11,.4327200E-09,.6231000E-09,.7158700E-09,.1678500E-10,&
     & .1783000E-11,.2493900E-09,.3003300E-09,.2970800E-09,.8548000E-11,&
     & .1641000E-11,.2724400E-09,.3446700E-09,.3487000E-09,.9564200E-11,&
     & .1520000E-11,.2950700E-09,.3902000E-09,.4150500E-09,.1076400E-10,&
     & .1415600E-11,.3173600E-09,.4369300E-09,.4875800E-09,.1201600E-10,&
     & .1324600E-11,.3440900E-09,.4924300E-09,.5618100E-09,.1326700E-10,&
     & .1507400E-11,.1974600E-09,.2344200E-09,.2302700E-09,.6694200E-11,&
     & .1383800E-11,.2175400E-09,.2681200E-09,.2686500E-09,.7509500E-11,&
     & .1278900E-11,.2344500E-09,.3051000E-09,.3191600E-09,.8462600E-11,&
     & .1188800E-11,.2530500E-09,.3449600E-09,.3790600E-09,.9496300E-11,&
     & .1110600E-11,.2735500E-09,.3868000E-09,.4388200E-09,.1048600E-10,&
     & .1272200E-11,.1567600E-09,.1843600E-09,.1818800E-09,.5260100E-11,&
     & .1164900E-11,.1725600E-09,.2103000E-09,.2087600E-09,.5946500E-11,&
     & .1074300E-11,.1870600E-09,.2402200E-09,.2465700E-09,.6675000E-11,&
     & .9968300E-12,.2026300E-09,.2720600E-09,.2933700E-09,.7501600E-11,&
     & .9297500E-12,.2179800E-09,.3042000E-09,.3420300E-09,.8318200E-11/

      data   absb(1051:1175, 1)   /                                     &
     & .1073300E-11,.1255200E-09,.1466800E-09,.1447700E-09,.4144000E-11,&
     & .9802700E-12,.1370500E-09,.1648000E-09,.1629800E-09,.4713700E-11,&
     & .9020900E-12,.1496600E-09,.1891100E-09,.1911700E-09,.5273000E-11,&
     & .8354600E-12,.1621000E-09,.2141000E-09,.2275800E-09,.5934900E-11,&
     & .7780000E-12,.1742300E-09,.2397200E-09,.2672500E-09,.6616300E-11,&
     & .9062400E-12,.9978600E-10,.1167400E-09,.1128600E-09,.3277600E-11,&
     & .8254700E-12,.1089700E-09,.1293800E-09,.1270500E-09,.3716100E-11,&
     & .7579100E-12,.1200900E-09,.1480400E-09,.1482700E-09,.4164600E-11,&
     & .7005800E-12,.1292500E-09,.1683700E-09,.1763000E-09,.4695700E-11,&
     & .6513100E-12,.1393700E-09,.1899900E-09,.2089900E-09,.5274200E-11,&
     & .7659500E-12,.7882200E-10,.9386300E-10,.8752900E-10,.2603300E-11,&
     & .6956800E-12,.8655500E-10,.1019000E-09,.1002200E-09,.2928800E-11,&
     & .6372100E-12,.9538500E-10,.1164000E-09,.1153300E-09,.3299400E-11,&
     & .5878200E-12,.1032500E-09,.1326400E-09,.1366300E-09,.3718600E-11,&
     & .5455200E-12,.1117300E-09,.1504200E-09,.1625300E-09,.4181900E-11,&
     & .6469300E-12,.6242100E-10,.7617400E-10,.6832400E-10,.2083300E-11,&
     & .5858800E-12,.6927800E-10,.8119700E-10,.7969100E-10,.2313000E-11,&
     & .5353500E-12,.7587300E-10,.9141400E-10,.9039800E-10,.2635000E-11,&
     & .4928500E-12,.8265300E-10,.1049300E-09,.1061700E-09,.2947300E-11,&
     & .4566000E-12,.8964300E-10,.1186100E-09,.1264700E-09,.3323100E-11,&
     & .5364500E-12,.5055700E-10,.6224000E-10,.5485400E-10,.1693000E-11,&
     & .4852300E-12,.5646800E-10,.6560400E-10,.6482700E-10,.1870200E-11,&
     & .4429400E-12,.6144200E-10,.7366700E-10,.7268800E-10,.2129200E-11,&
     & .4074400E-12,.6718400E-10,.8449000E-10,.8521700E-10,.2383200E-11,&
     & .3772000E-12,.7265700E-10,.9569200E-10,.1014400E-09,.2685200E-11/

      data   absb(  1:175, 2)   /                                       &
     & .1832300E-07,.8626600E-05,.1372800E-04,.1595200E-04,.7892700E-06,&
     & .1696100E-07,.9620300E-05,.1527500E-04,.1786300E-04,.9365900E-06,&
     & .1578700E-07,.1067500E-04,.1692400E-04,.1967500E-04,.1105000E-05,&
     & .1476400E-07,.1177900E-04,.1847400E-04,.2132000E-04,.1286800E-05,&
     & .1386500E-07,.1284300E-04,.2002900E-04,.2308000E-04,.1456500E-05,&
     & .1489300E-07,.7188200E-05,.1144300E-04,.1327100E-04,.6583500E-06,&
     & .1379300E-07,.8027600E-05,.1273200E-04,.1489000E-04,.7813200E-06,&
     & .1284400E-07,.8918900E-05,.1410100E-04,.1633900E-04,.9223300E-06,&
     & .1201700E-07,.9830900E-05,.1538400E-04,.1772100E-04,.1067600E-05,&
     & .1128900E-07,.1070400E-04,.1667900E-04,.1919400E-04,.1209100E-05,&
     & .1209600E-07,.5993400E-05,.9544900E-05,.1105800E-04,.5496900E-06,&
     & .1120900E-07,.6693200E-05,.1062300E-04,.1240700E-04,.6519700E-06,&
     & .1044300E-07,.7444900E-05,.1174600E-04,.1357100E-04,.7694000E-06,&
     & .9774700E-08,.8194800E-05,.1281400E-04,.1473200E-04,.8849600E-06,&
     & .9186400E-08,.8918800E-05,.1388300E-04,.1594600E-04,.1002300E-05,&
     & .9820500E-08,.5002700E-05,.7964400E-05,.9224800E-05,.4587600E-06,&
     & .9106100E-08,.5584600E-05,.8859300E-05,.1033100E-04,.5433400E-06,&
     & .8488200E-08,.6218300E-05,.9775700E-05,.1127900E-04,.6405000E-06,&
     & .7948600E-08,.6827600E-05,.1066400E-04,.1224400E-04,.7334400E-06,&
     & .7473200E-08,.7426800E-05,.1154500E-04,.1322400E-04,.8295500E-06,&
     & .7975300E-08,.4171300E-05,.6615300E-05,.7689200E-05,.3824700E-06,&
     & .7399400E-08,.4656200E-05,.7375300E-05,.8592200E-05,.4518800E-06,&
     & .6900800E-08,.5182900E-05,.8128000E-05,.9363800E-05,.5321100E-06,&
     & .6465000E-08,.5676500E-05,.8856000E-05,.1016800E-04,.6064700E-06,&
     & .6080700E-08,.6175800E-05,.9589400E-05,.1094900E-04,.6854900E-06,&
     & .6477900E-08,.3472700E-05,.5487500E-05,.6400000E-05,.3185100E-06,&
     & .6013500E-08,.3875100E-05,.6130800E-05,.7137200E-05,.3755200E-06,&
     & .5611100E-08,.4313200E-05,.6747800E-05,.7769800E-05,.4414100E-06,&
     & .5259000E-08,.4715000E-05,.7344400E-05,.8437000E-05,.5007500E-06,&
     & .4948300E-08,.5134500E-05,.7949500E-05,.9063700E-05,.5660100E-06,&
     & .5262700E-08,.2890800E-05,.4558600E-05,.5326000E-05,.2651600E-06,&
     & .4888200E-08,.3226000E-05,.5094000E-05,.5920000E-05,.3119000E-06,&
     & .4563200E-08,.3584300E-05,.5599200E-05,.6439700E-05,.3660800E-06,&
     & .4278600E-08,.3916700E-05,.6092300E-05,.6997200E-05,.4136100E-06,&
     & .4027300E-08,.4266700E-05,.6590400E-05,.7502100E-05,.4674100E-06/

      data   absb(176:350, 2)   /                                       &
     & .4270400E-08,.2407400E-05,.3799000E-05,.4445800E-05,.2210300E-06,&
     & .3969000E-08,.2694200E-05,.4239300E-05,.4912000E-05,.2601400E-06,&
     & .3707100E-08,.2982700E-05,.4657400E-05,.5348700E-05,.3041600E-06,&
     & .3477600E-08,.3260100E-05,.5063500E-05,.5807600E-05,.3430400E-06,&
     & .3274700E-08,.3551800E-05,.5470000E-05,.6219700E-05,.3875200E-06,&
     & .3464900E-08,.2009600E-05,.3168900E-05,.3707000E-05,.1842400E-06,&
     & .3222400E-08,.2250300E-05,.3527100E-05,.4078400E-05,.2170500E-06,&
     & .3011400E-08,.2483400E-05,.3873700E-05,.4443600E-05,.2522900E-06,&
     & .2826300E-08,.2714900E-05,.4208300E-05,.4806100E-05,.2846400E-06,&
     & .2662600E-08,.2955600E-05,.4539400E-05,.5156500E-05,.3214800E-06,&
     & .2797600E-08,.1691300E-05,.2663200E-05,.3110800E-05,.1553700E-06,&
     & .2604300E-08,.1890800E-05,.2955300E-05,.3406700E-05,.1831100E-06,&
     & .2435900E-08,.2081100E-05,.3239000E-05,.3711000E-05,.2109400E-06,&
     & .2287800E-08,.2274600E-05,.3514400E-05,.4000900E-05,.2381600E-06,&
     & .2156700E-08,.2471400E-05,.3783900E-05,.4291000E-05,.2688600E-06,&
     & .2258300E-08,.1423400E-05,.2238800E-05,.2608800E-05,.1314300E-06,&
     & .2104300E-08,.1586400E-05,.2477600E-05,.2849300E-05,.1546000E-06,&
     & .1969900E-08,.1743600E-05,.2707200E-05,.3100900E-05,.1766900E-06,&
     & .1851500E-08,.1905700E-05,.2935500E-05,.3330800E-05,.1995300E-06,&
     & .1746500E-08,.2065300E-05,.3154200E-05,.3566300E-05,.2250100E-06,&
     & .1822100E-08,.1200100E-05,.1880800E-05,.2186800E-05,.1113800E-06,&
     & .1699500E-08,.1331500E-05,.2077900E-05,.2382500E-05,.1306800E-06,&
     & .1592300E-08,.1462500E-05,.2264700E-05,.2588400E-05,.1482000E-06,&
     & .1497800E-08,.1597200E-05,.2451200E-05,.2777100E-05,.1673400E-06,&
     & .1413800E-08,.1725400E-05,.2629600E-05,.2964900E-05,.1884800E-06,&
     & .1469300E-08,.1011100E-05,.1580900E-05,.1828600E-05,.9436600E-07,&
     & .1371800E-08,.1119900E-05,.1743000E-05,.1995200E-05,.1104500E-06,&
     & .1286400E-08,.1227300E-05,.1895700E-05,.2157200E-05,.1244500E-06,&
     & .1211000E-08,.1337300E-05,.2046200E-05,.2316200E-05,.1404200E-06,&
     & .1143800E-08,.1441000E-05,.2191600E-05,.2464800E-05,.1576700E-06,&
     & .1183400E-08,.8533100E-06,.1331600E-05,.1532900E-05,.8021200E-07,&
     & .1106100E-08,.9419200E-06,.1462600E-05,.1672900E-05,.9277500E-07,&
     & .1038200E-08,.1031700E-05,.1589000E-05,.1800600E-05,.1046200E-06,&
     & .9781000E-09,.1120500E-05,.1709900E-05,.1931400E-05,.1180000E-06,&
     & .9245700E-09,.1203400E-05,.1827600E-05,.2050400E-05,.1320900E-06/

      data   absb(351:525, 2)   /                                       &
     & .9531100E-09,.7189000E-06,.1121800E-05,.1286700E-05,.6808400E-07,&
     & .8917700E-09,.7925800E-06,.1227500E-05,.1401900E-05,.7799900E-07,&
     & .8378100E-09,.8660500E-06,.1330800E-05,.1504000E-05,.8802700E-07,&
     & .7899800E-09,.9381500E-06,.1428500E-05,.1609100E-05,.9905700E-07,&
     & .7472900E-09,.1004600E-05,.1523600E-05,.1705900E-05,.1107400E-06,&
     & .7677100E-09,.6062300E-06,.9442700E-06,.1079600E-05,.5784400E-07,&
     & .7190400E-09,.6665300E-06,.1029800E-05,.1169900E-05,.6572600E-07,&
     & .6761400E-09,.7268900E-06,.1113100E-05,.1256800E-05,.7400900E-07,&
     & .6380600E-09,.7848200E-06,.1192700E-05,.1340600E-05,.8320000E-07,&
     & .6040100E-09,.8380600E-06,.1269500E-05,.1418600E-05,.9230600E-07,&
     & .6181700E-09,.5105300E-06,.7945000E-06,.9068700E-06,.4907800E-07,&
     & .5795800E-09,.5607100E-06,.8640300E-06,.9769000E-06,.5525700E-07,&
     & .5455100E-09,.6100900E-06,.9307700E-06,.1049500E-05,.6228600E-07,&
     & .5152000E-09,.6563100E-06,.9956700E-06,.1115900E-05,.6995400E-07,&
     & .4880600E-09,.6983000E-06,.1057600E-05,.1179000E-05,.7722800E-07,&
     & .4978300E-09,.4301400E-06,.6673200E-06,.7605800E-06,.4130600E-07,&
     & .4672300E-09,.4714800E-06,.7243400E-06,.8168200E-06,.4653700E-07,&
     & .4401500E-09,.5114600E-06,.7780900E-06,.8753700E-06,.5245100E-07,&
     & .4160300E-09,.5487700E-06,.8307300E-06,.9288400E-06,.5879300E-07,&
     & .3944000E-09,.5815900E-06,.8805400E-06,.9803000E-06,.6463000E-07,&
     & .4008700E-09,.3625600E-06,.5606200E-06,.6354900E-06,.3477800E-07,&
     & .3766000E-09,.3962000E-06,.6067400E-06,.6831300E-06,.3921300E-07,&
     & .3551000E-09,.4285500E-06,.6503700E-06,.7299900E-06,.4417600E-07,&
     & .3359000E-09,.4585200E-06,.6930800E-06,.7733800E-06,.4942400E-07,&
     & .3186600E-09,.4845900E-06,.7329700E-06,.8148400E-06,.5413700E-07,&
     & .3228000E-09,.3054700E-06,.4708300E-06,.5314700E-06,.2935300E-07,&
     & .3035600E-09,.3327100E-06,.5077000E-06,.5711200E-06,.3307800E-07,&
     & .2864700E-09,.3587600E-06,.5433900E-06,.6085200E-06,.3721100E-07,&
     & .2712000E-09,.3821400E-06,.5778100E-06,.6435400E-06,.4129500E-07,&
     & .2574600E-09,.4037400E-06,.6099000E-06,.6771100E-06,.4536200E-07,&
     & .2599700E-09,.2572300E-06,.3950500E-06,.4446700E-06,.2475400E-07,&
     & .2447100E-09,.2792600E-06,.4246900E-06,.4770200E-06,.2790700E-07,&
     & .2311300E-09,.3002400E-06,.4538400E-06,.5067200E-06,.3134400E-07,&
     & .2189800E-09,.3186200E-06,.4814200E-06,.5351500E-06,.3459000E-07,&
     & .2080300E-09,.3362400E-06,.5072100E-06,.5624200E-06,.3795700E-07/

      data   absb(526:700, 2)   /                                       &
     & .2098700E-09,.2156100E-06,.3303500E-06,.3709300E-06,.2080100E-07,&
     & .1977100E-09,.2335600E-06,.3543300E-06,.3972300E-06,.2344300E-07,&
     & .1868800E-09,.2504200E-06,.3780400E-06,.4212700E-06,.2627100E-07,&
     & .1771600E-09,.2651400E-06,.4002900E-06,.4443300E-06,.2886700E-07,&
     & .1684000E-09,.2794900E-06,.4210200E-06,.4661200E-06,.3162000E-07,&
     & .1704600E-09,.1790900E-06,.2739700E-06,.3073000E-06,.1728300E-07,&
     & .1606600E-09,.1937800E-06,.2936700E-06,.3286900E-06,.1945600E-07,&
     & .1519100E-09,.2071600E-06,.3128700E-06,.3482000E-06,.2175900E-07,&
     & .1440700E-09,.2193800E-06,.3309000E-06,.3670300E-06,.2385400E-07,&
     & .1369900E-09,.2311500E-06,.3477900E-06,.3845900E-06,.2612800E-07,&
     & .1393800E-09,.1474200E-06,.2253500E-06,.2526100E-06,.1414500E-07,&
     & .1313700E-09,.1594900E-06,.2415500E-06,.2701100E-06,.1591700E-07,&
     & .1242300E-09,.1703700E-06,.2572100E-06,.2860600E-06,.1780900E-07,&
     & .1178200E-09,.1804400E-06,.2720000E-06,.3014900E-06,.1953000E-07,&
     & .1120400E-09,.1901100E-06,.2858200E-06,.3158100E-06,.2139100E-07,&
     & .1150300E-09,.1199100E-06,.1833900E-06,.2054000E-06,.1139800E-07,&
     & .1083700E-09,.1298500E-06,.1967000E-06,.2199100E-06,.1282900E-07,&
     & .1024400E-09,.1389500E-06,.2096400E-06,.2331000E-06,.1437000E-07,&
     & .9712200E-10,.1472500E-06,.2218300E-06,.2458200E-06,.1579000E-07,&
     & .9232500E-10,.1552600E-06,.2332700E-06,.2576700E-06,.1730800E-07,&
     & .9499000E-10,.9743400E-07,.1491200E-06,.1669900E-06,.9174500E-08,&
     & .8944900E-10,.1056300E-06,.1600700E-06,.1789000E-06,.1033000E-07,&
     & .8451600E-10,.1132200E-06,.1707100E-06,.1898200E-06,.1158500E-07,&
     & .8009600E-10,.1200900E-06,.1808400E-06,.2003000E-06,.1275900E-07,&
     & .7611300E-10,.1267200E-06,.1902800E-06,.2101200E-06,.1399200E-07,&
     & .7844400E-10,.7914700E-07,.1212200E-06,.1357300E-06,.7384600E-08,&
     & .7383200E-10,.8590100E-07,.1302200E-06,.1455200E-06,.8318000E-08,&
     & .6972900E-10,.9223000E-07,.1389900E-06,.1545900E-06,.9337400E-08,&
     & .6605600E-10,.9791900E-07,.1473700E-06,.1632500E-06,.1030700E-07,&
     & .6274900E-10,.1034100E-06,.1551800E-06,.1713300E-06,.1131200E-07,&
     & .6520700E-10,.6371500E-07,.9775500E-07,.1094500E-06,.5883100E-08,&
     & .6131900E-10,.6928900E-07,.1052400E-06,.1175700E-06,.6623700E-08,&
     & .5786600E-10,.7469400E-07,.1124500E-06,.1251500E-06,.7445500E-08,&
     & .5477900E-10,.7935900E-07,.1194100E-06,.1322900E-06,.8259500E-08,&
     & .5200300E-10,.8395700E-07,.1259300E-06,.1390900E-06,.9069800E-08/

      data   absb(701:875, 2)   /                                       &
     & .5426300E-10,.5117200E-07,.7871400E-07,.8818300E-07,.4674000E-08,&
     & .5097800E-10,.5583800E-07,.8493900E-07,.9482800E-07,.5265500E-08,&
     & .4806600E-10,.6032800E-07,.9085900E-07,.1011800E-06,.5927300E-08,&
     & .4546700E-10,.6426000E-07,.9666700E-07,.1071000E-06,.6627900E-08,&
     & .4313400E-10,.6808900E-07,.1020800E-06,.1127900E-06,.7256000E-08,&
     & .4517000E-10,.4107300E-07,.6333300E-07,.7106700E-07,.3718500E-08,&
     & .4239300E-10,.4496400E-07,.6850700E-07,.7645500E-07,.4186400E-08,&
     & .3993600E-10,.4866700E-07,.7338200E-07,.8177200E-07,.4715700E-08,&
     & .3774700E-10,.5199600E-07,.7822000E-07,.8668900E-07,.5283400E-08,&
     & .3578400E-10,.5517800E-07,.8273900E-07,.9143400E-07,.5804000E-08,&
     & .3775400E-10,.3277000E-07,.5066200E-07,.5697900E-07,.2940500E-08,&
     & .3538700E-10,.3599700E-07,.5499300E-07,.6135700E-07,.3306600E-08,&
     & .3329900E-10,.3907300E-07,.5903600E-07,.6579300E-07,.3726500E-08,&
     & .3144200E-10,.4190300E-07,.6301400E-07,.6992000E-07,.4184800E-08,&
     & .2978100E-10,.4454600E-07,.6682200E-07,.7384900E-07,.4620400E-08,&
     & .3162300E-10,.2605400E-07,.4040500E-07,.4569200E-07,.2291800E-08,&
     & .2959900E-10,.2871300E-07,.4401000E-07,.4912200E-07,.2605300E-08,&
     & .2781700E-10,.3129600E-07,.4739100E-07,.5281500E-07,.2935300E-08,&
     & .2623600E-10,.3367700E-07,.5065500E-07,.5626400E-07,.3302300E-08,&
     & .2482500E-10,.3589600E-07,.5386300E-07,.5953800E-07,.3673000E-08,&
     & .2650200E-10,.2070500E-07,.3215200E-07,.3644800E-07,.1784100E-08,&
     & .2476800E-10,.2286800E-07,.3518500E-07,.3932900E-07,.2051500E-08,&
     & .2324700E-10,.2502700E-07,.3799900E-07,.4233800E-07,.2311800E-08,&
     & .2190000E-10,.2706300E-07,.4069400E-07,.4525800E-07,.2604800E-08,&
     & .2070100E-10,.2890100E-07,.4336100E-07,.4796900E-07,.2917400E-08,&
     & .2225800E-10,.1642200E-07,.2548000E-07,.2899100E-07,.1383000E-08,&
     & .2076700E-10,.1815800E-07,.2803300E-07,.3144200E-07,.1614400E-08,&
     & .1946200E-10,.1995300E-07,.3040200E-07,.3385000E-07,.1816200E-08,&
     & .1831200E-10,.2165800E-07,.3263900E-07,.3630800E-07,.2047500E-08,&
     & .1728900E-10,.2321100E-07,.3483900E-07,.3858300E-07,.2299400E-08,&
     & .1876400E-10,.1295100E-07,.2007200E-07,.2298200E-07,.1064500E-08,&
     & .1747300E-10,.1433500E-07,.2221400E-07,.2508500E-07,.1247900E-08,&
     & .1634700E-10,.1581300E-07,.2420500E-07,.2697100E-07,.1420700E-08,&
     & .1535700E-10,.1725500E-07,.2607500E-07,.2901500E-07,.1600800E-08,&
     & .1448000E-10,.1855800E-07,.2788500E-07,.3092200E-07,.1801800E-08/

      data   absb(876:1050, 2)   /                                      &
     & .1583300E-10,.1018600E-07,.1578200E-07,.1817400E-07,.8171800E-09,&
     & .1471300E-10,.1131200E-07,.1755500E-07,.1989100E-07,.9609800E-09,&
     & .1374000E-10,.1250900E-07,.1924800E-07,.2148200E-07,.1111200E-08,&
     & .1288700E-10,.1371900E-07,.2081100E-07,.2314000E-07,.1251500E-08,&
     & .1213400E-10,.1484000E-07,.2231300E-07,.2476900E-07,.1410500E-08,&
     & .1337200E-10,.8005000E-08,.1240300E-07,.1428300E-07,.6311700E-09,&
     & .1239800E-10,.8939300E-08,.1384000E-07,.1575100E-07,.7390700E-09,&
     & .1155600E-10,.9882100E-08,.1526100E-07,.1710400E-07,.8664500E-09,&
     & .1082100E-10,.1088400E-07,.1658200E-07,.1844100E-07,.9782900E-09,&
     & .1017400E-10,.1184100E-07,.1782800E-07,.1980400E-07,.1103400E-08,&
     & .1128200E-10,.6272600E-08,.9747800E-08,.1122600E-07,.4839800E-09,&
     & .1043700E-10,.7054600E-08,.1093200E-07,.1251800E-07,.5705100E-09,&
     & .9710100E-11,.7819300E-08,.1211400E-07,.1367100E-07,.6699700E-09,&
     & .9077200E-11,.8637900E-08,.1321800E-07,.1471200E-07,.7662500E-09,&
     & .8521600E-11,.9440500E-08,.1425800E-07,.1584400E-07,.8648800E-09,&
     & .9517900E-11,.4925100E-08,.7635600E-08,.8776100E-08,.3721300E-09,&
     & .8785600E-11,.5568100E-08,.8622000E-08,.9915900E-08,.4407400E-09,&
     & .8157600E-11,.6193000E-08,.9598900E-08,.1086500E-07,.5184300E-09,&
     & .7613100E-11,.6851800E-08,.1053900E-07,.1174600E-07,.6021000E-09,&
     & .7136600E-11,.7520600E-08,.1140300E-07,.1265900E-07,.6784100E-09,&
     & .8036800E-11,.3860800E-08,.6001000E-08,.6844100E-08,.2884800E-09,&
     & .7400700E-11,.4390300E-08,.6797400E-08,.7812700E-08,.3424500E-09,&
     & .6857700E-11,.4908600E-08,.7595900E-08,.8627900E-08,.4010600E-09,&
     & .6388600E-11,.5429700E-08,.8380100E-08,.9377800E-08,.4705800E-09,&
     & .5979500E-11,.5982900E-08,.9110300E-08,.1011100E-07,.5321800E-09,&
     & .6793200E-11,.3010200E-08,.4726900E-08,.5339500E-08,.2243200E-09,&
     & .6239700E-11,.3445000E-08,.5345800E-08,.6144900E-08,.2628200E-09,&
     & .5769300E-11,.3876800E-08,.6000600E-08,.6861300E-08,.3102000E-09,&
     & .5364700E-11,.4300200E-08,.6653300E-08,.7499000E-08,.3644600E-09,&
     & .5012900E-11,.4752700E-08,.7263800E-08,.8071700E-08,.4175900E-09,&
     & .5732000E-11,.2358200E-08,.3683700E-08,.4131700E-08,.1747700E-09,&
     & .5251800E-11,.2716300E-08,.4204800E-08,.4820800E-08,.2032100E-09,&
     & .4845700E-11,.3067800E-08,.4747200E-08,.5447200E-08,.2408100E-09,&
     & .4497800E-11,.3414800E-08,.5285100E-08,.5973500E-08,.2834500E-09,&
     & .4196300E-11,.3780400E-08,.5800100E-08,.6457000E-08,.3292300E-09/

      data   absb(1051:1175, 2)   /                                     &
     & .4834700E-11,.1853400E-08,.2853900E-08,.3182800E-08,.1349200E-09,&
     & .4418600E-11,.2139200E-08,.3319000E-08,.3781700E-08,.1584800E-09,&
     & .4068300E-11,.2432000E-08,.3759200E-08,.4315600E-08,.1883500E-09,&
     & .3769300E-11,.2715000E-08,.4199900E-08,.4762500E-08,.2207600E-09,&
     & .3511100E-11,.3008200E-08,.4634500E-08,.5173600E-08,.2591600E-09,&
     & .4081300E-11,.1454300E-08,.2203800E-08,.2448800E-08,.1032800E-09,&
     & .3720100E-11,.1681300E-08,.2637100E-08,.2964000E-08,.1244600E-09,&
     & .3417500E-11,.1919100E-08,.2974700E-08,.3411500E-08,.1460400E-09,&
     & .3160400E-11,.2157500E-08,.3334800E-08,.3801800E-08,.1720200E-09,&
     & .2939100E-11,.2393200E-08,.3693000E-08,.4147200E-08,.2023500E-09,&
     & .3448600E-11,.1148000E-08,.1703200E-08,.1877100E-08,.7878700E-10,&
     & .3134500E-11,.1318900E-08,.2061500E-08,.2312100E-08,.9683200E-10,&
     & .2872800E-11,.1514400E-08,.2344000E-08,.2686300E-08,.1131700E-09,&
     & .2651300E-11,.1710200E-08,.2643600E-08,.3024600E-08,.1339300E-09,&
     & .2461500E-11,.1902200E-08,.2939400E-08,.3313500E-08,.1577200E-09,&
     & .2911900E-11,.8950000E-09,.1302000E-08,.1438300E-08,.6025300E-10,&
     & .2639200E-11,.1036900E-08,.1603600E-08,.1789100E-08,.7576500E-10,&
     & .2413200E-11,.1197500E-08,.1852300E-08,.2111400E-08,.8826000E-10,&
     & .2222700E-11,.1357000E-08,.2095900E-08,.2404500E-08,.1047800E-09,&
     & .2060100E-11,.1513900E-08,.2340600E-08,.2645600E-08,.1233100E-09,&
     & .2414300E-11,.7185100E-09,.1036200E-08,.1151600E-08,.4808600E-10,&
     & .2185700E-11,.8373200E-09,.1289200E-08,.1433500E-08,.6063000E-10,&
     & .1996500E-11,.9681500E-09,.1499000E-08,.1703900E-08,.7120800E-10,&
     & .1837400E-11,.1098200E-08,.1697300E-08,.1945500E-08,.8451400E-10,&
     & .1701700E-11,.1227700E-08,.1898100E-08,.2147200E-08,.9928900E-10/

      data   absb(  1:175, 3)   /                                       &
     & .6949700E-07,.5590000E-04,.7763400E-04,.8330000E-04,.1441900E-04,&
     & .6492300E-07,.6177900E-04,.8523800E-04,.9045800E-04,.1599600E-04,&
     & .6092000E-07,.6753200E-04,.9281100E-04,.9758800E-04,.1762200E-04,&
     & .5737900E-07,.7287600E-04,.1004000E-03,.1045500E-03,.1914400E-04,&
     & .5422000E-07,.7769900E-04,.1078400E-03,.1113900E-03,.2067100E-04,&
     & .5653800E-07,.4662700E-04,.6467800E-04,.6939900E-04,.1208600E-04,&
     & .5283900E-07,.5153600E-04,.7108000E-04,.7529200E-04,.1341400E-04,&
     & .4960000E-07,.5633400E-04,.7738400E-04,.8127700E-04,.1473200E-04,&
     & .4673100E-07,.6075100E-04,.8375300E-04,.8706700E-04,.1597300E-04,&
     & .4417100E-07,.6479200E-04,.8996300E-04,.9261700E-04,.1721400E-04,&
     & .4596300E-07,.3892000E-04,.5392100E-04,.5782400E-04,.1008900E-04,&
     & .4297600E-07,.4298700E-04,.5928600E-04,.6261500E-04,.1123800E-04,&
     & .4035800E-07,.4701400E-04,.6455700E-04,.6765800E-04,.1229300E-04,&
     & .3803700E-07,.5064100E-04,.6987900E-04,.7248700E-04,.1331700E-04,&
     & .3596400E-07,.5400300E-04,.7506400E-04,.7708300E-04,.1432100E-04,&
     & .3735400E-07,.3247900E-04,.4495300E-04,.4814600E-04,.8407900E-05,&
     & .3494400E-07,.3587300E-04,.4945400E-04,.5212000E-04,.9340500E-05,&
     & .3282900E-07,.3922000E-04,.5386100E-04,.5629300E-04,.1020900E-04,&
     & .3095200E-07,.4215500E-04,.5830900E-04,.6026800E-04,.1108000E-04,&
     & .2927400E-07,.4501400E-04,.6261000E-04,.6418400E-04,.1188700E-04,&
     & .3036400E-07,.2708600E-04,.3745100E-04,.3999500E-04,.7004900E-05,&
     & .2842000E-07,.2991900E-04,.4119400E-04,.4336700E-04,.7773600E-05,&
     & .2671000E-07,.3265000E-04,.4490200E-04,.4681700E-04,.8483800E-05,&
     & .2519200E-07,.3509800E-04,.4860500E-04,.5010300E-04,.9203800E-05,&
     & .2383300E-07,.3750700E-04,.5218800E-04,.5341300E-04,.9860700E-05,&
     & .2468700E-07,.2258100E-04,.3121100E-04,.3322000E-04,.5836100E-05,&
     & .2311600E-07,.2494000E-04,.3430800E-04,.3607800E-04,.6464000E-05,&
     & .2173400E-07,.2718100E-04,.3741400E-04,.3893300E-04,.7034400E-05,&
     & .2050600E-07,.2921600E-04,.4051700E-04,.4165300E-04,.7627600E-05,&
     & .1940600E-07,.3123700E-04,.4349000E-04,.4439700E-04,.8175300E-05,&
     & .2007400E-07,.1881800E-04,.2599900E-04,.2759700E-04,.4866200E-05,&
     & .1880500E-07,.2078400E-04,.2856200E-04,.2999300E-04,.5372300E-05,&
     & .1768800E-07,.2262600E-04,.3115100E-04,.3236300E-04,.5837700E-05,&
     & .1669300E-07,.2430000E-04,.3376000E-04,.3462900E-04,.6317700E-05,&
     & .1580200E-07,.2598500E-04,.3622600E-04,.3690300E-04,.6774200E-05/

      data   absb(176:350, 3)   /                                       &
     & .1630600E-07,.1571700E-04,.2168600E-04,.2295900E-04,.4068000E-05,&
     & .1528300E-07,.1735500E-04,.2382600E-04,.2496900E-04,.4470600E-05,&
     & .1438100E-07,.1885600E-04,.2599200E-04,.2691200E-04,.4855000E-05,&
     & .1357800E-07,.2026500E-04,.2816200E-04,.2881700E-04,.5235500E-05,&
     & .1285700E-07,.2163800E-04,.3021500E-04,.3070700E-04,.5622400E-05,&
     & .1324500E-07,.1312800E-04,.1809100E-04,.1910700E-04,.3388000E-05,&
     & .1242000E-07,.1449500E-04,.1988700E-04,.2079400E-04,.3714400E-05,&
     & .1169200E-07,.1570400E-04,.2170300E-04,.2239700E-04,.4041500E-05,&
     & .1104300E-07,.1690400E-04,.2348900E-04,.2398400E-04,.4345500E-05,&
     & .1046000E-07,.1803700E-04,.2519700E-04,.2555800E-04,.4666200E-05,&
     & .1071100E-07,.1104400E-04,.1519900E-04,.1600800E-04,.2840900E-05,&
     & .1005200E-07,.1216500E-04,.1669900E-04,.1741900E-04,.3104100E-05,&
     & .9469000E-08,.1315800E-04,.1822400E-04,.1873300E-04,.3376800E-05,&
     & .8948400E-08,.1416600E-04,.1969100E-04,.2005000E-04,.3623700E-05,&
     & .8480400E-08,.1508500E-04,.2110500E-04,.2135600E-04,.3887600E-05,&
     & .8660600E-08,.9300200E-05,.1277500E-04,.1342000E-04,.2381500E-05,&
     & .8134000E-08,.1021000E-04,.1403400E-04,.1455200E-04,.2595500E-05,&
     & .7667100E-08,.1103500E-04,.1530200E-04,.1565700E-04,.2815000E-05,&
     & .7249700E-08,.1186300E-04,.1651300E-04,.1677000E-04,.3019800E-05,&
     & .6873900E-08,.1261600E-04,.1766400E-04,.1784800E-04,.3238700E-05,&
     & .6999500E-08,.7839200E-05,.1075400E-04,.1125000E-04,.1998600E-05,&
     & .6579000E-08,.8567400E-05,.1180600E-04,.1218600E-04,.2173800E-05,&
     & .6205500E-08,.9266400E-05,.1285000E-04,.1310300E-04,.2346900E-05,&
     & .5871000E-08,.9937600E-05,.1385400E-04,.1402700E-04,.2519500E-05,&
     & .5569600E-08,.1052700E-04,.1479500E-04,.1491700E-04,.2698800E-05,&
     & .5653800E-08,.6611800E-05,.9061000E-05,.9442200E-05,.1675300E-05,&
     & .5318400E-08,.7198600E-05,.9938900E-05,.1020900E-04,.1822300E-05,&
     & .5019900E-08,.7786300E-05,.1080000E-04,.1098100E-04,.1959000E-05,&
     & .4752200E-08,.8341500E-05,.1162600E-04,.1173700E-04,.2102000E-05,&
     & .4510500E-08,.8801700E-05,.1237300E-04,.1246800E-04,.2249800E-05,&
     & .4562000E-08,.5574900E-05,.7646300E-05,.7928500E-05,.1403200E-05,&
     & .4295000E-08,.6059000E-05,.8382200E-05,.8565200E-05,.1525500E-05,&
     & .4056900E-08,.6551000E-05,.9088800E-05,.9211600E-05,.1637300E-05,&
     & .3842900E-08,.6991800E-05,.9760400E-05,.9832900E-05,.1756100E-05,&
     & .3649500E-08,.7356600E-05,.1036700E-04,.1042900E-04,.1876900E-05/

      data   absb(351:525, 3)   /                                       &
     & .3680900E-08,.4696700E-05,.6458000E-05,.6664900E-05,.1178600E-05,&
     & .3468300E-08,.5107200E-05,.7066100E-05,.7190200E-05,.1275900E-05,&
     & .3278400E-08,.5505300E-05,.7647000E-05,.7725000E-05,.1369100E-05,&
     & .3107400E-08,.5847400E-05,.8189800E-05,.8234000E-05,.1466600E-05,&
     & .2952600E-08,.6149300E-05,.8681500E-05,.8719900E-05,.1565000E-05,&
     & .2970100E-08,.3958200E-05,.5457300E-05,.5595900E-05,.9903700E-06,&
     & .2800800E-08,.4304800E-05,.5954600E-05,.6040600E-05,.1065900E-05,&
     & .2649300E-08,.4627700E-05,.6428600E-05,.6474500E-05,.1144200E-05,&
     & .2512700E-08,.4894000E-05,.6861200E-05,.6891800E-05,.1225000E-05,&
     & .2388800E-08,.5142800E-05,.7265400E-05,.7286500E-05,.1304600E-05,&
     & .2395800E-08,.3340900E-05,.4611500E-05,.4705400E-05,.8302100E-06,&
     & .2261100E-08,.3626700E-05,.5019200E-05,.5075400E-05,.8915700E-06,&
     & .2140200E-08,.3880500E-05,.5405300E-05,.5429100E-05,.9566300E-06,&
     & .2031100E-08,.4092900E-05,.5752900E-05,.5768500E-05,.1023200E-05,&
     & .1932000E-08,.4301400E-05,.6068400E-05,.6086800E-05,.1087400E-05,&
     & .1932800E-08,.2821700E-05,.3895500E-05,.3956200E-05,.6946400E-06,&
     & .1825500E-08,.3051500E-05,.4229300E-05,.4260500E-05,.7458800E-06,&
     & .1729100E-08,.3249600E-05,.4539000E-05,.4550100E-05,.7995700E-06,&
     & .1641900E-08,.3423500E-05,.4819100E-05,.4824900E-05,.8537300E-06,&
     & .1562600E-08,.3599200E-05,.5063200E-05,.5080500E-05,.9058200E-06,&
     & .1559000E-08,.2384100E-05,.3289000E-05,.3328200E-05,.5810800E-06,&
     & .1473600E-08,.2568300E-05,.3560500E-05,.3575600E-05,.6240100E-06,&
     & .1396700E-08,.2722300E-05,.3805500E-05,.3811000E-05,.6686200E-06,&
     & .1327100E-08,.2865200E-05,.4029500E-05,.4033100E-05,.7121600E-06,&
     & .1263700E-08,.3006900E-05,.4225700E-05,.4239300E-05,.7552700E-06,&
     & .1257500E-08,.2009900E-05,.2776400E-05,.2798800E-05,.4861400E-06,&
     & .1189500E-08,.2155100E-05,.2996100E-05,.2999600E-05,.5219900E-06,&
     & .1128200E-08,.2277100E-05,.3190100E-05,.3189700E-05,.5586700E-06,&
     & .1072600E-08,.2396900E-05,.3365700E-05,.3369600E-05,.5941000E-06,&
     & .1021800E-08,.2507600E-05,.3523200E-05,.3536600E-05,.6296500E-06,&
     & .1014400E-08,.1694000E-05,.2342100E-05,.2351900E-05,.4070100E-06,&
     & .9602400E-09,.1806600E-05,.2514600E-05,.2514800E-05,.4365700E-06,&
     & .9113400E-09,.1905300E-05,.2673800E-05,.2669000E-05,.4665600E-06,&
     & .8668900E-09,.2003800E-05,.2810400E-05,.2813900E-05,.4951600E-06,&
     & .8263000E-09,.2086000E-05,.2929400E-05,.2948500E-05,.5246200E-06/

      data   absb(526:700, 3)   /                                       &
     & .8200000E-09,.1423100E-05,.1966100E-05,.1970000E-05,.3399600E-06,&
     & .7767200E-09,.1510500E-05,.2104700E-05,.2101800E-05,.3643500E-06,&
     & .7375700E-09,.1590800E-05,.2230700E-05,.2227200E-05,.3885100E-06,&
     & .7019300E-09,.1670600E-05,.2342100E-05,.2344800E-05,.4121900E-06,&
     & .6693600E-09,.1733000E-05,.2434200E-05,.2453800E-05,.4360700E-06,&
     & .6665200E-09,.1184200E-05,.1638300E-05,.1638200E-05,.2818300E-06,&
     & .6315600E-09,.1254100E-05,.1749900E-05,.1745700E-05,.3019500E-06,&
     & .5999000E-09,.1320900E-05,.1850400E-05,.1848400E-05,.3217400E-06,&
     & .5710700E-09,.1385000E-05,.1941800E-05,.1943800E-05,.3412400E-06,&
     & .5447000E-09,.1434800E-05,.2013500E-05,.2033500E-05,.3605000E-06,&
     & .5450600E-09,.9787000E-06,.1354100E-05,.1352000E-05,.2319000E-06,&
     & .5165000E-09,.1035400E-05,.1445500E-05,.1440400E-05,.2484500E-06,&
     & .4906300E-09,.1091000E-05,.1527000E-05,.1524700E-05,.2646400E-06,&
     & .4670700E-09,.1142700E-05,.1601200E-05,.1603300E-05,.2807200E-06,&
     & .4455300E-09,.1183300E-05,.1660400E-05,.1677400E-05,.2964300E-06,&
     & .4494900E-09,.8010400E-06,.1106600E-05,.1104500E-05,.1888800E-06,&
     & .4257900E-09,.8481000E-06,.1182600E-05,.1177700E-05,.2024200E-06,&
     & .4043500E-09,.8940400E-06,.1250400E-05,.1247700E-05,.2157800E-06,&
     & .3848300E-09,.9373700E-06,.1312400E-05,.1312900E-05,.2289200E-06,&
     & .3669900E-09,.9711800E-06,.1361700E-05,.1374400E-05,.2419000E-06,&
     & .3708800E-09,.6546800E-06,.9034700E-06,.9014000E-06,.1537100E-06,&
     & .3512000E-09,.6942100E-06,.9667600E-06,.9622200E-06,.1648400E-06,&
     & .3334000E-09,.7321100E-06,.1023300E-05,.1020300E-05,.1757800E-06,&
     & .3172100E-09,.7683800E-06,.1075000E-05,.1074500E-05,.1865600E-06,&
     & .3024200E-09,.7967600E-06,.1116400E-05,.1125600E-05,.1973000E-06,&
     & .3060300E-09,.5351800E-06,.7375500E-06,.7356000E-06,.1250900E-06,&
     & .2896700E-09,.5682700E-06,.7901700E-06,.7860700E-06,.1342000E-06,&
     & .2749000E-09,.5994600E-06,.8372900E-06,.8341800E-06,.1432000E-06,&
     & .2614800E-09,.6296800E-06,.8804500E-06,.8791900E-06,.1520600E-06,&
     & .2492200E-09,.6535800E-06,.9151100E-06,.9218700E-06,.1609200E-06,&
     & .2540100E-09,.4349100E-06,.5970600E-06,.5958300E-06,.1010300E-06,&
     & .2402700E-09,.4624300E-06,.6412700E-06,.6378000E-06,.1085400E-06,&
     & .2278800E-09,.4878900E-06,.6811100E-06,.6778500E-06,.1159600E-06,&
     & .2166400E-09,.5137000E-06,.7177600E-06,.7155700E-06,.1231800E-06,&
     & .2063900E-09,.5341300E-06,.7473800E-06,.7511600E-06,.1305500E-06/

      data   absb(701:875, 3)   /                                       &
     & .2110300E-09,.3516200E-06,.4824500E-06,.4818100E-06,.8155100E-07,&
     & .1994700E-09,.3757700E-06,.5197500E-06,.5168300E-06,.8762400E-07,&
     & .1890600E-09,.3968100E-06,.5537200E-06,.5501600E-06,.9377500E-07,&
     & .1796300E-09,.4185600E-06,.5840500E-06,.5817700E-06,.9967700E-07,&
     & .1710500E-09,.4361000E-06,.6095700E-06,.6115000E-06,.1057800E-06,&
     & .1753600E-09,.2841200E-06,.3894800E-06,.3893300E-06,.6579100E-07,&
     & .1656300E-09,.3050400E-06,.4209700E-06,.4185400E-06,.7075200E-07,&
     & .1568800E-09,.3226500E-06,.4495700E-06,.4462400E-06,.7579000E-07,&
     & .1489700E-09,.3408000E-06,.4750200E-06,.4726800E-06,.8064800E-07,&
     & .1417800E-09,.3558400E-06,.4969600E-06,.4974700E-06,.8568500E-07,&
     & .1462500E-09,.2282800E-06,.3126900E-06,.3130100E-06,.5285800E-07,&
     & .1380000E-09,.2466000E-06,.3394500E-06,.3373600E-06,.5687200E-07,&
     & .1306000E-09,.2615000E-06,.3638200E-06,.3605200E-06,.6100500E-07,&
     & .1239200E-09,.2763700E-06,.3849600E-06,.3826200E-06,.6506400E-07,&
     & .1178600E-09,.2896000E-06,.4043200E-06,.4034600E-06,.6913400E-07,&
     & .1222000E-09,.1829900E-06,.2500300E-06,.2507500E-06,.4242600E-07,&
     & .1151800E-09,.1985800E-06,.2727100E-06,.2712400E-06,.4560500E-07,&
     & .1089000E-09,.2114900E-06,.2930600E-06,.2905900E-06,.4898900E-07,&
     & .1032400E-09,.2236100E-06,.3112800E-06,.3090900E-06,.5235700E-07,&
     & .9811900E-10,.2354000E-06,.3278200E-06,.3265700E-06,.5565000E-07,&
     & .1021500E-09,.1460400E-06,.1995700E-06,.2007900E-06,.3400900E-07,&
     & .9616800E-10,.1597600E-06,.2185900E-06,.2178200E-06,.3656500E-07,&
     & .9083200E-10,.1709200E-06,.2359400E-06,.2340300E-06,.3933600E-07,&
     & .8603900E-10,.1807600E-06,.2514800E-06,.2494000E-06,.4212000E-07,&
     & .8170500E-10,.1909100E-06,.2657900E-06,.2641700E-06,.4480600E-07,&
     & .8554800E-10,.1160600E-06,.1585800E-06,.1602500E-06,.2713800E-07,&
     & .8043000E-10,.1277700E-06,.1746500E-06,.1743900E-06,.2926000E-07,&
     & .7588200E-10,.1377100E-06,.1895600E-06,.1880100E-06,.3151900E-07,&
     & .7180700E-10,.1459700E-06,.2028200E-06,.2008500E-06,.3379000E-07,&
     & .6813100E-10,.1545400E-06,.2148500E-06,.2132100E-06,.3601600E-07,&
     & .7188000E-10,.9167700E-07,.1251800E-06,.1271700E-06,.2154800E-07,&
     & .6747400E-10,.1018200E-06,.1388500E-06,.1388700E-06,.2336600E-07,&
     & .6357300E-10,.1103700E-06,.1515500E-06,.1503700E-06,.2514000E-07,&
     & .6008900E-10,.1175700E-06,.1629200E-06,.1611500E-06,.2702300E-07,&
     & .5695500E-10,.1246000E-06,.1730300E-06,.1715000E-06,.2888600E-07/

      data   absb(876:1050, 3)   /                                      &
     & .6043800E-10,.7237800E-07,.9858600E-07,.1007500E-06,.1706800E-07,&
     & .5663900E-10,.8066200E-07,.1100300E-06,.1105200E-06,.1864300E-07,&
     & .5328800E-10,.8839100E-07,.1206900E-06,.1200700E-06,.2005000E-07,&
     & .5030600E-10,.9462400E-07,.1304500E-06,.1291600E-06,.2158700E-07,&
     & .4763200E-10,.1002600E-06,.1391400E-06,.1377800E-06,.2313600E-07,&
     & .5085600E-10,.5704600E-07,.7756500E-07,.7974100E-07,.1351000E-07,&
     & .4757400E-10,.6373000E-07,.8693900E-07,.8782600E-07,.1481100E-07,&
     & .4469200E-10,.7036700E-07,.9597400E-07,.9570700E-07,.1599200E-07,&
     & .4213600E-10,.7595900E-07,.1044100E-06,.1033600E-06,.1723500E-07,&
     & .3985100E-10,.8066200E-07,.1117900E-06,.1106100E-06,.1850100E-07,&
     & .4275200E-10,.4499400E-07,.6093300E-07,.6317000E-07,.1073300E-07,&
     & .3992000E-10,.5041700E-07,.6869500E-07,.6977600E-07,.1177800E-07,&
     & .3744500E-10,.5612900E-07,.7638100E-07,.7629200E-07,.1279600E-07,&
     & .3525800E-10,.6091600E-07,.8353300E-07,.8274600E-07,.1377000E-07,&
     & .3330900E-10,.6501000E-07,.8993300E-07,.8882800E-07,.1480900E-07,&
     & .3593500E-10,.3527800E-07,.4790100E-07,.5005600E-07,.8523900E-08,&
     & .3349400E-10,.3989700E-07,.5424500E-07,.5543900E-07,.9356200E-08,&
     & .3136800E-10,.4457700E-07,.6068700E-07,.6085400E-07,.1022900E-07,&
     & .2949700E-10,.4890600E-07,.6666800E-07,.6621700E-07,.1100900E-07,&
     & .2783400E-10,.5234800E-07,.7215400E-07,.7132200E-07,.1185600E-07,&
     & .3022700E-10,.2757200E-07,.3752400E-07,.3975400E-07,.6719800E-08,&
     & .2811800E-10,.3156000E-07,.4281000E-07,.4403100E-07,.7426700E-08,&
     & .2629000E-10,.3531900E-07,.4807900E-07,.4848600E-07,.8151100E-08,&
     & .2468700E-10,.3904100E-07,.5314100E-07,.5289800E-07,.8804400E-08,&
     & .2326700E-10,.4212300E-07,.5783500E-07,.5717700E-07,.9493400E-08,&
     & .2544800E-10,.2145600E-07,.2932400E-07,.3145000E-07,.5303500E-08,&
     & .2362300E-10,.2492200E-07,.3370600E-07,.3490300E-07,.5902500E-08,&
     & .2204800E-10,.2793100E-07,.3800900E-07,.3854800E-07,.6484200E-08,&
     & .2067300E-10,.3111700E-07,.4229300E-07,.4218000E-07,.7045400E-08,&
     & .1945900E-10,.3380000E-07,.4631600E-07,.4577600E-07,.7588600E-08,&
     & .2139000E-10,.1674300E-07,.2303100E-07,.2487700E-07,.4177700E-08,&
     & .1981500E-10,.1961900E-07,.2655600E-07,.2772200E-07,.4703100E-08,&
     & .1846300E-10,.2216700E-07,.3010100E-07,.3070800E-07,.5164600E-08,&
     & .1728600E-10,.2480400E-07,.3370400E-07,.3371600E-07,.5644800E-08,&
     & .1625100E-10,.2718300E-07,.3704900E-07,.3669600E-07,.6078800E-08/

      data   absb(1051:1175, 3)   /                                     &
     & .1797300E-10,.1303700E-07,.1811500E-07,.1972700E-07,.3291200E-08,&
     & .1661500E-10,.1540200E-07,.2094200E-07,.2204300E-07,.3730100E-08,&
     & .1545400E-10,.1759200E-07,.2384800E-07,.2448500E-07,.4112900E-08,&
     & .1444800E-10,.1973300E-07,.2681500E-07,.2695400E-07,.4514600E-08,&
     & .1356500E-10,.2178500E-07,.2963300E-07,.2941400E-07,.4876200E-08,&
     & .1511200E-10,.1012700E-07,.1426200E-07,.1563700E-07,.2598600E-08,&
     & .1393900E-10,.1208200E-07,.1645800E-07,.1756700E-07,.2951000E-08,&
     & .1294200E-10,.1398000E-07,.1887600E-07,.1949900E-07,.3280300E-08,&
     & .1208000E-10,.1567700E-07,.2129800E-07,.2151600E-07,.3607800E-08,&
     & .1132800E-10,.1742100E-07,.2367400E-07,.2355200E-07,.3914200E-08,&
     & .1271600E-10,.7842300E-08,.1121800E-07,.1236800E-07,.2057200E-08,&
     & .1170300E-10,.9446800E-08,.1294400E-07,.1393900E-07,.2344900E-08,&
     & .1084400E-10,.1105400E-07,.1493300E-07,.1550400E-07,.2616100E-08,&
     & .1010500E-10,.1244500E-07,.1689500E-07,.1716700E-07,.2878900E-08,&
     & .9462900E-11,.1392900E-07,.1888400E-07,.1883700E-07,.3139100E-08,&
     & .1069400E-10,.6108400E-08,.8827800E-08,.9780100E-08,.1619200E-08,&
     & .9818000E-11,.7385900E-08,.1020600E-07,.1106400E-07,.1848600E-08,&
     & .9079700E-11,.8711300E-08,.1180100E-07,.1234800E-07,.2087200E-08,&
     & .8447800E-11,.9900300E-08,.1340900E-07,.1370700E-07,.2296700E-08,&
     & .7899800E-11,.1110800E-07,.1506900E-07,.1508200E-07,.2518200E-08,&
     & .8851300E-11,.4923900E-08,.7138600E-08,.7917000E-08,.1308900E-08,&
     & .8118700E-11,.5954100E-08,.8253900E-08,.8975800E-08,.1495200E-08,&
     & .7502100E-11,.7045000E-08,.9556700E-08,.1002400E-07,.1691900E-08,&
     & .6975300E-11,.8039100E-08,.1087400E-07,.1114300E-07,.1864600E-08,&
     & .6519100E-11,.9026500E-08,.1224000E-07,.1226800E-07,.2047100E-08/

      data   absb(  1:175, 4)   /                                       &
     & .4011500E-06,.2054200E-03,.2900600E-03,.2998300E-03,.8640400E-04,&
     & .3850300E-06,.2208600E-03,.3083300E-03,.3251800E-03,.9261100E-04,&
     & .3704200E-06,.2341500E-03,.3267400E-03,.3481400E-03,.9887800E-04,&
     & .3569900E-06,.2460700E-03,.3433500E-03,.3685800E-03,.1055900E-03,&
     & .3443900E-06,.2574100E-03,.3594300E-03,.3871600E-03,.1126200E-03,&
     & .3271700E-06,.1708600E-03,.2412100E-03,.2500100E-03,.7212000E-04,&
     & .3141100E-06,.1835800E-03,.2559900E-03,.2708900E-03,.7735100E-04,&
     & .3022500E-06,.1945600E-03,.2714900E-03,.2900200E-03,.8267600E-04,&
     & .2913200E-06,.2046000E-03,.2854400E-03,.3067800E-03,.8833100E-04,&
     & .2810500E-06,.2141000E-03,.2991900E-03,.3227100E-03,.9445100E-04,&
     & .2667300E-06,.1422100E-03,.2004800E-03,.2084300E-03,.6021300E-04,&
     & .2561500E-06,.1526800E-03,.2126400E-03,.2253100E-03,.6453100E-04,&
     & .2465200E-06,.1617000E-03,.2256700E-03,.2415700E-03,.6910200E-04,&
     & .2376400E-06,.1702300E-03,.2375000E-03,.2554300E-03,.7389100E-04,&
     & .2292600E-06,.1781500E-03,.2491200E-03,.2689200E-03,.7916700E-04,&
     & .2174000E-06,.1182400E-03,.1664400E-03,.1738100E-03,.5022700E-04,&
     & .2088400E-06,.1269600E-03,.1768300E-03,.1878000E-03,.5390300E-04,&
     & .2010400E-06,.1343400E-03,.1876400E-03,.2009400E-03,.5777700E-04,&
     & .1938100E-06,.1417700E-03,.1975700E-03,.2125900E-03,.6182100E-04,&
     & .1869800E-06,.1482800E-03,.2074700E-03,.2241600E-03,.6632600E-04,&
     & .1772400E-06,.9828700E-04,.1381300E-03,.1449600E-03,.4185100E-04,&
     & .1702900E-06,.1054800E-03,.1470100E-03,.1563600E-03,.4494300E-04,&
     & .1639600E-06,.1116900E-03,.1560500E-03,.1672700E-03,.4817600E-04,&
     & .1580900E-06,.1179500E-03,.1643500E-03,.1768400E-03,.5171600E-04,&
     & .1525100E-06,.1233200E-03,.1726500E-03,.1868300E-03,.5548000E-04,&
     & .1445000E-06,.8170300E-04,.1146400E-03,.1207900E-03,.3484900E-04,&
     & .1388700E-06,.8758000E-04,.1222400E-03,.1302800E-03,.3746500E-04,&
     & .1337300E-06,.9286900E-04,.1296600E-03,.1390600E-03,.4022500E-04,&
     & .1289600E-06,.9803400E-04,.1366800E-03,.1471200E-03,.4322700E-04,&
     & .1243900E-06,.1025000E-03,.1436700E-03,.1555700E-03,.4638300E-04,&
     & .1178200E-06,.6792200E-04,.9512100E-04,.1005900E-03,.2901400E-04,&
     & .1132500E-06,.7275900E-04,.1016600E-03,.1084300E-03,.3123900E-04,&
     & .1090900E-06,.7721300E-04,.1076900E-03,.1156300E-03,.3359400E-04,&
     & .1052100E-06,.8150700E-04,.1136700E-03,.1224400E-03,.3608700E-04,&
     & .1014600E-06,.8522900E-04,.1195100E-03,.1295500E-03,.3876000E-04/

      data   absb(176:350, 4)   /                                       &
     & .9600500E-07,.5653900E-04,.7904800E-04,.8363800E-04,.2418200E-04,&
     & .9231100E-07,.6052200E-04,.8450600E-04,.9030800E-04,.2605200E-04,&
     & .8892700E-07,.6430500E-04,.8960800E-04,.9622900E-04,.2807100E-04,&
     & .8577500E-07,.6781100E-04,.9465900E-04,.1020300E-03,.3018500E-04,&
     & .8270900E-07,.7096700E-04,.9955100E-04,.1079600E-03,.3243800E-04,&
     & .7822500E-07,.4709400E-04,.6573600E-04,.6970600E-04,.2016800E-04,&
     & .7523500E-07,.5038300E-04,.7030900E-04,.7521000E-04,.2174800E-04,&
     & .7249100E-07,.5358100E-04,.7457900E-04,.8010600E-04,.2344200E-04,&
     & .6992900E-07,.5644800E-04,.7886400E-04,.8510800E-04,.2524800E-04,&
     & .6741900E-07,.5909000E-04,.8293200E-04,.8987000E-04,.2709800E-04,&
     & .6357200E-07,.3942600E-04,.5497900E-04,.5844300E-04,.1691300E-04,&
     & .6116600E-07,.4209700E-04,.5877900E-04,.6293300E-04,.1825900E-04,&
     & .5895100E-07,.4479400E-04,.6233000E-04,.6695200E-04,.1967000E-04,&
     & .5687200E-07,.4714700E-04,.6596100E-04,.7123000E-04,.2122100E-04,&
     & .5482200E-07,.4937700E-04,.6923400E-04,.7503800E-04,.2274300E-04,&
     & .5165600E-07,.3295000E-04,.4605200E-04,.4908400E-04,.1418100E-04,&
     & .4972100E-07,.3525000E-04,.4910600E-04,.5267500E-04,.1533000E-04,&
     & .4793000E-07,.3743800E-04,.5215200E-04,.5609700E-04,.1653200E-04,&
     & .4624200E-07,.3941800E-04,.5516000E-04,.5967400E-04,.1781100E-04,&
     & .4457000E-07,.4128000E-04,.5789900E-04,.6265400E-04,.1910000E-04,&
     & .4196200E-07,.2760800E-04,.3852700E-04,.4116700E-04,.1190800E-04,&
     & .4040500E-07,.2954700E-04,.4109900E-04,.4408100E-04,.1288000E-04,&
     & .3895900E-07,.3132100E-04,.4368700E-04,.4705100E-04,.1391200E-04,&
     & .3758300E-07,.3299400E-04,.4619000E-04,.4989000E-04,.1496100E-04,&
     & .3622700E-07,.3456000E-04,.4847300E-04,.5235800E-04,.1603900E-04,&
     & .3407500E-07,.2315300E-04,.3230500E-04,.3456600E-04,.1001400E-04,&
     & .3282200E-07,.2476900E-04,.3443500E-04,.3696800E-04,.1082200E-04,&
     & .3165500E-07,.2624200E-04,.3663500E-04,.3948400E-04,.1169900E-04,&
     & .3052800E-07,.2762400E-04,.3864800E-04,.4177000E-04,.1258800E-04,&
     & .2943400E-07,.2895100E-04,.4063200E-04,.4377400E-04,.1347900E-04,&
     & .2765200E-07,.1946000E-04,.2711400E-04,.2905200E-04,.8437200E-05,&
     & .2664500E-07,.2078500E-04,.2893300E-04,.3109100E-04,.9130500E-05,&
     & .2570300E-07,.2201800E-04,.3074900E-04,.3317300E-04,.9856000E-05,&
     & .2478000E-07,.2316600E-04,.3240500E-04,.3497100E-04,.1059300E-04,&
     & .2390000E-07,.2428600E-04,.3404800E-04,.3666300E-04,.1134600E-04/

      data   absb(351:525, 4)   /                                       &
     & .2243900E-07,.1637000E-04,.2276300E-04,.2439700E-04,.7108400E-05,&
     & .2162800E-07,.1746100E-04,.2431800E-04,.2614000E-04,.7696100E-05,&
     & .2086500E-07,.1849300E-04,.2582600E-04,.2782400E-04,.8308000E-05,&
     & .2011300E-07,.1945200E-04,.2721500E-04,.2928700E-04,.8920700E-05,&
     & .1940500E-07,.2037600E-04,.2854300E-04,.3071000E-04,.9547700E-05,&
     & .1820800E-07,.1377700E-04,.1913800E-04,.2052400E-04,.5995200E-05,&
     & .1755500E-07,.1467900E-04,.2044800E-04,.2199600E-04,.6490700E-05,&
     & .1693600E-07,.1552200E-04,.2166100E-04,.2335700E-04,.6996600E-05,&
     & .1632500E-07,.1633400E-04,.2284600E-04,.2453700E-04,.7512400E-05,&
     & .1575500E-07,.1708800E-04,.2393100E-04,.2572200E-04,.8030500E-05,&
     & .1477200E-07,.1159800E-04,.1611700E-04,.1729400E-04,.5060200E-05,&
     & .1424600E-07,.1234600E-04,.1719800E-04,.1852100E-04,.5479900E-05,&
     & .1373900E-07,.1304700E-04,.1820500E-04,.1956600E-04,.5899200E-05,&
     & .1324700E-07,.1372900E-04,.1917300E-04,.2057400E-04,.6325100E-05,&
     & .1278900E-07,.1434100E-04,.2006800E-04,.2153600E-04,.6749600E-05,&
     & .1198400E-07,.9767900E-05,.1356800E-04,.1456500E-04,.4272100E-05,&
     & .1156000E-07,.1038900E-04,.1446100E-04,.1556000E-04,.4618200E-05,&
     & .1114400E-07,.1097300E-04,.1530200E-04,.1641000E-04,.4971500E-05,&
     & .1075000E-07,.1153200E-04,.1609100E-04,.1724400E-04,.5321200E-05,&
     & .1038100E-07,.1203400E-04,.1682100E-04,.1802300E-04,.5671200E-05,&
     & .9720500E-08,.8229600E-05,.1144000E-04,.1228800E-04,.3607900E-05,&
     & .9377700E-08,.8743100E-05,.1215800E-04,.1306700E-04,.3897400E-05,&
     & .9039100E-08,.9230000E-05,.1286200E-04,.1376900E-04,.4188500E-05,&
     & .8721900E-08,.9685400E-05,.1350800E-04,.1445400E-04,.4475600E-05,&
     & .8425100E-08,.1010100E-04,.1409000E-04,.1506400E-04,.4764300E-05,&
     & .7884200E-08,.6937900E-05,.9637800E-05,.1035000E-04,.3046200E-05,&
     & .7605800E-08,.7360500E-05,.1023000E-04,.1096400E-04,.3287900E-05,&
     & .7331200E-08,.7768500E-05,.1080600E-04,.1155200E-04,.3525800E-05,&
     & .7076100E-08,.8139300E-05,.1132900E-04,.1210900E-04,.3764200E-05,&
     & .6837500E-08,.8472300E-05,.1180600E-04,.1258200E-04,.4002700E-05,&
     & .6394600E-08,.5848400E-05,.8113200E-05,.8713800E-05,.2572200E-05,&
     & .6166800E-08,.6198500E-05,.8615200E-05,.9204300E-05,.2771600E-05,&
     & .5946000E-08,.6532900E-05,.9076100E-05,.9691400E-05,.2967200E-05,&
     & .5741000E-08,.6834900E-05,.9499100E-05,.1012800E-04,.3164900E-05,&
     & .5549000E-08,.7109800E-05,.9897200E-05,.1049400E-04,.3361300E-05/

      data   absb(526:700, 4)   /                                       &
     & .5193100E-08,.4915500E-05,.6812200E-05,.7300300E-05,.2164800E-05,&
     & .5006600E-08,.5210800E-05,.7229900E-05,.7712800E-05,.2328000E-05,&
     & .4828900E-08,.5483800E-05,.7607600E-05,.8107700E-05,.2490600E-05,&
     & .4663600E-08,.5727100E-05,.7946000E-05,.8449400E-05,.2653400E-05,&
     & .4508900E-08,.5954300E-05,.8275100E-05,.8742200E-05,.2815900E-05,&
     & .4217700E-08,.4110400E-05,.5691400E-05,.6085200E-05,.1807800E-05,&
     & .4079500E-08,.4358000E-05,.6033400E-05,.6427400E-05,.1942100E-05,&
     & .3935300E-08,.4581400E-05,.6340700E-05,.6744300E-05,.2076700E-05,&
     & .3801200E-08,.4780300E-05,.6619600E-05,.7014400E-05,.2211400E-05,&
     & .3675600E-08,.4968200E-05,.6888400E-05,.7253900E-05,.2345800E-05,&
     & .3416200E-08,.3414100E-05,.4722400E-05,.5041200E-05,.1496500E-05,&
     & .3319600E-08,.3621500E-05,.5003700E-05,.5324100E-05,.1607800E-05,&
     & .3219500E-08,.3806400E-05,.5258300E-05,.5581700E-05,.1718800E-05,&
     & .3109900E-08,.3971500E-05,.5488100E-05,.5800600E-05,.1830700E-05,&
     & .3007200E-08,.4127000E-05,.5706500E-05,.5997400E-05,.1941900E-05,&
     & .2800800E-08,.2809400E-05,.3880100E-05,.4139300E-05,.1224800E-05,&
     & .2712100E-08,.2983600E-05,.4115500E-05,.4374800E-05,.1316800E-05,&
     & .2648200E-08,.3139700E-05,.4328200E-05,.4588500E-05,.1408700E-05,&
     & .2557600E-08,.3277900E-05,.4520000E-05,.4770600E-05,.1501300E-05,&
     & .2472700E-08,.3408200E-05,.4702300E-05,.4935400E-05,.1593500E-05,&
     & .2298400E-08,.2310600E-05,.3185600E-05,.3396100E-05,.1001600E-05,&
     & .2222000E-08,.2456300E-05,.3382600E-05,.3591500E-05,.1077600E-05,&
     & .2158800E-08,.2588200E-05,.3560000E-05,.3769600E-05,.1153800E-05,&
     & .2104000E-08,.2703300E-05,.3719900E-05,.3921300E-05,.1230400E-05,&
     & .2033900E-08,.2812200E-05,.3872800E-05,.4059700E-05,.1306700E-05,&
     & .1886700E-08,.1899000E-05,.2614700E-05,.2785500E-05,.8189500E-06,&
     & .1821900E-08,.2021100E-05,.2778900E-05,.2948200E-05,.8816800E-06,&
     & .1767100E-08,.2132400E-05,.2927200E-05,.3096200E-05,.9447500E-06,&
     & .1720100E-08,.2228700E-05,.3060700E-05,.3223100E-05,.1008000E-05,&
     & .1672900E-08,.2319700E-05,.3188500E-05,.3338900E-05,.1071300E-05,&
     & .1556700E-08,.1548600E-05,.2129900E-05,.2269700E-05,.6641800E-06,&
     & .1503000E-08,.1652500E-05,.2268600E-05,.2406300E-05,.7159100E-06,&
     & .1455900E-08,.1746800E-05,.2393800E-05,.2530400E-05,.7680400E-06,&
     & .1414700E-08,.1828400E-05,.2506200E-05,.2637700E-05,.8205300E-06,&
     & .1380800E-08,.1905100E-05,.2614200E-05,.2736200E-05,.8728200E-06/

      data   absb(701:875, 4)   /                                       &
     & .1285500E-08,.1262200E-05,.1732600E-05,.1847200E-05,.5377800E-06,&
     & .1241400E-08,.1348600E-05,.1849400E-05,.1961200E-05,.5806200E-06,&
     & .1200600E-08,.1429000E-05,.1955100E-05,.2066500E-05,.6236500E-06,&
     & .1165500E-08,.1498400E-05,.2050600E-05,.2157000E-05,.6670700E-06,&
     & .1136600E-08,.1563000E-05,.2141600E-05,.2240400E-05,.7103500E-06,&
     & .1061600E-08,.1027300E-05,.1408500E-05,.1503400E-05,.4351400E-06,&
     & .1025000E-08,.1099700E-05,.1507000E-05,.1597300E-05,.4706100E-06,&
     & .9903000E-09,.1167900E-05,.1595300E-05,.1686100E-05,.5061300E-06,&
     & .9606200E-09,.1227200E-05,.1676600E-05,.1764200E-05,.5421600E-06,&
     & .9344300E-09,.1281700E-05,.1752900E-05,.1833300E-05,.5778400E-06,&
     & .8788300E-09,.8319900E-06,.1140200E-05,.1216800E-05,.3502500E-06,&
     & .8482900E-09,.8925400E-06,.1222500E-05,.1295800E-05,.3796000E-06,&
     & .8190300E-09,.9507200E-06,.1296600E-05,.1370800E-05,.4089300E-06,&
     & .7936900E-09,.1002000E-05,.1366500E-05,.1437900E-05,.4386200E-06,&
     & .7710800E-09,.1047800E-05,.1430200E-05,.1496200E-05,.4681700E-06,&
     & .7283800E-09,.6708000E-06,.9196200E-06,.9810200E-06,.2810700E-06,&
     & .7027900E-09,.7227100E-06,.9885300E-06,.1049100E-05,.3054000E-06,&
     & .6785800E-09,.7720300E-06,.1052200E-05,.1112200E-05,.3295800E-06,&
     & .6565100E-09,.8162500E-06,.1111500E-05,.1169700E-05,.3540700E-06,&
     & .6372700E-09,.8549500E-06,.1165100E-05,.1219200E-05,.3785000E-06,&
     & .6037900E-09,.5404700E-06,.7412800E-06,.7910000E-06,.2253500E-06,&
     & .5822800E-09,.5842900E-06,.7985400E-06,.8484100E-06,.2454800E-06,&
     & .5622000E-09,.6260600E-06,.8528300E-06,.9015700E-06,.2654100E-06,&
     & .5432000E-09,.6642300E-06,.9029600E-06,.9501500E-06,.2855800E-06,&
     & .5268600E-09,.6971900E-06,.9483000E-06,.9925400E-06,.3058000E-06,&
     & .5010800E-09,.4336100E-06,.5955400E-06,.6343200E-06,.1802300E-06,&
     & .4829100E-09,.4712700E-06,.6434900E-06,.6851000E-06,.1967700E-06,&
     & .4661200E-09,.5062100E-06,.6894200E-06,.7292900E-06,.2132500E-06,&
     & .4500400E-09,.5390300E-06,.7316900E-06,.7709500E-06,.2298600E-06,&
     & .4360000E-09,.5674000E-06,.7708100E-06,.8071400E-06,.2465400E-06,&
     & .4166500E-09,.3461400E-06,.4756500E-06,.5059800E-06,.1434500E-06,&
     & .4011900E-09,.3777900E-06,.5164200E-06,.5501500E-06,.1569700E-06,&
     & .3870200E-09,.4075200E-06,.5552500E-06,.5875900E-06,.1706400E-06,&
     & .3737200E-09,.4356300E-06,.5907600E-06,.6231800E-06,.1843000E-06,&
     & .3613900E-09,.4604100E-06,.6247400E-06,.6551500E-06,.1980000E-06/

      data   absb(876:1050, 4)   /                                      &
     & .3465800E-09,.2749400E-06,.3792400E-06,.4034900E-06,.1140700E-06,&
     & .3334000E-09,.3024300E-06,.4134000E-06,.4406600E-06,.1250800E-06,&
     & .3214300E-09,.3275600E-06,.4463500E-06,.4730800E-06,.1363600E-06,&
     & .3103200E-09,.3514900E-06,.4766900E-06,.5030900E-06,.1476300E-06,&
     & .2996900E-09,.3732300E-06,.5054200E-06,.5306200E-06,.1589200E-06,&
     & .2884600E-09,.2177800E-06,.3016800E-06,.3215800E-06,.9060700E-07,&
     & .2771600E-09,.2413900E-06,.3307100E-06,.3521400E-06,.9957700E-07,&
     & .2670000E-09,.2630400E-06,.3580400E-06,.3804900E-06,.1088100E-06,&
     & .2576700E-09,.2831300E-06,.3839400E-06,.4056600E-06,.1181400E-06,&
     & .2487500E-09,.3018300E-06,.4083200E-06,.4290600E-06,.1274200E-06,&
     & .2399400E-09,.1727600E-06,.2403900E-06,.2564400E-06,.7212400E-07,&
     & .2302800E-09,.1927500E-06,.2642900E-06,.2811700E-06,.7938800E-07,&
     & .2216200E-09,.2110600E-06,.2875800E-06,.3060600E-06,.8692200E-07,&
     & .2137700E-09,.2280700E-06,.3094700E-06,.3271400E-06,.9462800E-07,&
     & .2064100E-09,.2440000E-06,.3296200E-06,.3471500E-06,.1022800E-06,&
     & .1996000E-09,.1370900E-06,.1910100E-06,.2040700E-06,.5756600E-07,&
     & .1913200E-09,.1535200E-06,.2112000E-06,.2245500E-06,.6330800E-07,&
     & .1839500E-09,.1692400E-06,.2306800E-06,.2456300E-06,.6942100E-07,&
     & .1773200E-09,.1835200E-06,.2492600E-06,.2638300E-06,.7577300E-07,&
     & .1711700E-09,.1971500E-06,.2663000E-06,.2806800E-06,.8209700E-07,&
     & .1661600E-09,.1085200E-06,.1519100E-06,.1613700E-06,.4597500E-07,&
     & .1590100E-09,.1218800E-06,.1685500E-06,.1794300E-06,.5045400E-07,&
     & .1527200E-09,.1353500E-06,.1850200E-06,.1970500E-06,.5542400E-07,&
     & .1471100E-09,.1476500E-06,.2004200E-06,.2125200E-06,.6061600E-07,&
     & .1419400E-09,.1590700E-06,.2148400E-06,.2267300E-06,.6582600E-07,&
     & .1383900E-09,.8570400E-07,.1202300E-06,.1274400E-06,.3652300E-07,&
     & .1322300E-09,.9660000E-07,.1343800E-06,.1431200E-06,.4015200E-07,&
     & .1268400E-09,.1080400E-06,.1478400E-06,.1573500E-06,.4420700E-07,&
     & .1220800E-09,.1185100E-06,.1609300E-06,.1710800E-06,.4843500E-07,&
     & .1177200E-09,.1281100E-06,.1731300E-06,.1829300E-06,.5273300E-07,&
     & .1151600E-09,.6780200E-07,.9540000E-07,.1006500E-06,.2904000E-07,&
     & .1098600E-09,.7692700E-07,.1071200E-06,.1142800E-06,.3204400E-07,&
     & .1052600E-09,.8629500E-07,.1183300E-06,.1258000E-06,.3533500E-07,&
     & .1012200E-09,.9519000E-07,.1293600E-06,.1376300E-06,.3876400E-07,&
     & .9754500E-10,.1032400E-06,.1396900E-06,.1476800E-06,.4231200E-07/

      data   absb(1051:1175, 4)   /                                     &
     & .9583200E-10,.5361000E-07,.7598100E-07,.7966700E-07,.2311700E-07,&
     & .9128100E-10,.6113000E-07,.8532900E-07,.9100500E-07,.2565700E-07,&
     & .8733500E-10,.6882500E-07,.9473600E-07,.1007500E-06,.2826400E-07,&
     & .8389700E-10,.7643500E-07,.1039600E-06,.1107000E-06,.3105700E-07,&
     & .8081000E-10,.8327800E-07,.1126900E-06,.1192800E-06,.3397200E-07,&
     & .7981000E-10,.4232200E-07,.6037400E-07,.6292600E-07,.1840400E-07,&
     & .7586400E-10,.4855900E-07,.6803900E-07,.7224100E-07,.2054800E-07,&
     & .7248500E-10,.5477300E-07,.7579900E-07,.8075700E-07,.2258100E-07,&
     & .6956000E-10,.6122400E-07,.8355400E-07,.8896700E-07,.2485900E-07,&
     & .6695100E-10,.6708400E-07,.9079700E-07,.9624600E-07,.2724500E-07,&
     & .6652400E-10,.3332000E-07,.4751600E-07,.4954100E-07,.1462000E-07,&
     & .6308100E-10,.3844300E-07,.5402700E-07,.5714200E-07,.1637000E-07,&
     & .6018700E-10,.4358700E-07,.6061100E-07,.6460800E-07,.1803600E-07,&
     & .5768900E-10,.4897000E-07,.6691500E-07,.7123300E-07,.1989500E-07,&
     & .5547800E-10,.5392500E-07,.7304500E-07,.7761800E-07,.2183100E-07,&
     & .5544700E-10,.2624000E-07,.3743500E-07,.3898900E-07,.1160600E-07,&
     & .5244200E-10,.3046000E-07,.4295500E-07,.4523300E-07,.1304700E-07,&
     & .4996200E-10,.3476100E-07,.4841800E-07,.5168000E-07,.1443100E-07,&
     & .4782600E-10,.3915200E-07,.5362600E-07,.5704700E-07,.1593400E-07,&
     & .4595400E-10,.4338000E-07,.5878600E-07,.6253800E-07,.1751100E-07,&
     & .4574000E-10,.2124800E-07,.3026100E-07,.3157700E-07,.9422100E-08,&
     & .4321000E-10,.2473000E-07,.3494800E-07,.3672200E-07,.1060800E-07,&
     & .4114100E-10,.2828800E-07,.3941000E-07,.4205200E-07,.1175400E-07,&
     & .3936100E-10,.3190400E-07,.4374000E-07,.4652700E-07,.1298600E-07,&
     & .3780500E-10,.3544200E-07,.4802000E-07,.5108300E-07,.1427800E-07/

      data   absb(  1:175, 5)   /                                       &
     & .7904500E-05,.5851200E-03,.8096700E-03,.8857800E-03,.3482400E-03,&
     & .9280700E-05,.6181800E-03,.8608300E-03,.9243700E-03,.3787000E-03,&
     & .1062000E-04,.6502500E-03,.9061700E-03,.9629000E-03,.4063900E-03,&
     & .1184700E-04,.6808600E-03,.9476700E-03,.9981500E-03,.4314900E-03,&
     & .1289700E-04,.7094100E-03,.9842600E-03,.1029600E-02,.4545900E-03,&
     & .6568300E-05,.4879800E-03,.6757000E-03,.7375500E-03,.2917200E-03,&
     & .7695700E-05,.5155500E-03,.7185600E-03,.7697800E-03,.3170900E-03,&
     & .8785200E-05,.5424300E-03,.7560400E-03,.8014200E-03,.3406400E-03,&
     & .9762200E-05,.5681700E-03,.7903800E-03,.8309100E-03,.3609500E-03,&
     & .1061400E-04,.5920800E-03,.8201800E-03,.8569100E-03,.3809500E-03,&
     & .5464300E-05,.4066000E-03,.5637100E-03,.6128600E-03,.2443500E-03,&
     & .6388300E-05,.4297300E-03,.5991400E-03,.6409300E-03,.2647500E-03,&
     & .7259300E-05,.4521800E-03,.6300400E-03,.6665900E-03,.2848600E-03,&
     & .8052600E-05,.4737300E-03,.6584700E-03,.6912800E-03,.3020000E-03,&
     & .8736400E-05,.4939300E-03,.6831800E-03,.7132100E-03,.3193000E-03,&
     & .4548200E-05,.3387200E-03,.4700300E-03,.5092000E-03,.2047300E-03,&
     & .5289300E-05,.3578500E-03,.4989800E-03,.5330600E-03,.2217100E-03,&
     & .5998900E-05,.3767300E-03,.5246700E-03,.5546100E-03,.2380300E-03,&
     & .6642600E-05,.3947000E-03,.5482700E-03,.5751500E-03,.2524400E-03,&
     & .7175900E-05,.4119300E-03,.5690900E-03,.5936100E-03,.2677300E-03,&
     & .3769000E-05,.2817700E-03,.3915300E-03,.4226500E-03,.1715500E-03,&
     & .4379000E-05,.2977800E-03,.4153500E-03,.4428800E-03,.1856300E-03,&
     & .4954500E-05,.3137100E-03,.4366400E-03,.4609000E-03,.1990800E-03,&
     & .5456300E-05,.3288600E-03,.4563700E-03,.4785400E-03,.2109700E-03,&
     & .5887900E-05,.3435900E-03,.4741600E-03,.4938700E-03,.2244000E-03,&
     & .3126000E-05,.2343200E-03,.3259600E-03,.3508200E-03,.1434800E-03,&
     & .3618500E-05,.2478600E-03,.3456100E-03,.3676500E-03,.1554600E-03,&
     & .4073200E-05,.2612000E-03,.3634300E-03,.3832500E-03,.1662300E-03,&
     & .4483900E-05,.2740300E-03,.3799700E-03,.3981100E-03,.1765900E-03,&
     & .4830400E-05,.2867400E-03,.3951300E-03,.4111400E-03,.1881300E-03,&
     & .2585300E-05,.1948300E-03,.2712800E-03,.2912600E-03,.1196900E-03,&
     & .2983400E-05,.2062300E-03,.2874900E-03,.3054100E-03,.1297800E-03,&
     & .3354600E-05,.2175000E-03,.3026700E-03,.3187100E-03,.1387400E-03,&
     & .3686300E-05,.2284200E-03,.3165600E-03,.3311500E-03,.1478900E-03,&
     & .3963400E-05,.2394200E-03,.3294400E-03,.3422000E-03,.1574200E-03/

      data   absb(176:350, 5)   /                                       &
     & .2148600E-05,.1622100E-03,.2260900E-03,.2423600E-03,.1001400E-03,&
     & .2473500E-05,.1717800E-03,.2394900E-03,.2539900E-03,.1085700E-03,&
     & .2774400E-05,.1813600E-03,.2522600E-03,.2653500E-03,.1160400E-03,&
     & .3041400E-05,.1907600E-03,.2639700E-03,.2756600E-03,.1239800E-03,&
     & .3261000E-05,.2001900E-03,.2748800E-03,.2851000E-03,.1319900E-03,&
     & .1787700E-05,.1350700E-03,.1884500E-03,.2016300E-03,.8387900E-04,&
     & .2051800E-05,.1431900E-03,.1996800E-03,.2113400E-03,.9085800E-04,&
     & .2295300E-05,.1513700E-03,.2103800E-03,.2209400E-03,.9709800E-04,&
     & .2509800E-05,.1594700E-03,.2201900E-03,.2294800E-03,.1040200E-03,&
     & .2683000E-05,.1674900E-03,.2295000E-03,.2376100E-03,.1106900E-03,&
     & .1502200E-05,.1129700E-03,.1577300E-03,.1681900E-03,.7080800E-04,&
     & .1715700E-05,.1199800E-03,.1671400E-03,.1764400E-03,.7634500E-04,&
     & .1911100E-05,.1269000E-03,.1761100E-03,.1845400E-03,.8180400E-04,&
     & .2081200E-05,.1338500E-03,.1842500E-03,.1916400E-03,.8759100E-04,&
     & .2214400E-05,.1406600E-03,.1923500E-03,.1986700E-03,.9322800E-04,&
     & .1262100E-05,.9468100E-04,.1321000E-03,.1403200E-03,.5963000E-04,&
     & .1434600E-05,.1006100E-03,.1400900E-03,.1475100E-03,.6414200E-04,&
     & .1590800E-05,.1065500E-03,.1474900E-03,.1541300E-03,.6898800E-04,&
     & .1725300E-05,.1124600E-03,.1543700E-03,.1601300E-03,.7389300E-04,&
     & .1827200E-05,.1182500E-03,.1613000E-03,.1662700E-03,.7864900E-04,&
     & .1061100E-05,.7941900E-04,.1107900E-03,.1173600E-03,.5024400E-04,&
     & .1199600E-05,.8448600E-04,.1174900E-03,.1234500E-03,.5405400E-04,&
     & .1324100E-05,.8956500E-04,.1236400E-03,.1288300E-03,.5825900E-04,&
     & .1430000E-05,.9459200E-04,.1294900E-03,.1340600E-03,.6230000E-04,&
     & .1507000E-05,.9954500E-04,.1353800E-03,.1392800E-03,.6648500E-04,&
     & .8924800E-06,.6675700E-04,.9304300E-04,.9826900E-04,.4239600E-04,&
     & .1003400E-05,.7108200E-04,.9865200E-04,.1033200E-03,.4571900E-04,&
     & .1102400E-05,.7540400E-04,.1037500E-03,.1078200E-03,.4922800E-04,&
     & .1183500E-05,.7969400E-04,.1088200E-03,.1123400E-03,.5264900E-04,&
     & .1240600E-05,.8391500E-04,.1137600E-03,.1167700E-03,.5626400E-04,&
     & .7514700E-06,.5624800E-04,.7829100E-04,.8244100E-04,.3578100E-04,&
     & .8402500E-06,.5992900E-04,.8291700E-04,.8656200E-04,.3871500E-04,&
     & .9172800E-06,.6361800E-04,.8723100E-04,.9044200E-04,.4168200E-04,&
     & .9794700E-06,.6728500E-04,.9160600E-04,.9431100E-04,.4461000E-04,&
     & .1022200E-05,.7087900E-04,.9575900E-04,.9795400E-04,.4767800E-04/

      data   absb(351:525, 5)   /                                       &
     & .6322800E-06,.4744800E-04,.6592600E-04,.6921100E-04,.3031200E-04,&
     & .7018700E-06,.5058300E-04,.6975200E-04,.7262200E-04,.3284400E-04,&
     & .7626300E-06,.5372300E-04,.7343100E-04,.7594300E-04,.3528300E-04,&
     & .8097500E-06,.5688400E-04,.7713900E-04,.7923500E-04,.3783700E-04,&
     & .8409900E-06,.5993700E-04,.8067000E-04,.8219800E-04,.4039800E-04,&
     & .5306200E-06,.4005400E-04,.5553600E-04,.5810500E-04,.2568300E-04,&
     & .5858100E-06,.4274800E-04,.5871900E-04,.6098800E-04,.2782900E-04,&
     & .6328300E-06,.4544500E-04,.6192700E-04,.6380800E-04,.2991400E-04,&
     & .6684100E-06,.4814100E-04,.6503300E-04,.6656700E-04,.3210100E-04,&
     & .6913200E-06,.5074400E-04,.6800200E-04,.6902400E-04,.3419600E-04,&
     & .4448200E-06,.3387600E-04,.4681300E-04,.4882000E-04,.2182200E-04,&
     & .4883000E-06,.3617300E-04,.4953400E-04,.5126700E-04,.2359800E-04,&
     & .5252000E-06,.3850700E-04,.5226400E-04,.5370000E-04,.2540700E-04,&
     & .5514200E-06,.4079000E-04,.5487900E-04,.5593400E-04,.2723300E-04,&
     & .5682100E-06,.4300500E-04,.5737400E-04,.5799500E-04,.2893900E-04,&
     & .3724000E-06,.2868800E-04,.3949800E-04,.4108200E-04,.1854200E-04,&
     & .4069700E-06,.3066200E-04,.4183200E-04,.4313900E-04,.2003700E-04,&
     & .4351800E-06,.3265700E-04,.4413700E-04,.4519800E-04,.2159200E-04,&
     & .4545900E-06,.3460000E-04,.4634200E-04,.4703000E-04,.2309900E-04,&
     & .4667400E-06,.3646800E-04,.4842700E-04,.4875200E-04,.2452400E-04,&
     & .3116700E-06,.2432800E-04,.3336500E-04,.3457700E-04,.1577500E-04,&
     & .3388000E-06,.2603300E-04,.3537600E-04,.3634300E-04,.1703700E-04,&
     & .3602200E-06,.2772800E-04,.3730600E-04,.3802500E-04,.1833400E-04,&
     & .3744300E-06,.2938100E-04,.3915300E-04,.3955100E-04,.1955900E-04,&
     & .3830400E-06,.3094000E-04,.4088800E-04,.4101200E-04,.2073500E-04,&
     & .2606500E-06,.2066600E-04,.2822600E-04,.2912800E-04,.1340300E-04,&
     & .2817900E-06,.2213000E-04,.2993400E-04,.3064200E-04,.1449500E-04,&
     & .2979100E-06,.2357200E-04,.3155400E-04,.3200500E-04,.1557600E-04,&
     & .3081300E-06,.2496300E-04,.3310100E-04,.3327800E-04,.1658200E-04,&
     & .3141200E-06,.2626100E-04,.3451000E-04,.3449700E-04,.1753300E-04,&
     & .2176800E-06,.1758100E-04,.2390300E-04,.2455800E-04,.1140500E-04,&
     & .2340900E-06,.1882700E-04,.2533300E-04,.2581200E-04,.1233100E-04,&
     & .2459600E-06,.2005300E-04,.2669600E-04,.2694200E-04,.1320000E-04,&
     & .2533800E-06,.2121100E-04,.2797800E-04,.2801200E-04,.1403700E-04,&
     & .2573500E-06,.2229200E-04,.2911300E-04,.2901900E-04,.1478900E-04/

      data   absb(526:700, 5)   /                                       &
     & .1811100E-06,.1493300E-04,.2022100E-04,.2069100E-04,.9685300E-05,&
     & .1937300E-06,.1599300E-04,.2141300E-04,.2170800E-04,.1045900E-04,&
     & .2026200E-06,.1702500E-04,.2255100E-04,.2265100E-04,.1117000E-04,&
     & .2080200E-06,.1799200E-04,.2360300E-04,.2354000E-04,.1184700E-04,&
     & .2106800E-06,.1889200E-04,.2452200E-04,.2437000E-04,.1244400E-04,&
     & .1494000E-06,.1260400E-04,.1700100E-04,.1733900E-04,.8163000E-05,&
     & .1593400E-06,.1350500E-04,.1800800E-04,.1818300E-04,.8789000E-05,&
     & .1662600E-06,.1437400E-04,.1896300E-04,.1897600E-04,.9383800E-05,&
     & .1704000E-06,.1518700E-04,.1982400E-04,.1972000E-04,.9922900E-05,&
     & .1722900E-06,.1593900E-04,.2058200E-04,.2039400E-04,.1041200E-04,&
     & .1222400E-06,.1056000E-04,.1420300E-04,.1444700E-04,.6819600E-05,&
     & .1303000E-06,.1132600E-04,.1505500E-04,.1515600E-04,.7332100E-05,&
    